<?php
/**
 * Plugin Name: Ziina
 * Description: Accept payments via Ziina
 * Author: Ziina
 * Author URI: https://ziina.com/
 * Text Domain: ziina
 * Domain Path: /languages
 * WC requires at least: 4.8
 * WC tested up to: 8.7.0
 * Requires at least: 5.7
 * Requires PHP: 8.1
 * Version: 1.2.18
 *
 * @package ZiinaPayment
 */

namespace ZiinaPayment;

defined( 'ABSPATH' ) || exit();

/**
 * Class Main
 *
 * @package ZiinaPayment
 * @since   1.0.0
 */
class Main {
	/**
	 * Class instance
	 *
	 * @var Main|null
	 */
	private static $instance = null;

	/**
	 * Plugin id
	 *
	 * @var string
	 */
	public $plugin_id = 'ziina';

	/**
	 * Plugin version
	 *
	 * @var string
	 */
	public $version = '1.2.18';

	/**
	 * Plugin url
	 *
	 * @var string
	 */
	public $plugin_url;

	/**
	 * Plugin path
	 *
	 * @var string
	 */
	public $plugin_path;

	/**
	 * Assets url
	 *
	 * @var string
	 */
	public $assets_url;

	/**
	 * Settings array
	 *
	 * @var array|null
	 */
	private $settings = null;

	/**
	 * Admin class instance
	 *
	 * @var Api\Main|null
	 */
	private $api = null;

	/**
	 * Admin class instance
	 *
	 * @var Ajax\Main|null
	 */
	private $ajax;

	/**
	 * Main constructor.
	 */
	private function __construct() {
		$this->plugin_url  = plugin_dir_url( __FILE__ );
		$this->plugin_path = plugin_dir_path( __FILE__ );
		$this->assets_url  = $this->plugin_url . '/assets/';

		require_once 'vendor/autoload.php';

		load_plugin_textdomain( 'ziina', false, $this->plugin_path . 'languages/' );

		add_action('plugins_loaded', function() {
			set_error_handler([$this, 'handle_error'], E_ALL);
			register_shutdown_function([$this, 'handle_fatal_error']);
		});

		add_action( 'plugins_loaded', array( $this, 'action_plugins_loaded' ) );
		add_action( 'before_woocommerce_init', array( $this, 'before_woocommerce_hpos' ) );

		register_activation_hook( __FILE__, array( $this, 'activation_hook' ) );
		register_deactivation_hook( __FILE__, array( $this, 'deactivation_hook' ) );

		add_filter( 'woocommerce_payment_gateways', array( $this, 'filter_add_payment_gateway' ) );

		$this->ajax = new Ajax\Main();
	}

	/**
	 * Get api class instance.
	 *
	 * @return Api\Main
	 */
	public function api(): Api\Main {
		if ( is_null( $this->api ) ) {
			$this->api = new Api\Main();
		}

		return $this->api;
	}

	/**
	 * Get Ajax class instance
	 *
	 * @return Ajax\Main
	 */
	public function ajax(): Ajax\Main {
		return $this->ajax;
	}

	/**
	 * Disable plugin if Woocommerce not active
	 */
	public function action_plugins_loaded() {
		if ( ! class_exists( 'WC_Payment_Gateway' ) ) {
			$this->disable_plugin();
			return;
		}

		new Integrations\Main();

		if ( is_admin() ) {
			new Admin\OrderDetails();
		}
	}

	/**
	 * Declare compatibility with WooCommerce High-Performance Order Storage.
	 */
	public function before_woocommerce_hpos() {
		if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
			\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
		}
	}

	/**
	 * Add plugin gateway to Woo gateways list
	 *
	 * @param array $gateways Array of gateway classes or class names.
	 *
	 * @return array
	 */
	public function filter_add_payment_gateway( array $gateways ): array {
		$gateways[] = Gateway::class;

		return $gateways;
	}

	/**
	 * Activation hook
	 */
	public function activation_hook() {
	}

	/**
	 * Deactivation hook
	 */
	public function deactivation_hook() {
		$this->api()->delete_webhook();
		$this->gateway()->update_option('ziina_webhook_registered', false);
	}

	/**
	 * Get Gateway setting
	 *
	 * @param string $name Setting key.
	 *
	 * @return mixed
	 */
	public function get_setting( string $name ) {
		if ( is_null( $this->settings ) ) {
			$gateway = $this->gateway();

			if ( empty( $gateway ) ) {
				$this->log( 'Get_settings without gateway' );

				return null;
			}

			$this->settings = array();

			$is_test = $gateway->get_option( 'is_test' ) === 'yes';

			$this->settings['gateway_id']          = $gateway->id;
			$this->settings['enabled']             = $gateway->get_option( 'enabled' ) === 'yes';
			$this->settings['authorization_token'] = $gateway->get_option( 'authorization_token' );
			$this->settings['is_test']             = $is_test;
			$this->settings['logging']             = $is_test || $gateway->get_option( 'logging' ) === 'yes';
		}

		return $this->settings[ $name ] ?? null;
	}

	/**
	 * Get plugin gateway instance.
	 *
	 * @return Gateway|null Gateway instance.
	 */
	public function gateway(): ?Gateway {
		return WC()->payment_gateways()->payment_gateways()[ $this->plugin_id ] ?? null;
	}

	/**
	 * Wrapper of wc_get_template function
	 *
	 * @param string $template Template name.
	 * @param array  $args     Arguments.
	 * @param bool   $return   Return or echo. Echo by default.
	 *
	 * @return bool|string
	 */
	public function include_template( string $template, $args = array(), $return = false ) {
		if ( $return ) {
			ob_start();
		}

		wc_get_template(
			$template,
			$args,
			'',
			$this->plugin_path . 'templates/'
		);

		if ( $return ) {
			return ob_get_clean();
		}

		return true;
	}

	/**
	 * Disable plugin
	 */
	public function disable_plugin() {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';

		deactivate_plugins( $this->plugin_id . '/index.php' );
	}

	/**
	 * Add data to Woo logs
	 *
	 * @param array|string $data        Data to add to logs.
	 * @param string       $code_source Source of log in code.
	 */
	public function log( $data, string $code_source = '' ) {
		if ( ( defined( 'WP_DEBUG' ) && WP_DEBUG ) || $this->settings['logging'] ) {
			if ( empty( $code_source ) ) {
				$backtrace = debug_backtrace( DEBUG_BACKTRACE_IGNORE_ARGS, 2 )[1]; //phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_debug_backtrace

				$code_source  = isset( $backtrace['class'] ) ? $backtrace['class'] . '::' : '';
				$code_source .= $backtrace['function'] ?? '';
			}

			$data = array(
				'source' => $code_source,
				'data'   => $data,
			);

			wc_get_logger()->debug( print_r( $data, true ), array( 'source' => $this->plugin_id ) );// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r
		}
	}

	/**
	 * Get singleton instance
	 *
	 * @return Main
	 */
	public static function get_instance(): Main {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	public function handle_error($errno, $errstr, $errfile, $errline) {
		if (stripos($errfile, 'ziina') !== false) {
			if (class_exists('\\ZiinaPayment\\Logger\\Main')) {
				\ZiinaPayment\Logger\Main::init($this->api());
				\ZiinaPayment\Logger\Main::error("PHP error occured", [
					'error_type'	=> $errno,
					'error_message' => $errstr,
					'error_file' 		=> $errfile,
					'error_line' 		=> $errline
				]);
			} else {
				// Fallback to WC logging if Logger not available
				$this->log("PHP error: $errstr in $errfile on line $errline", 'handle_error');
			}
		}

    return false;
	}

	public function handle_fatal_error() {
    $error = error_get_last();

    if ($error !== null && stripos($error['file'], 'ziina') !== false) {
			if (class_exists('\\ZiinaPayment\\Logger\\Main')) {
				\ZiinaPayment\Logger\Main::init($this->api());
				\ZiinaPayment\Logger\Main::fatal("Fatal error", [
					'error_type' => $error['type'],
					'error_message' => $error['message'],
					'error_file' => $error['file'],
					'error_line' => $error['line']
				]);
			} else {
				// Fallback to WC logging if Logger not available
				$this->log("Fatal error: {$error['message']} in {$error['file']} on line {$error['line']}", 'handle_fatal_error');
			}
    }
	}
}

require_once 'main-class-shortcut.php';

ziina_payment();
