<?php
/**
 * Class Rex_WPVR_Telemetry
 *
 * Handles telemetry tracking for the WPVR plugin.
 *
 * @since 8.5.48
 */

class Rex_WPVR_Telemetry {

    /**
     * Rex_WPVR_Telemetry constructor.
     *
     * Initialize telemetry hooks for the plugin.
     *
     * @since 8.5.48
     */
    public function __construct() {
        add_action( 'wpvr_plugin_activated', array( $this, 'track_plugin_activation' ));
        add_action( 'transition_post_status', array( $this, 'track_first_tour_published' ), 10, 3);
        add_action( 'rex_wpvr_tour_created', array( $this, 'track_tour_created' ), 10, 1);
        add_action( 'current_screen', array( $this, 'track_page_view' ) );
        add_action( 'rex_wpvr_embadded_tour', array( $this, 'track_embadded_tour' ), 10, 1 );
    }

    /**
     * Track plugin activation
     *
     * Sends telemetry event when the plugin is activated.
     *
     * @since 8.5.48
     */
    public function track_plugin_activation() {
        coderex_telemetry_track(
            WPVR_FILE,
            'plugin_activation',
            array(
                'plugin_version' => defined('WPVR_VERSION') ? WPVR_VERSION : 'unknown',
                'activation_time' => current_time( 'c' ),
            )
        );
    }

    /**
     * Track the first published tour
     *
     * Sends telemetry when the first tour is published for the plugin.
     *
     * @param string $new_status The new post status
     * @param string $old_status The previous post status
     * @param object $post  \WP_Post The post object
     * @since 8.5.48
     */
    public function track_first_tour_published( $new_status, $old_status, $post ) {
        if ($post->post_type !== 'wpvr_item') {
            return;
        }


        if ($new_status === 'publish' && in_array($old_status, ['auto-draft', 'draft', 'new', ''])) {
            $tour_count = wp_count_posts('wpvr_item');
            $total_tours = $tour_count->publish + $tour_count->draft;
            do_action('rex_wpvr_tour_created', $post->post_title);
            if (1 === $total_tours) {
                coderex_telemetry_track(
                    WPVR_FILE,
                    'first_tour_published',
                    array(
                        'tour_title'    => $post->post_title,
                        'time'          => current_time('mysql'),
                    )
                );
            }
        } elseif ($new_status === 'publish' && $old_status === 'publish') {
            coderex_telemetry_track(
                WPVR_FILE,
                'tour_updated',
                array(
                    'tour_title' => $post->post_title,
                    'time' => current_time('mysql'),
                )
            );
        }
    }

    /**
     * Track tour creation
     *
     * Sends telemetry when a new tour is created.
     *
     * @param string $post_title The title of the created tour
     * @since 8.5.48
     */
    public function track_tour_created( $post_title ) {
        coderex_telemetry_track(
            WPVR_FILE,
            'tour_created',
            array(
                'tour_title' => isset( $post_title) ? $post_title : '',
                'time' => current_time('mysql'),
            )
        );
    }


    /**
     * Track page views
     *
     * Sends telemetry when specific admin pages for the plugin are viewed.
     *
     * @param WP_Screen $screen Current admin screen object
     * @return void
     * @since 8.5.48
     */
    public function track_page_view( $screen ) {
        if ( ! is_admin() || empty( $screen->id ) ) {
            return;
        }

        $page_map = array(
            '/wp-admin/edit.php?post_type=wpvr_item' => 'Tour list',
            '/wp-admin/admin.php?page=wpvr-settings' => 'Settings',
            '/wp-admin/admin.php?page=wpvr-analytics' => 'Analytics',
            '/wp-admin/admin.php?page=rex-wpvr-setup-wizard' => 'Setup wizard',
            '/wp-admin/admin.php?page=wpvrpro' => 'License',
        );

        $current_page = isset( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( $_SERVER['REQUEST_URI'] ) : '';
        if ( '' === $current_page ) {
            return;
        }

        $page_name = null;
        foreach ( $page_map as $fragment => $name ) {
            if ( strpos( $current_page, $fragment ) !== false ) {
                $page_name = $name;
                break;
            }
        }


        if ( null === $page_name && 
            strpos( $current_page, '/wp-admin/post.php' ) !== false && 
            strpos( $current_page, 'action=edit' ) !== false ) {
            if ( isset( $screen->post_type ) && 'wpvr_item' === $screen->post_type ) {
                $page_name = 'Tour edit';
            }
        }

        if ( null === $page_name ) {
            return;
        }

        $current_user = wp_get_current_user();
        if ( ! $current_user->exists() ) {
            return;
        }

        coderex_telemetry_track(
            WPVR_FILE,
            'page_view',
            array(
                'page' => $current_page,
                'page_name' => $page_name,
                'time' => current_time( 'mysql' ),
            )
        );
    }


    /**
     * Track embedded tour event
     *
     * Sends telemetry when a tour is embedded.
     *
     * @param int $tour_id The ID of the tour being embedded.
     * @return void
     * @since 8.5.48
     */
    public function track_embadded_tour( $tour_id ) {
        $tour = get_post( $tour_id );

        if ( ! $tour || $tour->post_type !== 'wpvr_item' ) {
            return;
        }

        coderex_telemetry_track(
            WPVR_FILE,
            'tour_embedded',
            array(
                'title' => $tour->post_title,
            )
        );
    }


}

new Rex_WPVR_Telemetry();