<?php
/*
 * License: GPLv3
 * License URI: https://www.gnu.org/licenses/gpl.txt
 * Copyright 2012-2025 Jean-Sebastien Morisset (https://wpsso.com/)
 */

if ( ! defined( 'ABSPATH' ) ) {

	die( 'These aren\'t the droids you\'re looking for.' );
}

if ( ! defined( 'WPSSO_PLUGINDIR' ) ) {

	die( 'Do. Or do not. There is no try.' );
}

if ( ! class_exists( 'WpssoAbstractWpMeta' ) ) {

	require_once WPSSO_PLUGINDIR . 'lib/abstract/wp-meta.php';
}

/*
 * Extended by the WpssoOpmPost class.
 */
if ( ! class_exists( 'WpssoPost' ) ) {

	class WpssoPost extends WpssoAbstractWpMeta {

		private static $saved_shortlink_url = null;	// Used by get_canonical_shortlink() and maybe_restore_shortlink().
		private static $cache_shortlinks    = array();	// Used by get_canonical_shortlink() and maybe_restore_shortlink().

		public function __construct( &$plugin ) {

			$this->p =& $plugin;

			if ( $this->p->debug->enabled ) {

				$this->p->debug->mark();
			}

			/*
			 * Maybe enable excerpts for pages.
			 */
			if ( ! empty( $this->p->options[ 'plugin_page_excerpt' ] ) ) {

				add_post_type_support( 'page', array( 'excerpt' ) );
			}

			/*
			 * Maybe register tags for pages.
			 */
			if ( $page_tag_taxonomy = SucomUtil::get_const( 'WPSSO_PAGE_TAG_TAXONOMY' ) ) {

				if ( ! empty( $this->p->options[ 'plugin_page_tags' ] ) ) {

					if ( ! taxonomy_exists( $page_tag_taxonomy ) ) {

						WpssoRegister::register_taxonomy_page_tag();
					}

				} elseif ( taxonomy_exists( $page_tag_taxonomy ) ) {

					unregister_taxonomy( $page_tag_taxonomy );
				}
			}

			/*
			 * This hook is fired once WordPress, plugins, and the theme are fully loaded and instantiated.
			 */
			add_action( 'wp_loaded', array( $this, 'add_wp_callbacks' ) );
		}

		/*
		 * Add WordPress action and filter callbacks.
		 */
		public function add_wp_callbacks() {

			if ( $this->p->debug->enabled ) {

				$this->p->debug->mark();
			}

			/*
			 * Since WPSSO Core v17.0.0.
			 *
			 * Register our post meta.
			 */
			$this->register_meta( $object_type = 'post', WPSSO_META_NAME );

			$is_admin = is_admin();	// Only check once.

			$doing_ajax = SucomUtilWP::doing_ajax();

			if ( $is_admin ) {

				$metabox_id = $this->p->cf[ 'meta' ][ 'id' ];

				add_action( 'wp_ajax_wpsso_get_metabox_postbox_id_' . $metabox_id . '_inside', array( $this, 'ajax_get_metabox_sso' ) );

				add_action( 'wp_ajax_wpsso_get_validate_submenu', array( $this, 'ajax_get_validate_submenu' ) );

				if ( ! empty( $_GET ) || 'post-new' === basename( $_SERVER[ 'PHP_SELF' ], '.php' ) ) {	// Skip some action hooks if no query argument(s).

					/*
					 * load_meta_page() priorities: 100 post, 200 user, 300 term.
					 *
					 * Sets the parent::$head_tags and parent::$head_info class properties.
					 */
					add_action( 'current_screen', array( $this, 'load_meta_page' ), 100, 1 );

					/*
					 * The 'add_meta_boxes' action fires after all built-in meta boxes have been added.
					 */
					add_action( 'add_meta_boxes', array( $this, 'add_meta_boxes' ), 10, 2 );
				}

				add_action( 'wp_after_insert_post', array( $this, 'after_insert_post' ), 10, 4 );

				/*
				 * The 'save_post' action is run after other post type specific actions, so we can use it to save
				 * post meta for any post type. Don't hook the 'clean_post_cache' action since 'save_post' is run
				 * after 'clean_post_cache' and our custom post meta has not been saved yet.
				 */
				add_action( 'save_post', array( $this, 'save_options' ), WPSSO_META_SAVE_PRIORITY );
				add_action( 'save_post', array( $this, 'clear_cache' ), WPSSO_META_CLEAR_PRIORITY );
				add_action( 'save_post', array( $this, 'refresh_cache' ), WPSSO_META_REFRESH_PRIORITY );

				/*
				 * The wp_insert_post() function returns after running the 'edit_attachment' action, so the
				 * 'save_post' action is never run for attachments.
				 */
				add_action( 'edit_attachment', array( $this, 'save_options' ), WPSSO_META_SAVE_PRIORITY );
				add_action( 'edit_attachment', array( $this, 'clear_cache' ), WPSSO_META_CLEAR_PRIORITY );
				add_action( 'edit_attachment', array( $this, 'refresh_cache' ), WPSSO_META_REFRESH_PRIORITY );
			}

			/*
			 * Add the columns when doing AJAX as well to allow Quick Edit to add the required columns.
			 */
			if ( $is_admin || $doing_ajax ) {

				/*
				 * Add edit table columns.
				 */
				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'adding column filters for posts' );
				}

				add_filter( 'manage_pages_columns', array( $this, 'add_page_column_headings' ), WPSSO_ADD_COLUMN_PRIORITY, 1 );
				add_filter( 'manage_posts_columns', array( $this, 'add_post_column_headings' ), WPSSO_ADD_COLUMN_PRIORITY, 2 );
				add_filter( 'manage_media_columns', array( $this, 'add_media_column_headings' ), WPSSO_ADD_COLUMN_PRIORITY, 1 );

				add_action( 'manage_pages_custom_column', array( $this, 'show_column_content' ), 10, 2 );
				add_action( 'manage_posts_custom_column', array( $this, 'show_column_content' ), 10, 2 );
				add_action( 'manage_media_custom_column', array( $this, 'show_column_content' ), 10, 2 );

				/*
				 * The 'parse_query' action is hooked once in the WpssoPost class to set the column orderby for
				 * post, term, and user edit tables.
				 */
				add_action( 'parse_query', array( $this, 'set_column_orderby' ), 10, 1 );
			}

			if ( ! empty( $this->p->options[ 'plugin_wp_shortlink' ] ) ) {	// Use Short URL for WP Shortlink.

				if ( ! empty( $this->p->options[ 'plugin_shortener' ] ) && $this->p->options[ 'plugin_shortener' ] !== 'none' ) {

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'adding pre_get_shortlink filters to shorten the url' );
					}

					add_filter( 'pre_get_shortlink', array( $this, 'get_canonical_shortlink' ), PHP_INT_MIN, 4 );
					add_filter( 'pre_get_shortlink', array( $this, 'maybe_restore_shortlink' ), PHP_INT_MAX, 4 );

					if ( function_exists( 'wpme_get_shortlink_handler' ) ) {

						if ( $this->p->debug->enabled ) {

							$this->p->debug->log( 'removing the jetpack pre_get_shortlink filter hook' );
						}

						remove_filter( 'pre_get_shortlink', 'wpme_get_shortlink_handler', 1 );
					}
				}
			}

			/*
			 * Maybe create or update the post column content.
			 */
			add_filter( 'get_post_metadata', array( $this, 'check_sortable_meta' ), 1000, 4 );

			/*
			 * Maybe inherit a featured image ID from the post/page parent, if the 'plugin_inherit_featured' option is
			 * enabled, and ignore saving the same featured image ID.
			 */
			add_filter( 'get_post_metadata', array( $this, 'get_post_metadata_thumbnail_id' ), PHP_INT_MAX, 4 );
			add_filter( 'update_post_metadata', array( $this, 'update_post_metadata_thumbnail_id' ), PHP_INT_MAX, 5 );
		}

		/*
		 * Get the $mod object for a post id.
		 */
		public function get_mod( $post_id ) {

			if ( $this->p->debug->enabled ) {

				$this->p->debug->mark_caller();

				$this->p->debug->log_args( array(
					'post_id' => $post_id,
				) );
			}

			static $local_fifo = array();

			/*
			 * Maybe return the array from the local cache.
			 */
			if ( isset( $local_fifo[ $post_id ] ) ) {

				if ( ! $this->md_cache_disabled ) {

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'exiting early: post id ' . $post_id . ' mod array from local cache' );
					}

					return $local_fifo[ $post_id ];

				} else unset( $local_fifo[ $post_id ] );
			}

			/*
			 * Maybe limit the number of array elements.
			 */
			$local_fifo = SucomUtil::array_slice_fifo( $local_fifo, WPSSO_CACHE_ARRAY_FIFO_MAX );

			$mod = self::get_mod_defaults();

			/*
			 * Common elements.
			 */
			$mod[ 'id' ]          = is_numeric( $post_id ) ? (int) $post_id : 0;	// Cast as integer.
			$mod[ 'name' ]        = 'post';
			$mod[ 'name_transl' ] = _x( 'post', 'module name', 'wpsso' );
			$mod[ 'obj' ]         =& $this;

			/*
			 * WpssoPost elements.
			 */
			$mod[ 'is_post' ]       = true;
			$mod[ 'is_home_page' ]  = SucomUtilWP::is_home_page( $post_id );					// Static front page (singular post).
			$mod[ 'is_home_posts' ] = $mod[ 'is_home_page' ] ? false : SucomUtilWP::is_home_posts( $post_id );	// Static posts page or blog archive page.
			$mod[ 'is_home' ]       = $mod[ 'is_home_page' ] || $mod[ 'is_home_posts' ] ? true : false;		// Home page (static or blog archive).

			if ( $mod[ 'id' ] ) {	// Just in case.

				$mod[ 'wp_obj' ] = SucomUtilWP::get_post_object( $mod[ 'id' ] );

				if ( $mod[ 'wp_obj' ] instanceof WP_Post ) {	// Just in case.

					$mod[ 'post_slug' ]               = get_post_field( 'post_name', $mod[ 'wp_obj' ] );		// Post name (aka slug).
					$mod[ 'post_type' ]               = get_post_type( $mod[ 'wp_obj' ] );				// Post type name.
					$mod[ 'post_mime_type' ]          = get_post_mime_type( $mod[ 'wp_obj' ] );			// Post mime type (ie. image/jpg).
					$mod[ 'post_status' ]             = get_post_status( $mod[ 'wp_obj' ] );			// Post status name.
					$mod[ 'post_author' ]             = (int) get_post_field( 'post_author', $mod[ 'wp_obj' ] );	// Post author id.
					$mod[ 'post_coauthors' ]          = array();
					$mod[ 'post_time' ]               = get_post_time( 'c', $gmt = true, $mod[ 'wp_obj' ] );		// ISO 8601 date or false.
					$mod[ 'post_timestamp' ]          = get_post_time( 'U', $gmt = true, $mod[ 'wp_obj' ] );		// Unix timestamp or false.
					$mod[ 'post_modified_time' ]      = get_post_modified_time( 'c', $gmt = true, $mod[ 'wp_obj' ] );	// ISO 8601 date or false.
					$mod[ 'post_modified_timestamp' ] = get_post_modified_time( 'U', $gmt = true, $mod[ 'wp_obj' ] );	// Unix timestamp or false.

					if ( is_array( $mod[ 'post_type' ] ) ) {	// Just in case.

						$notice_msg = sprintf( __( 'The <a href="%1$s">WordPress get_post_type() function</a> returned an array for WP_Post object ID %2$s.', 'wpsso' ), __( 'https://developer.wordpress.org/reference/functions/get_post_type/', 'wpsso' ), $mod[ 'id' ] ) . ' ';

						$notice_msg .= __( 'This function must not return an array, it must return false or a post type string.', 'wpsso' ) . ' ';

						$notice_msg .= '<pre><code>' . print_r( $mod[ 'post_type' ], true ) . '</code></pre>';

						if ( $this->p->notice->is_admin_pre_notices() ) {

							$this->p->notice->err( $notice_msg );
						}

						$error_pre  = sprintf( '%s error:', __METHOD__ );

						SucomUtil::safe_error_log( $error_pre . ' ' . $notice_msg, $strip_html = true );

						if ( $this->p->debug->enabled ) {

							$this->p->debug->log( 'get_post_type() returned an array for WP_Post object id ' . $mod[ 'id' ] );

							$this->p->debug->log_arr( 'post_type', $mod[ 'post_type' ] );
						}
					}

					/*
					 * See WpssoIntegEcomWooCommerce->filter_get_post_type().
					 */
					$mod[ 'post_type' ] = apply_filters( 'wpsso_get_post_type', $mod[ 'post_type' ], $post_id );

					if ( ! empty( $mod[ 'post_type' ] ) && is_string( $mod[ 'post_type' ] ) ) {	// Not false or empty string.

						$mod[ 'is_attachment' ] = 'attachment' === $mod[ 'post_type' ] ? true : false;		// Post type is 'attachment'.

						$post_type_obj = get_post_type_object( $mod[ 'post_type' ] );

						if ( $post_type_obj instanceof WP_Post_Type ) {	// Just in case.

							if ( isset( $post_type_obj->labels->name ) ) {

								$mod[ 'post_type_label_plural' ] = $post_type_obj->labels->name;
							}

							if ( isset( $post_type_obj->labels->singular_name ) ) {

								$mod[ 'post_type_label_single' ] = $post_type_obj->labels->singular_name;
							}

							if ( isset( $post_type_obj->public ) ) {

								$mod[ 'is_public' ] = $post_type_obj->public ? true : false;
							}

							$mod[ 'is_post_type_archive' ] = SucomUtilWP::is_post_type_archive( $post_type_obj, $mod[ 'post_slug' ] );

							$mod[ 'is_archive' ] = $mod[ 'is_post_type_archive' ];
						}

						unset( $post_type_obj );	// Done with $post_type_obj.
					}

					/*
					 * If we have a post with content (ie. singular), then check for paging in the content.
					 */
					if ( ! $mod[ 'is_post_type_archive' ] && ! $mod[ 'is_home_posts' ] ) {

						$mod[ 'paged_total' ] = substr_count( $mod[ 'wp_obj' ]->post_content, '<!--nextpage-->' ) + 1;
					}

					/*
					 * If the post has a parent, save the parent ID.
					 */
					if ( ! empty( $mod[ 'wp_obj' ]->post_parent ) ) {

						$mod[ 'post_parent' ] = $mod[ 'wp_obj' ]->post_parent;	// Post parent id.
					}

					/*
					 * The post type might be public, but if the post itself is private, then mark the post as not public.
					 *
					 * See https://wordpress.org/support/article/post-status/#default-statuses.
					 */
					if ( 'private' === $mod[ 'post_status' ] ) {

						$mod[ 'is_public' ] = false;
					}

					/*
					 * Find the post mime type group and subgroup values.
					 *
					 * See wp_post_mime_type_where() in wordpress/wp-includes/post.php.
					 */
					if ( ! empty( $mod[ 'post_mime_type' ] ) ) {

						if ( false !== $slashpos = strpos( $mod[ 'post_mime_type' ], '/' ) ) {

							$mod[ 'post_mime_group' ] = preg_replace( '/[^-*.a-zA-Z0-9]/', '',
								substr( $mod[ 'post_mime_type' ], 0, $slashpos ) );

							$mod[ 'post_mime_subgroup' ] = preg_replace( '/[^-*.+a-zA-Z0-9]/', '',
								substr( $mod[ 'post_mime_type' ], $slashpos + 1 ) );

						} else {

							$mod[ 'post_mime_group' ] = preg_replace( '/[^-*.a-zA-Z0-9]/', '', $mod[ 'post_mime_type' ] );

							$mod[ 'post_mime_subgroup' ] = '*';
						}
					}

				} else $mod[ 'wp_obj' ] = false;
			}

			/*
			 * Filter the post mod array.
			 *
			 * See WpssoIntegUserCoAuthors->filter_get_post_mod().
			 */
			$mod = apply_filters( 'wpsso_get_post_mod', $mod, $post_id );

			if ( $this->p->debug->enabled ) {

				$this->p->debug->log_arr( 'mod', $mod );
			}

			/*
			 * Maybe save the array to the local cache.
			 */
			if ( ! $this->md_cache_disabled ) {

				$local_fifo[ $post_id ] = $mod;

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log_size( 'local_fifo', $local_fifo );
				}
			}

			return $mod;
		}

		public function get_mod_wp_object( array $mod ) {

			if ( $mod[ 'wp_obj' ] instanceof WP_Post ) {

				return $mod[ 'wp_obj' ];
			}

			return SucomUtilWP::get_post_object( $mod[ 'id' ] );
		}

		/*
		 * Check if the post type matches a pre-defined Open Graph type.
		 *
		 * For example, a post type of 'organization' would return 'website' for the Open Graph type.
		 *
		 * Returns false or an Open Graph type string.
		 */
		public function get_post_type_og_type( $mod ) {

			if ( ! empty( $mod[ 'post_type' ] ) && is_string( $mod[ 'post_type' ] ) ) {	// Not false or empty string.

				if ( ! empty( $this->p->cf[ 'head' ][ 'og_type_by_post_type' ][ $mod[ 'post_type' ] ] ) ) {

					return $this->p->cf[ 'head' ][ 'og_type_by_post_type' ][ $mod[ 'post_type' ] ];
				}
			}

			return false;
		}

		/*
		 * Option handling methods:
		 *
		 *	get_defaults()
		 *	get_options()
		 *	save_options()
		 *	delete_options()
		 */
		public function get_options( $post_id, $md_key = false, $filter_opts = true, $merge_defs = false ) {

			if ( $this->p->debug->enabled ) {

				$this->p->debug->mark_caller();

				$this->p->debug->log_args( array(
					'post_id'     => $post_id,
					'md_key'      => $md_key,
					'filter_opts' => $filter_opts,
					'merge_defs'  => $merge_defs,
				) );
			}

			static $local_fifo = array();

			/*
			 * Use $post_id and $filter_opts to create the cache ID string, but DO NOT ADD $merge_defs.
			 */
			$cache_id = SucomUtil::get_assoc_salt( array( 'id' => $post_id, 'filter' => $filter_opts ) );

			/*
			 * Maybe initialize a new local cache element. Use isset() instead of empty() to allow for an empty array.
			 */
			if ( ! isset( $local_fifo[ $cache_id ] ) ) {

				/*
				 * Maybe limit the number of array elements.
				 */
				$local_fifo = SucomUtil::array_slice_fifo( $local_fifo, WPSSO_CACHE_ARRAY_FIFO_MAX );

				$local_fifo[ $cache_id ] = null;	// Create an element to reference.
			}

			$md_opts =& $local_fifo[ $cache_id ];	// Reference the local cache element.

			if ( null === $md_opts ) {	// Maybe read metadata into a new local cache element.

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'getting metadata for post id ' . $post_id );
				}

				$md_opts = self::get_meta( $post_id, WPSSO_META_NAME, $single = true );

				if ( ! is_array( $md_opts ) ) {

					$md_opts = array();	// WPSSO_META_NAME not found.
				}

				unset( $md_opts[ 'opt_filtered' ] );	// Just in case.

				/*
				 * Check if options need to be upgraded and saved.
				 */
				if ( $this->p->opt->is_upgrade_required( $md_opts ) ) {

					$md_opts = $this->upgrade_options( $md_opts, $post_id );

					self::update_meta( $post_id, WPSSO_META_NAME, $md_opts );
				}
			}

			if ( $filter_opts ) {

				if ( ! empty( $md_opts[ 'opt_filtered' ] ) ) {	// Set before calling filters to prevent recursion.

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'skipping filters: options already filtered' );
					}

				} else {

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'setting opt_filtered to 1' );
					}

					$md_opts[ 'opt_filtered' ] = 1;	// Set before calling filters to prevent recursion.

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'required call to WpssoPost->get_mod() for post ID ' . $post_id );
					}

					$mod = $this->get_mod( $post_id );

					/*
					 * See WpssoUtilBlocks->filter_import_content_blocks().
					 */
					if ( isset( $mod[ 'wp_obj' ]->post_content ) ) {

						$filter_name = 'wpsso_import_content_blocks';

						if ( $this->p->debug->enabled ) {

							$this->p->debug->log( 'applying filters "' . $filter_name . '"' );
						}

						$md_opts = apply_filters( $filter_name, $md_opts, $mod[ 'wp_obj' ]->post_content );

					} elseif ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'content property missing in post id ' . $post_id . ' object' );
					}

					/*
					 * The 'import_custom_fields' filter is executed before the 'wpsso_get_md_options' and
					 * 'wpsso_get_post_options' filters, custom field values may get overwritten by these
					 * filters.
					 *
					 * The 'import_custom_fields' filter is also executed before the 'wpsso_get_md_defaults'
					 * and 'wpsso_get_post_defaults' filters, so submitted form values that are identical to
					 * their defaults can be removed before saving the options array.
					 *
					 * See WpssoPost->get_options().
					 * See WpssoAbstractWpMeta->get_defaults().
					 * See WpssoUtilCustomFields->filter_import_custom_fields().
					 * See WpssoIntegEcomWooCommerce->add_mt_product().
					 * See WpssoIntegEcomWooAddGtin->filter_wc_variation_alt_options().
					 */
					$filter_name = 'wpsso_import_custom_fields';

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'applying filters "' . $filter_name . '"' );
					}

					$md_opts = apply_filters( $filter_name, $md_opts, $mod, self::get_meta( $post_id ) );

					/*
					 * Since WPSSO Core v14.2.0.
					 *
					 * See WpssoIntegEcomWooCommerce->add_mt_product().
					 */
					$filter_name = 'wpsso_import_product_attributes';

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'applying filters "' . $filter_name . '"' );
					}

					$md_opts = apply_filters( $filter_name, $md_opts, $mod, $mod[ 'wp_obj' ] );

					/*
					 * Since WPSSO Core v9.5.0.
					 *
					 * Overwrite parent options with those of the child, allowing only undefined child options
					 * to be inherited from the parent.
					 */
					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'inheriting parent metadata options for post id ' . $post_id );
					}

					$parent_opts = $this->get_inherited_md_opts( $mod );

					if ( ! empty( $parent_opts ) ) {

						$md_opts = array_merge( $parent_opts, $md_opts );
					}

					$filter_name = 'wpsso_get_md_options';

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'applying filters "' . $filter_name . '"' );
					}

					$md_opts = apply_filters( $filter_name, $md_opts, $mod );

					/*
					 * Hooked by several integration modules to provide information about the current content.
					 * e-Commerce integration modules will provide information on their product (price,
					 * condition, etc.) and disable these options in the Document SSO metabox.
					 */
					$filter_name = 'wpsso_get_' . $mod[ 'name' ] . '_options';

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'applying filters "' . $filter_name . '"' );
					}

					$md_opts = apply_filters( $filter_name, $md_opts, $post_id, $mod );

					$filter_name = 'wpsso_sanitize_md_options';;

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'applying filters "' . $filter_name . '"' );
					}

					$md_opts = apply_filters( $filter_name, $md_opts, $mod );
				}
			}

			/*
			 * Maybe save the array to the local cache.
			 */
			if ( $this->md_cache_disabled ) {

				$deref_md_opts = $local_fifo[ $cache_id ];	// Dereference.

				unset( $local_fifo, $md_opts );

				return $this->return_options( $post_id, $deref_md_opts, $md_key, $merge_defs );
			}

			return $this->return_options( $post_id, $md_opts, $md_key, $merge_defs );
		}

		/*
		 * Use $rel = false to extend WpssoAbstractWpMeta->save_options().
		 */
		public function save_options( $post_id, $rel = false ) {

			if ( $this->p->debug->enabled ) {

				$this->p->debug->log_args( array(
					'post_id' => $post_id,
				) );
			}

			if ( empty( $post_id ) ) {	// Just in case.

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: post id is empty' );
				}

				return;

			} elseif ( ! $this->verify_submit_nonce() ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: verify_submit_nonce failed' );
				}

				return;

			/*
			 * WpssoPost->post_can_have_meta() returns false:
			 *
			 *	If the $post argument is not numeric or an instance of WP_Post.
			 *	If the post ID is empty.
			 *	If the post type is empty.
			 *	If the post status is empty.
			 *	If the post type is 'revision'.
			 *	If the post status is 'trash'.
			 */
			} elseif ( ! $this->post_can_have_meta( $post_id ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: post id ' . $post_id . ' invalid for metadata' );
				}

				return;

			/*
			 * Check user capability for the post id.
			 */
			} elseif ( ! $this->user_can_edit( $post_id ) ) {

				if ( $this->p->debug->enabled ) {

					$user_id = get_current_user_id();

					$this->p->debug->log( 'exiting early: user id ' . $user_id . ' cannot edit post id ' . $post_id );
				}

				return;
			}

			$this->md_cache_disable();	// Disable the local cache.

			$mod = $this->get_mod( $post_id );

			$md_opts = $this->get_submit_opts( $mod );	// Merge previous + submitted options and then sanitize.

			$this->md_cache_enable();	// Re-enable the local cache.

			if ( false === $md_opts ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: returned submit options is false' );
				}

				return;
			}

			$md_opts = apply_filters( 'wpsso_save_md_options', $md_opts, $mod );
			$md_opts = apply_filters( 'wpsso_save_' . $mod[ 'name' ] . '_options', $md_opts, $post_id, $mod );

			return self::update_meta( $post_id, WPSSO_META_NAME, $md_opts );
		}

		/*
		 * Use $rel = false to extend WpssoAbstractWpMeta->delete_options().
		 */
		public function delete_options( $post_id, $rel = false ) {

			return self::delete_meta( $post_id, WPSSO_META_NAME );
		}

		public function after_insert_post( $post_id, $post_obj, $update, $post_before ) {

			if ( null === $post_before ) {

				if ( false === $update ) {

					if ( isset( $post_obj->post_status ) && 'auto-draft' === $post_obj->post_status ) {

						if ( ! empty( $this->p->options[ 'plugin_add_to_' . $post_obj->post_type ] ) ) {

							$mod = $this->get_mod( $post_id );

							list(
								parent::$head_tags,	// Used by WpssoAbstractWpMeta->is_meta_page().
								parent::$head_info	// Used by WpssoAbstractWpMeta->check_head_info().
							) = $this->p->util->cache->refresh_mod_head_meta( $mod );
						}
					}
				}
			}
		}

		/*
		 * Get all publicly accessible post ids using WP_Query->query().
		 */
		public static function get_public_ids( array $posts_args = array() ) {

			$wpsso =& Wpsso::get_instance();

			/*
			 * The WPML integration module returns true so WPML can filter posts for the current language.
			 *
			 * See WpssoAbstractWpMeta->get_column_meta_query_og_type().
			 * See WpssoIntegLangWpml->__construct().
			 * See WpssoIntegLangQtranslateXt->__construct().
			 */
			$filter_name = 'wpsso_post_public_ids_suppress_filters';

			if ( $wpsso->debug->enabled ) {

				$wpsso->debug->log( 'applying filters "' . $filter_name . '"' );
			}

			$suppress_filters = apply_filters( $filter_name, true );

			$posts_args = array_merge( array(
				'has_password'     => false,
				'order'            => 'DESC',		// Newest first.
				'orderby'          => 'date',
				'post_status'      => 'publish',	// Only 'publish' (not 'auto-draft', 'draft', 'future', 'inherit', 'pending', 'private', or 'trash').
				'post_type'        => 'any',		// Return any post, page, or custom post type.
				'paged'            => false,
				'posts_per_page'   => -1,		// The number of posts to query for. -1 to request all posts.
				'no_found_rows'    => true,		// Skip counting total rows found - should be enabled when pagination is not needed.
				'suppress_filters' => $suppress_filters,
			), $posts_args, array( 'fields' => 'ids' ) );	// Return an array of post ids.

			/*
			 * See WpssoIntegLangQtranslateXt->filter_post_public_ids_posts_args().
			 */
			$filter_name = 'wpsso_post_public_ids_posts_args';

			if ( $wpsso->debug->enabled ) {

				$wpsso->debug->log( 'applying filters "' . $filter_name . '"' );
			}

			$posts_args = apply_filters( $filter_name, $posts_args );

			if ( $wpsso->debug->enabled ) {

				$wpsso->debug->log_arr( 'posts_args', $posts_args );
			}

			$mtime_start = microtime( $get_float = true );

			$public_ids = SucomUtilWP::get_posts( $posts_args );

			$mtime_total = microtime( $get_float = true ) - $mtime_start;

			if ( $wpsso->debug->enabled ) {

				$wpsso->debug->log( count( $public_ids ) . ' post IDs returned in ' . sprintf( '%0.3f secs', $mtime_total ) );
			}

			$filter_name = 'wpsso_post_public_ids';

			if ( $wpsso->debug->enabled ) {

				$wpsso->debug->log( 'applying filters "' . $filter_name . '"' );
			}

			return apply_filters( $filter_name, $public_ids, $posts_args );
		}

		/*
		 * Get the children for a post.
		 *
		 * Returns an array of post ids for a given $mod object.
		 *
		 * The 'posts_per_page' value should be set for an archive page before calling this method.
		 *
		 * See WpssoPost->clear_cache().
		 * See WpssoAbstractWpMeta->get_posts_mods().
		 */
		public function get_posts_ids( array $mod ) {

			if ( $this->p->debug->enabled ) {

				$this->p->debug->mark();
			}

			if ( empty( $mod[ 'is_post' ] ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: ' . $mod[ 'name' ] . ' ID ' . $mod[ 'id' ] .  ' is not a post' );
				}

				return array();
			}

			$posts_args = array_merge( array(
				'has_password' => false,
				'order'        => 'DESC',		// Newest first.
				'orderby'      => 'date',
				'post_status'  => 'publish',		// Only 'publish', not 'auto-draft', 'draft', 'future', 'inherit', 'pending', 'private', or 'trash'.
				'post_type'    => 'any',		// Return posts, pages, or any custom post type.
				'post_parent'  => $mod[ 'id' ],
				'child_of'     => $mod[ 'id' ],		// Only include direct children.
			), $mod[ 'posts_args' ], array( 'fields' => 'ids' ) );	// Return an array of post ids.

			if ( $this->p->debug->enabled ) {

				$this->p->debug->log( 'getting posts for direct children of ' . $mod[ 'name' ] . ' ID ' . $mod[ 'id' ] );

				$this->p->debug->log_arr( 'posts_args', $posts_args );
			}

			$mtime_start = microtime( $get_float = true );

			$posts_ids = SucomUtilWP::get_posts( $posts_args );	// Alternative to get_posts() that does not exclude sticky posts.

			$mtime_total = microtime( $get_float = true ) - $mtime_start;

			if ( $this->p->debug->enabled ) {

				$this->p->debug->log( count( $posts_ids ) . ' post IDs returned in ' . sprintf( '%0.3f secs', $mtime_total ) );
			}

			return $posts_ids;
		}

		/*
		 * This method is hooked to the 'manage_pages_columns' filter.
		 *
		 * See https://core.trac.wordpress.org/browser/tags/5.8.1/src/wp-admin/includes/class-wp-posts-list-table.php#L711.
		 */
		public function add_page_column_headings( $columns ) {

			add_filter( 'manage_edit-page_sortable_columns', array( $this, 'add_sortable_columns' ), 10, 1 );

			return $this->add_column_headings( $columns, $post_type = 'page' );
		}

		/*
		 * This method is hooked to the 'manage_posts_columns' filter.
		 *
		 * Some plugins have been known to call the 'manage_posts_columns' filter with only one argument, so include a
		 * default value for the second argument to avoid a PHP fatal error.
		 *
		 * See https://core.trac.wordpress.org/browser/tags/5.8.1/src/wp-admin/includes/class-wp-posts-list-table.php#L722.
		 */
		public function add_post_column_headings( $columns, $post_type = 'post' ) {

			add_filter( 'manage_edit-' . $post_type . '_sortable_columns', array( $this, 'add_sortable_columns' ), 10, 1 );

			return $this->add_column_headings( $columns, $post_type );
		}

		/*
		 * This method is hooked to the 'manage_media_columns' filter.
		 *
		 * See https://core.trac.wordpress.org/browser/tags/5.8.1/src/wp-admin/includes/class-wp-media-list-table.php#L366.
		 */
		public function add_media_column_headings( $columns ) {

			add_filter( 'manage_upload_sortable_columns', array( $this, 'add_sortable_columns' ), 10, 1 );

			return $this->add_column_headings( $columns, $post_type = 'attachment' );
		}

		/*
		 * Hooked to the 'manage_pages_custom_column' action.
		 * Hooked to the 'manage_posts_custom_column' action.
		 * Hooked to the 'manage_media_custom_column' action.
		 */
		public function show_column_content( $column_name, $post_id ) {

			echo $this->get_column_content( '', $column_name, $post_id );
		}

		public function get_update_meta_cache( $post_id ) {

			return SucomUtilWP::get_update_meta_cache( $post_id, $meta_type = 'post' );
		}

		/*
		 * Hooked into the current_screen action.
		 *
		 * Sets the parent::$head_tags and parent::$head_info class properties.
		 */
		public function load_meta_page( $screen = false ) {

			if ( $this->p->debug->enabled ) {

				$this->p->debug->mark();
			}

			/*
			 * All meta modules set this property, so use it to optimize code execution.
			 */
			if ( false !== parent::$head_tags || ! isset( $screen->id ) ) {

				return;
			}

			if ( $this->p->debug->enabled ) {

				$this->p->debug->log( 'screen id = ' . $screen->id );
			}

			switch ( $screen->id ) {

				case 'upload':
				case ( 0 === strpos( $screen->id, 'edit-' ) ? true : false ):	// Posts list table.

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'exiting early: not a recognized post page' );
					}

					return;
			}

			/*
			 * Get the post object for sanity checks.
			 */
			if ( $this->p->debug->enabled ) {

				$this->p->debug->log( 'calling get_post_object( $use_post = true )' );
			}

			$post_obj = SucomUtilWP::get_post_object( $use_post = true );
			$post_id  = empty( $post_obj->ID ) ? 0 : $post_obj->ID;

			if ( ! $post_obj instanceof WP_Post ) {

				if ( $this->p->debug->enabled ) {

					if ( is_object( $post_obj ) ) {

						$this->p->debug->log( 'exiting early: ' . get_class( $post_obj ) . ' object is not an instance of WP_Post' );

					} else $this->p->debug->log( 'exiting early: post object is not an object' );
				}

				return;
			}

			/*
			 * Define parent::$head_tags and signal to other 'current_screen' actions that this is a post page.
			 */
			parent::$head_tags = array();	// Used by WpssoAbstractWpMeta->is_meta_page().

			/*
			 * WpssoPost->post_can_have_meta() returns false:
			 *
			 *	If the $post argument is not numeric or an instance of WP_Post.
			 *	If the post ID is empty.
			 *	If the post type is empty.
			 *	If the post status is empty.
			 *	If the post type is 'revision'.
			 *	If the post status is 'trash'.
			 */
			if ( ! $this->post_can_have_meta( $post_obj ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: post id ' . $post_id . ' invalid for metadata' );
				}

				return;

			} elseif ( isset( $_REQUEST[ 'action' ] ) && 'trash' === $_REQUEST[ 'action' ] ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: post id ' . $post_id . ' is being trashed' );
				}

				return;

			} elseif ( isset( $_REQUEST[ 'action' ] ) && 'delete' === $_REQUEST[ 'action' ] ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: post id ' . $post_id . ' is being deleted' );
				}

				return;
			}

			$mod = $this->get_mod( $post_id );

			if ( $this->p->debug->enabled ) {

				$this->p->debug->log( 'post id = ' . $post_id );
				$this->p->debug->log( 'home url = ' . get_option( 'home' ) );
				$this->p->debug->log( 'locale current = ' . SucomUtilWP::get_locale() );
				$this->p->debug->log( 'locale default = ' . SucomUtilWP::get_locale( 'default' ) );
				$this->p->debug->log( 'locale mod = ' . SucomUtilWP::get_locale( $mod ) );
				$this->p->debug->log( SucomUtil::get_array_pretty( $mod ) );
			}

			if ( SucomUtilWP::doing_block_editor() && ( ! empty( $_REQUEST[ 'meta-box-loader' ] ) || ! empty( $_REQUEST[ 'meta_box' ] ) ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'skipping head: doing block editor for metabox' );
				}

			} elseif ( empty( $this->p->options[ 'plugin_add_to_' . $post_obj->post_type ] ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'skipping head: metabox not enabled for post type ' . $post_obj->post_type );
				}

			} else {

				/*
				 * Hooked by woocommerce module to load front-end libraries and start a session.
				 */
				do_action( 'wpsso_admin_post_head', $mod );

				list(
					parent::$head_tags,	// Used by WpssoAbstractWpMeta->is_meta_page().
					parent::$head_info	// Used by WpssoAbstractWpMeta->check_head_info().
				) = $this->p->util->cache->refresh_mod_head_meta( $mod );

				/*
				 * Check for missing open graph image and description values.
				 */
				if ( $mod[ 'id' ] && $mod[ 'is_public' ] && 'publish' === $mod[ 'post_status' ] ) {

					$this->check_head_info( $mod );

					/*
					 * Check duplicates only when the post is available publicly and we have a valid permalink.
					 */
					if ( current_user_can( 'manage_options' ) ) {

						$check_head = empty( $this->p->options[ 'plugin_check_head' ] ) ? false : true;

						if ( apply_filters( 'wpsso_check_post_head', $check_head, $post_id, $post_obj ) ) {

							if ( $this->p->debug->enabled ) {

								$this->p->debug->log( 'checking post head' );
							}

							$this->check_post_head( $post_id, $post_obj );
						}
					}
				}
			}

			$action_query = 'wpsso-action';

			if ( ! empty( $_GET[ $action_query ] ) ) {

				$action_name = SucomUtil::sanitize_hookname( $_GET[ $action_query ] );

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'found action query: ' . $action_name );
				}

				if ( empty( $_GET[ WPSSO_NONCE_NAME ] ) ) {	// WPSSO_NONCE_NAME is an md5() string

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'nonce token query field missing' );
					}

				} elseif ( ! wp_verify_nonce( $_GET[ WPSSO_NONCE_NAME ], WpssoAdmin::get_nonce_action() ) ) {

					$this->p->notice->err( sprintf( __( 'Nonce token validation failed for %1$s action "%2$s".', 'wpsso' ), 'post', $action_name ) );

				} else {

					$_SERVER[ 'REQUEST_URI' ] = remove_query_arg( array( $action_query, WPSSO_NONCE_NAME ) );

					switch ( $action_name ) {

						default:

							do_action( 'wpsso_load_meta_page_post_' . $action_name, $post_id, $post_obj );

							break;
					}
				}
			}
		}

		public function check_post_head( $post_id = true, $post_obj = false ) {

			if ( $this->p->debug->enabled ) {

				$this->p->debug->mark();
			}

			if ( empty( $post_id ) ) {

				$post_id = true;
			}

			if ( ! is_object( $post_obj ) ) {

				$post_obj = SucomUtilWP::get_post_object( $post_id );

				if ( empty( $post_obj ) ) {

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'exiting early: unable to get the post object');
					}

					return;	// Stop here.
				}
			}

			if ( ! is_numeric( $post_id ) ) {	// Just in case the post_id is true/false.

				if ( empty( $post_obj->ID ) ) {

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'exiting early: post id in post object is empty');
					}

					return;	// Stop here.
				}

				$post_id = $post_obj->ID;
			}

			static $do_once = array();

			if ( isset( $do_once[ $post_id ] ) ) return;	// Stop here.

			$do_once[ $post_id ] = true;

			/*
			 * Only check publicly available posts.
			 */
			if ( ! isset( $post_obj->post_status ) || 'publish' !== $post_obj->post_status ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: post_status "' . $post_obj->post_status . '" is not publish' );
				}

				return;	// Stop here.
			}

			if ( empty( $post_obj->post_type ) || SucomUtilWP::is_post_type_public( $post_obj->post_type ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: post_type "' . $post_obj->post_type . '" not public' );
				}

				return;	// Stop here.
			}

			$exec_count = $this->p->debug->enabled ? 0 : (int) get_option( WPSSO_POST_CHECK_COUNT_NAME, $default = 0 );
			$max_count  = SucomUtil::get_const( 'WPSSO_DUPE_CHECK_HEADER_COUNT', 3 );

			if ( $exec_count >= $max_count ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: exec_count of ' . $exec_count . ' exceeds max_count of ' . $max_count );
				}

				return;	// Stop here.
			}

			if ( ini_get( 'open_basedir' ) ) {

				$check_url = $this->p->util->get_canonical_url( $post_id );

			} else $check_url = $this->p->util->get_shortlink( $post_id, $context = 'post' );

			$check_url_htmlenc = SucomUtil::encode_html_emoji( urldecode( $check_url ) );	// Does not double-encode.

			if ( empty( $check_url ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: invalid shortlink' );
				}

				return;	// Stop here.
			}

			/*
			 * Fetch the post HTML.
			 */
			$is_admin = is_admin();	// Call the function only once.

			if ( $this->p->debug->enabled ) {

				$this->p->debug->log( 'getting html for ' . $check_url );
			}

			if ( $is_admin ) {

				$this->p->notice->inf( sprintf( __( 'Checking %1$s for duplicate meta tags...', 'wpsso' ),
					'<a href="' . $check_url . '">' . $check_url_htmlenc . '</a>' ) );
			}

			/*
			 * Use the Facebook user agent to get Open Graph meta tags.
			 */
			$curl_opts = array(
				'CURLOPT_USERAGENT' => WPSSO_PHP_CURL_USERAGENT_FACEBOOK,
			);

			$this->p->cache->clear( $check_url );	// Clear the cached webpage.

			$exp_secs     = $this->p->debug->enabled ? false : null;
			$webpage_html = $this->p->cache->get( $check_url, $format = 'raw', $cache_type = 'transient', $exp_secs, $pre_ext = '', $curl_opts );
			$url_mtime    = $this->p->cache->get_url_mtime( $check_url );
			$html_size    = strlen( $webpage_html );
			$error_size   = (int) SucomUtil::get_const( 'WPSSO_DUPE_CHECK_ERROR_SIZE', 2500000 );
			$warning_time = (int) SucomUtil::get_const( 'WPSSO_DUPE_CHECK_WARNING_TIME', 2.5 );
			$timeout_time = (int) SucomUtil::get_const( 'WPSSO_DUPE_CHECK_TIMEOUT_TIME', 3.0 );

			if ( $html_size > $error_size ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'size of ' . $check_url . ' is ' . $html_size . ' bytes' );
				}

				/*
				 * If debug is enabled, the webpage may be larger than normal, so skip this warning.
				 */
				if ( $is_admin && ! $this->p->debug->enabled ) {

					$notice_msg = sprintf( __( 'The webpage HTML retrieved from %1$s is %2$s bytes.', 'wpsso' ),
						'<a href="' . $check_url . '">' . $check_url_htmlenc . '</a>', $html_size ) . ' ';

					$notice_msg .= sprintf( __( 'This exceeds the maximum limit of %1$s bytes imposed by the Google crawler.', 'wpsso' ),
						$error_size ) . ' ';

					$notice_msg .= __( 'If you do not reduce the webpage HTML size, Google will refuse to crawl this webpage.', 'wpsso' );

					$this->p->notice->err( $notice_msg );
				}
			}

			if ( true === $url_mtime ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'fetched ' . $check_url . ' from transient cache' );
				}

			} elseif ( false === $url_mtime ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'fetched ' . $check_url . ' returned a failure' );
				}

			} else {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'fetched ' . $check_url . ' in ' . $url_mtime . ' secs' );
				}

				if ( $is_admin && $url_mtime > $warning_time ) {

					$this->p->notice->warn(
						sprintf( __( 'Retrieving the webpage HTML for %1$s took %2$s seconds.', 'wpsso' ),
							'<a href="' . $check_url . '">' . $check_url_htmlenc . '</a>', $url_mtime ) . ' ' .
						sprintf( __( 'This exceeds the recommended limit of %1$s seconds (crawlers often time-out after %2$s seconds).',
							'wpsso' ), $warning_time, $timeout_time ) . ' ' .
						__( 'Please consider improving the speed of your site.', 'wpsso' ) . ' ' .
						__( 'As an added benefit, a faster site will also improve ranking in search results.', 'wpsso' ) . ' ;-)'
					);
				}
			}

			if ( empty( $webpage_html ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: error retrieving content from ' . $check_url );
				}

				if ( $is_admin ) {

					$this->p->notice->err( sprintf( __( 'Error retrieving content from <a href="%1$s">%1$s</a>.', 'wpsso' ), $check_url ) );
				}

				return;	// Stop here.

			} elseif ( stripos( $webpage_html, '<html' ) === false ) {	// Webpage must have an <html> tag.

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: <html> tag not found in ' . $check_url );
				}

				if ( $is_admin ) {

					$this->p->notice->err( sprintf( __( 'An %1$s tag was not found in <a href="%2$s">%2$s</a>.', 'wpsso' ),
						'&lt;html&gt;', $check_url ) );
				}

				return;	// Stop here

			} elseif ( ! preg_match( '/<meta[ \n]/i', $webpage_html ) ) {	// Webpage must have one or more <meta/> tags.

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: No <meta/> HTML tags were found in ' . $check_url );
				}

				if ( $is_admin ) {

					$this->p->notice->err( sprintf( __( 'No %1$s HTML tags were found in <a href="%2$s">%2$s</a>.', 'wpsso' ),
						'&lt;meta/&gt;', $check_url ) );
				}

				return;	// Stop here.

			} elseif ( false === strpos( $webpage_html, WPSSO_DATA_ID . ' begin' ) ) {	// Webpage should include our own meta tags.

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: ' . WPSSO_DATA_ID . ' not found in ' . $check_url );
				}

				if ( $is_admin ) {

					$short_name = $this->p->cf[ 'plugin' ][ 'wpsso' ][ 'short' ];

					$notice_msg = sprintf( __( 'The %1$s meta tags and Schema markup section was not found in <a href="%2$s">%2$s</a>.',
						'wpsso' ), $short_name, $check_url ) . ' ';

					$notice_msg .= __( 'Does a caching plugin or service need to be refreshed?', 'wpsso' );

					$this->p->notice->err( $notice_msg );
				}

				return;	// Stop here.
			}

			/*
			 * Remove the WPSSO meta tag and Schema markup section from the webpage to check for duplicate meta tags and markup.
			 */
			if ( $this->p->debug->enabled ) {

				$this->p->debug->log( 'removing the wpsso meta tag section from the webpage html' );
			}

			$mt_mark_preg = $this->p->head->get_mt_data( 'preg' );

			$count = null;

			$html_stripped = preg_replace( $mt_mark_preg, '', $webpage_html, $limit = -1, $count );

			if ( ! $count ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: preg_replace() function failed to remove the meta tag section' );
				}

				if ( $is_admin ) {

					$short_name = $this->p->cf[ 'plugin' ][ 'wpsso' ][ 'short' ];

					$notice_msg = sprintf( __( 'The PHP %1$s function failed to remove the %2$s meta tag section.', 'wpsso' ), '<code>preg_replace()</code>', $short_name ) . ' ';

					$notice_msg .= __( 'This could indicate a problem with PHP\'s PCRE library, or an optimization plugin / service corrupting the webpage HTML.', 'wpsso' ) . ' ';

					$notice_msg .= __( 'You should consider updating or having your hosting provider update your PHP installation and its PCRE library.', 'wpsso' );

					$this->p->notice->err( $notice_msg );
				}

				return;	// Stop here.
			}

			/*
			 * Check the stripped webpage HTML for duplicate html tags.
			 */
			if ( $this->p->debug->enabled ) {

				$this->p->debug->log( 'checking the stripped webpage html for duplicates' );
			}

			$metas = $this->p->util->get_html_head_meta( $html_stripped, $query = '/html/head/link|/html/head/meta', $libxml_errors = true );

			$check_opts = SucomUtil::preg_grep_keys( '/^add_/', $this->p->options, $invert = false, $replace = '' );

			$conflicts_msg = __( 'Conflict detected - your theme or another plugin is adding %1$s to the head section of this webpage.', 'wpsso' );

			$conflicts_found = 0;

			if ( is_array( $metas ) ) {

				if ( empty( $metas ) ) {	// No link or meta tags found.

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'error parsing head meta for ' . $check_url );
					}

					if ( $is_admin ) {

						$validator_url = 'https://validator.w3.org/nu/?doc=' . urlencode( $check_url );

						$settings_page_url = $this->p->util->get_admin_url( 'general#sucom-tabset_social_search-tab_pinterest' );

						$notice_msg = sprintf( __( 'An error occured parsing the head meta tags from <a href="%1$s">%1$s</a> (no "link" or "meta" HTML tags were found).', 'wpsso' ), $check_url ) . ' ';

						$notice_msg .= __( 'The webpage may contain HTML syntax errors preventing PHP from successfully parsing the HTML document.',
							'wpsso' ) . ' ';

						$notice_msg .= sprintf( __( 'Please review the <a href="%1$s">W3C Markup Validator</a> results and correct any syntax errors.',
							'wpsso' ), $validator_url ) . ' ';

						$notice_key = 'possible-html-syntax-errors-for-' . $check_url;

						$this->p->notice->err( $notice_msg, null, $notice_key );
					}

				} else {

					foreach( array(
						'link' => array( 'rel' ),
						'meta' => array( 'name', 'property', 'itemprop' ),
					) as $tag => $types ) {

						if ( isset( $metas[ $tag ] ) ) {

							foreach( $metas[ $tag ] as $meta ) {

								foreach( $types as $type ) {

									if ( isset( $meta[ $type ] ) && $meta[ $type ] !== 'generator' &&
										! empty( $check_opts[ $tag . '_' . $type . '_' . $meta[ $type ] ] ) ) {

										$conflicts_found++;

										$conflicts_tag = '<code>' . $tag . ' ' . $type . '="' . $meta[ $type ] . '"</code>';

										$this->p->notice->err( sprintf( $conflicts_msg, $conflicts_tag ) );
									}
								}
							}
						}
					}

					if ( $is_admin ) {

						$exec_count++;

						if ( $conflicts_found ) {

							$notice_key = 'duplicate-meta-tags-found';

							$notice_msg = sprintf( __( '%1$d duplicate meta tags found.', 'wpsso' ), $conflicts_found ) . ' ';

							$notice_msg .= sprintf( __( 'Check %1$d of %2$d failed (will retry)...', 'wpsso' ), $exec_count, $max_count );

							$this->p->notice->warn( $notice_msg, null, $notice_key );

						} else {

							$notice_key = 'no-duplicate-meta-tags-found';

							$notice_msg = __( 'Awesome! No duplicate meta tags found.', 'wpsso' ) . ' :-) ';

							if ( $this->p->debug->enabled ) {

								$notice_msg .= __( 'Debug option is enabled - will keep repeating duplicate check...', 'wpsso' );

							} else {

								$notice_msg .= sprintf( __( 'Check %1$d of %2$d successful...', 'wpsso' ), $exec_count, $max_count );
							}

							update_option( WPSSO_POST_CHECK_COUNT_NAME, $exec_count, $autoload = false );

							$this->p->notice->inf( $notice_msg, null, $notice_key );
						}
					}
				}
			}
		}

		/*
		 * Use $post_obj = false to extend WpssoAbstractWpMeta->add_meta_boxes().
		 *
		 * The Organization and Place post types do not have a Document SSO metabox as they have their own metabox.
		 *
		 * See WpssoOpmIntegAdminPost->add_meta_boxes().
		 */
		public function add_meta_boxes( $post_type, $post_obj = false ) {

			if ( $this->p->debug->enabled ) {

				$this->p->debug->mark();
			}

			$post_id = empty( $post_obj->ID ) ? 0 : $post_obj->ID;

			if ( empty( $this->p->options[ 'plugin_add_to_' . $post_type ] ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: cannot add metabox to post type "' . $post_type . '"' );
				}

				return;
			}

			$capability = 'page' === $post_type ? 'edit_page' : 'edit_post';

			if ( ! current_user_can( $capability, $post_id ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: cannot ' . $capability . ' for post id ' . $post_id );
				}

				return;
			}

			$metabox_id      = $this->p->cf[ 'meta' ][ 'id' ];
			$metabox_title   = _x( $this->p->cf[ 'meta' ][ 'title' ], 'metabox title', 'wpsso' );
			$metabox_screen  = $post_type;
			$metabox_context = 'normal';
			$metabox_prio    = 'default';
			$callback_args   = array(	// Second argument passed to the callback function / method.
				'metabox_id'                         => $metabox_id,
				'metabox_title'                      => $metabox_title,
				'__block_editor_compatible_meta_box' => true,
			);

			if ( $this->p->debug->enabled ) {

				$this->p->debug->log( 'adding metabox id wpsso_' . $metabox_id . ' for screen ' . $metabox_screen );
			}

			add_meta_box( 'wpsso_' . $metabox_id, $metabox_title, array( $this, 'show_metabox_' . $metabox_id ),
				$metabox_screen, $metabox_context, $metabox_prio, $callback_args );
		}

		public function ajax_get_validate_submenu() {

			$doing_ajax = SucomUtilWP::doing_ajax();

			if ( ! $doing_ajax ) {	// Just in case.

				return;
			}

			$post_obj = $this->die_or_get_ajax_post_obj();

			require_once ABSPATH . WPINC . '/class-wp-admin-bar.php';

			$admin_bar_class = apply_filters( 'wp_admin_bar_class', 'WP_Admin_Bar' );
			$ajax_admin_bar  = new $admin_bar_class;
			$parent_id       = $this->p->page->add_admin_bar_menu_validate( $ajax_admin_bar, $post_obj->ID );
			$metabox_html    = '';

			if ( empty( $parent_id ) ) {

				die( $metabox_html );
			}

			$nodes       = $ajax_admin_bar->get_nodes();
			$parent_node = $nodes[ $parent_id ];
			$menu_class  = 'ab-submenu' . ( empty( $parent_node->meta[ 'class' ] ) ? '' : $parent_node->meta[ 'class' ] );

			$metabox_html .= '<ul id="' . esc_attr( 'wp-admin-bar-' . $parent_node->id . '-default' ) . '"';
			$metabox_html .= $menu_class ? ' class="' . esc_attr( trim( $menu_class ) ) . '"' : '';
			$metabox_html .= '>';

			foreach ( $nodes as $key => $node ) {

				if ( $parent_id !== $node->parent ) {

					continue;
				}

				$has_link              = ! empty( $node->href );
				$is_parent             = ! empty( $node->children );
				$is_root_top_item      = 'root-default' === $node->parent;
				$is_top_secondary_item = 'top-secondary' === $node->parent;
				$tabindex              = isset( $node->meta[ 'tabindex' ] ) && is_numeric( $node->meta[ 'tabindex' ] ) ? (int) $node->meta[ 'tabindex' ] : '';
				$aria_attributes       = '' !== $tabindex ? ' tabindex="' . $tabindex . '"' : '';
				$menu_class            = empty( $node->meta[ 'class' ] ) ? '' : $node->meta[ 'class' ];
				$arrow                 = '';

				if ( ! $is_root_top_item && ! $is_top_secondary_item && $is_parent ) {

					$arrow = '<span class="wp-admin-bar-arrow" aria-hidden="true"></span>';
				}

				$link = $has_link ?
					'<a class="ab-item"' . $aria_attributes . ' href="' . esc_url( $node->href ) . '"' :
					'<div class="ab-item ab-empty-item"' . $aria_attributes;

				$attributes = array( 'onclick', 'target', 'title', 'rel', 'lang', 'dir' );

				foreach ( $attributes as $attribute ) {

					 if ( empty( $node->meta[ $attribute ] ) ) {

						 continue;
					 }

					$link .= ' ' . $attribute . '="';
					$link .= 'onclick' === $attribute ? esc_js( $node->meta[ $attribute ] ) : esc_attr( $node->meta[ $attribute ] );
					$link .= '"';
				}

				$link .= '>' . $arrow . $node->title;
				$link .= $has_link ? '</a>' : '</div>';

				$metabox_html .= '<li id="' . esc_attr( 'wp-admin-bar-' . $node->id ) . '"';
				$metabox_html .= $menu_class ? ' class="' . esc_attr( trim( $menu_class ) ) . '"' : '';
				$metabox_html .= '>' . $link;
				$metabox_html .= empty( $node->meta[ 'html' ] ) ? '' : $node->meta[ 'html' ];
				$metabox_html .=  '</li>' . "\n";
			}

			$metabox_html .= '</ul>';

			die( $metabox_html );
		}

		public function ajax_get_metabox_sso() {

			$doing_ajax = SucomUtilWP::doing_ajax();

			if ( ! $doing_ajax ) {	// Just in case.

				return;
			}

			$post_obj = $this->die_or_get_ajax_post_obj();

			if ( ! empty( $this->p->options[ 'plugin_add_to_' . $post_obj->post_type ] ) ) {

				$mod = $this->get_mod( $post_obj->ID );

				list(
					parent::$head_tags,	// Used by WpssoAbstractWpMeta->is_meta_page().
					parent::$head_info	// Used by WpssoAbstractWpMeta->check_head_info().
				) = $this->p->util->cache->refresh_mod_head_meta( $mod );

				/*
				 * Check for missing open graph image and description values.
				 */
				if ( $mod[ 'id' ] && $mod[ 'is_public' ] && 'publish' === $mod[ 'post_status' ] ) {

					$this->check_head_info( $mod );
				}
			}

			$metabox_html = $this->get_metabox_sso( $post_obj );

			die( $metabox_html );
		}

		private function die_or_get_ajax_post_obj() {

			$error_msg = '';

			if ( SucomUtil::get_const( 'DOING_AUTOSAVE' ) ) {

				die( -1 );

			} elseif ( ! check_ajax_referer( WPSSO_NONCE_NAME, '_ajax_nonce', $die = false ) ) {

				$error_msg = __( 'ajax request invalid nonce', 'wpsso' );
			}

			$post_id = isset( $_POST[ 'post_id' ] ) ? SucomUtil::sanitize_int( $_POST[ 'post_id' ] ) : 0;	// Returns integer or null.

			/*
			 * WpssoPost->post_can_have_meta() returns false:
			 *
			 *	If the $post argument is not numeric or an instance of WP_Post.
			 *	If the post ID is empty.
			 *	If the post type is empty.
			 *	If the post status is empty.
			 *	If the post type is 'revision'.
			 *	If the post status is 'trash'.
			 */
			if ( ! $this->post_can_have_meta( $post_id ) ) {

				$error_msg = sprintf( __( 'post id %s invalid for metadata', 'wpsso' ), $post_id );

			/*
			 * Check user capability for the post id.
			 */
			} elseif ( ! $this->user_can_edit( $post_id ) ) {

				$user_id = get_current_user_id();

				$error_msg = sprintf( __( 'user id %s cannot edit post id %s', 'wpsso' ), $user_id, $post_id );
			}

			if ( $error_msg ) {

				$stack     = debug_backtrace( DEBUG_BACKTRACE_IGNORE_ARGS );
				$from      = '';
				$class_seq = 1;
				$func_seq  = 1;

				if ( ! empty( $stack[ $class_seq ][ 'class' ] ) ) {

					$from .= $stack[ $class_seq ][ 'class' ] . '::';
				}

				if ( ! empty( $stack[ $func_seq ][ 'function' ] ) ) {

					$from .= $stack[ $func_seq ][ 'function' ];
				}

				$error_pre = trim( sprintf( __( '%s error:', 'wpsso' ), $from  ) );

				SucomUtil::safe_error_log( $error_pre . ' ' . $error_msg );

				die( -1 );
			}

			return SucomUtilWP::get_post_object( $post_id );
		}

		/*
		 * Use $rel = false to extend WpssoAbstractWpMeta->clear_cache().
		 *
		 * See WpssoIntegEcomWooCommerce->clear_product_cache().
		 */
		public function clear_cache( $post_id, $rel = false ) {

			if ( $this->p->debug->enabled ) {

				$this->p->debug->log_args( array(
					'post_id' => $post_id,
				) );
			}

			static $do_once = array();	// Just in case - prevent recursion.

			if ( isset( $do_once[ $post_id ] ) ) return;	// Stop here.

			$do_once[ $post_id ] = true;

			/*
			 * WpssoPost->post_can_have_meta() returns false:
			 *
			 *	If the $post argument is not numeric or an instance of WP_Post.
			 *	If the post ID is empty.
			 *	If the post type is empty.
			 *	If the post status is empty.
			 *	If the post type is 'revision'.
			 *	If the post status is 'trash'.
			 */
			if ( ! $this->post_can_have_meta( $post_id ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: post id ' . $post_id . ' invalid for metadata' );
				}

				return;
			}

			/*
			 * Clear the post meta, content, and head caches.
			 */
			$mod = $this->get_mod( $post_id );

			$this->clear_mod_cache( $mod );

			/*
			 * Clear the permalink, canonical / shortlink url cache.
			 */
			$permalink = get_permalink( $mod[ 'wp_obj' ] );

			$this->p->cache->clear( $permalink );

			if ( ini_get( 'open_basedir' ) ) {

				$other_url = $this->p->util->get_canonical_url( $mod );

			} else $other_url = $this->p->util->get_shortlink( $mod, $context = 'post' );

			if ( $permalink !== $other_url ) {

				$this->p->cache->clear( $other_url );
			}

			/*
			 * Clear post date, term, and author archive pages.
			 */
			$post_year  = get_the_time( 'Y', $mod[ 'wp_obj' ] );
			$post_month = get_the_time( 'm', $mod[ 'wp_obj' ] );
			$post_day   = get_the_time( 'd', $mod[ 'wp_obj' ] );

			$home_url       = home_url( '/' );
			$post_year_url  = get_year_link( $post_year );
			$post_month_url = get_month_link( $post_year, $post_month );
			$post_day_url   = get_day_link( $post_year, $post_month, $post_day );

			foreach ( array( $home_url, $post_year_url, $post_month_url, $post_day_url ) as $url ) {

				$this->p->head->clear_head_array( $archive_page_mod = false, $url );
			}

			/*
			 * Clear the post terms (categories, tags, etc.) for published (aka public) posts.
			 */
			if ( 'publish' === $mod[ 'post_status' ] ) {

				$post_taxonomies = get_post_taxonomies( $mod[ 'wp_obj' ] );

				if ( is_array( $post_taxonomies ) ) {	// Just in case.

					foreach ( $post_taxonomies as $tax_slug ) {

						$post_terms = wp_get_post_terms( $post_id, $tax_slug );	// Returns WP_Error if taxonomy does not exist.

						if ( is_array( $post_terms ) ) {	// Just in case.

							foreach ( $post_terms as $term_obj ) {

								$this->p->term->clear_cache( $term_obj->term_id, $tax_slug );
							}

							unset( $post_terms, $term_obj );
						}
					}

					unset( $post_taxonomies, $tax_slug );
				}
			}

			/*
			 * Clear the post author archive page.
			 */
			$this->p->user->clear_cache( $mod[ 'post_author' ] );

			/*
			 * The WPSSO FAQ add-on question shortcode attaches the post id to the question so the post cache can be
			 * cleared if/when a question is updated.
			 */
			$attached_ids = self::get_attached_ids( $post_id, 'post' );

			foreach ( $attached_ids as $attach_id => $bool ) {

				if ( $bool ) $this->clear_cache( $attach_id );
			}

			unset( $attached_ids, $attach_id, $bool );

			/*
			 * Clear the cache for any direct children as well.
			 */
			$children_ids = $this->get_posts_ids( $mod );

			foreach ( $children_ids as $child_id ) {

				$this->clear_cache( $child_id, $rel = false );
			}

			unset( $children_ids, $child_id );

			do_action( 'wpsso_clear_post_cache', $post_id, $mod );
		}

		/*
		 * Refresh the cache for a single post ID.
		 *
		 * This method will only execute once per post ID per page load.
		 *
		 * Use $rel = false to extend WpssoAbstractWpMeta->refresh_cache().
		 *
		 * See WpssoIntegEcomWooCommerce->refresh_product_cache().
		 */
		public function refresh_cache( $post_id, $rel = false ) {

			if ( $this->p->debug->enabled ) {

				$this->p->debug->log_args( array(
					'post_id' => $post_id,
				) );
			}

			static $do_once = array();	// Just in case - prevent recursion.

			if ( isset( $do_once[ $post_id ] ) ) return;	// Stop here.

			$do_once[ $post_id ] = true;

			/*
			 * WpssoPost->post_can_have_meta() returns false:
			 *
			 *	If the $post argument is not numeric or an instance of WP_Post.
			 *	If the post ID is empty.
			 *	If the post type is empty.
			 *	If the post status is empty.
			 *	If the post type is 'revision'.
			 *	If the post status is 'trash'.
			 */
			if ( ! $this->post_can_have_meta( $post_id ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: post id ' . $post_id . ' invalid for metadata' );
				}

				return;
			}

			$mod = $this->get_mod( $post_id );

			$this->p->util->cache->refresh_mod_head_meta( $mod );

			/*
			 * See WpssoCmcfActions->action_refresh_post_cache().
			 * See WpssoGmfActions->action_refresh_post_cache().
			 */
			do_action( 'wpsso_refresh_post_cache', $post_id, $mod );
		}

		/*
		 * Check user capability for the post id.
		 *
		 * Use $rel = false to extend WpssoAbstractWpMeta->user_can_edit().
		 */
		public function user_can_edit( $post_id, $rel = false ) {

			if ( ! $post_type = SucomUtil::get_request_value( 'post_type', 'POST' ) ) {	// Uses sanitize_text_field.

				$post_type = 'post';
			}

			$capability = 'page' === $post_type ? 'edit_page' : 'edit_post';

			if ( ! current_user_can( $capability, $post_id ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: cannot ' . $capability . ' for post id ' . $post_id );
				}

				/*
				 * Add notice only if the admin notices have not already been shown.
				 */
				if ( $this->p->notice->is_admin_pre_notices() ) {

					$this->p->notice->err( sprintf( __( 'Insufficient privileges to edit %1$s ID %2$s.', 'wpsso' ), $post_type, $post_id ) );
				}

				return false;
			}

			return true;
		}

		public function get_mt_reviews( $post_id, $rating_meta = 'rating', $worst_rating = 1, $best_rating = 5 ) {

			$reviews = array();

			if ( empty( $post_id ) ) {

				return $reviews;
			}

			$comments = get_comments( array(
				'post_id' => $post_id,
				'status'  => 'approve',
				'parent'  => 0,		// Parent ID of comment to retrieve children of (0 = don't get replies).
				'order'   => 'DESC',	// Newest first.
				'orderby' => 'date',
				'number'  => WPSSO_SCHEMA_REVIEWS_MAX,
			) );

			if ( is_array( $comments ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( count( $comments ) . ' comment objects' );
				}

				foreach( $comments as $num => $comment_obj ) {

					$comment_review = $this->get_mt_comment_review( $comment_obj, $rating_meta, $worst_rating, $best_rating );

					if ( ! empty( $comment_review ) ) {	// Just in case.

						$reviews[] = $comment_review;
					}
				}
			}

			if ( $this->p->debug->enabled ) {

				$this->p->debug->log_arr( 'reviews', $reviews );
			}

			return $reviews;
		}

		/*
		 * WpssoPost class specific methods.
		 *
		 * Filters the wp shortlink for a post - returns the shortened canonical URL.
		 *
		 * The wp_shortlink_wp_head() function calls wp_get_shortlink( 0, 'query' );
		 */
		public function get_canonical_shortlink( $shortlink = false, $post_id = 0, $context = 'post', $allow_slugs = true ) {

			if ( $this->p->debug->enabled ) {

				$this->p->debug->log_args( array(
					'shortlink'   => $shortlink,
					'post_id'     => $post_id,
					'context'     => $context,
					'allow_slugs' => $allow_slugs,
				) );
			}

			self::$saved_shortlink_url = null;	// Just in case.

			if ( isset( self::$cache_shortlinks[ $post_id ][ $context ][ $allow_slugs ] ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'returning shortlink from static cache = ' . self::$cache_shortlinks[ $post_id ][ $context ][ $allow_slugs ] );
				}

				return self::$saved_shortlink_url = self::$cache_shortlinks[ $post_id ][ $context ][ $allow_slugs ];
			}

			/*
			 * Check to make sure we have a plugin shortener selected.
			 */
			if ( empty( $this->p->options[ 'plugin_shortener' ] ) || $this->p->options[ 'plugin_shortener' ] === 'none' ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: no shortening service defined' );
				}

				return $shortlink;	// Return original shortlink.
			}

			/*
			 * The WordPress link-template.php functions call wp_get_shortlink() with a post id of 0.
			 *
			 * Use the same WordPress code to get a real post id and create a default shortlink (if required).
			 */
			if ( 0 === $post_id ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'provided post id is 0 (current post)' );
				}

				if ( 'query' === $context && is_singular() ) {	// wp_get_shortlink() uses the same logic.

					$post_id = get_queried_object_id();

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'setting post id ' . $post_id . ' from queried object' );
					}

				} elseif ( 'post' === $context ) {

					$post_obj = get_post();

					if ( empty( $post_obj->ID ) ) {

						if ( $this->p->debug->enabled ) {

							$this->p->debug->log( 'exiting early: post object ID is empty' );
						}

						return $shortlink;	// Return original shortlink.

					} else {

						$post_id = $post_obj->ID;

						if ( $this->p->debug->enabled ) {

							$this->p->debug->log( 'setting post id ' . $post_id . ' from post object' );
						}
					}
				}

				if ( empty( $post_id ) ) {

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'exiting early: unable to determine the post id' );
					}

					return $shortlink;	// Return original shortlink.
				}

				if ( empty( $shortlink ) ) {

					if ( 'page' === get_post_type( $post_id ) &&
						(int) $post_id === (int) get_option( 'page_on_front' ) &&
							'page' === get_option( 'show_on_front' ) ) {

						$shortlink = home_url( '/' );

					} else $shortlink = home_url( '?p=' . $post_id );
				}

			/*
			 * WpssoPost->post_can_have_meta() returns false:
			 *
			 *	If the $post argument is not numeric or an instance of WP_Post.
			 *	If the post ID is empty.
			 *	If the post type is empty.
			 *	If the post status is empty.
			 *	If the post type is 'revision'.
			 *	If the post status is 'trash'.
			 */
			} elseif ( ! $this->post_can_have_meta( $post_id ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: post id ' . $post_id . ' invalid for metadata' );
				}

				return $shortlink;	// Return original shortlink.
			}

			$mod = $this->get_mod( $post_id );

			if ( 'auto-draft' === $mod[ 'post_status' ] ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: post status is ' . $mod[ 'post_status' ] );
				}

				return $shortlink;	// Return original shortlink.
			}

			/*
			 * Shorten URL using the selected shortening service.
			 */
			$canonical_url = $this->p->util->get_canonical_url( $mod );
			$short_url     = $this->p->util->shorten_url( $canonical_url, $mod );

			if ( $short_url === $canonical_url ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: shortened URL same as canonical URL' );
				}

				return $shortlink;	// Return original shortlink.
			}

			if ( $this->p->debug->enabled ) {

				$this->p->debug->log( 'returning shortlink = ' . $short_url );
			}

			return self::$saved_shortlink_url = self::$cache_shortlinks[ $post_id ][ $context ][ $allow_slugs ] = $short_url;
		}

		public function maybe_restore_shortlink( $shortlink = false, $post_id = 0, $context = 'post', $allow_slugs = true ) {

			if ( self::$saved_shortlink_url === $shortlink ) {	// Shortlink value has not changed.

				self::$saved_shortlink_url = null;	// Just in case.

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: shortlink value has not changed' );
				}

				return $shortlink;
			}

			self::$saved_shortlink_url = null;	// Just in case.

			if ( isset( self::$cache_shortlinks[ $post_id ][ $context ][ $allow_slugs ] ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'restoring shortlink ' . $shortlink . ' to ' .
						self::$cache_shortlinks[ $post_id ][ $context ][ $allow_slugs ] );
				}

				return self::$cache_shortlinks[ $post_id ][ $context ][ $allow_slugs ];
			}

			return $shortlink;
		}

		/*
		 * Maybe inherit a featured image ID from the post/page parent, if the 'plugin_inherit_featured' option is enabled.
		 *
		 * See get_metadata_raw() in wordpress/wp-includes/meta.php.
		 * See metadata_exists() in wordpress/wp-includes/meta.php.
		 */
		public function get_post_metadata_thumbnail_id( $check, $post_id, $meta_key, $single ) {

			if ( '_thumbnail_id' !== $meta_key ) {	// Inherit only the featured image (aka '_thumbnail_id').

				return $check;	// Null by default.
			}

			if ( $this->p->debug->enabled ) {

				$this->p->debug->log( 'required call to WpssoPost->get_mod() for post ID ' . $post_id );
			}

			$mod = $this->get_mod( $post_id );	// Uses a local cache.

			if ( $mod[ 'is_attachment' ] ) {	// Attachments do not inherit metadata.

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: attachments do not inherit metadata' );
				}

				return $check;	// Null by default.
			}

			$inherit_featured = empty( $this->p->options[ 'plugin_inherit_featured' ] ) ? false : true;

			$inherit_featured = (bool) apply_filters( 'wpsso_inherit_featured_image', $inherit_featured, $mod );

			if ( $inherit_featured ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'inherit featured image is enabled' );
				}

			} else {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: inherit featured image is disabled' );
				}

				return $check;	// Null by default.
			}

			$metadata = $this->get_update_meta_cache( $post_id );

			/*
			 * If the meta key already has a value, then no need to check the parents.
			 */
			if ( ! empty( $metadata[ $meta_key ][ 0 ] ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: featured image = ' . $metadata[ $meta_key ][ 0 ] );
				}

				return $check;	// Null by default.
			}

			/*
			 * Make sure the post type is not false, empty string, or array.
			 */
			if ( empty( $mod[ 'post_type' ] ) || ! is_string( $mod[ 'post_type' ] ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: invalid post type' );
				}

				return $check;	// Null by default.
			}

			/*
			 * Start with the parent and work our way up - return the first value found.
			 */
			if ( $this->p->debug->enabled ) {

				$this->p->debug->log( 'getting ancestors for post type = ' . $mod[ 'post_type' ] );
			}

			$ancestor_ids = get_ancestors( $mod[ 'id' ], $mod[ 'post_type' ], $resource_type = 'post_type' );

			if ( empty( $ancestor_ids ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'no ancestors for ' . $mod[ 'name' ] . ' id ' . $mod[ 'id' ] );
				}

			} elseif ( is_array( $ancestor_ids ) ) {	// Just in case.

				foreach ( $ancestor_ids as $parent_id ) {

					/*
					 * Get the parent's metadata array.
					 */
					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'getting metadata for parent id ' . $parent_id );
					}

					$metadata = $this->get_update_meta_cache( $parent_id );

					if ( ! empty( $metadata[ $meta_key ][ 0 ] ) ) {	// Parent has a meta key value.

						if ( $this->p->debug->enabled ) {

							$this->p->debug->log( 'found parent ID ' . $parent_id . ' featured image = ' . $metadata[ $meta_key ][ 0 ] );
						}

						/*
						 * Return the parent's metadata single value or its array.
						 */
						if ( $single ) {

							return maybe_unserialize( $metadata[ $meta_key ][ 0 ] );
						}

						return array_map( 'maybe_unserialize', $metadata[ $meta_key ] );
					}
				}
			}

			return $check;	// Null by default.
		}

		/*
		 * Maybe inherit a featured image ID from the post/page parent, if the 'plugin_inherit_featured' option is enabled,
		 * and ignore saving the same featured image ID.
		 *
		 * See update_metadata() in wordpress/wp-includes/meta.php.
		 */
		public function update_post_metadata_thumbnail_id( $check, $post_id, $meta_key, $meta_value, $prev_value ) {

			if ( '_thumbnail_id' !== $meta_key ) {	// Inherit only the featured image (aka '_thumbnail_id').

				return $check;	// Null by default.
			}

			if ( $this->p->debug->enabled ) {

				$this->p->debug->log( 'required call to WpssoPost->get_mod() for post ID ' . $post_id );
			}

			$mod = $this->get_mod( $post_id );	// Uses a local cache.

			if ( $mod[ 'is_attachment' ] ) {	// Attachments do not inherit metadata.

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: attachments do not inherit metadata' );
				}

				return $check;	// Null by default.
			}

			$inherit_featured = empty( $this->p->options[ 'plugin_inherit_featured' ] ) ? false : true;

			$inherit_featured = (bool) apply_filters( 'wpsso_inherit_featured_image', $inherit_featured, $mod );

			if ( $inherit_featured ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'inherit featured image is enabled' );
				}

			} else {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: inherit featured image is disabled' );
				}

				return $check;	// Null by default.
			}

			if ( '' === $prev_value ) {	// No existing previous value.

				/*
				 * Make sure the post type is not false, empty string, or array.
				 */
				if ( empty( $mod[ 'post_type' ] ) || ! is_string( $mod[ 'post_type' ] ) ) {

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'exiting early: invalid post type' );
					}

					return $check;	// Null by default.
				}

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'getting ancestors for post type = ' . $mod[ 'post_type' ] );
				}

				$ancestor_ids = get_ancestors( $mod[ 'id' ], $mod[ 'post_type' ], $resource_type = 'post_type' );

				if ( empty( $ancestor_ids ) ) {

					if ( $this->p->debug->enabled ) {

						$this->p->debug->log( 'no ancestors for ' . $mod[ 'name' ] . ' id ' . $mod[ 'id' ] );
					}

				} elseif ( is_array( $ancestor_ids ) ) {	// Just in case.

					foreach ( $ancestor_ids as $parent_id ) {

						/*
						 * Get the parent's metadata array.
						 */
						if ( $this->p->debug->enabled ) {

							$this->p->debug->log( 'getting metadata for parent id ' . $parent_id );
						}

						$metadata = $this->get_update_meta_cache( $parent_id );

						if ( ! empty( $metadata[ $meta_key ][ 0 ] ) ) {	// Parent has a meta key value.

							$parent_value = maybe_unserialize( $metadata[ $meta_key ][ 0 ] );

							if ( $meta_value == $parent_value ) {	// Allow integer to numeric string comparison.

								return false;	// Do not save the meta key value.
							}
						}
					}
				}
			}

			if ( $this->p->debug->enabled ) {

				$this->p->debug->log_arr( 'check', $check );
			}

			return $check;	// Null by default.
		}

		/*
		 * Returns a custom or default term ID, or false if a term for the $tax_slug is not found.
		 */
		public function get_primary_term_id( array $mod, $tax_slug = 'category' ) {

			$primary_term_id = false;

			if ( $mod[ 'is_post' ] ) {	// Just in case.

				static $local_fifo = array();

				$post_id = $mod[ 'id' ];

				if ( isset( $local_fifo[ $post_id ][ $tax_slug ] ) ) {

					return $local_fifo[ $post_id ][ $tax_slug ];	// Return value from local cache.
				}

				/*
				 * Maybe limit the number of array elements.
				 */
				$local_fifo = SucomUtil::array_slice_fifo( $local_fifo, WPSSO_CACHE_ARRAY_FIFO_MAX );

				/*
				 * The 'wpsso_primary_tax_slug' filter is hooked by the WooCommerce integration module.
				 */
				$primary_tax_slug = apply_filters( 'wpsso_primary_tax_slug', $tax_slug, $mod );

				/*
				 * Returns null if a custom primary term ID has not been selected.
				 */
				$primary_term_id = $this->get_options( $post_id, $md_key = 'primary_term_id' );

				/*
				 * Make sure the term is not null or false, and still exists.
				 *
				 * Note that term_exists() requires an integer ID, not a string ID.
				 */
				if ( ! empty( $primary_term_id ) && term_exists( (int) $primary_term_id ) ) {

					$is_custom = true;

				} else {

					$is_custom = false;

					$primary_term_id = $this->get_default_term_id( $mod, $tax_slug );
				}

				$primary_term_id = apply_filters( 'wpsso_primary_term_id', $primary_term_id, $mod, $tax_slug, $is_custom );

				$local_fifo[ $post_id ][ $tax_slug ] = empty( $primary_term_id ) ? false : (int) $primary_term_id;
			}

			return $primary_term_id;
		}

		/*
		 * Returns the first taxonomy term ID, , or false if a term for the $tax_slug is not found.
		 */
		public function get_default_term_id( array $mod, $tax_slug = 'category' ) {

			$default_term_id = false;

			if ( $mod[ 'is_post' ] ) {	// Just in case.

				/*
				 * The 'wpsso_primary_tax_slug' filter is hooked by the WooCommerce integration module.
				 */
				$primary_tax_slug = apply_filters( 'wpsso_primary_tax_slug', $tax_slug, $mod );

				$post_terms = wp_get_post_terms( $mod[ 'id' ], $primary_tax_slug, $args = array( 'number' => 1 ) );

				if ( ! empty( $post_terms ) && is_array( $post_terms ) ) {	// Have one or more terms and taxonomy exists.

					foreach ( $post_terms as $term_obj ) {

						$default_term_id = (int) $term_obj->term_id;	// Use the first term ID found.

						break;
					}
				}

				$default_term_id = apply_filters( 'wpsso_default_term_id', $default_term_id, $mod, $tax_slug );
			}

			return $default_term_id;
		}

		/*
		 * Returns an associative array of term IDs and their names or objects.
		 *
		 * If the custom primary or default term ID exists in the post terms array, it will be moved to the top.
		 */
		public function get_primary_terms( array $mod, $tax_slug = 'category', $output = 'objects' ) {

			$primary_terms = array();

			if ( $mod[ 'is_post' ] ) {	// Just in case.

				$post_id = $mod[ 'id' ];

				/*
				 * Returns a custom or default term ID, or false if a term for the $tax_slug is not found.
				 */
				$primary_term_id = $this->p->post->get_primary_term_id( $mod, $tax_slug );	// Returns false or term ID.

				if ( $primary_term_id ) {

					/*
					 * The 'wpsso_primary_tax_slug' filter is hooked by the WooCommerce integration module.
					 */
					$primary_tax_slug = apply_filters( 'wpsso_primary_tax_slug', $tax_slug, $mod );

					$primary_term_obj = get_term_by( 'id', $primary_term_id, $primary_tax_slug, OBJECT, 'raw' );

					if ( $primary_term_obj ) {

						$post_terms = wp_get_post_terms( $post_id, $primary_tax_slug );

						if ( ! empty( $post_terms ) && is_array( $post_terms ) ) {	// Have one or more terms and taxonomy exists.

							/*
							 * If the primary or default term ID exists in the post terms array, move it to the top.
							 */
							foreach ( $post_terms as $num => $term_obj ) {

								if ( $primary_term_obj->term_id === $term_obj->term_id ) {

									unset( $post_terms[ $num ] );

									$post_terms = array_merge( array( $primary_term_obj ), $post_terms );

									break;	// No need to continue.
								}
							}

						} else {

							$post_terms = array( $primary_term_obj );
						}

						foreach ( $post_terms as $term_obj ) {

							switch ( $output ) {

								case 'ids':
								case 'term_ids':

									$primary_terms[ $term_obj->term_id ] = (int) $term_obj->term_id;

									break;

								case 'names':

									$primary_terms[ $term_obj->term_id ] = (string) $term_obj->name;

									break;

								case 'objects':

									$primary_terms[ $term_obj->term_id ] = $term_obj;

									break;
							}
						}
					}
				}
			}

			return apply_filters( 'wpsso_primary_terms', $primary_terms, $mod, $tax_slug, $output );
		}

		/*
		 * Since WPSSO Core v15.15.1.
		 *
		 * WpssoPost->post_can_have_meta() returns false:
		 *
		 *	If the $post argument is not numeric or an instance of WP_Post.
		 *	If the post ID is empty.
		 *	If the post type is empty.
		 *	If the post status is empty.
		 *	If the post type is 'revision'.
		 *	If the post status is 'trash'.
		 */
		public function post_can_have_meta( $post ) {

			if ( $post instanceof WP_Post ) {

				$post_id     = isset( $post->ID ) ? $post->ID : 0;
				$post_type   = isset( $post->post_type ) ? $post->post_type : '';
				$post_status = isset( $post->post_status ) ? $post->post_status : '';

			} elseif ( is_numeric( $post ) && $post > 0 ) {

				$post_id     = $post;
				$post_type   = get_post_type( $post_id );
				$post_status = get_post_status( $post_id );

			} else {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: post argument is not a valid number or an instance of WP_Post' );
				}

				return false;
			}

			if ( empty( $post_id ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: post id is empty' );
				}

				return false;

			} elseif ( empty( $post_type ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: post type is empty' );
				}

				return false;

			} elseif ( empty( $post_status ) ) {

				if ( $this->p->debug->enabled ) {

					$this->p->debug->log( 'exiting early: post status is empty' );
				}

				return false;
			}

			if ( $this->p->debug->enabled ) {

				$this->p->debug->log( 'post id ' . $post_id . ' post type is ' . $post_type );
				$this->p->debug->log( 'post id ' . $post_id . ' post status is ' . $post_status );
			}

			switch ( $post_type ) {

				case 'revision':

					return false;
			}

			switch ( $post_status ) {

				case 'trash':

					return false;
			}

			return true;
		}

		/*
		 * If $meta_key is en empty string, retrieves all metadata for the specified object ID.
		 *
		 * Use get_metadata() instead of get_post_meta() as the WordPress get_post_meta() function does not check
		 * wp_is_post_revision(), so it retrieves the revision metadata instead of the post metadata.
		 *
		 * See https://developer.wordpress.org/reference/functions/get_metadata/.
		 */
		public static function get_meta( $post_id, $meta_key = '', $single = false ) {

			return get_metadata( 'post', $post_id, $meta_key, $single );
		}

		public static function update_meta( $post_id, $meta_key, $value ) {

			return update_metadata( 'post', $post_id, $meta_key, $value );
		}

		public static function delete_meta( $post_id, $meta_key ) {

			return delete_metadata( 'post', $post_id, $meta_key );
		}
	}
}
