<?php

/**
 * Se encarga exclusivamente de las redirecciones de usuario (Login y Logout).
 *
 * Gestiona la lógica para redirigir a los usuarios a URLs personalizadas
 * después de iniciar sesión o cerrar sesión, con excepciones para administradores.
 *
 * @package    Wporlogin
 * @subpackage Wporlogin/public
 * @author     Oregoom <tu-email@oregoom.com>
 */
class Wporlogin_Public_Redirects {

    /**
     * El identificador único del plugin.
     *
     * @access private
     * @var    string    $plugin_name  El nombre (slug) del plugin.
     */
    private $plugin_name;

    /**
     * La versión actual del plugin.
     *
     * @access private
     * @var    string    $version      La versión actual del plugin.
     */
    private $version;

    /**
     * Inicializa la clase y asigna las propiedades principales.
     *
     * @param    string    $plugin_name    El nombre del plugin.
     * @param    string    $version        La versión del plugin.
     */
    public function __construct( $plugin_name, $version ) {
        $this->plugin_name = $plugin_name;
        $this->version     = $version;
    }

    /**
     * Redirecciona al usuario después de un inicio de sesión exitoso.
     * Hook: 'login_redirect' (Filter)
     *
     * @since    2.9.3
     * @param    string     $redirect_to    URL a la que WordPress planea redirigir por defecto.
     * @param    string     $request        URL desde la que vino el usuario.
     * @param    WP_User    $user           Objeto del usuario logueado.
     * @return   string                     URL final de redirección.
     */
    public function custom_login_redirect( $redirect_to, $request, $user ) {
        // 1. Si hubo error en el login o el usuario no es válido, devolvemos lo que había.
        if ( ! is_a( $user, 'WP_User' ) ) {
            return $redirect_to;
        }

        // 2. Obtener opciones de la base de datos
        $redirect_enabled      = get_option( 'wporlogin_enable_login_redirect' );
        $custom_redirect       = get_option( 'wporlogin_login_redirect' );
        $allow_admin_dashboard = get_option( 'wporlogin_enable_admin_redirect' ); // Checkbox "Permitir a admins ir al dashboard"

        // 3. Validar si la redirección está activa y si hay URL definida
        if ( $redirect_enabled != '1' || empty( $custom_redirect ) ) {
            return $redirect_to;
        }

        // 4. Lógica de Excepción para Administradores
        // Si el usuario es admin Y NO hemos forzado la redirección para admins, lo mandamos al escritorio.
        if ( isset( $user->roles ) && is_array( $user->roles ) && in_array( 'administrator', $user->roles ) ) {
            // Si la opción "Redirigir también a administradores" NO está marcada (valor '1'),
            // devolvemos admin_url() para que vayan al escritorio normalmente.
            if ( $allow_admin_dashboard != '1' ) {
                return admin_url();
            }
        }

        // 5. Retornar URL personalizada limpia
        return esc_url( $custom_redirect );
    }

    /**
     * Redirecciona al usuario al cerrar sesión.
     * Hook: 'wp_logout' (Action)
     *
     * @since    2.9.3
     */
    public function custom_logout_redirect() {
        // 1. Obtener opciones
        $redirect_enabled = get_option( 'wporlogin_enable_logout_redirect' );
        $custom_redirect  = get_option( 'wporlogin_logout_redirect' );

        // 2. Validar y Ejecutar
        if ( $redirect_enabled == '1' && ! empty( $custom_redirect ) ) {
            // Usamos wp_safe_redirect para seguridad (evita redirecciones a sitios maliciosos externos no permitidos)
            // Si quieres permitir dominios externos, usa wp_redirect() en su lugar, pero wp_safe_redirect es mejor práctica.
            wp_safe_redirect( esc_url( $custom_redirect ) );
            exit;
        }
    }
}