<?php

/**
 * Se encarga exclusivamente de la apariencia visual del login.
 *
 * Gestiona la inyección de CSS dinámico y la carga de hojas de estilo
 * para personalizar el formulario de acceso, registro y recuperación.
 *
 * @package    Wporlogin
 * @subpackage Wporlogin/public
 * @author     Oregoom <tu-email@oregoom.com>
 */
class Wporlogin_Public_Design {

    /**
     * El identificador único del plugin.
     *
     * @access private
     * @var    string    $plugin_name  El nombre (slug) del plugin.
     */
    private $plugin_name;

    /**
     * La versión actual del plugin.
     *
     * @access private
     * @var    string    $version      La versión actual del plugin.
     */
    private $version;

    /**
     * Inicializa la clase y asigna las propiedades principales.
     *
     * @param    string    $plugin_name    El nombre del plugin.
     * @param    string    $version        La versión del plugin.
     */
    public function __construct( $plugin_name, $version ) {
        $this->plugin_name = $plugin_name;
        $this->version     = $version;
    }

    /**
     * ==============================================================
     * 1. HOOKS DE MODIFICACIÓN (Filtros de WP)
     * ==============================================================
     */

    /**
     * Elimina el selector de idioma de la página de login.
     *
     * Se ejecuta solo si la opción 'remove_language_wporlogin' está activa.
     *
     * @since    2.7.0
     */
    public function remove_language_dropdown() {
        if ( get_option( 'remove_language_wporlogin' ) == 1 ) {
            add_filter( 'login_display_language_dropdown', '__return_false' );
        }
    }

    /**
     * Personaliza el título del enlace del logotipo (Tooltip).
     *
     * Cambia el texto "Funciona con WordPress" por el título del sitio o uno personalizado.
     *
     * @since    1.0.0
     * @param    string    $title    Título original del enlace.
     * @return   string              Título modificado.
     */
    public function custom_login_header_title( $title ) {
        $design = get_option( 'wporlogin_design', false );
        
        if ( $design && ( $design === 'wporlogin_design_standard' || $design === 'wporlogin_design_premium' ) ) {
            $custom_title = get_option( 'wporlogin_titulo_logotipo' );
            if ( $custom_title ) return esc_html( $custom_title );
        } else {
            if ( isset( $_GET['action'] ) ) {
                $action = sanitize_text_field( $_GET['action'] );
                if ( $action === 'register' )     return __( 'Sign Up', 'wporlogin' );
                if ( $action === 'lostpassword' ) return __( 'Reset Password', 'wporlogin' );
            }
            return __( 'Log In', 'wporlogin' );
        }
        return $title;
    }

    /**
     * Personaliza la URL del enlace del logotipo.
     *
     * Cambia el enlace de wordpress.org por la URL del sitio actual.
     *
     * @since    1.0.0
     * @param    string    $url    URL original del enlace.
     * @return   string            URL modificada.
     */
    public function custom_login_header_url( $url ) {
        $custom_url = get_option( 'wporlogin_ruta_url_logotipo' );
        if ( $custom_url ) return esc_url( $custom_url );

        if ( get_option( 'wporlogin_design' ) == 'wporlogin_design_basic' ) {
            return get_bloginfo( 'url' );
        }
        return $url; 
    }

    /**
     * ==============================================================
     * 2. GENERADORES DE CSS (Inyección de estilos Inline)
     * ==============================================================
     */

    /**
     * Inyecta variables CSS con los colores de marca configurados.
     *
     * @since    2.0.0
     */
    public function inject_brand_colors() {
        $color_main   = get_option( 'wporlogin_color_principal_marca' );
        $color_hover  = get_option( 'wporlogin_color_hover_marca' );
        $color_text   = get_option( 'wporlogin_color_principal_marca_text_submit' );
        ?>
        <style>
        :root {
            --wporlogin-color-principal-marca: <?php echo esc_attr( $color_main ); ?> !important;
            --wporlogin-color-hover-marca: <?php echo esc_attr( $color_hover ); ?> !important;
            --wporlogin-color-principal-marca-text-submit: <?php echo esc_attr( $color_text ); ?>;
        }
        </style>
        <?php 
    }

    /**
     * Inyecta estilos CSS para el logotipo personalizado.
     *
     * Configura el ancho, alto y posición del logo según las opciones del admin.
     *
     * @since    2.0.0
     */
    /*public function inject_logo_styles() {
        $design = get_option( 'wporlogin_design' );

        if ( $design === 'wporlogin_design_standard' || $design === 'wporlogin_design_premium' ) {

            $plugin_logo_url = get_option( "wporlogin_url_logotipo" );

            if ( $plugin_logo_url ) {
                $width_option  = get_option( 'wporlogin_width_logotipo_text' );
                $height_option = get_option( 'wporlogin_height_logotipo_text' );

                $width  = !empty($width_option) ? intval($width_option) : 84; 
                $height = !empty($height_option) ? intval($height_option) : 84;
                
                $pos_map  = [0=>'left top', 1=>'left center', 2=>'left bottom', 3=>'right top', 4=>'right center', 5=>'right bottom', 6=>'center top', 7=>'center center', 8=>'center bottom'];
                $size_map = [0=>'inherit', 1=>'cover', 2=>'contain'];

                $pos_idx  = get_option( 'wporlogin_background_position_logotipo_select', 0 );
                $size_idx = get_option( 'wporlogin_background_size_logotipo_select', 0 );

                $bg_pos  = isset( $pos_map[$pos_idx] ) ? $pos_map[$pos_idx] : 'left top';
                $bg_size = isset( $size_map[$size_idx] ) ? $size_map[$size_idx] : 'inherit';

                ?>
                <style>
                :root {
                    --wporlogin-logo: url('<?php echo esc_url( $plugin_logo_url ); ?>') !important;
                    --wporlogin-width: <?php echo $width; ?>px !important;
                    --wporlogin-height: <?php echo $height; ?>px !important;
                    --background-position: <?php echo $bg_pos; ?> !important;
                    --background-size: <?php echo $bg_size; ?> !important;
                }
                </style>
                <?php 
            } 
            else {
                $custom_logo_id = get_theme_mod( 'custom_logo' );
                $image = wp_get_attachment_image_src( $custom_logo_id, 'full' );    
                
                $url  = has_custom_logo() ? $image[0] : home_url( '/wp-admin/images/wordpress-logo.svg?ver=20131107' );
                $size = has_custom_logo() ? '200px' : '84px'; 

                echo "<style>:root{ --wporlogin-logo: url('" . esc_url( $url ) . "')!important; --wporlogin-width: $size; --wporlogin-height: $size; --background-size: $size !important; }</style>";
            }
        }
    }*/
    public function inject_logo_styles() {
        
        $design = get_option('wporlogin_design');

        // Solo aplicamos esto si el diseño lo soporta (Standard o Premium)
        if ( $design === 'wporlogin_design_standard' || $design === 'wporlogin_design_premium' ) {

            $plugin_logo_url = get_option("wporlogin_url_logotipo");
            $final_logo_url  = '';
            
            // Valores por defecto (se sobrescribirán con el cálculo inteligente)
            $width_css  = '84px';
            $height_css = '84px';

            if ( $plugin_logo_url ) {
                $final_logo_url = $plugin_logo_url;

                // --- LÓGICA INTELIGENTE DE AUTO-ADAPTACIÓN ---
                $attachment_id = attachment_url_to_postid($final_logo_url);
                $meta_data     = $attachment_id ? wp_get_attachment_metadata($attachment_id) : false;

                if ( $meta_data && isset($meta_data['width'], $meta_data['height']) ) {
                    $real_w = $meta_data['width'];
                    $real_h = $meta_data['height'];

                    if ( $real_w > $real_h ) {
                        // CASO A: Horizontal (Paisaje) -> Ancho 250px
                        $width_css  = '250px';
                        $ratio      = $real_h / $real_w;
                        $height_css = round($ratio * 250) . 'px';

                    } elseif ( $real_h > $real_w ) {
                        // CASO B: Vertical (Retrato) -> Alto 90px
                        $height_css = '90px';
                        $ratio      = $real_w / $real_h;
                        $width_css  = round($ratio * 90) . 'px';

                    } else {
                        // CASO C: Cuadrado (Lados iguales) -> Default WP (84px)
                        $width_css  = '84px';
                        $height_css = '84px';
                    }
                } else {
                    // FALLBACK: Si no podemos leer la imagen, usamos tu configuración manual
                    $w_opt = get_option('wporlogin_width_logotipo_text');
                    $h_opt = get_option('wporlogin_height_logotipo_text');
                    
                    $width_css  = !empty($w_opt) ? intval($w_opt) . 'px' : '250px';
                    $height_css = !empty($h_opt) ? intval($h_opt) . 'px' : 'auto';
                }

                // Recuperamos opciones de posición y tamaño manuales
                $pos_idx  = get_option( 'wporlogin_background_position_logotipo_select', 0 );
                $size_idx = get_option( 'wporlogin_background_size_logotipo_select', 2 ); // Default a 2 (contain)
                
                $pos_map  = [0=>'left top', 1=>'left center', 2=>'left bottom', 3=>'right top', 4=>'right center', 5=>'right bottom', 6=>'center top', 7=>'center center', 8=>'center bottom'];
                $size_map = [0=>'inherit', 1=>'cover', 2=>'contain'];

                $bg_pos  = isset($pos_map[$pos_idx]) ? $pos_map[$pos_idx] : 'center center';
                $bg_size = isset($size_map[$size_idx]) ? $size_map[$size_idx] : 'contain';

            } else {
                // LOGO DEL TEMA O WORDPRESS (Si el usuario no subió uno en el plugin)
                $custom_logo_id = get_theme_mod( 'custom_logo' );
                $image          = wp_get_attachment_image_src( $custom_logo_id, 'full' );
                
                if ( has_custom_logo() && $image ) {
                    $final_logo_url = $image[0];
                    // Si viene del tema, asumimos horizontal por seguridad, o dejamos 250px
                    $width_css  = '250px';
                    $height_css = 'auto'; 
                } else {
                    $final_logo_url = home_url( '/wp-admin/images/wordpress-logo.svg?ver=20131107' );
                    // Logo WP Default
                    $width_css  = '84px';
                    $height_css = '84px';
                }
                
                $bg_pos  = 'center center';
                $bg_size = 'contain';
            }

            // IMPRIMIMOS EL CSS DINÁMICO
            ?>
            <style>
            :root {
                --wporlogin-logo: url('<?php echo esc_url( $final_logo_url ); ?>') !important;
                --wporlogin-width: <?php echo $width_css; ?> !important;
                --wporlogin-height: <?php echo $height_css; ?> !important;
                --background-position: <?php echo $bg_pos; ?> !important;
                --background-size: contain !important; /* Siempre contain para evitar cortes */
            }
            
            /* Corrección para asegurar el centrado */
            body.login div#login h1 a {
                width: var(--wporlogin-width);
                height: var(--wporlogin-height);
                background-size: var(--background-size);
                background-position: var(--background-position);
                margin: 0 auto 25px auto;
                display: block;
            }
            </style>
            <?php 
        }
    }

    /**
     * Inyecta la imagen de fondo personalizada.
     *
     * @since    2.0.0
     */
    public function inject_background_url() {
        $design = get_option( 'wporlogin_design' );

        if ( $design === 'wporlogin_design_standard' || $design === 'wporlogin_design_premium' ) {
            $type = get_option( 'wporlogin_background_images' );
            $url  = false;

            if ( $type == 'wporlogin_free_images' ) {
                $url = get_option( 'wporlogin-background-free-image' );
            } elseif ( $type == 'wporlogin_my_images' ) {
                $url = get_option( 'wporlogin_url_img_fondo' );
            }

            if ( $url ) {
                // --- CORRECCIÓN VISUAL (Solo HTML/CSS) ---
                // Si la URL tiene la ruta vieja '/img/', la cambiamos a '/assets/img/' al vuelo.
                // Esto hace que la imagen cargue bien sin tocar la base de datos.
                if ( strpos( $url, '/plugins/wporlogin/img/' ) !== false ) {
                    $url = str_replace( '/plugins/wporlogin/img/', '/plugins/wporlogin/assets/img/', $url );
                }
                // -----------------------------------------

                echo "<style>:root { --wporlogin-img-fondo: url('" . esc_url( $url ) . "'); }</style>";
            }
        }
    }

    /**
     * ==============================================================
     * 3. ORQUESTADOR DE DISEÑO (Main Loader)
     * ==============================================================
     */

    /**
     * Aplica los estilos CSS correctos según el diseño seleccionado.
     *
     * Encola el archivo CSS (Básico, Estándar o Premium) correspondiente.
     *
     * @since    2.0.0
     */
    public function apply_design_styles() {
        $design = get_option( 'wporlogin_design' ); 
        
        $this->remove_language_dropdown();

        switch ( $design ) {
            case 'wporlogin_design_basic':
                wp_enqueue_style( 'wporlogin-basic', WPORLOGIN_URL . 'assets/css/wporlogin-style-design-basic.css', [], $this->version );
                break;

            case 'wporlogin_design_standard':
                wp_enqueue_style( 'wporlogin-standard', WPORLOGIN_URL . 'assets/css/wporlogin-style-design-standard.css', [], $this->version );
                break;

            case 'wporlogin_design_premium':
                $premium_design = get_option( 'wporlogin-design-img-premium' );
                $styles_map = [
                    'wporlogin_design_img_premium_one'   => 'wporlogin-style-design-premium-one.css',
                    'wporlogin_design_img_premium_two'   => 'wporlogin-style-design-premium-two.css',
                    'wporlogin_design_img_premium_three' => 'wporlogin-style-design-premium-three.css'
                ];

                if ( $premium_design && isset( $styles_map[$premium_design] ) ) {
                    wp_enqueue_style( 'wporlogin-premium', WPORLOGIN_URL . 'assets/css/' . $styles_map[$premium_design], [], $this->version );
                }
                break;
        }
    }
}