<?php

/**
 * La Clase Central del Plugin (The Core Class).
 *
 * Se encarga de cargar todas las dependencias, inicializar el orquestador de hooks 
 * y registrar los hooks (acciones y filtros) del plugin en WordPress.
 *
 * @package    Wporlogin
 * @subpackage Wporlogin/includes
 * @author     Oregoom <tu-email@oregoom.com>
 */
class Wporlogin {

    /**
     * El loader se encarga de coordinar todos los hooks del plugin.
     *
     * @access protected
     * @var    Wporlogin_Loader    $loader    Mantiene y registra los hooks.
     */
    protected $loader;

    /**
     * El identificador único del plugin.
     *
     * @access protected
     * @var    string    $plugin_name    El string usado para identificar este plugin.
     */
    protected $plugin_name;

    /**
     * La versión actual del plugin.
     *
     * @access protected
     * @var    string    $version    La versión actual del plugin.
     */
    protected $version;

    /**
     * Inicializa la clase y define las propiedades principales del plugin.
     *
     * @since    1.0.0
     */
    public function __construct() {
        if ( defined( 'WPORLOGIN_VERSION' ) ) {
            $this->version = WPORLOGIN_VERSION;
        } else {
            $this->version = '1.0.0';
        }
        $this->plugin_name = 'wporlogin';

        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();
    }

    /**
     * Carga todos los archivos dependientes del núcleo del plugin.
     *
     * @access private
     */
    private function load_dependencies() {

        // Clases del núcleo y utilidades
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-wporlogin-loader.php';
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-wporlogin-i18n.php';

        // Clases del área de administración
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-wporlogin-admin.php';
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-wporlogin-admin-notices.php'; // [NUEVO]

        // Clases de la parte pública (Frontend/Login)
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-wporlogin-public-design.php';
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-wporlogin-public-security.php';
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-wporlogin-public-redirects.php';
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-wporlogin-public-limit-login.php'; // [NUEVO]

        // [NUEVO] Módulo Hide Login
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-wporlogin-public-hide-login.php';

        $this->loader = new Wporlogin_Loader();
    }

    /**
     * Define la configuración regional para la internacionalización (i18n).
     *
     * @access private
     */
    private function set_locale() {
        $plugin_i18n = new Wporlogin_i18n();
        $this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );
    }

    /**
     * Registra todos los hooks (acciones y filtros) del área de administración.
     *
     * @access private
     */
    private function define_admin_hooks() {

        $plugin_admin   = new Wporlogin_Admin( $this->get_plugin_name(), $this->get_version() );
        $plugin_notices = new Wporlogin_Admin_Notices( $this->get_plugin_name(), $this->get_version() ); // [NUEVO]

        // --- GESTIÓN DE MENÚS Y AJUSTES (Admin Class) ---
        $this->loader->add_action( 'admin_menu', $plugin_admin, 'add_plugin_admin_menu' );
        $this->loader->add_action( 'admin_init', $plugin_admin, 'register_settings' );
        $this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_admin_assets' );

        // --- GESTIÓN DE AVISOS Y NOTIFICACIONES (Notices Class) ---
        
        // Alerta de configuración reCAPTCHA
        $this->loader->add_action( 'admin_notices', $plugin_notices, 'recaptcha_config_warning' );
        
        // Alerta de caché
        $this->loader->add_action( 'admin_notices', $plugin_notices, 'show_cache_warning' );

        // Aviso de Novedad (Hide Login)
        $this->loader->add_action( 'admin_notices', $plugin_notices, 'notice_new_feature_hide_login' );
        // AJAX para cerrar aviso de novedad
        $this->loader->add_action( 'wp_ajax_wporlogin_dismiss_notice_generic', $plugin_notices, 'ajax_dismiss_notice' );

        // Aviso de Reseña (5 Estrellas)
        $this->loader->add_action( 'admin_notices', $plugin_notices, 'show_review_notice' );
        // AJAX para cerrar aviso de reseña
        $this->loader->add_action( 'wp_ajax_delete-notice-wp', $plugin_notices, 'ajax_delete_notice' );
    }

    /**
     * Registra todos los hooks del área pública.
     *
     * @access private
     */
    private function define_public_hooks() {

        $public_design    = new Wporlogin_Public_Design( $this->get_plugin_name(), $this->get_version() );
        $public_security  = new Wporlogin_Public_Security( $this->get_plugin_name(), $this->get_version() );
        $public_redirects = new Wporlogin_Public_Redirects( $this->get_plugin_name(), $this->get_version() );
        $public_limit     = new Wporlogin_Public_Limit_Login( $this->get_plugin_name(), $this->get_version() ); // [NUEVO]

        // --- 1. REDIRECCIONES ---
        $this->loader->add_filter( 'login_redirect', $public_redirects, 'custom_login_redirect', 10, 3 );
        $this->loader->add_action( 'wp_logout', $public_redirects, 'custom_logout_redirect' );

        // --- 2. DISEÑO ---
        $this->loader->add_action( 'login_init', $public_design, 'remove_language_dropdown' );
        $this->loader->add_action( 'login_head', $public_design, 'inject_background_url' );
        $this->loader->add_action( 'login_head', $public_design, 'inject_brand_colors' );
        $this->loader->add_action( 'login_head', $public_design, 'inject_logo_styles' );
        $this->loader->add_action( 'login_enqueue_scripts', $public_design, 'apply_design_styles' );
        $this->loader->add_filter( 'login_headertext', $public_design, 'custom_login_header_title' ); 
        $this->loader->add_filter( 'login_headerurl', $public_design, 'custom_login_header_url' );

        // --- 3. SEGURIDAD (RECAPTCHA) ---
        $this->loader->add_action( 'login_enqueue_scripts', $public_security, 'enqueue_recaptcha_scripts' );
        $this->loader->add_action( 'login_form', $public_security, 'render_login_recaptcha' );
        $this->loader->add_filter( 'wp_authenticate_user', $public_security, 'verify_login_recaptcha', 10, 2 );
        $this->loader->add_action( 'register_form', $public_security, 'render_register_recaptcha' );
        $this->loader->add_filter( 'registration_errors', $public_security, 'verify_register_recaptcha', 10, 3 );
        $this->loader->add_action( 'lostpassword_form', $public_security, 'render_lostpassword_recaptcha' );
        $this->loader->add_filter( 'lostpassword_errors', $public_security, 'verify_lostpassword_recaptcha', 10, 1 );

        // --- 4. SEGURIDAD (LIMIT LOGIN ATTEMPTS) [NUEVO] ---
        // Verificar bloqueo antes de autenticar (Prioridad 1)
        $this->loader->add_filter( 'wp_authenticate_user', $public_limit, 'check_lockout', 1, 2 );
        // Registrar fallos
        $this->loader->add_action( 'wp_login_failed', $public_limit, 'handle_failed_login' );
        // Resetear contador al tener éxito
        $this->loader->add_action( 'wp_login', $public_limit, 'reset_counter_on_success', 10, 2 );
        // Mostrar aviso de intentos restantes
        $this->loader->add_filter( 'login_errors', $public_limit, 'show_remaining_attempts' );


        // -----------------------------------------------------------------
    // Lógica Hide Login (Ubicada en Public)
    // -----------------------------------------------------------------
    $plugin_hide_login = new Wporlogin_Public_Hide_Login( $this->get_plugin_name(), $this->get_version() );
    
    // Opción A: Si usas el método init() interno de la clase (recomendado para mantener limpio el orquestador)
    $plugin_hide_login->init();
    
    }

    /**
     * Ejecuta el loader para registrar todos los hooks con WordPress.
     * @since 1.0.0
     */
    public function run() {
        $this->loader->run();
    }

    /**
     * Devuelve el nombre del plugin.
     * @return string
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }

    /**
     * Devuelve el loader.
     * @return Wporlogin_Loader
     */
    public function get_loader() {
        return $this->loader;
    }

    /**
     * Devuelve la versión.
     * @return string
     */
    public function get_version() {
        return $this->version;
    }

    /**
     * Obtiene la lista de URLs de imágenes de fondo predeterminadas.
     * @static
     * @return array
     */
    public static function get_default_backgrounds() {
        return array(
            WPORLOGIN_URL . 'assets/img/wporlogin-img-fondo-0.jpg',
            WPORLOGIN_URL . 'assets/img/wporlogin-img-fondo-1.jpg',
            WPORLOGIN_URL . 'assets/img/wporlogin-img-fondo-2.jpg',
            WPORLOGIN_URL . 'assets/img/wporlogin-img-fondo-3.jpg',
            WPORLOGIN_URL . 'assets/img/wporlogin-img-fondo-4.jpg',
            WPORLOGIN_URL . 'assets/img/wporlogin-img-fondo-5.jpg',
            WPORLOGIN_URL . 'assets/img/wporlogin-img-fondo-6.jpg',
            WPORLOGIN_URL . 'assets/img/wporlogin-img-fondo-7.jpg',
            WPORLOGIN_URL . 'assets/img/wporlogin-img-fondo-8.jpg',
            WPORLOGIN_URL . 'assets/img/wporlogin-img-fondo-9.jpg',
            WPORLOGIN_URL . 'assets/img/wporlogin-img-fondo-10.jpg',
            WPORLOGIN_URL . 'assets/img/wporlogin-img-fondo-11.jpg',
            WPORLOGIN_URL . 'assets/img/wporlogin-img-fondo-12.jpg',
            WPORLOGIN_URL . 'assets/img/wporlogin-img-fondo-13.jpg'
        );
    }
}