<?php
/**
 * Security Class for WP Google Map Plugin
 * Handles sanitization and validation of shortcode attributes
 */

if ( ! class_exists( 'WPGMP_Security' ) ) {

    class WPGMP_Security {
    
        /**
         * Sanitize shortcode attributes
         *
         * @param array $atts Shortcode attributes
         * @return array Sanitized attributes
         */
        public static function wpgmp_sanitize_shortcode_atts( $atts ) {

            if ( ! is_array( $atts ) ) {
                return array();
            }
            
            $sanitized = array();
            
            foreach ( $atts as $key => $value ) {
                // Clean the attribute name - remove any non-alphanumeric except underscore and hyphen
                $key = preg_replace( '/[^a-zA-Z0-9_-]/', '', $key );
                
                if ( empty( $key ) ) {
                    continue; // Skip invalid keys
                }
                
                // Clean the value based on expected type
                switch ( $key ) {
                    case 'id':
                    case 'limit':
                    case 'perpage':
                    case 'zoom':
                        $sanitized[ $key ] = absint( $value );
                        break;
                        
                    case 'width':
                    case 'height':
                        $sanitized[ $key ] = self::wpgmp_sanitize_css_unit( $value );
                        break;
                        
                    case 'show_all_locations':
                    case 'hide_map':
                    case 'maps_only':
                        $sanitized[ $key ] = self::wpgmp_sanitize_boolean_like( $value );
                        break;
                        
                    case 'show':
                    case 'category':
                    case 'test': // For your example attribute
                    default:
                        $sanitized[ $key ] = self::wpgmp_sanitize_text_attribute( $value );
                        break;
                }
            }
            
            return $sanitized;
        }
        
        /**
         * Sanitize CSS units (width, height, etc.)
         *
         * @param string $value CSS value
         * @return string Sanitized CSS value
         */
        private static function wpgmp_sanitize_css_unit( $value ) {

            if ( empty( $value ) ) {
                return '';
            }
            
            // First, remove any JavaScript or malicious content
            $value = self::wpgmp_remove_malicious_content( $value );
            
            // Remove any characters that are not allowed in CSS values
            // Allowed: digits, decimal point, percentage, spaces, and common CSS units
            $value = preg_replace( '/[^a-zA-Z0-9%.\s-]/', '', $value );
            
            // Trim whitespace
            $value = trim( $value );
            
            // Validate CSS units pattern
            if ( preg_match( '/^(\d+(\.\d+)?)\s*(px|em|rem|%|vh|vw|vmin|vmax|cm|mm|in|pt|pc)?$/', $value ) ) {
                return $value;
            }
            
            // If it's just a number, assume pixels
            if ( is_numeric( $value ) ) {
                return $value . 'px';
            }
            
            return '';
        }
        
        /**
         * Sanitize boolean-like values
         *
         * @param string $value Boolean-like string
         * @return string Sanitized boolean string
         */
        private static function wpgmp_sanitize_boolean_like( $value ) {

            if ( empty( $value ) ) {
                return '';
            }
            
            $value = strtolower( trim( $value ) );
            $valid_values = array( 'true', 'false', '1', '0', 'yes', 'no', 'on', 'off' );
            
            return in_array( $value, $valid_values ) ? $value : '';
        }
        
        /**
         * Sanitize text attributes with strict filtering
         *
         * @param string $value Text value
         * @return string Sanitized text
         */
        private static function wpgmp_sanitize_text_attribute( $value ) {

            if ( empty( $value ) ) {
                return '';
            }
            
            // Remove malicious content
            $value = self::wpgmp_remove_malicious_content( $value );
            
            // Use WordPress core sanitization
            $value = sanitize_text_field( $value );
            
            return $value;
        }
        
        /**
         * Remove malicious content from strings
         *
         * @param string $value Input string
         * @return string Cleaned string
         */
        private static function wpgmp_remove_malicious_content( $value ) {

            if ( empty( $value ) ) {
                return '';
            }
            
            // Remove JavaScript event handlers (onclick, onfocus, etc.)
            $value = preg_replace( '/\s*on\w+\s*=\s*["\'][^"\']*["\']/i', '', $value );
            $value = preg_replace( '/\s*on\w+\s*=\s*[^"\'][^\s>]*/i', '', $value );
            
            // Remove JavaScript protocol
            $value = preg_replace( '/javascript\s*:/i', '', $value );
            
            // Remove data protocol
            $value = preg_replace( '/data\s*:/i', '', $value );
            
            // Remove VBScript
            $value = preg_replace( '/vbscript\s*:/i', '', $value );
            
            // Remove script tags and their content
            $value = preg_replace( '/<script\b[^>]*>(.*?)<\/script>/is', '', $value );
            
            // Remove iframe tags
            $value = preg_replace( '/<iframe\b[^>]*>(.*?)<\/iframe>/is', '', $value );
            
            // Remove object tags
            $value = preg_replace( '/<object\b[^>]*>(.*?)<\/object>/is', '', $value );
            
            // Remove embed tags
            $value = preg_replace( '/<embed\b[^>]*>/is', '', $value );
            
            // Remove CSS expressions
            $value = preg_replace( '/expression\s*\([^)]*\)/i', '', $value );
            
            // Remove CSS url() with javascript
            $value = preg_replace( '/url\s*\(\s*javascript\s*:/i', '', $value );
            
            return $value;
        }
        
        /**
         * Escape output for display
         *
         * @param string $output Output to escape
         * @return string Escaped output
         */
        public static function wpgmp_escape_output( $output ) {
            return wp_kses_post( $output );
        }
        
        /**
         * Validate map ID
         *
         * @param mixed $id Map ID
         * @return int Validated map ID or 0
         */
        public static function wpgmp_validate_map_id( $id ) {

            $id = absint( $id );
            return $id > 0 ? $id : 0;
        }
        
        /**
         * Sanitize array values recursively
         *
         * @param array $array Array to sanitize
         * @return array Sanitized array
         */
        public static function wpgmp_sanitize_array( $array ) {
            
            if ( ! is_array( $array ) ) {
                return self::wpgmp_sanitize_text_attribute( $array );
            }
            
            $sanitized = array();
            foreach ( $array as $key => $value ) {
                $clean_key = self::wpgmp_sanitize_text_attribute( $key );
                $sanitized[ $clean_key ] = is_array( $value ) ? self::wpgmp_sanitize_array( $value ) : self::wpgmp_sanitize_text_attribute( $value );
            }
            
            return $sanitized;
        }
    }

}

