<?php
/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://shapedplugin.com/
 * @since      2.0.0
 *
 * @package    WP_Tabs
 * @subpackage WP_Tabs/includes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      2.0.0
 * @package    WP_Tabs
 * @subpackage WP_Tabs/includes
 * @author     ShapedPlugin <help@shapedplugin.com>
 */
class SP_WP_Tabs_Free {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    2.0.0
	 * @access   protected
	 * @var      WP_Tabs_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * Stores the general tab settings loaded from the WordPress options table.
	 *
	 * @var array
	 */
	protected static $general_tab_settings = array();

	/**
	 * Stores the general tab settings loaded from the WordPress options table.
	 *
	 * @var array
	 */
	protected static $is_product_tabs_enabled = true;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    2.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    2.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    2.0.0
	 */
	public function __construct() {

		if ( defined( 'WP_TABS_VERSION' ) ) {

			$this->version = WP_TABS_VERSION;
		} else {

			$this->version = '3.0.0';
		}
		$this->plugin_name = 'wp-expand-tabs-free';

		self::$general_tab_settings    = get_option( 'sp-tab__settings' );
		self::$is_product_tabs_enabled = self::get_general_setting( 'enable_product_tabs', true );

		$this->load_dependencies();
		$this->define_admin_hooks();
		$this->define_public_hooks();
		$this->sptpro_wc_tab();
	}

	/**
	 * Check if WooCommerce is active.
	 *
	 * @since 2.2.13
	 *
	 * @return bool True if WooCommerce is active, false otherwise.
	 */
	public static function is_woocommerce_active() {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';

		return is_plugin_active( 'woocommerce/woocommerce.php' ) || is_plugin_active_for_network( 'woocommerce/woocommerce.php' );
	}

	/**
	 * Register WooCommerce hooks.
	 *
	 * @since 2.0.2
	 * @access private
	 */
	private function sptpro_wc_tab() {
		// Return if WooCommerce is not active.
		if ( ! self::is_woocommerce_active() ) {
			return;
		}

		$product_tab = new WP_Tabs_Product_Tab();

		// WooCommerce product tabs (Frontend).
		if ( self::$is_product_tabs_enabled ) {
			// Add the product tabs filter.
			$this->loader->add_filter( 'woocommerce_product_tabs', $product_tab, 'sptpro_woo_tab', 10, 2 );
		}

		$is_tab_group_enabled  = self::get_general_setting( 'sptpro_woo_tab', false );
		$is_version_compatible = version_compare( WP_TABS_FIRST_VERSION, '3.0.0', '<' );
		if ( $is_tab_group_enabled && $is_version_compatible ) {
			// Product tab.
			$this->loader->add_filter( 'woocommerce_product_tabs', $product_tab, 'sptpro_woo_tab_group', 10, 2 );
		}
	}

	/**
	 * Get General Tab Settings.
	 *
	 * @param string $key The setting key to retrieve.
	 * @param mixed  $option_default The default value to return if the key does not exist.
	 * @return mixed The value of the setting if it exists, otherwise the default value.
	 */
	public static function get_general_setting( $key, $option_default = null ) {
		return self::$general_tab_settings[ $key ] ?? $option_default;
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - WP_Tabs_Loader. Orchestrates the hooks of the plugin.
	 * - WP_Tabs_i18n. Defines internationalization functionality.
	 * - WP_Tabs_Admin. Defines all hooks for the admin area.
	 * - WP_Tabs_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    2.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		require_once plugin_dir_path( __DIR__ ) . 'includes/class-wp-tabs-updates.php';

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( __DIR__ ) . 'includes/class-wp-tabs-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		// require_once plugin_dir_path( __DIR__ ) . 'includes/class-wp-tabs-i18n.php';

		/**
		 * The class responsible for custom post type
		 * of the plugin.
		 */
		require_once plugin_dir_path( __DIR__ ) . 'includes/class-wp-tabs-cpt.php';

		/**
		 * The class responsible for admin sub-menu
		 * of the plugin.
		 */
		require_once plugin_dir_path( __DIR__ ) . 'includes/class-wp-tabs-admin-menu.php';

		/**
		 * The class responsible for Export and Import.
		 * of the plugin.
		 */
		require_once plugin_dir_path( __DIR__ ) . 'includes/class-wp-tabs-import-export.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( __DIR__ ) . 'admin/class-wp-tabs-admin.php';
		require_once plugin_dir_path( __DIR__ ) . 'admin/class-smart-tabs-free-admin.php';

		/**
		 * The class responsible for help page
		 */
		require_once WP_TABS_PATH . 'admin/help-page/help.php';

		/**
		 * The class responsible for shortcode
		 */
		require_once plugin_dir_path( __DIR__ ) . 'public/class-wp-tabs-shortcode.php';

		/**
		 * The class live preview.
		 */
		require_once WP_TABS_PATH . 'admin/preview/class-wp-tabs-preview.php';

		/**
		 * The class Elementor shortcode addons..
		 */
		if ( ( is_plugin_active( 'elementor/elementor.php' ) || is_plugin_active_for_network( 'elementor/elementor.php' ) ) ) {
			require_once WP_TABS_PATH . 'admin/ElementorAddons/class-wp-tabs-elementor-addons.php';
		}

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( __DIR__ ) . 'public/class-wp-tabs-public.php';
		require_once plugin_dir_path( __DIR__ ) . 'admin/partials/notices/review.php';
		require_once plugin_dir_path( __DIR__ ) . 'admin/partials/class-wp-tabs-widget.php';

		$this->loader = new WP_Tabs_Loader();

		require_once plugin_dir_path( __DIR__ ) . 'includes/class-wp-tabs-product-tab.php';
	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    2.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		// Plugin admin styles n scripts.
		$plugin_admin = new WP_Tabs_Admin( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'widgets_init', $plugin_admin, 'register_wptabs_widget' );
		$this->loader->add_action( 'admin_action_sp_duplicate_tabs', $plugin_admin, 'duplicate_wp_tabs' );
		$this->loader->add_filter( 'post_row_actions', $plugin_admin, 'sp_duplicate_tabs_link', 10, 2 );
		// Redirect after active.
		$this->loader->add_action( 'activated_plugin', $plugin_admin, 'sp_tabs_redirect_after_activation', 10, 2 );

		// Plugin admin custom post types.
		$plugin_admin_cpt = new WP_Tabs_CPT( $this->get_plugin_name(), $this->get_version() );
		$this->loader->add_action( 'init', $plugin_admin_cpt, 'sptpro_post_type' );

		/**
		 * If WooCommerce product tabs is enabled, then register the custom post type and the main class for product tabs.
		 */
		if ( self::$is_product_tabs_enabled && self::is_woocommerce_active() ) {
			$smart_tabs_admin = new Smart_Tabs_Free_Admin();
			// Register custom post type for product tabs.
			$this->loader->add_action( 'init', $plugin_admin_cpt, 'sp_product_tabs_post_type' );
			$this->loader->add_action( 'edit_form_top', $plugin_admin_cpt, 'sptpro_back_to_all_product_tabs' );
		}

		$this->loader->add_filter( 'post_updated_messages', $plugin_admin_cpt, 'sptpro_updated_messages', 10, 2 );
		$this->loader->add_filter( 'manage_sp_wp_tabs_posts_columns', $plugin_admin_cpt, 'sptpro_admin_column' );
		$this->loader->add_action( 'manage_sp_wp_tabs_posts_custom_column', $plugin_admin_cpt, 'sptpro_admin_field', 10, 2 );

		// Plugin admin menus.
		$plugin_admin_menu = new WP_Tabs_Admin_Menu( $this->get_plugin_name(), $this->get_version() );
		$this->loader->add_filter( 'plugin_action_links', $plugin_admin_menu, 'sptpro_plugin_action_links', 10, 2 );
		$this->loader->add_filter( 'admin_footer_text', $plugin_admin_menu, 'sptpro_review_text', 10, 2 );
		$this->loader->add_filter( 'update_footer', $plugin_admin_menu, 'sptpro_version_text', 11 );

		$plugin_review_notice = new WP_Tabs_Review( WP_TABS_NAME, WP_TABS_VERSION );
		$this->loader->add_action( 'admin_notices', $plugin_review_notice, 'display_admin_notice' );
		$this->loader->add_action( 'wp_ajax_sp-wptabs-never-show-review-notice', $plugin_review_notice, 'dismiss_review_notice' );

		// Export Import.
		$import_export = new Wp_Tabs_Import_Export( WP_TABS_NAME, WP_TABS_VERSION );
		$this->loader->add_action( 'wp_ajax_tabs_export_shortcode', $import_export, 'export_shortcode' );
		$this->loader->add_action( 'wp_ajax_tabs_import_shortcode', $import_export, 'import_shortcode' );

		/**
		 * Gutenberg block.
		 */
		if ( version_compare( $GLOBALS['wp_version'], '5.3', '>=' ) ) {
			require_once WP_TABS_PATH . 'admin/class-wp-tabs-free-gutenberg-block.php';
			new WP_Tabs_Free_Gutenberg_Block();
		}

		// Check if WooCommerce product tabs is enabled or not.
		if ( self::$is_product_tabs_enabled ) {
			add_action( 'admin_notices', array( $this, 'wc_smart_tabs_admin_notice' ) );
			add_action( 'wp_ajax_dismiss_smart_tabs_wc_notice', array( $this, 'dismiss_smart_tabs_wc_notice' ) );
		}

		if ( ! defined( 'SHAPEDPLIUGIN_OFFER_BANNER_LOADED' ) ) {
			define( 'SHAPEDPLIUGIN_OFFER_BANNER_LOADED', true );
			/**
			 * The class responsible for generating admin offer banner.
			 */
			require_once plugin_dir_path( __DIR__ ) . 'admin/partials/notices/offer-banner.php';
		}
	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    2.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		// Plugin public enqueue.
		$plugin_public = new WP_Tabs_Public( $this->get_plugin_name(), $this->get_version() );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_public, 'admin_enqueue_styles' );
		$this->loader->add_action( 'wp_loaded', $plugin_public, 'register_all_scripts' );

		$this->loader->add_filter( 'sp_wp_tabs_content', $plugin_public, 'sp_wp_tabs_markdown_to_html' );

		// Add Shortcode.
		$plugin_shortcode = new WP_Tabs_Shortcode( $this->get_plugin_name(), $this->get_version() );
		$this->loader->add_action( 'sptpro_action_tag_for_shortcode', $plugin_shortcode, 'sptpro_shortcode_execute' );
		add_shortcode( 'wptabs', array( $plugin_shortcode, 'sptpro_shortcode_execute' ) );
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    2.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     2.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     2.0.0
	 * @return    WP_Tabs_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     2.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

	/**
	 * Show admin notice if WooCommerce is not activated.
	 *
	 * @since 3.2.0
	 */
	public function wc_smart_tabs_admin_notice() {
		// Only show for admins.
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Show only if WooCommerce is not active and notice not dismissed.
		if ( class_exists( 'WooCommerce' ) || get_option( 'smart_tabs_wc_notice_dismiss_status', false ) ) {
			return;
		}

		add_thickbox();

		$link = esc_url(
			add_query_arg(
				array(
					'tab'       => 'plugin-information',
					'plugin'    => 'woocommerce',
					'TB_iframe' => 'true',
					'width'     => '640',
					'height'    => '500',
				),
				admin_url( 'plugin-install.php' )
			)
		);

		$settings_page_url = admin_url( 'edit.php?post_type=sp_wp_tabs&page=tabs_settings#tab=1' );

		$message = sprintf(
			wp_kses(
				/* translators: %1$s and %2$s are HTML links */
				__(
					'You must install and activate the %1$sWooCommerce%2$s plugin to run the <strong>Product Tabs</strong> smoothly in Smart Tabs. Don’t need it? Just disable the %3$sProduct Tabs%4$s option to hide this notice.',
					'wp-tabs-pro'
				),
				array(
					'a'      => array(
						'href'   => array(),
						'class'  => array(),
						'target' => array(),
					),
					'strong' => array(),
				)
			),
			'<a class="thickbox open-plugin-details-modal" href="' . esc_url( $link ) . '"><strong>',
			'</strong></a>',
			'<a href="' . esc_url( $settings_page_url ) . '"><strong>',
			'</strong></a>'
		);

		printf(
			'<div class="notice notice-error is-dismissible wc-smart-tabs-notice"><p>%s</p></div>',
			$message // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Message content escaped via wp_kses above.
		);
		$notice_nonce = wp_create_nonce( 'smart_tabs_wc_notice_nonce' );
		?>
		<script>
		jQuery(document).on('click', '.wc-smart-tabs-notice .notice-dismiss', function() {
			jQuery.post(ajaxurl, { 
				action: 'dismiss_smart_tabs_wc_notice',
				_wpnonce: '<?php echo esc_js( $notice_nonce ); ?>'
			});
		});
		</script>
		<?php
	}

	/**
	 * AJAX callback to persistently dismiss the WooCommerce dependency notice.
	 *
	 * Triggered when an admin clicks the dismiss button on the Smart Tabs notice.
	 * Stores a user meta flag so the notice remains hidden for that user on future page loads.
	 *
	 * @since 3.2.1
	 *
	 * @return void
	 */
	public function dismiss_smart_tabs_wc_notice() {
		// Verify nonce.
		if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['_wpnonce'] ), 'smart_tabs_wc_notice_nonce' ) ) { // phpcs:ignore -- WordPress.Security.NonceVerification.Recommended
			wp_die( -1 );
		}

		update_option( 'smart_tabs_wc_notice_dismiss_status', 1 );
		wp_send_json_success();
	}
}
