<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://www.quanticedgesolutions.com
 * @since      5.0
 *
 * @package    WPCD_Category_Discount
 * @subpackage WPCD_Category_Discount/admin
 */
class WPCD_Category_Discount_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    5.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    5.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * The slug of admin menu
	 * 
	 * @since	5.0
	 * @access	private
	 * @var		string		$menu_slug		The slug of the admin menu
	 */
	private static $menu_slug = 'woo-product-category-discount';

	/**
	 * wpdb object
	 * 
	 * @since	5.0
	 * @access  private
	 * @var		object 		$wpdb 			Object of wpdb
	 */
	private $wpdb;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    5.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		global $wpdb;
		$this->plugin_name = $plugin_name;
		$this->version = $version;
		$this->wpdb = $wpdb;
	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    5.0
	 */
	public function enqueue_scripts() {
		if( isset( $_GET['page'], $_GET['id'] ) && $_GET['page'] == self::$menu_slug && ( $_GET['action'] == 'add' || $_GET['action'] == 'edit' ) ) {
			wp_enqueue_script('wpcd-translation-fix', plugin_dir_url( __FILE__ ) . 'js/woo-product-category-discount-admin-translation-fix.js', array(), $this->version, 'all' );
			wp_enqueue_script('wpcd-react-app', plugin_dir_url( __FILE__ ) . 'components/wizard/build/bundle.js', array(), $this->version, 'all' );
			
			$excluded_taxonomies = ['product_type', 'product_visibility', 'product_shipping_class'];

			$taxonomies = get_object_taxonomies('product', 'objects');

			$taxonomies = array_filter($taxonomies, function($taxonomy) use ($excluded_taxonomies) {
				return !in_array($taxonomy->name, $excluded_taxonomies);
			});

			$taxonomies = array_map(function($taxonomy) {
				return [
					'label' => $taxonomy->labels->singular_name,
					'value' => $taxonomy->name
				];
			}, $taxonomies);
			$taxonomy_list = array_values($taxonomies);
			
			wp_localize_script('wpcd-react-app', 'wpcd_data', [
				'api_url'     => esc_url_raw(rest_url('wpcd/v1/')),
				'nonce'   	  => wp_create_nonce('wp_rest'),
				'redirect_url'=> admin_url( "admin.php?page=" . self::$menu_slug),
				'discount_id' => sanitize_text_field( $_GET['id'] ),
				'taxonomies'  => $taxonomy_list,
				'i18n'        => [
					'discount_type' 				=> __('Discount Type','woo-product-category-discount'), 
					'schedule' 						=> __('Schedule','woo-product-category-discount'), 
					'confirmation'  				=> __('Confirmation','woo-product-category-discount'),
					'finish'      					=> __('Finish', 'woo-product-category-discount'),	
					'next'     						=> __('Next', 'woo-product-category-discount'),
					'back'							=> __('Back', 'woo-product-category-discount'),
					'all_products' 					=> __('All Products', 'woo-product-category-discount'),
					'by_taxonomy' 					=> __('Taxonomy', 'woo-product-category-discount'),
					'match_type'					=> __('Match Type', 'woo-product-category-discount'),
					'match_all'						=> __('Match All', 'woo-product-category-discount'),
					'match_any'						=> __('Match Any', 'woo-product-category-discount'),
					'taxonomy'						=> __('Taxonomy', 'woo-product-category-discount'),
					'operator'						=> __('Operator', 'woo-product-category-discount'),
					'equal_to'						=> __('Equal To', 'woo-product-category-discount'),
					'not_equal_to'					=> __('Not Equal To', 'woo-product-category-discount'),
					'in' 							=> __('In', 'woo-product-category-discount'),
					'not_in' 						=> __('Not In', 'woo-product-category-discount'),
					'terms'							=> __('Terms', 'woo-product-category-discount'),
					'remove'						=> __('Remove', 'woo-product-category-discount'),
					'add_condition' 				=> __('+ Add Condition', 'woo-product-category-discount'),
					'select_schedule' 			  	=> __('Select Schedule', 'woo-product-category-discount'),
					'estimated_affected_products'	=> __('Estimated Affected Products: ', 'woo-product-category-discount'),
					'schedule_discount' 		  	=> $_GET['id'] == 'new' ? __('Schedule Discount', 'woo-product-category-discount') : __('Update Discount', 'woo-product-category-discount'),
					'start_date'					=> __('Start Date', 'woo-product-category-discount'),
					'end_date'						=> __('End Date', 'woo-product-category-discount'),
					'schedule_notice'				=> __('Note: Discounts take effect at 00:00 on the start date and end at 23:59 on the end date, based on server time. If no schedule is set, the discount will be applied immediately.', 'woo-product-category-discount'),
					'status_label'					=> __('Status', 'woo-product-category-discount'),
					'status_active'					=> __('Active', 'woo-product-category-discount'),
					'status_inactive'				=> __('Inactive', 'woo-product-category-discount'),
					'failed_schedule_message'		=> __('Failed to schedule the discount. Please try again.', 'woo-product-category-discount'),
					'discount_name'					=> __('Discount Name', 'woo-product-category-discount'),
					'discount_name_required'		=> __('Discount name is required.', 'woo-product-category-discount'),
					'at_least_one_rule_required'	=> __('At least one taxonomy rule is required.', 'woo-product-category-discount'),
					'required'						=> __('Required', 'woo-product-category-discount'),
					'discount_amount_label'			=> __('Discount Amount', 'woo-product-category-discount'),
					'amount'						=> __('Amount', 'woo-product-category-discount'),
					'amount_type'					=> __('Type','woo-product-category-discount'),
					'percent'						=> __('Percentage (%)','woo-product-category-discount'),
					'flat'							=> __('Flat Amount', 'woo-product-category-discount'),
					'discount_amount_required'		=> __('Discount amount is required.', 'woo-product-category-discount'),
					'discount_amount_type_required' => __('Discount amount type is required.', 'woo-product-category-discount'),
					'something_went_wrong'			=> __('Something went wrong','woo-product-category-discount'),
					'by_cart'						=> __('Cart Value', 'woo-product-category-discount'),
					'cart_discount_type_label'		=> __('Cart Discount Type', 'woo-product-category-discount'),
					'cart_value_label'				=> __('Cart Value Based', 'woo-product-category-discount'),
					'cart_discount_conditions_label'=> __('Cart Discount Conditions', 'woo-product-category-discount'),
					'min_cart_value_label' 			=> __('Minimum Cart Value (Optional)', 'woo-product-category-discount'),
					'max_cart_value_label' 			=> __('Maximum Cart Value (Optional)', 'woo-product-category-discount'),
					'cart_applicability_with_other_disc_label' => __('Discount Applicable with Other Discounts', 'woo-product-category-discount'),
					'yes_label'						=> __('Yes', 'woo-product-category-discount'),
					'no_label'						=> __('No', 'woo-product-category-discount'),
					'free_product_label' 			=> __('Free Product Giveaway Settings', 'woo-product-category-discount'),
					'automatically_add_label'		=> __('Whether it should be automatically added?', 'woo-product-category-discount'),
					'automatically_add_yes_label'	=> __('Yes, add it automatically', 'woo-product-category-discount'),
					'automatically_add_no_label'	=> __('No, prompt user to add it', 'woo-product-category-discount'),
					'products_label'				=> __('Products', 'woo-product-category-discount'),
					'max_value_should_be_greater_than_min_value' => __('Max value should be greater than min value', 'woo-product-category-discount'),
					'by_quantity' 					=> __('Quantity', 'woo-product-category-discount'),
					'quantity_discount_conditions_label' => __('Quantity Discount Conditions', 'woo-product-category-discount'),
					'min_quantity_label' 			=> __('Minimum Quantity', 'woo-product-category-discount'),
					'max_quantity_label' 			=> __('Maximum Quantity (Optional)', 'woo-product-category-discount'),
					'min_quantity_required' 		=> __('Minimum quanity is required.', 'woo-product-category-discount'),
					'do_not_go_away' => __('Please stay on this page while your discount is being applied. Leaving now may cause the discount to not apply correctly.', 'woo-product-category-discount'),
				],
			]);
		}

		if( isset( $_GET['page'], $_GET['id'], $_GET['action'] ) && $_GET['page'] == self::$menu_slug && $_GET['action'] == 'view-progress' ) {
			wp_enqueue_script( 'wpcd-discount-progress', plugin_dir_url( __FILE__ ) . 'components/discount-progress/build/bundle.js', array(), $this->version, true );
			wp_localize_script( 'wpcd-discount-progress', 'wpcd_data', [
				'api_url'     		=> esc_url_raw(rest_url('wpcd/v1/')),
				'nonce'   	  		=> wp_create_nonce('wp_rest'),
				'discount_id' 		=> sanitize_text_field( $_GET['id'] ),
				'discount_list_url' => admin_url( 'admin.php?page=woo-product-category-discount' ),
				'i18n'		  		=> [
					'discount_progress_loading_message' => __('Discount progress is loading...', 'woo-product-category-discount'),
					'back_button_text'					=> __('Back to Discount List', 'woo-product-category-discount'),
					'alert_message'						=> __('You may safely leave this page. The discount is being applied or removed in the background and will continue to process automatically.','woo-product-category-discount'),
				]
			]);
		}

		if( isset( $_GET['page'] ) && $_GET['page'] == self::$menu_slug ) {
			wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/woo-product-category-discount-admin.js', array('jquery'), $this->version, true );
			wp_localize_script( $this->plugin_name, 'wpcd', [
				'message' => __('You cannot edit the discount while it is being processed. Please press OK to fetch the latest status.', 'woo-product-category-discount'),
				'ajax_url' => admin_url( 'admin-ajax.php' ),
				'update_discount_nonce' => wp_create_nonce('update_discount_nonce'),
				'fetch_discount_status_nonce' => wp_create_nonce('fetch_discount_status_nonce'),
				'error' => __('There was an error. Please try again.', 'woo-product-category-discount'),
				'loader_img' => '<img class="loader" src="' . plugin_dir_url( __FILE__ ) . 'assets/images/loader.gif" alt="processing" height="50">',
				'check_discount_nonce' => wp_create_nonce('check_discount_nonce'),
				'view_more' => __('View More', 'woo-product-category-discount'),
				'view_less' => __('View Less', 'woo-product-category-discount'),
				'wp_rest_nonce' => wp_create_nonce('wp_rest'),
				'api_url'     => esc_url_raw(rest_url('wpcd/v1/')),
				'active' 	  => __('(Active)', 'woo-product-category-discount'),
				'inactive'    => __('(Inactive)', 'woo-product-category-discount'),
				'terminate_discount_nonce' => wp_create_nonce('terminate_discount_nonce'),
				'processing_terminate' => __('Your discount progress is being terminated. Please keep this page open while we finish.', 'woo-product-category-discount'),
				'confirm_terminate' => __('Alert: Terminating this process will not reverse the applied or removed discount. To complete the update, please activate the discount if it was applied, or mark it inactive if it was removed. Please wait until the process is fully completed.'),
				'processing_message' => __('Your discount is being updated. Please keep this page open while we finish.', 'woo-product-category-discount'),
			]);

			wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/woo-product-category-discount-admin.css', array(), $this->version);
		}
	}

	/**
	 * Add the admin menu.
	 *
	 * @since    5.0
	 */
	public function admin_menu() {
		add_submenu_page(
			'quanticedge',
			__( 'Simple Discount Rules', 'woo-product-category-discount' ),
			__( 'Simple Discount Rules', 'woo-product-category-discount' ),
			'manage_options',
			self::$menu_slug,
			array( $this, 'render_category_discount' ),
			6
		);
	}

	/**
	 * Render the category discount add or list page.
	 *
	 * @since    5.0
	 */
	public function render_category_discount() {
		if( get_option('wpcd_tables_created') != 'yes' ) {
			activate_wpcd_category_discount();
		}

		if( isset( $_GET['id'] ) ) {
			if( isset( $_GET['action'] ) && $_GET['action'] == 'delete' ) {
				$this->delete_discount_action();
			} else if ( isset( $_GET['action'] ) && $_GET['action'] == 'view-progress' ) {
				$this->show_discount_progress();
		 	} else {
				$this->render_add_discount();
			}
		} else if ( isset( $_GET['action'] ) && $_GET['action'] == 'settings' ){
			$this->render_settings();
		} else {
			$this->render_discount_list();
		}
	}

	/**
	 * Deletes a discount and its associated taxonomy rules.
	 *
	 * This function is called when the user clicks on the "Delete" link.
	 *
	 * @since 5.0
	 */
	public function delete_discount_action() {
		$id = absint($_GET['id']);
		$nonce = $_REQUEST['_wpnonce'] ?? '';

		if (!wp_verify_nonce($nonce, 'wpcd_delete_discount_' . $id)) {
			wp_die(__('Security check failed', 'woo-product-category-discount'));
		}

		$discount_data = $this->get_scheduled_discount_data( $id );
		$this->delete_discount( $discount_data );
		wp_redirect( admin_url( 'admin.php?wpcd_deleted=1&page=' . self::$menu_slug ) );
		exit();
	}

	/**
	 * Renders the add discount page.
	 *
	 * This function is called when the user clicks on the "Add Discount" link.
	 *
	 * @since 5.0
	 */
	public function render_add_discount() {
		include_once plugin_dir_path( __FILE__ ) . 'partials/woo-product-category-discount-add-discount.php';
	}

	/**
	 * Renders the discount list page.
	 *
	 * This function is called when the user is on the plugin's main page.
	 *
	 * @since 5.0
	 */
	public function render_discount_list() {
		$list_table = new WPCD_Discount_List_Table();
		include_once plugin_dir_path( __FILE__ ) . 'partials/woo-product-category-discount-list-table.php';
	}

	/**
	 * Renders the settings page.
	 *
	 * This function is called when the user is on the plugin's settings page.
	 *
	 * @since 5.8
	 */
	public function render_settings(){
		if (!current_user_can('manage_options')) {
			wp_die( __('You are not allowed to edit this section.', 'woo-product-category-discount'));
		}
		include_once plugin_dir_path( __FILE__ ) . 'partials/woo-product-category-discount-settings.php';
	}

	/**
	 * Outputs any admin notices.
	 *
	 * This function is called on the admin_notices action hook.
	 *
	 * @since 5.0
	 */
	public function admin_notices(){
		include_once plugin_dir_path( __FILE__ ) . 'partials/woo-product-category-discount-notices.php';
	}

	/**
	 * Renders the discount progress bar.
	 *
	 * This function is called when the user is on the add discount page and a scheduled discount is being processed.
	 *
	 * @since 5.0
	 */
	public function show_discount_progress(){
		include_once plugin_dir_path( __FILE__ ) . 'partials/woo-product-category-discount-progress.php';
	}

	/**
	 * Register the REST API endpoints used by the plugin.
	 *
	 * This function is called when the plugin is loaded.
	 *
	 * @since 5.0
	 */
	public function register_rest_routes(){
		register_rest_route('wpcd/v1', '/taxonomy-terms', [
			'methods'             => 'GET',
			'callback'            => [$this, 'get_taxonomies_terms'],
			'permission_callback' => function () {
				return current_user_can('manage_woocommerce');
			},
			'args' => [
				'taxonomy' => [
					'required' => true,
					'sanitize_callback' => 'sanitize_key',
				],
				'search' => [
					'required' => false,
					'sanitize_callback' => 'sanitize_text_field',
				]
			]
		]);

		register_rest_route('wpcd/v1', '/estimate-affected-products', [
			'methods'             => 'POST',
			'callback'            => [$this, 'estimate_products'],
			'permission_callback' => function () {
				return current_user_can('manage_woocommerce');
			}
		]);

		register_rest_route('wpcd/v1', '/schedule-discount', [
			'methods'             => 'POST',
			'callback'            => [$this, 'schedule_discount'],
			'permission_callback' => function () {
				return current_user_can('manage_woocommerce');
			}
		]);

		register_rest_route('wpcd/v1', '/process-discount', [
			'methods'			 	=> 'POST',
			'callback' 				=> [$this, 'process_discount'],
			'permission_callback' => function () {
				return current_user_can('manage_woocommerce');
			}
		]);

		register_rest_route('wpcd/v1', '/remove-discount', [
			'methods'             => 'POST',
			'callback'            => [$this, 'remove_discount_process'],
			'permission_callback' => function () {
				return current_user_can('manage_woocommerce');
			}
		]);

		register_rest_route('wpcd/v1', '/get-scheduled-discount/(?P<id>[a-zA-Z0-9_-]+)', [
			'methods'				=> 'GET',
			'callback'				=> [$this, 'get_scheduled_discount'],
			'permission_callback' 	=> function () {
				return current_user_can('manage_woocommerce');
			}
		]);

		register_rest_route('wpcd/v1', '/discount-progress/(?P<id>[a-zA-Z0-9_-]+)', [
			'methods' 				=> 'GET',
			'callback' 				=> [$this, 'get_discount_progress'],
			'permission_callback' 	=> function () {
				return current_user_can('manage_woocommerce');
			},
		]);

		register_rest_route('wpcd/v1', '/products', [
			'methods'             => 'GET',
			'callback'            => [$this, 'get_products'],
			'permission_callback' => function () {
				return current_user_can('manage_woocommerce');
			},
			'args' => [
				'search' => [
					'required' => false,
					'sanitize_callback' => 'sanitize_text_field',
				]
			]
		]);
	}

	/**
	 * Returns a list of terms of a given taxonomy.
	 *
	 * @param WP_REST_Request $request {
	 *     @type string $taxonomy The taxonomy to retrieve terms from.
	 *     @type string $search   Optional search query.
	 * }
	 *
	 * @return WP_Error|array Array of term objects on success.
	 */
	public function get_taxonomies_terms( WP_REST_Request $request ) {
		$taxonomy = $request['taxonomy'];
		$search   = $request['search'] ?? '';

		if ( ! taxonomy_exists( $taxonomy ) ) {
			return new WP_Error( 'invalid_taxonomy', __( 'Invalid taxonomy', 'woo-product-category-discount' ), [ 'status' => 400 ] );
		}

		$terms = get_terms( [
			'taxonomy'   => $taxonomy,
			'hide_empty' => false,
			'search'     => $search,
			'number'     => 50,
		] );

		if ( is_wp_error( $terms ) ) {
			return $terms;
		}

		$response = array_map( function ( $term ) {
			return [
				'id'    => $term->term_id,
				'label' => $term->name . ' (' . $term->slug . ')',
				'value' => $term->term_id,
			];
		}, $terms );

		return rest_ensure_response( $response );
	}

	/**
	 * Estimate the number of products that will be discounted with the given discount options.
	 *
	 * @param WP_REST_Request $request {
	 *     @type string $discountType Discount type: 'fixed' or 'percentage'.
	 *     @type string $matchType Match type: 'all' or 'any'.
	 *     @type array $taxonomyRules Taxonomy rules.
	 * }
	 *
	 * @return WP_Error|array {
	 *     @type int $count The number of products that will be discounted.
	 * }
	 */
	public function estimate_products(WP_REST_Request $request){
		$params = $request->get_json_params();

		$discount_type = $params['discountType'] ?? '';
		if ( $discount_type === 'cart' ){
			return rest_ensure_response(['count' => __('Cart discount does not apply to fixed number of products.', 'woo-product-category-discount')]);
		}

		if( $discount_type === 'quantity' ){
			return rest_ensure_response(['count' => __('Quantity discount does not apply to fixed number of products.', 'woo-product-category-discount')]);
		}

		$match_type = $params['matchType'] ?? '';
		$taxonomy_rules = $params['taxonomyRules'];

		$query = $this->get_products_query($discount_type, $match_type, $taxonomy_rules);

		return rest_ensure_response(['count' => $query->found_posts]);
	}
	
	/**
	 * Schedules a discount with the given options.
	 *
	 * @param WP_REST_Request $request {
	 *     @type string $discount_id Discount ID.
	 *     @type string $discountName Discount name.
	 *     @type string $discountType Discount type: 'fixed', 'percentage', 'cart', 'free_product'.
	 *     @type string $matchType Match type: 'all', 'any'.
	 *     @type array $taxonomyRules Taxonomy rules.
	 *     @type string $startDate Start date.
	 *     @type string $endDate End date.
	 *     @type string $discountAmountType Discount amount type: 'fixed', 'percentage'.
	 *     @type string $discountAmount Discount amount.
	 *     @type int $status Status.
	 *     @type int $cart_discount_type Cart discount type.
	 *     @type float $min_cart_value Minimum cart value.
	 *     @type float $max_cart_value Maximum cart value.
	 *     @type int $discount_applicable_with_other_discount Discount applicable with other discount.
	 *     @type int $automatically_add_type Automatically add type.
	 *     @type array $cart_rule_products Cart rule products.
	 * }
	 *
	 * @return WP_Error|array Array of term objects on success.
	 */
	public function schedule_discount(WP_REST_Request $request) {
		$params = $request->get_json_params();

		$discount_id = $params['discount_id'] ?? '';
		if( $discount_id == 'new' ){
			$discount_id = null;
		}

		$discount_name = $params['discountName'] ?? '';
		$discount_type = $params['discountType'] ?? '';
		$ruleType = $params['matchType'] ?? '';
		$start_date = $params['startDate'] ?? '';
		$end_date = $params['endDate'] ?? '';
		$discount_amount_type = $params['discountAmountType'] ?? '';
		$discount_amount = $params['discountAmount'] ?? '';
		$status = $params['status'] ?? 0;
		$cart_discount_type = $request['cartDiscountType'] == 'free_product' ? 1 : 0;
		$min_cart_value = $request['minCartValue'] ?? null;
		$max_cart_value = $request['maxCartValue'] ?? null;
		$discount_applicable_with_other_discount = $request['discApplicableWithOtherDisc'] == 'yes' ? 1 : 0;
		$automatically_add_type = $request['automaticallyAddType'] == 'yes' ? 1 : 0;
		$cart_rule_products = $request['selectedProducts'] ?? [];

		if( empty($discount_name) ) {
			return new WP_Error('invalid_data', __('Discount name is required', 'woo-product-category-discount'), ['status' => 400]);
		}
		
		if (empty($discount_type) ) {
			return new WP_Error('invalid_data', __('Invalid discount type provided', 'woo-product-category-discount'), ['status' => 400]);
		}

		if( $status == 1 ){
			$start_ts = !empty($start_date) ? strtotime($start_date) : null;
			$end_ts   = !empty($end_date)   ? strtotime($end_date)   : null;
			$now      = time();

			if ($start_ts !== null && $start_ts < $now) {
				return new WP_Error(
					'invalid_dates',
					__('Start date cannot be in the past. Please choose a date in the future.', 'woo-product-category-discount'),
					['status' => 400]
				);
			}

			if ($end_ts !== null && $end_ts < $now) {
				return new WP_Error(
					'invalid_dates',
					__('End date cannot be in the past. Please choose a date in the future.', 'woo-product-category-discount'),
					['status' => 400]
				);
			}

			if ($start_ts !== null && $end_ts !== null && $start_ts > $end_ts) {
				return new WP_Error(
					'invalid_dates',
					__('Start date must be before end date', 'woo-product-category-discount'),
					['status' => 400]
				);
			}
		}

		if ($discount_type === 'taxonomy') {
			$rules = $params['taxonomyRules'] ?? [];
			if (empty($rules)) {
				return new WP_Error('invalid_data', __('No taxonomy rules provided', 'woo-product-category-discount'), ['status' => 400]);
			}
		} else {
			$rules = $discount_type === 'quantity' ? $params['taxonomyRules'] : [];
		}

		if($discount_type === 'cart' || $discount_type === 'quantity' ){
			if( !empty( $min_cart_value ) && !empty( $max_cart_value ) && ( $min_cart_value > $max_cart_value ) ){
				return new WP_Error('invalid_data', __('Minimum cart value must be less than maximum cart value', 'woo-product-category-discount'), ['status' => 400]);
			}

			if( $cart_discount_type === 'free_product' && empty( $cart_rule_products ) ){
				return new WP_Error('invalid_data', __('Please select at least one product', 'woo-product-category-discount'), ['status' => 400]);
			} 
		}

		if( !empty( $discount_id ) ){
			$discount_data = $this->get_scheduled_discount_data( $discount_id );
			if( $this->get_process_method() == 'cron' ){
				$this->maybe_inactive_discount( $discount_data );
			} else {
				$previous_data = $discount_data;
				$schdule_discount_data = $this->prepare_schedule_discount_data( $discount_data );
				if( !empty( $discount_data['start_date'] ) && strtotime( $discount_data['start_date'] . ' 00:00:00' ) > time() ){
					wp_unschedule_event( strtotime( $discount_data['start_date'] . ' 00:00:00' ), 'wpcd_apply_discount_setup', [$schdule_discount_data] );
				}

				if( !empty( $discount_data['end_date'] ) && strtotime( $discount_data['end_date'] . ' 23:59:59' ) > time() ){
					wp_unschedule_event( strtotime( $discount_data['end_date'] . ' 23:59:59' ), 'wpcd_remove_discount_setup', [$schdule_discount_data] );
				}
			}

			$this->wpdb->delete(
				$this->wpdb->prefix . 'wpcd_taxonomy_discount_terms',
				['discount_id'=> $discount_id],
				['%d']
			);

			$this->wpdb->delete(
				$this->wpdb->prefix . 'wpcd_cart_discount_rules',
				['discount_id'=> $discount_id],
				['%d']
			);

			$this->wpdb->delete(
				$this->wpdb->prefix . 'wpcd_cart_discount_rules_products',
				['discount_id'=> $discount_id],
				['%d']
			);

			$this->wpdb->update(
				$this->wpdb->prefix . 'wpcd_discounts',
				[
					'name'					=> sanitize_text_field($discount_name),
					'discount_type'			=> $discount_type == 'all_products' ? 0 : ($discount_type == 'taxonomy' ? 1 : ($discount_type == 'cart' ? 2 : 3)),
					'rule_type'    			=> $ruleType == 'any' ? 1 : 0,
					'start_date'   			=> !empty($start_date) ? sanitize_text_field($start_date) : null,
					'end_date'     			=> !empty($end_date) ? sanitize_text_field($end_date) : null,
					'discount_amount_type' 	=> $discount_amount_type == 'percentage' ? 0 : 1,
					'discount_amount' 		=> sanitize_text_field($discount_amount),
					'status'        		=> intval($status),
					'total_chunks'			=> 1,
					'processed_chunks'		=> 0,
					'updated_at'			=> date('Y-m-d H:i:s'),
					'user_id'				=> get_current_user_id()
				],
				[
					'id'					=> $discount_id
				],
				['%s', '%d', '%d', '%s', '%s', '%d', '%f', '%d', '%d', '%d', '%s', '%d'],
				['%d']
			);
		} else {
			$this->wpdb->insert(
				$this->wpdb->prefix . 'wpcd_discounts',
				[
					'name'					=> sanitize_text_field($discount_name),
					'discount_type'			=> $discount_type == 'all_products' ? 0 : ($discount_type == 'taxonomy' ? 1 : ($discount_type == 'cart' ? 2 : 3)),
					'rule_type'    			=> $ruleType == 'any' ? 1 : 0,
					'start_date'   			=> !empty($start_date) ? sanitize_text_field($start_date) : null,
					'end_date'     			=> !empty($end_date) ? sanitize_text_field($end_date) : null,
					'discount_amount_type' 	=> $discount_amount_type == 'percentage' ? 0 : 1,
					'discount_amount' 		=> sanitize_text_field($discount_amount),
					'status'        		=> intval($status),
					'total_chunks'			=> 1,
					'processed_chunks' 		=> 0,
					'updated_at'			=> date('Y-m-d H:i:s'),
					'user_id'				=> get_current_user_id()
				],
				['%s', '%d', '%d', '%s', '%s', '%d', '%f', '%d', '%d', '%d', '%s', '%d']
			);

			$discount_id = $this->wpdb->insert_id;
		}

		if( $discount_type == 'taxonomy' || $discount_type == 'quantity' ){
			foreach( $rules as $rule ){
				if( empty( $rule['taxonomy'] ) || empty( $rule['terms'] ) || empty( $rule['operator'] ) ){
					continue;
				}
				if( !is_array( $rule['terms'] ) ){
					$rule['terms'] = [$rule['terms']];
				}
				$this->wpdb->insert(
					$this->wpdb->prefix . 'wpcd_taxonomy_discount_terms',
					[
						'discount_id' => $discount_id,
						'taxonomy' => $rule['taxonomy'],
						'terms' => implode(',', array_map('sanitize_text_field', $rule['terms'])),
						'operator' => $this->translate_operator_id( $rule['operator'] )
					],
					['%d', '%s', '%s', '%d']
				);
			}
		} 
		
		if ( $discount_type == 'cart' || $discount_type == 'quantity' ){
			$this->wpdb->insert(
				$this->wpdb->prefix . 'wpcd_cart_discount_rules',
				[
					'discount_id' => $discount_id,
					'cart_discount_type' => $cart_discount_type,
					'min_cart_value' => $min_cart_value,
					'max_cart_value' => $max_cart_value,
					'discount_applicable_with_other_discount' => $discount_applicable_with_other_discount,
					'automatically_add_type' => $automatically_add_type
				],
				['%d', '%d', '%f', '%f', '%d', '%d']
			);

			if( $cart_discount_type == 1 ){
				foreach( $cart_rule_products as $product_id ){
					$this->wpdb->insert(
						$this->wpdb->prefix . 'wpcd_cart_discount_rules_products',
						[
							'discount_id' => $discount_id,
							'product_id' => $product_id,
						],
						['%d', '%d']
					);
				}
			}
		}

		$discount_data = $this->get_scheduled_discount_data( $discount_id );

		if( $discount_data['status'] == 1 && ($discount_data['discount_type'] !== 'cart' && $discount_data['discount_type'] !== 'quantity') ){
			if( $this->get_process_method() == 'cron' ){
				$schedule = true;
			} else {
				if( !empty( $discount_data['start_date'] ) ){
					$schedule = true;
				}
			}

			$schdule_discount_data = $this->prepare_schedule_discount_data( $discount_data );

			if( $schedule ){
				$time = empty( $discount_data['start_date'] ) ? time() : strtotime($discount_data['start_date'] . ' 00:00:00');
				wp_schedule_single_event($time, 'wpcd_apply_discount_setup', [$schdule_discount_data]);

				if( !empty( $discount_data['end_date'] ) ){
					wp_schedule_single_event(strtotime($discount_data['end_date'] . ' 23:59:59'), 'wpcd_remove_discount_setup', [$schdule_discount_data]);
				}

				if( isset( $previous_data ) && ( empty( $previous_data['start_date'] ) || $previous_data['start_date'] <= date('Y-m-d') ) ){
					$previous_data['processed_chunks'] = 0;
					$query = $this->get_products_query( $previous_data['discount_type'], $previous_data['rule_type'], $previous_data['taxonomy_rules'] );
					$product_chunks = array_chunk($query->posts, 10);
					$previous_data['total_chunks'] = count($product_chunks);

					return rest_ensure_response([
						'data' => [
							'status' => 200,
							'discount_id' => $discount_data['id'],
							'discount_data' => $previous_data,
							'next' => 'remove',
							'processed' => 0
						],
						'message' => __('Discount is being updated', 'woo-product-category-discount')
					]);
				}

				return rest_ensure_response(['data' => ['status' => 200], 'message' => $params['discount_id'] !== 'new' ? __('Discount updated successfully.', 'woo-product-category-discount') : __('Discount scheduled successfully.', 'woo-product-category-discount')]);
			} else {
				if( !empty( $discount_data['end_date'] ) ){
					wp_schedule_single_event(strtotime($discount_data['end_date'] . ' 23:59:59'), 'wpcd_remove_discount_setup', [$schdule_discount_data]);
				}

				$query = $this->get_products_query( $discount_data['discount_type'], $discount_data['rule_type'], $discount_data['taxonomy_rules'] );
				$product_chunks = array_chunk($query->posts, 10);
				$this->wpdb->update( $this->wpdb->prefix . 'wpcd_discounts', ['total_chunks' => count( $product_chunks )], ['id' => $discount_data['id']] );

				if( isset( $previous_data ) ){
					$_previous_data = $previous_data;
					$_discount_data = $discount_data;
					unset( $_previous_data['total_chunks'], $_previous_data['processed_chunks'], $_previous_data['status'], $_discount_data['total_chunks'], $_discount_data['processed_chunks'], $_discount_data['status'] );

					if( empty( array_diff_assoc( $_previous_data, $_discount_data ) ) ){
						$next = 'process';
					} else if ( $previous_data['status'] == 0 ){
						$next = 'process';
					} else {
						$next = 'remove-process';
					}
				} else {
					$next = 'process';
				}

				return rest_ensure_response([
					'data' => [
						'status' => 200,
						'discount_id' => $discount_data['id'],
						'discount_data' => $previous_data,
						'next' => $next,
						'processed' => 0
					],
					'message' => __('Discount is being updated', 'woo-product-category-discount')
				]);
			}
		}

		if( isset( $previous_data ) && ( empty( $previous_data['start_date'] ) || $previous_data['start_date'] <= date('Y-m-d') ) ){
			if( !empty( $previous_data['end_date'] ) && $previous_data['end_date'] <= date('Y-m-d') ){
				$this->wpdb->update( $this->wpdb->prefix . 'wpcd_discounts', ['total_chunks' => 0], ['id' => $discount_data['id']] );
				return rest_ensure_response(['data' => ['status' => 200], 'message' => $params['discount_id'] !== 'new' ? __('Discount updated successfully.', 'woo-product-category-discount') : __('Discount scheduled successfully.', 'woo-product-category-discount')]);
			} else if ( !empty( $previous_data['end_date'] ) && $previous_data['end_date'] > date('Y-m-d') ) {
				$this->maybe_inactive_discount( $previous_data );
				return rest_ensure_response(['data' => ['status' => 200], 'message' => __('Discount updated successfully.', 'woo-product-category-discount')]);
			}

			$previous_data['processed_chunks'] = 0;
			$query = $this->get_products_query( $previous_data['discount_type'], $previous_data['rule_type'], $previous_data['taxonomy_rules'] );
			$product_chunks = array_chunk($query->posts, 10);
			$previous_data['total_chunks'] = count($product_chunks);
			
			return rest_ensure_response([
				'data' => [
					'status' => 200,
					'discount_id' => $discount_data['id'],
					'discount_data' => $previous_data,
					'next' => $discount_data['status'] == 0 ? 'remove' : 'remove-process',
					'processed' => 0
				],
				'message' => __('Discount is being updated', 'woo-product-category-discount')
			]);
		}

		return rest_ensure_response(['data' => ['status' => 200], 'message' => $params['discount_id'] !== 'new' ? __('Discount updated successfully.', 'woo-product-category-discount') : __('Discount scheduled successfully.', 'woo-product-category-discount')]);
	}

	/**
	 * Handles the request to get a scheduled discount data.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response The response object.
	 */
	public function get_scheduled_discount(WP_REST_Request $request){
		$id = $request['id'];
		$data = $this->get_scheduled_discount_data( $id );
		if( empty( $data ) ){
			return new WP_Error('not_found', __('Discount not found.','woo-product-category-discount'), ['status' => 404]);
		}
		return rest_ensure_response(['data' => ['status' => 200, 'data' => $data], 'message' => __('Discount fetched successfully.', 'woo-product-category-discount')]);
	}

	/**
	 * Handles the request to get a scheduled discount progress.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response The response object.
	 */
	public function get_discount_progress(WP_REST_Request $request){
		$id = $request['id'];
		$data = $this->get_scheduled_discount_data( $id );
		if( empty( $data ) ){
			return new WP_Error('not_found', __('Discount not found.','woo-product-category-discount'), ['status' => 404]);
		}
		return rest_ensure_response(['data' => ['status' => 200, 'data' => ['total' => $data['total_chunks'], 'processed' => $data['processed_chunks'], 'status' => $data['processed_chunks'] == $data['total_chunks'] ? __('Completed', 'woo-product-category-discount') : __('Processing', 'woo-product-category-discount')]], 'message' => __('Discount progress fetched successfully.', 'woo-product-category-discount')]);
	}

	/**
	 * Handles the request to search for products.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response The response object.
	 */
	public function get_products(WP_REST_Request $request){
		$search = sanitize_text_field($request['search']);

		$args = [
			'limit' => 50
		];

		if (!empty($search)) {
			$args['s'] = $search;
		}

		$found_products = wc_get_products($args);
		$products = [];

		foreach ($found_products as $product) {
			if( $product->is_type('variable') ){
				foreach( $product->get_children() as $child_product_id ){
					$products[] = [
						'id'    => $child_product_id,
						'label'  => get_the_title($child_product_id),
						'value' => $child_product_id
					];
				}
			} else {
				$products[] = [
					'id'    => $product->get_id(),
					'label'  => $product->get_name(),
					'value' => $product->get_id(),
				];
			}
		}

		return rest_ensure_response($products);
	}

	/**
	 * Applies a scheduled discount to a set of products.
	 *
	 * This function takes a WP_REST_Request object and applies the discount to the products.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response The response object.
	 * 
	 * @since 5.8
	 */
	public function process_discount(WP_REST_Request $request){
		$id = $request['discount_id'];

		$data = $this->get_scheduled_discount_data( $id );

		if( $data['processed_chunks'] == $data['total_chunks'] ){
			$this->wpdb->update( $this->wpdb->prefix . 'wpcd_discounts', ['processed_chunks' => 0], ['id' => $id] );
			$data = $this->get_scheduled_discount_data( $id );
		}

		$query = $this->get_products_query( $data['discount_type'], $data['rule_type'], $data['taxonomy_rules'], $data['processed_chunks']+1, 10 );

		$discount_applied = $this->apply_discount( $id, $data['discount_amount'], $data['discount_amount_type'], $query->posts );

		if( $discount_applied ){
			$processed = round( 100 * ( $data['processed_chunks'] + 1 ) / $data['total_chunks'], 2, PHP_ROUND_HALF_UP);
			return rest_ensure_response([
				'data' => [
					'status' => 200,
					'discount_id' => $data['id'],
					'processed' => $processed
				],
				'message' => $processed >= 100 ? __('Discount applied successfully', 'woo-product-category-discount') : __('Discount is being updated', 'woo-product-category-discount')
			]);
		} else {
			return rest_ensure_response([
				'data' => [
					'status' => 500,
				],
				'message' => __('Something went wrong', 'woo-product-category-discount')
			]);
		}
	}

	/**
	 * Removes a scheduled discount from a set of products.
	 *
	 * This function takes a WP_REST_Request object and removes the discount from the products.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response The response object.
	 * 
	 * @since 5.8
	 */
	public function remove_discount_process(WP_REST_Request $request){
		$data = $request['discount_data'];
		$query = $this->get_products_query( $data['discount_type'], $data['rule_type'], $data['taxonomy_rules'], $data['processed_chunks']+1, 10 );

		$discount_applied = $this->remove_discount( $data['id'], $query->posts, true );

		if( $discount_applied ){
			$processed = round( 100 * ( $data['processed_chunks'] + 1 ) / $data['total_chunks'], 2, PHP_ROUND_HALF_UP);
			return rest_ensure_response([
				'data' => [
					'status' => 200,
					'discount_id' => $data['id'],
					'processed' => $processed
				],
				'message' => $processed >= 100 ? __('Discount removed successfully', 'woo-product-category-discount') : __('Discount is being removed', 'woo-product-category-discount')
			]);
		} else {
			return rest_ensure_response([
				'data' => [
					'status' => 500,
				],
				'message' => __('Something went wrong', 'woo-product-category-discount')
			]);
		}
	}

	/**
	 * Apply discount to products.
	 *
	 * This function takes a query object based on the discount type and rules,
	 * and applies the discount to the products in chunks of 10.
	 *
	 * @param array $data Contains the discount id, discount amount, discount amount type, and taxonomy rules.
	 */
	public function apply_discount_setup( $data ){
		$query = $this->get_products_query( $data['discount_type'], $data['rule_type'], $data['taxonomy_rules'] );
		$product_chunks = array_chunk($query->posts, 10);
		$this->wpdb->update( $this->wpdb->prefix . 'wpcd_discounts', ['total_chunks' => count( $product_chunks )], ['id' => $data['id']] );
		$time = time();
		foreach( $product_chunks as $chunk ){
			wp_schedule_single_event($time, 'wpcd_apply_discount', [$data['id'], $data['discount_amount'], $data['discount_amount_type'], $chunk]);
			$time = $time + 10;
		}
	}

	/**
	 * Applies a discount to products.
	 *
	 * This function takes a discount id, discount amount, discount amount type, and a list of product ids,
	 * and applies the discount to the products.
	 *
	 * @param int $discount_id The id of the discount.
	 * @param float $discount_amount The amount of the discount.
	 * @param string $discount_amount_type The type of the discount amount, either 'percentage' or 'fixed'.
	 * @param array $product_ids The list of product ids to which the discount should be applied.
	 * @param bool $update_processed_chunks Whether to update the processed chunks or not.
	 * 
	 * @return bool
	 */
	public function apply_discount( $discount_id, $discount_amount, $discount_amount_type, $product_ids, $update_processed_chunks=true ){
		$custom_fields = self::custom_fields();
		foreach( $product_ids as $product_id ){
			$product = wc_get_product( $product_id );
			if( $product->is_type('variable') ){
				$variations = $product->get_children();
				foreach( $variations as $variation_id ){
					$discount_applied = get_post_meta( $variation_id, '_wpcd_discount_id', true );
					$variation = wc_get_product( $variation_id );
					
					$_price = $variation->get_price();
					if( empty( $_price ) ){
						continue;
					}

					$regular_price = $variation->get_regular_price();
					$price = $regular_price;

					if( !empty( $discount_applied ) ){
						$new_price = $discount_amount_type == 'percentage' ? ($price - ($price * $discount_amount / 100)) : ($price - $discount_amount);
						if( $new_price >= $_price ){
							continue;
						} else {
							$this->remove_discount( $discount_applied, [$variation_id], true, false );
						}
					}
					
					$variation = wc_get_product( $variation->get_id() );
					$sale_price = $variation->get_sale_price();
					$_price = $variation->get_price();

					$new_price = $discount_amount_type == 'percentage' ? ($price - ($price * $discount_amount / 100)) : ($price - $discount_amount);
					$variation->set_sale_price( $new_price );
					$variation->save();
					update_post_meta( $variation_id, '_wpcd_discount_id', $discount_id );
					update_post_meta( $variation_id, '_wpcd_original_regular_price', $regular_price );
					update_post_meta( $variation_id, '_wpcd_original_sale_price', $sale_price );
					update_post_meta( $variation_id, '_wpcd_original_price', $_price );

					if ( defined( 'ICL_SITEPRESS_VERSION' ) && has_filter( 'wpml_sync_custom_field' ) ) {
						foreach( $custom_fields as $field ){
							do_action('wpml_sync_custom_field', $variation_id, $field);
						}
					}
				}
			} else {
				$discount_applied = get_post_meta( $product_id, '_wpcd_discount_id', true );
				$_price = $product->get_price();
				if( empty( $_price ) ){
					continue;
				}

				$regular_price = $product->get_regular_price();
				$price = $regular_price;
				if( empty( $price ) ){
					continue;
				}

				if( !empty( $discount_applied ) ){
					$new_price = $discount_amount_type == 'percentage' ? ($price - ($price * $discount_amount / 100)) : ($price - $discount_amount);
					if( $new_price >= $_price ){
						continue;
					} else {
						$this->remove_discount( $discount_applied, [$product_id], true, false );
					}
				}
				
				$product = wc_get_product( $product->get_id() );
				$sale_price = $product->get_sale_price();
				$_price = $product->get_price();
				
				$new_price = $discount_amount_type == 'percentage' ? ($price - ($price * $discount_amount / 100)) : ($price - $discount_amount);
				$product->set_sale_price( $new_price );
				$product->save();
				update_post_meta( $product_id, '_wpcd_discount_id', $discount_id );
				update_post_meta( $product_id, '_wpcd_original_regular_price', $regular_price );
				update_post_meta( $product_id, '_wpcd_original_sale_price', $sale_price );
				update_post_meta( $product_id, '_wpcd_original_price', $_price );

				if ( defined( 'ICL_SITEPRESS_VERSION' ) && has_filter( 'wpml_sync_custom_field' ) ) {
					foreach( $custom_fields as $field ){
						do_action('wpml_sync_custom_field', $product_id, $field);
					}
				}
			}
		}

		if( $update_processed_chunks){
			$this->wpdb->query( $this->wpdb->prepare( "UPDATE {$this->wpdb->prefix}wpcd_discounts SET processed_chunks = processed_chunks + 1 WHERE id = %d", $discount_id ) );
		}

		return true;
	}

	/**
	 * Sets up the removal of a discount.
	 *
	 * This function takes the discount id, discount type, rule type, and taxonomy rules,
	 * and schedules the removal of the discount from the products in chunks of 10.
	 *
	 * @param array $data Contains the discount id, discount type, rule type, and taxonomy rules.
	 */
	public function remove_discount_setup( $data ){
		if( empty( $data ) ){
			return;
		}
		
		$query = $this->get_products_query( $data['discount_type'], $data['rule_type'], $data['taxonomy_rules'] );
		$product_chunks = array_chunk($query->posts, 10);
		$this->wpdb->update( $this->wpdb->prefix . 'wpcd_discounts', ['total_chunks' => count( $product_chunks ), 'processed_chunks' => 0], ['id' => $data['id']] );
		$time = time();
		foreach( $product_chunks as $chunk ){
			wp_schedule_single_event($time, 'wpcd_remove_discount', [$data['id'], $chunk]);
			$time = $time + 10;
		}
	}

	/**
	 * Removes a discount from a list of products.
	 *
	 * This function takes a discount id and a list of product ids, and removes the discount
	 * from the products. If the discount is not present on a product, the function
	 * does nothing.
	 *
	 * @param int $discount_id The id of the discount.
	 * @param array $product_ids The list of product ids from which the discount should be removed.
	 * @param bool $avoid_apply_latest_discount Whether to avoid applying the latest discount to the products.
	 * @param bool $update_processed_chunks Whether to update the processed chunks of the discount. 
	 * 
	 * @return bool
	*/
	public function remove_discount( $discount_id, $product_ids, $avoid_apply_latest_discount = false, $update_processed_chunks = true) {
		$custom_fields = self::custom_fields();
		foreach ( $product_ids as $product_id ) {
			$product = wc_get_product( $product_id );
			if ( ! $product ) continue;

			if ( $product->is_type( 'variable' ) ) {
				$variations = $product->get_children();
				foreach ( $variations as $variation_id ) {
					$discount_applied = get_post_meta( $variation_id, '_wpcd_discount_id', true );
					if ( empty( $discount_applied ) || $discount_applied != $discount_id ) {
						continue;
					}

					$variation = wc_get_product( $variation_id );
					if ( ! $variation ) continue;

					$regular_price = get_post_meta( $variation_id, '_wpcd_original_regular_price', true );
					$sale_price    = get_post_meta( $variation_id, '_wpcd_original_sale_price', true );

					$variation->set_regular_price( $regular_price );
					$variation->set_sale_price( $sale_price );

					$variation->save();

					delete_post_meta( $variation_id, '_wpcd_discount_id' );
					delete_post_meta( $variation_id, '_wpcd_original_regular_price' );
					delete_post_meta( $variation_id, '_wpcd_original_sale_price' );
					delete_post_meta( $variation_id, '_wpcd_original_price' );

					if ( defined( 'ICL_SITEPRESS_VERSION' ) && has_filter( 'wpml_sync_custom_field' ) ) {
						foreach( $custom_fields as $field ){
							do_action('wpml_sync_custom_field', $variation_id, $field);
						}
					}

					wc_delete_product_transients( $variation_id );
					if( !$avoid_apply_latest_discount ){
						$this->apply_latest_discount($variation);
					}
				}
			} else {
				$discount_applied = get_post_meta( $product_id, '_wpcd_discount_id', true );
				if ( empty( $discount_applied ) || $discount_applied != $discount_id ) {
					continue;
				}

				$regular_price = get_post_meta( $product_id, '_wpcd_original_regular_price', true );
				$sale_price    = get_post_meta( $product_id, '_wpcd_original_sale_price', true );

				$product->set_regular_price( $regular_price );
				$product->set_sale_price( $sale_price );
				
				$product->save();

				delete_post_meta( $product_id, '_wpcd_discount_id' );
				delete_post_meta( $product_id, '_wpcd_original_regular_price' );
				delete_post_meta( $product_id, '_wpcd_original_sale_price' );
				delete_post_meta( $product_id, '_wpcd_original_price' );

				if ( defined( 'ICL_SITEPRESS_VERSION' ) && has_filter( 'wpml_sync_custom_field' ) ) {
					foreach( $custom_fields as $field ){
						do_action('wpml_sync_custom_field', $product_id, $field);
					}
				}

				wc_delete_product_transients( $product_id );
				if( !$avoid_apply_latest_discount ){
					$this->apply_latest_discount($product);
				}
			}
		}

		if( $update_processed_chunks ){
			$this->wpdb->query( $this->wpdb->prepare( "UPDATE {$this->wpdb->prefix}wpcd_discounts SET processed_chunks = processed_chunks + 1 WHERE id = %d", $discount_id ) );
		}

		return true;
	}

	/**
	 * Get a WP_Query object based on the discount type and rules.
	 *
	 * This function takes a discount type, match type, and taxonomy rules, and returns a WP_Query object
	 * that can be used to fetch products based on the rules.
	 *
	 * @param string $discount_type The type of discount to apply, either 'taxonomy' or 'all'.
	 * @param string $match_type The type of match to apply, either 'all' or 'any'.
	 * @param array $taxonomy_rules The rules to apply for the taxonomy match.
	 * @param int|null $page The page number to fetch, or null to fetch all products.
	 * @param int $per_page The number of products to fetch per page, or -1 to fetch all products.
	 *
	 * @return WP_Query The WP_Query object based on the given parameters.
	 */
	private function get_products_query($discount_type, $match_type, $taxonomy_rules, $page=null, $per_page=-1){	
		$filter = [
			'post_type'      => 'product',
			'posts_per_page' => $per_page,
			'fields'         => 'ids',
			'orderby'        => 'ID',
            'order'          => 'DESC',
			'post_status'    => 'any',
		];

		if( !is_null( $page ) ){
			$filter['paged'] = $page;
		}

		$rules = $taxonomy_rules ?? [];
		if ($discount_type !== 'taxonomy' || empty($rules)) {
			return new WP_Query($filter);
		}

		$tax_query = ['relation' => $match_type == 'all' ? 'AND' : 'OR' ];

		foreach ($rules as $rule) {
			$taxonomy = sanitize_text_field($rule['taxonomy'] ?? '');
			$operator = sanitize_text_field($rule['operator'] ?? 'eq');
			$terms    = $rule['terms'] ?? [];

			if( !is_array( $terms ) ) {
				$terms = [$terms];
			}

			if (!$taxonomy || empty($terms)) continue;

			$tax_query[] = [
				'taxonomy' => $taxonomy,
				'field'    => 'term_id',
				'terms'    => array_map('sanitize_text_field', $terms),
				'operator' => $this->translate_operator($operator),
			];
		}

		$filter['tax_query'] = $tax_query;

		return new WP_Query($filter);
	}
	
	/**
	 * Retrieves a scheduled discount data.
	 *
	 * This function takes a discount id and retrieves the discount data from the database.
	 * The data includes the discount id, name, discount type, rule type, start date, end date,
	 * discount amount type, discount amount, status, and taxonomy rules.
	 *
	 * @param int $discount_id The id of the discount.
	 *
	 * @return array|false The discount data or false if the discount does not exist.
	 */
	public function get_scheduled_discount_data( $discount_id ){
		$discount_id = sanitize_text_field( $discount_id );

		$discount_row = $this->wpdb->get_row( $this->wpdb->prepare( "SELECT * FROM {$this->wpdb->prefix}wpcd_discounts WHERE id=%d", $discount_id ) );
		if( empty( $discount_row ) ){
			return false;
		}
		$discount_data = [
			'id' => $discount_id,
			'name' => $discount_row->name ?? '',
			'discount_type' => $discount_row->discount_type == 0 ? 'all_products' : ( $discount_row->discount_type == 1 ? 'taxonomy' : ($discount_row->discount_type == 2 ? 'cart' : 'quantity') ),
			'rule_type' => $discount_row->rule_type == 0 ? 'all' : 'any',
			'start_date' => $discount_row->start_date ?? '',
			'end_date' => $discount_row->end_date ?? '',
			'discount_amount_type' => $discount_row->discount_amount_type == 0 ? 'percentage' : 'flat',
			'discount_amount' => $discount_row->discount_amount ?? 0,
			'status' => $discount_row->status ?? 0,
			'total_chunks' => $discount_row->total_chunks ?? 0,
			'processed_chunks' => $discount_row->processed_chunks ?? 0,
			'taxonomy_rules' => [],
			'cart_discount_type' => 0,
			'min_cart_value' => null,
			'max_cart_value' => null,
			'discount_applicable_with_other_discount' => 1,
			'automatically_add_type' => 1,
			'free_products' => [],
		];

		if( $discount_row->discount_type == 1 || $discount_row->discount_type == 3 ){
			$taxonomy_rules = [];
			$discount_taxonomies = $this->wpdb->get_results( $this->wpdb->prepare( "SELECT * FROM {$this->wpdb->prefix}wpcd_taxonomy_discount_terms WHERE discount_id=%d", $discount_id ) );
			foreach( $discount_taxonomies as $discount_taxonomy ){
				$taxonomy_rules[] = [
					'taxonomy' => $discount_taxonomy->taxonomy ?? '',
					'terms' => empty( $discount_taxonomy->terms ) ? [] : explode(',', $discount_taxonomy->terms),
					'operator' => $this->translate_operator_key( $discount_taxonomy->operator ?? '' )
				];
			}

			$discount_data['taxonomy_rules'] = $taxonomy_rules;
		} 
		
		if ( $discount_row->discount_type == 2 || $discount_row->discount_type == 3 ) {
			$cart_discount_data = $this->wpdb->get_row( $this->wpdb->prepare( "SELECT * FROM {$this->wpdb->prefix}wpcd_cart_discount_rules WHERE discount_id=%d", $discount_id ) );
			$discount_data['cart_discount_type'] = $cart_discount_data->cart_discount_type == 1 ? 'free_product' : 'cart_value';
			$discount_data['min_cart_value'] = $cart_discount_data->min_cart_value ?? null;
			$discount_data['max_cart_value'] = $cart_discount_data->max_cart_value ?? null;
			$discount_data['discount_applicable_with_other_discount'] = $cart_discount_data->discount_applicable_with_other_discount == 1 ? 'yes' : 'no';
			$discount_data['automatically_add_type'] = $cart_discount_data->automatically_add_type == 1 ?'yes' : 'no';
			if( $cart_discount_data->cart_discount_type == 1 ){
				$free_products = $this->wpdb->get_col( $this->wpdb->prepare( "SELECT product_id FROM {$this->wpdb->prefix}wpcd_cart_discount_rules_products WHERE discount_id=%d", $discount_id ) );
				$discount_data['free_products'] = $free_products;
				$free_products_data = [];
				foreach( $free_products as $free_product_id ){
					$free_product = wc_get_product( $free_product_id );
					$free_products_data[] = [
						'value' => $free_product_id,
						'label' => get_the_title( $free_product_id ),
					];
				}
				$discount_data['free_products_data'] = $free_products_data;
			}
		}

		return $discount_data;
	}

	/**
	 * Hides the WPML menu on the plugin's settings page.
	 *
	 * We do this to prevent the user from switching languages while on the plugin's settings page, which can cause problems with the plugin's functionality.
	 */
	public function hide_wpml_menu(){
		$screen = get_current_screen();
		if( $screen && $screen->id === 'quanticedge_page_woo-product-category-discount' ){
			echo '<style>#wp-admin-bar-WPML_ALS { display: none !important; }</style>';
		}
	}

	/**
	 * Force the default language for the plugin's settings page.
	 *
	 * We do this to ensure that the plugin's settings page is always displayed
	 * in the default language, even if the user switches languages.
	 *
	 * @param object $screen The WP_Screen object for the current page.
	 */
	public function force_wpml_language( $screen ){
		if( $screen->id === 'quanticedge_page_woo-product-category-discount' ){
			$default_lang = apply_filters( 'wpml_default_language', null );
        	do_action( 'wpml_switch_language', $default_lang );
		}
	}

	/**
	 * Schedule or unschedule discount events based on the discount's status and dates.
	 *
	 * This function checks the status of a discount and its start and end dates
	 * to determine whether to schedule or unschedule events related to discount
	 * application and removal.
	 *
	 * @param array $discount_data The discount data which includes status, start date, and end date.
	 */
	private function maybe_inactive_discount( $discount_data ){
		if( empty( $discount_data ) || !is_array( $discount_data ) || !isset( $discount_data['status'], $discount_data['start_date'], $discount_data['end_date'] ) || $discount_data['discount_type'] == 'cart' || $discount_data['discount_type'] == 'quantity' ){
			return;
		}

		if( $discount_data['status'] == 1 ){
			$this->wpdb->update( $this->wpdb->prefix . 'wpcd_discounts', ['total_chunks' => 0], ['id' => $discount_data['id']] );
			$schdule_discount_data = $this->prepare_schedule_discount_data( $discount_data );
			if( !empty( $discount_data['start_date'] ) && strtotime( $discount_data['start_date'] . ' 00:00:00' ) > time() ){
				wp_unschedule_event( strtotime( $discount_data['start_date'] . ' 00:00:00' ), 'wpcd_apply_discount_setup', [$schdule_discount_data] );
			} else {
				wp_schedule_single_event(time(), 'wpcd_remove_discount_setup', [$schdule_discount_data]);
			}

			if( !empty( $discount_data['end_date'] ) && strtotime( $discount_data['end_date'] . ' 23:59:59' ) > time() ){
				wp_unschedule_event( strtotime( $discount_data['end_date'] . ' 23:59:59' ), 'wpcd_remove_discount_setup', [$schdule_discount_data] );
			}
		}
	}

	/**
	 * Deletes a discount and any associated taxonomy rules.
	 *
	 * @param array $discount_data The discount data to delete.
	 */
	private function delete_discount( $discount_data ){
		if( empty( $discount_data ) || !is_array( $discount_data ) || !isset( $discount_data['id'] ) ){
			return;
		}

		$this->maybe_inactive_discount( $discount_data );
		$this->wpdb->delete( $this->wpdb->prefix . 'wpcd_discount_taxonomies', [ 'discount_id' => $discount_data['id'] ] );
		$this->wpdb->delete( $this->wpdb->prefix . 'wpcd_discounts', [ 'id' => $discount_data['id'] ] );
	}

	/**
	 * Translate the given operator from the frontend to the value expected by WordPress's taxonomy queries.
	 *
	 * @param string $operator The operator to translate. Can be 'eq', 'neq', 'in', 'not_in'.
	 *
	 * @return string The translated operator.
	 */
	private function translate_operator(string $operator): string {
		switch ($operator) {
			case 'eq':
			case 'in':
				return 'IN';
			case 'neq':
			case 'not_in':
				return 'NOT IN';
			default:
				return 'IN';
		}
	}

	/**
	 * Translate the given operator from the frontend to the id expected by the database.
	 *
	 * @param string $operator The operator to translate. Can be 'eq', 'neq', 'in', 'not_in'.
	 *
	 * @return int The translated id.
	 */
	private function translate_operator_id($operator){
		switch ($operator) {
			case 'eq': return 1;
			case 'neq': return 0;
			case 'in': return 2;
			case 'not_in': return 3;
			default: return 2;
		}
	}

	/**
	 * Translate the given id to the corresponding operator key used in taxonomy queries.
	 *
	 * @param int $id The id to translate. Can be 1, 0, 2, or 3.
	 *
	 * @return string The translated operator key. Returns 'eq', 'neq', 'in', or 'not_in'.
	 */
	private function translate_operator_key($id){
		switch ($id) {
			case 1: return 'eq';
			case 0: return 'neq';
			case 2: return 'in';
			case 3: return 'not_in';
			default: return 'in';
		}
	}

	/**
	 * The custom fields used to save the original prices of a product.
	 *
	 * @return array The custom fields.
	 */
	private static function custom_fields(){
		return [
			'_wpcd_discount_id',
			'_wpcd_original_regular_price',
			'_wpcd_original_sale_price',
			'_wpcd_original_price',
			'_price',
			'_regular_price',
			'_sale_price'
		];
	}

	/**
	 * Applies the latest discount to a product.
	 *
	 * This function takes a product or product id, and applies the latest discount to it.
	 *
	 * @param WC_Product|integer $product The product to apply the discount to.
	 *
	 * @since 5.4
	 */
	private function apply_latest_discount($product){
		if( is_int( $product ) ){
			$product = wc_get_product( $product );
		}

		$product_id = $product->get_id();

		if( $product->is_type('variation') ){
			$product = wc_get_product( $product->get_parent_id() );
		}

		$discounts = $this->wpdb->get_col( $this->wpdb->prepare("SELECT id FROM {$this->wpdb->prefix}wpcd_discounts WHERE status = 1 AND discount_type IN (0,1) AND (start_date IS NULL OR %s >= start_date) AND (end_date IS NULL OR %s <= end_date)", date('Y-m-d H:i:s'), date('Y-m-d H:i:s')) );
		
		if( !empty( $discounts ) ){
			$applicable_discounts = null;
			$last_price = PHP_INT_MAX;
			foreach( $discounts as $discount_id ){
				$discount = $this->get_scheduled_discount_data( $discount_id );
				$products = $this->get_products_query( $discount['discount_type'], $discount['rule_type'], $discount['taxonomy_rules'] );
				if( in_array( $product->get_id(), $products->posts ) ){
					$price = floatval($product->get_price());
					if( empty( $price ) || empty( $discount['discount_amount'] ) ){
						return;
					}
					$new_price = $discount['discount_amount_type'] == 'percentage' ? ($price - ($price * $discount['discount_amount'] / 100)) : ($price - $discount['discount_amount']);

					if( $new_price < $last_price ){
						$last_price = $new_price;
						$applicable_discounts = [
							'id' => $discount['id'],
							'discount_amount' => $discount['discount_amount'],
							'discount_amount_type' => $discount['discount_amount_type'],
						];
					}
				}
			}

			if( !is_null( $applicable_discounts ) ){
				$this->apply_discount( $applicable_discounts['id'], $applicable_discounts['discount_amount'], $applicable_discounts['discount_amount_type'], [$product_id], false );
			} else {
				$discount_applied = get_post_meta( $product_id, '_wpcd_discount_id', true );
				if( !empty( $discount_applied ) ){
					$this->remove_discount( $discount_applied, [$product_id], true, false );
				}
			}
		}
	}

	/**
	 * Applies the latest discount to a product when it is saved.
	 *
	 * This function is called when a product is saved in the admin interface.
	 * It takes a product id, post object, and update boolean as parameters.
	 *
	 * @param int $product_id The id of the product to apply the discount to.
	 * @param WP_Post $post The post object of the product.
	 *
	 * @since 5.4
	 */
	public function apply_discount_price_on_product_save($product_id, $post){
		if( !is_admin() ){
			return;
		}

		$product = wc_get_product( $product_id );

		if( $product->is_type('variable') ){
			foreach( $product->get_children() as $variation_id ){
				$this->apply_latest_discount( $variation_id );
			}
		} else {
			$this->apply_latest_discount( $product->get_id() );
		}
	}

	/**
	 * When a product's price is changed, update the original regular price saved by the plugin.
	 *
	 * This function is called when a product's price is updated in the admin interface.
	 * It takes a meta id, post id, meta key, and meta value as parameters.
	 *
	 * @param int $meta_id The id of the meta field being updated.
	 * @param int $post_id The id of the post being updated.
	 * @param string $meta_key The key of the meta field being updated.
	 * @param string $meta_value The value of the meta field being updated.
	 *
	 * @since 5.4
	 */
	public function change_price_keys( $meta_id, $post_id, $meta_key, $meta_value ){
		if( $meta_key !== '_regular_price' ){
			return;
		}
		
		$wpcd_price = get_post_meta( $post_id, '_wpcd_original_regular_price', true );
		if( !empty( $wpcd_price ) && $wpcd_price != $meta_value ){
			update_post_meta( $post_id, '_wpcd_original_regular_price', $meta_value );
		}
	}

	/**
	 * Prepares the discount data for scheduling.
	 *
	 * @param array $discount_data The discount data to prepare.
	 *
	 * @return array The prepared discount data.
	 */
	private function prepare_schedule_discount_data( $discount_data ){
		unset( $discount_data['total_chunks'] );
		unset( $discount_data['processed_chunks'] );
		return $discount_data;
	}

	/**
	 * Adds a settings link to the plugin list table.
	 *
	 * @param array $links The links to add the settings link to.
	 *
	 * @return array The links with the settings link added.
	 *
	 * @since 5.6
	 */
	public function add_settings_link( $links ){
		$settings_link = '<a href="' . admin_url( 'admin.php?page=' . self::$menu_slug ) . '">' . __( 'Settings', 'woo-product-category-discount' ) . '</a>';
		array_unshift( $links, $settings_link );
		return $links;
	}

	/**
	 * Updates the status of a discount.
	 *
	 * This function is called by the AJAX request to update the status of a discount.
	 *
	 * @since 5.8
	 */
	public function update_discount_status(){
		if ( !isset( $_REQUEST['nonce'] ) || !wp_verify_nonce( sanitize_text_field(wp_unslash( $_REQUEST['nonce'] )), 'update_discount_nonce' ) ) {
			wp_send_json_error( array( 'message' => __('Invalid request.', 'woo-product-category-discount') ), 403 );
			wp_die();
    	}

		if ( !current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __('Unauthorized access.', 'woo-product-category-discount') ), 403 );
			wp_die();
		}

		if( !isset( $_REQUEST['discount_id'])){
			wp_send_json_error( array( 'message' => __('Discount id is required.', 'woo-product-category-discount') ), 403 );
			wp_die();
		}

		$discount_data = $this->get_scheduled_discount_data( $_REQUEST['discount_id'] );

		$discount_status = wpcd_get_admin_discount_status( $discount_data );

		if( $discount_status != 'active' && $discount_status != 'inactive' ){
			wp_send_json_error( array( 'message' => __('This discount cannot be changed.', 'woo-product-category-discount') ), 403 );
			wp_die();
		}

		if( $discount_status == 'active' ){
			$this->wpdb->update( $this->wpdb->prefix . 'wpcd_discounts', [ 'status' => 0, 'total_chunks' => 1, 'processed_chunks' => 0, 'user_id' => get_current_user_id(), 'updated_at' => date( 'Y-m-d H:i:s') ], [ 'id' => $_REQUEST['discount_id'] ] );
			if( $discount_data['discount_type'] !== 'cart' && $discount_data['discount_type'] !== 'quantity'){
				if( (!empty( $discount_data['end_date'] ) ) || $this->get_process_method() == 'cron' ){
					$this->maybe_inactive_discount( $discount_data );
				} else {
					$query = $this->get_products_query( $discount_data['discount_type'], $discount_data['rule_type'], $discount_data['taxonomy_rules'] );
					$product_chunks = array_chunk($query->posts, 10);

					$discount_data['total_chunks'] = count( $product_chunks );

					wp_send_json_success( array( 'message' => __('Discount updation started successfully.', 'woo-product-category-discount'), 'next' => 'remove', 'discount_data' => $discount_data) );
				}
			}
		} else {
			if( !empty( $discount_data['end_date'] ) && $discount_data['end_date'] < date( 'Y-m-d' ) ){
				wp_send_json_error( array( 'message' => __('This discount cannot be activated because end date is already passed.', 'woo-product-category-discount') ), 403 );
				wp_die();
			}
			if( !empty( $discount_data['start_date'] ) && $discount_data['start_date'] < date( 'Y-m-d' ) ){
				wp_send_json_error( array( 'message' => __('This discount cannot be activated because start date is already passed.', 'woo-product-category-discount') ), 403 );
				wp_die();
			}
			$this->wpdb->update( $this->wpdb->prefix . 'wpcd_discounts', [ 'status' => 1, 'total_chunks' => 1, 'processed_chunks' => 0, 'user_id' => get_current_user_id(), 'updated_at' => date( 'Y-m-d H:i:s' ) ], [ 'id' => $_REQUEST['discount_id'] ] );
			if( $discount_data['discount_type'] !== 'cart' && $discount_data['discount_type'] !== 'quantity'){
				if( !empty( $discount_data['end_date'] ) ){
					wp_schedule_single_event(strtotime($discount_data['end_date'] . ' 23:59:59'), 'wpcd_remove_discount_setup', [$schdule_discount_data]);
				}

				if( $this->get_process_method() == 'cron' ){
					$time = empty( $discount_data['start_date'] ) ? time() : strtotime($discount_data['start_date'] . ' 00:00:00');
					$schdule_discount_data = $this->prepare_schedule_discount_data( $discount_data );
					wp_schedule_single_event($time, 'wpcd_apply_discount_setup', [$schdule_discount_data]);
				} else {
					$query = $this->get_products_query( $discount_data['discount_type'], $discount_data['rule_type'], $discount_data['taxonomy_rules'] );
					$product_chunks = array_chunk($query->posts, 10);
					$this->wpdb->update( $this->wpdb->prefix . 'wpcd_discounts', ['total_chunks' => count( $product_chunks )], ['id' => $discount_data['id']] );

					wp_send_json_success( array( 'message' => __('Discount updation started successfully.', 'woo-product-category-discount'), 'next' => 'process') );
				}
			}
		}

		$discount_data = $this->get_scheduled_discount_data( $_REQUEST['discount_id'] );
		wp_send_json_success( array( 'message' => __('Discount status updated successfully.', 'woo-product-category-discount'), 'html' => wpcd_get_admin_discount_status_html( $discount_data ) ) );
 	}

	/**
	 * Terminates a scheduled discount progress.
	 *
	 * This function is called when the user wants to terminate a scheduled discount progress.
	 *
	 * @since 5.11
	 */
	public function terminate_discount_progress() {
		if ( !isset( $_REQUEST['nonce'] ) || !wp_verify_nonce( sanitize_text_field(wp_unslash( $_REQUEST['nonce'] )), 'terminate_discount_nonce' ) ) {
			wp_send_json_error( array( 'message' => __('Invalid request.', 'woo-product-category-discount') ), 403 );
			wp_die();
    	}

		if ( !current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __('Unauthorized access.', 'woo-product-category-discount') ), 403 );
			wp_die();
		}

		if( !isset( $_REQUEST['discount_id'])){
			wp_send_json_error( array( 'message' => __('Discount id is required.', 'woo-product-category-discount') ), 403 );
			wp_die();
		}

		$discount_data = $this->get_scheduled_discount_data( $_REQUEST['discount_id'] );

		if( empty( $discount_data ) ){
			wp_send_json_error( array( 'message' => __('Discount not found.', 'woo-product-category-discount') ), 403 );
			wp_die();
		}

		if( empty($discount_data['end_date']) && $this->get_process_method() == 'ajax' ){
			$this->wpdb->update( $this->wpdb->prefix . 'wpcd_discounts', [ 'status' => !$discount_data['status'] ?? 0, 'processed_chunks' => $discount_data['total_chunks'] ?? 0, 'updated_at' => date( 'Y-m-d H:i:s' ) ], [ 'id' => $_REQUEST['discount_id'] ] );

			wp_send_json_success( array( 'message' => __('Discount process terminated successfully.', 'woo-product-category-discount'), 'html' => wpcd_get_admin_discount_status_html( $discount_data ) ) );
		} else {
			wp_send_json_error( array( 'message' => __('Discount process cannot be terminated.', 'woo-product-category-discount') ), 403 );
			wp_die();
		}
	}

	/**
	 * Upgrades the table schema if the plugin was installed before 5.8
	 *
	 * This function is called on plugin activation and checks if the installed version is false.
	 * If it is, it adds user_id and updated_at columns to the wpcd_discounts table and updates the installed version to the current version.
	 *
	 * @since 5.8
	 */
	public function maybe_upgrade_table_schema(){
		if( get_option('wpcd_tables_created') != 'yes' ) {
			activate_wpcd_category_discount();
		}
		
		$installed_version = get_option('wpcd_category_discount_version');
		if ($installed_version === false) {
			$table = $this->wpdb->prefix . 'wpcd_discounts';
			$this->wpdb->query( "ALTER TABLE $table ADD COLUMN user_id INT DEFAULT NULL" );
			$this->wpdb->query( "ALTER TABLE $table ADD COLUMN updated_at datetime DEFAULT NULL" );
			update_option('wpcd_category_discount_version', WPCD_CATEGORY_DISCOUNT_VERSION);
		} elseif (version_compare($installed_version, WPCD_CATEGORY_DISCOUNT_VERSION, '<')) {
			update_option('wpcd_category_discount_version', WPCD_CATEGORY_DISCOUNT_VERSION);
		}
	}

	/**
	 * Registers the plugin's settings.
	 *
	 * This function is called when the plugin is initialized and registers the 'wpcd_process_method' setting.
	 * It also adds a settings section and field to the plugin's settings page.
	 *
	 * @since 5.8
	 */
	public function register_settings(){
		register_setting('wpcd_settings_group', 'wpcd_process_method',[
			'type' => 'string',
			'sanitize_callback' => 'sanitize_text_field',
			'default' => 'ajax'
		]);

		add_settings_section(
			'wpcd_settings_section',
			__('General Settings', 'woo-product-category-discount'),
			array( $this, 'settings_section_callback'),
			'wpcd-settings'
		);

		add_settings_field(
			'wpcd_process_method',
			__('Process Method', 'woo-product-category-discount'),
			array( $this, 'process_method_field_callback'),
			'wpcd-settings',
			'wpcd_settings_section'
		);
	}

	/**
	 * Includes the settings section partial.
	 *
	 * This function is called as a callback for the 'wpcd_settings_section' settings section
	 * and includes the 'partials/woo-product-category-discount-settings-sections.php' partial.
	 *
	 * @since 5.8
	 */
	public function settings_section_callback(){
		include_once plugin_dir_path( __FILE__ ) . 'partials/woo-product-category-discount-settings-sections.php';
	}

	/**
	 * Includes the process method field partial.
	 *
	 * This function is called as a callback for the 'wpcd_process_method' settings field
	 * and includes the 'partials/woo-product-category-discount-process-method.php' partial.
	 *
	 * @since 5.8
	 */
	public function process_method_field_callback(){
		$value = $this->get_process_method();
		include_once plugin_dir_path( __FILE__ ) . 'partials/woo-product-category-discount-process-method.php';
	}

	/**
	 * Retrieves the process method setting.
	 *
	 * This function retrieves the 'wpcd_process_method' setting, which determines how discounts are processed in the store.
	 * If the setting does not exist, it defaults to 'ajax'.
	 *
	 * @return string The process method setting.
	 * 
	 * @since 5.8
	 */
	public function get_process_method(){
		return get_option('wpcd_process_method', 'ajax');
	}

	/**
	 * Retrieves the latest discount status.
	 *
	 * This function retrieves the latest discount status for each discount id passed in the request.
	 * It will return an array of discount statuses, each corresponding to the discount id passed in the request.
	 *
	 * @since 5.13
	 *
	 * @return array The array of discount statuses, each corresponding to the discount id passed in the request.
	 */
	public function get_latest_discount_status(){
		if ( !isset( $_REQUEST['nonce'] ) || !wp_verify_nonce( sanitize_text_field(wp_unslash( $_REQUEST['nonce'] )), 'fetch_discount_status_nonce' ) ) {
			wp_send_json_error( array( 'message' => __('Invalid request.', 'woo-product-category-discount') ), 403 );
			wp_die();
    	}

		if ( !current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __('Unauthorized access.', 'woo-product-category-discount') ), 403 );
			wp_die();
		}

		if( !isset( $_REQUEST['discount_ids'])){
			wp_send_json_error( array( 'message' => __('Discount ids are required.', 'woo-product-category-discount') ), 403 );
			wp_die();
		}
		
		$discounts = wc_clean( $_REQUEST['discount_ids'] );
		$discount_statuses = array();
		
		foreach( $discounts as $discount_id ){
			$discount_data = $this->get_scheduled_discount_data( $discount_id );
			$discount_statuses[$discount_id] = wpcd_get_admin_discount_status_html( $discount_data );
		}

		wp_send_json_success( array( 'discount_statuses' => $discount_statuses ) );
	}
}
