<?php
/**
 * Admin dashboard settings
 *
 * @package    HidePostsPlugin
 */

namespace MartinCV\WHP\Admin;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin dashboard class
 */
class Dashboard {
	use \MartinCV\WHP\Traits\Singleton;

	/**
	 * Initialize class
	 *
	 * @return  void
	 */
	private function initialize() {
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_menu', array( $this, 'menu' ) );
		add_action( 'admin_notices', array( $this, 'migrate_data_notice' ) );
		add_action( 'admin_init', array( $this, 'handle_migration_action' ) );
	}

	/**
	 * Register new submenu and Settings
	 *
	 * @return  void
	 */
	public function menu() {
		add_submenu_page(
			'options-general.php',
			__( 'Hide Posts', 'whp-hide-posts' ),
			__( 'Hide Posts', 'whp-hide-posts' ),
			'administrator',
			'whp-settings',
			array( $this, 'settings' )
		);
	}

	/**
	 * Show the settings form with options
	 *
	 * @return  void
	 */
	public function settings() {
		$post_types                   = get_post_types( array( 'public' => true ), 'object' );
		$enabled_post_types           = whp_plugin()->get_enabled_post_types();
		$whp_disable_hidden_on_column = get_option( 'whp_disable_hidden_on_column', false );

		require_once WHP_PLUGIN_DIR . 'views/admin/template-admin-dashboard.php';
	}

	/**
	 * Register plugin settings
	 *
	 * @return  void
	 */
	public function register_settings() {
		register_setting( 'whp-settings-group', 'whp_enabled_post_types' );
		register_setting( 'whp-settings-group', 'whp_disable_hidden_on_column' );
	}

	/**
	 * Migrate hide posts data from meta to table
	 * Uses transient lock to prevent race conditions
	 *
	 * @return void
	 */
	public function migrate_meta_to_table() {
		// Use a transient lock to prevent concurrent migrations.
		if ( false !== get_transient( 'whp_migration_lock' ) ) {
			return; // Migration already in progress.
		}

		set_transient( 'whp_migration_lock', true, 5 * MINUTE_IN_SECONDS );

		$data_migrated = get_option( 'whp_data_migrated', false );

		if ( $data_migrated ) {
			delete_transient( 'whp_migration_lock' );
			return;
		}

		global $wpdb;

		$table_name = esc_sql( $wpdb->prefix . 'whp_posts_visibility' );

		$table_exists = $wpdb->get_var(
			$wpdb->prepare(
				"SHOW TABLES LIKE %s",
				$table_name
			)
		);

		if ( $wpdb->last_error ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( sprintf( 'WHP: Failed to check table existence: %s', $wpdb->last_error ) );
			delete_transient( 'whp_migration_lock' );
			return;
		}

		if ( $table_exists !== $table_name ) {
			return;
		}

		$meta_keys = array(
			'_whp_hide_on_frontpage'            => 'hide_on_frontpage',
			'_whp_hide_on_blog_page'            => 'hide_on_blog_page',
			'_whp_hide_on_cpt_archive'          => 'hide_on_cpt_archive',
			'_whp_hide_on_categories'           => 'hide_on_categories',
			'_whp_hide_on_search'               => 'hide_on_search',
			'_whp_hide_on_tags'                 => 'hide_on_tags',
			'_whp_hide_on_authors'              => 'hide_on_authors',
			'_whp_hide_on_date'                 => 'hide_on_date',
			'_whp_hide_in_rss_feed'             => 'hide_in_rss_feed',
			'_whp_hide_on_store'                => 'hide_on_store',
			'_whp_hide_on_product_category'     => 'hide_on_product_category',
			'_whp_hide_on_single_post_page'     => 'hide_on_single_post_page',
			'_whp_hide_on_post_navigation'      => 'hide_on_post_navigation',
			'_whp_hide_on_recent_posts'         => 'hide_on_recent_posts',
			'_whp_hide_on_archive'              => 'hide_on_archive',
			'_whp_hide_on_rest_api'             => 'hide_on_rest_api',
			'_whp_hide_on_xml_sitemap'          => 'hide_on_xml_sitemap',
			'_whp_hide_on_yoast_sitemap'        => 'hide_on_yoast_sitemap',
			'_whp_hide_on_yoast_breadcrumbs'    => 'hide_on_yoast_breadcrumbs',
			'_whp_hide_on_yoast_internal_links' => 'hide_on_yoast_internal_links',
		);

		// Use optimized INSERT ... SELECT for bulk migration (faster than individual inserts).
		foreach ( $meta_keys as $meta_key => $condition ) {
			// Use INSERT IGNORE to skip duplicates automatically.
			$sql = $wpdb->prepare(
				"INSERT IGNORE INTO {$table_name} (post_id, `condition`)
				SELECT post_id, %s
				FROM {$wpdb->postmeta}
				WHERE meta_key = %s",
				$condition,
				$meta_key
			);

			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
			$wpdb->query( $sql );

			if ( $wpdb->last_error ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( sprintf( 'WHP: Migration error for %s: %s', $meta_key, $wpdb->last_error ) );
				continue;
			}

			// Delete migrated postmeta entries.
			$wpdb->query(
				$wpdb->prepare(
					"DELETE FROM {$wpdb->postmeta} WHERE meta_key = %s",
					$meta_key
				)
			);

			if ( $wpdb->last_error ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( sprintf( 'WHP: Failed to delete legacy meta %s: %s', $meta_key, $wpdb->last_error ) );
			}
		}

		update_option( 'whp_data_migrated', true );

		// Release the lock.
		delete_transient( 'whp_migration_lock' );
	}

	/**
	 * Check if there's any legacy postmeta data that needs migration
	 *
	 * @return bool True if legacy data exists, false otherwise
	 */
	private function has_legacy_data() {
		global $wpdb;

		// Check if any of the legacy meta keys exist in postmeta table.
		$legacy_meta_keys = array(
			'_whp_hide_on_frontpage',
			'_whp_hide_on_blog_page',
			'_whp_hide_on_cpt_archive',
			'_whp_hide_on_categories',
			'_whp_hide_on_search',
			'_whp_hide_on_tags',
			'_whp_hide_on_authors',
			'_whp_hide_on_date',
			'_whp_hide_in_rss_feed',
			'_whp_hide_on_store',
			'_whp_hide_on_product_category',
			'_whp_hide_on_single_post_page',
			'_whp_hide_on_post_navigation',
			'_whp_hide_on_recent_posts',
			'_whp_hide_on_archive',
			'_whp_hide_on_rest_api',
			'_whp_hide_on_xml_sitemap',
			'_whp_hide_on_yoast_sitemap',
			'_whp_hide_on_yoast_breadcrumbs',
			'_whp_hide_on_yoast_internal_links',
		);

		// Use a single query to check if ANY of these meta keys exist.
		$placeholders = implode( ', ', array_fill( 0, count( $legacy_meta_keys ), '%s' ) );
		$sql          = "SELECT COUNT(*) FROM {$wpdb->postmeta} WHERE meta_key IN ($placeholders) LIMIT 1";

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
		$count = (int) $wpdb->get_var( $wpdb->prepare( $sql, $legacy_meta_keys ) );

		return $count > 0;
	}

	/**
	 * Notice to migrate data
	 *
	 * @return void
	 */
	public function migrate_data_notice() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$data_migrated_notice_closed = get_option( 'whp_data_migrated_notice_closed', false );

		if ( $data_migrated_notice_closed ) {
			return;
		}

		$data_migrated = get_option( 'whp_data_migrated', false );

		if ( $data_migrated ) {
			$action_url = add_query_arg(
				array(
					'action' => 'whp_hide_posts_migration_complete_notice_close',
					'__nonce' => wp_create_nonce( 'whp-hide-posts-migration-complete-nonce' ),
				),
				admin_url()
			);

			echo '<div class="notice notice-success">';
			echo '<p><strong>' . esc_html__( 'Hide Posts Plugin:', 'whp-hide-posts' ) . '</strong> ' . esc_html__( 'Migration Complete.', 'whp-hide-posts' ) . '</p>';
			echo '<p><a href="' . esc_url( $action_url ) . '" class="button button-primary">' . esc_html__( 'Close Notice', 'whp-hide-posts' ) . '</a></p>';
			echo '</div>';
			return;
		}

		// Check if there's actually any legacy data to migrate.
		if ( ! $this->has_legacy_data() ) {
			// No legacy data found, mark as migrated and don't show notice.
			update_option( 'whp_data_migrated', true );
			return;
		}

		$action_url = add_query_arg(
			array(
				'action' => 'whp_hide_posts_migrate_data',
				'__nonce' => wp_create_nonce( 'whp-hide-posts-migrate-data-nonce' ),
			),
			admin_url()
		);

		echo '<div class="notice notice-warning is-dismissible">';
		echo '<p><strong>' . esc_html__( 'Hide Posts Plugin:', 'whp-hide-posts' ) . '</strong> ' . esc_html__( 'Important: We implemented a new table for managing hide flags which optimizes queries and improves overall performance.', 'whp-hide-posts' ) . ' <strong>' . esc_html__( 'Please create a database backup before proceeding, just in case.', 'whp-hide-posts' ) . '</strong></p>';
		echo '<p><a href="' . esc_url( $action_url ) . '" class="button button-primary">' . esc_html__( 'Migrate Hide Post Data', 'whp-hide-posts' ) . '</a></p>';
		echo '</div>';
	}

	/**
	 * Handle the migration action
	 *
	 * @return void
	 */
	public function handle_migration_action() {
		// Check user capability first.
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$data_migrated = get_option( 'whp_data_migrated', false );

		if ( $data_migrated ) {
			$data_migrated_notice_closed = get_option( 'whp_data_migrated_notice_closed', false );

			if ( ! $data_migrated_notice_closed ) {
				if ( ! isset( $_GET['action'] ) || 'whp_hide_posts_migration_complete_notice_close' !== sanitize_text_field( wp_unslash( $_GET['action'] ) ) ) {
					return;
				}

				if ( ! isset( $_GET['__nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['__nonce'] ) ), 'whp-hide-posts-migration-complete-nonce' ) ) {
					return;
				}

				update_option( 'whp_data_migrated_notice_closed', true );

				wp_safe_redirect( remove_query_arg( array( 'action', '__nonce' ) ) );
				exit;
			}

			return;
		}

		if ( ! isset( $_GET['action'] ) || 'whp_hide_posts_migrate_data' !== sanitize_text_field( wp_unslash( $_GET['action'] ) ) ) {
			return;
		}

		if ( ! isset( $_GET['__nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['__nonce'] ) ), 'whp-hide-posts-migrate-data-nonce' ) ) {
			return;
		}

		$this->migrate_meta_to_table();

		wp_safe_redirect( remove_query_arg( array( 'action', '__nonce' ) ) );
		exit;
	}
}
