<?php
/*
Plugin Name: weDocs
Plugin URI: https://wedocs.co/
Description: A documentation plugin for WordPress
Version: 2.1.15
Author: weDevs
Author URI: https://wedocs.co/?utm_source=wporg&utm_medium=banner&utm_campaign=author-uri
License: GPL2
Text Domain: wedocs
Domain Path: /languages
*/

/*
 * Copyright (c) 2021 Tareq Hasan (email: tareq@wedevs.com). All rights reserved.
 *
 * Released under the GPL license
 * http://www.opensource.org/licenses/gpl-license.php
 *
 * This is an add-on for WordPress
 * http://wordpress.org/
 *
 * **********************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * **********************************************************************
 */

// don't call the file directly
use Appsero\Client;
use WeDevs\WeDocs\Capability;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

require_once __DIR__ . '/vendor/autoload.php';
require_once plugin_dir_path(__FILE__) . 'assets/build/blocks/DocsGrid/render.php';

/**
 * WeDocs class.
 *
 * @class WeDocs The class that holds the entire WeDocs plugin
 */
final class WeDocs {


    /**
     * Plugin version.
     *
     * @var string
     */
    const VERSION = '2.1.15';

    /**
     * The plugin url.
     *
     * @var string
     */
    public $plugin_url;

    /**
     * The plugin path.
     *
     * @var string
     */
    public $plugin_path;

    /**
     * The theme directory path.
     *
     * @var string
     */
    public $theme_dir_path;

    /**
     * The post type name.
     *
     * @var string
     */
    private $post_type = 'docs';

    /**
     * Holds various class instances
     *
     * @var array
     */
    private $container = [];

    /**
     * Class constructor.
     */
    private function __construct() {
        $this->define_constants();
        $this->init_actions();

        register_activation_hook( __FILE__, [ $this, 'activate' ] );

        add_action( 'after_setup_theme', [ $this, 'init_classes' ] );

        $this->init_action_scheduler();
    }

    /**
     * Initializes the WeDocs() class.
     *
     * Checks for an existing WeDocs() instance
     * and if it doesn't find one, creates it.
     */
    public static function init() {
        static $instance = false;

        if ( ! $instance ) {
            $instance = new WeDocs();
        }

        return $instance;
    }

    /**
     * Define the required plugin constants
     *
     * @return void
     */
    public function define_constants() {
        define( 'WEDOCS_VERSION', self::VERSION );
        define( 'WEDOCS_FILE', __FILE__ );
        define( 'WEDOCS_PATH', __DIR__ );
        define( 'WEDOCS_URL', plugins_url( '', WEDOCS_FILE ) );
        define( 'WEDOCS_ASSETS', WEDOCS_URL . '/assets' );
        define( 'WEDOCS_PROMO_KEY', 'wedocs_promo_notices' );
        define( 'WEDOCS_PROMO_URL', 'https://raw.githubusercontent.com/weDevsOfficial/wedocs-util/refs/heads/master/promotion.json' );
        define( 'WEDOCS_LOGO_URL', WEDOCS_URL . '/assets/img/wedocs-icon-128x128.png' );
    }

    /**
     * Magic getter to bypass referencing plugin.
     *
     * @param $prop
     *
     * @return mixed
     */
    public function __get( $prop ) {
        if ( array_key_exists( $prop, $this->container ) ) {
            return $this->container[ $prop ];
        }

        return $this->{$prop};
    }

    /**
     * Initialize the plugin.
     *
     * @return void
     */
    public function init_actions() {
        $this->theme_dir_path = apply_filters( 'wedocs_theme_dir_path', 'wedocs/' );

        // Localize our plugin
        add_action( 'init', [ $this, 'localization_setup' ] );
        add_action('init', [$this, 'register_blocks']);

        // registeer our widget
        add_action( 'widgets_init', [ $this, 'register_widget' ] );
    }

    public function register_blocks() {
        // Register the DocsGrid block
        register_block_type(
            plugin_dir_path(__FILE__) . 'assets/build/blocks/DocsGrid',
            array(
                'render_callback' => 'render_wedocs_docs_grid'
            )
        );
    }

    /**
     * The plugin activation function.
     *
     * @since 1.3
     *
     * @return void
     */
    public function activate() {
        $installer = new WeDevs\WeDocs\Installer();
        $installer->run();

        // Set the redirect option to true when the plugin is activated.
        update_option( 'wedocs_activation_redirect', true );
    }

    /**
     * Initialize the classes.
     *
     * @since 1.4
     *
     * @return void
     */
    public function init_classes() {
        $this->container['post_type'] = new WeDevs\WeDocs\Post_Types();

        if ( is_admin() ) {
            $this->container['admin'] = new WeDevs\WeDocs\Admin();
        } else {
            $this->container['frontend'] = new WeDevs\WeDocs\Frontend();
        }

        if ( wp_doing_ajax() ) {
            $this->container['ajax'] = new WeDevs\WeDocs\Ajax();
        }

        $this->container['api']      = new WeDevs\WeDocs\API();
        $this->container['assets']   = new WeDevs\WeDocs\Assets();
        $this->container['migrate']  = new WeDevs\WeDocs\Admin\Migrate();
        $this->container['upgrader'] = new WeDevs\WeDocs\Upgrader\Upgrader();
        $this->container['capability'] = new Capability();
    }

    /**
     * Initialize plugin for localization.
     *
     * @uses load_plugin_textdomain()
     */
    public function localization_setup() {
        load_plugin_textdomain( 'wedocs', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
    }

    /**
     * Register the search widget.
     *
     * @return void
     */
    public function register_widget() {
        register_widget( 'WeDevs\WeDocs\Widget' );
    }

    /**
     * Get the plugin url.
     *
     * @return string
     */
    public function plugin_url() {
        if ( $this->plugin_url ) {
            return $this->plugin_url;
        }

        return $this->plugin_url = untrailingslashit( plugins_url( '/', __FILE__ ) );
    }

    /**
     * Get the plugin path.
     *
     * @return string
     */
    public function plugin_path() {
        if ( $this->plugin_path ) {
            return $this->plugin_path;
        }

        return $this->plugin_path = untrailingslashit( plugin_dir_path( __FILE__ ) );
    }

    /**
     * Initialize action scheduler.
     *
     * @since 2.0.0
     *
     * @return void
     */
    public function init_action_scheduler() {
        require_once( __DIR__ . '/vendor/woocommerce/action-scheduler/action-scheduler.php' );
    }

    /**
     * Get the template path.
     *
     * @return string
     */
    public function template_path() {
        return $this->plugin_path() . '/templates/';
    }

    /**
     * Get the theme directory path.
     *
     * @return string
     */
    public function theme_dir_path() {
        return $this->theme_dir_path;
    }
} // WeDocs

/**
 * Initialize the plugin.
 *
 * @return \WeDocs
 */
function wedocs() {
    return WeDocs::init();
}

// kick it off
wedocs();
