<?php

namespace kirillbdev\WCUkrShipping\Foundation;

use kirillbdev\WCUkrShipping\Component\Carriers\NovaPoshta\ShippingOptionDefaultMapper;
use kirillbdev\WCUkrShipping\Factories\Rates\NovaPoshta\NovaPoshtaCheckoutOrderFactory;
use kirillbdev\WCUkrShipping\Helpers\WCUSHelper;
use kirillbdev\WCUkrShipping\Services\CalculationService;

if ( ! defined('ABSPATH')) {
    exit;
}

class NovaPoshtaShipping extends \WC_Shipping_Method
{
    public function __construct($instance_id = 0)
    {
        parent::__construct($instance_id);

        $this->id = WC_UKR_SHIPPING_NP_SHIPPING_NAME;
        $this->method_title = WC_UKR_SHIPPING_NP_SHIPPING_TITLE;
        $this->method_description = 'Nova Poshta by WC Ukraine Shipping';

        $this->supports = array(
            'shipping-zones',
            'instance-settings',
            'instance-settings-modal',
        );

        $this->init();
    }

    public function __get($name)
    {
        return $this->$name;
    }

    /**
     * Init your settings
     *
     * @access public
     * @return void
     */
    function init()
    {
        $this->init_settings();
        $this->init_form_fields();

        $this->title = $this->get_option('title');

        // @phpstan-ignore-next-line
        add_action('woocommerce_update_options_shipping_' . $this->id, array($this, 'process_admin_options'));
    }

    public function init_form_fields(): void
    {
        $optionMapper = new ShippingOptionDefaultMapper();
        $this->instance_form_fields = [
            'title' => [
                'title' => __('Name', 'woocommerce' ),
                'type' => 'text',
                'description' => '',
                'default' => wc_ukr_shipping_get_option('wc_ukr_shipping_np_method_title'),
            ],
            'delivery_methods' => [
                'title' => __('Delivery types', 'wc-ukr-shipping-i18n'),
                'type' => 'multiselect',
                'options' => [
                    'warehouse' => __('To warehouse', 'wc-ukr-shipping-i18n'),
                    'doors'  => __('By courier', 'wc-ukr-shipping-i18n'),
                    'poshtomat'  => __('To poshtomat', 'wc-ukr-shipping-i18n'),
                ],
                'default' => $optionMapper->getDeliveryMethodsDefault(),
                'desc_tip' => true,
                'description' => __('You can choose which types of delivery will be processed by this shipping method or create several shipping methods with concrete delivery types separately.', 'wc-ukr-shipping-i18n'),
                'sanitize_callback' => [$this, 'sanitizeDeliveryMethods'],
            ],
            'combine_poshtomats' => [
                'label' => __('Combine poshtomats and warehouses', 'wc-ukr-shipping-i18n'),
                'type' => 'checkbox',
                'default' =>  (int)wc_ukr_shipping_get_option('wcus_combine_poshtomats') === 1 ? 'yes' : 'no',
                'desc_tip' => true,
            ],
            'cost_calculation_type' => [
                'title' => __('Type of shipping cost calculation', 'wc-ukr-shipping-i18n'),
                'type' => 'select',
                'description' => __('Access to the SmartyParcel Rates API is only available on paid plans', 'wc-ukr-shipping-i18n'),
                'desc_tip' => true,
                'class' => 'j-wcus-shipping-calc-type',
                'options' => [
                    'fixed' => __('Fixed', 'wc-ukr-shipping-i18n'),
                    'rates_api'  => __('SmartyParcel Rates API', 'wc-ukr-shipping-i18n'),
                ],
                'default' => $optionMapper->getShippingCalculationTypeDefault(),
            ],
            'fixed_cost' => [
                'title' => __('Fixed shipping cost', 'wc-ukr-shipping-i18n'),
                'type' => 'text',
                'default' =>  (float)wc_ukr_shipping_get_option('wc_ukr_shipping_np_price'),
                'desc_tip' => true,
                'class' => 'j-wcus-shipping-fixed-cost',
                'sanitize_callback' => [$this, 'sanitizePrice'],
            ],
            'include_cod' => [
                'label' => __('Include COD when calculating shipping cost', 'wc-ukr-shipping-i18n'),
                'type' => 'checkbox',
                'description' => __('If checked and function supported by carrier, the COD service will be included in shipping cost.', 'wc-ukr-shipping-i18n'),
                'default' =>  (int)wc_ukr_shipping_get_option('wcus_cod_payment_active') === 1 ? 'yes' : 'no',
                'desc_tip' => true,
                'class' => 'j-wcus-shipping-option-cod',
            ],
            'add_cost_to_order' => [
                'label' => __('Include shipping cost to order total', 'wc-ukr-shipping-i18n'),
                'type' => 'checkbox',
                'description' => __('If checked, the shipping cost will be added to the order total.', 'wc-ukr-shipping-i18n'),
                'default' =>  (int)wc_ukr_shipping_get_option('wcus_cost_view_only') === 1 ? 'no' : 'yes',
                'desc_tip' => true,
            ],
            'enable_free_shipping' => [
                'label' => __('Enable free shipping rule', 'wc-ukr-shipping-i18n'),
                'type' => 'checkbox',
                'description' => __('If checked, free shipping would be available based on order amount.', 'wc-ukr-shipping-i18n'),
                'default' => $optionMapper->getFreeShippingConditionDefault(),
                'desc_tip' => true,
            ],
            'free_shipping_min_amount' => [
                'title' => __('Minimum amount for free shipping', 'wc-ukr-shipping-i18n'),
                'type' => 'text',
                'default' => $optionMapper->getFreeShippingMinAmountDefault(),
                'desc_tip' => true,
                'sanitize_callback' => [$this, 'sanitizePrice'],
            ],
            'free_shipping_title' => [
                'title' => __('Free shipping method name', 'wc-ukr-shipping-i18n'),
                'type' => 'text',
                'default' => __('Free shipping', 'wc-ukr-shipping-i18n'),
            ],
            'options_message' => [
                'type' => 'wcus_message',
                'text' => __('Additional settings also available at', 'wc-ukr-shipping-i18n'),
                'link' => home_url('wp-admin/admin.php?page=wc_ukr_shipping_options&section=nova_poshta'),
            ],
        ];
    }

    public function generate_wcus_message_html(string $key, array $data): string
    {
        return sprintf(
            '<div style="margin-bottom: 16px;">%s <a href="%s" target="_blank">%s</a></div>',
            esc_html($data['text']),
            esc_attr($data['link']),
            __('plugin settings page', 'wc-ukr-shipping-i18n')
        );
    }

    public function calculate_shipping($package = []): void
    {
        if ( ! $this->shouldCalculated()) {
            $this->add_rate([
                'label' => $this->title,
                'cost' => 0,
                'package' => $package,
            ]);
            return;
        }

        $service = new CalculationService();
        $factory = new NovaPoshtaCheckoutOrderFactory();
        $orderInfo = $factory->createOrderInfo();
        $orderInfo->setShippingMethodInstance($this);
        $shippingCost = $service->calculateRates($orderInfo);

        $rate = [
            'label' => $this->get_option('enable_free_shipping') === 'yes' && $shippingCost <= 0
                ? $this->get_option('free_shipping_title')
                : $this->title,
            'cost' => $shippingCost,
            'package' => $package,
        ];
        $this->add_rate($rate);
    }

    /**
     * Is this method available?
     * @param array $package
     * @return bool
     */
    public function is_available($package)
    {
        return $this->is_enabled();
    }

    /**
     * @param mixed $value
     * @return mixed
     */
    public function sanitizePrice($value)
    {
        if ($value !== '' && !is_numeric($value)) {
            throw new \InvalidArgumentException('Field value must be an integer or a float');
        }

        return $value;
    }

    /**
     * @param mixed $value
     * @return mixed
     */
    public function sanitizeDeliveryMethods($value)
    {
        if (!is_array($value) || count($value) < 1) {
            throw new \InvalidArgumentException('You should select at least one delivery method');
        }

        return $value;
    }

    private function shouldCalculated(): bool
    {
        if ( ! isset($_GET['wc-ajax'])) {
            return false;
        }

        if (!WCUSHelper::hasChosenShippingMethodInstance($this)) {
            return false;
        }

        return ($_GET['wc-ajax'] === 'update_order_review' && ! empty($_POST['post_data']))
            || $_GET['wc-ajax'] === 'checkout';
    }
}
