<?php

declare(strict_types=1);

namespace kirillbdev\WCUkrShipping\Factories\Rates\NovaPoshta;

use kirillbdev\WCUkrShipping\Component\Rates\ApiRatesCalculator;
use kirillbdev\WCUkrShipping\Component\Rates\FixedRatesCalculator;
use kirillbdev\WCUkrShipping\Component\Rates\FreeShippingRatesCalculator;
use kirillbdev\WCUkrShipping\Contracts\Rates\RatesCalculatorInterface;
use kirillbdev\WCUkrShipping\Dto\Rates\OrderInfoDto;
use kirillbdev\WCUkrShipping\Factories\Rates\RatesCalculatorFactoryInterface;
use kirillbdev\WCUkrShipping\Services\SmartyParcelService;

class NovaPoshtaRatesCalculatorFactory implements RatesCalculatorFactoryInterface
{
    private SmartyParcelService $smartyParcelService;

    public function __construct(SmartyParcelService $smartyParcelService)
    {
        $this->smartyParcelService = $smartyParcelService;
    }

    public function getRatesCalculator(OrderInfoDto $orderInfo): RatesCalculatorInterface
    {
        if ($orderInfo->shippingMethodInstance === null) {
            throw new \LogicException('Shipping method instance not set');
        }

        $calculator = $this->getRealRatesCalculator($orderInfo);
        if ($orderInfo->shippingMethodInstance->get_option('enable_free_shipping') === 'yes') {
            return new FreeShippingRatesCalculator(
                $calculator,
                (float)$orderInfo->shippingMethodInstance->get_option('free_shipping_min_amount')
            );
        } else {
            return $calculator;
        }
    }

    private function getRealRatesCalculator(OrderInfoDto $orderInfo): RatesCalculatorInterface
    {
        $calcType = $orderInfo->shippingMethodInstance->get_option('cost_calculation_type');
        switch ($calcType) {
            case 'fixed':
                return new FixedRatesCalculator(
                    (float)$orderInfo->shippingMethodInstance->get_option('fixed_cost')
                );
            case 'rates_api':
                return new ApiRatesCalculator(
                    $this->smartyParcelService,
                    (string)wc_ukr_shipping_get_option('wc_ukr_shipping_np_sender_city'),
                    (string)get_option('wcus_nova_poshta_default_carrier'),
                    $orderInfo->shippingMethodInstance->get_option('include_cod') === 'yes'
                        && wc_ukr_shipping_get_option('wcus_cod_payment_id') === $orderInfo->paymentMethod
                );
            default:
                throw new \InvalidArgumentException('Unsupported calculation type ' . $calcType);
        }
    }
}
