<?php
/**
 * Plugin Name: GSheetConnector for WC
 * Plugin URI: https://wordpress.org/plugins/wc-gsheetconnector/
 * Description: Send your WooCommerce data to your Google Sheets spreadsheet.
 * Author: GSheetConnector
 * Author URI: https://www.gsheetconnector.com/
 * Version: 1.4.6
 * Text Domain: wc-gsheetconnector
 * Domain Path:  /languages
 * WooCommerce requires at least: 3.2.0
 * Requires Plugins: woocommerce
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

if (wc_gsheetconnector_Init::gscwoo_is_pugin_active('wc_gsheetconnector_Init_Pro')) {
    return;
}

/*freemius*/
if (function_exists('is_plugin_active') && is_plugin_active('wc-gsheetconnector/wc-gsheetconnector.php')) {
    if (!function_exists('gs_woofree')) {
        // Create a helper function for easy SDK access.
        function gs_woofree()
        {
            global $gs_woofree;

            if (!isset($gs_woofree)) {
                // Activate multisite network integration.
                if (!defined('WP_FS__PRODUCT_9480_MULTISITE')) {
                    define('WP_FS__PRODUCT_9480_MULTISITE', true);
                }

                // Include Freemius SDK.
                require_once dirname(__FILE__) . '/lib/vendor/freemius/start.php';

                $gs_woofree = fs_dynamic_init(array(
                    'id' => '9480',
                    'slug' => 'wc-gsheetconnector',
                    'type' => 'plugin',
                    'public_key' => 'pk_487f703ba4a974974c9d344111193',
                    'is_premium' => false,
                    'has_addons' => false,
                    'has_paid_plans' => false,
                    'is_org_compliant' => true, 
                    'menu' => array(
                        'slug' => 'wc-gsheetconnector-config',
                        'first-path' => (!is_multisite() ? 'admin.php?page=wc-gsheetconnector-config' : 'plugins.php'),
                        'account' => false,
                    ),
                ));
            }

            return $gs_woofree;
        }

        // Init Freemius.
        gs_woofree();
        // Signal that SDK was initiated.
        do_action('gs_woofree_loaded');
    }
}
/*freemius*/

// Declare some global constants
define('WC_GSHEETCONNECTOR_VERSION', '1.4.6');
define('WC_GSHEETCONNECTOR_DB_VERSION', '1.4.6');
define('WC_GSHEETCONNECTOR_ROOT', dirname(__FILE__));
define('WC_GSHEETCONNECTOR_URL', plugins_url('/', __FILE__));
define('WC_GSHEETCONNECTOR_BASE_FILE', basename(dirname(__FILE__)) . '/wc-gsheetconnector.php');
define('WC_GSHEETCONNECTOR_BASE_NAME', plugin_basename(__FILE__));
define('WC_GSHEETCONNECTOR_PATH', plugin_dir_path(__FILE__)); //use for include files to other files
define('WC_GSHEETCONNECTOR_CURRENT_THEME', get_stylesheet_directory());
define('WC_GSHEETCONNECTOR_API_URL', 'https://oauth.gsheetconnector.com/api-cred.php');

load_plugin_textdomain('wc-gsheetconnector', false, basename(dirname(__FILE__)) . '/languages');

/*
 * include utility classes
 */
if (!class_exists('wc_gsheetconnector_utility')) {
    include(WC_GSHEETCONNECTOR_ROOT . '/includes/class-wc-gsheetconnector-utility.php');
}
//Include Library Files
require_once WC_GSHEETCONNECTOR_ROOT . '/lib/vendor/autoload.php';

include_once(WC_GSHEETCONNECTOR_ROOT . '/lib/google-sheets.php');

if (!class_exists('wc_gsheetconnector_Service')) {
    include_once(WC_GSHEETCONNECTOR_PATH . 'includes/class-wc-gsheetconnector-services.php');
}

class wc_gsheetconnector_Init
{

    /**
     *  Set things up.
     *  @since 1.0
     */
    public function __construct()
    {

        //run on activation of plugin
        register_activation_hook(__FILE__, array($this, 'wcgsc_activate'));

        //run on deactivation of plugin
        register_deactivation_hook(__FILE__, array($this, 'wcgsc_deactivate'));

        //run on uninstall
        register_uninstall_hook(__FILE__, array('wc_gsheetconnector_Init', 'wcgsc_free_uninstall'));

        // clear debug logs method using ajax for system status tab
        add_action('wp_ajax_wcgsc_clear_debug_logs', array($this, 'wcgsc_clear_debug_logs'));

        // validate is woocommerce plugin exist
        add_action('admin_init', array($this, 'validate_parent_plugin_exists'));

        // register admin menu under "Contact" > "Integration"
        add_action('admin_menu', array($this, 'register_gs_menu_pages'), 70);

        // load the js and css files
        add_action('init', array($this, 'load_css_and_js_files'));

        // load the classes
        add_action('init', array($this, 'load_all_classes'));

        add_filter('plugin_row_meta', [$this, 'plugin_row_meta'], 10, 2);
        // run upgradation
        add_action('admin_init', array($this, 'run_on_upgrade'));


    }


    /**
     * Plugin row meta.
     *
     * Adds row meta links to the plugin list table
     *
     * Fired by `plugin_row_meta` filter.
     *
     * @since 1.1.4
     * @access public
     *
     * @param array  $plugin_meta An array of the plugin's metadata, including
     *                            the version, author, author URI, and plugin URI.
     * @param string $plugin_file Path to the plugin file, relative to the plugins
     *                            directory.
     *
     * @return array An array of plugin row meta links.
     */
    public function plugin_row_meta($plugin_meta, $plugin_file)
    {
        if (WC_GSHEETCONNECTOR_BASE_NAME === $plugin_file) {
            $row_meta = [
                'docs' => '<a href="https://support.gsheetconnector.com/kb-category/woocommerce-gsheetconnector" aria-label="' . esc_attr(esc_html__('View Documentation', 'wc-gsheetconnector')) . '" target="_blank">' . esc_html__('Docs', 'wc-gsheetconnector') . '</a>',
                'ideo' => '<a href="https://www.gsheetconnector.com/support" aria-label="' . esc_attr(esc_html__('Get Support', 'wc-gsheetconnector')) . '" target="_blank">' . esc_html__('Support', 'wc-gsheetconnector') . '</a>',
            ];

            $plugin_meta = array_merge($plugin_meta, $row_meta);
        }

        return $plugin_meta;
    }

    /**
     * Do things on plugin activation
     * @since 1.0
     */
    public function wcgsc_activate($network_wide)
    {
        global $wpdb;
        $this->run_on_activation();

        if (function_exists('is_multisite') && is_multisite()) {
            if ($network_wide) {
                $sites = get_sites(array('fields' => 'ids'));
                foreach ($sites as $blog_id) {
                    switch_to_blog($blog_id);
                    $this->run_for_site();
                    restore_current_blog();
                }
                return;
            }
        }

        $this->run_for_site();
    }


    /**
     * Called on activation.
     * Creates the site_options (required for all the sites in a multi-site setup)
     * If the current version doesn't match the new version, runs the upgrade
     * @since 1.0
     */
    private function run_on_activation()
    {
        try {
            $plugin_options = get_site_option('WC_GS_info');

            if (false === $plugin_options) {
                $Wc_GS_info = array(
                    'version' => WC_GSHEETCONNECTOR_VERSION,
                    'db_version' => WC_GSHEETCONNECTOR_DB_VERSION
                );
                update_site_option('WC_GS_info', $Wc_GS_info);
            } else if (WC_GSHEETCONNECTOR_DB_VERSION != $plugin_options['version']) {
                $this->run_on_upgrade();
            }
           // Fetch and save the API credentails.
           wc_gsheetconnector_utility::instance()->save_api_credentials();

          } catch (Exception $e) {
            
        }
    }

    /**
     * called on upgrade. 
     * checks the current version and applies the necessary upgrades from that version onwards
     * @since 1.0
     */
    public function run_on_upgrade()
    {
        $plugin_options = get_site_option('WC_GS_info');
        if ($plugin_options['version'] == '1.3.18') {
            $this->upgrade_database_18();
        }

        // update the version value
        $google_sheet_info = array(
            'version'     => WC_GSHEETCONNECTOR_VERSION,
            'db_version'  => WC_GSHEETCONNECTOR_DB_VERSION
        );

        update_site_option('WC_GS_info', $google_sheet_info);
    }

    public function upgrade_database_18()
    {
        // look through each of the blogs and upgrade the DB
        if (function_exists('is_multisite') && is_multisite()) {
            // Use core function to get all blog IDs (cached and safe)
            $blog_ids = get_sites(array('fields' => 'ids'));

            foreach ($blog_ids as $blog_id) {
                switch_to_blog($blog_id);
                $this->upgrade_helper_18();
                restore_current_blog();
            }
        }

        // Run on current site (non-multisite or base site)
        $this->upgrade_helper_18();
    }

    public function upgrade_helper_18()
    {
        // Fetch and save the API credentails.
        wc_gsheetconnector_utility::instance()->save_api_credentials();
    }

   
    /**
     * deactivate the plugin
     * @since 1.0
     */
    public function wcgsc_deactivate($network_wide)
    {

    }

    /**
     *  Runs on plugin uninstall.
     *  a static class method or function can be used in an uninstall hook
     *
     *  @since 1.0
     */
    public static function wcgsc_free_uninstall()
    {
        wc_gsheetconnector_Init::run_on_uninstall();

        if (function_exists('is_multisite') && is_multisite()) {
            // Use core WordPress function to safely get all blog IDs
            $blog_ids = get_sites(array('fields' => 'ids'));

            foreach ($blog_ids as $blog_id) {
                switch_to_blog($blog_id);
                wc_gsheetconnector_Init::delete_for_site();
                restore_current_blog();
            }
            return;
        }

        wc_gsheetconnector_Init::delete_for_site();
    }


    /**
     * Called on uninstall - deletes site_options
     *
     * @since 1.5
     */
    private static function run_on_uninstall()
    {
        if (!defined('ABSPATH') && !defined('WP_UNINSTALL_PLUGIN'))
            exit();

        delete_site_option('WC_GS_info');
    }

    /**
     * Called on uninstall - deletes site specific options
     *
     * @since 1.0
     */
    private static function delete_for_site()
    {
        delete_option('wcgsc_access_code');
        delete_option('wcgsc_verify');
        delete_option('wcgsc_token');
        delete_option('wcgsc_feeds');
        delete_option('wcgsc_sheetId');
        delete_post_meta_by_key('wcgsc_settings');
    }

    /**
     * Validate parent Plugin WooCommerce exist and activated
     * @access public
     * @since 1.0
     */
    public function validate_parent_plugin_exists()
    {
        $plugin = plugin_basename(__FILE__);

        if ((!is_plugin_active('woocommerce/woocommerce.php')) || (!file_exists(plugin_dir_path(__DIR__) . 'woocommerce/woocommerce.php'))) {
            add_action('admin_notices', array($this, 'wc_gsheetconnector_missing_notice'));
            add_action('network_admin_notices', array($this, 'wc_gsheetconnector_missing_notice'));

            deactivate_plugins($plugin);

            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Activation context, handled by WordPress core
            if (isset($_GET['activate']) && $_GET['activate'] === 'true') {
                unset($_GET['activate']);
            }

            // Optional: redirect safely
            // wp_safe_redirect(admin_url('plugins.php'));
            // exit;
        }
    }


    /**
     * If WooCommerce plugin is not installed or activated then throw the error
     *
     * @access public
     * @return mixed error_message, an array containing the error message
     *
     * @since 1.0 initial version
     */
    public function wc_gsheetconnector_missing_notice()
    {
        $plugin_error = wc_gsheetconnector_utility::instance()->admin_notice(array(
            'type' => 'error',
            'message' => __('GSheetConnector WooCommerce Add-on requires WooCommerce plugin to be installed and activated.', 'wc-gsheetconnector')
        ));

        echo wp_kses_post( $plugin_error );
    }

    /**
     * Called on activation.
     * Creates the options and DB (required by per site)
     * @since 1.0
     */
    // phpcs:disable WordPress.Security.NonceVerification.Recommended -- This function does not process user input.
    private function run_for_site()
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Not processing user input, just setting defaults.
        if (!get_option('wcgsc_access_code')) {
            update_option('wcgsc_access_code', '');
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if (!get_option('wcgsc_verify')) {
            update_option('wcgsc_verify', 'invalid');
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if (!get_option('wcgsc_token')) {
            update_option('wcgsc_token', '');
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if (!get_option('wcgsc_feeds')) {
            update_option('wcgsc_feeds', '');
        }
        if (!get_option('wcgsc_sheetId')) {
            update_option('wcgsc_sheetId', '');
        }
        if (!get_option('wcgsc_settings')) {
            update_option('wcgsc_settings', '');
        }
        if (!get_option('wcgsc_checkbox_settings')) {
            update_option('wcgsc_checkbox_settings', array());
        }
        if (!get_option('wcgsc_tab_roles_setting')) {
            update_option("wcgsc_tab_roles_setting", array());
        }
    }
    // phpcs:enable


    public function load_css_and_js_files()
    {
        add_action('admin_print_styles', array($this, 'add_css_files'));
        add_action('admin_print_scripts', array($this, 'add_js_files'));
    }

    /**
     * enqueue CSS files
     * @since 1.0
     */
    public function add_css_files()
    {
        if (is_admin() && (
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Safe usage for enqueue logic based on admin page check
            isset($_GET['page']) && $_GET['page'] == 'wc-gsheetconnector-config'
        )) {
            wp_enqueue_style(
                'gs-woocommerce-connector-css',
                WC_GSHEETCONNECTOR_URL . 'assets/css/gs-woocommerce-connector.css',
                [],
                WC_GSHEETCONNECTOR_VERSION,
                'all'
            );
			wp_enqueue_style(
                'gs-fontawesome-css',
                WC_GSHEETCONNECTOR_URL . 'assets/css/fontawesome.css',
                [],
                WC_GSHEETCONNECTOR_VERSION,
                'all'
            );
            wp_enqueue_style(
                'wc-gsheetconnector-debug-css',
                WC_GSHEETCONNECTOR_URL . 'assets/css/system-debug.css',
                [],
                WC_GSHEETCONNECTOR_VERSION,
                'all'
            );
        }

    }

    /**
     * enqueue JS files
     * @since 1.0
     */
    public function add_js_files()
    {
        if (is_admin() &&
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Safe usage for enqueue logic based on admin page check
            isset($_GET['page']) && $_GET['page'] === 'wc-gsheetconnector-config'
        ) {
            wp_enqueue_script(
                'gs-connector-js',
                WC_GSHEETCONNECTOR_URL . 'assets/js/gs-connector.js',
                array('jquery'),
                WC_GSHEETCONNECTOR_VERSION,
                true
            );

            wp_enqueue_script(
                'wc-gsheetconnector-popup-js',
                WC_GSHEETCONNECTOR_URL . 'assets/js/wc-gsheet-popup.js',
                array('jquery'),
                WC_GSHEETCONNECTOR_VERSION,
                true
            );

            wp_enqueue_script(
                'wc-gsheetconnector-debug-js',
                WC_GSHEETCONNECTOR_URL . 'assets/js/system-debug.js',
                array('jquery'),
                WC_GSHEETCONNECTOR_VERSION,
                true
            );
        }
    }



    /**
     * Create/Register menu items for the plugin.
     * @since 1.0
     */
    public function register_gs_menu_pages()
    {
        $current_role = wc_gsheetconnector_utility::instance()->get_current_user_role();
        $gs_woo_roles = get_option('wcgsc_page_roles_setting');

        if ($current_role === "administrator" || is_super_admin()) {
            // if (( is_array($gs_woo_roles) && array_key_exists($current_role, $gs_woo_roles) ) || $current_role === "administrator" || is_super_admin()) {
            add_submenu_page('woocommerce', 'Google Sheets', 'Google Sheets', 'manage_options', 'wc-gsheetconnector-config', array($this, 'google_sheet_configuration'));
        }

    }

    /**
     * Google Sheets page action.
     * This method is called when the menu item "Google Sheets" is clicked.
     * @since 1.0
     */
    public function google_sheet_configuration()
    {
        include(WC_GSHEETCONNECTOR_PATH . "includes/pages/google-sheet-settings.php");
    }

    /**
     * Load all the classes - as part of init action hook
     * @since 1.0
     */
    public function load_all_classes()
    {
        if (!class_exists('GS_Processes')) {
            include(WC_GSHEETCONNECTOR_PATH . 'includes/class-wc-gsheetconnector-processes.php');
        }
        if (!class_exists('wc_gsheetconnector_role_settings_free')) {
            include(WC_GSHEETCONNECTOR_PATH . 'includes/class-wc-gsheetconnector-role-settings-free.php');
        }
    }

    /**
     * Add custom link for the plugin beside activate/deactivate links
     * @param array $links Array of links to display below our plugin listing.
     * @return array Amended array of links.    * 
     * @since 1.5
     */
    public function wc_gsheet_setting_link($links)
    {
        // We shouldn't encourage editing our plugin directly.
        unset($links['edit']);

        // Add our custom links to the returned array value.
        return array_merge(array(
            '<a href="' . admin_url('admin.php?page=wc-gsheetconnector-config') . '">' . __('Settings', 'wc-gsheetconnector') . '</a>',
            '<a class="upgradeProSet" style="color: red;font-weight: 600;font-style: italic;" href="https://www.gsheetconnector.com/woocommerce-google-sheet-connector-pro?gsheetconnector-ref=17"  target="__blank">' . __('Upgrade to PRO', 'wc-gsheetconnector') . '</a>',
        ), $links);
    }

    /**
     * Add function to check plugins is Activate or not
     * @param string $class of plugins main class .
     * @return true/false    * 
     * @since 2.0.2
     */
    public static function gscwoo_is_pugin_active($class)
    {
        if (class_exists($class)) {
            return true;
        }
        return false;
    }

    /**
     * Build System Information String
     * @global object $wpdb
     * @return string
     * @since 1.2
     */
    public function get_wcfree_system_info()
    {
        global $wpdb;

        // Get WordPress version
        $wp_version = get_bloginfo('version');

        // Get theme info
        $theme_data = wp_get_theme();
        $theme_name_version = $theme_data->get('Name') . ' ' . $theme_data->get('Version');
        $parent_theme = $theme_data->get('Template');

        if (!empty($parent_theme)) {
            $parent_theme_data = wp_get_theme($parent_theme);
            $parent_theme_name_version = $parent_theme_data->get('Name') . ' ' . $parent_theme_data->get('Version');
        } else {
            $parent_theme_name_version = 'N/A';
        }

        // Check plugin version and subscription plan
        $plugin_version = defined('WC_GSHEETCONNECTOR_VERSION') ? WC_GSHEETCONNECTOR_VERSION : 'N/A';
        $subscription_plan = 'FREE';
      
        $plugin_name = 'GSheetConnector for WC';


        // Check Google Account Authentication
        $api_token_auto = get_option('wcgsc_token');

        if (!empty($api_token_auto)) {
            // The user is authenticated through the auto method
            $google_sheet_auto = new GSCWOO_googlesheet();
            $email_account_auto = $google_sheet_auto->gsheet_print_google_account_email();
            $connected_email = !empty($email_account_auto) ? esc_html($email_account_auto) : 'Not Auth';
        } else {
            // Auto authentication is the only method available
            $connected_email = 'Not Auth';
        }

        // Check Google Permission
        $gs_verify_status = get_option('wcgsc_verify');
        $search_permission = ($gs_verify_status === 'valid') ? 'Given' : 'Not Given';

        // Create the system info HTML
        $system_info = '<div class="wcgsc-system-status">';
        $system_info .= '<h4><button id="show-info-button" class="info-button">GSheetConnector<span class="dashicons dashicons-arrow-down"></span></h4>';
        $system_info .= '<div id="info-container" class="info-content" style="display:none;">';
        $system_info .= '<h3>GSheetConnector</h3>';
        $system_info .= '<table>';
         $system_info .= '<tr><td>Plugin Name</td><td>' . esc_html($plugin_name) . '</td></tr>';
        $system_info .= '<tr><td>Plugin Version</td><td>' . esc_html($plugin_version) . '</td></tr>';
        $system_info .= '<tr><td>Plugin Subscription Plan</td><td>' . esc_html($subscription_plan) . '</td></tr>';
        $system_info .= '<tr><td>Connected Email Account</td><td>' . $connected_email . '</td></tr>';

        $gscpclass = 'wcgsc-permission-notgiven';
        if ($search_permission == "Given") {
            $gscpclass = 'wcgsc-permission-given';
        }
        $system_info .= '<tr><td>Google Drive Permission</td><td class="' . $gscpclass . '">' . esc_html($search_permission) . '</td></tr>';

        $system_info .= '<tr><td>Google Sheet Permission</td><td class="' . $gscpclass . '">' . esc_html($search_permission) . '</td></tr>';
        $system_info .= '</table>';
        $system_info .= '</div>';
        // Add WordPress info
        // Create a button for WordPress info
        $system_info .= '<h2><button id="show-wordpress-info-button" class="info-button">WordPress Info<span class="dashicons dashicons-arrow-down"></span></h2>';
        $system_info .= '<div id="wordpress-info-container" class="info-content" style="display:none;">';
        $system_info .= '<h3>WordPress Info</h3>';
        $system_info .= '<table>';

        $system_info .= '<tr><td>Version</td><td>' . esc_html(get_bloginfo('version')) . '</td></tr>';
        $system_info .= '<tr><td>Site Language</td><td>' . esc_html(get_bloginfo('language')) . '</td></tr>';
        $system_info .= '<tr><td>Debug Mode</td><td>' . (WP_DEBUG ? 'Enabled' : 'Disabled') . '</td></tr>';
        $system_info .= '<tr><td>Home URL</td><td>' . esc_url(get_home_url()) . '</td></tr>';
        $system_info .= '<tr><td>Site URL</td><td>' . esc_url(get_site_url()) . '</td></tr>';
        $system_info .= '<tr><td>Permalink structure</td><td>' . esc_html(get_option('permalink_structure')) . '</td></tr>';
        $system_info .= '<tr><td>Is this site using HTTPS?</td><td>' . (is_ssl() ? 'Yes' : 'No') . '</td></tr>';
        $system_info .= '<tr><td>Is this a multisite?</td><td>' . (is_multisite() ? 'Yes' : 'No') . '</td></tr>';
        $system_info .= '<tr><td>Can anyone register on this site?</td><td>' . (get_option('users_can_register') ? 'Yes' : 'No') . '</td></tr>';
        $system_info .= '<tr><td>Is this site discouraging search engines?</td><td>' . (get_option('blog_public') ? 'No' : 'Yes') . '</td></tr>';
        $system_info .= '<tr><td>Default comment status</td><td>' . esc_html(get_option('default_comment_status')) . '</td></tr>';

        // Validate and sanitize $_SERVER['REMOTE_ADDR']
        $server_ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '';
        $environment_type = ( $server_ip === '127.0.0.1' || $server_ip === '::1' ) ? 'localhost' : 'production';
        $system_info .= '<tr><td>Environment type</td><td>' . esc_html($environment_type) . '</td></tr>';

        // User count
        $user_count = count_users();
        $total_users = $user_count['total_users'];
        $system_info .= '<tr><td>User Count</td><td>' . esc_html($total_users) . '</td></tr>';

        // Safe fallback for blog_publicize option
        $system_info .= '<tr><td>Communication with WordPress.org</td><td>' . (get_option('blog_publicize') ? 'Yes' : 'No') . '</td></tr>';

        // Validate and sanitize $_SERVER['SERVER_SOFTWARE']
        $server_software = isset($_SERVER['SERVER_SOFTWARE']) ? sanitize_text_field(wp_unslash($_SERVER['SERVER_SOFTWARE'])) : 'Unavailable';
        $system_info .= '<tr><td>Web Server</td><td>' . esc_html($server_software) . '</td></tr>';

        $system_info .= '</table>';
        $system_info .= '</div>';

        // info about active theme
        $active_theme = wp_get_theme();

        $system_info .= '<h2><button id="show-active-info-button" class="info-button">Active Theme<span class="dashicons dashicons-arrow-down"></span></h2>';
        $system_info .= '<div id="active-info-container" class="info-content" style="display:none;">';
        $system_info .= '<h3>Active Theme</h3>';
        $system_info .= '<table>';
        $system_info .= '<tr><td>Name</td><td>' . $active_theme->get('Name') . '</td></tr>';
        $system_info .= '<tr><td>Version</td><td>' . $active_theme->get('Version') . '</td></tr>';
        $system_info .= '<tr><td>Author</td><td>' . $active_theme->get('Author') . '</td></tr>';
        $system_info .= '<tr><td>Author website</td><td>' . $active_theme->get('AuthorURI') . '</td></tr>';
        $system_info .= '<tr><td>Theme directory location</td><td>' . $active_theme->get_template_directory() . '</td></tr>';
        $system_info .= '</table>';
        $system_info .= '</div>';

        // Get a list of other plugins you want to check compatibility with
        $other_plugins = array(
            'plugin-folder/plugin-file.php', // Replace with the actual plugin slug
            // Add more plugins as needed
        );

        // Network Active Plugins
        if (is_multisite()) {
            $network_active_plugins = get_site_option('active_sitewide_plugins', array());
            if (!empty($network_active_plugins)) {
                $system_info .= '<h2><button id="show-netplug-info-button" class="info-button">Network Active plugins<span class="dashicons dashicons-arrow-down"></span></h2>';
                $system_info .= '<div id="netplug-info-container" class="info-content" style="display:none;">';
                $system_info .= '<h3>Network Active plugins</h3>';
                $system_info .= '<table>';
                foreach ($network_active_plugins as $plugin => $plugin_data) {
                    $plugin_data = get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin);
                    $system_info .= '<tr><td>' . $plugin_data['Name'] . '</td><td>' . $plugin_data['Version'] . '</td></tr>';
                }
                // Add more network active plugin statuses here...
                $system_info .= '</table>';
                $system_info .= '</div>';
            }
        }
        // Active plugins
        $system_info .= '<h2><button id="show-acplug-info-button" class="info-button">Active plugins<span class="dashicons dashicons-arrow-down"></span></h2>';
        $system_info .= '<div id="acplug-info-container" class="info-content" style="display:none;">';
        $system_info .= '<h3>Active plugins</h3>';
        $system_info .= '<table>';

        // Retrieve all active plugins data
        $active_plugins_data = array();
        $active_plugins = get_option('active_plugins', array());
        foreach ($active_plugins as $plugin) {
            $plugin_data = get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin);
            $active_plugins_data[$plugin] = array(
                'name' => $plugin_data['Name'],
                'version' => $plugin_data['Version'],
                'count' => 0, // Initialize the count to zero
            );
        }

        // Count the number of active installations for each plugin
        $all_plugins = get_plugins();
        foreach ($all_plugins as $plugin_file => $plugin_data) {
            if (array_key_exists($plugin_file, $active_plugins_data)) {
                $active_plugins_data[$plugin_file]['count']++;
            }
        }

        // Sort plugins based on the number of active installations (descending order)
        uasort($active_plugins_data, function ($a, $b) {
            return $b['count'] - $a['count'];
        });

        // Display the top 5 most used plugins
        $counter = 0;
        foreach ($active_plugins_data as $plugin_data) {
            $system_info .= '<tr><td>' . $plugin_data['name'] . '</td><td>' . $plugin_data['version'] . '</td></tr>';
            // $counter++;
            // if ($counter >= 5) {
            //     break;
            // }
        }
        $system_info .= '</table>';
        $system_info .= '</div>';
        // Webserver Configuration
        // Load WP_Filesystem for file permission check
        if ( ! function_exists('WP_Filesystem') ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        global $wp_filesystem;
        WP_Filesystem();

        // Check .htaccess writable status using WP_Filesystem
        $htaccess_path = ABSPATH . '.htaccess';
        $htaccess_writable = $wp_filesystem->is_writable( $htaccess_path ) ? 'Writable' : 'Non Writable';

        // Get current server time using gmdate() (timezone-safe)
        $current_server_time = gmdate('Y-m-d H:i:s');

        $system_info .= '<h2><button id="show-server-info-button" class="info-button">Server<span class="dashicons dashicons-arrow-down"></span></h2>';
        $system_info .= '<div id="server-info-container" class="info-content" style="display:none;">';
        $system_info .= '<h3>Server</h3>';
        $system_info .= '<table>';
        $system_info .= '<p>The options shown below relate to your server setup. If changes are required, you may need your web host’s assistance.</p>';

        $system_info .= '<tr><td>Server Architecture</td><td>' . esc_html(php_uname('s')) . '</td></tr>';
        $web_server = isset($_SERVER['SERVER_SOFTWARE']) ? sanitize_text_field(wp_unslash($_SERVER['SERVER_SOFTWARE'])) : 'Unavailable';
        $system_info .= '<tr><td>Web Server</td><td>' . esc_html($web_server) . '</td></tr>';
        $system_info .= '<tr><td>PHP Version</td><td>' . esc_html(phpversion()) . '</td></tr>';
        $system_info .= '<tr><td>PHP SAPI</td><td>' . esc_html(php_sapi_name()) . '</td></tr>';
        $system_info .= '<tr><td>PHP Max Input Variables</td><td>' . esc_html(ini_get('max_input_vars')) . '</td></tr>';
        $system_info .= '<tr><td>PHP Time Limit</td><td>' . esc_html(ini_get('max_execution_time')) . ' seconds</td></tr>';
        $system_info .= '<tr><td>PHP Memory Limit</td><td>' . esc_html(ini_get('memory_limit')) . '</td></tr>';
        $system_info .= '<tr><td>Max Input Time</td><td>' . esc_html(ini_get('max_input_time')) . ' seconds</td></tr>';
        $system_info .= '<tr><td>Upload Max Filesize</td><td>' . esc_html(ini_get('upload_max_filesize')) . '</td></tr>';
        $system_info .= '<tr><td>PHP Post Max Size</td><td>' . esc_html(ini_get('post_max_size')) . '</td></tr>';
        $system_info .= '<tr><td>cURL Version</td><td>' . esc_html(curl_version()['version']) . '</td></tr>';
        $system_info .= '<tr><td>Is SUHOSIN Installed?</td><td>' . (extension_loaded('suhosin') ? 'Yes' : 'No') . '</td></tr>';
        $system_info .= '<tr><td>Is the Imagick Library Available?</td><td>' . (extension_loaded('imagick') ? 'Yes' : 'No') . '</td></tr>';
        $system_info .= '<tr><td>Are Pretty Permalinks Supported?</td><td>' . (get_option('permalink_structure') ? 'Yes' : 'No') . '</td></tr>';
        $system_info .= '<tr><td>.htaccess Rules</td><td>' . esc_html($htaccess_writable) . '</td></tr>';
        $system_info .= '<tr><td>Current Time</td><td>' . esc_html(current_time('mysql')) . '</td></tr>';
        $system_info .= '<tr><td>Current UTC Time</td><td>' . esc_html(current_time('mysql', true)) . '</td></tr>';
        $system_info .= '<tr><td>Current Server Time</td><td>' . esc_html($current_server_time) . '</td></tr>';

        $system_info .= '</table>';
        $system_info .= '</div>';


        // Database Configuration
        $system_info .= '<h2><button id="show-database-info-button" class="info-button">Database<span class="dashicons dashicons-arrow-down"></span></h2>';
        $system_info .= '<div id="database-info-container" class="info-content" style="display:none;">';
        $system_info .= '<h3>Database</h3>';
        $system_info .= '<table>';

        $database_extension = 'mysqli';

        // Cached queries to avoid PHPCS warnings
        $database_server_version = wp_cache_get('gs_db_server_version', 'gsc');
        if (false === $database_server_version) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Safe, read-only server info
            $database_server_version = $wpdb->get_var("SELECT VERSION() as version");
            wp_cache_set('gs_db_server_version', $database_server_version, 'gsc', 3600);
        }

        $max_allowed_packet_size = wp_cache_get('gs_max_allowed_packet', 'gsc');
        if (false === $max_allowed_packet_size) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Safe, used for diagnostics only
            $max_allowed_packet_size = $wpdb->get_var("SHOW VARIABLES LIKE 'max_allowed_packet'");
            wp_cache_set('gs_max_allowed_packet', $max_allowed_packet_size, 'gsc', 3600);
        }

        $max_connections_number = wp_cache_get('gs_max_connections', 'gsc');
        if (false === $max_connections_number) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Safe, used for diagnostics only
            $max_connections_number = $wpdb->get_var("SHOW VARIABLES LIKE 'max_connections'");
            wp_cache_set('gs_max_connections', $max_connections_number, 'gsc', 3600);
        }

        $database_client_version = $wpdb->db_version();
        $database_username = DB_USER;
        $database_host = DB_HOST;
        $database_name = DB_NAME;
        $table_prefix = $wpdb->prefix;
        $database_charset = $wpdb->charset;
        $database_collation = $wpdb->collate;

        $system_info .= '<tr><td>Extension</td><td>' . esc_html($database_extension) . '</td></tr>';
        $system_info .= '<tr><td>Server Version</td><td>' . esc_html($database_server_version) . '</td></tr>';
        $system_info .= '<tr><td>Client Version</td><td>' . esc_html($database_client_version) . '</td></tr>';
        $system_info .= '<tr><td>Database Username</td><td>' . esc_html($database_username) . '</td></tr>';
        $system_info .= '<tr><td>Database Host</td><td>' . esc_html($database_host) . '</td></tr>';
        $system_info .= '<tr><td>Database Name</td><td>' . esc_html($database_name) . '</td></tr>';
        $system_info .= '<tr><td>Table Prefix</td><td>' . esc_html($table_prefix) . '</td></tr>';
        $system_info .= '<tr><td>Database Charset</td><td>' . esc_html($database_charset) . '</td></tr>';
        $system_info .= '<tr><td>Database Collation</td><td>' . esc_html($database_collation) . '</td></tr>';
        $system_info .= '<tr><td>Max Allowed Packet Size</td><td>' . esc_html($max_allowed_packet_size) . '</td></tr>';
        $system_info .= '<tr><td>Max Connections Number</td><td>' . esc_html($max_connections_number) . '</td></tr>';
        $system_info .= '</table>';
        $system_info .= '</div>';

        // wordpress constants
        $system_info .= '<h2><button id="show-wrcons-info-button" class="info-button">WordPress Constants<span class="dashicons dashicons-arrow-down"></span></h2>';
        $system_info .= '<div id="wrcons-info-container" class="info-content" style="display:none;">';
        $system_info .= '<h3>WordPress Constants</h3>';
        $system_info .= '<table>';
        // Add WordPress Constants information
        $system_info .= '<tr><td>ABSPATH</td><td>' . esc_html(ABSPATH) . '</td></tr>';
        $system_info .= '<tr><td>WP_HOME</td><td>' . esc_html(home_url()) . '</td></tr>';
        $system_info .= '<tr><td>WP_SITEURL</td><td>' . esc_html(site_url()) . '</td></tr>';
        $system_info .= '<tr><td>WP_CONTENT_DIR</td><td>' . esc_html(WP_CONTENT_DIR) . '</td></tr>';
        $system_info .= '<tr><td>WP_PLUGIN_DIR</td><td>' . esc_html(WP_PLUGIN_DIR) . '</td></tr>';
        $system_info .= '<tr><td>WP_MEMORY_LIMIT</td><td>' . esc_html(WP_MEMORY_LIMIT) . '</td></tr>';
        $system_info .= '<tr><td>WP_MAX_MEMORY_LIMIT</td><td>' . esc_html(WP_MAX_MEMORY_LIMIT) . '</td></tr>';
        $system_info .= '<tr><td>WP_DEBUG</td><td>' . (defined('WP_DEBUG') && WP_DEBUG ? 'Yes' : 'No') . '</td></tr>';
        $system_info .= '<tr><td>WP_DEBUG_DISPLAY</td><td>' . (defined('WP_DEBUG_DISPLAY') && WP_DEBUG_DISPLAY ? 'Yes' : 'No') . '</td></tr>';
        $system_info .= '<tr><td>SCRIPT_DEBUG</td><td>' . (defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? 'Yes' : 'No') . '</td></tr>';
        $system_info .= '<tr><td>WP_CACHE</td><td>' . (defined('WP_CACHE') && WP_CACHE ? 'Yes' : 'No') . '</td></tr>';
        $system_info .= '<tr><td>CONCATENATE_SCRIPTS</td><td>' . (defined('CONCATENATE_SCRIPTS') && CONCATENATE_SCRIPTS ? 'Yes' : 'No') . '</td></tr>';
        $system_info .= '<tr><td>COMPRESS_SCRIPTS</td><td>' . (defined('COMPRESS_SCRIPTS') && COMPRESS_SCRIPTS ? 'Yes' : 'No') . '</td></tr>';
        $system_info .= '<tr><td>COMPRESS_CSS</td><td>' . (defined('COMPRESS_CSS') && COMPRESS_CSS ? 'Yes' : 'No') . '</td></tr>';
        // Manually define the environment type (example values: 'development', 'staging', 'production')
        $environment_type = 'development';

        // Display the environment type
        $system_info .= '<tr><td>WP_ENVIRONMENT_TYPE</td><td>' . esc_html($environment_type) . '</td></tr>';

        $system_info .= '<tr><td>WP_DEVELOPMENT_MODE</td><td>' . (defined('WP_DEVELOPMENT_MODE') && WP_DEVELOPMENT_MODE ? 'Yes' : 'No') . '</td></tr>';
        $system_info .= '<tr><td>DB_CHARSET</td><td>' . esc_html(DB_CHARSET) . '</td></tr>';
        $system_info .= '<tr><td>DB_COLLATE</td><td>' . esc_html(DB_COLLATE) . '</td></tr>';

        $system_info .= '</table>';
        $system_info .= '</div>';

        // Filesystem Permission
        // Load WP_Filesystem if not already available
        if ( ! function_exists('WP_Filesystem') ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        global $wp_filesystem;
        WP_Filesystem();

        // Get directory paths
        $upload_dir = wp_upload_dir()['basedir'];
        $theme_root = get_theme_root();

        // Check writability using WP_Filesystem
        $main_dir_writable     = $wp_filesystem->is_writable( ABSPATH ) ? 'Writable' : 'Not Writable';
        $wp_content_writable   = $wp_filesystem->is_writable( WP_CONTENT_DIR ) ? 'Writable' : 'Not Writable';
        $upload_dir_writable   = $wp_filesystem->is_writable( $upload_dir ) ? 'Writable' : 'Not Writable';
        $plugin_dir_writable   = $wp_filesystem->is_writable( WP_PLUGIN_DIR ) ? 'Writable' : 'Not Writable';
        $theme_dir_writable    = $wp_filesystem->is_writable( $theme_root ) ? 'Writable' : 'Not Writable';

        $system_info .= '<h2><button id="show-ftps-info-button" class="info-button">Filesystem Permission <span class="dashicons dashicons-arrow-down"></span></button></h2>';
        $system_info .= '<div id="ftps-info-container" class="info-content" style="display:none;">';
        $system_info .= '<h3>Filesystem Permission</h3>';
        $system_info .= '<p>Shows whether WordPress is able to write to the directories it needs access to.</p>';
        $system_info .= '<table>';

        $system_info .= '<tr><td>The main WordPress directory</td><td>' . esc_html(ABSPATH) . '</td><td>' . esc_html($main_dir_writable) . '</td></tr>';
        $system_info .= '<tr><td>The wp-content directory</td><td>' . esc_html(WP_CONTENT_DIR) . '</td><td>' . esc_html($wp_content_writable) . '</td></tr>';
        $system_info .= '<tr><td>The uploads directory</td><td>' . esc_html($upload_dir) . '</td><td>' . esc_html($upload_dir_writable) . '</td></tr>';
        $system_info .= '<tr><td>The plugins directory</td><td>' . esc_html(WP_PLUGIN_DIR) . '</td><td>' . esc_html($plugin_dir_writable) . '</td></tr>';
        $system_info .= '<tr><td>The themes directory</td><td>' . esc_html($theme_root) . '</td><td>' . esc_html($theme_dir_writable) . '</td></tr>';

        $system_info .= '</table>';
        $system_info .= '</div>';

        return $system_info;
    }

    public function display_error_log()
    {
        // Define the path to your debug log file
        $debug_log_file = WP_CONTENT_DIR . '/debug.log';

        // Check if the debug log file exists
        if (file_exists($debug_log_file)) {
            // Read the contents of the debug log file
            $debug_log_contents = file_get_contents($debug_log_file);

            // Split the log content into an array of lines
            $log_lines = explode("\n", $debug_log_contents);

            // Get the last 100 lines in reversed order
            $last_100_lines = array_slice(array_reverse($log_lines), 0, 100);

            // Join the lines back together with line breaks
            $last_100_log = implode("\n", $last_100_lines);

            // Output the last 100 lines in reversed order in a textarea
            ?>
            <textarea class="errorlog" rows="20" cols="80"><?php echo esc_textarea($last_100_log); ?></textarea>
            <?php
        } else {
            echo 'Debug log file not found.';
        }
    }

}

// Initialize the google sheet connector class
$init = new wc_gsheetconnector_Init();

add_filter('plugin_action_links_' . WC_GSHEETCONNECTOR_BASE_NAME, array($init, 'wc_gsheet_setting_link'));