<?php

namespace Buckaroo\Woocommerce\Admin;

use Buckaroo\Woocommerce\Gateways\AbstractPaymentGateway;
use Buckaroo\Woocommerce\Services\LoggerStorage;
use WC_Tax;

class PaymentMethodSettings extends AbstractPaymentGateway
{
    public function __construct()
    {
        $this->id = 'buckaroo_mastersettings';
        $this->title = 'Master Settings';
        $this->has_fields = false;
        $this->method_title = __(
            'Buckaroo Master Settings',
            'wc-buckaroo-bpe-gateway'
        );
        parent::__construct();
    }

    /**
     * Custom validation for textarea fields to prevent HTML entity encoding
     *
     * @param string $key
     * @param string $value
     * @return string
     */
    public function validate_textarea_field($key, $value)
    {
        if (in_array($key, ['transactiondescription', 'refund_description'])) {
            return is_null($value) ? '' : wp_unslash(trim($value));
        }

        return parent::validate_textarea_field($key, $value);
    }

    /**
        public
     */
    public function init_form_fields()
    {
        $this->id = (! isset($this->id) ? '' : $this->id);

        // Start Certificate fields
        $this->form_fields['merchantkey'] = [
            'title' => __('Store key', 'wc-buckaroo-bpe-gateway'),
            'type' => 'password',
            'description' => __('This is your Buckaroo Payment Plaza <a href="https://plaza.buckaroo.nl/Configuration/Merchant/ApiKeys" target="_blank">store key</a> (Settings -> API Key -> Store keys).', 'wc-buckaroo-bpe-gateway'),
            'default' => '',
            'custom_attributes' => [
                'required' => 'required',
            ],
        ];
        $this->form_fields['secretkey'] = [
            'title' => __('Secret key', 'wc-buckaroo-bpe-gateway'),
            'type' => 'password',
            'description' => __('This is your Buckaroo Payment Plaza <a href="https://plaza.buckaroo.nl/Configuration/Merchant/ApiKeys" target="_blank">secret key</a> (Settings -> API Key -> Secret key).', 'wc-buckaroo-bpe-gateway'),
            'default' => '',
            'custom_attributes' => [
                'required' => 'required',
            ],
        ];
        $this->form_fields['test_credentials'] = [
            'title' => __('Test credentials', 'wc-buckaroo-bpe-gateway'),
            'type' => 'button',
            'description' => __('Click here to verify store key & secret key.', 'wc-buckaroo-bpe-gateway'),
            'custom_attributes' => [
                'title' => __('Test', 'wc-buckaroo-bpe-gateway'),
            ],
            'default' => '',
        ];

        $this->form_fields['auto_configure'] = [
            'title' => __('Automatic payment method configuration', 'wc-buckaroo-bpe-gateway'),
            'type' => 'button',
            'description' => __('Automatically configure the Buckaroo plugin based on your active subscriptions. When you use this option, the plugin will connect to your Buckaroo account, check which payment methods are active, and enable them in Live mode. You will be asked to confirm before changes are applied.', 'wc-buckaroo-bpe-gateway'),
            'custom_attributes' => [
                'title' => __('Auto-configure', 'wc-buckaroo-bpe-gateway'),
            ],
            'value' => __('Auto-configure', 'wc-buckaroo-bpe-gateway'),
            'default' => '',
        ];

        $taxes = $this->getTaxClasses();
        $this->form_fields['feetax'] = [
            'title' => __('Select tax class for fee', 'wc-buckaroo-bpe-gateway'),
            'type' => 'select',
            'options' => $taxes,
            'description' => __('Fee tax class', 'wc-buckaroo-bpe-gateway'),
            'default' => '',
        ];
        $this->form_fields['paymentfeevat'] = [
            'title' => __('Payment fee display', 'wc-buckaroo-bpe-gateway'),
            'type' => 'select',
            'options' => [
                'off' => 'Excluding VAT',
                'on' => 'Including VAT',
            ],
            'description' => __('Select if payment fee is displayed including / excluding VAT', 'wc-buckaroo-bpe-gateway'),
            'default' => 'off',
        ];
        $this->form_fields['culture'] = [
            'title' => __('Language', 'wc-buckaroo-bpe-gateway'),
            'type' => 'select',
            'description' => __(
                "The chosen language (culture) for the Buckaroo payment engine <br>page.
When this is set to “Dynamic language” the plugin will <br>automatically determine the language based on the
language <br>settings of the customer's web browser. Please note that we only <br>support the following languages: English, Dutch, German and French.<br>
English will be used as a fallback language for unknown language types.",
                'wc-buckaroo-bpe-gateway'
            ),
            'options' => [
                'dynamic' => 'Dynamic language (based on the web browser language)',
                'en-US' => 'English',
                'nl-NL' => 'Dutch',
                'fr-FR' => 'French',
                'de-DE' => 'German',
            ],
            'default' => 'dynamic',
            'id' => 'woocommerce_buckaroo_mastersettings_culture',
        ];

        $this->form_fields['debugmode'] = [
            'title' => __('Debug mode', 'wc-buckaroo-bpe-gateway'),
            'type' => 'select',
            'description' => __('Toggle debug mode on/off', 'wc-buckaroo-bpe-gateway'),
            'options' => [
                'on' => 'On',
                'off' => 'Off',
            ],
            'default' => 'off',
        ];

        $this->form_fields['logstorage'] = [
            'title' => __('Debug data storage', 'wc-buckaroo-bpe-gateway'),
            'type' => 'select',
            'description' => __('Select where to store debug data', 'wc-buckaroo-bpe-gateway'),
            'options' => [
                LoggerStorage::STORAGE_ALL => __('Both'),
                LoggerStorage::STORAGE_FILE => __('File'),
                LoggerStorage::STORAGE_DB => __('Database'),
            ],
            'default' => LoggerStorage::STORAGE_ALL,
        ];

        $this->form_fields['transactiondescription'] = [
            'title' => __('Transaction description', 'wc-buckaroo-bpe-gateway'),
            'type' => 'textarea',
            'description' => __('Transaction description', 'wc-buckaroo-bpe-gateway'),
            'desc_tip' => __('Transaction description can be filled with static text and tags like: {order_number}, {shop_name} and {product_name} for first product found.'),
            'default' => '',
        ];

        $this->form_fields['refund_description'] = [
            'title' => __('Refund description', 'wc-buckaroo-bpe-gateway'),
            'type' => 'textarea',
            'description' => __('Refund description', 'wc-buckaroo-bpe-gateway'),
            'desc_tip' => __('Refund description can be filled with static text and tags like: {order_number}, {shop_name} and {product_name} for first product found.'),
            'default' => '',
        ];

        $this->apply_filter_or_error('append_subscription_configurationCode_in_setting_field', $this);

        $this->form_fields['useidin'] = [
            'title' => __('iDIN mode', 'wc-buckaroo-bpe-gateway'),
            'type' => 'select',
            'description' => __('By turning on this setting age verification with iDIN will be in use', 'wc-buckaroo-bpe-gateway'),
            'options' => [
                '0' => 'Off',
                'live' => 'Live',
                'test' => 'Test',
            ],
            'default' => '0',
        ];

        $idinCategories = [];
        if (
            $categories = get_terms(
                [
                'taxonomy' => 'product_cat',
                'hide_empty' => false,
                ]
            )
        ) {
            foreach ($categories as $category) {
                $idinCategories[$category->term_id] = $category->name;
            }
        }
        $this->form_fields['idincategories'] = [
            'title' => __('iDIN specific product categories', 'wc-buckaroo-bpe-gateway'),
            'type' => 'multiselect',
            'options' => $idinCategories,
            'description' => __("Select for what product categories iDIN verification should be applied. Don't select anything if want to apply iDIN to any product", 'wc-buckaroo-bpe-gateway'),
            'default' => [],
        ];
    }

    protected function getTaxClasses()
    {
        $allTaxRates = [];
        $taxClasses = WC_Tax::get_tax_classes(); // Retrieve all tax classes.
        if (! in_array('', $taxClasses)) {
            // Make sure "Standard rate" (empty class name) is present.
            array_unshift($taxClasses, '');
        }
        foreach ($taxClasses as $taxClass) {
            // For each tax class, get all rates.
            $taxes = WC_Tax::get_rates_for_tax_class($taxClass);
            foreach ($taxes as $tax) {
                $allTaxRates[$tax->{'tax_rate_class'}] = $tax->{'tax_rate_name'};
                if (empty($allTaxRates[$tax->{'tax_rate_class'}])) {
                    $allTaxRates[$tax->{'tax_rate_class'}] = 'Standard Rate';
                }
            }
        }

        return $allTaxRates;
    }
}
