<?php
/** 
 * @package     VikBooking
 * @subpackage  core
 * @author      E4J s.r.l.
 * @copyright   Copyright (C) 2025 E4J s.r.l. All Rights Reserved.
 * @license     http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 * @link        https://vikwp.com
 */

// No direct access
defined('ABSPATH') or die('No script kiddies please!');

/**
 * Obtain vars from arguments received in the layout file.
 * 
 * @var string $tool         The operator tool identifier calling this layout file.
 * @var array  $operator     The operator record accessing the tool.
 * @var object $permissions  The operator-tool permissions registry.
 * @var string $tool_uri     The base URI for rendering this tool.
 * @var array  $data         The data for rendering the tasks of a given month within a calendar.
 * @var int    $no_tasks     True for just displaying the day with no tasks, can be undefined.
 */
extract($displayData);

if (!$operator || empty($operator['id'])) {
    throw new Exception('Missing operator details', 400);
}

// check no-loading flag
$no_loading = (bool) ($no_tasks ?? 0);

// get the filters for rendering the calendar tasks
$filters = (array) ($data['filters'] ?? []);

// get the current operator permissions
$accept_tasks = (bool) ($permissions ? $permissions->get('accept_tasks', 0) : 0);

// determine the day to load
$dayYmd = $filters['calendar_day'] ?? date('Y-m-d');
if (($filters['dates'] ?? null) && !($filters['calendar_day'] ?? null)) {
    // when no calendar day filter set, use the current dates filter
    list($fromDt, $toDt) = VBOTaskModelTask::getInstance()->getFilterDatesInterval((string) $filters['dates'], $local = true, $sql = false);
    if ($fromDt) {
        $dayYmd = date('Y-m-d', strtotime($fromDt));
    }
}

// build the pool of day tasks
$dayTasks = [];

if (!$no_loading) {
    // save current filters in the user state
    JFactory::getApplication()->setUserState('vbo.tm.filters', $filters);

    // check from the permissions whether tasks can be accepted by the operator,
    // hence null assigness should be included - use a different filter otherwise
    $operatorTasksFilterName = $accept_tasks ? 'operator' : 'assignee';

    // build filter options for fetching the tasks
    $fetchOptions = array_merge($filters, [
        $operatorTasksFilterName => $operator['id'],
        'dates' => $dayYmd . ':' . $dayYmd,
        'calendar' => true,
    ]);

    // load tasks according to filters, by always forcing/injecting the dates and the assignee/operator filter
    foreach (VBOTaskModelTask::getInstance()->filterItems($fetchOptions, 0, 0) as $taskRecord) {
        // push task registry
        $dayTasks[] = VBOTaskTaskregistry::getInstance((array) $taskRecord);
    }
}

// obtain the current day information
$dayDate = getdate(strtotime($dayYmd));

// get navigation dates
$todayYmd = date('Y-m-d');
$dayBack  = date('Y-m-d', strtotime('-1 day', $dayDate[0]));
$dayNext  = date('Y-m-d', strtotime('+1 day', $dayDate[0]));

// build the task objects, if any, for JS positioning
$taskObjects = [];
foreach ($dayTasks as $dayTask) {
    $duration = $dayTask->getDuration();

    $start = $dayTask->getDueDate(true, 'Y-m-d\TH:i:s');
    $begin = $dayTask->getBeginDate(true, 'Y-m-d\TH:i:s');

    if ($begin) {
        $start = $begin;
    }

    $startDate = JFactory::getDate($start);

    // get selected date at midnight
    $dateMidnight = JFactory::getDate($dayYmd . ' 00:00:00');

    // check whether the event starts on a previous date
    if ($startDate < $dateMidnight) {
        // decrease duration by the difference between today and the real start date
        $diff = $startDate->diff($dateMidnight);
        $duration -= $diff->days * 1440 + $diff->h * 60 + $diff->i;

        // use today at midnight
        $startDate = clone $dateMidnight;
    }

    if ($duration <= 0) {
        // ignore the event in case of invalid duration
        continue;
    }

    $endDate = clone $startDate;
    $endDate->modify('+' . $duration . ' minutes');

    // extract hours and minutes
    list($hour, $min) = explode(':', $startDate->format('G:i'));

    $taskObjects[] = [
        'id'       => $dayTask->getID(),
        'start'    => $startDate->format('Y-m-d\TH:i:s'),
        'end'      => $endDate->format('Y-m-d\TH:i:s'),
        'hour'     => (int) $hour,
        'min'      => (int) $min,
        'duration' => (int) $duration,
        'html'     => JLayoutHelper::render('taskmanager.calendar.day.task', ['task' => $dayTask], null, [
            'component' => 'com_vikbooking',
            'client'    => 'site',
        ]),
    ];
}

?>

<div class="vbo-tm-calendar-wrap">
    <div class="vbo-tm-calendar-head">
        <div class="vbo-tm-calendar-info">
            <a href="javascript:void(0)" class="vbo-tm-calendar-day-back" data-month="<?php echo date('Y-m-01', $dayDate[0]); ?>">
                <?php echo VikBookingIcons::e('chevron-left'); ?>
            </a>
            <span class="vbo-tm-calendar-day-day"><?php echo VikBooking::sayWeekDay($dayDate['wday']) . ', ' . $dayDate['mday']; ?></span>
            <span class="vbo-tm-calendar-day-month"><?php echo VikBooking::sayMonth($dayDate['mon']); ?></span>
            <span class="vbo-tm-calendar-day-year"><?php echo $dayDate['year']; ?></span>
        </div>

        <div class="vbo-tm-calendar-nav">
            <div class="vbo-tm-calendar-nav-btns">
                <span class="vbo-tm-calendar-nav-btn vbo-tm-calendar-nav-back" data-day="<?php echo $dayBack; ?>"><?php VikBookingIcons::e('chevron-left'); ?></span>
                <span class="vbo-tm-calendar-nav-btn vbo-tm-calendar-nav-today" data-day="<?php echo $todayYmd; ?>"><?php echo JText::translate('VBTODAY'); ?></span>
                <span class="vbo-tm-calendar-nav-btn vbo-tm-calendar-nav-next" data-day="<?php echo $dayNext; ?>"><?php VikBookingIcons::e('chevron-right'); ?></span>
            </div>
        </div>
    </div>

    <div class="vbo-tm-calendar-day-timeline">
        <div class="vbo-tm-calendar-day-timeline-week">

        <?php
        // display up to 3 week-days in the past
        for ($d = 3; $d > 0; $d--) {
            $timeline_date = getdate(strtotime(sprintf('-%d %s', $d, ($d == 1 ? 'day' : 'days')), $dayDate[0]));
            ?>
            <div class="vbo-tm-calendar-day-timeline-week-day">
                <span class="vbo-tm-calendar-nav-btn vbo-tm-calendar-day-timeline-week-nav" data-day="<?php echo date('Y-m-d', $timeline_date[0]); ?>">
                    <span class="vbo-tm-calendar-day-timeline-week-nav-mday"><?php echo $timeline_date['mday']; ?></span>
                    <span class="vbo-tm-calendar-day-timeline-week-nav-wday"><?php echo VikBooking::sayWeekDay($timeline_date['wday'], true); ?></span>
                </span>
            </div>
            <?php
        }
        ?>
            <div class="vbo-tm-calendar-day-timeline-week-day">
                <span class="vbo-tm-calendar-nav-btn vbo-tm-calendar-day-timeline-week-nav vbo-tm-calendar-day-timeline-week-today" data-day="<?php echo $todayYmd; ?>">
                    <span class="vbo-tm-calendar-day-timeline-week-nav-mday"><?php echo $dayDate['mday']; ?></span>
                    <span class="vbo-tm-calendar-day-timeline-week-nav-wday"><?php echo VikBooking::sayWeekDay($dayDate['wday'], true); ?></span>
                </span>
            </div>
        <?php
        // display up to 3 week-days ahead
        for ($d = 1; $d < 4; $d++) {
            $timeline_date = getdate(strtotime(sprintf('+%d %s', $d, ($d == 1 ? 'day' : 'days')), $dayDate[0]));
            ?>
            <div class="vbo-tm-calendar-day-timeline-week-day">
                <span class="vbo-tm-calendar-nav-btn vbo-tm-calendar-day-timeline-week-nav" data-day="<?php echo date('Y-m-d', $timeline_date[0]); ?>">
                    <span class="vbo-tm-calendar-day-timeline-week-nav-mday"><?php echo $timeline_date['mday']; ?></span>
                    <span class="vbo-tm-calendar-day-timeline-week-nav-wday"><?php echo VikBooking::sayWeekDay($timeline_date['wday'], true); ?></span>
                </span>
            </div>
            <?php
        }
        ?>
        </div>
        <div class="vbo-tm-calendar-day-timeline-rows">
        <?php
        for ($h = 0; $h < 24; $h++) {
            ?>
            <div class="vbo-tm-calendar-day-timeline-row" data-hour="<?php echo $h; ?>">
                <div class="vbo-tm-calendar-day-hour">
                    <strong><?php echo str_pad($h, 2, '0', STR_PAD_LEFT) . ':' . '00'; ?></strong>
                </div>
                <div class="vbo-tm-calendar-day-tasks"></div>
            </div>
            <?php
        }
        ?>
        </div>
    </div>
</div>

<script type="text/javascript">
    jQuery(function($) {
        // setup the task (events) within the calendar daily layout
        window.taskManagerSetupCalendar(<?php echo json_encode($taskObjects); ?>);

        /**
         * Register listeners for day navigation buttons.
         */
        document
            .querySelectorAll('.vbo-tm-calendar-nav-btn')
            .forEach((nav) => {
                // get the navigation day
                const day = nav.getAttribute('data-day');

                nav.addEventListener('click', () => {
                    VBOCore.emitEvent('vbo-tm-apply-filters', {
                        filters: {
                            calendar_day: day,
                        },
                    });
                });
            });

        /**
         * Register listeners for month calendar type.
         */
        document
            .querySelectorAll('.vbo-tm-calendar-day-back')
            .forEach((mname) => {
                // get the navigation month
                const month = mname.getAttribute('data-month');

                mname.addEventListener('click', () => {
                    VBOCore.emitEvent('vbo-tm-apply-filters', {
                        filters: {
                            calendar_type: 'month',
                            calendar_month: month,
                            calendar_day: '',
                        },
                    });
                });
            });

        /**
         * Register listeners for viewing an existing task.
         */
        document
            .querySelectorAll('.vbo-tm-calendar-day-task[data-task-id]')
            .forEach((taskElement) => {
                if (taskElement.clickListener) {
                    // listener added already
                    return;
                }

                // get the clicked task and area IDs
                const taskId = taskElement.getAttribute('data-task-id');
                const areaId = taskElement.getAttribute('data-area-id');

                if (!taskId || !areaId) {
                    // missing task details
                    return;
                }

                taskElement.addEventListener('click', (event) => {
                    event.preventDefault();
                    event.stopPropagation();

                    VBOCore.emitEvent('vbo-tm-apply-filters', {
                        filters: {
                            calendar_back_type: 'day',
                            calendar_back_day: '<?php echo $dayYmd; ?>',
                            calendar_type: 'taskdetails',
                            task_id: taskId,
                        },
                    });
                });

                // turn flag on for listener set
                taskElement.clickListener = true;
            });

    });
</script>