<?php
/** 
 * @package     VikBooking
 * @subpackage  core
 * @author      E4J s.r.l.
 * @copyright   Copyright (C) 2021 E4J s.r.l. All Rights Reserved.
 * @license     http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 * @link        https://vikwp.com
 */

// No direct access
defined('ABSPATH') or die('No script kiddies please!');

/**
 * Wizard pre-checkin help instruction.
 * 
 * @since 1.18.2 (J) - 1.8.2 (WP)
 */
class VBOHelpWizardDriverGeneralCronPrecheckin extends VBOHelpWizardInstructionaware
{
    /**
     * @inheritDoc
     */
    public function getID()
    {
        return 'general.cron_precheckin';
    }

    /**
     * @inheritDoc
     */
    public function getIcon()
    {
        return VikBookingIcons::i('user-check');
    }

    /**
     * @inheritDoc
     */
    public function getPriority()
    {
        return 9999;
    }

    /**
     * @inheritDoc
     */
    public function isConfigured()
    {
        if (!VBOFactory::getConfig()->getBool('precheckinenabled')) {
            // pre-checkin disabled
            return false;
        }

        // check whether a published cron job exists to collect pre-checkins
        return (bool) (new VBOModelCronjob)->getItem([
            'class_file' => 'precheckin_reminder',
            'published' => 1,
        ]);
    }

    /**
     * @inheritDoc
     */
    public function isProcessable(?string &$btnText = null)
    {
        $btnText = JText::translate('VBO_SCHEDULE_CRONJOB');
        
        return true;
    }

    /**
     * @inheritDoc
     */
    public function process(array $args = [])
    {
        $config = VBOFactory::getConfig();
        $config->set('precheckinenabled', 1);
        $config->set('precheckinminoffset', 0);

        switch ($config->get('maincountry')) {
            case 'IT':
                $config->set('checkindata', 'italy');
                break;

            case 'PT':
                $config->set('checkindata', 'portugal');
                break;

            case 'ES':
                // in case of states under "Catalunya" (Barcelona, Girona, Lleida, Tarragona) use "spain" driver, otherwise fallback to ses hospedajes
                $config->set('checkindata', in_array($config->get('mainstate'), ['08', '17', '25', '43']) ? 'spain' : 'spainseshospedajes');
                break;
        }

        $cron = (new VBOModelCronjob)->getItem([
            'class_file' => 'precheckin_reminder',
        ]);

        if (!$cron) {
            // create the cron job
            $cron = new stdClass;
            $cron->cron_name = 'Pre-Checkin Reminder';
            $cron->class_file = 'precheckin_reminder';
            $cron->params = [
                'remindbefored' => 7,
                'less_days_advance' => 1,
                'subject' => 'Pre-Checkin Reminder',
                'tpl_text' => '',
            ];

            // attempt to create a new job instance
            $job = VBOFactory::getCronFactory()->createInstance($cron->class_file, $cron);

            // use the default e-mail template generated by the cron
            $cron->params['tpl_text'] = $job->getDefaultTemplate();

            $cron->params = json_encode($cron->params);
        }

        // make sure the cron job is published
        $cron->published = 1;

        // insert or update the cron
        if (!(new VBOModelCronjob)->save($cron)) {
            throw new RuntimeException('Unable to save the pre-checkin cron!', 500);
        }
    }
}
