<?php
/** 
 * @package     VikBooking
 * @subpackage  core
 * @author      Alessio Gaggii - E4J s.r.l.
 * @copyright   Copyright (C) 2024 E4J s.r.l. All Rights Reserved.
 * @license     http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 * @link        https://vikwp.com
 */

// No direct access
defined('ABSPATH') or die('No script kiddies please!');

/**
 * Defines the handler for a pax field of type "spain_municipio".
 * 
 * @since 	1.17.2 (J) - 1.7.2 (WP)
 */
final class VBOCheckinPaxfieldTypeSpainMunicipio extends VBOCheckinPaxfieldType
{
	/**
	 * Renders the current pax field HTML.
	 * 
	 * @return 	string 	the HTML string to render the field.
	 */
	public function render()
	{
		// get the field unique ID
		$field_id = $this->getFieldIdAttr();

		// get the guest number
		$guest_number = $this->field->getGuestNumber();

		// get the field class attribute
		$pax_field_class = $this->getFieldClassAttr();

		// get field name attribute
		$name = $this->getFieldNameAttr();

		// get the field value attribute
		$value = $this->getFieldValueAttr();

		// compose HTML content for the field
		$field_html = '';
		$field_html .= "<select id=\"$field_id\" data-gind=\"$guest_number\" class=\"$pax_field_class\" name=\"$name\">\n";
		$field_html .= '<option></option>' . "\n";

		foreach (self::loadMunicipioCodes() as $codigo => $municipio) {
			$field_html .= '<option value="' . htmlspecialchars($codigo, ENT_QUOTES, 'UTF-8') . '"' . ($value == $codigo ? ' selected="selected"' : '') . '>' . $municipio . '</option>' . "\n";
		}

		$field_html .= '</select>';

		// append select2 JS script for rendering the field
		$field_html .= <<<HTML
<script>
	jQuery(function() {

		jQuery("#$field_id").select2({
			width: "100%",
			placeholder: "-----",
			allowClear: true
		});

	});
</script>
HTML;

		// return the necessary HTML string to display the field
		return $field_html;
	}

	/**
	 * Helper method that returns a list of "municipio codes".
	 *
	 * @return 	array
	 */
	public static function loadMunicipioCodes()
	{
		return [
			'01051' => "Salvatierra/Agurain",
			'01001' => "Alegría-Dulantzi",
			'01002' => "Amurrio",
			'01049' => "Añana",
			'01003' => "Aramaio",
			'01006' => "Armiñón",
			'01037' => "Arraia-Maeztu",
			'01008' => "Arratzua-Ubarrundia",
			'01004' => "Artziniega",
			'01009' => "Asparrena",
			'01010' => "Ayala/Aiara",
			'01011' => "Baños de Ebro/Mañueta",
			'01013' => "Barrundia",
			'01014' => "Berantevilla",
			'01016' => "Bernedo",
			'01017' => "Campezo/Kanpezu",
			'01021' => "Elburgo/Burgelu",
			'01022' => "Elciego",
			'01023' => "Elvillar/Bilar",
			'01046' => "Erriberagoitia/Ribera Alta",
			'01056' => "Harana/Valle de Arana",
			'01901' => "Iruña Oka/Iruña de Oca",
			'01027' => "Iruraiz-Gauna",
			'01019' => "Kripan",
			'01020' => "Kuartango",
			'01028' => "Labastida/Bastida",
			'01030' => "Lagrán",
			'01031' => "Laguardia",
			'01032' => "Lanciego/Lantziego",
			'01902' => "Lantarón",
			'01033' => "Lapuebla de Labarca",
			'01036' => "Laudio/Llodio",
			'01058' => "Legutio",
			'01034' => "Leza",
			'01039' => "Moreda de Álava/Moreda Araba",
			'01041' => "Navaridas",
			'01042' => "Okondo",
			'01043' => "Oyón-Oion",
			'01044' => "Peñacerrada-Urizaharra",
			'01047' => "Ribera Baja/Erriberabeitia",
			'01052' => "Samaniego",
			'01053' => "San Millán/Donemiliaga",
			'01054' => "Urkabustaiz",
			'01055' => "Valdegovía/Gaubea",
			'01057' => "Villabuena de Álava/Eskuernaga",
			'01059' => "Vitoria-Gasteiz",
			'01060' => "Yécora/Iekora",
			'01061' => "Zalduondo",
			'01062' => "Zambrana",
			'01018' => "Zigoitia",
			'01063' => "Zuia",
			'02001' => "Abengibre",
			'02002' => "Alatoz",
			'02003' => "Albacete",
			'02004' => "Albatana",
			'02005' => "Alborea",
			'02006' => "Alcadozo",
			'02007' => "Alcalá del Júcar",
			'02008' => "Alcaraz",
			'02009' => "Almansa",
			'02010' => "Alpera",
			'02011' => "Ayna",
			'02012' => "Balazote",
			'02014' => "Ballestero, El",
			'02013' => "Balsa de Ves",
			'02015' => "Barrax",
			'02016' => "Bienservida",
			'02017' => "Bogarra",
			'02018' => "Bonete",
			'02019' => "Bonillo, El",
			'02020' => "Carcelén",
			'02021' => "Casas de Juan Núñez",
			'02022' => "Casas de Lázaro",
			'02023' => "Casas de Ves",
			'02024' => "Casas-Ibáñez",
			'02025' => "Caudete",
			'02026' => "Cenizate",
			'02029' => "Chinchilla de Monte-Aragón",
			'02027' => "Corral-Rubio",
			'02028' => "Cotillas",
			'02030' => "Elche de la Sierra",
			'02031' => "Férez",
			'02032' => "Fuensanta",
			'02033' => "Fuente-Álamo",
			'02034' => "Fuentealbilla",
			'02035' => "Gineta, La",
			'02036' => "Golosalvo",
			'02037' => "Hellín",
			'02038' => "Herrera, La",
			'02039' => "Higueruela",
			'02040' => "Hoya-Gonzalo",
			'02041' => "Jorquera",
			'02042' => "Letur",
			'02043' => "Lezuza",
			'02044' => "Liétor",
			'02045' => "Madrigueras",
			'02046' => "Mahora",
			'02047' => "Masegoso",
			'02048' => "Minaya",
			'02049' => "Molinicos",
			'02050' => "Montalvos",
			'02051' => "Montealegre del Castillo",
			'02052' => "Motilleja",
			'02053' => "Munera",
			'02054' => "Navas de Jorquera",
			'02055' => "Nerpio",
			'02056' => "Ontur",
			'02057' => "Ossa de Montiel",
			'02058' => "Paterna del Madera",
			'02060' => "Peñas de San Pedro",
			'02059' => "Peñascosa",
			'02061' => "Pétrola",
			'02062' => "Povedilla",
			'02901' => "Pozo Cañada",
			'02063' => "Pozohondo",
			'02064' => "Pozo-Lorente",
			'02065' => "Pozuelo",
			'02066' => "Recueja, La",
			'02067' => "Riópar",
			'02068' => "Robledo",
			'02069' => "Roda, La",
			'02070' => "Salobre",
			'02071' => "San Pedro",
			'02072' => "Socovos",
			'02073' => "Tarazona de la Mancha",
			'02074' => "Tobarra",
			'02075' => "Valdeganga",
			'02076' => "Vianos",
			'02077' => "Villa de Ves",
			'02078' => "Villalgordo del Júcar",
			'02079' => "Villamalea",
			'02080' => "Villapalacios",
			'02081' => "Villarrobledo",
			'02082' => "Villatoya",
			'02083' => "Villavaliente",
			'02084' => "Villaverde de Guadalimar",
			'02085' => "Viveros",
			'02086' => "Yeste",
			'03002' => "Agost",
			'03003' => "Agres",
			'03004' => "Aigües",
			'03014' => "Alacant/Alicante",
			'03005' => "Albatera",
			'03006' => "Alcalalí",
			'03009' => "Alcoi/Alcoy",
			'03008' => "Alcoleja",
			'03007' => "Alcosser",
			'03010' => "Alfafara",
			'03011' => "Alfàs del Pi, l'",
			'03012' => "Algorfa",
			'03013' => "Algueña",
			'03015' => "Almoradí",
			'03016' => "Almudaina",
			'03017' => "Alqueria d'Asnar, l'",
			'03018' => "Altea",
			'03019' => "Aspe",
			'03001' => "Atzúbia, l'",
			'03020' => "Balones",
			'03021' => "Banyeres de Mariola",
			'03022' => "Benasau",
			'03023' => "Beneixama",
			'03024' => "Benejúzar",
			'03025' => "Benferri",
			'03026' => "Beniarbeig",
			'03027' => "Beniardà",
			'03028' => "Beniarrés",
			'03030' => "Benidoleig",
			'03031' => "Benidorm",
			'03032' => "Benifallim",
			'03033' => "Benifato",
			'03029' => "Benigembla",
			'03034' => "Benijófar",
			'03035' => "Benilloba",
			'03036' => "Benillup",
			'03037' => "Benimantell",
			'03038' => "Benimarfull",
			'03039' => "Benimassot",
			'03040' => "Benimeli",
			'03041' => "Benissa",
			'03043' => "Biar",
			'03044' => "Bigastro",
			'03045' => "Bolulla",
			'03046' => "Busot",
			'03049' => "Callosa de Segura",
			'03048' => "Callosa d'en Sarrià",
			'03047' => "Calp",
			'03051' => "Camp de Mirra, el/Campo de Mirra",
			'03050' => "Campello, el",
			'03052' => "Cañada",
			'03053' => "Castalla",
			'03054' => "Castell de Castells",
			'03075' => "Castell de Guadalest, el",
			'03055' => "Catral",
			'03056' => "Cocentaina",
			'03057' => "Confrides",
			'03058' => "Cox",
			'03059' => "Crevillent",
			'03061' => "Daya Nueva",
			'03062' => "Daya Vieja",
			'03063' => "Dénia",
			'03064' => "Dolores",
			'03066' => "Elda",
			'03065' => "Elx/Elche",
			'03067' => "Fageca",
			'03068' => "Famorca",
			'03069' => "Finestrat",
			'03077' => "Fondó de les Neus, el/Hondón de las Nieves",
			'03070' => "Formentera del Segura",
			'03072' => "Gaianes",
			'03071' => "Gata de Gorgos",
			'03073' => "Gorga",
			'03074' => "Granja de Rocamora",
			'03076' => "Guardamar del Segura",
			'03078' => "Hondón de los Frailes",
			'03079' => "Ibi",
			'03080' => "Jacarilla",
			'03085' => "Llíber",
			'03086' => "Millena",
			'03088' => "Monforte del Cid",
			'03089' => "Monòver/Monóvar",
			'03903' => "Montesinos, Los",
			'03091' => "Murla",
			'03092' => "Muro de Alcoy",
			'03090' => "Mutxamel",
			'03093' => "Novelda",
			'03094' => "Nucia, la",
			'03095' => "Ondara",
			'03096' => "Onil",
			'03097' => "Orba",
			'03099' => "Orihuela",
			'03084' => "Orxa, l'/Lorcha",
			'03098' => "Orxeta",
			'03100' => "Parcent",
			'03101' => "Pedreguer",
			'03102' => "Pego",
			'03103' => "Penàguila",
			'03104' => "Petrer",
			'03902' => "Pilar de la Horadada",
			'03105' => "Pinós, el/Pinoso",
			'03106' => "Planes",
			'03042' => "Poble Nou de Benitatxell, el/Benitachell",
			'03901' => "Poblets, els",
			'03107' => "Polop",
			'03060' => "Quatretondeta",
			'03109' => "Rafal",
			'03110' => "Ràfol d'Almúnia, el",
			'03111' => "Redován",
			'03112' => "Relleu",
			'03113' => "Rojales",
			'03114' => "Romana, la",
			'03115' => "Sagra",
			'03116' => "Salinas",
			'03118' => "San Fulgencio",
			'03904' => "San Isidro",
			'03120' => "San Miguel de Salinas",
			'03117' => "Sanet y Negrals",
			'03119' => "Sant Joan d'Alacant",
			'03122' => "Sant Vicent del Raspeig/San Vicente del Raspeig",
			'03121' => "Santa Pola",
			'03123' => "Sax",
			'03124' => "Sella",
			'03125' => "Senija",
			'03127' => "Tàrbena",
			'03128' => "Teulada",
			'03129' => "Tibi",
			'03130' => "Tollos",
			'03131' => "Tormos",
			'03132' => "Torre de les Maçanes, la/Torremanzanas",
			'03133' => "Torrevieja",
			'03134' => "Vall d'Alcalà, la",
			'03136' => "Vall de Gallinera, la",
			'03137' => "Vall de Laguar, la",
			'03135' => "Vall d'Ebo, la",
			'03138' => "Verger, el",
			'03139' => "Vila Joiosa, la/Villajoyosa",
			'03140' => "Villena",
			'03082' => "Xàbia/Jávea",
			'03081' => "Xaló",
			'03083' => "Xixona/Jijona",
			'04001' => "Abla",
			'04002' => "Abrucena",
			'04003' => "Adra",
			'04004' => "Albanchez",
			'04005' => "Alboloduy",
			'04006' => "Albox",
			'04007' => "Alcolea",
			'04008' => "Alcóntar",
			'04009' => "Alcudia de Monteagud",
			'04010' => "Alhabia",
			'04011' => "Alhama de Almería",
			'04012' => "Alicún",
			'04013' => "Almería",
			'04014' => "Almócita",
			'04015' => "Alsodux",
			'04016' => "Antas",
			'04017' => "Arboleas",
			'04018' => "Armuña de Almanzora",
			'04019' => "Bacares",
			'04904' => "Balanegra",
			'04020' => "Bayárcal",
			'04021' => "Bayarque",
			'04022' => "Bédar",
			'04023' => "Beires",
			'04024' => "Benahadux",
			'04026' => "Benitagla",
			'04027' => "Benizalón",
			'04028' => "Bentarique",
			'04029' => "Berja",
			'04030' => "Canjáyar",
			'04031' => "Cantoria",
			'04032' => "Carboneras",
			'04033' => "Castro de Filabres",
			'04036' => "Chercos",
			'04037' => "Chirivel",
			'04034' => "Cóbdar",
			'04035' => "Cuevas del Almanzora",
			'04038' => "Dalías",
			'04902' => "Ejido, El",
			'04041' => "Enix",
			'04043' => "Felix",
			'04044' => "Fines",
			'04045' => "Fiñana",
			'04046' => "Fondón",
			'04047' => "Gádor",
			'04048' => "Gallardos, Los",
			'04049' => "Garrucha",
			'04050' => "Gérgal",
			'04051' => "Huécija",
			'04052' => "Huércal de Almería",
			'04053' => "Huércal-Overa",
			'04054' => "Íllar",
			'04055' => "Instinción",
			'04056' => "Laroya",
			'04057' => "Láujar de Andarax",
			'04058' => "Líjar",
			'04059' => "Lubrín",
			'04060' => "Lucainena de las Torres",
			'04061' => "Lúcar",
			'04062' => "Macael",
			'04063' => "María",
			'04064' => "Mojácar",
			'04903' => "Mojonera, La",
			'04065' => "Nacimiento",
			'04066' => "Níjar",
			'04067' => "Ohanes",
			'04068' => "Olula de Castro",
			'04069' => "Olula del Río",
			'04070' => "Oria",
			'04071' => "Padules",
			'04072' => "Partaloa",
			'04073' => "Paterna del Río",
			'04074' => "Pechina",
			'04075' => "Pulpí",
			'04076' => "Purchena",
			'04077' => "Rágol",
			'04078' => "Rioja",
			'04079' => "Roquetas de Mar",
			'04080' => "Santa Cruz de Marchena",
			'04081' => "Santa Fe de Mondújar",
			'04082' => "Senés",
			'04083' => "Serón",
			'04084' => "Sierro",
			'04085' => "Somontín",
			'04086' => "Sorbas",
			'04087' => "Suflí",
			'04088' => "Tabernas",
			'04089' => "Taberno",
			'04090' => "Tahal",
			'04091' => "Terque",
			'04092' => "Tíjola",
			'04901' => "Tres Villas, Las",
			'04093' => "Turre",
			'04094' => "Turrillas",
			'04095' => "Uleila del Campo",
			'04096' => "Urrácal",
			'04097' => "Velefique",
			'04098' => "Vélez-Blanco",
			'04099' => "Vélez-Rubio",
			'04100' => "Vera",
			'04101' => "Viator",
			'04102' => "Vícar",
			'04103' => "Zurgena",
			'05001' => "Adanero",
			'05002' => "Adrada, La",
			'05005' => "Albornos",
			'05007' => "Aldeanueva de Santa Cruz",
			'05008' => "Aldeaseca",
			'05010' => "Aldehuela, La",
			'05012' => "Amavida",
			'05013' => "Arenal, El",
			'05014' => "Arenas de San Pedro",
			'05015' => "Arevalillo",
			'05016' => "Arévalo",
			'05017' => "Aveinte",
			'05018' => "Avellaneda",
			'05019' => "Ávila",
			'05021' => "Barco de Ávila, El",
			'05022' => "Barraco, El",
			'05023' => "Barromán",
			'05024' => "Becedas",
			'05025' => "Becedillas",
			'05026' => "Bercial de Zapardiel",
			'05027' => "Berlanas, Las",
			'05029' => "Bernuy-Zapardiel",
			'05030' => "Berrocalejo de Aragona",
			'05033' => "Blascomillán",
			'05034' => "Blasconuño de Matacabras",
			'05035' => "Blascosancho",
			'05036' => "Bohodón, El",
			'05037' => "Bohoyo",
			'05038' => "Bonilla de la Sierra",
			'05039' => "Brabos",
			'05040' => "Bularros",
			'05041' => "Burgohondo",
			'05042' => "Cabezas de Alambre",
			'05043' => "Cabezas del Pozo",
			'05044' => "Cabezas del Villar",
			'05045' => "Cabizuela",
			'05046' => "Canales",
			'05047' => "Candeleda",
			'05048' => "Cantiveros",
			'05049' => "Cardeñosa",
			'05051' => "Carrera, La",
			'05052' => "Casas del Puerto",
			'05053' => "Casasola",
			'05054' => "Casavieja",
			'05055' => "Casillas",
			'05056' => "Castellanos de Zapardiel",
			'05057' => "Cebreros",
			'05058' => "Cepeda la Mora",
			'05067' => "Chamartín",
			'05059' => "Cillán",
			'05060' => "Cisla",
			'05061' => "Colilla, La",
			'05062' => "Collado de Contreras",
			'05063' => "Collado del Mirón",
			'05064' => "Constanzana",
			'05065' => "Crespos",
			'05066' => "Cuevas del Valle",
			'05903' => "Diego del Carpio",
			'05069' => "Donjimeno",
			'05070' => "Donvidas",
			'05072' => "Espinosa de los Caballeros",
			'05073' => "Flores de Ávila",
			'05074' => "Fontiveros",
			'05075' => "Fresnedilla",
			'05076' => "Fresno, El",
			'05077' => "Fuente el Saúz",
			'05078' => "Fuentes de Año",
			'05079' => "Gallegos de Altamiros",
			'05080' => "Gallegos de Sobrinos",
			'05081' => "Garganta del Villar",
			'05082' => "Gavilanes",
			'05083' => "Gemuño",
			'05085' => "Gil García",
			'05084' => "Gilbuena",
			'05086' => "Gimialcón",
			'05087' => "Gotarrendura",
			'05088' => "Grandes y San Martín",
			'05089' => "Guisando",
			'05090' => "Gutierre-Muñoz",
			'05092' => "Hernansancho",
			'05093' => "Herradón de Pinares",
			'05094' => "Herreros de Suso",
			'05095' => "Higuera de las Dueñas",
			'05096' => "Hija de Dios, La",
			'05097' => "Horcajada, La",
			'05099' => "Horcajo de las Torres",
			'05100' => "Hornillo, El",
			'05102' => "Hoyo de Pinares, El",
			'05101' => "Hoyocasero",
			'05103' => "Hoyorredondo",
			'05106' => "Hoyos de Miguel Muñoz",
			'05104' => "Hoyos del Collado",
			'05105' => "Hoyos del Espino",
			'05107' => "Hurtumpascual",
			'05108' => "Junciana",
			'05109' => "Langa",
			'05110' => "Lanzahíta",
			'05113' => "Llanos de Tormes, Los",
			'05112' => "Losar del Barco, El",
			'05114' => "Madrigal de las Altas Torres",
			'05115' => "Maello",
			'05116' => "Malpartida de Corneja",
			'05117' => "Mamblas",
			'05118' => "Mancera de Arriba",
			'05119' => "Manjabálago y Ortigosa de Rioalmar",
			'05120' => "Marlín",
			'05121' => "Martiherrero",
			'05122' => "Martínez",
			'05123' => "Mediana de Voltoya",
			'05124' => "Medinilla",
			'05125' => "Mengamuñoz",
			'05126' => "Mesegar de Corneja",
			'05127' => "Mijares",
			'05128' => "Mingorría",
			'05129' => "Mirón, El",
			'05130' => "Mironcillo",
			'05131' => "Mirueña de los Infanzones",
			'05132' => "Mombeltrán",
			'05133' => "Monsalupe",
			'05134' => "Moraleja de Matacabras",
			'05135' => "Muñana",
			'05136' => "Muñico",
			'05138' => "Muñogalindo",
			'05139' => "Muñogrande",
			'05140' => "Muñomer del Peco",
			'05141' => "Muñopepe",
			'05142' => "Muñosancho",
			'05143' => "Muñotello",
			'05144' => "Narrillos del Álamo",
			'05145' => "Narrillos del Rebollar",
			'05149' => "Narros de Saldueña",
			'05147' => "Narros del Castillo",
			'05148' => "Narros del Puerto",
			'05152' => "Nava de Arévalo",
			'05153' => "Nava del Barco",
			'05151' => "Navacepedilla de Corneja",
			'05154' => "Navadijos",
			'05155' => "Navaescurial",
			'05156' => "Navahondilla",
			'05157' => "Navalacruz",
			'05158' => "Navalmoral",
			'05159' => "Navalonguilla",
			'05160' => "Navalosa",
			'05161' => "Navalperal de Pinares",
			'05162' => "Navalperal de Tormes",
			'05163' => "Navaluenga",
			'05164' => "Navaquesera",
			'05165' => "Navarredonda de Gredos",
			'05166' => "Navarredondilla",
			'05167' => "Navarrevisca",
			'05168' => "Navas del Marqués, Las",
			'05169' => "Navatalgordo",
			'05170' => "Navatejares",
			'05171' => "Neila de San Miguel",
			'05172' => "Niharra",
			'05173' => "Ojos-Albos",
			'05174' => "Orbita",
			'05175' => "Oso, El",
			'05176' => "Padiernos",
			'05177' => "Pajares de Adaja",
			'05178' => "Palacios de Goda",
			'05179' => "Papatrigo",
			'05180' => "Parral, El",
			'05181' => "Pascualcobo",
			'05182' => "Pedro Bernardo",
			'05183' => "Pedro-Rodríguez",
			'05184' => "Peguerinos",
			'05185' => "Peñalba de Ávila",
			'05186' => "Piedrahíta",
			'05187' => "Piedralaves",
			'05188' => "Poveda",
			'05189' => "Poyales del Hoyo",
			'05190' => "Pozanco",
			'05191' => "Pradosegar",
			'05192' => "Puerto Castilla",
			'05193' => "Rasueros",
			'05194' => "Riocabado",
			'05195' => "Riofrío",
			'05196' => "Rivilla de Barajas",
			'05197' => "Salobral",
			'05198' => "Salvadiós",
			'05199' => "San Bartolomé de Béjar",
			'05200' => "San Bartolomé de Corneja",
			'05201' => "San Bartolomé de Pinares",
			'05206' => "San Esteban de los Patos",
			'05208' => "San Esteban de Zapardiel",
			'05207' => "San Esteban del Valle",
			'05209' => "San García de Ingelmos",
			'05901' => "San Juan de Gredos",
			'05210' => "San Juan de la Encinilla",
			'05211' => "San Juan de la Nava",
			'05212' => "San Juan del Molinillo",
			'05213' => "San Juan del Olmo",
			'05214' => "San Lorenzo de Tormes",
			'05215' => "San Martín de la Vega del Alberche",
			'05216' => "San Martín del Pimpollar",
			'05217' => "San Miguel de Corneja",
			'05218' => "San Miguel de Serrezuela",
			'05219' => "San Pascual",
			'05220' => "San Pedro del Arroyo",
			'05231' => "San Vicente de Arévalo",
			'05204' => "Sanchidrián",
			'05205' => "Sanchorreja",
			'05222' => "Santa Cruz de Pinares",
			'05221' => "Santa Cruz del Valle",
			'05226' => "Santa María de los Caballeros",
			'05224' => "Santa María del Arroyo",
			'05225' => "Santa María del Berrocal",
			'05902' => "Santa María del Cubillo",
			'05227' => "Santa María del Tiétar",
			'05228' => "Santiago del Collado",
			'05904' => "Santiago del Tormes",
			'05229' => "Santo Domingo de las Posadas",
			'05230' => "Santo Tomé de Zabarcos",
			'05232' => "Serrada, La",
			'05233' => "Serranillos",
			'05234' => "Sigeres",
			'05235' => "Sinlabajos",
			'05236' => "Solana de Ávila",
			'05237' => "Solana de Rioalmar",
			'05238' => "Solosancho",
			'05239' => "Sotalbo",
			'05240' => "Sotillo de la Adrada",
			'05241' => "Tiemblo, El",
			'05242' => "Tiñosillos",
			'05243' => "Tolbaños",
			'05244' => "Tormellas",
			'05245' => "Tornadizos de Ávila",
			'05247' => "Torre, La",
			'05246' => "Tórtoles",
			'05249' => "Umbrías",
			'05251' => "Vadillo de la Sierra",
			'05252' => "Valdecasa",
			'05253' => "Vega de Santa María",
			'05254' => "Velayos",
			'05256' => "Villaflor",
			'05257' => "Villafranca de la Sierra",
			'05905' => "Villanueva de Ávila",
			'05258' => "Villanueva de Gómez",
			'05259' => "Villanueva del Aceral",
			'05260' => "Villanueva del Campillo",
			'05261' => "Villar de Corneja",
			'05262' => "Villarejo del Valle",
			'05263' => "Villatoro",
			'05264' => "Viñegra de Moraña",
			'05265' => "Vita",
			'05266' => "Zapardiel de la Cañada",
			'05267' => "Zapardiel de la Ribera",
			'06001' => "Acedera",
			'06002' => "Aceuchal",
			'06003' => "Ahillones",
			'06004' => "Alange",
			'06005' => "Albuera, La",
			'06006' => "Alburquerque",
			'06007' => "Alconchel",
			'06008' => "Alconera",
			'06009' => "Aljucén",
			'06010' => "Almendral",
			'06011' => "Almendralejo",
			'06012' => "Arroyo de San Serván",
			'06013' => "Atalaya",
			'06014' => "Azuaga",
			'06015' => "Badajoz",
			'06016' => "Barcarrota",
			'06017' => "Baterno",
			'06018' => "Benquerencia de la Serena",
			'06019' => "Berlanga",
			'06020' => "Bienvenida",
			'06021' => "Bodonal de la Sierra",
			'06022' => "Burguillos del Cerro",
			'06023' => "Cabeza del Buey",
			'06024' => "Cabeza la Vaca",
			'06025' => "Calamonte",
			'06026' => "Calera de León",
			'06027' => "Calzadilla de los Barros",
			'06028' => "Campanario",
			'06029' => "Campillo de Llerena",
			'06030' => "Capilla",
			'06031' => "Carmonita",
			'06032' => "Carrascalejo, El",
			'06033' => "Casas de Don Pedro",
			'06034' => "Casas de Reina",
			'06035' => "Castilblanco",
			'06036' => "Castuera",
			'06042' => "Cheles",
			'06037' => "Codosera, La",
			'06038' => "Cordobilla de Lácara",
			'06039' => "Coronada, La",
			'06040' => "Corte de Peleas",
			'06041' => "Cristina",
			'06043' => "Don Álvaro",
			'06044' => "Don Benito",
			'06045' => "Entrín Bajo",
			'06046' => "Esparragalejo",
			'06047' => "Esparragosa de la Serena",
			'06048' => "Esparragosa de Lares",
			'06049' => "Feria",
			'06050' => "Fregenal de la Sierra",
			'06051' => "Fuenlabrada de los Montes",
			'06052' => "Fuente de Cantos",
			'06053' => "Fuente del Arco",
			'06054' => "Fuente del Maestre",
			'06055' => "Fuentes de León",
			'06056' => "Garbayuela",
			'06057' => "Garlitos",
			'06058' => "Garrovilla, La",
			'06059' => "Granja de Torrehermosa",
			'06903' => "Guadiana",
			'06060' => "Guareña",
			'06061' => "Haba, La",
			'06062' => "Helechosa de los Montes",
			'06063' => "Herrera del Duque",
			'06064' => "Higuera de la Serena",
			'06065' => "Higuera de Llerena",
			'06066' => "Higuera de Vargas",
			'06067' => "Higuera la Real",
			'06068' => "Hinojosa del Valle",
			'06069' => "Hornachos",
			'06070' => "Jerez de los Caballeros",
			'06071' => "Lapa, La",
			'06073' => "Llera",
			'06074' => "Llerena",
			'06072' => "Lobón",
			'06075' => "Magacela",
			'06076' => "Maguilla",
			'06077' => "Malcocinado",
			'06078' => "Malpartida de la Serena",
			'06079' => "Manchita",
			'06080' => "Medellín",
			'06081' => "Medina de las Torres",
			'06082' => "Mengabril",
			'06083' => "Mérida",
			'06084' => "Mirandilla",
			'06085' => "Monesterio",
			'06086' => "Montemolín",
			'06087' => "Monterrubio de la Serena",
			'06088' => "Montijo",
			'06089' => "Morera, La",
			'06090' => "Nava de Santiago, La",
			'06091' => "Navalvillar de Pela",
			'06092' => "Nogales",
			'06093' => "Oliva de la Frontera",
			'06094' => "Oliva de Mérida",
			'06095' => "Olivenza",
			'06096' => "Orellana de la Sierra",
			'06097' => "Orellana la Vieja",
			'06098' => "Palomas",
			'06099' => "Parra, La",
			'06100' => "Peñalsordo",
			'06101' => "Peraleda del Zaucejo",
			'06102' => "Puebla de Alcocer",
			'06103' => "Puebla de la Calzada",
			'06104' => "Puebla de la Reina",
			'06107' => "Puebla de Obando",
			'06108' => "Puebla de Sancho Pérez",
			'06105' => "Puebla del Maestre",
			'06106' => "Puebla del Prior",
			'06902' => "Pueblonuevo del Guadiana",
			'06109' => "Quintana de la Serena",
			'06110' => "Reina",
			'06111' => "Rena",
			'06112' => "Retamal de Llerena",
			'06113' => "Ribera del Fresno",
			'06114' => "Risco",
			'06115' => "Roca de la Sierra, La",
			'06116' => "Salvaleón",
			'06117' => "Salvatierra de los Barros",
			'06119' => "San Pedro de Mérida",
			'06123' => "San Vicente de Alcántara",
			'06118' => "Sancti-Spíritus",
			'06120' => "Santa Amalia",
			'06121' => "Santa Marta",
			'06122' => "Santos de Maimona, Los",
			'06124' => "Segura de León",
			'06125' => "Siruela",
			'06126' => "Solana de los Barros",
			'06127' => "Talarrubias",
			'06128' => "Talavera la Real",
			'06129' => "Táliga",
			'06130' => "Tamurejo",
			'06131' => "Torre de Miguel Sesmero",
			'06132' => "Torremayor",
			'06133' => "Torremejía",
			'06134' => "Trasierra",
			'06135' => "Trujillanos",
			'06136' => "Usagre",
			'06137' => "Valdecaballeros",
			'06901' => "Valdelacalzada",
			'06138' => "Valdetorres",
			'06139' => "Valencia de las Torres",
			'06140' => "Valencia del Mombuey",
			'06141' => "Valencia del Ventoso",
			'06146' => "Valle de la Serena",
			'06147' => "Valle de Matamoros",
			'06148' => "Valle de Santa Ana",
			'06142' => "Valverde de Burguillos",
			'06143' => "Valverde de Leganés",
			'06144' => "Valverde de Llerena",
			'06145' => "Valverde de Mérida",
			'06149' => "Villafranca de los Barros",
			'06150' => "Villagarcía de la Torre",
			'06151' => "Villagonzalo",
			'06152' => "Villalba de los Barros",
			'06153' => "Villanueva de la Serena",
			'06154' => "Villanueva del Fresno",
			'06156' => "Villar de Rena",
			'06155' => "Villar del Rey",
			'06157' => "Villarta de los Montes",
			'06158' => "Zafra",
			'06159' => "Zahínos",
			'06160' => "Zalamea de la Serena",
			'06162' => "Zarza, La",
			'06161' => "Zarza-Capilla",
			'07002' => "Alaior",
			'07001' => "Alaró",
			'07003' => "Alcúdia",
			'07004' => "Algaida",
			'07005' => "Andratx",
			'07901' => "Ariany",
			'07006' => "Artà",
			'07007' => "Banyalbufar",
			'07008' => "Binissalem",
			'07009' => "Búger",
			'07010' => "Bunyola",
			'07011' => "Calvià",
			'07012' => "Campanet",
			'07013' => "Campos",
			'07014' => "Capdepera",
			'07064' => "Castell, Es",
			'07015' => "Ciutadella de Menorca",
			'07016' => "Consell",
			'07017' => "Costitx",
			'07018' => "Deià",
			'07026' => "Eivissa",
			'07019' => "Escorca",
			'07020' => "Esporles",
			'07021' => "Estellencs",
			'07022' => "Felanitx",
			'07023' => "Ferreries",
			'07024' => "Formentera",
			'07025' => "Fornalutx",
			'07027' => "Inca",
			'07028' => "Lloret de Vistalegre",
			'07029' => "Lloseta",
			'07030' => "Llubí",
			'07031' => "Llucmajor",
			'07033' => "Manacor",
			'07034' => "Mancor de la Vall",
			'07032' => "Maó",
			'07035' => "Maria de la Salut",
			'07036' => "Marratxí",
			'07037' => "Mercadal, Es",
			'07902' => "Migjorn Gran, Es",
			'07038' => "Montuïri",
			'07039' => "Muro",
			'07040' => "Palma",
			'07041' => "Petra",
			'07044' => "Pobla, Sa",
			'07042' => "Pollença",
			'07043' => "Porreres",
			'07045' => "Puigpunyent",
			'07059' => "Salines, Ses",
			'07046' => "Sant Antoni de Portmany",
			'07049' => "Sant Joan",
			'07050' => "Sant Joan de Labritja",
			'07048' => "Sant Josep de sa Talaia",
			'07051' => "Sant Llorenç des Cardassar",
			'07052' => "Sant Lluís",
			'07053' => "Santa Eugènia",
			'07054' => "Santa Eulària des Riu",
			'07055' => "Santa Margalida",
			'07056' => "Santa María del Camí",
			'07057' => "Santanyí",
			'07058' => "Selva",
			'07047' => "Sencelles",
			'07060' => "Sineu",
			'07061' => "Sóller",
			'07062' => "Son Servera",
			'07063' => "Valldemossa",
			'07065' => "Vilafranca de Bonany",
			'08001' => "Abrera",
			'08002' => "Aguilar de Segarra",
			'08014' => "Aiguafreda",
			'08003' => "Alella",
			'08004' => "Alpens",
			'08005' => "Ametlla del Vallès, L'",
			'08006' => "Arenys de Mar",
			'08007' => "Arenys de Munt",
			'08008' => "Argençola",
			'08009' => "Argentona",
			'08010' => "Artés",
			'08011' => "Avià",
			'08012' => "Avinyó",
			'08013' => "Avinyonet del Penedès",
			'08015' => "Badalona",
			'08904' => "Badia del Vallès",
			'08016' => "Bagà",
			'08017' => "Balenyà",
			'08018' => "Balsareny",
			'08252' => "Barberà del Vallès",
			'08019' => "Barcelona",
			'08020' => "Begues",
			'08021' => "Bellprat",
			'08022' => "Berga",
			'08023' => "Bigues i Riells del Fai",
			'08024' => "Borredà",
			'08025' => "Bruc, El",
			'08026' => "Brull, El",
			'08027' => "Cabanyes, Les",
			'08028' => "Cabrera d'Anoia",
			'08029' => "Cabrera de Mar",
			'08030' => "Cabrils",
			'08031' => "Calaf",
			'08034' => "Calders",
			'08033' => "Caldes de Montbui",
			'08032' => "Caldes d'Estrac",
			'08035' => "Calella",
			'08037' => "Calldetenes",
			'08038' => "Callús",
			'08036' => "Calonge de Segarra",
			'08039' => "Campins",
			'08040' => "Canet de Mar",
			'08041' => "Canovelles",
			'08042' => "Cànoves i Samalús",
			'08043' => "Canyelles",
			'08044' => "Capellades",
			'08045' => "Capolat",
			'08046' => "Cardedeu",
			'08047' => "Cardona",
			'08048' => "Carme",
			'08049' => "Casserres",
			'08057' => "Castell de l'Areny",
			'08052' => "Castellar de n'Hug",
			'08050' => "Castellar del Riu",
			'08051' => "Castellar del Vallès",
			'08053' => "Castellbell i el Vilar",
			'08054' => "Castellbisbal",
			'08055' => "Castellcir",
			'08056' => "Castelldefels",
			'08058' => "Castellet i la Gornal",
			'08060' => "Castellfollit de Riubregós",
			'08059' => "Castellfollit del Boix",
			'08061' => "Castellgalí",
			'08062' => "Castellnou de Bages",
			'08063' => "Castellolí",
			'08064' => "Castellterçol",
			'08065' => "Castellví de la Marca",
			'08066' => "Castellví de Rosanes",
			'08067' => "Centelles",
			'08268' => "Cercs",
			'08266' => "Cerdanyola del Vallès",
			'08068' => "Cervelló",
			'08069' => "Collbató",
			'08070' => "Collsuspina",
			'08071' => "Copons",
			'08072' => "Corbera de Llobregat",
			'08073' => "Cornellà de Llobregat",
			'08074' => "Cubelles",
			'08075' => "Dosrius",
			'08076' => "Esparreguera",
			'08077' => "Esplugues de Llobregat",
			'08078' => "Espunyola, L'",
			'08254' => "Esquirol, L'",
			'08079' => "Estany, L'",
			'08134' => "Figaró-Montmany",
			'08080' => "Fígols",
			'08082' => "Fogars de la Selva",
			'08081' => "Fogars de Montclús",
			'08083' => "Folgueroles",
			'08084' => "Fonollosa",
			'08085' => "Font-rubí",
			'08086' => "Franqueses del Vallès, Les",
			'08090' => "Gaià",
			'08087' => "Gallifa",
			'08088' => "Garriga, La",
			'08089' => "Gavà",
			'08091' => "Gelida",
			'08092' => "Gironella",
			'08093' => "Gisclareny",
			'08094' => "Granada, La",
			'08095' => "Granera",
			'08096' => "Granollers",
			'08097' => "Gualba",
			'08099' => "Guardiola de Berguedà",
			'08100' => "Gurb",
			'08101' => "Hospitalet de Llobregat, L'",
			'08162' => "Hostalets de Pierola, Els",
			'08102' => "Igualada",
			'08103' => "Jorba",
			'08104' => "Llacuna, La",
			'08105' => "Llagosta, La",
			'08107' => "Lliçà d'Amunt",
			'08108' => "Lliçà de Vall",
			'08106' => "Llinars del Vallès",
			'08109' => "Lluçà",
			'08110' => "Malgrat de Mar",
			'08111' => "Malla",
			'08112' => "Manlleu",
			'08113' => "Manresa",
			'08242' => "Marganell",
			'08114' => "Martorell",
			'08115' => "Martorelles",
			'08116' => "Masies de Roda, Les",
			'08117' => "Masies de Voltregà, Les",
			'08118' => "Masnou, El",
			'08119' => "Masquefa",
			'08120' => "Matadepera",
			'08121' => "Mataró",
			'08122' => "Mediona",
			'08138' => "Moià",
			'08123' => "Molins de Rei",
			'08124' => "Mollet del Vallès",
			'08128' => "Monistrol de Calders",
			'08127' => "Monistrol de Montserrat",
			'08125' => "Montcada i Reixac",
			'08130' => "Montclar",
			'08131' => "Montesquiu",
			'08126' => "Montgat",
			'08132' => "Montmajor",
			'08133' => "Montmaneu",
			'08135' => "Montmeló",
			'08136' => "Montornès del Vallès",
			'08137' => "Montseny",
			'08129' => "Muntanyola",
			'08139' => "Mura",
			'08140' => "Navarcles",
			'08141' => "Navàs",
			'08142' => "Nou de Berguedà, La",
			'08143' => "Òdena",
			'08145' => "Olèrdola",
			'08146' => "Olesa de Bonesvalls",
			'08147' => "Olesa de Montserrat",
			'08148' => "Olivella",
			'08149' => "Olost",
			'08144' => "Olvan",
			'08150' => "Orís",
			'08151' => "Oristà",
			'08152' => "Orpí",
			'08153' => "Òrrius",
			'08154' => "Pacs del Penedès",
			'08155' => "Palafolls",
			'08156' => "Palau-solità i Plegamans",
			'08157' => "Pallejà",
			'08905' => "Palma de Cervelló, La",
			'08158' => "Papiol, El",
			'08159' => "Parets del Vallès",
			'08160' => "Perafita",
			'08161' => "Piera",
			'08163' => "Pineda de Mar",
			'08164' => "Pla del Penedès, El",
			'08165' => "Pobla de Claramunt, La",
			'08166' => "Pobla de Lillet, La",
			'08167' => "Polinyà",
			'08182' => "Pont de Vilomara i Rocafort, El",
			'08168' => "Pontons",
			'08169' => "Prat de Llobregat, El",
			'08171' => "Prats de Lluçanès",
			'08170' => "Prats de Rei, Els",
			'08230' => "Premià de Dalt",
			'08172' => "Premià de Mar",
			'08174' => "Puigdàlber",
			'08175' => "Puig-reig",
			'08176' => "Pujalt",
			'08177' => "Quar, La",
			'08178' => "Rajadell",
			'08179' => "Rellinars",
			'08180' => "Ripollet",
			'08181' => "Roca del Vallès, La",
			'08183' => "Roda de Ter",
			'08184' => "Rubí",
			'08185' => "Rubió",
			'08901' => "Rupit i Pruit",
			'08187' => "Sabadell",
			'08188' => "Sagàs",
			'08190' => "Saldes",
			'08191' => "Sallent",
			'08194' => "Sant Adrià de Besòs",
			'08195' => "Sant Agustí de Lluçanès",
			'08196' => "Sant Andreu de la Barca",
			'08197' => "Sant Andreu de Llavaneres",
			'08198' => "Sant Antoni de Vilamajor",
			'08199' => "Sant Bartomeu del Grau",
			'08200' => "Sant Boi de Llobregat",
			'08201' => "Sant Boi de Lluçanès",
			'08203' => "Sant Cebrià de Vallalta",
			'08202' => "Sant Celoni",
			'08204' => "Sant Climent de Llobregat",
			'08205' => "Sant Cugat del Vallès",
			'08206' => "Sant Cugat Sesgarrigues",
			'08207' => "Sant Esteve de Palautordera",
			'08208' => "Sant Esteve Sesrovires",
			'08210' => "Sant Feliu de Codines",
			'08211' => "Sant Feliu de Llobregat",
			'08212' => "Sant Feliu Sasserra",
			'08209' => "Sant Fost de Campsentelles",
			'08213' => "Sant Fruitós de Bages",
			'08215' => "Sant Hipòlit de Voltregà",
			'08193' => "Sant Iscle de Vallalta",
			'08216' => "Sant Jaume de Frontanyà",
			'08218' => "Sant Joan de Vilatorrada",
			'08217' => "Sant Joan Despí",
			'08903' => "Sant Julià de Cerdanyola",
			'08220' => "Sant Julià de Vilatorta",
			'08221' => "Sant Just Desvern",
			'08222' => "Sant Llorenç d'Hortons",
			'08223' => "Sant Llorenç Savall",
			'08225' => "Sant Martí d'Albars",
			'08224' => "Sant Martí de Centelles",
			'08226' => "Sant Martí de Tous",
			'08227' => "Sant Martí Sarroca",
			'08228' => "Sant Martí Sesgueioles",
			'08229' => "Sant Mateu de Bages",
			'08231' => "Sant Pere de Ribes",
			'08232' => "Sant Pere de Riudebitlles",
			'08233' => "Sant Pere de Torelló",
			'08234' => "Sant Pere de Vilamajor",
			'08189' => "Sant Pere Sallavinera",
			'08235' => "Sant Pol de Mar",
			'08236' => "Sant Quintí de Mediona",
			'08237' => "Sant Quirze de Besora",
			'08238' => "Sant Quirze del Vallès",
			'08239' => "Sant Quirze Safaja",
			'08240' => "Sant Sadurní d'Anoia",
			'08241' => "Sant Sadurní d'Osormort",
			'08098' => "Sant Salvador de Guardiola",
			'08262' => "Sant Vicenç de Castellet",
			'08264' => "Sant Vicenç de Montalt",
			'08265' => "Sant Vicenç de Torelló",
			'08263' => "Sant Vicenç dels Horts",
			'08243' => "Santa Cecília de Voltregà",
			'08244' => "Santa Coloma de Cervelló",
			'08245' => "Santa Coloma de Gramenet",
			'08246' => "Santa Eugènia de Berga",
			'08247' => "Santa Eulàlia de Riuprimer",
			'08248' => "Santa Eulàlia de Ronçana",
			'08249' => "Santa Fe del Penedès",
			'08250' => "Santa Margarida de Montbui",
			'08251' => "Santa Margarida i els Monjos",
			'08253' => "Santa Maria de Besora",
			'08256' => "Santa Maria de Martorelles",
			'08255' => "Santa Maria de Merlès",
			'08257' => "Santa Maria de Miralles",
			'08259' => "Santa Maria de Palautordera",
			'08258' => "Santa Maria d'Oló",
			'08260' => "Santa Perpètua de Mogoda",
			'08261' => "Santa Susanna",
			'08192' => "Santpedor",
			'08267' => "Sentmenat",
			'08269' => "Seva",
			'08270' => "Sitges",
			'08271' => "Sobremunt",
			'08272' => "Sora",
			'08273' => "Subirats",
			'08274' => "Súria",
			'08276' => "Tagamanent",
			'08277' => "Talamanca",
			'08278' => "Taradell",
			'08275' => "Tavèrnoles",
			'08280' => "Tavertet",
			'08281' => "Teià",
			'08279' => "Terrassa",
			'08282' => "Tiana",
			'08283' => "Tona",
			'08284' => "Tordera",
			'08285' => "Torelló",
			'08286' => "Torre de Claramunt, La",
			'08287' => "Torrelavit",
			'08288' => "Torrelles de Foix",
			'08289' => "Torrelles de Llobregat",
			'08290' => "Ullastrell",
			'08291' => "Vacarisses",
			'08292' => "Vallbona d'Anoia",
			'08293' => "Vallcebre",
			'08294' => "Vallgorguina",
			'08295' => "Vallirana",
			'08296' => "Vallromanes",
			'08297' => "Veciana",
			'08298' => "Vic",
			'08299' => "Vilada",
			'08301' => "Viladecans",
			'08300' => "Viladecavalls",
			'08305' => "Vilafranca del Penedès",
			'08306' => "Vilalba Sasserra",
			'08303' => "Vilanova de Sau",
			'08302' => "Vilanova del Camí",
			'08902' => "Vilanova del Vallès",
			'08307' => "Vilanova i la Geltrú",
			'08214' => "Vilassar de Dalt",
			'08219' => "Vilassar de Mar",
			'08304' => "Vilobí del Penedès",
			'08308' => "Viver i Serrateix",
			'09001' => "Abajas",
			'09003' => "Adrada de Haza",
			'09006' => "Aguas Cándidas",
			'09007' => "Aguilar de Bureba",
			'09009' => "Albillos",
			'09010' => "Alcocero de Mola",
			'09011' => "Alfoz de Bricia",
			'09907' => "Alfoz de Quintanadueñas",
			'09012' => "Alfoz de Santa Gadea",
			'09013' => "Altable",
			'09014' => "Altos, Los",
			'09016' => "Ameyugo",
			'09017' => "Anguix",
			'09018' => "Aranda de Duero",
			'09019' => "Arandilla",
			'09020' => "Arauzo de Miel",
			'09021' => "Arauzo de Salce",
			'09022' => "Arauzo de Torre",
			'09023' => "Arcos",
			'09024' => "Arenillas de Riopisuerga",
			'09025' => "Arija",
			'09026' => "Arlanzón",
			'09027' => "Arraya de Oca",
			'09029' => "Atapuerca",
			'09030' => "Ausines, Los",
			'09032' => "Avellanosa de Muñó",
			'09033' => "Bahabón de Esgueva",
			'09034' => "Balbases, Los",
			'09035' => "Baños de Valdearados",
			'09036' => "Bañuelos de Bureba",
			'09037' => "Barbadillo de Herreros",
			'09038' => "Barbadillo del Mercado",
			'09039' => "Barbadillo del Pez",
			'09041' => "Barrio de Muñó",
			'09043' => "Barrios de Bureba, Los",
			'09044' => "Barrios de Colina",
			'09045' => "Basconcillos del Tozo",
			'09046' => "Bascuñana",
			'09047' => "Belbimbre",
			'09048' => "Belorado",
			'09050' => "Berberana",
			'09051' => "Berlangas de Roa",
			'09052' => "Berzosa de Bureba",
			'09054' => "Bozoó",
			'09055' => "Brazacorta",
			'09056' => "Briviesca",
			'09057' => "Bugedo",
			'09058' => "Buniel",
			'09059' => "Burgos",
			'09060' => "Busto de Bureba",
			'09061' => "Cabañes de Esgueva",
			'09062' => "Cabezón de la Sierra",
			'09064' => "Caleruega",
			'09065' => "Campillo de Aranda",
			'09066' => "Campolara",
			'09067' => "Canicosa de la Sierra",
			'09068' => "Cantabrana",
			'09070' => "Carazo",
			'09071' => "Carcedo de Bureba",
			'09072' => "Carcedo de Burgos",
			'09073' => "Cardeñadijo",
			'09074' => "Cardeñajimeno",
			'09075' => "Cardeñuela Riopico",
			'09076' => "Carrias",
			'09077' => "Cascajares de Bureba",
			'09078' => "Cascajares de la Sierra",
			'09079' => "Castellanos de Castro",
			'09083' => "Castil de Peones",
			'09082' => "Castildelgado",
			'09084' => "Castrillo de la Reina",
			'09085' => "Castrillo de la Vega",
			'09088' => "Castrillo de Riopisuerga",
			'09086' => "Castrillo del Val",
			'09090' => "Castrillo Mota de Judíos",
			'09091' => "Castrojeriz",
			'09063' => "Cavia",
			'09093' => "Cayuela",
			'09094' => "Cebrecos",
			'09095' => "Celada del Camino",
			'09098' => "Cerezo de Río Tirón",
			'09100' => "Cerratón de Juarros",
			'09101' => "Ciadoncha",
			'09102' => "Cillaperlata",
			'09103' => "Cilleruelo de Abajo",
			'09104' => "Cilleruelo de Arriba",
			'09105' => "Ciruelos de Cervera",
			'09108' => "Cogollos",
			'09109' => "Condado de Treviño",
			'09110' => "Contreras",
			'09112' => "Coruña del Conde",
			'09113' => "Covarrubias",
			'09114' => "Cubillo del Campo",
			'09115' => "Cubo de Bureba",
			'09117' => "Cueva de Roa, La",
			'09119' => "Cuevas de San Clemente",
			'09120' => "Encío",
			'09122' => "Espinosa de Cervera",
			'09124' => "Espinosa de los Monteros",
			'09123' => "Espinosa del Camino",
			'09125' => "Estépar",
			'09127' => "Fontioso",
			'09128' => "Frandovínez",
			'09129' => "Fresneda de la Sierra Tirón",
			'09130' => "Fresneña",
			'09131' => "Fresnillo de las Dueñas",
			'09132' => "Fresno de Río Tirón",
			'09133' => "Fresno de Rodilla",
			'09134' => "Frías",
			'09135' => "Fuentebureba",
			'09136' => "Fuentecén",
			'09137' => "Fuentelcésped",
			'09138' => "Fuentelisendo",
			'09139' => "Fuentemolinos",
			'09140' => "Fuentenebro",
			'09141' => "Fuentespina",
			'09143' => "Galbarros",
			'09144' => "Gallega, La",
			'09148' => "Grijalba",
			'09149' => "Grisaleña",
			'09151' => "Gumiel de Izán",
			'09152' => "Gumiel de Mercado",
			'09154' => "Hacinas",
			'09155' => "Haza",
			'09159' => "Hontanas",
			'09160' => "Hontangas",
			'09162' => "Hontoria de la Cantera",
			'09164' => "Hontoria de Valdearados",
			'09163' => "Hontoria del Pinar",
			'09166' => "Hormazas, Las",
			'09167' => "Hornillos del Camino",
			'09168' => "Horra, La",
			'09169' => "Hortigüela",
			'09170' => "Hoyales de Roa",
			'09172' => "Huérmeces",
			'09173' => "Huerta de Arriba",
			'09174' => "Huerta de Rey",
			'09175' => "Humada",
			'09176' => "Hurones",
			'09177' => "Ibeas de Juarros",
			'09178' => "Ibrillos",
			'09179' => "Iglesiarrubia",
			'09180' => "Iglesias",
			'09181' => "Isar",
			'09182' => "Itero del Castillo",
			'09183' => "Jaramillo de la Fuente",
			'09184' => "Jaramillo Quemado",
			'09189' => "Junta de Traslaloma",
			'09190' => "Junta de Villalba de Losa",
			'09191' => "Jurisdicción de Lara",
			'09192' => "Jurisdicción de San Zadornil",
			'09194' => "Lerma",
			'09195' => "Llano de Bureba",
			'09196' => "Madrigal del Monte",
			'09197' => "Madrigalejo del Monte",
			'09198' => "Mahamud",
			'09199' => "Mambrilla de Castrejón",
			'09200' => "Mambrillas de Lara",
			'09201' => "Mamolar",
			'09202' => "Manciles",
			'09206' => "Mazuela",
			'09208' => "Mecerreyes",
			'09209' => "Medina de Pomar",
			'09211' => "Melgar de Fernamental",
			'09213' => "Merindad de Cuesta-Urria",
			'09214' => "Merindad de Montija",
			'09906' => "Merindad de Río Ubierna",
			'09215' => "Merindad de Sotoscueva",
			'09216' => "Merindad de Valdeporres",
			'09217' => "Merindad de Valdivielso",
			'09218' => "Milagros",
			'09219' => "Miranda de Ebro",
			'09220' => "Miraveche",
			'09221' => "Modúbar de la Emparedada",
			'09223' => "Monasterio de la Sierra",
			'09224' => "Monasterio de Rodilla",
			'09225' => "Moncalvillo",
			'09226' => "Monterrubio de la Demanda",
			'09227' => "Montorio",
			'09228' => "Moradillo de Roa",
			'09229' => "Nava de Roa",
			'09230' => "Navas de Bureba",
			'09231' => "Nebreda",
			'09232' => "Neila",
			'09235' => "Olmedillo de Roa",
			'09236' => "Olmillos de Muñó",
			'09238' => "Oña",
			'09239' => "Oquillas",
			'09241' => "Orbaneja Riopico",
			'09242' => "Padilla de Abajo",
			'09243' => "Padilla de Arriba",
			'09244' => "Padrones de Bureba",
			'09246' => "Palacios de la Sierra",
			'09247' => "Palacios de Riopisuerga",
			'09248' => "Palazuelos de la Sierra",
			'09249' => "Palazuelos de Muñó",
			'09250' => "Pampliega",
			'09251' => "Pancorbo",
			'09253' => "Pardilla",
			'09255' => "Partido de la Sierra en Tobalina",
			'09256' => "Pedrosa de Duero",
			'09259' => "Pedrosa de Río Úrbel",
			'09257' => "Pedrosa del Páramo",
			'09258' => "Pedrosa del Príncipe",
			'09261' => "Peñaranda de Duero",
			'09262' => "Peral de Arlanza",
			'09265' => "Piérnigas",
			'09266' => "Pineda de la Sierra",
			'09267' => "Pineda Trasmonte",
			'09268' => "Pinilla de los Barruecos",
			'09269' => "Pinilla de los Moros",
			'09270' => "Pinilla Trasmonte",
			'09272' => "Poza de la Sal",
			'09273' => "Prádanos de Bureba",
			'09274' => "Pradoluengo",
			'09275' => "Presencio",
			'09276' => "Puebla de Arganzón, La",
			'09277' => "Puentedura",
			'09279' => "Quemada",
			'09281' => "Quintana del Pidio",
			'09280' => "Quintanabureba",
			'09283' => "Quintanaélez",
			'09287' => "Quintanaortuño",
			'09288' => "Quintanapalla",
			'09289' => "Quintanar de la Sierra",
			'09292' => "Quintanavides",
			'09294' => "Quintanilla de la Mata",
			'09901' => "Quintanilla del Agua y Tordueles",
			'09295' => "Quintanilla del Coco",
			'09298' => "Quintanilla San García",
			'09301' => "Quintanilla Vivar",
			'09297' => "Quintanillas, Las",
			'09302' => "Rabanera del Pinar",
			'09303' => "Rábanos",
			'09304' => "Rabé de las Calzadas",
			'09306' => "Rebolledo de la Torre",
			'09307' => "Redecilla del Camino",
			'09308' => "Redecilla del Campo",
			'09309' => "Regumiel de la Sierra",
			'09310' => "Reinoso",
			'09311' => "Retuerta",
			'09314' => "Revilla del Campo",
			'09316' => "Revilla Vallejera",
			'09312' => "Revilla y Ahedo, La",
			'09315' => "Revillarruz",
			'09317' => "Rezmondo",
			'09318' => "Riocavado de la Sierra",
			'09321' => "Roa",
			'09323' => "Rojas",
			'09325' => "Royuela de Río Franco",
			'09326' => "Rubena",
			'09327' => "Rublacedo de Abajo",
			'09328' => "Rucandio",
			'09329' => "Salas de Bureba",
			'09330' => "Salas de los Infantes",
			'09332' => "Saldaña de Burgos",
			'09334' => "Salinillas de Bureba",
			'09335' => "San Adrián de Juarros",
			'09337' => "San Juan del Monte",
			'09338' => "San Mamés de Burgos",
			'09339' => "San Martín de Rubiales",
			'09340' => "San Millán de Lara",
			'09360' => "San Vicente del Valle",
			'09343' => "Santa Cecilia",
			'09345' => "Santa Cruz de la Salceda",
			'09346' => "Santa Cruz del Valle Urbión",
			'09347' => "Santa Gadea del Cid",
			'09348' => "Santa Inés",
			'09350' => "Santa María del Campo",
			'09351' => "Santa María del Invierno",
			'09352' => "Santa María del Mercadillo",
			'09353' => "Santa María Ribarredonda",
			'09354' => "Santa Olalla de Bureba",
			'09355' => "Santibáñez de Esgueva",
			'09356' => "Santibáñez del Val",
			'09358' => "Santo Domingo de Silos",
			'09361' => "Sargentes de la Lora",
			'09362' => "Sarracín",
			'09363' => "Sasamón",
			'09365' => "Sequera de Haza, La",
			'09366' => "Solarana",
			'09368' => "Sordillos",
			'09369' => "Sotillo de la Ribera",
			'09372' => "Sotragero",
			'09373' => "Sotresgudo",
			'09374' => "Susinos del Páramo",
			'09375' => "Tamarón",
			'09377' => "Tardajos",
			'09378' => "Tejada",
			'09380' => "Terradillos de Esgueva",
			'09381' => "Tinieblas de la Sierra",
			'09382' => "Tobar",
			'09384' => "Tordómar",
			'09386' => "Torrecilla del Monte",
			'09387' => "Torregalindo",
			'09388' => "Torrelara",
			'09389' => "Torrepadre",
			'09390' => "Torresandino",
			'09391' => "Tórtoles de Esgueva",
			'09392' => "Tosantos",
			'09394' => "Trespaderne",
			'09395' => "Tubilla del Agua",
			'09396' => "Tubilla del Lago",
			'09398' => "Úrbel del Castillo",
			'09400' => "Vadocondes",
			'09403' => "Valdeande",
			'09405' => "Valdezate",
			'09406' => "Valdorros",
			'09408' => "Vallarta de Bureba",
			'09904' => "Valle de las Navas",
			'09908' => "Valle de Losa",
			'09409' => "Valle de Manzanedo",
			'09410' => "Valle de Mena",
			'09411' => "Valle de Oca",
			'09902' => "Valle de Santibáñez",
			'09905' => "Valle de Sedano",
			'09412' => "Valle de Tobalina",
			'09413' => "Valle de Valdebezana",
			'09414' => "Valle de Valdelaguna",
			'09415' => "Valle de Valdelucio",
			'09416' => "Valle de Zamanzas",
			'09417' => "Vallejera",
			'09418' => "Valles de Palenzuela",
			'09419' => "Valluércanes",
			'09407' => "Valmala",
			'09422' => "Vid de Bureba, La",
			'09421' => "Vid y Barrios, La",
			'09423' => "Vileña",
			'09427' => "Villadiego",
			'09428' => "Villaescusa de Roa",
			'09429' => "Villaescusa la Sombría",
			'09430' => "Villaespasa",
			'09431' => "Villafranca Montes de Oca",
			'09432' => "Villafruela",
			'09433' => "Villagalijo",
			'09434' => "Villagonzalo Pedernales",
			'09437' => "Villahoz",
			'09438' => "Villalba de Duero",
			'09439' => "Villalbilla de Burgos",
			'09440' => "Villalbilla de Gumiel",
			'09441' => "Villaldemiro",
			'09442' => "Villalmanzo",
			'09443' => "Villamayor de los Montes",
			'09444' => "Villamayor de Treviño",
			'09445' => "Villambistia",
			'09446' => "Villamedianilla",
			'09447' => "Villamiel de la Sierra",
			'09448' => "Villangómez",
			'09449' => "Villanueva de Argaño",
			'09450' => "Villanueva de Carazo",
			'09451' => "Villanueva de Gumiel",
			'09454' => "Villanueva de Teba",
			'09455' => "Villaquirán de la Puebla",
			'09456' => "Villaquirán de los Infantes",
			'09903' => "Villarcayo de Merindad de Castilla la Vieja",
			'09458' => "Villariezo",
			'09460' => "Villasandino",
			'09463' => "Villasur de Herreros",
			'09464' => "Villatuelda",
			'09466' => "Villaverde del Monte",
			'09467' => "Villaverde-Mogina",
			'09471' => "Villayerno Morquillas",
			'09472' => "Villazopeque",
			'09473' => "Villegas",
			'09476' => "Villoruebo",
			'09424' => "Viloria de Rioja",
			'09425' => "Vilviestre del Pinar",
			'09478' => "Vizcaínos",
			'09480' => "Zael",
			'09482' => "Zarzosa de Río Pisuerga",
			'09483' => "Zazuar",
			'09485' => "Zuñeda",
			'10001' => "Abadía",
			'10002' => "Abertura",
			'10003' => "Acebo",
			'10004' => "Acehúche",
			'10005' => "Aceituna",
			'10006' => "Ahigal",
			'10903' => "Alagón del Río",
			'10007' => "Albalá",
			'10008' => "Alcántara",
			'10009' => "Alcollarín",
			'10010' => "Alcuéscar",
			'10012' => "Aldea del Cano",
			'10013' => "Aldea del Obispo, La",
			'10011' => "Aldeacentenera",
			'10014' => "Aldeanueva de la Vera",
			'10015' => "Aldeanueva del Camino",
			'10016' => "Aldehuela de Jerte",
			'10017' => "Alía",
			'10018' => "Aliseda",
			'10019' => "Almaraz",
			'10020' => "Almoharín",
			'10021' => "Arroyo de la Luz",
			'10023' => "Arroyomolinos",
			'10022' => "Arroyomolinos de la Vera",
			'10024' => "Baños de Montemayor",
			'10025' => "Barrado",
			'10026' => "Belvís de Monroy",
			'10027' => "Benquerencia",
			'10028' => "Berrocalejo",
			'10029' => "Berzocana",
			'10030' => "Bohonal de Ibor",
			'10031' => "Botija",
			'10032' => "Brozas",
			'10033' => "Cabañas del Castillo",
			'10034' => "Cabezabellosa",
			'10035' => "Cabezuela del Valle",
			'10036' => "Cabrero",
			'10037' => "Cáceres",
			'10038' => "Cachorrilla",
			'10039' => "Cadalso",
			'10040' => "Calzadilla",
			'10041' => "Caminomorisco",
			'10042' => "Campillo de Deleitosa",
			'10043' => "Campo Lugar",
			'10044' => "Cañamero",
			'10045' => "Cañaveral",
			'10046' => "Carbajo",
			'10047' => "Carcaboso",
			'10048' => "Carrascalejo",
			'10049' => "Casar de Cáceres",
			'10050' => "Casar de Palomero",
			'10051' => "Casares de las Hurdes",
			'10052' => "Casas de Don Antonio",
			'10053' => "Casas de Don Gómez",
			'10056' => "Casas de Millán",
			'10057' => "Casas de Miravete",
			'10054' => "Casas del Castañar",
			'10055' => "Casas del Monte",
			'10058' => "Casatejada",
			'10059' => "Casillas de Coria",
			'10060' => "Castañar de Ibor",
			'10061' => "Ceclavín",
			'10062' => "Cedillo",
			'10063' => "Cerezo",
			'10064' => "Cilleros",
			'10065' => "Collado de la Vera",
			'10066' => "Conquista de la Sierra",
			'10067' => "Coria",
			'10068' => "Cuacos de Yuste",
			'10069' => "Cumbre, La",
			'10070' => "Deleitosa",
			'10071' => "Descargamaría",
			'10072' => "Eljas",
			'10073' => "Escurial",
			'10075' => "Fresnedoso de Ibor",
			'10076' => "Galisteo",
			'10077' => "Garciaz",
			'10079' => "Garganta la Olla",
			'10078' => "Garganta, La",
			'10080' => "Gargantilla",
			'10081' => "Gargüera",
			'10082' => "Garrovillas de Alconétar",
			'10083' => "Garvín",
			'10084' => "Gata",
			'10085' => "Gordo, El",
			'10086' => "Granja, La",
			'10087' => "Guadalupe",
			'10088' => "Guijo de Coria",
			'10089' => "Guijo de Galisteo",
			'10090' => "Guijo de Granadilla",
			'10091' => "Guijo de Santa Bárbara",
			'10092' => "Herguijuela",
			'10093' => "Hernán-Pérez",
			'10094' => "Herrera de Alcántara",
			'10095' => "Herreruela",
			'10096' => "Hervás",
			'10097' => "Higuera de Albalat",
			'10098' => "Hinojal",
			'10099' => "Holguera",
			'10100' => "Hoyos",
			'10101' => "Huélaga",
			'10102' => "Ibahernando",
			'10103' => "Jaraicejo",
			'10104' => "Jaraíz de la Vera",
			'10105' => "Jarandilla de la Vera",
			'10106' => "Jarilla",
			'10107' => "Jerte",
			'10108' => "Ladrillar",
			'10109' => "Logrosán",
			'10110' => "Losar de la Vera",
			'10111' => "Madrigal de la Vera",
			'10112' => "Madrigalejo",
			'10113' => "Madroñera",
			'10114' => "Majadas",
			'10115' => "Malpartida de Cáceres",
			'10116' => "Malpartida de Plasencia",
			'10117' => "Marchagaz",
			'10118' => "Mata de Alcántara",
			'10119' => "Membrío",
			'10120' => "Mesas de Ibor",
			'10121' => "Miajadas",
			'10122' => "Millanes",
			'10123' => "Mirabel",
			'10124' => "Mohedas de Granadilla",
			'10125' => "Monroy",
			'10126' => "Montánchez",
			'10127' => "Montehermoso",
			'10128' => "Moraleja",
			'10129' => "Morcillo",
			'10130' => "Navaconcejo",
			'10131' => "Navalmoral de la Mata",
			'10132' => "Navalvillar de Ibor",
			'10133' => "Navas del Madroño",
			'10134' => "Navezuelas",
			'10135' => "Nuñomoral",
			'10136' => "Oliva de Plasencia",
			'10137' => "Palomero",
			'10138' => "Pasarón de la Vera",
			'10139' => "Pedroso de Acim",
			'10140' => "Peraleda de la Mata",
			'10141' => "Peraleda de San Román",
			'10142' => "Perales del Puerto",
			'10143' => "Pescueza",
			'10144' => "Pesga, La",
			'10145' => "Piedras Albas",
			'10146' => "Pinofranqueado",
			'10147' => "Piornal",
			'10148' => "Plasencia",
			'10149' => "Plasenzuela",
			'10150' => "Portaje",
			'10151' => "Portezuelo",
			'10152' => "Pozuelo de Zarzón",
			'10905' => "Pueblonuevo de Miramontes",
			'10153' => "Puerto de Santa Cruz",
			'10154' => "Rebollar",
			'10155' => "Riolobos",
			'10156' => "Robledillo de Gata",
			'10157' => "Robledillo de la Vera",
			'10158' => "Robledillo de Trujillo",
			'10159' => "Robledollano",
			'10160' => "Romangordo",
			'10901' => "Rosalejo",
			'10161' => "Ruanes",
			'10162' => "Salorino",
			'10163' => "Salvatierra de Santiago",
			'10164' => "San Martín de Trevejo",
			'10165' => "Santa Ana",
			'10166' => "Santa Cruz de la Sierra",
			'10167' => "Santa Cruz de Paniagua",
			'10168' => "Santa Marta de Magasca",
			'10169' => "Santiago de Alcántara",
			'10170' => "Santiago del Campo",
			'10171' => "Santibáñez el Alto",
			'10172' => "Santibáñez el Bajo",
			'10173' => "Saucedilla",
			'10174' => "Segura de Toro",
			'10175' => "Serradilla",
			'10176' => "Serrejón",
			'10177' => "Sierra de Fuentes",
			'10178' => "Talaván",
			'10179' => "Talaveruela de la Vera",
			'10180' => "Talayuela",
			'10181' => "Tejeda de Tiétar",
			'10904' => "Tiétar",
			'10182' => "Toril",
			'10183' => "Tornavacas",
			'10184' => "Torno, El",
			'10187' => "Torre de Don Miguel",
			'10188' => "Torre de Santa María",
			'10185' => "Torrecilla de los Ángeles",
			'10186' => "Torrecillas de la Tiesa",
			'10190' => "Torrejón el Rubio",
			'10189' => "Torrejoncillo",
			'10191' => "Torremenga",
			'10192' => "Torremocha",
			'10193' => "Torreorgaz",
			'10194' => "Torrequemada",
			'10195' => "Trujillo",
			'10196' => "Valdastillas",
			'10197' => "Valdecañas de Tajo",
			'10198' => "Valdefuentes",
			'10199' => "Valdehúncar",
			'10200' => "Valdelacasa de Tajo",
			'10201' => "Valdemorales",
			'10202' => "Valdeobispo",
			'10203' => "Valencia de Alcántara",
			'10204' => "Valverde de la Vera",
			'10205' => "Valverde del Fresno",
			'10902' => "Vegaviana",
			'10206' => "Viandar de la Vera",
			'10207' => "Villa del Campo",
			'10208' => "Villa del Rey",
			'10209' => "Villamesías",
			'10210' => "Villamiel",
			'10211' => "Villanueva de la Sierra",
			'10212' => "Villanueva de la Vera",
			'10214' => "Villar de Plasencia",
			'10213' => "Villar del Pedroso",
			'10215' => "Villasbuenas de Gata",
			'10216' => "Zarza de Granadilla",
			'10217' => "Zarza de Montánchez",
			'10218' => "Zarza la Mayor",
			'10219' => "Zorita",
			'11001' => "Alcalá de los Gazules",
			'11002' => "Alcalá del Valle",
			'11003' => "Algar",
			'11004' => "Algeciras",
			'11005' => "Algodonales",
			'11006' => "Arcos de la Frontera",
			'11007' => "Barbate",
			'11008' => "Barrios, Los",
			'11901' => "Benalup-Casas Viejas",
			'11009' => "Benaocaz",
			'11010' => "Bornos",
			'11011' => "Bosque, El",
			'11012' => "Cádiz",
			'11013' => "Castellar de la Frontera",
			'11015' => "Chiclana de la Frontera",
			'11016' => "Chipiona",
			'11014' => "Conil de la Frontera",
			'11017' => "Espera",
			'11018' => "Gastor, El",
			'11019' => "Grazalema",
			'11020' => "Jerez de la Frontera",
			'11021' => "Jimena de la Frontera",
			'11022' => "Línea de la Concepción, La",
			'11023' => "Medina Sidonia",
			'11024' => "Olvera",
			'11025' => "Paterna de Rivera",
			'11026' => "Prado del Rey",
			'11027' => "Puerto de Santa María, El",
			'11028' => "Puerto Real",
			'11029' => "Puerto Serrano",
			'11030' => "Rota",
			'11031' => "San Fernando",
			'11902' => "San José del Valle",
			'11903' => "San Martín del Tesorillo",
			'11033' => "San Roque",
			'11032' => "Sanlúcar de Barrameda",
			'11034' => "Setenil de las Bodegas",
			'11035' => "Tarifa",
			'11036' => "Torre Alháquime",
			'11037' => "Trebujena",
			'11038' => "Ubrique",
			'11039' => "Vejer de la Frontera",
			'11040' => "Villaluenga del Rosario",
			'11041' => "Villamartín",
			'11042' => "Zahara",
			'12002' => "Aín",
			'12003' => "Albocàsser",
			'12004' => "Alcalà de Xivert",
			'12005' => "Alcora, l'",
			'12006' => "Alcudia de Veo",
			'12007' => "Alfondeguilla",
			'12008' => "Algimia de Almonacid",
			'12009' => "Almassora",
			'12010' => "Almedíjar",
			'12011' => "Almenara",
			'12901' => "Alqueries, les/Alquerías del Niño Perdido",
			'12012' => "Altura",
			'12013' => "Arañuel",
			'12014' => "Ares del Maestrat",
			'12015' => "Argelita",
			'12016' => "Artana",
			'12001' => "Atzeneta del Maestrat",
			'12017' => "Ayódar",
			'12018' => "Azuébar",
			'12020' => "Barracas",
			'12022' => "Bejís",
			'12024' => "Benafer",
			'12025' => "Benafigos",
			'12026' => "Benassal",
			'12027' => "Benicarló",
			'12028' => "Benicàssim/Benicasim",
			'12029' => "Benlloc",
			'12021' => "Betxí",
			'12032' => "Borriana/Burriana",
			'12031' => "Borriol",
			'12033' => "Cabanes",
			'12034' => "Càlig",
			'12036' => "Canet lo Roig",
			'12037' => "Castell de Cabres",
			'12038' => "Castellfort",
			'12039' => "Castellnovo",
			'12040' => "Castelló de la Plana",
			'12041' => "Castillo de Villamalefa",
			'12042' => "Catí",
			'12043' => "Caudiel",
			'12044' => "Cervera del Maestre",
			'12053' => "Chilches/Xilxes",
			'12056' => "Chóvar",
			'12045' => "Cinctorres",
			'12046' => "Cirat",
			'12048' => "Cortes de Arenoso",
			'12049' => "Costur",
			'12050' => "Coves de Vinromà, les",
			'12051' => "Culla",
			'12057' => "Eslida",
			'12058' => "Espadilla",
			'12059' => "Fanzara",
			'12060' => "Figueroles",
			'12061' => "Forcall",
			'12063' => "Fuente la Reina",
			'12064' => "Fuentes de Ayódar",
			'12065' => "Gaibiel",
			'12067' => "Geldo",
			'12068' => "Herbers",
			'12069' => "Higueras",
			'12070' => "Jana, la",
			'12071' => "Jérica",
			'12074' => "Llosa, la",
			'12072' => "Llucena/Lucena del Cid",
			'12073' => "Ludiente",
			'12075' => "Mata de Morella, la",
			'12076' => "Matet",
			'12077' => "Moncofa",
			'12078' => "Montán",
			'12079' => "Montanejos",
			'12080' => "Morella",
			'12081' => "Navajas",
			'12082' => "Nules",
			'12083' => "Olocau del Rey",
			'12084' => "Onda",
			'12085' => "Orpesa/Oropesa del Mar",
			'12087' => "Palanques",
			'12088' => "Pavías",
			'12089' => "Peníscola/Peñíscola",
			'12090' => "Pina de Montalgrao",
			'12093' => "Pobla de Benifassà, la",
			'12094' => "Pobla Tornesa, la",
			'12091' => "Portell de Morella",
			'12092' => "Puebla de Arenoso",
			'12095' => "Ribesalbes",
			'12096' => "Rossell",
			'12097' => "Sacañet",
			'12098' => "Salzadella, la",
			'12101' => "San Rafael del Río",
			'12902' => "Sant Joan de Moró",
			'12099' => "Sant Jordi/San Jorge",
			'12100' => "Sant Mateu",
			'12102' => "Santa Magdalena de Pulpis",
			'12104' => "Segorbe",
			'12103' => "Serratella, la",
			'12105' => "Sierra Engarcerán",
			'12106' => "Soneja",
			'12107' => "Sot de Ferrer",
			'12108' => "Suera/Sueras",
			'12109' => "Tales",
			'12110' => "Teresa",
			'12111' => "Tírig",
			'12112' => "Todolella",
			'12113' => "Toga",
			'12114' => "Torás",
			'12115' => "Toro, El",
			'12116' => "Torralba del Pinar",
			'12119' => "Torre d'en Besora, la",
			'12120' => "Torre d'en Doménec, la",
			'12117' => "Torreblanca",
			'12118' => "Torrechiva",
			'12121' => "Traiguera",
			'12122' => "Useres, les/Useras",
			'12124' => "Vall d'Alba",
			'12125' => "Vall de Almonacid",
			'12126' => "Vall d'Uixó, la",
			'12123' => "Vallat",
			'12127' => "Vallibona",
			'12128' => "Vilafamés",
			'12129' => "Vilafranca/Villafranca del Cid",
			'12132' => "Vilanova d'Alcolea",
			'12134' => "Vilar de Canes",
			'12135' => "Vila-real",
			'12136' => "Vilavella, la",
			'12130' => "Villahermosa del Río",
			'12131' => "Villamalur",
			'12133' => "Villanueva de Viver",
			'12137' => "Villores",
			'12138' => "Vinaròs",
			'12139' => "Vistabella del Maestrat",
			'12140' => "Viver",
			'12052' => "Xert",
			'12055' => "Xodos/Chodos",
			'12141' => "Zorita del Maestrazgo",
			'12142' => "Zucaina",
			'13001' => "Abenójar",
			'13002' => "Agudo",
			'13003' => "Alamillo",
			'13004' => "Albaladejo",
			'13005' => "Alcázar de San Juan",
			'13006' => "Alcoba",
			'13007' => "Alcolea de Calatrava",
			'13008' => "Alcubillas",
			'13009' => "Aldea del Rey",
			'13010' => "Alhambra",
			'13011' => "Almadén",
			'13012' => "Almadenejos",
			'13013' => "Almagro",
			'13014' => "Almedina",
			'13015' => "Almodóvar del Campo",
			'13016' => "Almuradiel",
			'13017' => "Anchuras",
			'13903' => "Arenales de San Gregorio",
			'13018' => "Arenas de San Juan",
			'13019' => "Argamasilla de Alba",
			'13020' => "Argamasilla de Calatrava",
			'13021' => "Arroba de los Montes",
			'13022' => "Ballesteros de Calatrava",
			'13023' => "Bolaños de Calatrava",
			'13024' => "Brazatortas",
			'13025' => "Cabezarados",
			'13026' => "Cabezarrubias del Puerto",
			'13027' => "Calzada de Calatrava",
			'13028' => "Campo de Criptana",
			'13029' => "Cañada de Calatrava",
			'13030' => "Caracuel de Calatrava",
			'13031' => "Carrión de Calatrava",
			'13032' => "Carrizosa",
			'13033' => "Castellar de Santiago",
			'13038' => "Chillón",
			'13034' => "Ciudad Real",
			'13035' => "Corral de Calatrava",
			'13036' => "Cortijos, Los",
			'13037' => "Cózar",
			'13039' => "Daimiel",
			'13040' => "Fernán Caballero",
			'13041' => "Fontanarejo",
			'13042' => "Fuencaliente",
			'13043' => "Fuenllana",
			'13044' => "Fuente el Fresno",
			'13045' => "Granátula de Calatrava",
			'13046' => "Guadalmez",
			'13047' => "Herencia",
			'13048' => "Hinojosas de Calatrava",
			'13049' => "Horcajo de los Montes",
			'13050' => "Labores, Las",
			'13904' => "Llanos del Caudillo",
			'13051' => "Luciana",
			'13052' => "Malagón",
			'13053' => "Manzanares",
			'13054' => "Membrilla",
			'13055' => "Mestanza",
			'13056' => "Miguelturra",
			'13057' => "Montiel",
			'13058' => "Moral de Calatrava",
			'13059' => "Navalpino",
			'13060' => "Navas de Estena",
			'13061' => "Pedro Muñoz",
			'13062' => "Picón",
			'13063' => "Piedrabuena",
			'13064' => "Poblete",
			'13065' => "Porzuna",
			'13066' => "Pozuelo de Calatrava",
			'13067' => "Pozuelos de Calatrava, Los",
			'13068' => "Puebla de Don Rodrigo",
			'13069' => "Puebla del Príncipe",
			'13070' => "Puerto Lápice",
			'13071' => "Puertollano",
			'13072' => "Retuerta del Bullaque",
			'13901' => "Robledo, El",
			'13902' => "Ruidera",
			'13073' => "Saceruela",
			'13074' => "San Carlos del Valle",
			'13075' => "San Lorenzo de Calatrava",
			'13076' => "Santa Cruz de los Cáñamos",
			'13077' => "Santa Cruz de Mudela",
			'13078' => "Socuéllamos",
			'13080' => "Solana del Pino",
			'13079' => "Solana, La",
			'13081' => "Terrinches",
			'13082' => "Tomelloso",
			'13083' => "Torralba de Calatrava",
			'13084' => "Torre de Juan Abad",
			'13085' => "Torrenueva",
			'13086' => "Valdemanco del Esteras",
			'13087' => "Valdepeñas",
			'13088' => "Valenzuela de Calatrava",
			'13089' => "Villahermosa",
			'13090' => "Villamanrique",
			'13091' => "Villamayor de Calatrava",
			'13092' => "Villanueva de la Fuente",
			'13093' => "Villanueva de los Infantes",
			'13094' => "Villanueva de San Carlos",
			'13095' => "Villar del Pozo",
			'13096' => "Villarrubia de los Ojos",
			'13097' => "Villarta de San Juan",
			'13098' => "Viso del Marqués",
			'14001' => "Adamuz",
			'14002' => "Aguilar de la Frontera",
			'14003' => "Alcaracejos",
			'14004' => "Almedinilla",
			'14005' => "Almodóvar del Río",
			'14006' => "Añora",
			'14007' => "Baena",
			'14008' => "Belalcázar",
			'14009' => "Belmez",
			'14010' => "Benamejí",
			'14011' => "Blázquez, Los",
			'14012' => "Bujalance",
			'14013' => "Cabra",
			'14014' => "Cañete de las Torres",
			'14015' => "Carcabuey",
			'14016' => "Cardeña",
			'14017' => "Carlota, La",
			'14018' => "Carpio, El",
			'14019' => "Castro del Río",
			'14020' => "Conquista",
			'14021' => "Córdoba",
			'14022' => "Doña Mencía",
			'14023' => "Dos Torres",
			'14024' => "Encinas Reales",
			'14025' => "Espejo",
			'14026' => "Espiel",
			'14027' => "Fernán-Núñez",
			'14901' => "Fuente Carreteros",
			'14028' => "Fuente la Lancha",
			'14029' => "Fuente Obejuna",
			'14030' => "Fuente Palmera",
			'14031' => "Fuente-Tójar",
			'14032' => "Granjuela, La",
			'14033' => "Guadalcázar",
			'14902' => "Guijarrosa, La",
			'14034' => "Guijo, El",
			'14035' => "Hinojosa del Duque",
			'14036' => "Hornachuelos",
			'14037' => "Iznájar",
			'14038' => "Lucena",
			'14039' => "Luque",
			'14040' => "Montalbán de Córdoba",
			'14041' => "Montemayor",
			'14042' => "Montilla",
			'14043' => "Montoro",
			'14044' => "Monturque",
			'14045' => "Moriles",
			'14046' => "Nueva Carteya",
			'14047' => "Obejo",
			'14048' => "Palenciana",
			'14049' => "Palma del Río",
			'14050' => "Pedro Abad",
			'14051' => "Pedroche",
			'14052' => "Peñarroya-Pueblonuevo",
			'14053' => "Posadas",
			'14054' => "Pozoblanco",
			'14055' => "Priego de Córdoba",
			'14056' => "Puente Genil",
			'14057' => "Rambla, La",
			'14058' => "Rute",
			'14059' => "San Sebastián de los Ballesteros",
			'14061' => "Santa Eufemia",
			'14060' => "Santaella",
			'14062' => "Torrecampo",
			'14063' => "Valenzuela",
			'14064' => "Valsequillo",
			'14065' => "Victoria, La",
			'14066' => "Villa del Río",
			'14067' => "Villafranca de Córdoba",
			'14068' => "Villaharta",
			'14069' => "Villanueva de Córdoba",
			'14070' => "Villanueva del Duque",
			'14071' => "Villanueva del Rey",
			'14072' => "Villaralto",
			'14073' => "Villaviciosa de Córdoba",
			'14074' => "Viso, El",
			'14075' => "Zuheros",
			'15001' => "Abegondo",
			'15002' => "Ames",
			'15003' => "Aranga",
			'15004' => "Ares",
			'15005' => "Arteixo",
			'15006' => "Arzúa",
			'15007' => "Baña, A",
			'15008' => "Bergondo",
			'15009' => "Betanzos",
			'15010' => "Boimorto",
			'15011' => "Boiro",
			'15012' => "Boqueixón",
			'15013' => "Brión",
			'15014' => "Cabana de Bergantiños",
			'15015' => "Cabanas",
			'15016' => "Camariñas",
			'15017' => "Cambre",
			'15018' => "Capela, A",
			'15019' => "Carballo",
			'15901' => "Cariño",
			'15020' => "Carnota",
			'15021' => "Carral",
			'15022' => "Cedeira",
			'15023' => "Cee",
			'15024' => "Cerceda",
			'15025' => "Cerdido",
			'15027' => "Coirós",
			'15028' => "Corcubión",
			'15029' => "Coristanco",
			'15030' => "Coruña, A",
			'15031' => "Culleredo",
			'15032' => "Curtis",
			'15033' => "Dodro",
			'15034' => "Dumbría",
			'15035' => "Fene",
			'15036' => "Ferrol",
			'15037' => "Fisterra",
			'15038' => "Frades",
			'15039' => "Irixoa",
			'15041' => "Laracha, A",
			'15040' => "Laxe",
			'15042' => "Lousame",
			'15043' => "Malpica de Bergantiños",
			'15044' => "Mañón",
			'15045' => "Mazaricos",
			'15046' => "Melide",
			'15047' => "Mesía",
			'15048' => "Miño",
			'15049' => "Moeche",
			'15050' => "Monfero",
			'15051' => "Mugardos",
			'15053' => "Muros",
			'15052' => "Muxía",
			'15054' => "Narón",
			'15055' => "Neda",
			'15056' => "Negreira",
			'15057' => "Noia",
			'15058' => "Oleiros",
			'15059' => "Ordes",
			'15060' => "Oroso",
			'15061' => "Ortigueira",
			'15062' => "Outes",
			'15902' => "Oza-Cesuras",
			'15064' => "Paderne",
			'15065' => "Padrón",
			'15066' => "Pino, O",
			'15067' => "Pobra do Caramiñal, A",
			'15068' => "Ponteceso",
			'15069' => "Pontedeume",
			'15070' => "Pontes de García Rodríguez, As",
			'15071' => "Porto do Son",
			'15072' => "Rianxo",
			'15073' => "Ribeira",
			'15074' => "Rois",
			'15075' => "Sada",
			'15076' => "San Sadurniño",
			'15077' => "Santa Comba",
			'15078' => "Santiago de Compostela",
			'15079' => "Santiso",
			'15080' => "Sobrado",
			'15081' => "Somozas, As",
			'15082' => "Teo",
			'15083' => "Toques",
			'15084' => "Tordoia",
			'15085' => "Touro",
			'15086' => "Trazo",
			'15088' => "Val do Dubra",
			'15087' => "Valdoviño",
			'15089' => "Vedra",
			'15091' => "Vilarmaior",
			'15090' => "Vilasantar",
			'15092' => "Vimianzo",
			'15093' => "Zas",
			'16001' => "Abia de la Obispalía",
			'16002' => "Acebrón, El",
			'16003' => "Alarcón",
			'16004' => "Albaladejo del Cuende",
			'16005' => "Albalate de las Nogueras",
			'16006' => "Albendea",
			'16007' => "Alberca de Záncara, La",
			'16008' => "Alcalá de la Vega",
			'16009' => "Alcantud",
			'16010' => "Alcázar del Rey",
			'16011' => "Alcohujate",
			'16012' => "Alconchel de la Estrella",
			'16013' => "Algarra",
			'16014' => "Aliaguilla",
			'16015' => "Almarcha, La",
			'16016' => "Almendros",
			'16017' => "Almodóvar del Pinar",
			'16018' => "Almonacid del Marquesado",
			'16019' => "Altarejos",
			'16020' => "Arandilla del Arroyo",
			'16905' => "Arcas",
			'16022' => "Arcos de la Sierra",
			'16024' => "Arguisuelas",
			'16025' => "Arrancacepas",
			'16026' => "Atalaya del Cañavate",
			'16027' => "Barajas de Melo",
			'16029' => "Barchín del Hoyo",
			'16030' => "Bascuñana de San Pedro",
			'16031' => "Beamud",
			'16032' => "Belinchón",
			'16033' => "Belmonte",
			'16034' => "Belmontejo",
			'16035' => "Beteta",
			'16036' => "Boniches",
			'16038' => "Buciegas",
			'16039' => "Buenache de Alarcón",
			'16040' => "Buenache de la Sierra",
			'16041' => "Buendía",
			'16042' => "Campillo de Altobuey",
			'16043' => "Campillos-Paravientos",
			'16044' => "Campillos-Sierra",
			'16901' => "Campos del Paraíso",
			'16045' => "Canalejas del Arroyo",
			'16046' => "Cañada del Hoyo",
			'16047' => "Cañada Juncosa",
			'16048' => "Cañamares",
			'16049' => "Cañavate, El",
			'16050' => "Cañaveras",
			'16051' => "Cañaveruelas",
			'16052' => "Cañete",
			'16053' => "Cañizares",
			'16055' => "Carboneras de Guadazaón",
			'16056' => "Cardenete",
			'16057' => "Carrascosa",
			'16058' => "Carrascosa de Haro",
			'16060' => "Casas de Benítez",
			'16061' => "Casas de Fernando Alonso",
			'16062' => "Casas de Garcimolina",
			'16063' => "Casas de Guijarro",
			'16064' => "Casas de Haro",
			'16065' => "Casas de los Pinos",
			'16066' => "Casasimarro",
			'16067' => "Castejón",
			'16068' => "Castillejo de Iniesta",
			'16070' => "Castillejo-Sierra",
			'16072' => "Castillo de Garcimuñoz",
			'16071' => "Castillo-Albaráñez",
			'16073' => "Cervera del Llano",
			'16023' => "Chillarón de Cuenca",
			'16081' => "Chumillas",
			'16074' => "Cierva, La",
			'16078' => "Cuenca",
			'16079' => "Cueva del Hierro",
			'16082' => "Enguídanos",
			'16083' => "Fresneda de Altarejos",
			'16084' => "Fresneda de la Sierra",
			'16085' => "Frontera, La",
			'16086' => "Fuente de Pedro Naharro",
			'16087' => "Fuentelespino de Haro",
			'16088' => "Fuentelespino de Moya",
			'16904' => "Fuentenava de Jábaga",
			'16089' => "Fuentes",
			'16091' => "Fuertescusa",
			'16092' => "Gabaldón",
			'16093' => "Garaballa",
			'16094' => "Gascueña",
			'16095' => "Graja de Campalbo",
			'16096' => "Graja de Iniesta",
			'16097' => "Henarejos",
			'16098' => "Herrumblar, El",
			'16099' => "Hinojosa, La",
			'16100' => "Hinojosos, Los",
			'16101' => "Hito, El",
			'16102' => "Honrubia",
			'16103' => "Hontanaya",
			'16104' => "Hontecillas",
			'16106' => "Horcajo de Santiago",
			'16107' => "Huélamo",
			'16108' => "Huelves",
			'16109' => "Huérguina",
			'16110' => "Huerta de la Obispalía",
			'16111' => "Huerta del Marquesado",
			'16112' => "Huete",
			'16113' => "Iniesta",
			'16115' => "Laguna del Marquesado",
			'16116' => "Lagunaseca",
			'16117' => "Landete",
			'16118' => "Ledaña",
			'16119' => "Leganiel",
			'16121' => "Majadas, Las",
			'16122' => "Mariana",
			'16123' => "Masegosa",
			'16124' => "Mesas, Las",
			'16125' => "Minglanilla",
			'16126' => "Mira",
			'16128' => "Monreal del Llano",
			'16129' => "Montalbanejo",
			'16130' => "Montalbo",
			'16131' => "Monteagudo de las Salinas",
			'16132' => "Mota de Altarejos",
			'16133' => "Mota del Cuervo",
			'16134' => "Motilla del Palancar",
			'16135' => "Moya",
			'16137' => "Narboneta",
			'16139' => "Olivares de Júcar",
			'16140' => "Olmeda de la Cuesta",
			'16141' => "Olmeda del Rey",
			'16142' => "Olmedilla de Alarcón",
			'16143' => "Olmedilla de Eliz",
			'16145' => "Osa de la Vega",
			'16146' => "Pajarón",
			'16147' => "Pajaroncillo",
			'16148' => "Palomares del Campo",
			'16149' => "Palomera",
			'16150' => "Paracuellos",
			'16151' => "Paredes",
			'16152' => "Parra de las Vegas, La",
			'16153' => "Pedernoso, El",
			'16154' => "Pedroñeras, Las",
			'16155' => "Peral, El",
			'16156' => "Peraleja, La",
			'16157' => "Pesquera, La",
			'16158' => "Picazo, El",
			'16159' => "Pinarejo",
			'16160' => "Pineda de Gigüela",
			'16161' => "Piqueras del Castillo",
			'16162' => "Portalrubio de Guadamejud",
			'16163' => "Portilla",
			'16165' => "Poyatos",
			'16166' => "Pozoamargo",
			'16908' => "Pozorrubielos de la Mancha",
			'16167' => "Pozorrubio de Santiago",
			'16169' => "Pozuelo, El",
			'16170' => "Priego",
			'16171' => "Provencio, El",
			'16172' => "Puebla de Almenara",
			'16174' => "Puebla del Salvador",
			'16175' => "Quintanar del Rey",
			'16176' => "Rada de Haro",
			'16177' => "Reíllo",
			'16181' => "Rozalén del Monte",
			'16185' => "Saceda-Trasierra",
			'16186' => "Saelices",
			'16187' => "Salinas del Manzano",
			'16188' => "Salmeroncillos",
			'16189' => "Salvacañete",
			'16190' => "San Clemente",
			'16191' => "San Lorenzo de la Parrilla",
			'16192' => "San Martín de Boniches",
			'16193' => "San Pedro Palmiches",
			'16194' => "Santa Cruz de Moya",
			'16196' => "Santa María de los Llanos",
			'16195' => "Santa María del Campo Rus",
			'16197' => "Santa María del Val",
			'16198' => "Sisante",
			'16199' => "Solera de Gabaldón",
			'16909' => "Sotorribas",
			'16202' => "Talayuelas",
			'16203' => "Tarancón",
			'16204' => "Tébar",
			'16205' => "Tejadillos",
			'16206' => "Tinajas",
			'16209' => "Torralba",
			'16211' => "Torrejoncillo del Rey",
			'16212' => "Torrubia del Campo",
			'16213' => "Torrubia del Castillo",
			'16215' => "Tragacete",
			'16216' => "Tresjuncos",
			'16217' => "Tribaldos",
			'16218' => "Uclés",
			'16219' => "Uña",
			'16906' => "Valdecolmenas, Los",
			'16224' => "Valdemeca",
			'16225' => "Valdemorillo de la Sierra",
			'16227' => "Valdemoro-Sierra",
			'16228' => "Valdeolivas",
			'16902' => "Valdetórtola",
			'16903' => "Valeras, Las",
			'16231' => "Valhermoso de la Fuente",
			'16173' => "Valle de Altomira, El",
			'16234' => "Valsalobre",
			'16236' => "Valverde de Júcar",
			'16237' => "Valverdejo",
			'16238' => "Vara de Rey",
			'16239' => "Vega del Codorno",
			'16240' => "Vellisca",
			'16242' => "Villaconejos de Trabaque",
			'16243' => "Villaescusa de Haro",
			'16244' => "Villagarcía del Llano",
			'16245' => "Villalba de la Sierra",
			'16246' => "Villalba del Rey",
			'16247' => "Villalgordo del Marquesado",
			'16248' => "Villalpardo",
			'16249' => "Villamayor de Santiago",
			'16250' => "Villanueva de Guadamejud",
			'16251' => "Villanueva de la Jara",
			'16253' => "Villar de Cañas",
			'16254' => "Villar de Domingo García",
			'16255' => "Villar de la Encina",
			'16263' => "Villar de Olalla",
			'16258' => "Villar del Humo",
			'16259' => "Villar del Infantado",
			'16910' => "Villar y Velasco",
			'16264' => "Villarejo de Fuentes",
			'16265' => "Villarejo de la Peñuela",
			'16266' => "Villarejo-Periesteban",
			'16269' => "Villares del Saz",
			'16270' => "Villarrubio",
			'16271' => "Villarta",
			'16272' => "Villas de la Ventosa",
			'16273' => "Villaverde y Pasaconsol",
			'16274' => "Víllora",
			'16275' => "Vindel",
			'16276' => "Yémeda",
			'16277' => "Zafra de Záncara",
			'16278' => "Zafrilla",
			'16279' => "Zarza de Tajo",
			'16280' => "Zarzuela",
			'17001' => "Agullana",
			'17002' => "Aiguaviva",
			'17003' => "Albanyà",
			'17004' => "Albons",
			'17006' => "Alp",
			'17007' => "Amer",
			'17008' => "Anglès",
			'17009' => "Arbúcies",
			'17010' => "Argelaguer",
			'17011' => "Armentera, L'",
			'17012' => "Avinyonet de Puigventós",
			'17015' => "Banyoles",
			'17016' => "Bàscara",
			'17013' => "Begur",
			'17018' => "Bellcaire d'Empordà",
			'17019' => "Besalú",
			'17020' => "Bescanó",
			'17021' => "Beuda",
			'17022' => "Bisbal d'Empordà, La",
			'17234' => "Biure",
			'17023' => "Blanes",
			'17029' => "Boadella i les Escaules",
			'17024' => "Bolvir",
			'17025' => "Bordils",
			'17026' => "Borrassà",
			'17027' => "Breda",
			'17028' => "Brunyola i Sant Martí Sapresa",
			'17031' => "Cabanelles",
			'17030' => "Cabanes",
			'17032' => "Cadaqués",
			'17033' => "Caldes de Malavella",
			'17034' => "Calonge i Sant Antoni",
			'17035' => "Camós",
			'17036' => "Campdevànol",
			'17037' => "Campelles",
			'17038' => "Campllong",
			'17039' => "Camprodon",
			'17040' => "Canet d'Adri",
			'17041' => "Cantallops",
			'17042' => "Capmany",
			'17044' => "Cassà de la Selva",
			'17048' => "Castell d'Aro, Platja d'Aro i s'Agaró",
			'17046' => "Castellfollit de la Roca",
			'17047' => "Castelló d'Empúries",
			'17189' => "Cellera de Ter, La",
			'17049' => "Celrà",
			'17050' => "Cervià de Ter",
			'17051' => "Cistella",
			'17054' => "Colera",
			'17055' => "Colomers",
			'17057' => "Corçà",
			'17056' => "Cornellà del Terri",
			'17058' => "Crespià",
			'17901' => "Cruïlles, Monells i Sant Sadurní de l'Heura",
			'17060' => "Darnius",
			'17061' => "Das",
			'17062' => "Escala, L'",
			'17063' => "Espinelves",
			'17064' => "Espolla",
			'17065' => "Esponellà",
			'17005' => "Far d'Empordà, El",
			'17066' => "Figueres",
			'17067' => "Flaçà",
			'17068' => "Foixà",
			'17069' => "Fontanals de Cerdanya",
			'17070' => "Fontanilles",
			'17071' => "Fontcoberta",
			'17902' => "Forallac",
			'17073' => "Fornells de la Selva",
			'17074' => "Fortià",
			'17075' => "Garrigàs",
			'17076' => "Garrigoles",
			'17077' => "Garriguella",
			'17078' => "Ger",
			'17079' => "Girona",
			'17080' => "Gombrèn",
			'17081' => "Gualta",
			'17082' => "Guils de Cerdanya",
			'17083' => "Hostalric",
			'17084' => "Isòvol",
			'17085' => "Jafre",
			'17086' => "Jonquera, La",
			'17087' => "Juià",
			'17088' => "Lladó",
			'17089' => "Llagostera",
			'17090' => "Llambilles",
			'17091' => "Llanars",
			'17092' => "Llançà",
			'17093' => "Llers",
			'17094' => "Llívia",
			'17095' => "Lloret de Mar",
			'17096' => "Llosses, Les",
			'17102' => "Maçanet de Cabrenys",
			'17103' => "Maçanet de la Selva",
			'17097' => "Madremanya",
			'17098' => "Maià de Montcal",
			'17100' => "Masarac",
			'17101' => "Massanes",
			'17099' => "Meranges",
			'17105' => "Mieres",
			'17106' => "Mollet de Peralada",
			'17107' => "Molló",
			'17109' => "Montagut i Oix",
			'17110' => "Mont-ras",
			'17111' => "Navata",
			'17112' => "Ogassa",
			'17114' => "Olot",
			'17115' => "Ordis",
			'17116' => "Osor",
			'17117' => "Palafrugell",
			'17118' => "Palamós",
			'17119' => "Palau de Santa Eulàlia",
			'17121' => "Palau-sator",
			'17120' => "Palau-saverdera",
			'17123' => "Palol de Revardit",
			'17124' => "Pals",
			'17125' => "Pardines",
			'17126' => "Parlavà",
			'17128' => "Pau",
			'17129' => "Pedret i Marzà",
			'17130' => "Pera, La",
			'17132' => "Peralada",
			'17133' => "Planes d'Hostoles, Les",
			'17134' => "Planoles",
			'17135' => "Pont de Molins",
			'17136' => "Pontós",
			'17137' => "Porqueres",
			'17140' => "Port de la Selva, El",
			'17138' => "Portbou",
			'17139' => "Preses, Les",
			'17141' => "Puigcerdà",
			'17142' => "Quart",
			'17043' => "Queralbs",
			'17143' => "Rabós",
			'17144' => "Regencós",
			'17145' => "Ribes de Freser",
			'17146' => "Riells i Viabrea",
			'17147' => "Ripoll",
			'17148' => "Riudarenes",
			'17149' => "Riudaura",
			'17150' => "Riudellots de la Selva",
			'17151' => "Riumors",
			'17152' => "Roses",
			'17153' => "Rupià",
			'17154' => "Sales de Llierca",
			'17155' => "Salt",
			'17157' => "Sant Andreu Salou",
			'17183' => "Sant Aniol de Finestres",
			'17158' => "Sant Climent Sescebes",
			'17159' => "Sant Feliu de Buixalleu",
			'17160' => "Sant Feliu de Guíxols",
			'17161' => "Sant Feliu de Pallerols",
			'17162' => "Sant Ferriol",
			'17163' => "Sant Gregori",
			'17164' => "Sant Hilari Sacalm",
			'17165' => "Sant Jaume de Llierca",
			'17167' => "Sant Joan de les Abadesses",
			'17168' => "Sant Joan de Mollet",
			'17185' => "Sant Joan les Fonts",
			'17166' => "Sant Jordi Desvalls",
			'17169' => "Sant Julià de Ramis",
			'17903' => "Sant Julià del Llor i Bonmatí",
			'17171' => "Sant Llorenç de la Muga",
			'17172' => "Sant Martí de Llémena",
			'17173' => "Sant Martí Vell",
			'17174' => "Sant Miquel de Campmajor",
			'17175' => "Sant Miquel de Fluvià",
			'17176' => "Sant Mori",
			'17177' => "Sant Pau de Segúries",
			'17178' => "Sant Pere Pescador",
			'17180' => "Santa Coloma de Farners",
			'17181' => "Santa Cristina d'Aro",
			'17182' => "Santa Llogaia d'Àlguema",
			'17184' => "Santa Pau",
			'17186' => "Sarrià de Ter",
			'17187' => "Saus, Camallera i Llampaies",
			'17188' => "Selva de Mar, La",
			'17190' => "Serinyà",
			'17191' => "Serra de Daró",
			'17192' => "Setcases",
			'17193' => "Sils",
			'17052' => "Siurana",
			'17194' => "Susqueda",
			'17195' => "Tallada d'Empordà, La",
			'17196' => "Terrades",
			'17197' => "Torrent",
			'17198' => "Torroella de Fluvià",
			'17199' => "Torroella de Montgrí",
			'17200' => "Tortellà",
			'17201' => "Toses",
			'17202' => "Tossa de Mar",
			'17204' => "Ullà",
			'17205' => "Ullastret",
			'17203' => "Ultramort",
			'17206' => "Urús",
			'17014' => "Vajol, La",
			'17208' => "Vall de Bianya, La",
			'17207' => "Vall d'en Bas, La",
			'17170' => "Vallfogona de Ripollès",
			'17209' => "Vall-llobrega",
			'17210' => "Ventalló",
			'17211' => "Verges",
			'17212' => "Vidrà",
			'17213' => "Vidreres",
			'17214' => "Vilabertran",
			'17215' => "Vilablareix",
			'17217' => "Viladamat",
			'17216' => "Viladasens",
			'17218' => "Vilademuls",
			'17220' => "Viladrau",
			'17221' => "Vilafant",
			'17223' => "Vilajuïga",
			'17224' => "Vilallonga de Ter",
			'17225' => "Vilamacolum",
			'17226' => "Vilamalla",
			'17227' => "Vilamaniscle",
			'17228' => "Vilanant",
			'17230' => "Vila-sacra",
			'17222' => "Vilaür",
			'17233' => "Vilobí d'Onyar",
			'17232' => "Vilopriu",
			'18001' => "Agrón",
			'18002' => "Alamedilla",
			'18003' => "Albolote",
			'18004' => "Albondón",
			'18005' => "Albuñán",
			'18006' => "Albuñol",
			'18007' => "Albuñuelas",
			'18010' => "Aldeire",
			'18011' => "Alfacar",
			'18012' => "Algarinejo",
			'18013' => "Alhama de Granada",
			'18014' => "Alhendín",
			'18015' => "Alicún de Ortega",
			'18016' => "Almegíjar",
			'18017' => "Almuñécar",
			'18904' => "Alpujarra de la Sierra",
			'18018' => "Alquife",
			'18020' => "Arenas del Rey",
			'18021' => "Armilla",
			'18022' => "Atarfe",
			'18023' => "Baza",
			'18024' => "Beas de Granada",
			'18025' => "Beas de Guadix",
			'18027' => "Benalúa",
			'18028' => "Benalúa de las Villas",
			'18029' => "Benamaurel",
			'18030' => "Bérchules",
			'18032' => "Bubión",
			'18033' => "Busquístar",
			'18034' => "Cacín",
			'18035' => "Cádiar",
			'18036' => "Cájar",
			'18114' => "Calahorra, La",
			'18037' => "Calicasas",
			'18038' => "Campotéjar",
			'18039' => "Caniles",
			'18040' => "Cáñar",
			'18042' => "Capileira",
			'18043' => "Carataunas",
			'18044' => "Cástaras",
			'18045' => "Castilléjar",
			'18046' => "Castril",
			'18047' => "Cenes de la Vega",
			'18059' => "Chauchina",
			'18061' => "Chimeneas",
			'18062' => "Churriana de la Vega",
			'18048' => "Cijuela",
			'18049' => "Cogollos de Guadix",
			'18050' => "Cogollos de la Vega",
			'18051' => "Colomera",
			'18053' => "Cortes de Baza",
			'18054' => "Cortes y Graena",
			'18912' => "Cuevas del Campo",
			'18056' => "Cúllar",
			'18057' => "Cúllar Vega",
			'18063' => "Darro",
			'18064' => "Dehesas de Guadix",
			'18065' => "Dehesas Viejas",
			'18066' => "Deifontes",
			'18067' => "Diezma",
			'18068' => "Dílar",
			'18069' => "Dólar",
			'18915' => "Domingo Pérez de Granada",
			'18070' => "Dúdar",
			'18071' => "Dúrcal",
			'18072' => "Escúzar",
			'18074' => "Ferreira",
			'18076' => "Fonelas",
			'18077' => "Fornes",
			'18078' => "Freila",
			'18079' => "Fuente Vaqueros",
			'18905' => "Gabias, Las",
			'18082' => "Galera",
			'18083' => "Gobernador",
			'18084' => "Gójar",
			'18085' => "Gor",
			'18086' => "Gorafe",
			'18087' => "Granada",
			'18088' => "Guadahortuna",
			'18089' => "Guadix",
			'18906' => "Guájares, Los",
			'18093' => "Gualchos",
			'18094' => "Güéjar Sierra",
			'18095' => "Güevéjar",
			'18096' => "Huélago",
			'18097' => "Huéneja",
			'18098' => "Huéscar",
			'18099' => "Huétor de Santillán",
			'18100' => "Huétor Tájar",
			'18101' => "Huétor Vega",
			'18102' => "Íllora",
			'18103' => "Ítrabo",
			'18105' => "Iznalloz",
			'18106' => "Játar",
			'18107' => "Jayena",
			'18108' => "Jérez del Marquesado",
			'18109' => "Jete",
			'18111' => "Jun",
			'18112' => "Juviles",
			'18115' => "Láchar",
			'18116' => "Lanjarón",
			'18117' => "Lanteira",
			'18119' => "Lecrín",
			'18120' => "Lentegí",
			'18121' => "Lobras",
			'18122' => "Loja",
			'18123' => "Lugros",
			'18124' => "Lújar",
			'18126' => "Malahá, La",
			'18127' => "Maracena",
			'18128' => "Marchal",
			'18132' => "Moclín",
			'18133' => "Molvízar",
			'18134' => "Monachil",
			'18135' => "Montefrío",
			'18136' => "Montejícar",
			'18137' => "Montillana",
			'18138' => "Moraleda de Zafayona",
			'18909' => "Morelábor",
			'18140' => "Motril",
			'18141' => "Murtas",
			'18903' => "Nevada",
			'18143' => "Nigüelas",
			'18144' => "Nívar",
			'18145' => "Ogíjares",
			'18146' => "Orce",
			'18147' => "Órgiva",
			'18148' => "Otívar",
			'18150' => "Padul",
			'18151' => "Pampaneira",
			'18152' => "Pedro Martínez",
			'18153' => "Peligros",
			'18154' => "Peza, La",
			'18910' => "Pinar, El",
			'18157' => "Pinos Genil",
			'18158' => "Pinos Puente",
			'18159' => "Píñar",
			'18161' => "Polícar",
			'18162' => "Polopos",
			'18163' => "Pórtugos",
			'18164' => "Puebla de Don Fadrique",
			'18165' => "Pulianas",
			'18167' => "Purullena",
			'18168' => "Quéntar",
			'18170' => "Rubite",
			'18171' => "Salar",
			'18173' => "Salobreña",
			'18174' => "Santa Cruz del Comercio",
			'18175' => "Santa Fe",
			'18176' => "Soportújar",
			'18177' => "Sorvilán",
			'18901' => "Taha, La",
			'18178' => "Torre-Cardela",
			'18916' => "Torrenueva Costa",
			'18179' => "Torvizcón",
			'18180' => "Trevélez",
			'18181' => "Turón",
			'18182' => "Ugíjar",
			'18914' => "Valderrubio",
			'18907' => "Valle del Zalabí",
			'18902' => "Valle, El",
			'18183' => "Válor",
			'18911' => "Vegas del Genil",
			'18184' => "Vélez de Benaudalla",
			'18185' => "Ventas de Huelma",
			'18149' => "Villa de Otura",
			'18908' => "Villamena",
			'18187' => "Villanueva de las Torres",
			'18188' => "Villanueva Mesía",
			'18189' => "Víznar",
			'18192' => "Zafarraya",
			'18913' => "Zagra",
			'18193' => "Zubia, La",
			'18194' => "Zújar",
			'19001' => "Abánades",
			'19002' => "Ablanque",
			'19003' => "Adobes",
			'19004' => "Alaminos",
			'19005' => "Alarilla",
			'19006' => "Albalate de Zorita",
			'19007' => "Albares",
			'19008' => "Albendiego",
			'19009' => "Alcocer",
			'19010' => "Alcolea de las Peñas",
			'19011' => "Alcolea del Pinar",
			'19013' => "Alcoroches",
			'19015' => "Aldeanueva de Guadalajara",
			'19016' => "Algar de Mesa",
			'19017' => "Algora",
			'19018' => "Alhóndiga",
			'19019' => "Alique",
			'19020' => "Almadrones",
			'19021' => "Almoguera",
			'19022' => "Almonacid de Zorita",
			'19023' => "Alocén",
			'19024' => "Alovera",
			'19027' => "Alustante",
			'19031' => "Angón",
			'19032' => "Anguita",
			'19033' => "Anquela del Ducado",
			'19034' => "Anquela del Pedregal",
			'19036' => "Aranzueque",
			'19037' => "Arbancón",
			'19038' => "Arbeteta",
			'19039' => "Argecilla",
			'19040' => "Armallones",
			'19041' => "Armuña de Tajuña",
			'19042' => "Arroyo de las Fraguas",
			'19043' => "Atanzón",
			'19044' => "Atienza",
			'19045' => "Auñón",
			'19046' => "Azuqueca de Henares",
			'19047' => "Baides",
			'19048' => "Baños de Tajo",
			'19049' => "Bañuelos",
			'19050' => "Barriopedro",
			'19051' => "Berninches",
			'19052' => "Bodera, La",
			'19053' => "Brihuega",
			'19054' => "Budia",
			'19055' => "Bujalaro",
			'19057' => "Bustares",
			'19058' => "Cabanillas del Campo",
			'19059' => "Campillo de Dueñas",
			'19060' => "Campillo de Ranas",
			'19061' => "Campisábalos",
			'19064' => "Canredondo",
			'19065' => "Cantalojas",
			'19066' => "Cañizar",
			'19067' => "Cardoso de la Sierra, El",
			'19070' => "Casa de Uceda",
			'19071' => "Casar, El",
			'19073' => "Casas de San Galindo",
			'19074' => "Caspueñas",
			'19075' => "Castejón de Henares",
			'19076' => "Castellar de la Muela",
			'19078' => "Castilforte",
			'19079' => "Castilnuevo",
			'19080' => "Cendejas de Enmedio",
			'19081' => "Cendejas de la Torre",
			'19082' => "Centenera",
			'19103' => "Checa",
			'19104' => "Chequilla",
			'19106' => "Chillarón del Rey",
			'19105' => "Chiloeches",
			'19086' => "Cifuentes",
			'19087' => "Cincovillas",
			'19088' => "Ciruelas",
			'19089' => "Ciruelos del Pinar",
			'19090' => "Cobeta",
			'19091' => "Cogollor",
			'19092' => "Cogolludo",
			'19095' => "Condemios de Abajo",
			'19096' => "Condemios de Arriba",
			'19097' => "Congostrina",
			'19098' => "Copernal",
			'19099' => "Corduente",
			'19102' => "Cubillo de Uceda, El",
			'19107' => "Driebes",
			'19108' => "Durón",
			'19109' => "Embid",
			'19110' => "Escamilla",
			'19111' => "Escariche",
			'19112' => "Escopete",
			'19113' => "Espinosa de Henares",
			'19114' => "Esplegares",
			'19115' => "Establés",
			'19116' => "Estriégana",
			'19117' => "Fontanar",
			'19118' => "Fuembellida",
			'19119' => "Fuencemillán",
			'19120' => "Fuentelahiguera de Albatages",
			'19121' => "Fuentelencina",
			'19122' => "Fuentelsaz",
			'19123' => "Fuentelviejo",
			'19124' => "Fuentenovilla",
			'19125' => "Gajanejos",
			'19126' => "Galápagos",
			'19127' => "Galve de Sorbe",
			'19129' => "Gascueña de Bornova",
			'19130' => "Guadalajara",
			'19132' => "Henche",
			'19133' => "Heras de Ayuso",
			'19134' => "Herrería",
			'19135' => "Hiendelaencina",
			'19136' => "Hijes",
			'19138' => "Hita",
			'19139' => "Hombrados",
			'19142' => "Hontoba",
			'19143' => "Horche",
			'19145' => "Hortezuela de Océn",
			'19146' => "Huerce, La",
			'19147' => "Huérmeces del Cerro",
			'19148' => "Huertahernando",
			'19150' => "Hueva",
			'19151' => "Humanes",
			'19152' => "Illana",
			'19153' => "Iniéstola",
			'19154' => "Inviernas, Las",
			'19155' => "Irueste",
			'19156' => "Jadraque",
			'19157' => "Jirueque",
			'19159' => "Ledanca",
			'19160' => "Loranca de Tajuña",
			'19161' => "Lupiana",
			'19162' => "Luzaga",
			'19163' => "Luzón",
			'19165' => "Majaelrayo",
			'19166' => "Málaga del Fresno",
			'19167' => "Malaguilla",
			'19168' => "Mandayona",
			'19169' => "Mantiel",
			'19170' => "Maranchón",
			'19171' => "Marchamalo",
			'19172' => "Masegoso de Tajuña",
			'19173' => "Matarrubia",
			'19174' => "Matillas",
			'19175' => "Mazarete",
			'19176' => "Mazuecos",
			'19177' => "Medranda",
			'19178' => "Megina",
			'19179' => "Membrillera",
			'19181' => "Miedes de Atienza",
			'19182' => "Mierla, La",
			'19184' => "Millana",
			'19183' => "Milmarcos",
			'19185' => "Miñosa, La",
			'19186' => "Mirabueno",
			'19187' => "Miralrío",
			'19188' => "Mochales",
			'19189' => "Mohernando",
			'19190' => "Molina de Aragón",
			'19191' => "Monasterio",
			'19192' => "Mondéjar",
			'19193' => "Montarrón",
			'19194' => "Moratilla de los Meleros",
			'19195' => "Morenilla",
			'19196' => "Muduex",
			'19197' => "Navas de Jadraque, Las",
			'19198' => "Negredo",
			'19199' => "Ocentejo",
			'19200' => "Olivar, El",
			'19201' => "Olmeda de Cobeta",
			'19202' => "Olmeda de Jadraque, La",
			'19203' => "Ordial, El",
			'19204' => "Orea",
			'19208' => "Pálmaces de Jadraque",
			'19209' => "Pardos",
			'19210' => "Paredes de Sigüenza",
			'19211' => "Pareja",
			'19212' => "Pastrana",
			'19213' => "Pedregal, El",
			'19214' => "Peñalén",
			'19215' => "Peñalver",
			'19216' => "Peralejos de las Truchas",
			'19217' => "Peralveche",
			'19218' => "Pinilla de Jadraque",
			'19219' => "Pinilla de Molina",
			'19220' => "Pioz",
			'19221' => "Piqueras",
			'19222' => "Pobo de Dueñas, El",
			'19223' => "Poveda de la Sierra",
			'19224' => "Pozo de Almoguera",
			'19225' => "Pozo de Guadalajara",
			'19226' => "Prádena de Atienza",
			'19227' => "Prados Redondos",
			'19228' => "Puebla de Beleña",
			'19229' => "Puebla de Valles",
			'19230' => "Quer",
			'19231' => "Rebollosa de Jadraque",
			'19232' => "Recuenco, El",
			'19233' => "Renera",
			'19234' => "Retiendas",
			'19235' => "Riba de Saelices",
			'19237' => "Rillo de Gallo",
			'19238' => "Riofrío del Llano",
			'19239' => "Robledillo de Mohernando",
			'19240' => "Robledo de Corpes",
			'19241' => "Romanillos de Atienza",
			'19242' => "Romanones",
			'19243' => "Rueda de la Sierra",
			'19244' => "Sacecorbo",
			'19245' => "Sacedón",
			'19246' => "Saelices de la Sal",
			'19247' => "Salmerón",
			'19248' => "San Andrés del Congosto",
			'19249' => "San Andrés del Rey",
			'19250' => "Santiuste",
			'19251' => "Saúca",
			'19252' => "Sayatón",
			'19254' => "Selas",
			'19901' => "Semillas",
			'19255' => "Setiles",
			'19256' => "Sienes",
			'19257' => "Sigüenza",
			'19258' => "Solanillos del Extremo",
			'19259' => "Somolinos",
			'19260' => "Sotillo, El",
			'19261' => "Sotodosos",
			'19262' => "Tamajón",
			'19263' => "Taragudo",
			'19264' => "Taravilla",
			'19265' => "Tartanedo",
			'19266' => "Tendilla",
			'19267' => "Terzaga",
			'19268' => "Tierzo",
			'19269' => "Toba, La",
			'19271' => "Tordellego",
			'19270' => "Tordelrábano",
			'19272' => "Tordesilos",
			'19274' => "Torija",
			'19279' => "Torre del Burgo",
			'19277' => "Torrecuadrada de Molina",
			'19278' => "Torrecuadradilla",
			'19280' => "Torrejón del Rey",
			'19281' => "Torremocha de Jadraque",
			'19282' => "Torremocha del Campo",
			'19283' => "Torremocha del Pinar",
			'19284' => "Torremochuela",
			'19285' => "Torrubia",
			'19286' => "Tórtola de Henares",
			'19287' => "Tortuera",
			'19288' => "Tortuero",
			'19289' => "Traíd",
			'19290' => "Trijueque",
			'19291' => "Trillo",
			'19293' => "Uceda",
			'19294' => "Ujados",
			'19296' => "Utande",
			'19297' => "Valdarachas",
			'19298' => "Valdearenas",
			'19299' => "Valdeavellano",
			'19300' => "Valdeaveruelo",
			'19301' => "Valdeconcha",
			'19302' => "Valdegrudas",
			'19303' => "Valdelcubo",
			'19304' => "Valdenuño Fernández",
			'19305' => "Valdepeñas de la Sierra",
			'19306' => "Valderrebollo",
			'19307' => "Valdesotos",
			'19308' => "Valfermoso de Tajuña",
			'19309' => "Valhermoso",
			'19310' => "Valtablado del Río",
			'19311' => "Valverde de los Arroyos",
			'19314' => "Viana de Jadraque",
			'19317' => "Villanueva de Alcorón",
			'19318' => "Villanueva de Argecilla",
			'19319' => "Villanueva de la Torre",
			'19321' => "Villares de Jadraque",
			'19322' => "Villaseca de Henares",
			'19323' => "Villaseca de Uceda",
			'19324' => "Villel de Mesa",
			'19325' => "Viñuelas",
			'19326' => "Yebes",
			'19327' => "Yebra",
			'19329' => "Yélamos de Abajo",
			'19330' => "Yélamos de Arriba",
			'19331' => "Yunquera de Henares",
			'19332' => "Yunta, La",
			'19333' => "Zaorejas",
			'19334' => "Zarzuela de Jadraque",
			'19335' => "Zorita de los Canes",
			'20001' => "Abaltzisketa",
			'20002' => "Aduna",
			'20016' => "Aia",
			'20003' => "Aizarnazabal",
			'20004' => "Albiztur",
			'20005' => "Alegia",
			'20006' => "Alkiza",
			'20906' => "Altzaga",
			'20007' => "Altzo",
			'20008' => "Amezketa",
			'20009' => "Andoain",
			'20010' => "Anoeta",
			'20011' => "Antzuola",
			'20012' => "Arama",
			'20013' => "Aretxabaleta",
			'20055' => "Arrasate/Mondragón",
			'20014' => "Asteasu",
			'20903' => "Astigarraga",
			'20015' => "Ataun",
			'20017' => "Azkoitia",
			'20018' => "Azpeitia",
			'20904' => "Baliarrain",
			'20019' => "Beasain",
			'20020' => "Beizama",
			'20021' => "Belauntza",
			'20022' => "Berastegi",
			'20074' => "Bergara",
			'20023' => "Berrobi",
			'20024' => "Bidania-Goiatz",
			'20029' => "Deba",
			'20069' => "Donostia/San Sebastián",
			'20030' => "Eibar",
			'20031' => "Elduain",
			'20033' => "Elgeta",
			'20032' => "Elgoibar",
			'20067' => "Errenteria",
			'20066' => "Errezil",
			'20034' => "Eskoriatza",
			'20035' => "Ezkio-Itsaso",
			'20038' => "Gabiria",
			'20037' => "Gaintza",
			'20907' => "Gaztelu",
			'20039' => "Getaria",
			'20040' => "Hernani",
			'20041' => "Hernialde",
			'20036' => "Hondarribia",
			'20042' => "Ibarra",
			'20043' => "Idiazabal",
			'20044' => "Ikaztegieta",
			'20045' => "Irun",
			'20046' => "Irura",
			'20047' => "Itsasondo",
			'20048' => "Larraul",
			'20902' => "Lasarte-Oria",
			'20049' => "Lazkao",
			'20050' => "Leaburu",
			'20051' => "Legazpi",
			'20052' => "Legorreta",
			'20068' => "Leintz-Gatzaga",
			'20053' => "Lezo",
			'20054' => "Lizartza",
			'20901' => "Mendaro",
			'20057' => "Mutiloa",
			'20056' => "Mutriku",
			'20063' => "Oiartzun",
			'20058' => "Olaberria",
			'20059' => "Oñati",
			'20076' => "Ordizia",
			'20905' => "Orendain",
			'20060' => "Orexa",
			'20061' => "Orio",
			'20062' => "Ormaiztegi",
			'20064' => "Pasaia",
			'20070' => "Segura",
			'20065' => "Soraluze-Placencia de las Armas",
			'20071' => "Tolosa",
			'20072' => "Urnieta",
			'20077' => "Urretxu",
			'20073' => "Usurbil",
			'20075' => "Villabona",
			'20078' => "Zaldibia",
			'20079' => "Zarautz",
			'20025' => "Zegama",
			'20026' => "Zerain",
			'20027' => "Zestoa",
			'20028' => "Zizurkil",
			'20081' => "Zumaia",
			'20080' => "Zumarraga",
			'21001' => "Alájar",
			'21002' => "Aljaraque",
			'21003' => "Almendro, El",
			'21004' => "Almonaster la Real",
			'21005' => "Almonte",
			'21006' => "Alosno",
			'21007' => "Aracena",
			'21008' => "Aroche",
			'21009' => "Arroyomolinos de León",
			'21010' => "Ayamonte",
			'21011' => "Beas",
			'21012' => "Berrocal",
			'21013' => "Bollullos Par del Condado",
			'21014' => "Bonares",
			'21015' => "Cabezas Rubias",
			'21016' => "Cala",
			'21017' => "Calañas",
			'21018' => "Campillo, El",
			'21019' => "Campofrío",
			'21020' => "Cañaveral de León",
			'21021' => "Cartaya",
			'21022' => "Castaño del Robledo",
			'21023' => "Cerro de Andévalo, El",
			'21030' => "Chucena",
			'21024' => "Corteconcepción",
			'21025' => "Cortegana",
			'21026' => "Cortelazor",
			'21027' => "Cumbres de Enmedio",
			'21028' => "Cumbres de San Bartolomé",
			'21029' => "Cumbres Mayores",
			'21031' => "Encinasola",
			'21032' => "Escacena del Campo",
			'21033' => "Fuenteheridos",
			'21034' => "Galaroza",
			'21035' => "Gibraleón",
			'21036' => "Granada de Río-Tinto, La",
			'21037' => "Granado, El",
			'21038' => "Higuera de la Sierra",
			'21039' => "Hinojales",
			'21040' => "Hinojos",
			'21041' => "Huelva",
			'21042' => "Isla Cristina",
			'21043' => "Jabugo",
			'21044' => "Lepe",
			'21045' => "Linares de la Sierra",
			'21046' => "Lucena del Puerto",
			'21047' => "Manzanilla",
			'21048' => "Marines, Los",
			'21049' => "Minas de Riotinto",
			'21050' => "Moguer",
			'21051' => "Nava, La",
			'21052' => "Nerva",
			'21053' => "Niebla",
			'21054' => "Palma del Condado, La",
			'21055' => "Palos de la Frontera",
			'21056' => "Paterna del Campo",
			'21057' => "Paymogo",
			'21058' => "Puebla de Guzmán",
			'21059' => "Puerto Moral",
			'21060' => "Punta Umbría",
			'21061' => "Rociana del Condado",
			'21062' => "Rosal de la Frontera",
			'21063' => "San Bartolomé de la Torre",
			'21064' => "San Juan del Puerto",
			'21066' => "San Silvestre de Guzmán",
			'21065' => "Sanlúcar de Guadiana",
			'21067' => "Santa Ana la Real",
			'21068' => "Santa Bárbara de Casa",
			'21069' => "Santa Olalla del Cala",
			'21070' => "Trigueros",
			'21071' => "Valdelarco",
			'21072' => "Valverde del Camino",
			'21073' => "Villablanca",
			'21074' => "Villalba del Alcor",
			'21075' => "Villanueva de las Cruces",
			'21076' => "Villanueva de los Castillejos",
			'21077' => "Villarrasa",
			'21078' => "Zalamea la Real",
			'21902' => "Zarza-Perrunal, La",
			'21079' => "Zufre",
			'22001' => "Abiego",
			'22002' => "Abizanda",
			'22003' => "Adahuesca",
			'22004' => "Agüero",
			'22907' => "Aínsa-Sobrarbe",
			'22006' => "Aisa",
			'22007' => "Albalate de Cinca",
			'22008' => "Albalatillo",
			'22009' => "Albelda",
			'22011' => "Albero Alto",
			'22012' => "Albero Bajo",
			'22013' => "Alberuela de Tubo",
			'22014' => "Alcalá de Gurrea",
			'22015' => "Alcalá del Obispo",
			'22016' => "Alcampell",
			'22017' => "Alcolea de Cinca",
			'22018' => "Alcubierre",
			'22019' => "Alerre",
			'22020' => "Alfántega",
			'22021' => "Almudévar",
			'22022' => "Almunia de San Juan",
			'22023' => "Almuniente",
			'22024' => "Alquézar",
			'22025' => "Altorricón",
			'22027' => "Angüés",
			'22028' => "Ansó",
			'22029' => "Antillón",
			'22032' => "Aragüés del Puerto",
			'22035' => "Arén",
			'22036' => "Argavieso",
			'22037' => "Arguis",
			'22039' => "Ayerbe",
			'22040' => "Azanuy-Alins",
			'22041' => "Azara",
			'22042' => "Azlor",
			'22043' => "Baélls",
			'22044' => "Bailo",
			'22045' => "Baldellou",
			'22046' => "Ballobar",
			'22047' => "Banastás",
			'22048' => "Barbastro",
			'22049' => "Barbués",
			'22050' => "Barbuñales",
			'22051' => "Bárcabo",
			'22052' => "Belver de Cinca",
			'22053' => "Benabarre",
			'22054' => "Benasque",
			'22246' => "Beranuy",
			'22055' => "Berbegal",
			'22057' => "Bielsa",
			'22058' => "Bierge",
			'22059' => "Biescas",
			'22060' => "Binaced",
			'22061' => "Binéfar",
			'22062' => "Bisaurri",
			'22063' => "Biscarrués",
			'22064' => "Blecua y Torres",
			'22066' => "Boltaña",
			'22067' => "Bonansa",
			'22068' => "Borau",
			'22069' => "Broto",
			'22072' => "Caldearenas",
			'22074' => "Campo",
			'22075' => "Camporrélls",
			'22076' => "Canal de Berdún",
			'22077' => "Candasnos",
			'22078' => "Canfranc",
			'22079' => "Capdesaso",
			'22080' => "Capella",
			'22081' => "Casbas de Huesca",
			'22083' => "Castejón de Monegros",
			'22084' => "Castejón de Sos",
			'22082' => "Castejón del Puente",
			'22085' => "Castelflorite",
			'22086' => "Castiello de Jaca",
			'22087' => "Castigaleu",
			'22088' => "Castillazuelo",
			'22089' => "Castillonroy",
			'22094' => "Chalamera",
			'22095' => "Chía",
			'22096' => "Chimillas",
			'22090' => "Colungo",
			'22099' => "Esplús",
			'22102' => "Estada",
			'22103' => "Estadilla",
			'22105' => "Estopiñán del Castillo",
			'22106' => "Fago",
			'22107' => "Fanlo",
			'22109' => "Fiscal",
			'22110' => "Fonz",
			'22111' => "Foradada del Toscar",
			'22112' => "Fraga",
			'22113' => "Fueva, La",
			'22114' => "Gistaín",
			'22115' => "Grado, El",
			'22116' => "Grañén",
			'22117' => "Graus",
			'22119' => "Gurrea de Gállego",
			'22122' => "Hoz de Jaca",
			'22908' => "Hoz y Costean",
			'22124' => "Huerto",
			'22125' => "Huesca",
			'22126' => "Ibieca",
			'22127' => "Igriés",
			'22128' => "Ilche",
			'22129' => "Isábena",
			'22130' => "Jaca",
			'22131' => "Jasa",
			'22133' => "Labuerda",
			'22135' => "Laluenga",
			'22136' => "Lalueza",
			'22137' => "Lanaja",
			'22139' => "Laperdiguera",
			'22141' => "Lascellas-Ponzano",
			'22142' => "Lascuarre",
			'22143' => "Laspaúles",
			'22144' => "Laspuña",
			'22149' => "Loarre",
			'22150' => "Loporzano",
			'22151' => "Loscorrales",
			'22905' => "Lupiñén-Ortilla",
			'22155' => "Monesma y Cajigar",
			'22156' => "Monflorite-Lascasas",
			'22157' => "Montanuy",
			'22158' => "Monzón",
			'22160' => "Naval",
			'22162' => "Novales",
			'22163' => "Nueno",
			'22164' => "Olvena",
			'22165' => "Ontiñena",
			'22167' => "Osso de Cinca",
			'22168' => "Palo",
			'22170' => "Panticosa",
			'22172' => "Peñalba",
			'22173' => "Peñas de Riglos, Las",
			'22174' => "Peralta de Alcofea",
			'22175' => "Peralta de Calasanz",
			'22176' => "Peraltilla",
			'22177' => "Perarrúa",
			'22178' => "Pertusa",
			'22181' => "Piracés",
			'22182' => "Plan",
			'22184' => "Poleñino",
			'22186' => "Pozán de Vero",
			'22187' => "Puebla de Castro, La",
			'22188' => "Puente de Montañana",
			'22902' => "Puente la Reina de Jaca",
			'22189' => "Puértolas",
			'22190' => "Pueyo de Araguás, El",
			'22193' => "Pueyo de Santa Cruz",
			'22195' => "Quicena",
			'22197' => "Robres",
			'22199' => "Sabiñánigo",
			'22200' => "Sahún",
			'22201' => "Salas Altas",
			'22202' => "Salas Bajas",
			'22203' => "Salillas",
			'22204' => "Sallent de Gállego",
			'22205' => "San Esteban de Litera",
			'22207' => "San Juan de Plan",
			'22903' => "San Miguel del Cinca",
			'22206' => "Sangarrén",
			'22208' => "Santa Cilia",
			'22209' => "Santa Cruz de la Serós",
			'22906' => "Santa María de Dulcis",
			'22212' => "Santaliestra y San Quílez",
			'22213' => "Sariñena",
			'22214' => "Secastilla",
			'22215' => "Seira",
			'22217' => "Sena",
			'22218' => "Senés de Alcubierre",
			'22220' => "Sesa",
			'22221' => "Sesué",
			'22222' => "Siétamo",
			'22223' => "Sopeira",
			'22904' => "Sotonera, La",
			'22225' => "Tamarite de Litera",
			'22226' => "Tardienta",
			'22227' => "Tella-Sin",
			'22228' => "Tierz",
			'22229' => "Tolva",
			'22230' => "Torla-Ordesa",
			'22232' => "Torralba de Aragón",
			'22233' => "Torre la Ribera",
			'22234' => "Torrente de Cinca",
			'22235' => "Torres de Alcanadre",
			'22236' => "Torres de Barbués",
			'22239' => "Tramaced",
			'22242' => "Valfarta",
			'22243' => "Valle de Bardají",
			'22901' => "Valle de Hecho",
			'22244' => "Valle de Lierp",
			'22245' => "Velilla de Cinca",
			'22909' => "Vencillón",
			'22247' => "Viacamp y Litera",
			'22248' => "Vicién",
			'22249' => "Villanova",
			'22250' => "Villanúa",
			'22251' => "Villanueva de Sigena",
			'22252' => "Yebra de Basa",
			'22253' => "Yésero",
			'22254' => "Zaidín",
			'23001' => "Albanchez de Mágina",
			'23002' => "Alcalá la Real",
			'23003' => "Alcaudete",
			'23004' => "Aldeaquemada",
			'23005' => "Andújar",
			'23006' => "Arjona",
			'23007' => "Arjonilla",
			'23008' => "Arquillos",
			'23905' => "Arroyo del Ojanco",
			'23009' => "Baeza",
			'23010' => "Bailén",
			'23011' => "Baños de la Encina",
			'23012' => "Beas de Segura",
			'23902' => "Bedmar y Garcíez",
			'23014' => "Begíjar",
			'23015' => "Bélmez de la Moraleda",
			'23016' => "Benatae",
			'23017' => "Cabra del Santo Cristo",
			'23018' => "Cambil",
			'23019' => "Campillo de Arenas",
			'23020' => "Canena",
			'23021' => "Carboneros",
			'23901' => "Cárcheles",
			'23024' => "Carolina, La",
			'23025' => "Castellar",
			'23026' => "Castillo de Locubín",
			'23027' => "Cazalilla",
			'23028' => "Cazorla",
			'23029' => "Chiclana de Segura",
			'23030' => "Chilluévar",
			'23031' => "Escañuela",
			'23032' => "Espeluy",
			'23033' => "Frailes",
			'23034' => "Fuensanta de Martos",
			'23035' => "Fuerte del Rey",
			'23037' => "Génave",
			'23038' => "Guardia de Jaén, La",
			'23039' => "Guarromán",
			'23041' => "Higuera de Calatrava",
			'23042' => "Hinojares",
			'23043' => "Hornos",
			'23044' => "Huelma",
			'23045' => "Huesa",
			'23046' => "Ibros",
			'23047' => "Iruela, La",
			'23048' => "Iznatoraf",
			'23049' => "Jabalquinto",
			'23050' => "Jaén",
			'23051' => "Jamilena",
			'23052' => "Jimena",
			'23053' => "Jódar",
			'23040' => "Lahiguera",
			'23054' => "Larva",
			'23055' => "Linares",
			'23056' => "Lopera",
			'23057' => "Lupión",
			'23058' => "Mancha Real",
			'23059' => "Marmolejo",
			'23060' => "Martos",
			'23061' => "Mengíbar",
			'23062' => "Montizón",
			'23063' => "Navas de San Juan",
			'23064' => "Noalejo",
			'23065' => "Orcera",
			'23066' => "Peal de Becerro",
			'23067' => "Pegalajar",
			'23069' => "Porcuna",
			'23070' => "Pozo Alcón",
			'23071' => "Puente de Génave",
			'23072' => "Puerta de Segura, La",
			'23073' => "Quesada",
			'23074' => "Rus",
			'23075' => "Sabiote",
			'23076' => "Santa Elena",
			'23077' => "Santiago de Calatrava",
			'23904' => "Santiago-Pontones",
			'23079' => "Santisteban del Puerto",
			'23080' => "Santo Tomé",
			'23081' => "Segura de la Sierra",
			'23082' => "Siles",
			'23084' => "Sorihuela del Guadalimar",
			'23085' => "Torreblascopedro",
			'23086' => "Torredelcampo",
			'23087' => "Torredonjimeno",
			'23088' => "Torreperogil",
			'23090' => "Torres",
			'23091' => "Torres de Albánchez",
			'23092' => "Úbeda",
			'23093' => "Valdepeñas de Jaén",
			'23094' => "Vilches",
			'23095' => "Villacarrillo",
			'23096' => "Villanueva de la Reina",
			'23097' => "Villanueva del Arzobispo",
			'23098' => "Villardompardo",
			'23099' => "Villares, Los",
			'23101' => "Villarrodrigo",
			'23903' => "Villatorres",
			'24001' => "Acebedo",
			'24002' => "Algadefe",
			'24003' => "Alija del Infantado",
			'24004' => "Almanza",
			'24005' => "Antigua, La",
			'24006' => "Ardón",
			'24007' => "Arganza",
			'24008' => "Astorga",
			'24009' => "Balboa",
			'24010' => "Bañeza, La",
			'24011' => "Barjas",
			'24012' => "Barrios de Luna, Los",
			'24014' => "Bembibre",
			'24015' => "Benavides",
			'24016' => "Benuza",
			'24017' => "Bercianos del Páramo",
			'24018' => "Bercianos del Real Camino",
			'24019' => "Berlanga del Bierzo",
			'24020' => "Boca de Huérgano",
			'24021' => "Boñar",
			'24022' => "Borrenes",
			'24023' => "Brazuelo",
			'24024' => "Burgo Ranero, El",
			'24025' => "Burón",
			'24026' => "Bustillo del Páramo",
			'24027' => "Cabañas Raras",
			'24028' => "Cabreros del Río",
			'24029' => "Cabrillanes",
			'24030' => "Cacabelos",
			'24031' => "Calzada del Coto",
			'24032' => "Campazas",
			'24033' => "Campo de Villavidel",
			'24034' => "Camponaraya",
			'24037' => "Cármenes",
			'24038' => "Carracedelo",
			'24039' => "Carrizo",
			'24040' => "Carrocera",
			'24041' => "Carucedo",
			'24042' => "Castilfalé",
			'24043' => "Castrillo de Cabrera",
			'24044' => "Castrillo de la Valduerna",
			'24046' => "Castrocalbón",
			'24047' => "Castrocontrigo",
			'24049' => "Castropodame",
			'24050' => "Castrotierra de Valmadrigal",
			'24051' => "Cea",
			'24052' => "Cebanico",
			'24053' => "Cebrones del Río",
			'24065' => "Chozas de Abajo",
			'24054' => "Cimanes de la Vega",
			'24055' => "Cimanes del Tejar",
			'24056' => "Cistierna",
			'24057' => "Congosto",
			'24058' => "Corbillos de los Oteros",
			'24059' => "Corullón",
			'24060' => "Crémenes",
			'24061' => "Cuadros",
			'24062' => "Cubillas de los Oteros",
			'24063' => "Cubillas de Rueda",
			'24064' => "Cubillos del Sil",
			'24066' => "Destriana",
			'24067' => "Encinedo",
			'24068' => "Ercina, La",
			'24069' => "Escobar de Campos",
			'24070' => "Fabero",
			'24071' => "Folgoso de la Ribera",
			'24073' => "Fresno de la Vega",
			'24074' => "Fuentes de Carbajal",
			'24076' => "Garrafe de Torío",
			'24077' => "Gordaliza del Pino",
			'24078' => "Gordoncillo",
			'24079' => "Gradefes",
			'24080' => "Grajal de Campos",
			'24081' => "Gusendos de los Oteros",
			'24082' => "Hospital de Órbigo",
			'24083' => "Igüeña",
			'24084' => "Izagre",
			'24086' => "Joarilla de las Matas",
			'24087' => "Laguna Dalga",
			'24088' => "Laguna de Negrillos",
			'24089' => "León",
			'24092' => "Llamas de la Ribera",
			'24090' => "Lucillo",
			'24091' => "Luyego",
			'24093' => "Magaz de Cepeda",
			'24094' => "Mansilla de las Mulas",
			'24095' => "Mansilla Mayor",
			'24096' => "Maraña",
			'24097' => "Matadeón de los Oteros",
			'24098' => "Matallana de Torío",
			'24099' => "Matanza",
			'24100' => "Molinaseca",
			'24101' => "Murias de Paredes",
			'24102' => "Noceda del Bierzo",
			'24103' => "Oencia",
			'24104' => "Omañas, Las",
			'24105' => "Onzonilla",
			'24106' => "Oseja de Sajambre",
			'24107' => "Pajares de los Oteros",
			'24108' => "Palacios de la Valduerna",
			'24109' => "Palacios del Sil",
			'24110' => "Páramo del Sil",
			'24112' => "Peranzanes",
			'24113' => "Pobladura de Pelayo García",
			'24114' => "Pola de Gordón, La",
			'24115' => "Ponferrada",
			'24116' => "Posada de Valdeón",
			'24117' => "Pozuelo del Páramo",
			'24118' => "Prado de la Guzpeña",
			'24119' => "Priaranza del Bierzo",
			'24120' => "Prioro",
			'24121' => "Puebla de Lillo",
			'24122' => "Puente de Domingo Flórez",
			'24123' => "Quintana del Castillo",
			'24124' => "Quintana del Marco",
			'24125' => "Quintana y Congosto",
			'24127' => "Regueras de Arriba",
			'24129' => "Reyero",
			'24130' => "Riaño",
			'24131' => "Riego de la Vega",
			'24132' => "Riello",
			'24133' => "Rioseco de Tapia",
			'24134' => "Robla, La",
			'24136' => "Roperuelos del Páramo",
			'24137' => "Sabero",
			'24139' => "Sahagún",
			'24141' => "San Adrián del Valle",
			'24142' => "San Andrés del Rabanedo",
			'24144' => "San Cristóbal de la Polantera",
			'24145' => "San Emiliano",
			'24146' => "San Esteban de Nogales",
			'24148' => "San Justo de la Vega",
			'24149' => "San Millán de los Caballeros",
			'24150' => "San Pedro Bercianos",
			'24143' => "Sancedo",
			'24151' => "Santa Colomba de Curueño",
			'24152' => "Santa Colomba de Somoza",
			'24153' => "Santa Cristina de Valmadrigal",
			'24154' => "Santa Elena de Jamuz",
			'24155' => "Santa María de la Isla",
			'24158' => "Santa María de Ordás",
			'24156' => "Santa María del Monte de Cea",
			'24157' => "Santa María del Páramo",
			'24159' => "Santa Marina del Rey",
			'24160' => "Santas Martas",
			'24161' => "Santiago Millas",
			'24162' => "Santovenia de la Valdoncina",
			'24163' => "Sariegos",
			'24164' => "Sena de Luna",
			'24165' => "Sobrado",
			'24166' => "Soto de la Vega",
			'24167' => "Soto y Amío",
			'24168' => "Toral de los Guzmanes",
			'24206' => "Toral de los Vados",
			'24169' => "Toreno",
			'24170' => "Torre del Bierzo",
			'24171' => "Trabadelo",
			'24172' => "Truchas",
			'24173' => "Turcia",
			'24174' => "Urdiales del Páramo",
			'24185' => "Val de San Lorenzo",
			'24175' => "Valdefresno",
			'24176' => "Valdefuentes del Páramo",
			'24177' => "Valdelugueros",
			'24178' => "Valdemora",
			'24179' => "Valdepiélago",
			'24180' => "Valdepolo",
			'24181' => "Valderas",
			'24182' => "Valderrey",
			'24183' => "Valderrueda",
			'24184' => "Valdesamario",
			'24187' => "Valdevimbre",
			'24188' => "Valencia de Don Juan",
			'24036' => "Valle de Ancares",
			'24191' => "Vallecillo",
			'24189' => "Valverde de la Virgen",
			'24190' => "Valverde-Enrique",
			'24193' => "Vecilla, La",
			'24196' => "Vega de Espinareda",
			'24197' => "Vega de Infanzones",
			'24198' => "Vega de Valcarce",
			'24194' => "Vegacervera",
			'24199' => "Vegaquemada",
			'24201' => "Vegas del Condado",
			'24202' => "Villablino",
			'24203' => "Villabraz",
			'24205' => "Villadangos del Páramo",
			'24207' => "Villademor de la Vega",
			'24209' => "Villafranca del Bierzo",
			'24210' => "Villagatón",
			'24211' => "Villamandos",
			'24901' => "Villamanín",
			'24212' => "Villamañán",
			'24213' => "Villamartín de Don Sancho",
			'24214' => "Villamejil",
			'24215' => "Villamol",
			'24216' => "Villamontán de la Valduerna",
			'24217' => "Villamoratiel de las Matas",
			'24218' => "Villanueva de las Manzanas",
			'24219' => "Villaobispo de Otero",
			'24902' => "Villaornate y Castro",
			'24221' => "Villaquejida",
			'24222' => "Villaquilambre",
			'24223' => "Villarejo de Órbigo",
			'24224' => "Villares de Órbigo",
			'24225' => "Villasabariego",
			'24226' => "Villaselán",
			'24227' => "Villaturiel",
			'24228' => "Villazala",
			'24229' => "Villazanzo de Valderaduey",
			'24230' => "Zotes del Páramo",
			'25001' => "Abella de la Conca",
			'25002' => "Àger",
			'25003' => "Agramunt",
			'25038' => "Aitona",
			'25004' => "Alamús, Els",
			'25005' => "Alàs i Cerc",
			'25006' => "Albagés, L'",
			'25007' => "Albatàrrec",
			'25008' => "Albesa",
			'25009' => "Albi, L'",
			'25010' => "Alcanó",
			'25011' => "Alcarràs",
			'25012' => "Alcoletge",
			'25013' => "Alfarràs",
			'25014' => "Alfés",
			'25015' => "Algerri",
			'25016' => "Alguaire",
			'25017' => "Alins",
			'25019' => "Almacelles",
			'25020' => "Almatret",
			'25021' => "Almenar",
			'25022' => "Alòs de Balaguer",
			'25023' => "Alpicat",
			'25024' => "Alt Àneu",
			'25027' => "Anglesola",
			'25029' => "Arbeca",
			'25031' => "Arres",
			'25032' => "Arsèguel",
			'25033' => "Artesa de Lleida",
			'25034' => "Artesa de Segre",
			'25036' => "Aspa",
			'25037' => "Avellanes i Santa Linya, Les",
			'25039' => "Baix Pallars",
			'25040' => "Balaguer",
			'25041' => "Barbens",
			'25042' => "Baronia de Rialb, La",
			'25044' => "Bassella",
			'25045' => "Bausen",
			'25046' => "Belianes",
			'25170' => "Bellaguarda",
			'25047' => "Bellcaire d'Urgell",
			'25048' => "Bell-lloc d'Urgell",
			'25049' => "Bellmunt d'Urgell",
			'25050' => "Bellpuig",
			'25051' => "Bellver de Cerdanya",
			'25052' => "Bellvís",
			'25053' => "Benavent de Segrià",
			'25055' => "Biosca",
			'25057' => "Bòrdes, Es",
			'25058' => "Borges Blanques, Les",
			'25059' => "Bossòst",
			'25056' => "Bovera",
			'25060' => "Cabanabona",
			'25061' => "Cabó",
			'25062' => "Camarasa",
			'25063' => "Canejan",
			'25904' => "Castell de Mur",
			'25064' => "Castellar de la Ribera",
			'25067' => "Castelldans",
			'25068' => "Castellnou de Seana",
			'25069' => "Castelló de Farfanya",
			'25070' => "Castellserà",
			'25071' => "Cava",
			'25072' => "Cervera",
			'25073' => "Cervià de les Garrigues",
			'25074' => "Ciutadilla",
			'25075' => "Clariana de Cardener",
			'25076' => "Cogul, El",
			'25077' => "Coll de Nargó",
			'25163' => "Coma i la Pedra, La",
			'25161' => "Conca de Dalt",
			'25078' => "Corbins",
			'25079' => "Cubells",
			'25081' => "Espluga Calba, L'",
			'25082' => "Espot",
			'25088' => "Estamariu",
			'25085' => "Estaràs",
			'25086' => "Esterri d'Àneu",
			'25087' => "Esterri de Cardós",
			'25089' => "Farrera",
			'25908' => "Fígols i Alinyà",
			'25092' => "Floresta, La",
			'25093' => "Fondarella",
			'25094' => "Foradada",
			'25096' => "Fuliola, La",
			'25097' => "Fulleda",
			'25098' => "Gavet de la Conca",
			'25912' => "Gimenells i el Pla de la Font",
			'25099' => "Golmés",
			'25100' => "Gósol",
			'25101' => "Granadella, La",
			'25102' => "Granja d'Escarp, La",
			'25103' => "Granyanella",
			'25105' => "Granyena de les Garrigues",
			'25104' => "Granyena de Segarra",
			'25109' => "Guimerà",
			'25903' => "Guingueta d'Àneu, La",
			'25110' => "Guissona",
			'25111' => "Guixers",
			'25115' => "Isona i Conca Dellà",
			'25112' => "Ivars de Noguera",
			'25113' => "Ivars d'Urgell",
			'25114' => "Ivorra",
			'25910' => "Josa i Tuixén",
			'25118' => "Juncosa",
			'25119' => "Juneda",
			'25121' => "Les",
			'25122' => "Linyola",
			'25123' => "Lladorre",
			'25124' => "Lladurs",
			'25125' => "Llardecans",
			'25126' => "Llavorsí",
			'25120' => "Lleida",
			'25127' => "Lles de Cerdanya",
			'25128' => "Llimiana",
			'25129' => "Llobera",
			'25133' => "Maials",
			'25130' => "Maldà",
			'25131' => "Massalcoreig",
			'25132' => "Massoteres",
			'25134' => "Menàrguens",
			'25135' => "Miralcamp",
			'25137' => "Mollerussa",
			'25136' => "Molsosa, La",
			'25139' => "Montellà i Martinet",
			'25140' => "Montferrer i Castellbò",
			'25138' => "Montgai",
			'25142' => "Montoliu de Lleida",
			'25141' => "Montoliu de Segarra",
			'25143' => "Montornès de Segarra",
			'25145' => "Nalec",
			'25025' => "Naut Aran",
			'25146' => "Navès",
			'25148' => "Odèn",
			'25149' => "Oliana",
			'25150' => "Oliola",
			'25151' => "Olius",
			'25152' => "Oluges, Les",
			'25153' => "Omellons, Els",
			'25154' => "Omells de na Gaia, Els",
			'25155' => "Organyà",
			'25156' => "Os de Balaguer",
			'25157' => "Ossó de Sió",
			'25158' => "Palau d'Anglesola, El",
			'25164' => "Penelles",
			'25165' => "Peramola",
			'25166' => "Pinell de Solsonès",
			'25167' => "Pinós",
			'25911' => "Plans de Sió, Els",
			'25168' => "Poal, El",
			'25169' => "Pobla de Cérvoles, La",
			'25171' => "Pobla de Segur, La",
			'25030' => "Pont de Bar, El",
			'25173' => "Pont de Suert, El",
			'25172' => "Ponts",
			'25174' => "Portella, La",
			'25175' => "Prats i Sansor",
			'25176' => "Preixana",
			'25177' => "Preixens",
			'25179' => "Prullans",
			'25180' => "Puiggròs",
			'25181' => "Puigverd d'Agramunt",
			'25182' => "Puigverd de Lleida",
			'25183' => "Rialp",
			'25905' => "Ribera d'Ondara",
			'25185' => "Ribera d'Urgellet",
			'25186' => "Riner",
			'25913' => "Riu de Cerdanya",
			'25189' => "Rosselló",
			'25190' => "Salàs de Pallars",
			'25191' => "Sanaüja",
			'25196' => "Sant Esteve de la Sarga",
			'25192' => "Sant Guim de Freixenet",
			'25197' => "Sant Guim de la Plana",
			'25193' => "Sant Llorenç de Morunys",
			'25902' => "Sant Martí de Riucorb",
			'25194' => "Sant Ramon",
			'25201' => "Sarroca de Bellera",
			'25200' => "Sarroca de Lleida",
			'25202' => "Senterada",
			'25035' => "Sentiu de Sió, La",
			'25204' => "Seròs",
			'25203' => "Seu d'Urgell, La",
			'25205' => "Sidamon",
			'25206' => "Soleràs, El",
			'25207' => "Solsona",
			'25208' => "Soriguera",
			'25209' => "Sort",
			'25210' => "Soses",
			'25211' => "Sudanell",
			'25212' => "Sunyer",
			'25215' => "Talarn",
			'25216' => "Talavera",
			'25217' => "Tàrrega",
			'25218' => "Tarrés",
			'25219' => "Tarroja de Segarra",
			'25220' => "Térmens",
			'25221' => "Tírvia",
			'25222' => "Tiurana",
			'25223' => "Torà",
			'25224' => "Torms, Els",
			'25225' => "Tornabous",
			'25227' => "Torre de Cabdella, La",
			'25226' => "Torrebesses",
			'25228' => "Torrefarrera",
			'25907' => "Torrefeta i Florejacs",
			'25230' => "Torregrossa",
			'25231' => "Torrelameu",
			'25232' => "Torres de Segre",
			'25233' => "Torre-serona",
			'25234' => "Tremp",
			'25043' => "Vall de Boí, La",
			'25901' => "Vall de Cardós",
			'25238' => "Vallbona de les Monges",
			'25240' => "Vallfogona de Balaguer",
			'25906' => "Valls d'Aguilar, Les",
			'25239' => "Valls de Valira, Les",
			'25909' => "Vansa i Fórnols, La",
			'25242' => "Verdú",
			'25243' => "Vielha e Mijaran",
			'25244' => "Vilagrassa",
			'25245' => "Vilaller",
			'25247' => "Vilamòs",
			'25248' => "Vilanova de Bellpuig",
			'25254' => "Vilanova de la Barca",
			'25249' => "Vilanova de l'Aguda",
			'25250' => "Vilanova de Meià",
			'25251' => "Vilanova de Segrià",
			'25252' => "Vila-sana",
			'25253' => "Vilosell, El",
			'25255' => "Vinaixa",
			'26001' => "Ábalos",
			'26002' => "Agoncillo",
			'26003' => "Aguilar del Río Alhama",
			'26004' => "Ajamil de Cameros",
			'26005' => "Albelda de Iregua",
			'26006' => "Alberite",
			'26007' => "Alcanadre",
			'26008' => "Aldeanueva de Ebro",
			'26009' => "Alesanco",
			'26010' => "Alesón",
			'26011' => "Alfaro",
			'26012' => "Almarza de Cameros",
			'26013' => "Anguciana",
			'26014' => "Anguiano",
			'26015' => "Arenzana de Abajo",
			'26016' => "Arenzana de Arriba",
			'26017' => "Arnedillo",
			'26018' => "Arnedo",
			'26019' => "Arrúbal",
			'26020' => "Ausejo",
			'26021' => "Autol",
			'26022' => "Azofra",
			'26023' => "Badarán",
			'26024' => "Bañares",
			'26026' => "Baños de Río Tobía",
			'26025' => "Baños de Rioja",
			'26027' => "Berceo",
			'26028' => "Bergasa",
			'26029' => "Bergasillas Bajera",
			'26030' => "Bezares",
			'26031' => "Bobadilla",
			'26032' => "Brieva de Cameros",
			'26033' => "Briñas",
			'26034' => "Briones",
			'26035' => "Cabezón de Cameros",
			'26036' => "Calahorra",
			'26037' => "Camprovín",
			'26038' => "Canales de la Sierra",
			'26039' => "Canillas de Río Tuerto",
			'26040' => "Cañas",
			'26041' => "Cárdenas",
			'26042' => "Casalarreina",
			'26043' => "Castañares de Rioja",
			'26044' => "Castroviejo",
			'26045' => "Cellorigo",
			'26046' => "Cenicero",
			'26047' => "Cervera del Río Alhama",
			'26048' => "Cidamón",
			'26049' => "Cihuri",
			'26050' => "Cirueña",
			'26051' => "Clavijo",
			'26052' => "Cordovín",
			'26053' => "Corera",
			'26054' => "Cornago",
			'26055' => "Corporales",
			'26056' => "Cuzcurrita de Río Tirón",
			'26057' => "Daroca de Rioja",
			'26058' => "Enciso",
			'26059' => "Entrena",
			'26060' => "Estollo",
			'26061' => "Ezcaray",
			'26062' => "Foncea",
			'26063' => "Fonzaleche",
			'26064' => "Fuenmayor",
			'26065' => "Galbárruli",
			'26066' => "Galilea",
			'26067' => "Gallinero de Cameros",
			'26068' => "Gimileo",
			'26069' => "Grañón",
			'26070' => "Grávalos",
			'26071' => "Haro",
			'26072' => "Herce",
			'26073' => "Herramélluri",
			'26074' => "Hervías",
			'26075' => "Hormilla",
			'26076' => "Hormilleja",
			'26077' => "Hornillos de Cameros",
			'26078' => "Hornos de Moncalvillo",
			'26079' => "Huércanos",
			'26080' => "Igea",
			'26081' => "Jalón de Cameros",
			'26082' => "Laguna de Cameros",
			'26083' => "Lagunilla del Jubera",
			'26084' => "Lardero",
			'26086' => "Ledesma de la Cogolla",
			'26087' => "Leiva",
			'26088' => "Leza de Río Leza",
			'26089' => "Logroño",
			'26091' => "Lumbreras de Cameros",
			'26092' => "Manjarrés",
			'26093' => "Mansilla de la Sierra",
			'26094' => "Manzanares de Rioja",
			'26095' => "Matute",
			'26096' => "Medrano",
			'26098' => "Munilla",
			'26099' => "Murillo de Río Leza",
			'26100' => "Muro de Aguas",
			'26101' => "Muro en Cameros",
			'26102' => "Nájera",
			'26103' => "Nalda",
			'26104' => "Navajún",
			'26105' => "Navarrete",
			'26106' => "Nestares",
			'26107' => "Nieva de Cameros",
			'26109' => "Ochánduri",
			'26108' => "Ocón",
			'26110' => "Ojacastro",
			'26111' => "Ollauri",
			'26112' => "Ortigosa de Cameros",
			'26113' => "Pazuengos",
			'26114' => "Pedroso",
			'26115' => "Pinillos",
			'26117' => "Pradejón",
			'26118' => "Pradillo",
			'26119' => "Préjano",
			'26120' => "Quel",
			'26121' => "Rabanera",
			'26122' => "Rasillo de Cameros, El",
			'26123' => "Redal, El",
			'26124' => "Ribafrecha",
			'26125' => "Rincón de Soto",
			'26126' => "Robres del Castillo",
			'26127' => "Rodezno",
			'26128' => "Sajazarra",
			'26129' => "San Asensio",
			'26130' => "San Millán de la Cogolla",
			'26131' => "San Millán de Yécora",
			'26132' => "San Román de Cameros",
			'26139' => "San Torcuato",
			'26142' => "San Vicente de la Sonsierra",
			'26134' => "Santa Coloma",
			'26135' => "Santa Engracia del Jubera",
			'26136' => "Santa Eulalia Bajera",
			'26138' => "Santo Domingo de la Calzada",
			'26140' => "Santurde de Rioja",
			'26141' => "Santurdejo",
			'26143' => "Sojuela",
			'26144' => "Sorzano",
			'26145' => "Sotés",
			'26146' => "Soto en Cameros",
			'26147' => "Terroba",
			'26148' => "Tirgo",
			'26149' => "Tobía",
			'26150' => "Tormantos",
			'26153' => "Torre en Cameros",
			'26151' => "Torrecilla en Cameros",
			'26152' => "Torrecilla sobre Alesanco",
			'26154' => "Torremontalbo",
			'26155' => "Treviana",
			'26157' => "Tricio",
			'26158' => "Tudelilla",
			'26160' => "Uruñuela",
			'26161' => "Valdemadera",
			'26162' => "Valgañón",
			'26163' => "Ventosa",
			'26164' => "Ventrosa",
			'26165' => "Viguera",
			'26166' => "Villalba de Rioja",
			'26167' => "Villalobar de Rioja",
			'26168' => "Villamediana de Iregua",
			'26169' => "Villanueva de Cameros",
			'26170' => "Villar de Arnedo, El",
			'26171' => "Villar de Torre",
			'26172' => "Villarejo",
			'26173' => "Villarroya",
			'26174' => "Villarta-Quintana",
			'26175' => "Villavelayo",
			'26176' => "Villaverde de Rioja",
			'26177' => "Villoslada de Cameros",
			'26178' => "Viniegra de Abajo",
			'26179' => "Viniegra de Arriba",
			'26180' => "Zarratón",
			'26181' => "Zarzosa",
			'26183' => "Zorraquín",
			'27001' => "Abadín",
			'27002' => "Alfoz",
			'27003' => "Antas de Ulla",
			'27004' => "Baleira",
			'27901' => "Baralla",
			'27005' => "Barreiros",
			'27006' => "Becerreá",
			'27007' => "Begonte",
			'27008' => "Bóveda",
			'27902' => "Burela",
			'27009' => "Carballedo",
			'27010' => "Castro de Rei",
			'27011' => "Castroverde",
			'27012' => "Cervantes",
			'27013' => "Cervo",
			'27016' => "Chantada",
			'27014' => "Corgo, O",
			'27015' => "Cospeito",
			'27017' => "Folgoso do Courel",
			'27018' => "Fonsagrada, A",
			'27019' => "Foz",
			'27020' => "Friol",
			'27022' => "Guitiriz",
			'27023' => "Guntín",
			'27024' => "Incio, O",
			'27026' => "Láncara",
			'27027' => "Lourenzá",
			'27028' => "Lugo",
			'27029' => "Meira",
			'27030' => "Mondoñedo",
			'27031' => "Monforte de Lemos",
			'27032' => "Monterroso",
			'27033' => "Muras",
			'27034' => "Navia de Suarna",
			'27035' => "Negueira de Muñiz",
			'27037' => "Nogais, As",
			'27038' => "Ourol",
			'27039' => "Outeiro de Rei",
			'27040' => "Palas de Rei",
			'27041' => "Pantón",
			'27042' => "Paradela",
			'27043' => "Páramo, O",
			'27044' => "Pastoriza, A",
			'27045' => "Pedrafita do Cebreiro",
			'27047' => "Pobra do Brollón, A",
			'27046' => "Pol",
			'27048' => "Pontenova, A",
			'27049' => "Portomarín",
			'27050' => "Quiroga",
			'27056' => "Rábade",
			'27051' => "Ribadeo",
			'27052' => "Ribas de Sil",
			'27053' => "Ribeira de Piquín",
			'27054' => "Riotorto",
			'27055' => "Samos",
			'27057' => "Sarria",
			'27058' => "Saviñao, O",
			'27059' => "Sober",
			'27060' => "Taboada",
			'27061' => "Trabada",
			'27062' => "Triacastela",
			'27063' => "Valadouro, O",
			'27064' => "Vicedo, O",
			'27065' => "Vilalba",
			'27066' => "Viveiro",
			'27021' => "Xermade",
			'27025' => "Xove",
			'28001' => "Acebeda, La",
			'28002' => "Ajalvir",
			'28003' => "Alameda del Valle",
			'28004' => "Álamo, El",
			'28005' => "Alcalá de Henares",
			'28006' => "Alcobendas",
			'28007' => "Alcorcón",
			'28008' => "Aldea del Fresno",
			'28009' => "Algete",
			'28010' => "Alpedrete",
			'28011' => "Ambite",
			'28012' => "Anchuelo",
			'28013' => "Aranjuez",
			'28014' => "Arganda del Rey",
			'28015' => "Arroyomolinos",
			'28016' => "Atazar, El",
			'28017' => "Batres",
			'28018' => "Becerril de la Sierra",
			'28019' => "Belmonte de Tajo",
			'28021' => "Berrueco, El",
			'28020' => "Berzosa del Lozoya",
			'28022' => "Boadilla del Monte",
			'28023' => "Boalo, El",
			'28024' => "Braojos",
			'28025' => "Brea de Tajo",
			'28026' => "Brunete",
			'28027' => "Buitrago del Lozoya",
			'28028' => "Bustarviejo",
			'28029' => "Cabanillas de la Sierra",
			'28030' => "Cabrera, La",
			'28031' => "Cadalso de los Vidrios",
			'28032' => "Camarma de Esteruelas",
			'28033' => "Campo Real",
			'28034' => "Canencia",
			'28035' => "Carabaña",
			'28036' => "Casarrubuelos",
			'28037' => "Cenicientos",
			'28038' => "Cercedilla",
			'28039' => "Cervera de Buitrago",
			'28051' => "Chapinería",
			'28052' => "Chinchón",
			'28040' => "Ciempozuelos",
			'28041' => "Cobeña",
			'28046' => "Collado Mediano",
			'28047' => "Collado Villalba",
			'28043' => "Colmenar de Oreja",
			'28042' => "Colmenar del Arroyo",
			'28045' => "Colmenar Viejo",
			'28044' => "Colmenarejo",
			'28048' => "Corpa",
			'28049' => "Coslada",
			'28050' => "Cubas de la Sagra",
			'28053' => "Daganzo de Arriba",
			'28054' => "Escorial, El",
			'28055' => "Estremera",
			'28056' => "Fresnedillas de la Oliva",
			'28057' => "Fresno de Torote",
			'28058' => "Fuenlabrada",
			'28059' => "Fuente el Saz de Jarama",
			'28060' => "Fuentidueña de Tajo",
			'28061' => "Galapagar",
			'28062' => "Garganta de los Montes",
			'28063' => "Gargantilla del Lozoya y Pinilla de Buitrago",
			'28064' => "Gascones",
			'28065' => "Getafe",
			'28066' => "Griñón",
			'28067' => "Guadalix de la Sierra",
			'28068' => "Guadarrama",
			'28069' => "Hiruela, La",
			'28070' => "Horcajo de la Sierra-Aoslos",
			'28071' => "Horcajuelo de la Sierra",
			'28072' => "Hoyo de Manzanares",
			'28073' => "Humanes de Madrid",
			'28074' => "Leganés",
			'28075' => "Loeches",
			'28076' => "Lozoya",
			'28901' => "Lozoyuela-Navas-Sieteiglesias",
			'28078' => "Madarcos",
			'28079' => "Madrid",
			'28080' => "Majadahonda",
			'28082' => "Manzanares el Real",
			'28083' => "Meco",
			'28084' => "Mejorada del Campo",
			'28085' => "Miraflores de la Sierra",
			'28086' => "Molar, El",
			'28087' => "Molinos, Los",
			'28088' => "Montejo de la Sierra",
			'28089' => "Moraleja de Enmedio",
			'28090' => "Moralzarzal",
			'28091' => "Morata de Tajuña",
			'28092' => "Móstoles",
			'28093' => "Navacerrada",
			'28094' => "Navalafuente",
			'28095' => "Navalagamella",
			'28096' => "Navalcarnero",
			'28097' => "Navarredonda y San Mamés",
			'28099' => "Navas del Rey",
			'28100' => "Nuevo Baztán",
			'28101' => "Olmeda de las Fuentes",
			'28102' => "Orusco de Tajuña",
			'28104' => "Paracuellos de Jarama",
			'28106' => "Parla",
			'28107' => "Patones",
			'28108' => "Pedrezuela",
			'28109' => "Pelayos de la Presa",
			'28110' => "Perales de Tajuña",
			'28111' => "Pezuela de las Torres",
			'28112' => "Pinilla del Valle",
			'28113' => "Pinto",
			'28114' => "Piñuécar-Gandullas",
			'28115' => "Pozuelo de Alarcón",
			'28116' => "Pozuelo del Rey",
			'28117' => "Prádena del Rincón",
			'28118' => "Puebla de la Sierra",
			'28902' => "Puentes Viejas",
			'28119' => "Quijorna",
			'28120' => "Rascafría",
			'28121' => "Redueña",
			'28122' => "Ribatejada",
			'28123' => "Rivas-Vaciamadrid",
			'28124' => "Robledillo de la Jara",
			'28125' => "Robledo de Chavela",
			'28126' => "Robregordo",
			'28127' => "Rozas de Madrid, Las",
			'28128' => "Rozas de Puerto Real",
			'28129' => "San Agustín del Guadalix",
			'28130' => "San Fernando de Henares",
			'28131' => "San Lorenzo de El Escorial",
			'28132' => "San Martín de la Vega",
			'28133' => "San Martín de Valdeiglesias",
			'28134' => "San Sebastián de los Reyes",
			'28135' => "Santa María de la Alameda",
			'28136' => "Santorcaz",
			'28137' => "Santos de la Humosa, Los",
			'28138' => "Serna del Monte, La",
			'28140' => "Serranillos del Valle",
			'28141' => "Sevilla la Nueva",
			'28143' => "Somosierra",
			'28144' => "Soto del Real",
			'28145' => "Talamanca de Jarama",
			'28146' => "Tielmes",
			'28147' => "Titulcia",
			'28148' => "Torrejón de Ardoz",
			'28149' => "Torrejón de la Calzada",
			'28150' => "Torrejón de Velasco",
			'28151' => "Torrelaguna",
			'28152' => "Torrelodones",
			'28153' => "Torremocha de Jarama",
			'28154' => "Torres de la Alameda",
			'28903' => "Tres Cantos",
			'28155' => "Valdaracete",
			'28156' => "Valdeavero",
			'28157' => "Valdelaguna",
			'28158' => "Valdemanco",
			'28159' => "Valdemaqueda",
			'28160' => "Valdemorillo",
			'28161' => "Valdemoro",
			'28162' => "Valdeolmos-Alalpardo",
			'28163' => "Valdepiélagos",
			'28164' => "Valdetorres de Jarama",
			'28165' => "Valdilecha",
			'28166' => "Valverde de Alcalá",
			'28167' => "Velilla de San Antonio",
			'28168' => "Vellón, El",
			'28169' => "Venturada",
			'28171' => "Villa del Prado",
			'28170' => "Villaconejos",
			'28172' => "Villalbilla",
			'28173' => "Villamanrique de Tajo",
			'28174' => "Villamanta",
			'28175' => "Villamantilla",
			'28176' => "Villanueva de la Cañada",
			'28178' => "Villanueva de Perales",
			'28177' => "Villanueva del Pardillo",
			'28179' => "Villar del Olmo",
			'28180' => "Villarejo de Salvanés",
			'28181' => "Villaviciosa de Odón",
			'28182' => "Villavieja del Lozoya",
			'28183' => "Zarzalejo",
			'29001' => "Alameda",
			'29002' => "Alcaucín",
			'29003' => "Alfarnate",
			'29004' => "Alfarnatejo",
			'29005' => "Algarrobo",
			'29006' => "Algatocín",
			'29007' => "Alhaurín de la Torre",
			'29008' => "Alhaurín el Grande",
			'29009' => "Almáchar",
			'29010' => "Almargen",
			'29011' => "Almogía",
			'29012' => "Álora",
			'29013' => "Alozaina",
			'29014' => "Alpandeire",
			'29015' => "Antequera",
			'29016' => "Árchez",
			'29017' => "Archidona",
			'29018' => "Ardales",
			'29019' => "Arenas",
			'29020' => "Arriate",
			'29021' => "Atajate",
			'29022' => "Benadalid",
			'29023' => "Benahavís",
			'29024' => "Benalauría",
			'29025' => "Benalmádena",
			'29026' => "Benamargosa",
			'29027' => "Benamocarra",
			'29028' => "Benaoján",
			'29029' => "Benarrabá",
			'29030' => "Borge, El",
			'29031' => "Burgo, El",
			'29032' => "Campillos",
			'29033' => "Canillas de Aceituno",
			'29034' => "Canillas de Albaida",
			'29035' => "Cañete la Real",
			'29036' => "Carratraca",
			'29037' => "Cartajima",
			'29038' => "Cártama",
			'29039' => "Casabermeja",
			'29040' => "Casarabonela",
			'29041' => "Casares",
			'29042' => "Coín",
			'29043' => "Colmenar",
			'29044' => "Comares",
			'29045' => "Cómpeta",
			'29046' => "Cortes de la Frontera",
			'29047' => "Cuevas Bajas",
			'29049' => "Cuevas de San Marcos",
			'29048' => "Cuevas del Becerro",
			'29050' => "Cútar",
			'29051' => "Estepona",
			'29052' => "Faraján",
			'29053' => "Frigiliana",
			'29054' => "Fuengirola",
			'29055' => "Fuente de Piedra",
			'29056' => "Gaucín",
			'29057' => "Genalguacil",
			'29058' => "Guaro",
			'29059' => "Humilladero",
			'29060' => "Igualeja",
			'29061' => "Istán",
			'29062' => "Iznate",
			'29063' => "Jimera de Líbar",
			'29064' => "Jubrique",
			'29065' => "Júzcar",
			'29066' => "Macharaviaya",
			'29067' => "Málaga",
			'29068' => "Manilva",
			'29069' => "Marbella",
			'29070' => "Mijas",
			'29071' => "Moclinejo",
			'29072' => "Mollina",
			'29073' => "Monda",
			'29903' => "Montecorto",
			'29074' => "Montejaque",
			'29075' => "Nerja",
			'29076' => "Ojén",
			'29077' => "Parauta",
			'29079' => "Periana",
			'29080' => "Pizarra",
			'29081' => "Pujerra",
			'29082' => "Rincón de la Victoria",
			'29083' => "Riogordo",
			'29084' => "Ronda",
			'29085' => "Salares",
			'29086' => "Sayalonga",
			'29087' => "Sedella",
			'29904' => "Serrato",
			'29088' => "Sierra de Yeguas",
			'29089' => "Teba",
			'29090' => "Tolox",
			'29901' => "Torremolinos",
			'29091' => "Torrox",
			'29092' => "Totalán",
			'29093' => "Valle de Abdalajís",
			'29094' => "Vélez-Málaga",
			'29095' => "Villanueva de Algaidas",
			'29902' => "Villanueva de la Concepción",
			'29098' => "Villanueva de Tapia",
			'29096' => "Villanueva del Rosario",
			'29097' => "Villanueva del Trabuco",
			'29099' => "Viñuela",
			'29100' => "Yunquera",
			'30001' => "Abanilla",
			'30002' => "Abarán",
			'30003' => "Águilas",
			'30004' => "Albudeite",
			'30005' => "Alcantarilla",
			'30902' => "Alcázares, Los",
			'30006' => "Aledo",
			'30007' => "Alguazas",
			'30008' => "Alhama de Murcia",
			'30009' => "Archena",
			'30010' => "Beniel",
			'30011' => "Blanca",
			'30012' => "Bullas",
			'30013' => "Calasparra",
			'30014' => "Campos del Río",
			'30015' => "Caravaca de la Cruz",
			'30016' => "Cartagena",
			'30017' => "Cehegín",
			'30018' => "Ceutí",
			'30019' => "Cieza",
			'30020' => "Fortuna",
			'30021' => "Fuente Álamo de Murcia",
			'30022' => "Jumilla",
			'30023' => "Librilla",
			'30024' => "Lorca",
			'30025' => "Lorquí",
			'30026' => "Mazarrón",
			'30027' => "Molina de Segura",
			'30028' => "Moratalla",
			'30029' => "Mula",
			'30030' => "Murcia",
			'30031' => "Ojós",
			'30032' => "Pliego",
			'30033' => "Puerto Lumbreras",
			'30034' => "Ricote",
			'30035' => "San Javier",
			'30036' => "San Pedro del Pinatar",
			'30901' => "Santomera",
			'30037' => "Torre-Pacheco",
			'30038' => "Torres de Cotillas, Las",
			'30039' => "Totana",
			'30040' => "Ulea",
			'30041' => "Unión, La",
			'30042' => "Villanueva del Río Segura",
			'30043' => "Yecla",
			'31001' => "Abáigar",
			'31002' => "Abárzuza/Abartzuza",
			'31003' => "Abaurregaina/Abaurrea Alta",
			'31004' => "Abaurrepea/Abaurrea Baja",
			'31005' => "Aberin",
			'31006' => "Ablitas",
			'31007' => "Adiós",
			'31008' => "Aguilar de Codés",
			'31009' => "Aibar/Oibar",
			'31011' => "Allín/Allin",
			'31012' => "Allo",
			'31010' => "Altsasu/Alsasua",
			'31013' => "Améscoa Baja",
			'31014' => "Ancín/Antzin",
			'31015' => "Andosilla",
			'31016' => "Ansoáin/Antsoain",
			'31017' => "Anue",
			'31018' => "Añorbe",
			'31019' => "Aoiz/Agoitz",
			'31020' => "Araitz",
			'31025' => "Arakil",
			'31021' => "Aranarache/Aranaratxe",
			'31023' => "Aranguren",
			'31024' => "Arano",
			'31022' => "Arantza",
			'31026' => "Aras",
			'31027' => "Arbizu",
			'31028' => "Arce/Artzi",
			'31029' => "Arcos, Los",
			'31030' => "Arellano",
			'31031' => "Areso",
			'31032' => "Arguedas",
			'31033' => "Aria",
			'31034' => "Aribe",
			'31035' => "Armañanzas",
			'31036' => "Arróniz",
			'31037' => "Arruazu",
			'31038' => "Artajona",
			'31039' => "Artazu",
			'31040' => "Atetz",
			'31058' => "Auritz/Burguete",
			'31041' => "Ayegui/Aiegi",
			'31042' => "Azagra",
			'31043' => "Azuelo",
			'31044' => "Bakaiku",
			'31901' => "Barañáin/Barañain",
			'31045' => "Barásoain",
			'31046' => "Barbarin",
			'31047' => "Bargota",
			'31048' => "Barillas",
			'31049' => "Basaburua",
			'31050' => "Baztan",
			'31137' => "Beintza-Labaien",
			'31051' => "Beire",
			'31052' => "Belascoáin",
			'31250' => "Bera",
			'31053' => "Berbinzana",
			'31905' => "Beriáin",
			'31902' => "Berrioplano/Berriobeiti",
			'31903' => "Berriozar",
			'31054' => "Bertizarana",
			'31055' => "Betelu",
			'31253' => "Bidaurreta",
			'31056' => "Biurrun-Olcoz",
			'31057' => "Buñuel",
			'31059' => "Burgui/Burgi",
			'31060' => "Burlada/Burlata",
			'31061' => "Busto, El",
			'31062' => "Cabanillas",
			'31063' => "Cabredo",
			'31064' => "Cadreita",
			'31065' => "Caparroso",
			'31066' => "Cárcar",
			'31067' => "Carcastillo",
			'31068' => "Cascante",
			'31069' => "Cáseda",
			'31070' => "Castejón",
			'31071' => "Castillonuevo",
			'31193' => "Cendea de Olza/Oltza Zendea",
			'31072' => "Cintruénigo",
			'31074' => "Cirauqui/Zirauki",
			'31075' => "Ciriza/Ziritza",
			'31076' => "Cizur",
			'31077' => "Corella",
			'31078' => "Cortes",
			'31079' => "Desojo",
			'31080' => "Dicastillo",
			'31081' => "Donamaria",
			'31221' => "Doneztebe/Santesteban",
			'31083' => "Echarri/Etxarri",
			'31087' => "Elgorriaga",
			'31089' => "Enériz/Eneritz",
			'31090' => "Eratsun",
			'31091' => "Ergoiena",
			'31092' => "Erro",
			'31094' => "Eslava",
			'31095' => "Esparza de Salazar/Espartza Zaraitzu",
			'31096' => "Espronceda",
			'31097' => "Estella-Lizarra",
			'31098' => "Esteribar",
			'31099' => "Etayo",
			'31082' => "Etxalar",
			'31084' => "Etxarri Aranatz",
			'31085' => "Etxauri",
			'31100' => "Eulate",
			'31101' => "Ezcabarte",
			'31093' => "Ezcároz/Ezkaroze",
			'31102' => "Ezkurra",
			'31103' => "Ezprogui",
			'31104' => "Falces",
			'31105' => "Fitero",
			'31106' => "Fontellas",
			'31107' => "Funes",
			'31108' => "Fustiñana",
			'31109' => "Galar",
			'31110' => "Gallipienzo/Galipentzu",
			'31111' => "Gallués/Galoze",
			'31112' => "Garaioa",
			'31113' => "Garde",
			'31114' => "Garínoain",
			'31115' => "Garralda",
			'31116' => "Genevilla",
			'31117' => "Goizueta",
			'31118' => "Goñi",
			'31119' => "Güesa/Gorza",
			'31120' => "Guesálaz/Gesalatz",
			'31121' => "Guirguillano",
			'31256' => "Hiriberri/Villanueva de Aezkoa",
			'31122' => "Huarte/Uharte",
			'31124' => "Ibargoiti",
			'31259' => "Igantzi",
			'31125' => "Igúzquiza",
			'31126' => "Imotz",
			'31127' => "Irañeta",
			'31904' => "Irurtzun",
			'31128' => "Isaba/Izaba",
			'31129' => "Ituren",
			'31130' => "Iturmendi",
			'31131' => "Iza/Itza",
			'31132' => "Izagaondoa",
			'31133' => "Izalzu/Itzaltzu",
			'31134' => "Jaurrieta",
			'31135' => "Javier",
			'31136' => "Juslapeña/Txulapain",
			'31138' => "Lakuntza",
			'31139' => "Lana",
			'31140' => "Lantz",
			'31141' => "Lapoblación",
			'31142' => "Larraga",
			'31143' => "Larraona",
			'31144' => "Larraun",
			'31145' => "Lazagurría",
			'31146' => "Leache/Leatxe",
			'31147' => "Legarda",
			'31148' => "Legaria",
			'31149' => "Leitza",
			'31908' => "Lekunberri",
			'31150' => "Leoz/Leotz",
			'31151' => "Lerga",
			'31152' => "Lerín",
			'31153' => "Lesaka",
			'31154' => "Lezaun",
			'31155' => "Liédena",
			'31156' => "Lizoain-Arriasgoiti/Lizoainibar-Arriasgoiti",
			'31157' => "Lodosa",
			'31158' => "Lónguida/Longida",
			'31159' => "Lumbier",
			'31160' => "Luquin",
			'31248' => "Luzaide/Valcarlos",
			'31161' => "Mañeru",
			'31162' => "Marañón",
			'31163' => "Marcilla",
			'31164' => "Mélida",
			'31165' => "Mendavia",
			'31166' => "Mendaza",
			'31167' => "Mendigorria",
			'31168' => "Metauten",
			'31169' => "Milagro",
			'31170' => "Mirafuentes",
			'31171' => "Miranda de Arga",
			'31172' => "Monreal/Elo",
			'31173' => "Monteagudo",
			'31174' => "Morentin",
			'31175' => "Mues",
			'31176' => "Murchante",
			'31177' => "Murieta",
			'31178' => "Murillo el Cuende",
			'31179' => "Murillo el Fruto",
			'31180' => "Muruzábal",
			'31181' => "Navascués/Nabaskoze",
			'31182' => "Nazar",
			'31088' => "Noáin (Valle de Elorz)/Noain (Elortzibar)",
			'31183' => "Obanos",
			'31185' => "Ochagavía/Otsagabia",
			'31184' => "Oco",
			'31186' => "Odieta",
			'31187' => "Oiz",
			'31188' => "Olaibar",
			'31189' => "Olazti/Olazagutía",
			'31190' => "Olejua",
			'31191' => "Olite/Erriberri",
			'31192' => "Olóriz/Oloritz",
			'31195' => "Orbaizeta",
			'31196' => "Orbara",
			'31197' => "Orísoain",
			'31906' => "Orkoien",
			'31198' => "Oronz/Orontze",
			'31199' => "Oroz-Betelu/Orotz-Betelu",
			'31211' => "Orreaga/Roncesvalles",
			'31200' => "Oteiza",
			'31201' => "Pamplona/Iruña",
			'31202' => "Peralta/Azkoien",
			'31203' => "Petilla de Aragón",
			'31204' => "Piedramillera",
			'31205' => "Pitillas",
			'31206' => "Puente la Reina/Gares",
			'31207' => "Pueyo/Puiu",
			'31208' => "Ribaforada",
			'31209' => "Romanzado/Erromantzatua",
			'31210' => "Roncal/Erronkari",
			'31212' => "Sada",
			'31213' => "Saldías",
			'31214' => "Salinas de Oro/Jaitz",
			'31215' => "San Adrián",
			'31217' => "San Martín de Unx",
			'31216' => "Sangüesa/Zangoza",
			'31219' => "Sansol",
			'31220' => "Santacara",
			'31222' => "Sarriés/Sartze",
			'31223' => "Sartaguda",
			'31224' => "Sesma",
			'31225' => "Sorlada",
			'31226' => "Sunbilla",
			'31227' => "Tafalla",
			'31228' => "Tiebas-Muruarte de Reta",
			'31229' => "Tirapu",
			'31230' => "Torralba del Río",
			'31231' => "Torres del Río",
			'31232' => "Tudela",
			'31233' => "Tulebras",
			'31234' => "Ucar",
			'31123' => "Uharte Arakil",
			'31235' => "Ujué/Uxue",
			'31236' => "Ultzama",
			'31237' => "Unciti",
			'31238' => "Unzué/Untzue",
			'31239' => "Urdazubi/Urdax",
			'31240' => "Urdiain",
			'31241' => "Urraul Alto",
			'31242' => "Urraul Bajo",
			'31244' => "Urroz",
			'31243' => "Urroz-Villa",
			'31245' => "Urzainqui/Urzainki",
			'31246' => "Uterga",
			'31247' => "Uztárroz/Uztarroze",
			'31086' => "Valle de Egüés/Eguesibar",
			'31194' => "Valle de Ollo/Ollaran",
			'31260' => "Valle de Yerri/Deierri",
			'31249' => "Valtierra",
			'31251' => "Viana",
			'31252' => "Vidángoz/Bidankoze",
			'31254' => "Villafranca",
			'31255' => "Villamayor de Monjardín",
			'31257' => "Villatuerta",
			'31258' => "Villava/Atarrabia",
			'31261' => "Yesa",
			'31262' => "Zabalza/Zabaltza",
			'31073' => "Ziordia",
			'31907' => "Zizur Mayor/Zizur Nagusia",
			'31263' => "Zubieta",
			'31264' => "Zugarramurdi",
			'31265' => "Zúñiga",
			'32001' => "Allariz",
			'32002' => "Amoeiro",
			'32003' => "Arnoia, A",
			'32004' => "Avión",
			'32005' => "Baltar",
			'32006' => "Bande",
			'32007' => "Baños de Molgas",
			'32008' => "Barbadás",
			'32009' => "Barco de Valdeorras, O",
			'32010' => "Beade",
			'32011' => "Beariz",
			'32012' => "Blancos, Os",
			'32013' => "Boborás",
			'32014' => "Bola, A",
			'32015' => "Bolo, O",
			'32016' => "Calvos de Randín",
			'32018' => "Carballeda de Avia",
			'32017' => "Carballeda de Valdeorras",
			'32019' => "Carballiño, O",
			'32020' => "Cartelle",
			'32022' => "Castrelo de Miño",
			'32021' => "Castrelo do Val",
			'32023' => "Castro Caldelas",
			'32024' => "Celanova",
			'32025' => "Cenlle",
			'32029' => "Chandrexa de Queixa",
			'32026' => "Coles",
			'32027' => "Cortegada",
			'32028' => "Cualedro",
			'32030' => "Entrimo",
			'32031' => "Esgos",
			'32033' => "Gomesende",
			'32034' => "Gudiña, A",
			'32035' => "Irixo, O",
			'32038' => "Larouco",
			'32039' => "Laza",
			'32040' => "Leiro",
			'32041' => "Lobeira",
			'32042' => "Lobios",
			'32043' => "Maceda",
			'32044' => "Manzaneda",
			'32045' => "Maside",
			'32046' => "Melón",
			'32047' => "Merca, A",
			'32048' => "Mezquita, A",
			'32049' => "Montederramo",
			'32050' => "Monterrei",
			'32051' => "Muíños",
			'32052' => "Nogueira de Ramuín",
			'32053' => "Oímbra",
			'32054' => "Ourense",
			'32055' => "Paderne de Allariz",
			'32056' => "Padrenda",
			'32057' => "Parada de Sil",
			'32058' => "Pereiro de Aguiar, O",
			'32059' => "Peroxa, A",
			'32060' => "Petín",
			'32061' => "Piñor",
			'32063' => "Pobra de Trives, A",
			'32064' => "Pontedeva",
			'32062' => "Porqueira",
			'32065' => "Punxín",
			'32066' => "Quintela de Leirado",
			'32067' => "Rairiz de Veiga",
			'32068' => "Ramirás",
			'32069' => "Ribadavia",
			'32071' => "Riós",
			'32072' => "Rúa, A",
			'32073' => "Rubiá",
			'32074' => "San Amaro",
			'32075' => "San Cibrao das Viñas",
			'32076' => "San Cristovo de Cea",
			'32070' => "San Xoán de Río",
			'32077' => "Sandiás",
			'32078' => "Sarreaus",
			'32079' => "Taboadela",
			'32080' => "Teixeira, A",
			'32081' => "Toén",
			'32082' => "Trasmiras",
			'32083' => "Veiga, A",
			'32084' => "Verea",
			'32085' => "Verín",
			'32086' => "Viana do Bolo",
			'32087' => "Vilamarín",
			'32088' => "Vilamartín de Valdeorras",
			'32089' => "Vilar de Barrio",
			'32090' => "Vilar de Santos",
			'32091' => "Vilardevós",
			'32092' => "Vilariño de Conso",
			'32032' => "Xinzo de Limia",
			'32036' => "Xunqueira de Ambía",
			'32037' => "Xunqueira de Espadanedo",
			'33001' => "Allande",
			'33002' => "Aller",
			'33003' => "Amieva",
			'33004' => "Avilés",
			'33005' => "Belmonte de Miranda",
			'33006' => "Bimenes",
			'33007' => "Boal",
			'33008' => "Cabrales",
			'33009' => "Cabranes",
			'33010' => "Candamo",
			'33012' => "Cangas de Onís",
			'33011' => "Cangas del Narcea",
			'33013' => "Caravia",
			'33014' => "Carreño",
			'33015' => "Caso",
			'33016' => "Castrillón",
			'33017' => "Castropol",
			'33018' => "Coaña",
			'33019' => "Colunga",
			'33020' => "Corvera de Asturias",
			'33021' => "Cudillero",
			'33022' => "Degaña",
			'33023' => "Franco, El",
			'33024' => "Gijón",
			'33025' => "Gozón",
			'33026' => "Grado",
			'33027' => "Grandas de Salime",
			'33028' => "Ibias",
			'33029' => "Illano",
			'33030' => "Illas",
			'33031' => "Langreo",
			'33032' => "Laviana",
			'33033' => "Lena",
			'33035' => "Llanera",
			'33036' => "Llanes",
			'33037' => "Mieres",
			'33038' => "Morcín",
			'33039' => "Muros de Nalón",
			'33040' => "Nava",
			'33041' => "Navia",
			'33042' => "Noreña",
			'33043' => "Onís",
			'33044' => "Oviedo",
			'33045' => "Parres",
			'33046' => "Peñamellera Alta",
			'33047' => "Peñamellera Baja",
			'33048' => "Pesoz",
			'33049' => "Piloña",
			'33050' => "Ponga",
			'33051' => "Pravia",
			'33052' => "Proaza",
			'33053' => "Quirós",
			'33054' => "Regueras, Las",
			'33055' => "Ribadedeva",
			'33056' => "Ribadesella",
			'33057' => "Ribera de Arriba",
			'33058' => "Riosa",
			'33059' => "Salas",
			'33061' => "San Martín de Oscos",
			'33060' => "San Martín del Rey Aurelio",
			'33063' => "San Tirso de Abres",
			'33062' => "Santa Eulalia de Oscos",
			'33064' => "Santo Adriano",
			'33065' => "Sariego",
			'33066' => "Siero",
			'33067' => "Sobrescobio",
			'33068' => "Somiedo",
			'33069' => "Soto del Barco",
			'33070' => "Tapia de Casariego",
			'33071' => "Taramundi",
			'33072' => "Teverga",
			'33073' => "Tineo",
			'33034' => "Valdés",
			'33074' => "Vegadeo",
			'33075' => "Villanueva de Oscos",
			'33076' => "Villaviciosa",
			'33077' => "Villayón",
			'33078' => "Yernes y Tameza",
			'34001' => "Abarca de Campos",
			'34003' => "Abia de las Torres",
			'34004' => "Aguilar de Campoo",
			'34005' => "Alar del Rey",
			'34006' => "Alba de Cerrato",
			'34009' => "Amayuelas de Arriba",
			'34010' => "Ampudia",
			'34011' => "Amusco",
			'34012' => "Antigüedad",
			'34015' => "Arconada",
			'34017' => "Astudillo",
			'34018' => "Autilla del Pino",
			'34019' => "Autillo de Campos",
			'34020' => "Ayuela",
			'34022' => "Baltanás",
			'34024' => "Baquerín de Campos",
			'34025' => "Bárcena de Campos",
			'34027' => "Barruelo de Santullán",
			'34028' => "Báscones de Ojeda",
			'34029' => "Becerril de Campos",
			'34031' => "Belmonte de Campos",
			'34032' => "Berzosilla",
			'34033' => "Boada de Campos",
			'34035' => "Boadilla de Rioseco",
			'34034' => "Boadilla del Camino",
			'34036' => "Brañosera",
			'34037' => "Buenavista de Valdavia",
			'34038' => "Bustillo de la Vega",
			'34039' => "Bustillo del Páramo de Carrión",
			'34041' => "Calahorra de Boedo",
			'34042' => "Calzada de los Molinos",
			'34045' => "Capillas",
			'34046' => "Cardeñosa de Volpejera",
			'34047' => "Carrión de los Condes",
			'34048' => "Castil de Vela",
			'34049' => "Castrejón de la Peña",
			'34050' => "Castrillo de Don Juan",
			'34051' => "Castrillo de Onielo",
			'34052' => "Castrillo de Villavega",
			'34053' => "Castromocho",
			'34055' => "Cervatos de la Cueza",
			'34056' => "Cervera de Pisuerga",
			'34057' => "Cevico de la Torre",
			'34058' => "Cevico Navero",
			'34059' => "Cisneros",
			'34060' => "Cobos de Cerrato",
			'34061' => "Collazos de Boedo",
			'34062' => "Congosto de Valdavia",
			'34063' => "Cordovilla la Real",
			'34066' => "Cubillas de Cerrato",
			'34067' => "Dehesa de Montejo",
			'34068' => "Dehesa de Romanos",
			'34069' => "Dueñas",
			'34070' => "Espinosa de Cerrato",
			'34071' => "Espinosa de Villagonzalo",
			'34072' => "Frechilla",
			'34073' => "Fresno del Río",
			'34074' => "Frómista",
			'34076' => "Fuentes de Nava",
			'34077' => "Fuentes de Valdepero",
			'34079' => "Grijota",
			'34080' => "Guardo",
			'34081' => "Guaza de Campos",
			'34082' => "Hérmedes de Cerrato",
			'34083' => "Herrera de Pisuerga",
			'34084' => "Herrera de Valdecañas",
			'34086' => "Hontoria de Cerrato",
			'34087' => "Hornillos de Cerrato",
			'34088' => "Husillos",
			'34089' => "Itero de la Vega",
			'34091' => "Lagartos",
			'34092' => "Lantadilla",
			'34094' => "Ledigos",
			'34903' => "Loma de Ucieza",
			'34096' => "Lomas",
			'34098' => "Magaz de Pisuerga",
			'34099' => "Manquillos",
			'34100' => "Mantinos",
			'34101' => "Marcilla de Campos",
			'34102' => "Mazariegos",
			'34103' => "Mazuecos de Valdeginate",
			'34104' => "Melgar de Yuso",
			'34106' => "Meneses de Campos",
			'34107' => "Micieces de Ojeda",
			'34108' => "Monzón de Campos",
			'34109' => "Moratinos",
			'34110' => "Mudá",
			'34112' => "Nogal de las Huertas",
			'34113' => "Olea de Boedo",
			'34114' => "Olmos de Ojeda",
			'34116' => "Osornillo",
			'34901' => "Osorno la Mayor",
			'34120' => "Palencia",
			'34121' => "Palenzuela",
			'34122' => "Páramo de Boedo",
			'34123' => "Paredes de Nava",
			'34124' => "Payo de Ojeda",
			'34125' => "Pedraza de Campos",
			'34126' => "Pedrosa de la Vega",
			'34127' => "Perales",
			'34904' => "Pernía, La",
			'34129' => "Pino del Río",
			'34130' => "Piña de Campos",
			'34131' => "Población de Arroyo",
			'34132' => "Población de Campos",
			'34133' => "Población de Cerrato",
			'34134' => "Polentinos",
			'34135' => "Pomar de Valdivia",
			'34136' => "Poza de la Vega",
			'34137' => "Pozo de Urama",
			'34139' => "Prádanos de Ojeda",
			'34140' => "Puebla de Valdavia, La",
			'34141' => "Quintana del Puente",
			'34143' => "Quintanilla de Onsoña",
			'34146' => "Reinoso de Cerrato",
			'34147' => "Renedo de la Vega",
			'34149' => "Requena de Campos",
			'34151' => "Respenda de la Peña",
			'34152' => "Revenga de Campos",
			'34154' => "Revilla de Collazos",
			'34155' => "Ribas de Campos",
			'34156' => "Riberos de la Cueza",
			'34157' => "Saldaña",
			'34158' => "Salinas de Pisuerga",
			'34159' => "San Cebrián de Campos",
			'34160' => "San Cebrián de Mudá",
			'34161' => "San Cristóbal de Boedo",
			'34163' => "San Mamés de Campos",
			'34165' => "San Román de la Cuba",
			'34167' => "Santa Cecilia del Alcor",
			'34168' => "Santa Cruz de Boedo",
			'34169' => "Santervás de la Vega",
			'34170' => "Santibáñez de Ecla",
			'34171' => "Santibáñez de la Peña",
			'34174' => "Santoyo",
			'34175' => "Serna, La",
			'34177' => "Soto de Cerrato",
			'34176' => "Sotobañado y Priorato",
			'34178' => "Tabanera de Cerrato",
			'34179' => "Tabanera de Valdavia",
			'34180' => "Támara de Campos",
			'34181' => "Tariego de Cerrato",
			'34182' => "Torquemada",
			'34184' => "Torremormojón",
			'34185' => "Triollo",
			'34186' => "Valbuena de Pisuerga",
			'34189' => "Valdeolmillos",
			'34190' => "Valderrábano",
			'34192' => "Valde-Ucieza",
			'34196' => "Valle de Cerrato",
			'34902' => "Valle del Retortillo",
			'34199' => "Velilla del Río Carrión",
			'34023' => "Venta de Baños",
			'34201' => "Vertavillo",
			'34093' => "Vid de Ojeda, La",
			'34202' => "Villabasta de Valdavia",
			'34204' => "Villacidaler",
			'34205' => "Villaconancio",
			'34206' => "Villada",
			'34208' => "Villaeles de Valdavia",
			'34210' => "Villahán",
			'34211' => "Villaherreros",
			'34213' => "Villalaco",
			'34214' => "Villalba de Guardo",
			'34215' => "Villalcázar de Sirga",
			'34216' => "Villalcón",
			'34217' => "Villalobón",
			'34218' => "Villaluenga de la Vega",
			'34220' => "Villamartín de Campos",
			'34221' => "Villamediana",
			'34222' => "Villameriel",
			'34223' => "Villamoronta",
			'34224' => "Villamuera de la Cueza",
			'34225' => "Villamuriel de Cerrato",
			'34227' => "Villanueva del Rebollar",
			'34228' => "Villanuño de Valdavia",
			'34229' => "Villaprovedo",
			'34230' => "Villarmentero de Campos",
			'34231' => "Villarrabé",
			'34232' => "Villarramiel",
			'34233' => "Villasarracino",
			'34234' => "Villasila de Valdavia",
			'34236' => "Villaturde",
			'34237' => "Villaumbrales",
			'34238' => "Villaviudas",
			'34240' => "Villerías de Campos",
			'34241' => "Villodre",
			'34242' => "Villodrigo",
			'34243' => "Villoldo",
			'34245' => "Villota del Páramo",
			'34246' => "Villovieco",
			'35001' => "Agaete",
			'35002' => "Agüimes",
			'35020' => "Aldea de San Nicolás, La",
			'35003' => "Antigua",
			'35004' => "Arrecife",
			'35005' => "Artenara",
			'35006' => "Arucas",
			'35007' => "Betancuria",
			'35008' => "Firgas",
			'35009' => "Gáldar",
			'35010' => "Haría",
			'35011' => "Ingenio",
			'35012' => "Mogán",
			'35013' => "Moya",
			'35014' => "Oliva, La",
			'35015' => "Pájara",
			'35016' => "Palmas de Gran Canaria, Las",
			'35017' => "Puerto del Rosario",
			'35018' => "San Bartolomé",
			'35019' => "San Bartolomé de Tirajana",
			'35021' => "Santa Brígida",
			'35022' => "Santa Lucía de Tirajana",
			'35023' => "Santa María de Guía de Gran Canaria",
			'35024' => "Teguise",
			'35025' => "Tejeda",
			'35026' => "Telde",
			'35027' => "Teror",
			'35028' => "Tías",
			'35029' => "Tinajo",
			'35030' => "Tuineje",
			'35032' => "Valleseco",
			'35031' => "Valsequillo de Gran Canaria",
			'35033' => "Vega de San Mateo",
			'35034' => "Yaiza",
			'36020' => "Agolada",
			'36001' => "Arbo",
			'36003' => "Baiona",
			'36002' => "Barro",
			'36004' => "Bueu",
			'36005' => "Caldas de Reis",
			'36006' => "Cambados",
			'36007' => "Campo Lameiro",
			'36008' => "Cangas",
			'36009' => "Cañiza, A",
			'36010' => "Catoira",
			'36902' => "Cerdedo-Cotobade",
			'36013' => "Covelo",
			'36014' => "Crecente",
			'36015' => "Cuntis",
			'36016' => "Dozón",
			'36017' => "Estrada, A",
			'36018' => "Forcarei",
			'36019' => "Fornelos de Montes",
			'36021' => "Gondomar",
			'36022' => "Grove, O",
			'36023' => "Guarda, A",
			'36901' => "Illa de Arousa, A",
			'36024' => "Lalín",
			'36025' => "Lama, A",
			'36026' => "Marín",
			'36027' => "Meaño",
			'36028' => "Meis",
			'36029' => "Moaña",
			'36030' => "Mondariz",
			'36031' => "Mondariz-Balneario",
			'36032' => "Moraña",
			'36033' => "Mos",
			'36034' => "Neves, As",
			'36035' => "Nigrán",
			'36036' => "Oia",
			'36037' => "Pazos de Borbén",
			'36041' => "Poio",
			'36043' => "Ponte Caldelas",
			'36042' => "Ponteareas",
			'36044' => "Pontecesures",
			'36038' => "Pontevedra",
			'36039' => "Porriño, O",
			'36040' => "Portas",
			'36045' => "Redondela",
			'36046' => "Ribadumia",
			'36047' => "Rodeiro",
			'36048' => "Rosal, O",
			'36049' => "Salceda de Caselas",
			'36050' => "Salvaterra de Miño",
			'36051' => "Sanxenxo",
			'36052' => "Silleda",
			'36053' => "Soutomaior",
			'36054' => "Tomiño",
			'36055' => "Tui",
			'36056' => "Valga",
			'36057' => "Vigo",
			'36059' => "Vila de Cruces",
			'36058' => "Vilaboa",
			'36060' => "Vilagarcía de Arousa",
			'36061' => "Vilanova de Arousa",
			'37001' => "Abusejo",
			'37002' => "Agallas",
			'37003' => "Ahigal de los Aceiteros",
			'37004' => "Ahigal de Villarino",
			'37005' => "Alameda de Gardón, La",
			'37006' => "Alamedilla, La",
			'37007' => "Alaraz",
			'37008' => "Alba de Tormes",
			'37009' => "Alba de Yeltes",
			'37010' => "Alberca, La",
			'37011' => "Alberguería de Argañán, La",
			'37012' => "Alconada",
			'37015' => "Aldea del Obispo",
			'37013' => "Aldeacipreste",
			'37014' => "Aldeadávila de la Ribera",
			'37016' => "Aldealengua",
			'37017' => "Aldeanueva de Figueroa",
			'37018' => "Aldeanueva de la Sierra",
			'37019' => "Aldearrodrigo",
			'37020' => "Aldearrubia",
			'37021' => "Aldeaseca de Alba",
			'37022' => "Aldeaseca de la Frontera",
			'37023' => "Aldeatejada",
			'37024' => "Aldeavieja de Tormes",
			'37025' => "Aldehuela de la Bóveda",
			'37026' => "Aldehuela de Yeltes",
			'37027' => "Almenara de Tormes",
			'37028' => "Almendra",
			'37029' => "Anaya de Alba",
			'37030' => "Añover de Tormes",
			'37031' => "Arabayona de Mógica",
			'37032' => "Arapiles",
			'37033' => "Arcediano",
			'37034' => "Arco, El",
			'37035' => "Armenteros",
			'37037' => "Atalaya, La",
			'37038' => "Babilafuente",
			'37039' => "Bañobárez",
			'37040' => "Barbadillo",
			'37041' => "Barbalos",
			'37042' => "Barceo",
			'37044' => "Barruecopardo",
			'37045' => "Bastida, La",
			'37046' => "Béjar",
			'37047' => "Beleña",
			'37049' => "Bermellar",
			'37050' => "Berrocal de Huebra",
			'37051' => "Berrocal de Salvatierra",
			'37052' => "Boada",
			'37054' => "Bodón, El",
			'37055' => "Bogajo",
			'37056' => "Bouza, La",
			'37057' => "Bóveda del Río Almar",
			'37058' => "Brincones",
			'37059' => "Buenamadre",
			'37060' => "Buenavista",
			'37061' => "Cabaco, El",
			'37063' => "Cabeza de Béjar, La",
			'37065' => "Cabeza del Caballo",
			'37062' => "Cabezabellosa de la Calzada",
			'37067' => "Cabrerizos",
			'37068' => "Cabrillas",
			'37069' => "Calvarrasa de Abajo",
			'37070' => "Calvarrasa de Arriba",
			'37071' => "Calzada de Béjar, La",
			'37072' => "Calzada de Don Diego",
			'37073' => "Calzada de Valdunciel",
			'37074' => "Campillo de Azaba",
			'37077' => "Campo de Peñaranda, El",
			'37078' => "Candelario",
			'37079' => "Canillas de Abajo",
			'37080' => "Cantagallo",
			'37081' => "Cantalapiedra",
			'37082' => "Cantalpino",
			'37083' => "Cantaracillo",
			'37085' => "Carbajosa de la Sagrada",
			'37086' => "Carpio de Azaba",
			'37087' => "Carrascal de Barregas",
			'37088' => "Carrascal del Obispo",
			'37089' => "Casafranca",
			'37090' => "Casas del Conde, Las",
			'37091' => "Casillas de Flores",
			'37092' => "Castellanos de Moriscos",
			'37185' => "Castellanos de Villiquera",
			'37096' => "Castillejo de Martín Viejo",
			'37097' => "Castraz",
			'37098' => "Cepeda",
			'37099' => "Cereceda de la Sierra",
			'37100' => "Cerezal de Peñahorcada",
			'37101' => "Cerralbo",
			'37102' => "Cerro, El",
			'37103' => "Cespedosa de Tormes",
			'37114' => "Chagarcía Medianero",
			'37104' => "Cilleros de la Bastida",
			'37106' => "Cipérez",
			'37107' => "Ciudad Rodrigo",
			'37108' => "Coca de Alba",
			'37109' => "Colmenar de Montemayor",
			'37110' => "Cordovilla",
			'37112' => "Cristóbal",
			'37113' => "Cubo de Don Sancho, El",
			'37115' => "Dios le Guarde",
			'37116' => "Doñinos de Ledesma",
			'37117' => "Doñinos de Salamanca",
			'37118' => "Éjeme",
			'37120' => "Encina de San Silvestre",
			'37119' => "Encina, La",
			'37121' => "Encinas de Abajo",
			'37122' => "Encinas de Arriba",
			'37123' => "Encinasola de los Comendadores",
			'37124' => "Endrinal",
			'37125' => "Escurial de la Sierra",
			'37126' => "Espadaña",
			'37127' => "Espeja",
			'37128' => "Espino de la Orbada",
			'37129' => "Florida de Liébana",
			'37130' => "Forfoleda",
			'37131' => "Frades de la Sierra",
			'37132' => "Fregeneda, La",
			'37133' => "Fresnedoso",
			'37134' => "Fresno Alhándiga",
			'37135' => "Fuente de San Esteban, La",
			'37136' => "Fuenteguinaldo",
			'37137' => "Fuenteliante",
			'37138' => "Fuenterroble de Salvatierra",
			'37139' => "Fuentes de Béjar",
			'37140' => "Fuentes de Oñoro",
			'37141' => "Gajates",
			'37142' => "Galindo y Perahuy",
			'37143' => "Galinduste",
			'37144' => "Galisancho",
			'37145' => "Gallegos de Argañán",
			'37146' => "Gallegos de Solmirón",
			'37147' => "Garcibuey",
			'37148' => "Garcihernández",
			'37149' => "Garcirrey",
			'37150' => "Gejuelo del Barro",
			'37151' => "Golpejas",
			'37152' => "Gomecello",
			'37154' => "Guadramiro",
			'37155' => "Guijo de Ávila",
			'37156' => "Guijuelo",
			'37157' => "Herguijuela de Ciudad Rodrigo",
			'37158' => "Herguijuela de la Sierra",
			'37159' => "Herguijuela del Campo",
			'37160' => "Hinojosa de Duero",
			'37161' => "Horcajo de Montemayor",
			'37162' => "Horcajo Medianero",
			'37163' => "Hoya, La",
			'37164' => "Huerta",
			'37165' => "Iruelos",
			'37166' => "Ituero de Azaba",
			'37167' => "Juzbado",
			'37168' => "Lagunilla",
			'37169' => "Larrodrigo",
			'37170' => "Ledesma",
			'37171' => "Ledrada",
			'37172' => "Linares de Riofrío",
			'37173' => "Lumbrales",
			'37175' => "Machacón",
			'37174' => "Macotera",
			'37176' => "Madroñal",
			'37177' => "Maíllo, El",
			'37178' => "Malpartida",
			'37179' => "Mancera de Abajo",
			'37180' => "Manzano, El",
			'37181' => "Martiago",
			'37183' => "Martín de Yeltes",
			'37182' => "Martinamor",
			'37184' => "Masueco",
			'37186' => "Mata de Ledesma, La",
			'37187' => "Matilla de los Caños del Río",
			'37188' => "Maya, La",
			'37189' => "Membribe de la Sierra",
			'37190' => "Mieza",
			'37191' => "Milano, El",
			'37192' => "Miranda de Azán",
			'37193' => "Miranda del Castañar",
			'37194' => "Mogarraz",
			'37195' => "Molinillo",
			'37196' => "Monforte de la Sierra",
			'37197' => "Monleón",
			'37198' => "Monleras",
			'37199' => "Monsagro",
			'37200' => "Montejo",
			'37201' => "Montemayor del Río",
			'37202' => "Monterrubio de Armuña",
			'37203' => "Monterrubio de la Sierra",
			'37204' => "Morasverdes",
			'37205' => "Morille",
			'37206' => "Moríñigo",
			'37207' => "Moriscos",
			'37208' => "Moronta",
			'37209' => "Mozárbez",
			'37211' => "Narros de Matalayegua",
			'37213' => "Nava de Béjar",
			'37214' => "Nava de Francia",
			'37215' => "Nava de Sotrobal",
			'37212' => "Navacarros",
			'37216' => "Navales",
			'37217' => "Navalmoral de Béjar",
			'37218' => "Navamorales",
			'37219' => "Navarredonda de la Rinconada",
			'37221' => "Navasfrías",
			'37222' => "Negrilla de Palencia",
			'37223' => "Olmedo de Camaces",
			'37224' => "Orbada, La",
			'37225' => "Pajares de la Laguna",
			'37226' => "Palacios del Arzobispo",
			'37228' => "Palaciosrubios",
			'37229' => "Palencia de Negrilla",
			'37230' => "Parada de Arriba",
			'37231' => "Parada de Rubiales",
			'37232' => "Paradinas de San Juan",
			'37233' => "Pastores",
			'37234' => "Payo, El",
			'37235' => "Pedraza de Alba",
			'37236' => "Pedrosillo de Alba",
			'37237' => "Pedrosillo de los Aires",
			'37238' => "Pedrosillo el Ralo",
			'37239' => "Pedroso de la Armuña, El",
			'37240' => "Pelabravo",
			'37241' => "Pelarrodríguez",
			'37242' => "Pelayos",
			'37243' => "Peña, La",
			'37244' => "Peñacaballera",
			'37245' => "Peñaparda",
			'37246' => "Peñaranda de Bracamonte",
			'37247' => "Peñarandilla",
			'37248' => "Peralejos de Abajo",
			'37249' => "Peralejos de Arriba",
			'37250' => "Pereña de la Ribera",
			'37251' => "Peromingo",
			'37252' => "Pinedas",
			'37253' => "Pino de Tormes, El",
			'37254' => "Pitiegua",
			'37255' => "Pizarral",
			'37256' => "Poveda de las Cintas",
			'37257' => "Pozos de Hinojo",
			'37258' => "Puebla de Azaba",
			'37259' => "Puebla de San Medel",
			'37260' => "Puebla de Yeltes",
			'37261' => "Puente del Congosto",
			'37262' => "Puertas",
			'37263' => "Puerto de Béjar",
			'37264' => "Puerto Seguro",
			'37265' => "Rágama",
			'37266' => "Redonda, La",
			'37267' => "Retortillo",
			'37268' => "Rinconada de la Sierra, La",
			'37269' => "Robleda",
			'37270' => "Robliza de Cojos",
			'37271' => "Rollán",
			'37272' => "Saelices el Chico",
			'37273' => "Sagrada, La",
			'37303' => "Sahugo, El",
			'37274' => "Salamanca",
			'37275' => "Saldeana",
			'37276' => "Salmoral",
			'37277' => "Salvatierra de Tormes",
			'37278' => "San Cristóbal de la Cuesta",
			'37284' => "San Esteban de la Sierra",
			'37285' => "San Felices de los Gallegos",
			'37286' => "San Martín del Castañar",
			'37287' => "San Miguel de Valero",
			'37036' => "San Miguel del Robledo",
			'37288' => "San Morales",
			'37289' => "San Muñoz",
			'37291' => "San Pedro de Rozados",
			'37290' => "San Pedro del Valle",
			'37292' => "San Pelayo de Guareña",
			'37280' => "Sanchón de la Ribera",
			'37281' => "Sanchón de la Sagrada",
			'37282' => "Sanchotello",
			'37279' => "Sancti-Spíritus",
			'37283' => "Sando",
			'37293' => "Santa María de Sando",
			'37294' => "Santa Marta de Tormes",
			'37296' => "Santiago de la Puebla",
			'37297' => "Santibáñez de Béjar",
			'37298' => "Santibáñez de la Sierra",
			'37299' => "Santiz",
			'37300' => "Santos, Los",
			'37301' => "Sardón de los Frailes",
			'37302' => "Saucelle",
			'37304' => "Sepulcro-Hilario",
			'37305' => "Sequeros",
			'37306' => "Serradilla del Arroyo",
			'37307' => "Serradilla del Llano",
			'37309' => "Sierpe, La",
			'37310' => "Sieteiglesias de Tormes",
			'37311' => "Sobradillo",
			'37312' => "Sorihuela",
			'37313' => "Sotoserrano",
			'37314' => "Tabera de Abajo",
			'37315' => "Tala, La",
			'37316' => "Tamames",
			'37317' => "Tarazona de Guareña",
			'37318' => "Tardáguila",
			'37319' => "Tejado, El",
			'37320' => "Tejeda y Segoyuela",
			'37321' => "Tenebrón",
			'37322' => "Terradillos",
			'37323' => "Topas",
			'37324' => "Tordillos",
			'37325' => "Tornadizo, El",
			'37327' => "Torresmenudas",
			'37328' => "Trabanca",
			'37329' => "Tremedal de Tormes",
			'37330' => "Valdecarros",
			'37331' => "Valdefuentes de Sangusín",
			'37332' => "Valdehijaderos",
			'37333' => "Valdelacasa",
			'37334' => "Valdelageve",
			'37335' => "Valdelosa",
			'37336' => "Valdemierque",
			'37337' => "Valderrodrigo",
			'37338' => "Valdunciel",
			'37339' => "Valero",
			'37343' => "Vallejera de Riofrío",
			'37340' => "Valsalabroso",
			'37341' => "Valverde de Valdelacasa",
			'37342' => "Valverdón",
			'37344' => "Vecinos",
			'37345' => "Vega de Tirados",
			'37346' => "Veguillas, Las",
			'37347' => "Vellés, La",
			'37348' => "Ventosa del Río Almar",
			'37349' => "Vídola, La",
			'37351' => "Villaflores",
			'37352' => "Villagonzalo de Tormes",
			'37353' => "Villalba de los Llanos",
			'37354' => "Villamayor",
			'37355' => "Villanueva del Conde",
			'37356' => "Villar de Argañán",
			'37357' => "Villar de Ciervo",
			'37358' => "Villar de Gallimazo",
			'37359' => "Villar de la Yegua",
			'37360' => "Villar de Peralonso",
			'37361' => "Villar de Samaniego",
			'37362' => "Villares de la Reina",
			'37363' => "Villares de Yeltes",
			'37364' => "Villarino de los Aires",
			'37365' => "Villarmayor",
			'37366' => "Villarmuerto",
			'37367' => "Villasbuenas",
			'37368' => "Villasdardo",
			'37369' => "Villaseco de los Gamitos",
			'37370' => "Villaseco de los Reyes",
			'37371' => "Villasrubias",
			'37372' => "Villaverde de Guareña",
			'37373' => "Villavieja de Yeltes",
			'37374' => "Villoria",
			'37375' => "Villoruela",
			'37350' => "Vilvestre",
			'37376' => "Vitigudino",
			'37377' => "Yecla de Yeltes",
			'37378' => "Zamarra",
			'37379' => "Zamayón",
			'37380' => "Zarapicos",
			'37381' => "Zarza de Pumareda, La",
			'37382' => "Zorita de la Frontera",
			'38001' => "Adeje",
			'38002' => "Agulo",
			'38003' => "Alajeró",
			'38004' => "Arafo",
			'38005' => "Arico",
			'38006' => "Arona",
			'38007' => "Barlovento",
			'38008' => "Breña Alta",
			'38009' => "Breña Baja",
			'38010' => "Buenavista del Norte",
			'38011' => "Candelaria",
			'38012' => "Fasnia",
			'38013' => "Frontera",
			'38014' => "Fuencaliente de la Palma",
			'38015' => "Garachico",
			'38016' => "Garafía",
			'38017' => "Granadilla de Abona",
			'38018' => "Guancha, La",
			'38019' => "Guía de Isora",
			'38020' => "Güímar",
			'38021' => "Hermigua",
			'38022' => "Icod de los Vinos",
			'38024' => "Llanos de Aridane, Los",
			'38025' => "Matanza de Acentejo, La",
			'38026' => "Orotava, La",
			'38027' => "Paso, El",
			'38901' => "Pinar de El Hierro, El",
			'38028' => "Puerto de la Cruz",
			'38029' => "Puntagorda",
			'38030' => "Puntallana",
			'38031' => "Realejos, Los",
			'38032' => "Rosario, El",
			'38033' => "San Andrés y Sauces",
			'38023' => "San Cristóbal de La Laguna",
			'38034' => "San Juan de la Rambla",
			'38035' => "San Miguel de Abona",
			'38036' => "San Sebastián de la Gomera",
			'38037' => "Santa Cruz de la Palma",
			'38038' => "Santa Cruz de Tenerife",
			'38039' => "Santa Úrsula",
			'38040' => "Santiago del Teide",
			'38041' => "Sauzal, El",
			'38042' => "Silos, Los",
			'38043' => "Tacoronte",
			'38044' => "Tanque, El",
			'38045' => "Tazacorte",
			'38046' => "Tegueste",
			'38047' => "Tijarafe",
			'38049' => "Valle Gran Rey",
			'38050' => "Vallehermoso",
			'38048' => "Valverde",
			'38051' => "Victoria de Acentejo, La",
			'38052' => "Vilaflor de Chasna",
			'38053' => "Villa de Mazo",
			'39001' => "Alfoz de Lloredo",
			'39002' => "Ampuero",
			'39003' => "Anievas",
			'39004' => "Arenas de Iguña",
			'39005' => "Argoños",
			'39006' => "Arnuero",
			'39007' => "Arredondo",
			'39008' => "Astillero, El",
			'39009' => "Bárcena de Cicero",
			'39010' => "Bárcena de Pie de Concha",
			'39011' => "Bareyo",
			'39012' => "Cabezón de la Sal",
			'39013' => "Cabezón de Liébana",
			'39014' => "Cabuérniga",
			'39015' => "Camaleño",
			'39016' => "Camargo",
			'39027' => "Campoo de Enmedio",
			'39017' => "Campoo de Yuso",
			'39018' => "Cartes",
			'39019' => "Castañeda",
			'39020' => "Castro-Urdiales",
			'39021' => "Cieza",
			'39022' => "Cillorigo de Liébana",
			'39023' => "Colindres",
			'39024' => "Comillas",
			'39025' => "Corrales de Buelna, Los",
			'39026' => "Corvera de Toranzo",
			'39028' => "Entrambasaguas",
			'39029' => "Escalante",
			'39030' => "Guriezo",
			'39031' => "Hazas de Cesto",
			'39032' => "Hermandad de Campoo de Suso",
			'39033' => "Herrerías",
			'39034' => "Lamasón",
			'39035' => "Laredo",
			'39036' => "Liendo",
			'39037' => "Liérganes",
			'39038' => "Limpias",
			'39039' => "Luena",
			'39040' => "Marina de Cudeyo",
			'39041' => "Mazcuerras",
			'39042' => "Medio Cudeyo",
			'39043' => "Meruelo",
			'39044' => "Miengo",
			'39045' => "Miera",
			'39046' => "Molledo",
			'39047' => "Noja",
			'39048' => "Penagos",
			'39049' => "Peñarrubia",
			'39050' => "Pesaguero",
			'39051' => "Pesquera",
			'39052' => "Piélagos",
			'39053' => "Polaciones",
			'39054' => "Polanco",
			'39055' => "Potes",
			'39056' => "Puente Viesgo",
			'39057' => "Ramales de la Victoria",
			'39058' => "Rasines",
			'39059' => "Reinosa",
			'39060' => "Reocín",
			'39061' => "Ribamontán al Mar",
			'39062' => "Ribamontán al Monte",
			'39063' => "Rionansa",
			'39064' => "Riotuerto",
			'39065' => "Rozas de Valdearroyo, Las",
			'39066' => "Ruente",
			'39067' => "Ruesga",
			'39068' => "Ruiloba",
			'39069' => "San Felices de Buelna",
			'39070' => "San Miguel de Aguayo",
			'39071' => "San Pedro del Romeral",
			'39072' => "San Roque de Riomiera",
			'39080' => "San Vicente de la Barquera",
			'39073' => "Santa Cruz de Bezana",
			'39074' => "Santa María de Cayón",
			'39075' => "Santander",
			'39076' => "Santillana del Mar",
			'39077' => "Santiurde de Reinosa",
			'39078' => "Santiurde de Toranzo",
			'39079' => "Santoña",
			'39081' => "Saro",
			'39082' => "Selaya",
			'39083' => "Soba",
			'39084' => "Solórzano",
			'39085' => "Suances",
			'39086' => "Tojos, Los",
			'39087' => "Torrelavega",
			'39088' => "Tresviso",
			'39089' => "Tudanca",
			'39090' => "Udías",
			'39095' => "Val de San Vicente",
			'39091' => "Valdáliga",
			'39092' => "Valdeolea",
			'39093' => "Valdeprado del Río",
			'39094' => "Valderredible",
			'39101' => "Valle de Villaverde",
			'39096' => "Vega de Liébana",
			'39097' => "Vega de Pas",
			'39098' => "Villacarriedo",
			'39099' => "Villaescusa",
			'39100' => "Villafufre",
			'39102' => "Voto",
			'40001' => "Abades",
			'40002' => "Adrada de Pirón",
			'40003' => "Adrados",
			'40004' => "Aguilafuente",
			'40005' => "Alconada de Maderuelo",
			'40012' => "Aldea Real",
			'40006' => "Aldealcorvo",
			'40007' => "Aldealengua de Pedraza",
			'40008' => "Aldealengua de Santa María",
			'40009' => "Aldeanueva de la Serrezuela",
			'40010' => "Aldeanueva del Codonal",
			'40013' => "Aldeasoña",
			'40014' => "Aldehorno",
			'40015' => "Aldehuela del Codonal",
			'40016' => "Aldeonte",
			'40017' => "Anaya",
			'40018' => "Añe",
			'40019' => "Arahuetes",
			'40020' => "Arcones",
			'40021' => "Arevalillo de Cega",
			'40022' => "Armuña",
			'40024' => "Ayllón",
			'40025' => "Barbolla",
			'40026' => "Basardilla",
			'40028' => "Bercial",
			'40029' => "Bercimuel",
			'40030' => "Bernardos",
			'40031' => "Bernuy de Porreros",
			'40032' => "Boceguillas",
			'40033' => "Brieva",
			'40034' => "Caballar",
			'40035' => "Cabañas de Polendos",
			'40036' => "Cabezuela",
			'40037' => "Calabazas de Fuentidueña",
			'40039' => "Campo de San Pedro",
			'40040' => "Cantalejo",
			'40041' => "Cantimpalos",
			'40161' => "Carabias",
			'40043' => "Carbonero el Mayor",
			'40044' => "Carrascal del Río",
			'40045' => "Casla",
			'40046' => "Castillejo de Mesleón",
			'40047' => "Castro de Fuentidueña",
			'40048' => "Castrojimeno",
			'40049' => "Castroserna de Abajo",
			'40051' => "Castroserracín",
			'40052' => "Cedillo de la Torre",
			'40053' => "Cerezo de Abajo",
			'40054' => "Cerezo de Arriba",
			'40065' => "Chañe",
			'40055' => "Cilleruelo de San Mamés",
			'40056' => "Cobos de Fuentidueña",
			'40057' => "Coca",
			'40058' => "Codorniz",
			'40059' => "Collado Hermoso",
			'40060' => "Condado de Castilnovo",
			'40061' => "Corral de Ayllón",
			'40902' => "Cozuelos de Fuentidueña",
			'40062' => "Cubillo",
			'40063' => "Cuéllar",
			'40905' => "Cuevas de Provanco",
			'40068' => "Domingo García",
			'40069' => "Donhierro",
			'40070' => "Duruelo",
			'40071' => "Encinas",
			'40072' => "Encinillas",
			'40073' => "Escalona del Prado",
			'40074' => "Escarabajosa de Cabezas",
			'40075' => "Escobar de Polendos",
			'40076' => "Espinar, El",
			'40077' => "Espirdo",
			'40078' => "Fresneda de Cuéllar",
			'40079' => "Fresno de Cantespino",
			'40080' => "Fresno de la Fuente",
			'40081' => "Frumales",
			'40082' => "Fuente de Santa Cruz",
			'40083' => "Fuente el Olmo de Fuentidueña",
			'40084' => "Fuente el Olmo de Íscar",
			'40086' => "Fuentepelayo",
			'40087' => "Fuentepiñel",
			'40088' => "Fuenterrebollo",
			'40089' => "Fuentesaúco de Fuentidueña",
			'40091' => "Fuentesoto",
			'40092' => "Fuentidueña",
			'40093' => "Gallegos",
			'40094' => "Garcillán",
			'40095' => "Gomezserracín",
			'40097' => "Grajera",
			'40099' => "Honrubia de la Cuesta",
			'40100' => "Hontalbilla",
			'40101' => "Hontanares de Eresma",
			'40103' => "Huertos, Los",
			'40104' => "Ituero y Lama",
			'40105' => "Juarros de Riomoros",
			'40106' => "Juarros de Voltoya",
			'40107' => "Labajos",
			'40108' => "Laguna de Contreras",
			'40109' => "Languilla",
			'40110' => "Lastras de Cuéllar",
			'40111' => "Lastras del Pozo",
			'40112' => "Lastrilla, La",
			'40113' => "Losa, La",
			'40115' => "Maderuelo",
			'40903' => "Marazoleja",
			'40118' => "Marazuela",
			'40119' => "Martín Miguel",
			'40120' => "Martín Muñoz de la Dehesa",
			'40121' => "Martín Muñoz de las Posadas",
			'40122' => "Marugán",
			'40124' => "Mata de Cuéllar",
			'40123' => "Matabuena",
			'40125' => "Matilla, La",
			'40126' => "Melque de Cercos",
			'40127' => "Membibre de la Hoz",
			'40128' => "Migueláñez",
			'40129' => "Montejo de Arévalo",
			'40130' => "Montejo de la Vega de la Serrezuela",
			'40131' => "Monterrubio",
			'40132' => "Moral de Hornuez",
			'40134' => "Mozoncillo",
			'40135' => "Muñopedro",
			'40136' => "Muñoveros",
			'40138' => "Nava de la Asunción",
			'40139' => "Navafría",
			'40140' => "Navalilla",
			'40141' => "Navalmanzano",
			'40142' => "Navares de Ayuso",
			'40143' => "Navares de Enmedio",
			'40144' => "Navares de las Cuevas",
			'40145' => "Navas de Oro",
			'40904' => "Navas de Riofrío",
			'40146' => "Navas de San Antonio",
			'40148' => "Nieva",
			'40149' => "Olombrada",
			'40150' => "Orejana",
			'40151' => "Ortigosa de Pestaño",
			'40901' => "Ortigosa del Monte",
			'40152' => "Otero de Herreros",
			'40154' => "Pajarejos",
			'40155' => "Palazuelos de Eresma",
			'40156' => "Pedraza",
			'40157' => "Pelayos del Arroyo",
			'40158' => "Perosillo",
			'40159' => "Pinarejos",
			'40160' => "Pinarnegrillo",
			'40162' => "Prádena",
			'40163' => "Puebla de Pedraza",
			'40164' => "Rapariegos",
			'40181' => "Real Sitio de San Ildefonso",
			'40165' => "Rebollo",
			'40166' => "Remondo",
			'40168' => "Riaguas de San Bartolomé",
			'40170' => "Riaza",
			'40171' => "Ribota",
			'40172' => "Riofrío de Riaza",
			'40173' => "Roda de Eresma",
			'40174' => "Sacramenia",
			'40176' => "Samboal",
			'40177' => "San Cristóbal de Cuéllar",
			'40178' => "San Cristóbal de la Vega",
			'40906' => "San Cristóbal de Segovia",
			'40182' => "San Martín y Mudrián",
			'40183' => "San Miguel de Bernuy",
			'40184' => "San Pedro de Gaíllos",
			'40179' => "Sanchonuño",
			'40180' => "Sangarcía",
			'40185' => "Santa María la Real de Nieva",
			'40186' => "Santa Marta del Cerro",
			'40188' => "Santiuste de Pedraza",
			'40189' => "Santiuste de San Juan Bautista",
			'40190' => "Santo Domingo de Pirón",
			'40191' => "Santo Tomé del Puerto",
			'40192' => "Sauquillo de Cabezas",
			'40193' => "Sebúlcor",
			'40194' => "Segovia",
			'40195' => "Sepúlveda",
			'40196' => "Sequera de Fresno",
			'40198' => "Sotillo",
			'40199' => "Sotosalbos",
			'40200' => "Tabanera la Luenga",
			'40201' => "Tolocirio",
			'40206' => "Torre Val de San Pedro",
			'40202' => "Torreadrada",
			'40203' => "Torrecaballeros",
			'40204' => "Torrecilla del Pinar",
			'40205' => "Torreiglesias",
			'40207' => "Trescasas",
			'40208' => "Turégano",
			'40210' => "Urueñas",
			'40211' => "Valdeprados",
			'40212' => "Valdevacas de Montejo",
			'40213' => "Valdevacas y Guijar",
			'40218' => "Valle de Tabladillo",
			'40219' => "Vallelado",
			'40220' => "Valleruela de Pedraza",
			'40221' => "Valleruela de Sepúlveda",
			'40214' => "Valseca",
			'40215' => "Valtiendas",
			'40216' => "Valverde del Majano",
			'40222' => "Veganzones",
			'40223' => "Vegas de Matute",
			'40224' => "Ventosilla y Tejadilla",
			'40225' => "Villacastín",
			'40228' => "Villaverde de Íscar",
			'40229' => "Villaverde de Montejo",
			'40230' => "Villeguillo",
			'40231' => "Yanguas de Eresma",
			'40233' => "Zarzuela del Monte",
			'40234' => "Zarzuela del Pinar",
			'41001' => "Aguadulce",
			'41002' => "Alanís",
			'41003' => "Albaida del Aljarafe",
			'41004' => "Alcalá de Guadaíra",
			'41005' => "Alcalá del Río",
			'41006' => "Alcolea del Río",
			'41007' => "Algaba, La",
			'41008' => "Algámitas",
			'41009' => "Almadén de la Plata",
			'41010' => "Almensilla",
			'41011' => "Arahal",
			'41012' => "Aznalcázar",
			'41013' => "Aznalcóllar",
			'41014' => "Badolatosa",
			'41015' => "Benacazón",
			'41016' => "Bollullos de la Mitación",
			'41017' => "Bormujos",
			'41018' => "Brenes",
			'41019' => "Burguillos",
			'41020' => "Cabezas de San Juan, Las",
			'41021' => "Camas",
			'41022' => "Campana, La",
			'41023' => "Cantillana",
			'41901' => "Cañada Rosal",
			'41024' => "Carmona",
			'41025' => "Carrión de los Céspedes",
			'41026' => "Casariche",
			'41027' => "Castilblanco de los Arroyos",
			'41028' => "Castilleja de Guzmán",
			'41029' => "Castilleja de la Cuesta",
			'41030' => "Castilleja del Campo",
			'41031' => "Castillo de las Guardas, El",
			'41032' => "Cazalla de la Sierra",
			'41033' => "Constantina",
			'41034' => "Coria del Río",
			'41035' => "Coripe",
			'41036' => "Coronil, El",
			'41037' => "Corrales, Los",
			'41903' => "Cuervo de Sevilla, El",
			'41038' => "Dos Hermanas",
			'41039' => "Écija",
			'41040' => "Espartinas",
			'41041' => "Estepa",
			'41042' => "Fuentes de Andalucía",
			'41043' => "Garrobo, El",
			'41044' => "Gelves",
			'41045' => "Gerena",
			'41046' => "Gilena",
			'41047' => "Gines",
			'41048' => "Guadalcanal",
			'41049' => "Guillena",
			'41050' => "Herrera",
			'41051' => "Huévar del Aljarafe",
			'41902' => "Isla Mayor",
			'41052' => "Lantejuela",
			'41053' => "Lebrija",
			'41054' => "Lora de Estepa",
			'41055' => "Lora del Río",
			'41056' => "Luisiana, La",
			'41057' => "Madroño, El",
			'41058' => "Mairena del Alcor",
			'41059' => "Mairena del Aljarafe",
			'41060' => "Marchena",
			'41061' => "Marinaleda",
			'41062' => "Martín de la Jara",
			'41063' => "Molares, Los",
			'41064' => "Montellano",
			'41065' => "Morón de la Frontera",
			'41066' => "Navas de la Concepción, Las",
			'41067' => "Olivares",
			'41068' => "Osuna",
			'41069' => "Palacios y Villafranca, Los",
			'41904' => "Palmar de Troya, El",
			'41070' => "Palomares del Río",
			'41071' => "Paradas",
			'41072' => "Pedrera",
			'41073' => "Pedroso, El",
			'41074' => "Peñaflor",
			'41075' => "Pilas",
			'41076' => "Pruna",
			'41077' => "Puebla de Cazalla, La",
			'41078' => "Puebla de los Infantes, La",
			'41079' => "Puebla del Río, La",
			'41080' => "Real de la Jara, El",
			'41081' => "Rinconada, La",
			'41082' => "Roda de Andalucía, La",
			'41083' => "Ronquillo, El",
			'41084' => "Rubio, El",
			'41085' => "Salteras",
			'41086' => "San Juan de Aznalfarache",
			'41088' => "San Nicolás del Puerto",
			'41087' => "Sanlúcar la Mayor",
			'41089' => "Santiponce",
			'41090' => "Saucejo, El",
			'41091' => "Sevilla",
			'41092' => "Tocina",
			'41093' => "Tomares",
			'41094' => "Umbrete",
			'41095' => "Utrera",
			'41096' => "Valencina de la Concepción",
			'41097' => "Villamanrique de la Condesa",
			'41100' => "Villanueva de San Juan",
			'41098' => "Villanueva del Ariscal",
			'41099' => "Villanueva del Río y Minas",
			'41101' => "Villaverde del Río",
			'41102' => "Viso del Alcor, El",
			'42001' => "Abejar",
			'42003' => "Adradas",
			'42004' => "Ágreda",
			'42006' => "Alconaba",
			'42007' => "Alcubilla de Avellaneda",
			'42008' => "Alcubilla de las Peñas",
			'42009' => "Aldealafuente",
			'42010' => "Aldealices",
			'42011' => "Aldealpozo",
			'42012' => "Aldealseñor",
			'42013' => "Aldehuela de Periáñez",
			'42014' => "Aldehuelas, Las",
			'42015' => "Alentisque",
			'42016' => "Aliud",
			'42017' => "Almajano",
			'42018' => "Almaluez",
			'42019' => "Almarza",
			'42020' => "Almazán",
			'42021' => "Almazul",
			'42022' => "Almenar de Soria",
			'42023' => "Alpanseque",
			'42024' => "Arancón",
			'42025' => "Arcos de Jalón",
			'42026' => "Arenillas",
			'42027' => "Arévalo de la Sierra",
			'42028' => "Ausejo de la Sierra",
			'42029' => "Baraona",
			'42030' => "Barca",
			'42031' => "Barcones",
			'42032' => "Bayubas de Abajo",
			'42033' => "Bayubas de Arriba",
			'42034' => "Beratón",
			'42035' => "Berlanga de Duero",
			'42036' => "Blacos",
			'42037' => "Bliecos",
			'42038' => "Borjabad",
			'42039' => "Borobia",
			'42041' => "Buberos",
			'42042' => "Buitrago",
			'42043' => "Burgo de Osma-Ciudad de Osma",
			'42044' => "Cabrejas del Campo",
			'42045' => "Cabrejas del Pinar",
			'42046' => "Calatañazor",
			'42048' => "Caltojar",
			'42049' => "Candilichera",
			'42050' => "Cañamaque",
			'42051' => "Carabantes",
			'42052' => "Caracena",
			'42053' => "Carrascosa de Abajo",
			'42054' => "Carrascosa de la Sierra",
			'42055' => "Casarejos",
			'42056' => "Castilfrío de la Sierra",
			'42058' => "Castillejo de Robledo",
			'42057' => "Castilruiz",
			'42059' => "Centenera de Andaluz",
			'42060' => "Cerbón",
			'42061' => "Cidones",
			'42062' => "Cigudosa",
			'42063' => "Cihuela",
			'42064' => "Ciria",
			'42065' => "Cirujales del Río",
			'42068' => "Coscurita",
			'42069' => "Covaleda",
			'42070' => "Cubilla",
			'42071' => "Cubo de la Solana",
			'42073' => "Cueva de Ágreda",
			'42075' => "Dévanos",
			'42076' => "Deza",
			'42078' => "Duruelo de la Sierra",
			'42079' => "Escobosa de Almazán",
			'42080' => "Espeja de San Marcelino",
			'42081' => "Espejón",
			'42082' => "Estepa de San Juan",
			'42083' => "Frechilla de Almazán",
			'42084' => "Fresno de Caracena",
			'42085' => "Fuentearmegil",
			'42086' => "Fuentecambrón",
			'42087' => "Fuentecantos",
			'42088' => "Fuentelmonge",
			'42089' => "Fuentelsaz de Soria",
			'42090' => "Fuentepinilla",
			'42092' => "Fuentes de Magaña",
			'42093' => "Fuentestrún",
			'42094' => "Garray",
			'42095' => "Golmayo",
			'42096' => "Gómara",
			'42097' => "Gormaz",
			'42098' => "Herrera de Soria",
			'42100' => "Hinojosa del Campo",
			'42103' => "Langa de Duero",
			'42105' => "Liceras",
			'42106' => "Losilla, La",
			'42107' => "Magaña",
			'42108' => "Maján",
			'42110' => "Matalebreras",
			'42111' => "Matamala de Almazán",
			'42113' => "Medinaceli",
			'42115' => "Miño de Medinaceli",
			'42116' => "Miño de San Esteban",
			'42117' => "Molinos de Duero",
			'42118' => "Momblona",
			'42119' => "Monteagudo de las Vicarías",
			'42120' => "Montejo de Tiermes",
			'42121' => "Montenegro de Cameros",
			'42123' => "Morón de Almazán",
			'42124' => "Muriel de la Fuente",
			'42125' => "Muriel Viejo",
			'42127' => "Nafría de Ucero",
			'42128' => "Narros",
			'42129' => "Navaleno",
			'42130' => "Nepas",
			'42131' => "Nolay",
			'42132' => "Noviercas",
			'42134' => "Ólvega",
			'42135' => "Oncala",
			'42139' => "Pinilla del Campo",
			'42140' => "Portillo de Soria",
			'42141' => "Póveda de Soria, La",
			'42142' => "Pozalmuro",
			'42144' => "Quintana Redonda",
			'42145' => "Quintanas de Gormaz",
			'42148' => "Quiñonería",
			'42149' => "Rábanos, Los",
			'42151' => "Rebollar",
			'42152' => "Recuerda",
			'42153' => "Rello",
			'42154' => "Renieblas",
			'42155' => "Retortillo de Soria",
			'42156' => "Reznos",
			'42157' => "Riba de Escalote, La",
			'42158' => "Rioseco de Soria",
			'42159' => "Rollamienta",
			'42160' => "Royo, El",
			'42161' => "Salduero",
			'42162' => "San Esteban de Gormaz",
			'42163' => "San Felices",
			'42164' => "San Leonardo de Yagüe",
			'42165' => "San Pedro Manrique",
			'42166' => "Santa Cruz de Yanguas",
			'42167' => "Santa María de Huerta",
			'42168' => "Santa María de las Hoyas",
			'42171' => "Serón de Nágima",
			'42172' => "Soliedra",
			'42173' => "Soria",
			'42174' => "Sotillo del Rincón",
			'42175' => "Suellacabras",
			'42176' => "Tajahuerce",
			'42177' => "Tajueco",
			'42178' => "Talveila",
			'42181' => "Tardelcuende",
			'42182' => "Taroda",
			'42183' => "Tejado",
			'42184' => "Torlengua",
			'42185' => "Torreblacos",
			'42187' => "Torrubia de Soria",
			'42188' => "Trévago",
			'42189' => "Ucero",
			'42190' => "Vadillo",
			'42191' => "Valdeavellano de Tera",
			'42192' => "Valdegeña",
			'42193' => "Valdelagua del Cerro",
			'42194' => "Valdemaluque",
			'42195' => "Valdenebro",
			'42196' => "Valdeprado",
			'42197' => "Valderrodilla",
			'42198' => "Valtajeros",
			'42200' => "Velamazán",
			'42201' => "Velilla de la Sierra",
			'42202' => "Velilla de los Ajos",
			'42204' => "Viana de Duero",
			'42205' => "Villaciervos",
			'42206' => "Villanueva de Gormaz",
			'42207' => "Villar del Ala",
			'42208' => "Villar del Campo",
			'42209' => "Villar del Río",
			'42211' => "Villares de Soria, Los",
			'42212' => "Villasayas",
			'42213' => "Villaseca de Arciel",
			'42215' => "Vinuesa",
			'42216' => "Vizmanos",
			'42217' => "Vozmediano",
			'42218' => "Yanguas",
			'42219' => "Yelo",
			'43001' => "Aiguamúrcia",
			'43002' => "Albinyana",
			'43003' => "Albiol, L'",
			'43004' => "Alcanar",
			'43005' => "Alcover",
			'43904' => "Aldea, L'",
			'43006' => "Aldover",
			'43007' => "Aleixar, L'",
			'43008' => "Alfara de Carles",
			'43009' => "Alforja",
			'43010' => "Alió",
			'43011' => "Almoster",
			'43012' => "Altafulla",
			'43013' => "Ametlla de Mar, L'",
			'43906' => "Ampolla, L'",
			'43014' => "Amposta",
			'43016' => "Arboç, L'",
			'43015' => "Arbolí",
			'43017' => "Argentera, L'",
			'43018' => "Arnes",
			'43019' => "Ascó",
			'43020' => "Banyeres del Penedès",
			'43021' => "Barberà de la Conca",
			'43022' => "Batea",
			'43023' => "Bellmunt del Priorat",
			'43024' => "Bellvei",
			'43025' => "Benifallet",
			'43026' => "Benissanet",
			'43027' => "Bisbal de Falset, La",
			'43028' => "Bisbal del Penedès, La",
			'43029' => "Blancafort",
			'43030' => "Bonastre",
			'43031' => "Borges del Camp, Les",
			'43032' => "Bot",
			'43033' => "Botarell",
			'43034' => "Bràfim",
			'43035' => "Cabacés",
			'43036' => "Cabra del Camp",
			'43037' => "Calafell",
			'43903' => "Camarles",
			'43038' => "Cambrils",
			'43907' => "Canonja, La",
			'43039' => "Capafonts",
			'43040' => "Capçanes",
			'43041' => "Caseres",
			'43042' => "Castellvell del Camp",
			'43043' => "Catllar, El",
			'43045' => "Colldejou",
			'43046' => "Conesa",
			'43047' => "Constantí",
			'43048' => "Corbera d'Ebre",
			'43049' => "Cornudella de Montsant",
			'43050' => "Creixell",
			'43051' => "Cunit",
			'43901' => "Deltebre",
			'43053' => "Duesaigües",
			'43054' => "Espluga de Francolí, L'",
			'43055' => "Falset",
			'43056' => "Fatarella, La",
			'43057' => "Febró, La",
			'43058' => "Figuera, La",
			'43059' => "Figuerola del Camp",
			'43060' => "Flix",
			'43061' => "Forès",
			'43062' => "Freginals",
			'43063' => "Galera, La",
			'43064' => "Gandesa",
			'43065' => "Garcia",
			'43066' => "Garidells, Els",
			'43067' => "Ginestar",
			'43068' => "Godall",
			'43069' => "Gratallops",
			'43070' => "Guiamets, Els",
			'43071' => "Horta de Sant Joan",
			'43072' => "Lloar, El",
			'43073' => "Llorac",
			'43074' => "Llorenç del Penedès",
			'43076' => "Marçà",
			'43075' => "Margalef",
			'43077' => "Mas de Barberans",
			'43078' => "Masdenverge",
			'43079' => "Masllorenç",
			'43080' => "Masó, La",
			'43081' => "Maspujols",
			'43082' => "Masroig, El",
			'43083' => "Milà, El",
			'43084' => "Miravet",
			'43085' => "Molar, El",
			'43086' => "Montblanc",
			'43088' => "Montbrió del Camp",
			'43089' => "Montferri",
			'43090' => "Montmell, El",
			'43091' => "Mont-ral",
			'43092' => "Mont-roig del Camp",
			'43093' => "Móra d'Ebre",
			'43094' => "Móra la Nova",
			'43095' => "Morell, El",
			'43096' => "Morera de Montsant, La",
			'43097' => "Nou de Gaià, La",
			'43098' => "Nulles",
			'43100' => "Pallaresos, Els",
			'43099' => "Palma d'Ebre, La",
			'43101' => "Passanant i Belltall",
			'43102' => "Paüls",
			'43103' => "Perafort",
			'43104' => "Perelló, El",
			'43105' => "Piles, Les",
			'43106' => "Pinell de Brai, El",
			'43107' => "Pira",
			'43108' => "Pla de Santa Maria, El",
			'43109' => "Pobla de Mafumet, La",
			'43110' => "Pobla de Massaluca, La",
			'43111' => "Pobla de Montornès, La",
			'43112' => "Poboleda",
			'43113' => "Pont d'Armentera, El",
			'43141' => "Pontils",
			'43114' => "Porrera",
			'43115' => "Pradell de la Teixeta",
			'43116' => "Prades",
			'43117' => "Prat de Comte",
			'43118' => "Pratdip",
			'43119' => "Puigpelat",
			'43120' => "Querol",
			'43136' => "Ràpita, La",
			'43121' => "Rasquera",
			'43122' => "Renau",
			'43123' => "Reus",
			'43124' => "Riba, La",
			'43125' => "Riba-roja d'Ebre",
			'43126' => "Riera de Gaià, La",
			'43127' => "Riudecanyes",
			'43128' => "Riudecols",
			'43129' => "Riudoms",
			'43130' => "Rocafort de Queralt",
			'43131' => "Roda de Berà",
			'43132' => "Rodonyà",
			'43133' => "Roquetes",
			'43134' => "Rourell, El",
			'43135' => "Salomó",
			'43905' => "Salou",
			'43137' => "Sant Jaume dels Domenys",
			'43902' => "Sant Jaume d'Enveja",
			'43138' => "Santa Bàrbara",
			'43139' => "Santa Coloma de Queralt",
			'43140' => "Santa Oliva",
			'43142' => "Sarral",
			'43143' => "Savallà del Comtat",
			'43144' => "Secuita, La",
			'43145' => "Selva del Camp, La",
			'43146' => "Senan",
			'43044' => "Sénia, La",
			'43147' => "Solivella",
			'43148' => "Tarragona",
			'43149' => "Tivenys",
			'43150' => "Tivissa",
			'43151' => "Torre de Fontaubella, La",
			'43152' => "Torre de l'Espanyol, La",
			'43153' => "Torredembarra",
			'43154' => "Torroja del Priorat",
			'43155' => "Tortosa",
			'43156' => "Ulldecona",
			'43157' => "Ulldemolins",
			'43158' => "Vallclara",
			'43159' => "Vallfogona de Riucorb",
			'43160' => "Vallmoll",
			'43161' => "Valls",
			'43162' => "Vandellòs i l'Hospitalet de l'Infant",
			'43163' => "Vendrell, El",
			'43164' => "Vespella de Gaià",
			'43165' => "Vilabella",
			'43175' => "Vilalba dels Arcs",
			'43166' => "Vilallonga del Camp",
			'43168' => "Vilanova de Prades",
			'43167' => "Vilanova d'Escornalbou",
			'43169' => "Vilaplana",
			'43170' => "Vila-rodona",
			'43171' => "Vila-seca",
			'43172' => "Vilaverd",
			'43173' => "Vilella Alta, La",
			'43174' => "Vilella Baixa, La",
			'43176' => "Vimbodí i Poblet",
			'43177' => "Vinebre",
			'43178' => "Vinyols i els Arcs",
			'43052' => "Xerta",
			'44001' => "Ababuj",
			'44002' => "Abejuela",
			'44003' => "Aguatón",
			'44004' => "Aguaviva",
			'44005' => "Aguilar del Alfambra",
			'44006' => "Alacón",
			'44007' => "Alba",
			'44008' => "Albalate del Arzobispo",
			'44009' => "Albarracín",
			'44010' => "Albentosa",
			'44011' => "Alcaine",
			'44012' => "Alcalá de la Selva",
			'44013' => "Alcañiz",
			'44014' => "Alcorisa",
			'44016' => "Alfambra",
			'44017' => "Aliaga",
			'44021' => "Allepuz",
			'44022' => "Alloza",
			'44023' => "Allueva",
			'44018' => "Almohaja",
			'44019' => "Alobras",
			'44020' => "Alpeñés",
			'44024' => "Anadón",
			'44025' => "Andorra",
			'44026' => "Arcos de las Salinas",
			'44027' => "Arens de Lledó",
			'44028' => "Argente",
			'44029' => "Ariño",
			'44031' => "Azaila",
			'44032' => "Bádenas",
			'44033' => "Báguena",
			'44034' => "Bañón",
			'44035' => "Barrachina",
			'44036' => "Bea",
			'44037' => "Beceite",
			'44039' => "Bello",
			'44038' => "Belmonte de San José",
			'44040' => "Berge",
			'44041' => "Bezas",
			'44042' => "Blancas",
			'44043' => "Blesa",
			'44044' => "Bordón",
			'44045' => "Bronchales",
			'44046' => "Bueña",
			'44047' => "Burbáguena",
			'44048' => "Cabra de Mora",
			'44049' => "Calaceite",
			'44050' => "Calamocha",
			'44051' => "Calanda",
			'44052' => "Calomarde",
			'44053' => "Camañas",
			'44054' => "Camarena de la Sierra",
			'44055' => "Camarillas",
			'44056' => "Caminreal",
			'44059' => "Cantavieja",
			'44060' => "Cañada de Benatanduz",
			'44061' => "Cañada de Verich, La",
			'44062' => "Cañada Vellida",
			'44063' => "Cañizar del Olivar",
			'44064' => "Cascante del Río",
			'44065' => "Castejón de Tornos",
			'44066' => "Castel de Cabra",
			'44070' => "Castellar, El",
			'44071' => "Castellote",
			'44067' => "Castelnou",
			'44068' => "Castelserás",
			'44074' => "Cedrillas",
			'44075' => "Celadas",
			'44076' => "Cella",
			'44077' => "Cerollera, La",
			'44080' => "Codoñera, La",
			'44082' => "Corbalán",
			'44084' => "Cortes de Aragón",
			'44085' => "Cosa",
			'44086' => "Cretas",
			'44087' => "Crivillén",
			'44088' => "Cuba, La",
			'44089' => "Cubla",
			'44090' => "Cucalón",
			'44092' => "Cuervo, El",
			'44093' => "Cuevas de Almudén",
			'44094' => "Cuevas Labradas",
			'44096' => "Ejulve",
			'44097' => "Escorihuela",
			'44099' => "Escucha",
			'44100' => "Estercuel",
			'44101' => "Ferreruela de Huerva",
			'44102' => "Fonfría",
			'44103' => "Formiche Alto",
			'44105' => "Fórnoles",
			'44106' => "Fortanete",
			'44107' => "Foz-Calanda",
			'44108' => "Fresneda, La",
			'44109' => "Frías de Albarracín",
			'44110' => "Fuenferrada",
			'44111' => "Fuentes Calientes",
			'44112' => "Fuentes Claras",
			'44113' => "Fuentes de Rubielos",
			'44114' => "Fuentespalda",
			'44115' => "Galve",
			'44116' => "Gargallo",
			'44117' => "Gea de Albarracín",
			'44118' => "Ginebrosa, La",
			'44119' => "Griegos",
			'44120' => "Guadalaviar",
			'44121' => "Gúdar",
			'44122' => "Híjar",
			'44123' => "Hinojosa de Jarque",
			'44124' => "Hoz de la Vieja, La",
			'44125' => "Huesa del Común",
			'44126' => "Iglesuela del Cid, La",
			'44127' => "Jabaloyas",
			'44128' => "Jarque de la Val",
			'44129' => "Jatiel",
			'44130' => "Jorcas",
			'44131' => "Josa",
			'44132' => "Lagueruela",
			'44133' => "Lanzuela",
			'44135' => "Libros",
			'44136' => "Lidón",
			'44137' => "Linares de Mora",
			'44141' => "Lledó",
			'44138' => "Loscos",
			'44142' => "Maicas",
			'44143' => "Manzanera",
			'44144' => "Martín del Río",
			'44145' => "Mas de las Matas",
			'44146' => "Mata de los Olmos, La",
			'44147' => "Mazaleón",
			'44148' => "Mezquita de Jarque",
			'44149' => "Mirambel",
			'44150' => "Miravete de la Sierra",
			'44151' => "Molinos",
			'44152' => "Monforte de Moyuela",
			'44153' => "Monreal del Campo",
			'44154' => "Monroyo",
			'44155' => "Montalbán",
			'44156' => "Monteagudo del Castillo",
			'44157' => "Monterde de Albarracín",
			'44158' => "Mora de Rubielos",
			'44159' => "Moscardón",
			'44160' => "Mosqueruela",
			'44161' => "Muniesa",
			'44163' => "Noguera de Albarracín",
			'44164' => "Nogueras",
			'44165' => "Nogueruelas",
			'44167' => "Obón",
			'44168' => "Odón",
			'44169' => "Ojos Negros",
			'44171' => "Olba",
			'44172' => "Oliete",
			'44173' => "Olmos, Los",
			'44174' => "Orihuela del Tremedal",
			'44175' => "Orrios",
			'44176' => "Palomar de Arroyos",
			'44177' => "Pancrudo",
			'44178' => "Parras de Castellote, Las",
			'44179' => "Peñarroya de Tastavins",
			'44180' => "Peracense",
			'44181' => "Peralejos",
			'44182' => "Perales del Alfambra",
			'44183' => "Pitarque",
			'44184' => "Plou",
			'44185' => "Pobo, El",
			'44187' => "Portellada, La",
			'44189' => "Pozondón",
			'44190' => "Pozuel del Campo",
			'44191' => "Puebla de Híjar, La",
			'44192' => "Puebla de Valverde, La",
			'44193' => "Puertomingalvo",
			'44194' => "Ráfales",
			'44195' => "Rillo",
			'44196' => "Riodeva",
			'44197' => "Ródenas",
			'44198' => "Royuela",
			'44199' => "Rubiales",
			'44200' => "Rubielos de la Cérida",
			'44201' => "Rubielos de Mora",
			'44203' => "Salcedillo",
			'44204' => "Saldón",
			'44205' => "Samper de Calanda",
			'44206' => "San Agustín",
			'44207' => "San Martín del Río",
			'44208' => "Santa Cruz de Nogueras",
			'44209' => "Santa Eulalia",
			'44210' => "Sarrión",
			'44211' => "Segura de los Baños",
			'44212' => "Seno",
			'44213' => "Singra",
			'44215' => "Terriente",
			'44216' => "Teruel",
			'44217' => "Toril y Masegoso",
			'44218' => "Tormón",
			'44219' => "Tornos",
			'44220' => "Torralba de los Sisones",
			'44223' => "Torre de Arcas",
			'44224' => "Torre de las Arcas",
			'44225' => "Torre del Compte",
			'44227' => "Torre los Negros",
			'44221' => "Torrecilla de Alcañiz",
			'44222' => "Torrecilla del Rebollar",
			'44226' => "Torrelacárcel",
			'44228' => "Torremocha de Jiloca",
			'44229' => "Torres de Albarracín",
			'44230' => "Torrevelilla",
			'44231' => "Torrijas",
			'44232' => "Torrijo del Campo",
			'44234' => "Tramacastiel",
			'44235' => "Tramacastilla",
			'44236' => "Tronchón",
			'44237' => "Urrea de Gaén",
			'44238' => "Utrillas",
			'44239' => "Valacloche",
			'44240' => "Valbona",
			'44241' => "Valdealgorfa",
			'44243' => "Valdecuenca",
			'44244' => "Valdelinares",
			'44245' => "Valdeltormo",
			'44246' => "Valderrobres",
			'44247' => "Valjunquera",
			'44249' => "Vallecillo, El",
			'44250' => "Veguillas de la Sierra",
			'44251' => "Villafranca del Campo",
			'44252' => "Villahermosa del Campo",
			'44256' => "Villanueva del Rebollar de la Sierra",
			'44257' => "Villar del Cobo",
			'44258' => "Villar del Salz",
			'44260' => "Villarluengo",
			'44261' => "Villarquemado",
			'44262' => "Villarroya de los Pinares",
			'44263' => "Villastar",
			'44264' => "Villel",
			'44265' => "Vinaceite",
			'44266' => "Visiedo",
			'44267' => "Vivel del Río Martín",
			'44268' => "Zoma, La",
			'45001' => "Ajofrín",
			'45002' => "Alameda de la Sagra",
			'45003' => "Albarreal de Tajo",
			'45004' => "Alcabón",
			'45005' => "Alcañizo",
			'45006' => "Alcaudete de la Jara",
			'45007' => "Alcolea de Tajo",
			'45008' => "Aldea en Cabo",
			'45009' => "Aldeanueva de Barbarroya",
			'45010' => "Aldeanueva de San Bartolomé",
			'45011' => "Almendral de la Cañada",
			'45012' => "Almonacid de Toledo",
			'45013' => "Almorox",
			'45014' => "Añover de Tajo",
			'45015' => "Arcicóllar",
			'45016' => "Argés",
			'45017' => "Azután",
			'45018' => "Barcience",
			'45019' => "Bargas",
			'45020' => "Belvís de la Jara",
			'45021' => "Borox",
			'45022' => "Buenaventura",
			'45023' => "Burguillos de Toledo",
			'45024' => "Burujón",
			'45025' => "Cabañas de la Sagra",
			'45026' => "Cabañas de Yepes",
			'45027' => "Cabezamesada",
			'45028' => "Calera y Chozas",
			'45029' => "Caleruela",
			'45030' => "Calzada de Oropesa",
			'45031' => "Camarena",
			'45032' => "Camarenilla",
			'45033' => "Campillo de la Jara, El",
			'45034' => "Camuñas",
			'45035' => "Cardiel de los Montes",
			'45036' => "Carmena",
			'45037' => "Carpio de Tajo, El",
			'45038' => "Carranque",
			'45039' => "Carriches",
			'45040' => "Casar de Escalona, El",
			'45041' => "Casarrubios del Monte",
			'45042' => "Casasbuenas",
			'45043' => "Castillo de Bayuela",
			'45045' => "Cazalegas",
			'45046' => "Cebolla",
			'45047' => "Cedillo del Condado",
			'45048' => "Cerralbos, Los",
			'45049' => "Cervera de los Montes",
			'45056' => "Chozas de Canales",
			'45057' => "Chueca",
			'45050' => "Ciruelos",
			'45051' => "Cobeja",
			'45052' => "Cobisa",
			'45053' => "Consuegra",
			'45054' => "Corral de Almaguer",
			'45055' => "Cuerva",
			'45058' => "Domingo Pérez",
			'45059' => "Dosbarrios",
			'45060' => "Erustes",
			'45061' => "Escalona",
			'45062' => "Escalonilla",
			'45063' => "Espinoso del Rey",
			'45064' => "Esquivias",
			'45065' => "Estrella, La",
			'45066' => "Fuensalida",
			'45067' => "Gálvez",
			'45068' => "Garciotum",
			'45069' => "Gerindote",
			'45070' => "Guadamur",
			'45071' => "Guardia, La",
			'45072' => "Herencias, Las",
			'45073' => "Herreruela de Oropesa",
			'45074' => "Hinojosa de San Vicente",
			'45075' => "Hontanar",
			'45076' => "Hormigos",
			'45077' => "Huecas",
			'45078' => "Huerta de Valdecarábanos",
			'45079' => "Iglesuela del Tiétar, La",
			'45080' => "Illán de Vacas",
			'45081' => "Illescas",
			'45082' => "Lagartera",
			'45083' => "Layos",
			'45084' => "Lillo",
			'45085' => "Lominchar",
			'45086' => "Lucillos",
			'45087' => "Madridejos",
			'45088' => "Magán",
			'45089' => "Malpica de Tajo",
			'45090' => "Manzaneque",
			'45091' => "Maqueda",
			'45092' => "Marjaliza",
			'45093' => "Marrupe",
			'45094' => "Mascaraque",
			'45095' => "Mata, La",
			'45096' => "Mazarambroz",
			'45097' => "Mejorada",
			'45098' => "Menasalbas",
			'45099' => "Méntrida",
			'45100' => "Mesegar de Tajo",
			'45101' => "Miguel Esteban",
			'45102' => "Mocejón",
			'45103' => "Mohedas de la Jara",
			'45104' => "Montearagón",
			'45105' => "Montesclaros",
			'45106' => "Mora",
			'45107' => "Nambroca",
			'45108' => "Nava de Ricomalillo, La",
			'45109' => "Navahermosa",
			'45110' => "Navalcán",
			'45111' => "Navalmoralejo",
			'45112' => "Navalmorales, Los",
			'45113' => "Navalucillos, Los",
			'45114' => "Navamorcuende",
			'45115' => "Noblejas",
			'45116' => "Noez",
			'45117' => "Nombela",
			'45118' => "Novés",
			'45119' => "Numancia de la Sagra",
			'45120' => "Nuño Gómez",
			'45121' => "Ocaña",
			'45122' => "Olías del Rey",
			'45123' => "Ontígola",
			'45124' => "Orgaz",
			'45125' => "Oropesa",
			'45126' => "Otero",
			'45127' => "Palomeque",
			'45128' => "Pantoja",
			'45129' => "Paredes de Escalona",
			'45130' => "Parrillas",
			'45131' => "Pelahustán",
			'45132' => "Pepino",
			'45133' => "Polán",
			'45134' => "Portillo de Toledo",
			'45135' => "Puebla de Almoradiel, La",
			'45136' => "Puebla de Montalbán, La",
			'45137' => "Pueblanueva, La",
			'45138' => "Puente del Arzobispo, El",
			'45139' => "Puerto de San Vicente",
			'45140' => "Pulgar",
			'45141' => "Quero",
			'45142' => "Quintanar de la Orden",
			'45143' => "Quismondo",
			'45144' => "Real de San Vicente, El",
			'45145' => "Recas",
			'45146' => "Retamoso de la Jara",
			'45147' => "Rielves",
			'45148' => "Robledo del Mazo",
			'45149' => "Romeral, El",
			'45150' => "San Bartolomé de las Abiertas",
			'45151' => "San Martín de Montalbán",
			'45152' => "San Martín de Pusa",
			'45153' => "San Pablo de los Montes",
			'45154' => "San Román de los Montes",
			'45155' => "Santa Ana de Pusa",
			'45156' => "Santa Cruz de la Zarza",
			'45157' => "Santa Cruz del Retamar",
			'45158' => "Santa Olalla",
			'45901' => "Santo Domingo-Caudilla",
			'45159' => "Sartajada",
			'45160' => "Segurilla",
			'45161' => "Seseña",
			'45162' => "Sevilleja de la Jara",
			'45163' => "Sonseca",
			'45164' => "Sotillo de las Palomas",
			'45165' => "Talavera de la Reina",
			'45166' => "Tembleque",
			'45167' => "Toboso, El",
			'45168' => "Toledo",
			'45169' => "Torralba de Oropesa",
			'45171' => "Torre de Esteban Hambrán, La",
			'45170' => "Torrecilla de la Jara",
			'45172' => "Torrico",
			'45173' => "Torrijos",
			'45174' => "Totanés",
			'45175' => "Turleque",
			'45176' => "Ugena",
			'45177' => "Urda",
			'45179' => "Valdeverdeja",
			'45180' => "Valmojado",
			'45181' => "Velada",
			'45182' => "Ventas con Peña Aguilera, Las",
			'45183' => "Ventas de Retamosa, Las",
			'45184' => "Ventas de San Julián, Las",
			'45186' => "Villa de Don Fadrique, La",
			'45185' => "Villacañas",
			'45187' => "Villafranca de los Caballeros",
			'45188' => "Villaluenga de la Sagra",
			'45189' => "Villamiel de Toledo",
			'45190' => "Villaminaya",
			'45191' => "Villamuelas",
			'45192' => "Villanueva de Alcardete",
			'45193' => "Villanueva de Bogas",
			'45194' => "Villarejo de Montalbán",
			'45195' => "Villarrubia de Santiago",
			'45196' => "Villaseca de la Sagra",
			'45197' => "Villasequilla",
			'45198' => "Villatobas",
			'45199' => "Viso de San Juan, El",
			'45200' => "Yébenes, Los",
			'45201' => "Yeles",
			'45202' => "Yepes",
			'45203' => "Yuncler",
			'45204' => "Yunclillos",
			'45205' => "Yuncos",
			'46001' => "Ademuz",
			'46002' => "Ador",
			'46004' => "Agullent",
			'46042' => "Aielo de Malferit",
			'46043' => "Aielo de Rugat",
			'46005' => "Alaquàs",
			'46006' => "Albaida",
			'46007' => "Albal",
			'46008' => "Albalat de la Ribera",
			'46009' => "Albalat dels Sorells",
			'46010' => "Albalat dels Tarongers",
			'46011' => "Alberic",
			'46012' => "Alborache",
			'46013' => "Alboraia/Alboraya",
			'46014' => "Albuixech",
			'46016' => "Alcàntera de Xúquer",
			'46015' => "Alcàsser",
			'46018' => "Alcublas",
			'46020' => "Alcúdia de Crespins, l'",
			'46019' => "Alcúdia, l'",
			'46021' => "Aldaia",
			'46022' => "Alfafar",
			'46024' => "Alfara de la Baronia",
			'46025' => "Alfara del Patriarca",
			'46026' => "Alfarb",
			'46027' => "Alfarrasí",
			'46023' => "Alfauir",
			'46028' => "Algar de Palància",
			'46029' => "Algemesí",
			'46030' => "Algímia d'Alfara",
			'46031' => "Alginet",
			'46032' => "Almàssera",
			'46033' => "Almiserà",
			'46034' => "Almoines",
			'46035' => "Almussafes",
			'46036' => "Alpuente",
			'46037' => "Alqueria de la Comtessa, l'",
			'46017' => "Alzira",
			'46038' => "Andilla",
			'46039' => "Anna",
			'46040' => "Antella",
			'46041' => "Aras de los Olmos",
			'46003' => "Atzeneta d'Albaida",
			'46044' => "Ayora",
			'46046' => "Barx",
			'46045' => "Barxeta",
			'46047' => "Bèlgida",
			'46048' => "Bellreguard",
			'46049' => "Bellús",
			'46050' => "Benagéber",
			'46051' => "Benaguasil",
			'46052' => "Benavites",
			'46053' => "Beneixida",
			'46054' => "Benetússer",
			'46055' => "Beniarjó",
			'46056' => "Beniatjar",
			'46057' => "Benicolet",
			'46904' => "Benicull de Xúquer",
			'46060' => "Benifaió",
			'46059' => "Benifairó de la Valldigna",
			'46058' => "Benifairó de les Valls",
			'46061' => "Beniflá",
			'46062' => "Benigànim",
			'46063' => "Benimodo",
			'46064' => "Benimuslem",
			'46065' => "Beniparrell",
			'46066' => "Benirredrà",
			'46067' => "Benissanó",
			'46068' => "Benissoda",
			'46069' => "Benissuera",
			'46070' => "Bétera",
			'46071' => "Bicorp",
			'46072' => "Bocairent",
			'46073' => "Bolbaite",
			'46074' => "Bonrepòs i Mirambell",
			'46075' => "Bufali",
			'46076' => "Bugarra",
			'46077' => "Buñol",
			'46078' => "Burjassot",
			'46079' => "Calles",
			'46080' => "Camporrobles",
			'46081' => "Canals",
			'46082' => "Canet d'En Berenguer",
			'46083' => "Carcaixent",
			'46084' => "Càrcer",
			'46085' => "Carlet",
			'46086' => "Carrícola",
			'46087' => "Casas Altas",
			'46088' => "Casas Bajas",
			'46089' => "Casinos",
			'46257' => "Castelló",
			'46090' => "Castelló de Rugat",
			'46091' => "Castellonet de la Conquesta",
			'46092' => "Castielfabib",
			'46093' => "Catadau",
			'46094' => "Catarroja",
			'46095' => "Caudete de las Fuentes",
			'46096' => "Cerdà",
			'46107' => "Chella",
			'46106' => "Chelva",
			'46108' => "Chera",
			'46109' => "Cheste",
			'46111' => "Chiva",
			'46112' => "Chulilla",
			'46097' => "Cofrentes",
			'46098' => "Corbera",
			'46099' => "Cortes de Pallás",
			'46100' => "Cotes",
			'46105' => "Cullera",
			'46113' => "Daimús",
			'46114' => "Domeño",
			'46115' => "Dos Aguas",
			'46116' => "Eliana, l'",
			'46117' => "Emperador",
			'46118' => "Enguera",
			'46119' => "Énova, l'",
			'46120' => "Estivella",
			'46121' => "Estubeny",
			'46122' => "Faura",
			'46123' => "Favara",
			'46126' => "Foios",
			'46128' => "Font de la Figuera, la",
			'46127' => "Font d'en Carròs, la",
			'46124' => "Fontanars dels Alforins",
			'46125' => "Fortaleny",
			'46129' => "Fuenterrobles",
			'46131' => "Gandia",
			'46902' => "Gátova",
			'46130' => "Gavarda",
			'46132' => "Genovés, el",
			'46133' => "Gestalgar",
			'46134' => "Gilet",
			'46135' => "Godella",
			'46136' => "Godelleta",
			'46137' => "Granja de la Costera, la",
			'46138' => "Guadasséquies",
			'46139' => "Guadassuar",
			'46140' => "Guardamar de la Safor",
			'46141' => "Higueruelas",
			'46142' => "Jalance",
			'46144' => "Jarafuel",
			'46154' => "Llanera de Ranes",
			'46155' => "Llaurí",
			'46147' => "Llíria",
			'46152' => "Llocnou de la Corona",
			'46153' => "Llocnou de Sant Jeroni",
			'46151' => "Llocnou d'En Fenollet",
			'46156' => "Llombai",
			'46157' => "Llosa de Ranes, la",
			'46150' => "Llutxent",
			'46148' => "Loriguilla",
			'46149' => "Losa del Obispo",
			'46158' => "Macastre",
			'46159' => "Manises",
			'46160' => "Manuel",
			'46161' => "Marines",
			'46162' => "Massalavés",
			'46163' => "Massalfassar",
			'46164' => "Massamagrell",
			'46165' => "Massanassa",
			'46166' => "Meliana",
			'46167' => "Millares",
			'46168' => "Miramar",
			'46169' => "Mislata",
			'46170' => "Moixent/Mogente",
			'46171' => "Moncada",
			'46173' => "Montaverner",
			'46174' => "Montesa",
			'46175' => "Montitxelvo/Montichelvo",
			'46176' => "Montroi/Montroy",
			'46172' => "Montserrat",
			'46177' => "Museros",
			'46178' => "Nàquera/Náquera",
			'46179' => "Navarrés",
			'46180' => "Novetlè",
			'46181' => "Oliva",
			'46183' => "Olleria, l'",
			'46182' => "Olocau",
			'46184' => "Ontinyent",
			'46185' => "Otos",
			'46186' => "Paiporta",
			'46187' => "Palma de Gandía",
			'46188' => "Palmera",
			'46189' => "Palomar, el",
			'46190' => "Paterna",
			'46191' => "Pedralba",
			'46192' => "Petrés",
			'46193' => "Picanya",
			'46194' => "Picassent",
			'46195' => "Piles",
			'46196' => "Pinet",
			'46199' => "Pobla de Farnals, la",
			'46202' => "Pobla de Vallbona, la",
			'46200' => "Pobla del Duc, la",
			'46203' => "Pobla Llarga, la",
			'46197' => "Polinyà de Xúquer",
			'46198' => "Potries",
			'46205' => "Puçol",
			'46201' => "Puebla de San Miguel",
			'46204' => "Puig de Santa Maria, el",
			'46101' => "Quart de les Valls",
			'46102' => "Quart de Poblet",
			'46103' => "Quartell",
			'46104' => "Quatretonda",
			'46206' => "Quesa",
			'46207' => "Rafelbunyol",
			'46208' => "Rafelcofer",
			'46209' => "Rafelguaraf",
			'46210' => "Ráfol de Salem",
			'46212' => "Real",
			'46211' => "Real de Gandia, el",
			'46213' => "Requena",
			'46214' => "Riba-roja de Túria",
			'46215' => "Riola",
			'46216' => "Rocafort",
			'46217' => "Rotglà i Corberà",
			'46218' => "Ròtova",
			'46219' => "Rugat",
			'46220' => "Sagunt/Sagunto",
			'46221' => "Salem",
			'46903' => "San Antonio de Benagéber",
			'46222' => "Sant Joanet",
			'46223' => "Sedaví",
			'46224' => "Segart",
			'46225' => "Sellent",
			'46226' => "Sempere",
			'46227' => "Senyera",
			'46228' => "Serra",
			'46229' => "Siete Aguas",
			'46230' => "Silla",
			'46231' => "Simat de la Valldigna",
			'46232' => "Sinarcas",
			'46233' => "Sollana",
			'46234' => "Sot de Chera",
			'46235' => "Sueca",
			'46236' => "Sumacàrcer",
			'46237' => "Tavernes Blanques",
			'46238' => "Tavernes de la Valldigna",
			'46239' => "Teresa de Cofrentes",
			'46240' => "Terrateig",
			'46241' => "Titaguas",
			'46242' => "Torrebaja",
			'46243' => "Torrella",
			'46244' => "Torrent",
			'46245' => "Torres Torres",
			'46246' => "Tous",
			'46247' => "Tuéjar",
			'46248' => "Turís",
			'46249' => "Utiel",
			'46250' => "València",
			'46251' => "Vallada",
			'46252' => "Vallanca",
			'46253' => "Vallés",
			'46254' => "Venta del Moro",
			'46255' => "Vilallonga/Villalonga",
			'46256' => "Vilamarxant",
			'46258' => "Villar del Arzobispo",
			'46259' => "Villargordo del Cabriel",
			'46260' => "Vinalesa",
			'46145' => "Xàtiva",
			'46143' => "Xeraco",
			'46146' => "Xeresa",
			'46110' => "Xirivella",
			'46261' => "Yátova",
			'46262' => "Yesa, La",
			'46263' => "Zarra",
			'47001' => "Adalia",
			'47002' => "Aguasal",
			'47003' => "Aguilar de Campos",
			'47004' => "Alaejos",
			'47005' => "Alcazarén",
			'47006' => "Aldea de San Miguel",
			'47007' => "Aldeamayor de San Martín",
			'47008' => "Almenara de Adaja",
			'47009' => "Amusquillo",
			'47010' => "Arroyo de la Encomienda",
			'47011' => "Ataquines",
			'47012' => "Bahabón",
			'47013' => "Barcial de la Loma",
			'47014' => "Barruelo del Valle",
			'47015' => "Becilla de Valderaduey",
			'47016' => "Benafarces",
			'47017' => "Bercero",
			'47018' => "Berceruelo",
			'47019' => "Berrueces",
			'47020' => "Bobadilla del Campo",
			'47021' => "Bocigas",
			'47022' => "Bocos de Duero",
			'47023' => "Boecillo",
			'47024' => "Bolaños de Campos",
			'47025' => "Brahojos de Medina",
			'47026' => "Bustillo de Chaves",
			'47027' => "Cabezón de Pisuerga",
			'47028' => "Cabezón de Valderaduey",
			'47029' => "Cabreros del Monte",
			'47030' => "Campaspero",
			'47031' => "Campillo, El",
			'47032' => "Camporredondo",
			'47033' => "Canalejas de Peñafiel",
			'47034' => "Canillas de Esgueva",
			'47035' => "Carpio",
			'47036' => "Casasola de Arión",
			'47037' => "Castrejón de Trabancos",
			'47038' => "Castrillo de Duero",
			'47039' => "Castrillo-Tejeriego",
			'47040' => "Castrobol",
			'47041' => "Castrodeza",
			'47042' => "Castromembibre",
			'47043' => "Castromonte",
			'47044' => "Castronuevo de Esgueva",
			'47045' => "Castronuño",
			'47046' => "Castroponce",
			'47047' => "Castroverde de Cerrato",
			'47048' => "Ceinos de Campos",
			'47049' => "Cervillego de la Cruz",
			'47050' => "Cigales",
			'47051' => "Ciguñuela",
			'47052' => "Cistérniga",
			'47053' => "Cogeces de Íscar",
			'47054' => "Cogeces del Monte",
			'47055' => "Corcos",
			'47056' => "Corrales de Duero",
			'47057' => "Cubillas de Santa Marta",
			'47058' => "Cuenca de Campos",
			'47059' => "Curiel de Duero",
			'47060' => "Encinas de Esgueva",
			'47061' => "Esguevillas de Esgueva",
			'47062' => "Fombellida",
			'47063' => "Fompedraza",
			'47064' => "Fontihoyuelo",
			'47065' => "Fresno el Viejo",
			'47066' => "Fuensaldaña",
			'47067' => "Fuente el Sol",
			'47068' => "Fuente-Olmedo",
			'47069' => "Gallegos de Hornija",
			'47070' => "Gatón de Campos",
			'47071' => "Geria",
			'47073' => "Herrín de Campos",
			'47074' => "Hornillos de Eresma",
			'47075' => "Íscar",
			'47076' => "Laguna de Duero",
			'47077' => "Langayo",
			'47079' => "Llano de Olmedo",
			'47078' => "Lomoviejo",
			'47080' => "Manzanillo",
			'47081' => "Marzales",
			'47082' => "Matapozuelos",
			'47083' => "Matilla de los Caños",
			'47084' => "Mayorga",
			'47086' => "Medina de Rioseco",
			'47085' => "Medina del Campo",
			'47087' => "Megeces",
			'47088' => "Melgar de Abajo",
			'47089' => "Melgar de Arriba",
			'47090' => "Mojados",
			'47091' => "Monasterio de Vega",
			'47092' => "Montealegre de Campos",
			'47093' => "Montemayor de Pililla",
			'47094' => "Moral de la Reina",
			'47095' => "Moraleja de las Panaderas",
			'47096' => "Morales de Campos",
			'47097' => "Mota del Marqués",
			'47098' => "Mucientes",
			'47099' => "Mudarra, La",
			'47100' => "Muriel",
			'47101' => "Nava del Rey",
			'47102' => "Nueva Villa de las Torres",
			'47103' => "Olivares de Duero",
			'47104' => "Olmedo",
			'47105' => "Olmos de Esgueva",
			'47106' => "Olmos de Peñafiel",
			'47109' => "Palazuelo de Vedija",
			'47110' => "Parrilla, La",
			'47111' => "Pedraja de Portillo, La",
			'47112' => "Pedrajas de San Esteban",
			'47113' => "Pedrosa del Rey",
			'47114' => "Peñafiel",
			'47115' => "Peñaflor de Hornija",
			'47116' => "Pesquera de Duero",
			'47117' => "Piña de Esgueva",
			'47118' => "Piñel de Abajo",
			'47119' => "Piñel de Arriba",
			'47121' => "Pollos",
			'47122' => "Portillo",
			'47123' => "Pozal de Gallinas",
			'47124' => "Pozaldez",
			'47125' => "Pozuelo de la Orden",
			'47126' => "Puras",
			'47127' => "Quintanilla de Arriba",
			'47129' => "Quintanilla de Onésimo",
			'47130' => "Quintanilla de Trigueros",
			'47128' => "Quintanilla del Molar",
			'47131' => "Rábano",
			'47132' => "Ramiro",
			'47133' => "Renedo de Esgueva",
			'47134' => "Roales de Campos",
			'47135' => "Robladillo",
			'47137' => "Roturas",
			'47138' => "Rubí de Bracamonte",
			'47139' => "Rueda",
			'47140' => "Saelices de Mayorga",
			'47141' => "Salvador de Zapardiel",
			'47142' => "San Cebrián de Mazote",
			'47143' => "San Llorente",
			'47144' => "San Martín de Valvení",
			'47145' => "San Miguel del Arroyo",
			'47146' => "San Miguel del Pino",
			'47147' => "San Pablo de la Moraleja",
			'47148' => "San Pedro de Latarce",
			'47149' => "San Pelayo",
			'47150' => "San Román de Hornija",
			'47151' => "San Salvador",
			'47156' => "San Vicente del Palacio",
			'47152' => "Santa Eufemia del Arroyo",
			'47153' => "Santervás de Campos",
			'47154' => "Santibáñez de Valcorba",
			'47155' => "Santovenia de Pisuerga",
			'47157' => "Sardón de Duero",
			'47158' => "Seca, La",
			'47159' => "Serrada",
			'47160' => "Siete Iglesias de Trabancos",
			'47161' => "Simancas",
			'47162' => "Tamariz de Campos",
			'47163' => "Tiedra",
			'47164' => "Tordehumos",
			'47165' => "Tordesillas",
			'47169' => "Torre de Esgueva",
			'47170' => "Torre de Peñafiel",
			'47166' => "Torrecilla de la Abadesa",
			'47167' => "Torrecilla de la Orden",
			'47168' => "Torrecilla de la Torre",
			'47171' => "Torrelobatón",
			'47172' => "Torrescárcela",
			'47173' => "Traspinedo",
			'47174' => "Trigueros del Valle",
			'47175' => "Tudela de Duero",
			'47176' => "Unión de Campos, La",
			'47177' => "Urones de Castroponce",
			'47178' => "Urueña",
			'47179' => "Valbuena de Duero",
			'47180' => "Valdearcos de la Vega",
			'47181' => "Valdenebro de los Valles",
			'47182' => "Valdestillas",
			'47183' => "Valdunquillo",
			'47186' => "Valladolid",
			'47184' => "Valoria la Buena",
			'47185' => "Valverde de Campos",
			'47187' => "Vega de Ruiponce",
			'47188' => "Vega de Valdetronco",
			'47189' => "Velascálvaro",
			'47190' => "Velilla",
			'47191' => "Velliza",
			'47192' => "Ventosa de la Cuesta",
			'47193' => "Viana de Cega",
			'47195' => "Villabáñez",
			'47196' => "Villabaruz de Campos",
			'47197' => "Villabrágima",
			'47198' => "Villacarralón",
			'47199' => "Villacid de Campos",
			'47200' => "Villaco",
			'47203' => "Villafrades de Campos",
			'47204' => "Villafranca de Duero",
			'47205' => "Villafrechós",
			'47206' => "Villafuerte",
			'47207' => "Villagarcía de Campos",
			'47208' => "Villagómez la Nueva",
			'47209' => "Villalán de Campos",
			'47210' => "Villalar de los Comuneros",
			'47211' => "Villalba de la Loma",
			'47212' => "Villalba de los Alcores",
			'47213' => "Villalbarba",
			'47214' => "Villalón de Campos",
			'47215' => "Villamuriel de Campos",
			'47216' => "Villán de Tordesillas",
			'47217' => "Villanubla",
			'47218' => "Villanueva de Duero",
			'47219' => "Villanueva de la Condesa",
			'47220' => "Villanueva de los Caballeros",
			'47221' => "Villanueva de los Infantes",
			'47222' => "Villanueva de San Mancio",
			'47223' => "Villardefrades",
			'47224' => "Villarmentero de Esgueva",
			'47225' => "Villasexmir",
			'47226' => "Villavaquerín",
			'47227' => "Villavellid",
			'47228' => "Villaverde de Medina",
			'47229' => "Villavicencio de los Caballeros",
			'47194' => "Viloria",
			'47230' => "Wamba",
			'47231' => "Zaratán",
			'47232' => "Zarza, La",
			'48001' => "Abadiño",
			'48002' => "Abanto y Ciérvana-Abanto Zierbena",
			'48911' => "Ajangiz",
			'48912' => "Alonsotegi",
			'48003' => "Amorebieta-Etxano",
			'48004' => "Amoroto",
			'48005' => "Arakaldo",
			'48006' => "Arantzazu",
			'48093' => "Areatza",
			'48009' => "Arrankudiaga-Zollo",
			'48914' => "Arratzu",
			'48010' => "Arrieta",
			'48011' => "Arrigorriaga",
			'48023' => "Artea",
			'48008' => "Artzentales",
			'48091' => "Atxondo",
			'48070' => "Aulesti",
			'48012' => "Bakio",
			'48090' => "Balmaseda",
			'48013' => "Barakaldo",
			'48014' => "Barrika",
			'48015' => "Basauri",
			'48092' => "Bedia",
			'48016' => "Berango",
			'48017' => "Bermeo",
			'48018' => "Berriatua",
			'48019' => "Berriz",
			'48020' => "Bilbao",
			'48021' => "Busturia",
			'48901' => "Derio",
			'48026' => "Dima",
			'48027' => "Durango",
			'48028' => "Ea",
			'48031' => "Elantxobe",
			'48032' => "Elorrio",
			'48902' => "Erandio",
			'48033' => "Ereño",
			'48034' => "Ermua",
			'48079' => "Errigoiti",
			'48029' => "Etxebarri",
			'48030' => "Etxebarria",
			'48906' => "Forua",
			'48035' => "Fruiz",
			'48036' => "Galdakao",
			'48037' => "Galdames",
			'48038' => "Gamiz-Fika",
			'48039' => "Garai",
			'48040' => "Gatika",
			'48041' => "Gautegiz Arteaga",
			'48046' => "Gernika-Lumo",
			'48044' => "Getxo",
			'48047' => "Gizaburuaga",
			'48042' => "Gordexola",
			'48043' => "Gorliz",
			'48045' => "Güeñes",
			'48048' => "Ibarrangelu",
			'48094' => "Igorre",
			'48049' => "Ispaster",
			'48910' => "Iurreta",
			'48050' => "Izurtza",
			'48022' => "Karrantza Harana/Valle de Carranza",
			'48907' => "Kortezubi",
			'48051' => "Lanestosa",
			'48052' => "Larrabetzu",
			'48053' => "Laukiz",
			'48054' => "Leioa",
			'48057' => "Lekeitio",
			'48055' => "Lemoa",
			'48056' => "Lemoiz",
			'48081' => "Lezama",
			'48903' => "Loiu",
			'48058' => "Mallabia",
			'48059' => "Mañaria",
			'48060' => "Markina-Xemein",
			'48061' => "Maruri-Jatabe",
			'48062' => "Mendata",
			'48063' => "Mendexa",
			'48064' => "Meñaka",
			'48066' => "Morga",
			'48068' => "Mundaka",
			'48069' => "Mungia",
			'48007' => "Munitibar-Arbatzegi Gerrikaitz",
			'48908' => "Murueta",
			'48071' => "Muskiz",
			'48067' => "Muxika",
			'48909' => "Nabarniz",
			'48073' => "Ondarroa",
			'48075' => "Orozko",
			'48083' => "Ortuella",
			'48072' => "Otxandio",
			'48077' => "Plentzia",
			'48078' => "Portugalete",
			'48082' => "Santurtzi",
			'48084' => "Sestao",
			'48904' => "Sondika",
			'48085' => "Sopela",
			'48086' => "Sopuerta",
			'48076' => "Sukarrieta",
			'48087' => "Trucios-Turtzioz",
			'48088' => "Ubide",
			'48065' => "Ugao-Miraballes",
			'48089' => "Urduliz",
			'48074' => "Urduña/Orduña",
			'48916' => "Usansolo",
			'48080' => "Valle de Trápaga-Trapagaran",
			'48095' => "Zaldibar",
			'48096' => "Zalla",
			'48905' => "Zamudio",
			'48097' => "Zaratamo",
			'48024' => "Zeanuri",
			'48025' => "Zeberio",
			'48913' => "Zierbena",
			'48915' => "Ziortza-Bolibar",
			'49002' => "Abezames",
			'49003' => "Alcañices",
			'49004' => "Alcubilla de Nogales",
			'49005' => "Alfaraz de Sayago",
			'49006' => "Algodre",
			'49007' => "Almaraz de Duero",
			'49008' => "Almeida de Sayago",
			'49009' => "Andavías",
			'49010' => "Arcenillas",
			'49011' => "Arcos de la Polvorosa",
			'49012' => "Argañín",
			'49013' => "Argujillo",
			'49014' => "Arquillinos",
			'49015' => "Arrabalde",
			'49016' => "Aspariegos",
			'49017' => "Asturianos",
			'49018' => "Ayoó de Vidriales",
			'49019' => "Barcial del Barco",
			'49020' => "Belver de los Montes",
			'49021' => "Benavente",
			'49022' => "Benegiles",
			'49023' => "Bermillo de Sayago",
			'49024' => "Bóveda de Toro, La",
			'49025' => "Bretó",
			'49026' => "Bretocino",
			'49027' => "Brime de Sog",
			'49028' => "Brime de Urz",
			'49029' => "Burganes de Valverde",
			'49030' => "Bustillo del Oro",
			'49031' => "Cabañas de Sayago",
			'49032' => "Calzadilla de Tera",
			'49033' => "Camarzana de Tera",
			'49034' => "Cañizal",
			'49035' => "Cañizo",
			'49036' => "Carbajales de Alba",
			'49037' => "Carbellino",
			'49038' => "Casaseca de Campeán",
			'49039' => "Casaseca de las Chanas",
			'49040' => "Castrillo de la Guareña",
			'49041' => "Castrogonzalo",
			'49042' => "Castronuevo",
			'49043' => "Castroverde de Campos",
			'49044' => "Cazurra",
			'49046' => "Cerecinos de Campos",
			'49047' => "Cerecinos del Carrizal",
			'49048' => "Cernadilla",
			'49050' => "Cobreros",
			'49052' => "Coomonte",
			'49053' => "Coreses",
			'49054' => "Corrales del Vino",
			'49055' => "Cotanes del Monte",
			'49056' => "Cubillos",
			'49057' => "Cubo de Benavente",
			'49058' => "Cubo de Tierra del Vino, El",
			'49059' => "Cuelgamures",
			'49061' => "Entrala",
			'49062' => "Espadañedo",
			'49063' => "Faramontanos de Tábara",
			'49064' => "Fariza",
			'49065' => "Fermoselle",
			'49066' => "Ferreras de Abajo",
			'49067' => "Ferreras de Arriba",
			'49068' => "Ferreruela",
			'49069' => "Figueruela de Arriba",
			'49071' => "Fonfría",
			'49075' => "Fresno de la Polvorosa",
			'49076' => "Fresno de la Ribera",
			'49077' => "Fresno de Sayago",
			'49078' => "Friera de Valverde",
			'49079' => "Fuente Encalada",
			'49080' => "Fuentelapeña",
			'49082' => "Fuentes de Ropel",
			'49081' => "Fuentesaúco",
			'49083' => "Fuentesecas",
			'49084' => "Fuentespreadas",
			'49085' => "Galende",
			'49086' => "Gallegos del Pan",
			'49087' => "Gallegos del Río",
			'49088' => "Gamones",
			'49090' => "Gema",
			'49091' => "Granja de Moreruela",
			'49092' => "Granucillo",
			'49093' => "Guarrate",
			'49094' => "Hermisende",
			'49095' => "Hiniesta, La",
			'49096' => "Jambrina",
			'49097' => "Justel",
			'49098' => "Losacino",
			'49099' => "Losacio",
			'49100' => "Lubián",
			'49101' => "Luelmo",
			'49102' => "Maderal, El",
			'49103' => "Madridanos",
			'49104' => "Mahide",
			'49105' => "Maire de Castroponce",
			'49107' => "Malva",
			'49108' => "Manganeses de la Lampreana",
			'49109' => "Manganeses de la Polvorosa",
			'49110' => "Manzanal de Arriba",
			'49112' => "Manzanal de los Infantes",
			'49111' => "Manzanal del Barco",
			'49113' => "Matilla de Arzón",
			'49114' => "Matilla la Seca",
			'49115' => "Mayalde",
			'49116' => "Melgar de Tera",
			'49117' => "Micereces de Tera",
			'49118' => "Milles de la Polvorosa",
			'49119' => "Molacillos",
			'49120' => "Molezuelas de la Carballeda",
			'49121' => "Mombuey",
			'49122' => "Monfarracinos",
			'49123' => "Montamarta",
			'49124' => "Moral de Sayago",
			'49126' => "Moraleja de Sayago",
			'49125' => "Moraleja del Vino",
			'49128' => "Morales de Rey",
			'49129' => "Morales de Toro",
			'49130' => "Morales de Valverde",
			'49127' => "Morales del Vino",
			'49131' => "Moralina",
			'49132' => "Moreruela de los Infanzones",
			'49133' => "Moreruela de Tábara",
			'49134' => "Muelas de los Caballeros",
			'49135' => "Muelas del Pan",
			'49136' => "Muga de Sayago",
			'49137' => "Navianos de Valverde",
			'49138' => "Olmillos de Castro",
			'49139' => "Otero de Bodas",
			'49141' => "Pajares de la Lampreana",
			'49143' => "Palacios de Sanabria",
			'49142' => "Palacios del Pan",
			'49145' => "Pedralba de la Pradería",
			'49146' => "Pego, El",
			'49147' => "Peleagonzalo",
			'49148' => "Peleas de Abajo",
			'49149' => "Peñausende",
			'49150' => "Peque",
			'49151' => "Perdigón, El",
			'49152' => "Pereruela",
			'49153' => "Perilla de Castro",
			'49154' => "Pías",
			'49155' => "Piedrahita de Castro",
			'49156' => "Pinilla de Toro",
			'49157' => "Pino del Oro",
			'49158' => "Piñero, El",
			'49160' => "Pobladura de Valderaduey",
			'49159' => "Pobladura del Valle",
			'49162' => "Porto",
			'49163' => "Pozoantiguo",
			'49164' => "Pozuelo de Tábara",
			'49165' => "Prado",
			'49166' => "Puebla de Sanabria",
			'49167' => "Pueblica de Valverde",
			'49170' => "Quintanilla de Urz",
			'49168' => "Quintanilla del Monte",
			'49169' => "Quintanilla del Olmo",
			'49171' => "Quiruelas de Vidriales",
			'49172' => "Rabanales",
			'49173' => "Rábano de Aliste",
			'49174' => "Requejo",
			'49175' => "Revellinos",
			'49176' => "Riofrío de Aliste",
			'49177' => "Rionegro del Puente",
			'49178' => "Roales",
			'49179' => "Robleda-Cervantes",
			'49180' => "Roelos de Sayago",
			'49181' => "Rosinos de la Requejada",
			'49183' => "Salce",
			'49184' => "Samir de los Caños",
			'49185' => "San Agustín del Pozo",
			'49186' => "San Cebrián de Castro",
			'49187' => "San Cristóbal de Entreviñas",
			'49188' => "San Esteban del Molar",
			'49189' => "San Justo",
			'49190' => "San Martín de Valderaduey",
			'49191' => "San Miguel de la Ribera",
			'49192' => "San Miguel del Valle",
			'49193' => "San Pedro de Ceque",
			'49194' => "San Pedro de la Nave-Almendra",
			'49208' => "San Vicente de la Cabeza",
			'49209' => "San Vitero",
			'49197' => "Santa Clara de Avedillo",
			'49199' => "Santa Colomba de las Monjas",
			'49200' => "Santa Cristina de la Polvorosa",
			'49201' => "Santa Croya de Tera",
			'49202' => "Santa Eufemia del Barco",
			'49203' => "Santa María de la Vega",
			'49204' => "Santa María de Valverde",
			'49205' => "Santibáñez de Tera",
			'49206' => "Santibáñez de Vidriales",
			'49207' => "Santovenia",
			'49210' => "Sanzoles",
			'49214' => "Tábara",
			'49216' => "Tapioles",
			'49219' => "Toro",
			'49220' => "Torre del Valle, La",
			'49221' => "Torregamones",
			'49222' => "Torres del Carrizal",
			'49223' => "Trabazos",
			'49224' => "Trefacio",
			'49225' => "Uña de Quintana",
			'49226' => "Vadillo de la Guareña",
			'49227' => "Valcabado",
			'49228' => "Valdefinjas",
			'49229' => "Valdescorriel",
			'49230' => "Vallesa de la Guareña",
			'49231' => "Vega de Tera",
			'49232' => "Vega de Villalobos",
			'49233' => "Vegalatrave",
			'49234' => "Venialbo",
			'49235' => "Vezdemarbán",
			'49236' => "Vidayanes",
			'49237' => "Videmala",
			'49238' => "Villabrázaro",
			'49239' => "Villabuena del Puente",
			'49240' => "Villadepera",
			'49241' => "Villaescusa",
			'49242' => "Villafáfila",
			'49243' => "Villaferrueña",
			'49244' => "Villageriz",
			'49245' => "Villalazán",
			'49246' => "Villalba de la Lampreana",
			'49247' => "Villalcampo",
			'49248' => "Villalobos",
			'49249' => "Villalonso",
			'49250' => "Villalpando",
			'49251' => "Villalube",
			'49252' => "Villamayor de Campos",
			'49255' => "Villamor de los Escuderos",
			'49256' => "Villanázar",
			'49257' => "Villanueva de Azoague",
			'49258' => "Villanueva de Campeán",
			'49259' => "Villanueva de las Peras",
			'49260' => "Villanueva del Campo",
			'49263' => "Villar de Fallaves",
			'49264' => "Villar del Buey",
			'49261' => "Villaralbo",
			'49262' => "Villardeciervos",
			'49265' => "Villardiegua de la Ribera",
			'49266' => "Villárdiga",
			'49267' => "Villardondiego",
			'49268' => "Villarrín de Campos",
			'49269' => "Villaseco del Pan",
			'49270' => "Villavendimio",
			'49272' => "Villaveza de Valverde",
			'49271' => "Villaveza del Agua",
			'49273' => "Viñas",
			'49275' => "Zamora",
			'50001' => "Abanto",
			'50002' => "Acered",
			'50003' => "Agón",
			'50004' => "Aguarón",
			'50005' => "Aguilón",
			'50006' => "Ainzón",
			'50007' => "Aladrén",
			'50008' => "Alagón",
			'50009' => "Alarba",
			'50010' => "Alberite de San Juan",
			'50011' => "Albeta",
			'50012' => "Alborge",
			'50013' => "Alcalá de Ebro",
			'50014' => "Alcalá de Moncayo",
			'50015' => "Alconchel de Ariza",
			'50016' => "Aldehuela de Liestos",
			'50017' => "Alfajarín",
			'50018' => "Alfamén",
			'50019' => "Alforque",
			'50020' => "Alhama de Aragón",
			'50021' => "Almochuel",
			'50022' => "Almolda, La",
			'50023' => "Almonacid de la Cuba",
			'50024' => "Almonacid de la Sierra",
			'50025' => "Almunia de Doña Godina, La",
			'50026' => "Alpartir",
			'50027' => "Ambel",
			'50028' => "Anento",
			'50029' => "Aniñón",
			'50030' => "Añón de Moncayo",
			'50031' => "Aranda de Moncayo",
			'50032' => "Arándiga",
			'50033' => "Ardisa",
			'50034' => "Ariza",
			'50035' => "Artieda",
			'50036' => "Asín",
			'50037' => "Atea",
			'50038' => "Ateca",
			'50039' => "Azuara",
			'50040' => "Badules",
			'50041' => "Bagüés",
			'50042' => "Balconchán",
			'50043' => "Bárboles",
			'50044' => "Bardallur",
			'50045' => "Belchite",
			'50046' => "Belmonte de Gracián",
			'50047' => "Berdejo",
			'50048' => "Berrueco",
			'50901' => "Biel",
			'50050' => "Bijuesca",
			'50051' => "Biota",
			'50052' => "Bisimbre",
			'50053' => "Boquiñeni",
			'50054' => "Bordalba",
			'50055' => "Borja",
			'50056' => "Botorrita",
			'50057' => "Brea de Aragón",
			'50058' => "Bubierca",
			'50059' => "Bujaraloz",
			'50060' => "Bulbuente",
			'50061' => "Bureta",
			'50062' => "Burgo de Ebro, El",
			'50063' => "Buste, El",
			'50064' => "Cabañas de Ebro",
			'50065' => "Cabolafuente",
			'50066' => "Cadrete",
			'50067' => "Calatayud",
			'50068' => "Calatorao",
			'50069' => "Calcena",
			'50070' => "Calmarza",
			'50071' => "Campillo de Aragón",
			'50072' => "Carenas",
			'50073' => "Cariñena",
			'50074' => "Caspe",
			'50075' => "Castejón de Alarba",
			'50076' => "Castejón de las Armas",
			'50077' => "Castejón de Valdejasa",
			'50078' => "Castiliscar",
			'50079' => "Cervera de la Cañada",
			'50080' => "Cerveruela",
			'50081' => "Cetina",
			'50092' => "Chiprana",
			'50093' => "Chodes",
			'50082' => "Cimballa",
			'50083' => "Cinco Olivas",
			'50084' => "Clarés de Ribota",
			'50085' => "Codo",
			'50086' => "Codos",
			'50087' => "Contamina",
			'50088' => "Cosuenda",
			'50089' => "Cuarte de Huerva",
			'50090' => "Cubel",
			'50091' => "Cuerlas, Las",
			'50094' => "Daroca",
			'50095' => "Ejea de los Caballeros",
			'50096' => "Embid de Ariza",
			'50098' => "Encinacorba",
			'50099' => "Épila",
			'50100' => "Erla",
			'50101' => "Escatrón",
			'50102' => "Fabara",
			'50104' => "Farlete",
			'50105' => "Fayón",
			'50106' => "Fayos, Los",
			'50107' => "Figueruelas",
			'50108' => "Fombuena",
			'50109' => "Frago, El",
			'50110' => "Frasno, El",
			'50111' => "Fréscano",
			'50113' => "Fuendejalón",
			'50114' => "Fuendetodos",
			'50115' => "Fuentes de Ebro",
			'50116' => "Fuentes de Jiloca",
			'50117' => "Gallocanta",
			'50118' => "Gallur",
			'50119' => "Gelsa",
			'50120' => "Godojos",
			'50121' => "Gotor",
			'50122' => "Grisel",
			'50123' => "Grisén",
			'50124' => "Herrera de los Navarros",
			'50125' => "Ibdes",
			'50126' => "Illueca",
			'50128' => "Isuerre",
			'50129' => "Jaraba",
			'50130' => "Jarque de Moncayo",
			'50131' => "Jaulín",
			'50132' => "Joyosa, La",
			'50133' => "Lagata",
			'50134' => "Langa del Castillo",
			'50135' => "Layana",
			'50136' => "Lécera",
			'50138' => "Lechón",
			'50137' => "Leciñena",
			'50139' => "Letux",
			'50140' => "Litago",
			'50141' => "Lituénigo",
			'50142' => "Lobera de Onsella",
			'50143' => "Longares",
			'50144' => "Longás",
			'50146' => "Lucena de Jalón",
			'50147' => "Luceni",
			'50148' => "Luesia",
			'50149' => "Luesma",
			'50150' => "Lumpiaque",
			'50151' => "Luna",
			'50152' => "Maella",
			'50153' => "Magallón",
			'50154' => "Mainar",
			'50155' => "Malanquilla",
			'50156' => "Maleján",
			'50160' => "Mallén",
			'50157' => "Malón",
			'50159' => "Maluenda",
			'50161' => "Manchones",
			'50162' => "Mara",
			'50163' => "María de Huerva",
			'50902' => "Marracos",
			'50164' => "Mediana de Aragón",
			'50165' => "Mequinenza",
			'50166' => "Mesones de Isuela",
			'50167' => "Mezalocha",
			'50168' => "Mianos",
			'50169' => "Miedes de Aragón",
			'50170' => "Monegrillo",
			'50171' => "Moneva",
			'50172' => "Monreal de Ariza",
			'50173' => "Monterde",
			'50174' => "Montón",
			'50175' => "Morata de Jalón",
			'50176' => "Morata de Jiloca",
			'50177' => "Morés",
			'50178' => "Moros",
			'50179' => "Moyuela",
			'50180' => "Mozota",
			'50181' => "Muel",
			'50182' => "Muela, La",
			'50183' => "Munébrega",
			'50184' => "Murero",
			'50185' => "Murillo de Gállego",
			'50186' => "Navardún",
			'50187' => "Nigüella",
			'50188' => "Nombrevilla",
			'50189' => "Nonaspe",
			'50190' => "Novallas",
			'50191' => "Novillas",
			'50192' => "Nuévalos",
			'50193' => "Nuez de Ebro",
			'50194' => "Olvés",
			'50195' => "Orcajo",
			'50196' => "Orera",
			'50197' => "Orés",
			'50198' => "Oseja",
			'50199' => "Osera de Ebro",
			'50200' => "Paniza",
			'50201' => "Paracuellos de Jiloca",
			'50202' => "Paracuellos de la Ribera",
			'50203' => "Pastriz",
			'50204' => "Pedrola",
			'50205' => "Pedrosas, Las",
			'50206' => "Perdiguera",
			'50207' => "Piedratajada",
			'50208' => "Pina de Ebro",
			'50209' => "Pinseque",
			'50210' => "Pintanos, Los",
			'50211' => "Plasencia de Jalón",
			'50212' => "Pleitas",
			'50213' => "Plenas",
			'50214' => "Pomer",
			'50215' => "Pozuel de Ariza",
			'50216' => "Pozuelo de Aragón",
			'50217' => "Pradilla de Ebro",
			'50218' => "Puebla de Albortón",
			'50219' => "Puebla de Alfindén, La",
			'50220' => "Puendeluna",
			'50221' => "Purujosa",
			'50222' => "Quinto",
			'50223' => "Remolinos",
			'50224' => "Retascón",
			'50225' => "Ricla",
			'50227' => "Romanos",
			'50228' => "Rueda de Jalón",
			'50229' => "Ruesca",
			'50241' => "Sabiñán",
			'50230' => "Sádaba",
			'50231' => "Salillas de Jalón",
			'50232' => "Salvatierra de Esca",
			'50233' => "Samper del Salz",
			'50234' => "San Martín de la Virgen de Moncayo",
			'50235' => "San Mateo de Gállego",
			'50236' => "Santa Cruz de Grío",
			'50237' => "Santa Cruz de Moncayo",
			'50238' => "Santa Eulalia de Gállego",
			'50239' => "Santed",
			'50240' => "Sástago",
			'50242' => "Sediles",
			'50243' => "Sestrica",
			'50244' => "Sierra de Luna",
			'50245' => "Sigüés",
			'50246' => "Sisamón",
			'50247' => "Sobradiel",
			'50248' => "Sos del Rey Católico",
			'50249' => "Tabuenca",
			'50250' => "Talamantes",
			'50251' => "Tarazona",
			'50252' => "Tauste",
			'50253' => "Terrer",
			'50254' => "Tierga",
			'50255' => "Tobed",
			'50256' => "Torralba de los Frailes",
			'50257' => "Torralba de Ribota",
			'50258' => "Torralbilla",
			'50259' => "Torrehermosa",
			'50260' => "Torrelapaja",
			'50261' => "Torrellas",
			'50262' => "Torres de Berrellén",
			'50263' => "Torrijo de la Cañada",
			'50264' => "Tosos",
			'50265' => "Trasmoz",
			'50266' => "Trasobares",
			'50267' => "Uncastillo",
			'50268' => "Undués de Lerda",
			'50269' => "Urrea de Jalón",
			'50270' => "Urriés",
			'50271' => "Used",
			'50272' => "Utebo",
			'50274' => "Val de San Martín",
			'50273' => "Valdehorna",
			'50275' => "Valmadrid",
			'50276' => "Valpalmas",
			'50277' => "Valtorres",
			'50278' => "Velilla de Ebro",
			'50279' => "Velilla de Jiloca",
			'50280' => "Vera de Moncayo",
			'50281' => "Vierlas",
			'50283' => "Villadoz",
			'50284' => "Villafeliche",
			'50285' => "Villafranca de Ebro",
			'50286' => "Villalba de Perejil",
			'50287' => "Villalengua",
			'50903' => "Villamayor de Gállego",
			'50288' => "Villanueva de Gállego",
			'50290' => "Villanueva de Huerva",
			'50289' => "Villanueva de Jiloca",
			'50291' => "Villar de los Navarros",
			'50292' => "Villarreal de Huerva",
			'50293' => "Villarroya de la Sierra",
			'50294' => "Villarroya del Campo",
			'50282' => "Vilueña, La",
			'50295' => "Vistabella",
			'50296' => "Zaida, La",
			'50297' => "Zaragoza",
			'50298' => "Zuera",
			'51001' => "Ceuta",
			'52001' => "Melilla",
		];
	}

	/**
	 * Helper method that returns a list of "municipio codes" with 6 digits.
	 *
	 * @return 	array
	 * 
	 * @since 	1.18.0 (J) - 1.8.0 (WP)
	 */
	public static function loadMunicipioSixDigitCodes()
	{
		return [
			'010014' => "Alegría-Dulantzi",
			'010029' => "Amurrio",
			'010493' => "Añana",
			'010035' => "Aramaio",
			'010066' => "Armiñón",
			'010376' => "Arraia-Maeztu",
			'010088' => "Arrazua-Ubarrundia",
			'010040' => "Artziniega",
			'010091' => "Asparrena",
			'010105' => "Ayala/Aiara",
			'010112' => "Baños de Ebro/Mañueta",
			'010133' => "Barrundia",
			'010148' => "Berantevilla",
			'010164' => "Bernedo",
			'010170' => "Campezo/Kanpezu",
			'010210' => "Elburgo/Burgelu",
			'010225' => "Elciego",
			'010231' => "Elvillar/Bilar",
			'010468' => "Erriberagoitia/Ribera Alta",
			'010565' => "Harana/Valle de Arana",
			'019015' => "Iruña Oka/Iruña de Oca",
			'010278' => "Iruraiz-Gauna",
			'010199' => "Kripan",
			'010203' => "Kuartango",
			'010284' => "Labastida/Bastida",
			'010301' => "Lagrán",
			'010318' => "Laguardia",
			'010323' => "Lanciego/Lantziego",
			'019020' => "Lantarón",
			'010339' => "Lapuebla de Labarca",
			'010360' => "Laudio/Llodio",
			'010587' => "Legutiano",
			'010344' => "Leza",
			'010395' => "Moreda de Álava",
			'010416' => "Navaridas",
			'010421' => "Okondo",
			'010437' => "Oyón-Oion",
			'010442' => "Peñacerrada-Urizaharra",
			'010474' => "Ribera Baja/Erribera Beitia",
			'010513' => "Salvatierra/Agurain",
			'010528' => "Samaniego",
			'010534' => "San Millán/Donemiliaga",
			'010549' => "Urkabustaiz",
			'010552' => "Valdegovía/Gaubea",
			'010571' => "Villabuena de Álava/Eskuernaga",
			'010590' => "Vitoria-Gasteiz",
			'010604' => "Yécora/Iekora",
			'010611' => "Zalduondo",
			'010626' => "Zambrana",
			'010186' => "Zigoitia",
			'010632' => "Zuia",
			'020019' => "Abengibre",
			'020024' => "Alatoz",
			'020030' => "Albacete",
			'020045' => "Albatana",
			'020058' => "Alborea",
			'020061' => "Alcadozo",
			'020077' => "Alcalá del Júcar",
			'020083' => "Alcaraz",
			'020096' => "Almansa",
			'020100' => "Alpera",
			'020117' => "Ayna",
			'020122' => "Balazote",
			'020143' => "Ballestero (El)",
			'020138' => "Balsa de Ves",
			'020156' => "Barrax",
			'020169' => "Bienservida",
			'020175' => "Bogarra",
			'020181' => "Bonete",
			'020194' => "Bonillo (El)",
			'020208' => "Carcelén",
			'020215' => "Casas de Juan Núñez",
			'020220' => "Casas de Lázaro",
			'020236' => "Casas de Ves",
			'020241' => "Casas-Ibáñez",
			'020254' => "Caudete",
			'020267' => "Cenizate",
			'020292' => "Chinchilla de Monte-Aragón",
			'020273' => "Corral-Rubio",
			'020289' => "Cotillas",
			'020306' => "Elche de la Sierra",
			'020313' => "Férez",
			'020328' => "Fuensanta",
			'020334' => "Fuente-Álamo",
			'020349' => "Fuentealbilla",
			'020352' => "Gineta (La)",
			'020365' => "Golosalvo",
			'020371' => "Hellín",
			'020387' => "Herrera (La)",
			'020390' => "Higueruela",
			'020404' => "Hoya-Gonzalo",
			'020411' => "Jorquera",
			'020426' => "Letur",
			'020432' => "Lezuza",
			'020447' => "Liétor",
			'020450' => "Madrigueras",
			'020463' => "Mahora",
			'020479' => "Masegoso",
			'020485' => "Minaya",
			'020498' => "Molinicos",
			'020501' => "Montalvos",
			'020518' => "Montealegre del Castillo",
			'020523' => "Motilleja",
			'020539' => "Munera",
			'020544' => "Navas de Jorquera",
			'020557' => "Nerpio",
			'020560' => "Ontur",
			'020576' => "Ossa de Montiel",
			'020582' => "Paterna del Madera",
			'020609' => "Peñas de San Pedro",
			'020595' => "Peñascosa",
			'020616' => "Pétrola",
			'020621' => "Povedilla",
			'029010' => "Pozo Cañada",
			'020637' => "Pozohondo",
			'020642' => "Pozo-Lorente",
			'020655' => "Pozuelo",
			'020668' => "Recueja (La)",
			'020674' => "Riópar",
			'020680' => "Robledo",
			'020693' => "Roda (La)",
			'020707' => "Salobre",
			'020714' => "San Pedro",
			'020729' => "Socovos",
			'020735' => "Tarazona de la Mancha",
			'020740' => "Tobarra",
			'020753' => "Valdeganga",
			'020766' => "Vianos",
			'020772' => "Villa de Ves",
			'020788' => "Villalgordo del Júcar",
			'020791' => "Villamalea",
			'020805' => "Villapalacios",
			'020812' => "Villarrobledo",
			'020827' => "Villatoya",
			'020833' => "Villavaliente",
			'020848' => "Villaverde de Guadalimar",
			'020851' => "Viveros",
			'020864' => "Yeste",
			'030015' => "Adsubia",
			'030020' => "Agost",
			'030036' => "Agres",
			'030041' => "Aigües",
			'030054' => "Albatera",
			'030067' => "Alcalalí",
			'030073' => "Alcocer de Planes",
			'030089' => "Alcoleja",
			'030092' => "Alcoy/Alcoi",
			'030106' => "Alfafara",
			'030113' => "Alfàs del Pi (l')",
			'030128' => "Algorfa",
			'030134' => "Algueña",
			'030149' => "Alicante/Alacant",
			'030152' => "Almoradí",
			'030165' => "Almudaina",
			'030171' => "Alqueria d'Asnar (l')",
			'030187' => "Altea",
			'030190' => "Aspe",
			'030204' => "Balones",
			'030211' => "Banyeres de Mariola",
			'030226' => "Benasau",
			'030232' => "Beneixama",
			'030247' => "Benejúzar",
			'030250' => "Benferri",
			'030263' => "Beniarbeig",
			'030279' => "Beniardá",
			'030285' => "Beniarrés",
			'030302' => "Benidoleig",
			'030319' => "Benidorm",
			'030324' => "Benifallim",
			'030330' => "Benifato",
			'030298' => "Benigembla",
			'030345' => "Benijófar",
			'030358' => "Benilloba",
			'030361' => "Benillup",
			'030377' => "Benimantell",
			'030383' => "Benimarfull",
			'030396' => "Benimassot",
			'030400' => "Benimeli",
			'030417' => "Benissa",
			'030422' => "Benitachell/Poble Nou de Benitatxell (el)",
			'030438' => "Biar",
			'030443' => "Bigastro",
			'030456' => "Bolulla",
			'030469' => "Busot",
			'030494' => "Callosa de Segura",
			'030481' => "Callosa d'En Sarrià",
			'030475' => "Calp",
			'030507' => "Campello (el)",
			'030514' => "Campo de Mirra/Camp de Mirra (el)",
			'030529' => "Cañada",
			'030535' => "Castalla",
			'030540' => "Castell de Castells",
			'030759' => "Castell de Guadalest (el)",
			'030553' => "Catral",
			'030566' => "Cocentaina",
			'030572' => "Confrides",
			'030588' => "Cox",
			'030591' => "Crevillent",
			'030612' => "Daya Nueva",
			'030627' => "Daya Vieja",
			'030633' => "Dénia",
			'030648' => "Dolores",
			'030651' => "Elche/Elx",
			'030664' => "Elda",
			'030670' => "Facheca",
			'030686' => "Famorca",
			'030699' => "Finestrat",
			'030778' => "Fondó de les Neus (el)/Hondón de las Nieves",
			'030703' => "Formentera del Segura",
			'030725' => "Gaianes",
			'030710' => "Gata de Gorgos",
			'030731' => "Gorga",
			'030746' => "Granja de Rocamora",
			'030762' => "Guardamar del Segura",
			'030784' => "Hondón de los Frailes",
			'030797' => "Ibi",
			'030801' => "Jacarilla",
			'030823' => "Jávea/Xàbia",
			'030839' => "Jijona/Xixona",
			'030857' => "Llíber",
			'030844' => "Lorcha/Orxa (l')",
			'030860' => "Millena",
			'030882' => "Monforte del Cid",
			'030895' => "Monóvar/Monòver",
			'039037' => "Montesinos (Los)",
			'030916' => "Murla",
			'030921' => "Muro de Alcoy",
			'030909' => "Mutxamel",
			'030937' => "Novelda",
			'030942' => "Nucia (la)",
			'030955' => "Ondara",
			'030968' => "Onil",
			'030974' => "Orba",
			'030993' => "Orihuela",
			'030980' => "Orxeta",
			'031007' => "Parcent",
			'031014' => "Pedreguer",
			'031029' => "Pego",
			'031035' => "Penàguila",
			'031040' => "Petrer",
			'039021' => "Pilar de la Horadada",
			'031053' => "Pinós (el)/Pinoso",
			'031066' => "Planes",
			'039016' => "Poblets (els)",
			'031072' => "Polop",
			'030605' => "Quatretondeta",
			'031091' => "Rafal",
			'031105' => "Ràfol d'Almúnia, el",
			'031112' => "Redován",
			'031127' => "Relleu",
			'031133' => "Rojales",
			'031148' => "Romana (la)",
			'031151' => "Sagra",
			'031164' => "Salinas",
			'031186' => "San Fulgencio",
			'039042' => "San Isidro",
			'031203' => "San Miguel de Salinas",
			'031225' => "San Vicente del Raspeig/Sant Vicent del Raspeig",
			'031170' => "Sanet y Negrals",
			'031199' => "Sant Joan d'Alacant",
			'031210' => "Santa Pola",
			'031231' => "Sax",
			'031246' => "Sella",
			'031259' => "Senija",
			'031278' => "Tàrbena",
			'031284' => "Teulada",
			'031297' => "Tibi",
			'031301' => "Tollos",
			'031318' => "Tormos",
			'031323' => "Torremanzanas/Torre de les Maçanes (la)",
			'031339' => "Torrevieja",
			'031344' => "Vall d'Alcalà (la)",
			'031360' => "Vall de Gallinera",
			'031376' => "Vall de Laguar (la)",
			'031357' => "Vall d'Ebo (la)",
			'031382' => "Verger (el)",
			'031395' => "Villajoyosa/Vila Joiosa (la)",
			'031409' => "Villena",
			'030818' => "Xaló",
			'040010' => "Abla",
			'040025' => "Abrucena",
			'040031' => "Adra",
			'040046' => "Albánchez",
			'040059' => "Alboloduy",
			'040062' => "Albox",
			'040078' => "Alcolea",
			'040084' => "Alcóntar",
			'040097' => "Alcudia de Monteagud",
			'040101' => "Alhabia",
			'040118' => "Alhama de Almería",
			'040123' => "Alicún",
			'040139' => "Almería",
			'040144' => "Almócita",
			'040157' => "Alsodux",
			'040160' => "Antas",
			'040176' => "Arboleas",
			'040182' => "Armuña de Almanzora",
			'040195' => "Bacares",
			'040209' => "Bayárcal",
			'040216' => "Bayarque",
			'040221' => "Bédar",
			'040237' => "Beires",
			'040242' => "Benahadux",
			'040268' => "Benitagla",
			'040274' => "Benizalón",
			'040280' => "Bentarique",
			'040293' => "Berja",
			'040307' => "Canjáyar",
			'040314' => "Cantoria",
			'040329' => "Carboneras",
			'040335' => "Castro de Filabres",
			'040366' => "Chercos",
			'040372' => "Chirivel",
			'040340' => "Cóbdar",
			'040353' => "Cuevas del Almanzora",
			'040388' => "Dalías",
			'049026' => "Ejido (El)",
			'040412' => "Enix",
			'040433' => "Felix",
			'040448' => "Fines",
			'040451' => "Fiñana",
			'040464' => "Fondón",
			'040470' => "Gádor",
			'040486' => "Gallardos (Los)",
			'040499' => "Garrucha",
			'040502' => "Gérgal",
			'040519' => "Huécija",
			'040524' => "Huércal de Almería",
			'040530' => "Huércal-Overa",
			'040545' => "Illar",
			'040558' => "Instinción",
			'040561' => "Laroya",
			'040577' => "Láujar de Andarax",
			'040583' => "Líjar",
			'040596' => "Lubrín",
			'040600' => "Lucainena de las Torres",
			'040617' => "Lúcar",
			'040622' => "Macael",
			'040638' => "María",
			'040643' => "Mojácar",
			'049032' => "Mojonera (La)",
			'040656' => "Nacimiento",
			'040669' => "Níjar",
			'040675' => "Ohanes",
			'040681' => "Olula de Castro",
			'040694' => "Olula del Río",
			'040708' => "Oria",
			'040715' => "Padules",
			'040720' => "Partaloa",
			'040736' => "Paterna del Río",
			'040741' => "Pechina",
			'040754' => "Pulpí",
			'040767' => "Purchena",
			'040773' => "Rágol",
			'040789' => "Rioja",
			'040792' => "Roquetas de Mar",
			'040806' => "Santa Cruz de Marchena",
			'040813' => "Santa Fe de Mondújar",
			'040828' => "Senés",
			'040834' => "Serón",
			'040849' => "Sierro",
			'040852' => "Somontín",
			'040865' => "Sorbas",
			'040871' => "Suflí",
			'040887' => "Tabernas",
			'040890' => "Taberno",
			'040904' => "Tahal",
			'040911' => "Terque",
			'040926' => "Tíjola",
			'049011' => "Tres Villas (Las)",
			'040932' => "Turre",
			'040947' => "Turrillas",
			'040950' => "Uleila del Campo",
			'040963' => "Urrácal",
			'040979' => "Velefique",
			'040985' => "Vélez-Blanco",
			'040998' => "Vélez-Rubio",
			'041002' => "Vera",
			'041019' => "Viator",
			'041024' => "Vícar",
			'041030' => "Zurgena",
			'050013' => "Adanero",
			'050028' => "Adrada (La)",
			'050052' => "Albornos",
			'050071' => "Aldeanueva de Santa Cruz",
			'050087' => "Aldeaseca",
			'050104' => "Aldehuela (La)",
			'050126' => "Amavida",
			'050132' => "Arenal (El)",
			'050147' => "Arenas de San Pedro",
			'050150' => "Arevalillo",
			'050163' => "Arévalo",
			'050179' => "Aveinte",
			'050185' => "Avellaneda",
			'050198' => "Ávila",
			'050219' => "Barco de Ávila (El)",
			'050224' => "Barraco (El)",
			'050230' => "Barromán",
			'050245' => "Becedas",
			'050258' => "Becedillas",
			'050261' => "Bercial de Zapardiel",
			'050277' => "Berlanas (Las)",
			'050296' => "Bernuy-Zapardiel",
			'050300' => "Berrocalejo de Aragona",
			'050338' => "Blascomillán",
			'050343' => "Blasconuño de Matacabras",
			'050356' => "Blascosancho",
			'050369' => "Bohodón (El)",
			'050375' => "Bohoyo",
			'050381' => "Bonilla de la Sierra",
			'050394' => "Brabos",
			'050408' => "Bularros",
			'050415' => "Burgohondo",
			'050420' => "Cabezas de Alambre",
			'050436' => "Cabezas del Pozo",
			'050441' => "Cabezas del Villar",
			'050454' => "Cabizuela",
			'050467' => "Canales",
			'050473' => "Candeleda",
			'050489' => "Cantiveros",
			'050492' => "Cardeñosa",
			'050512' => "Carrera (La)",
			'050527' => "Casas del Puerto",
			'050533' => "Casasola",
			'050548' => "Casavieja",
			'050551' => "Casillas",
			'050564' => "Castellanos de Zapardiel",
			'050570' => "Cebreros",
			'050586' => "Cepeda la Mora",
			'050678' => "Chamartín",
			'050599' => "Cillán",
			'050603' => "Cisla",
			'050610' => "Colilla (La)",
			'050625' => "Collado de Contreras",
			'050631' => "Collado del Mirón",
			'050646' => "Constanzana",
			'050659' => "Crespos",
			'050662' => "Cuevas del Valle",
			'059035' => "Diego del Carpio",
			'050697' => "Donjimeno",
			'050701' => "Donvidas",
			'050723' => "Espinosa de los Caballeros",
			'050739' => "Flores de Ávila",
			'050744' => "Fontiveros",
			'050757' => "Fresnedilla",
			'050760' => "Fresno (El)",
			'050776' => "Fuente el Saúz",
			'050782' => "Fuentes de Año",
			'050795' => "Gallegos de Altamiros",
			'050809' => "Gallegos de Sobrinos",
			'050816' => "Garganta del Villar",
			'050821' => "Gavilanes",
			'050837' => "Gemuño",
			'050855' => "Gil García",
			'050842' => "Gilbuena",
			'050868' => "Gimialcón",
			'050874' => "Gotarrendura",
			'050880' => "Grandes y San Martín",
			'050893' => "Guisando",
			'050907' => "Gutierre-Muñoz",
			'050929' => "Hernansancho",
			'050935' => "Herradón de Pinares",
			'050940' => "Herreros de Suso",
			'050953' => "Higuera de las Dueñas",
			'050966' => "Hija de Dios (La)",
			'050972' => "Horcajada (La)",
			'050991' => "Horcajo de las Torres",
			'051005' => "Hornillo (El)",
			'051027' => "Hoyo de Pinares (El)",
			'051012' => "Hoyocasero",
			'051033' => "Hoyorredondo",
			'051064' => "Hoyos de Miguel Muñoz",
			'051048' => "Hoyos del Collado",
			'051051' => "Hoyos del Espino",
			'051070' => "Hurtumpascual",
			'051086' => "Junciana",
			'051099' => "Langa",
			'051103' => "Lanzahíta",
			'051131' => "Llanos de Tormes (Los)",
			'051125' => "Losar del Barco (El)",
			'051146' => "Madrigal de las Altas Torres",
			'051159' => "Maello",
			'051162' => "Malpartida de Corneja",
			'051178' => "Mamblas",
			'051184' => "Mancera de Arriba",
			'051197' => "Manjabálago",
			'051201' => "Marlín",
			'051218' => "Martiherrero",
			'051223' => "Martínez",
			'051239' => "Mediana de Voltoya",
			'051244' => "Medinilla",
			'051257' => "Mengamuñoz",
			'051260' => "Mesegar de Corneja",
			'051276' => "Mijares",
			'051282' => "Mingorría",
			'051295' => "Mirón (El)",
			'051309' => "Mironcillo",
			'051316' => "Mirueña de los Infanzones",
			'051321' => "Mombeltrán",
			'051337' => "Monsalupe",
			'051342' => "Moraleja de Matacabras",
			'051355' => "Muñana",
			'051368' => "Muñico",
			'051380' => "Muñogalindo",
			'051393' => "Muñogrande",
			'051407' => "Muñomer del Peco",
			'051414' => "Muñopepe",
			'051429' => "Muñosancho",
			'051435' => "Muñotello",
			'051440' => "Narrillos del Álamo",
			'051453' => "Narrillos del Rebollar",
			'051491' => "Narros de Saldueña",
			'051472' => "Narros del Castillo",
			'051488' => "Narros del Puerto",
			'051526' => "Nava de Arévalo",
			'051532' => "Nava del Barco",
			'051511' => "Navacepedilla de Corneja",
			'051547' => "Navadijos",
			'051550' => "Navaescurial",
			'051563' => "Navahondilla",
			'051579' => "Navalacruz",
			'051585' => "Navalmoral",
			'051598' => "Navalonguilla",
			'051602' => "Navalosa",
			'051619' => "Navalperal de Pinares",
			'051624' => "Navalperal de Tormes",
			'051630' => "Navaluenga",
			'051645' => "Navaquesera",
			'051658' => "Navarredonda de Gredos",
			'051661' => "Navarredondilla",
			'051677' => "Navarrevisca",
			'051683' => "Navas del Marqués (Las)",
			'051696' => "Navatalgordo",
			'051700' => "Navatejares",
			'051717' => "Neila de San Miguel",
			'051722' => "Niharra",
			'051738' => "Ojos-Albos",
			'051743' => "Orbita",
			'051756' => "Oso (El)",
			'051769' => "Padiernos",
			'051775' => "Pajares de Adaja",
			'051781' => "Palacios de Goda",
			'051794' => "Papatrigo",
			'051808' => "Parral (El)",
			'051815' => "Pascualcobo",
			'051820' => "Pedro Bernardo",
			'051836' => "Pedro-Rodríguez",
			'051841' => "Peguerinos",
			'051854' => "Peñalba de Ávila",
			'051867' => "Piedrahíta",
			'051873' => "Piedralaves",
			'051889' => "Poveda",
			'051892' => "Poyales del Hoyo",
			'051906' => "Pozanco",
			'051913' => "Pradosegar",
			'051928' => "Puerto Castilla",
			'051934' => "Rasueros",
			'051949' => "Riocabado",
			'051952' => "Riofrío",
			'051965' => "Rivilla de Barajas",
			'051971' => "Salobral",
			'051987' => "Salvadiós",
			'051990' => "San Bartolomé de Béjar",
			'052004' => "San Bartolomé de Corneja",
			'052011' => "San Bartolomé de Pinares",
			'052063' => "San Esteban de los Patos",
			'052085' => "San Esteban de Zapardiel",
			'052079' => "San Esteban del Valle",
			'052098' => "San García de Ingelmos",
			'059014' => "San Juan de Gredos",
			'052102' => "San Juan de la Encinilla",
			'052119' => "San Juan de la Nava",
			'052124' => "San Juan del Molinillo",
			'052130' => "San Juan del Olmo",
			'052145' => "San Lorenzo de Tormes",
			'052158' => "San Martín de la Vega del Alberche",
			'052161' => "San Martín del Pimpollar",
			'052177' => "San Miguel de Corneja",
			'052183' => "San Miguel de Serrezuela",
			'052196' => "San Pascual",
			'052200' => "San Pedro del Arroyo",
			'052315' => "San Vicente de Arévalo",
			'052047' => "Sanchidrián",
			'052050' => "Sanchorreja",
			'052222' => "Santa Cruz de Pinares",
			'052217' => "Santa Cruz del Valle",
			'052269' => "Santa María de los Caballeros",
			'052243' => "Santa María del Arroyo",
			'052256' => "Santa María del Berrocal",
			'059029' => "Santa María del Cubillo",
			'052275' => "Santa María del Tiétar",
			'052281' => "Santiago del Collado",
			'059040' => "Santiago del Tormes",
			'052294' => "Santo Domingo de las Posadas",
			'052308' => "Santo Tomé de Zabarcos",
			'052320' => "Serrada (La)",
			'052336' => "Serranillos",
			'052341' => "Sigeres",
			'052354' => "Sinlabajos",
			'052367' => "Solana de Ávila",
			'052373' => "Solana de Rioalmar",
			'052389' => "Solosancho",
			'052392' => "Sotalbo",
			'052406' => "Sotillo de la Adrada",
			'052413' => "Tiemblo (El)",
			'052428' => "Tiñosillos",
			'052434' => "Tolbaños",
			'052449' => "Tormellas",
			'052452' => "Tornadizos de Ávila",
			'052471' => "Torre (La)",
			'052465' => "Tórtoles",
			'052490' => "Umbrías",
			'052510' => "Vadillo de la Sierra",
			'052525' => "Valdecasa",
			'052531' => "Vega de Santa María",
			'052546' => "Velayos",
			'052562' => "Villaflor",
			'052578' => "Villafranca de la Sierra",
			'059053' => "Villanueva de Ávila",
			'052584' => "Villanueva de Gómez",
			'052597' => "Villanueva del Aceral",
			'052601' => "Villanueva del Campillo",
			'052618' => "Villar de Corneja",
			'052623' => "Villarejo del Valle",
			'052639' => "Villatoro",
			'052644' => "Viñegra de Moraña",
			'052657' => "Vita",
			'052660' => "Zapardiel de la Cañada",
			'052676' => "Zapardiel de la Ribera",
			'060016' => "Acedera",
			'060021' => "Aceuchal",
			'060037' => "Ahillones",
			'060042' => "Alange",
			'060055' => "Albuera (La)",
			'060068' => "Alburquerque",
			'060074' => "Alconchel",
			'060080' => "Alconera",
			'060093' => "Aljucén",
			'060107' => "Almendral",
			'060114' => "Almendralejo",
			'060129' => "Arroyo de San Serván",
			'060135' => "Atalaya",
			'060140' => "Azuaga",
			'060153' => "Badajoz",
			'060166' => "Barcarrota",
			'060172' => "Baterno",
			'060188' => "Benquerencia de la Serena",
			'060191' => "Berlanga",
			'060205' => "Bienvenida",
			'060212' => "Bodonal de la Sierra",
			'060227' => "Burguillos del Cerro",
			'060233' => "Cabeza del Buey",
			'060248' => "Cabeza la Vaca",
			'060251' => "Calamonte",
			'060264' => "Calera de León",
			'060270' => "Calzadilla de los Barros",
			'060286' => "Campanario",
			'060299' => "Campillo de Llerena",
			'060303' => "Capilla",
			'060310' => "Carmonita",
			'060325' => "Carrascalejo (El)",
			'060331' => "Casas de Don Pedro",
			'060346' => "Casas de Reina",
			'060359' => "Castilblanco",
			'060362' => "Castuera",
			'060423' => "Cheles",
			'060378' => "Codosera (La)",
			'060384' => "Cordobilla de Lácara",
			'060397' => "Coronada (La)",
			'060401' => "Corte de Peleas",
			'060418' => "Cristina",
			'060439' => "Don Álvaro",
			'060444' => "Don Benito",
			'060457' => "Entrín Bajo",
			'060460' => "Esparragalejo",
			'060476' => "Esparragosa de la Serena",
			'060482' => "Esparragosa de Lares",
			'060495' => "Feria",
			'060508' => "Fregenal de la Sierra",
			'060515' => "Fuenlabrada de los Montes",
			'060520' => "Fuente de Cantos",
			'060536' => "Fuente del Arco",
			'060541' => "Fuente del Maestre",
			'060554' => "Fuentes de León",
			'060567' => "Garbayuela",
			'060573' => "Garlitos",
			'060589' => "Garrovilla (La)",
			'060592' => "Granja de Torrehermosa",
			'060606' => "Guareña",
			'060613' => "Haba (La)",
			'060628' => "Helechosa de los Montes",
			'060634' => "Herrera del Duque",
			'060649' => "Higuera de la Serena",
			'060652' => "Higuera de Llerena",
			'060665' => "Higuera de Vargas",
			'060671' => "Higuera la Real",
			'060687' => "Hinojosa del Valle",
			'060690' => "Hornachos",
			'060704' => "Jerez de los Caballeros",
			'060711' => "Lapa (La)",
			'060732' => "Llera",
			'060747' => "Llerena",
			'060726' => "Lobón",
			'060750' => "Magacela",
			'060763' => "Maguilla",
			'060779' => "Malcocinado",
			'060785' => "Malpartida de la Serena",
			'060798' => "Manchita",
			'060802' => "Medellín",
			'060819' => "Medina de las Torres",
			'060824' => "Mengabril",
			'060830' => "Mérida",
			'060845' => "Mirandilla",
			'060858' => "Monesterio",
			'060861' => "Montemolín",
			'060877' => "Monterrubio de la Serena",
			'060883' => "Montijo",
			'060896' => "Morera (La)",
			'060900' => "Nava de Santiago (La)",
			'060917' => "Navalvillar de Pela",
			'060922' => "Nogales",
			'060938' => "Oliva de la Frontera",
			'060943' => "Oliva de Mérida",
			'060956' => "Olivenza",
			'060969' => "Orellana de la Sierra",
			'060975' => "Orellana la Vieja",
			'060981' => "Palomas",
			'060994' => "Parra (La)",
			'061008' => "Peñalsordo",
			'061015' => "Peraleda del Zaucejo",
			'061020' => "Puebla de Alcocer",
			'061036' => "Puebla de la Calzada",
			'061041' => "Puebla de la Reina",
			'061073' => "Puebla de Obando",
			'061089' => "Puebla de Sancho Pérez",
			'061054' => "Puebla del Maestre",
			'061067' => "Puebla del Prior",
			'069022' => "Pueblonuevo del Guadiana",
			'061092' => "Quintana de la Serena",
			'061106' => "Reina",
			'061113' => "Rena",
			'061128' => "Retamal de Llerena",
			'061134' => "Ribera del Fresno",
			'061149' => "Risco",
			'061152' => "Roca de la Sierra (La)",
			'061165' => "Salvaleón",
			'061171' => "Salvatierra de los Barros",
			'061190' => "San Pedro de Mérida",
			'061232' => "San Vicente de Alcántara",
			'061187' => "Sancti-Spíritus",
			'061204' => "Santa Amalia",
			'061211' => "Santa Marta",
			'061226' => "Santos de Maimona (Los)",
			'061247' => "Segura de León",
			'061250' => "Siruela",
			'061263' => "Solana de los Barros",
			'061279' => "Talarrubias",
			'061285' => "Talavera la Real",
			'061298' => "Táliga",
			'061302' => "Tamurejo",
			'061319' => "Torre de Miguel Sesmero",
			'061324' => "Torremayor",
			'061330' => "Torremejía",
			'061345' => "Trasierra",
			'061358' => "Trujillanos",
			'061361' => "Usagre",
			'061377' => "Valdecaballeros",
			'069017' => "Valdelacalzada",
			'061383' => "Valdetorres",
			'061396' => "Valencia de las Torres",
			'061400' => "Valencia del Mombuey",
			'061417' => "Valencia del Ventoso",
			'061469' => "Valle de la Serena",
			'061475' => "Valle de Matamoros",
			'061481' => "Valle de Santa Ana",
			'061422' => "Valverde de Burguillos",
			'061438' => "Valverde de Leganés",
			'061443' => "Valverde de Llerena",
			'061456' => "Valverde de Mérida",
			'061494' => "Villafranca de los Barros",
			'061507' => "Villagarcía de la Torre",
			'061514' => "Villagonzalo",
			'061529' => "Villalba de los Barros",
			'061535' => "Villanueva de la Serena",
			'061540' => "Villanueva del Fresno",
			'061566' => "Villar de Rena",
			'061553' => "Villar del Rey",
			'061572' => "Villarta de los Montes",
			'061588' => "Zafra",
			'061591' => "Zahínos",
			'061605' => "Zalamea de la Serena",
			'061627' => "Zarza (La)",
			'061612' => "Zarza-Capilla",
			'070027' => "Alaior",
			'070012' => "Alaró",
			'070033' => "Alcúdia",
			'070048' => "Algaida",
			'070051' => "Andratx",
			'079013' => "Ariany",
			'070064' => "Artà",
			'070070' => "Banyalbufar",
			'070086' => "Binissalem",
			'070099' => "Búger",
			'070103' => "Bunyola",
			'070110' => "Calvià",
			'070125' => "Campanet",
			'070131' => "Campos",
			'070146' => "Capdepera",
			'070645' => "Castell (Es)",
			'070159' => "Ciutadella de Menorca",
			'070162' => "Consell",
			'070178' => "Costitx",
			'070184' => "Deyá",
			'070260' => "Eivissa",
			'070197' => "Escorca",
			'070201' => "Esporles",
			'070218' => "Estellencs",
			'070223' => "Felanitx",
			'070239' => "Ferreries",
			'070244' => "Formentera",
			'070257' => "Fornalutx",
			'070276' => "Inca",
			'070282' => "Lloret de Vistalegre",
			'070295' => "Lloseta",
			'070309' => "Llubí",
			'070316' => "Llucmajor",
			'070337' => "Manacor",
			'070342' => "Mancor de la Vall",
			'070321' => "Maó",
			'070355' => "Maria de la Salut",
			'070368' => "Marratxí",
			'070374' => "Mercadal (Es)",
			'079028' => "Migjorn Gran (Es)",
			'070380' => "Montuïri",
			'070393' => "Muro",
			'070407' => "Palma",
			'070414' => "Petra",
			'070440' => "Pobla (Sa)",
			'070429' => "Pollença",
			'070435' => "Porreres",
			'070453' => "Puigpunyent",
			'070598' => "Salines (Ses)",
			'070466' => "Sant Antoni de Portmany",
			'070491' => "Sant Joan",
			'070504' => "Sant Joan de Labritja",
			'070488' => "Sant Josep de sa Talaia",
			'070511' => "Sant Llorenç des Cardassar",
			'070526' => "Sant Lluís",
			'070532' => "Santa Eugènia",
			'070547' => "Santa Eulalia del Río",
			'070550' => "Santa Margalida",
			'070563' => "Santa María del Camí",
			'070579' => "Santanyí",
			'070585' => "Selva",
			'070472' => "Sencelles",
			'070602' => "Sineu",
			'070619' => "Sóller",
			'070624' => "Son Servera",
			'070630' => "Valldemossa",
			'070658' => "Vilafranca de Bonany",
			'080018' => "Abrera",
			'080023' => "Aguilar de Segarra",
			'080142' => "Aiguafreda",
			'080039' => "Alella",
			'080044' => "Alpens",
			'080057' => "Ametlla del Vallès (L')",
			'080060' => "Arenys de Mar",
			'080076' => "Arenys de Munt",
			'080082' => "Argençola",
			'080095' => "Argentona",
			'080109' => "Artés",
			'080116' => "Avià",
			'080121' => "Avinyó",
			'080137' => "Avinyonet del Penedès",
			'080155' => "Badalona",
			'089045' => "Badia del Vallès",
			'080168' => "Bagà",
			'080174' => "Balenyà",
			'080180' => "Balsareny",
			'082520' => "Barberà del Vallès",
			'080193' => "Barcelona",
			'080207' => "Begues",
			'080214' => "Bellprat",
			'080229' => "Berga",
			'080235' => "Bigues i Riells",
			'080240' => "Borredà",
			'080253' => "Bruc (El)",
			'080266' => "Brull (El)",
			'080272' => "Cabanyes (Les)",
			'080288' => "Cabrera d'Anoia",
			'080291' => "Cabrera de Mar",
			'080305' => "Cabrils",
			'080312' => "Calaf",
			'080348' => "Calders",
			'080333' => "Caldes de Montbui",
			'080327' => "Caldes d'Estrac",
			'080351' => "Calella",
			'080370' => "Calldetenes",
			'080386' => "Callús",
			'080364' => "Calonge de Segarra",
			'080399' => "Campins",
			'080403' => "Canet de Mar",
			'080410' => "Canovelles",
			'080425' => "Cànoves i Samalús",
			'080431' => "Canyelles",
			'080446' => "Capellades",
			'080459' => "Capolat",
			'080462' => "Cardedeu",
			'080478' => "Cardona",
			'080484' => "Carme",
			'080497' => "Casserres",
			'080575' => "Castell de l'Areny",
			'080522' => "Castellar de n'Hug",
			'080500' => "Castellar del Riu",
			'080517' => "Castellar del Vallès",
			'080538' => "Castellbell i el Vilar",
			'080543' => "Castellbisbal",
			'080556' => "Castellcir",
			'080569' => "Castelldefels",
			'080581' => "Castellet i la Gornal",
			'080608' => "Castellfollit de Riubregós",
			'080594' => "Castellfollit del Boix",
			'080615' => "Castellgalí",
			'080620' => "Castellnou de Bages",
			'080636' => "Castellolí",
			'080641' => "Castellterçol",
			'080654' => "Castellví de la Marca",
			'080667' => "Castellví de Rosanes",
			'080673' => "Centelles",
			'082687' => "Cercs",
			'082665' => "Cerdanyola del Vallès",
			'080689' => "Cervelló",
			'080692' => "Collbató",
			'080706' => "Collsuspina",
			'080713' => "Copons",
			'080728' => "Corbera de Llobregat",
			'080734' => "Cornellà de Llobregat",
			'080749' => "Cubelles",
			'080752' => "Dosrius",
			'080765' => "Esparreguera",
			'080771' => "Esplugues de Llobregat",
			'080787' => "Espunyola (L')",
			'080790' => "Estany (L')",
			'081347' => "Figaró-Montmany",
			'080804' => "Fígols",
			'080826' => "Fogars de la Selva",
			'080811' => "Fogars de Montclús",
			'080832' => "Folgueroles",
			'080847' => "Fonollosa",
			'080850' => "Font-rubí",
			'080863' => "Franqueses del Vallès (Les)",
			'080902' => "Gaià",
			'080879' => "Gallifa",
			'080885' => "Garriga (La)",
			'080898' => "Gavà",
			'080919' => "Gelida",
			'080924' => "Gironella",
			'080930' => "Gisclareny",
			'080945' => "Granada (La)",
			'080958' => "Granera",
			'080961' => "Granollers",
			'080977' => "Gualba",
			'080996' => "Guardiola de Berguedà",
			'081000' => "Gurb",
			'081017' => "Hospitalet de Llobregat (L')",
			'081629' => "Hostalets de Pierola (Els)",
			'081022' => "Igualada",
			'081038' => "Jorba",
			'081043' => "Llacuna (La)",
			'081056' => "Llagosta (La)",
			'081075' => "Lliçà d'Amunt",
			'081081' => "Lliçà de Vall",
			'081069' => "Llinars del Vallès",
			'081094' => "Lluçà",
			'081108' => "Malgrat de Mar",
			'081115' => "Malla",
			'081120' => "Manlleu",
			'081136' => "Manresa",
			'082423' => "Marganell",
			'081141' => "Martorell",
			'081154' => "Martorelles",
			'081167' => "Masies de Roda (Les)",
			'081173' => "Masies de Voltregà (Les)",
			'081189' => "Masnou (El)",
			'081192' => "Masquefa",
			'081206' => "Matadepera",
			'081213' => "Mataró",
			'081228' => "Mediona",
			'081385' => "Moià",
			'081234' => "Molins de Rei",
			'081249' => "Mollet del Vallès",
			'081287' => "Monistrol de Calders",
			'081271' => "Monistrol de Montserrat",
			'081252' => "Montcada i Reixac",
			'081304' => "Montclar",
			'081311' => "Montesquiu",
			'081265' => "Montgat",
			'081326' => "Montmajor",
			'081332' => "Montmaneu",
			'081350' => "Montmeló",
			'081363' => "Montornès del Vallès",
			'081379' => "Montseny",
			'081290' => "Muntanyola",
			'081398' => "Mura",
			'081402' => "Navarcles",
			'081419' => "Navàs",
			'081424' => "Nou de Berguedà (La)",
			'081430' => "Òdena",
			'081458' => "Olèrdola",
			'081461' => "Olesa de Bonesvalls",
			'081477' => "Olesa de Montserrat",
			'081483' => "Olivella",
			'081496' => "Olost",
			'081445' => "Olvan",
			'081509' => "Orís",
			'081516' => "Oristà",
			'081521' => "Orpí",
			'081537' => "Òrrius",
			'081542' => "Pacs del Penedès",
			'081555' => "Palafolls",
			'081568' => "Palau-solità i Plegamans",
			'081574' => "Pallejà",
			'089058' => "Palma de Cervelló (La)",
			'081580' => "Papiol (El)",
			'081593' => "Parets del Vallès",
			'081607' => "Perafita",
			'081614' => "Piera",
			'081635' => "Pineda de Mar",
			'081640' => "Pla del Penedès (El)",
			'081653' => "Pobla de Claramunt (La)",
			'081666' => "Pobla de Lillet (La)",
			'081672' => "Polinyà",
			'081825' => "Pont de Vilomara i Rocafort (El)",
			'081688' => "Pontons",
			'081691' => "Prat de Llobregat (El)",
			'081712' => "Prats de Lluçanès",
			'081705' => "Prats de Rei (Els)",
			'082303' => "Premià de Dalt",
			'081727' => "Premià de Mar",
			'081748' => "Puigdàlber",
			'081751' => "Puig-reig",
			'081764' => "Pujalt",
			'081770' => "Quar (La)",
			'081786' => "Rajadell",
			'081799' => "Rellinars",
			'081803' => "Ripollet",
			'081810' => "Roca del Vallès (La)",
			'081831' => "Roda de Ter",
			'081846' => "Rubí",
			'081859' => "Rubió",
			'089019' => "Rupit i Pruit",
			'081878' => "Sabadell",
			'081884' => "Sagàs",
			'081901' => "Saldes",
			'081918' => "Sallent",
			'081944' => "Sant Adrià de Besòs",
			'081957' => "Sant Agustí de Lluçanès",
			'081960' => "Sant Andreu de la Barca",
			'081976' => "Sant Andreu de Llavaneres",
			'081982' => "Sant Antoni de Vilamajor",
			'081995' => "Sant Bartomeu del Grau",
			'082009' => "Sant Boi de Llobregat",
			'082016' => "Sant Boi de Lluçanès",
			'082037' => "Sant Cebrià de Vallalta",
			'082021' => "Sant Celoni",
			'082042' => "Sant Climent de Llobregat",
			'082055' => "Sant Cugat del Vallès",
			'082068' => "Sant Cugat Sesgarrigues",
			'082074' => "Sant Esteve de Palautordera",
			'082080' => "Sant Esteve Sesrovires",
			'082107' => "Sant Feliu de Codines",
			'082114' => "Sant Feliu de Llobregat",
			'082129' => "Sant Feliu Sasserra",
			'082093' => "Sant Fost de Campsentelles",
			'082135' => "Sant Fruitós de Bages",
			'082153' => "Sant Hipòlit de Voltregà",
			'081939' => "Sant Iscle de Vallalta",
			'082166' => "Sant Jaume de Frontanyà",
			'082188' => "Sant Joan de Vilatorrada",
			'082172' => "Sant Joan Despí",
			'089030' => "Sant Julià de Cerdanyola",
			'082205' => "Sant Julià de Vilatorta",
			'082212' => "Sant Just Desvern",
			'082227' => "Sant Llorenç d'Hortons",
			'082233' => "Sant Llorenç Savall",
			'082251' => "Sant Martí d'Albars",
			'082248' => "Sant Martí de Centelles",
			'082264' => "Sant Martí de Tous",
			'082270' => "Sant Martí Sarroca",
			'082286' => "Sant Martí Sesgueioles",
			'082299' => "Sant Mateu de Bages",
			'082310' => "Sant Pere de Ribes",
			'082325' => "Sant Pere de Riudebitlles",
			'082331' => "Sant Pere de Torelló",
			'082346' => "Sant Pere de Vilamajor",
			'081897' => "Sant Pere Sallavinera",
			'082359' => "Sant Pol de Mar",
			'082362' => "Sant Quintí de Mediona",
			'082378' => "Sant Quirze de Besora",
			'082384' => "Sant Quirze del Vallès",
			'082397' => "Sant Quirze Safaja",
			'082401' => "Sant Sadurní d'Anoia",
			'082418' => "Sant Sadurní d'Osormort",
			'080983' => "Sant Salvador de Guardiola",
			'082628' => "Sant Vicenç de Castellet",
			'082649' => "Sant Vicenç de Montalt",
			'082652' => "Sant Vicenç de Torelló",
			'082634' => "Sant Vicenç dels Horts",
			'082439' => "Santa Cecília de Voltregà",
			'082444' => "Santa Coloma de Cervelló",
			'082457' => "Santa Coloma de Gramenet",
			'082460' => "Santa Eugènia de Berga",
			'082476' => "Santa Eulàlia de Riuprimer",
			'082482' => "Santa Eulàlia de Ronçana",
			'082495' => "Santa Fe del Penedès",
			'082508' => "Santa Margarida de Montbui",
			'082515' => "Santa Margarida i els Monjos",
			'082536' => "Santa Maria de Besora",
			'082541' => "Santa Maria de Corcó",
			'082567' => "Santa Maria de Martorelles",
			'082554' => "Santa Maria de Merlès",
			'082573' => "Santa Maria de Miralles",
			'082592' => "Santa Maria de Palautordera",
			'082589' => "Santa Maria d'Oló",
			'082606' => "Santa Perpètua de Mogoda",
			'082613' => "Santa Susanna",
			'081923' => "Santpedor",
			'082671' => "Sentmenat",
			'082690' => "Seva",
			'082704' => "Sitges",
			'082711' => "Sobremunt",
			'082726' => "Sora",
			'082732' => "Subirats",
			'082747' => "Súria",
			'082763' => "Tagamanent",
			'082779' => "Talamanca",
			'082785' => "Taradell",
			'082750' => "Tavèrnoles",
			'082802' => "Tavertet",
			'082819' => "Teià",
			'082798' => "Terrassa",
			'082824' => "Tiana",
			'082830' => "Tona",
			'082845' => "Tordera",
			'082858' => "Torelló",
			'082861' => "Torre de Claramunt (La)",
			'082877' => "Torrelavit",
			'082883' => "Torrelles de Foix",
			'082896' => "Torrelles de Llobregat",
			'082900' => "Ullastrell",
			'082917' => "Vacarisses",
			'082922' => "Vallbona d'Anoia",
			'082938' => "Vallcebre",
			'082943' => "Vallgorguina",
			'082956' => "Vallirana",
			'082969' => "Vallromanes",
			'082975' => "Veciana",
			'082981' => "Vic",
			'082994' => "Vilada",
			'083015' => "Viladecans",
			'083008' => "Viladecavalls",
			'083054' => "Vilafranca del Penedès",
			'083067' => "Vilalba Sasserra",
			'083036' => "Vilanova de Sau",
			'083020' => "Vilanova del Camí",
			'089024' => "Vilanova del Vallès",
			'083073' => "Vilanova i la Geltrú",
			'082140' => "Vilassar de Dalt",
			'082191' => "Vilassar de Mar",
			'083041' => "Vilobí del Penedès",
			'083089' => "Viver i Serrateix",
			'090011' => "Abajas",
			'090032' => "Adrada de Haza",
			'090063' => "Aguas Cándidas",
			'090079' => "Aguilar de Bureba",
			'090098' => "Albillos",
			'090102' => "Alcocero de Mola",
			'090119' => "Alfoz de Bricia",
			'099070' => "Alfoz de Quintanadueñas",
			'090124' => "Alfoz de Santa Gadea",
			'090130' => "Altable",
			'090145' => "Altos (Los)",
			'090161' => "Ameyugo",
			'090177' => "Anguix",
			'090183' => "Aranda de Duero",
			'090196' => "Arandilla",
			'090200' => "Arauzo de Miel",
			'090217' => "Arauzo de Salce",
			'090222' => "Arauzo de Torre",
			'090238' => "Arcos",
			'090243' => "Arenillas de Riopisuerga",
			'090256' => "Arija",
			'090269' => "Arlanzón",
			'090275' => "Arraya de Oca",
			'090294' => "Atapuerca",
			'090308' => "Ausines (Los)",
			'090320' => "Avellanosa de Muñó",
			'090336' => "Bahabón de Esgueva",
			'090341' => "Balbases (Los)",
			'090354' => "Baños de Valdearados",
			'090367' => "Bañuelos de Bureba",
			'090373' => "Barbadillo de Herreros",
			'090389' => "Barbadillo del Mercado",
			'090392' => "Barbadillo del Pez",
			'090413' => "Barrio de Muñó",
			'090434' => "Barrios de Bureba (Los)",
			'090449' => "Barrios de Colina",
			'090452' => "Basconcillos del Tozo",
			'090465' => "Bascuñana",
			'090471' => "Belbimbre",
			'090487' => "Belorado",
			'090503' => "Berberana",
			'090510' => "Berlangas de Roa",
			'090525' => "Berzosa de Bureba",
			'090546' => "Bozoó",
			'090559' => "Brazacorta",
			'090562' => "Briviesca",
			'090578' => "Bugedo",
			'090584' => "Buniel",
			'090597' => "Burgos",
			'090601' => "Busto de Bureba",
			'090618' => "Cabañes de Esgueva",
			'090623' => "Cabezón de la Sierra",
			'090644' => "Caleruega",
			'090657' => "Campillo de Aranda",
			'090660' => "Campolara",
			'090676' => "Canicosa de la Sierra",
			'090682' => "Cantabrana",
			'090709' => "Carazo",
			'090716' => "Carcedo de Bureba",
			'090721' => "Carcedo de Burgos",
			'090737' => "Cardeñadijo",
			'090742' => "Cardeñajimeno",
			'090755' => "Cardeñuela Riopico",
			'090768' => "Carrias",
			'090774' => "Cascajares de Bureba",
			'090780' => "Cascajares de la Sierra",
			'090793' => "Castellanos de Castro",
			'090835' => "Castil de Peones",
			'090829' => "Castildelgado",
			'090840' => "Castrillo de la Reina",
			'090853' => "Castrillo de la Vega",
			'090888' => "Castrillo de Riopisuerga",
			'090866' => "Castrillo del Val",
			'090905' => "Castrillo Matajudíos",
			'090912' => "Castrojeriz",
			'090639' => "Cavia",
			'090933' => "Cayuela",
			'090948' => "Cebrecos",
			'090951' => "Celada del Camino",
			'090986' => "Cerezo de Río Tirón",
			'091003' => "Cerratón de Juarros",
			'091010' => "Ciadoncha",
			'091025' => "Cillaperlata",
			'091031' => "Cilleruelo de Abajo",
			'091046' => "Cilleruelo de Arriba",
			'091059' => "Ciruelos de Cervera",
			'091084' => "Cogollos",
			'091097' => "Condado de Treviño",
			'091101' => "Contreras",
			'091123' => "Coruña del Conde",
			'091139' => "Covarrubias",
			'091144' => "Cubillo del Campo",
			'091157' => "Cubo de Bureba",
			'091176' => "Cueva de Roa (La)",
			'091195' => "Cuevas de San Clemente",
			'091209' => "Encío",
			'091221' => "Espinosa de Cervera",
			'091242' => "Espinosa de los Monteros",
			'091237' => "Espinosa del Camino",
			'091255' => "Estépar",
			'091274' => "Fontioso",
			'091280' => "Frandovínez",
			'091293' => "Fresneda de la Sierra Tirón",
			'091307' => "Fresneña",
			'091314' => "Fresnillo de las Dueñas",
			'091329' => "Fresno de Río Tirón",
			'091335' => "Fresno de Rodilla",
			'091340' => "Frías",
			'091353' => "Fuentebureba",
			'091366' => "Fuentecén",
			'091372' => "Fuentelcésped",
			'091388' => "Fuentelisendo",
			'091391' => "Fuentemolinos",
			'091405' => "Fuentenebro",
			'091412' => "Fuentespina",
			'091433' => "Galbarros",
			'091448' => "Gallega (La)",
			'091486' => "Grijalba",
			'091499' => "Grisaleña",
			'091519' => "Gumiel de Izán",
			'091524' => "Gumiel de Mercado",
			'091545' => "Hacinas",
			'091558' => "Haza",
			'091596' => "Hontanas",
			'091600' => "Hontangas",
			'091622' => "Hontoria de la Cantera",
			'091643' => "Hontoria de Valdearados",
			'091638' => "Hontoria del Pinar",
			'091669' => "Hormazas (Las)",
			'091675' => "Hornillos del Camino",
			'091681' => "Horra (La)",
			'091694' => "Hortigüela",
			'091708' => "Hoyales de Roa",
			'091720' => "Huérmeces",
			'091736' => "Huerta de Arriba",
			'091741' => "Huerta de Rey",
			'091754' => "Humada",
			'091767' => "Hurones",
			'091773' => "Ibeas de Juarros",
			'091789' => "Ibrillos",
			'091792' => "Iglesiarrubia",
			'091806' => "Iglesias",
			'091813' => "Isar",
			'091828' => "Itero del Castillo",
			'091834' => "Jaramillo de la Fuente",
			'091849' => "Jaramillo Quemado",
			'091890' => "Junta de Traslaloma",
			'091904' => "Junta de Villalba de Losa",
			'091911' => "Jurisdicción de Lara",
			'091926' => "Jurisdicción de San Zadornil",
			'091947' => "Lerma",
			'091950' => "Llano de Bureba",
			'091963' => "Madrigal del Monte",
			'091979' => "Madrigalejo del Monte",
			'091985' => "Mahamud",
			'091998' => "Mambrilla de Castrejón",
			'092002' => "Mambrillas de Lara",
			'092019' => "Mamolar",
			'092024' => "Manciles",
			'092061' => "Mazuela",
			'092083' => "Mecerreyes",
			'092096' => "Medina de Pomar",
			'092117' => "Melgar de Fernamental",
			'092138' => "Merindad de Cuesta-Urria",
			'092143' => "Merindad de Montija",
			'099064' => "Merindad de Río Ubierna",
			'092156' => "Merindad de Sotoscueva",
			'092169' => "Merindad de Valdeporres",
			'092175' => "Merindad de Valdivielso",
			'092181' => "Milagros",
			'092194' => "Miranda de Ebro",
			'092208' => "Miraveche",
			'092215' => "Modúbar de la Emparedada",
			'092236' => "Monasterio de la Sierra",
			'092241' => "Monasterio de Rodilla",
			'092254' => "Moncalvillo",
			'092267' => "Monterrubio de la Demanda",
			'092273' => "Montorio",
			'092289' => "Moradillo de Roa",
			'092292' => "Nava de Roa",
			'092306' => "Navas de Bureba",
			'092313' => "Nebreda",
			'092328' => "Neila",
			'092352' => "Olmedillo de Roa",
			'092365' => "Olmillos de Muñó",
			'092387' => "Oña",
			'092390' => "Oquillas",
			'092411' => "Orbaneja Riopico",
			'092426' => "Padilla de Abajo",
			'092432' => "Padilla de Arriba",
			'092447' => "Padrones de Bureba",
			'092463' => "Palacios de la Sierra",
			'092479' => "Palacios de Riopisuerga",
			'092485' => "Palazuelos de la Sierra",
			'092498' => "Palazuelos de Muñó",
			'092501' => "Pampliega",
			'092518' => "Pancorbo",
			'092539' => "Pardilla",
			'092557' => "Partido de la Sierra en Tobalina",
			'092560' => "Pedrosa de Duero",
			'092595' => "Pedrosa de Río Úrbel",
			'092576' => "Pedrosa del Páramo",
			'092582' => "Pedrosa del Príncipe",
			'092616' => "Peñaranda de Duero",
			'092621' => "Peral de Arlanza",
			'092655' => "Piérnigas",
			'092668' => "Pineda de la Sierra",
			'092674' => "Pineda Trasmonte",
			'092680' => "Pinilla de los Barruecos",
			'092693' => "Pinilla de los Moros",
			'092707' => "Pinilla Trasmonte",
			'092729' => "Poza de la Sal",
			'092735' => "Prádanos de Bureba",
			'092740' => "Pradoluengo",
			'092753' => "Presencio",
			'092766' => "Puebla de Arganzón (La)",
			'092772' => "Puentedura",
			'092791' => "Quemada",
			'092812' => "Quintana del Pidio",
			'092805' => "Quintanabureba",
			'092833' => "Quintanaélez",
			'092870' => "Quintanaortuño",
			'092886' => "Quintanapalla",
			'092899' => "Quintanar de la Sierra",
			'092925' => "Quintanavides",
			'092946' => "Quintanilla de la Mata",
			'099012' => "Quintanilla del Agua y Tordueles",
			'092959' => "Quintanilla del Coco",
			'092984' => "Quintanilla San García",
			'093018' => "Quintanilla Vivar",
			'092978' => "Quintanillas (Las)",
			'093023' => "Rabanera del Pinar",
			'093039' => "Rábanos",
			'093044' => "Rabé de las Calzadas",
			'093060' => "Rebolledo de la Torre",
			'093076' => "Redecilla del Camino",
			'093082' => "Redecilla del Campo",
			'093095' => "Regumiel de la Sierra",
			'093109' => "Reinoso",
			'093116' => "Retuerta",
			'093142' => "Revilla del Campo",
			'093168' => "Revilla Vallejera",
			'093121' => "Revilla y Ahedo (La)",
			'093155' => "Revillarruz",
			'093174' => "Rezmondo",
			'093180' => "Riocavado de la Sierra",
			'093214' => "Roa",
			'093235' => "Rojas",
			'093253' => "Royuela de Río Franco",
			'093266' => "Rubena",
			'093272' => "Rublacedo de Abajo",
			'093288' => "Rucandio",
			'093291' => "Salas de Bureba",
			'093305' => "Salas de los Infantes",
			'093327' => "Saldaña de Burgos",
			'093348' => "Salinillas de Bureba",
			'093351' => "San Adrián de Juarros",
			'093370' => "San Juan del Monte",
			'093386' => "San Mamés de Burgos",
			'093399' => "San Martín de Rubiales",
			'093403' => "San Millán de Lara",
			'093608' => "San Vicente del Valle",
			'093431' => "Santa Cecilia",
			'093459' => "Santa Cruz de la Salceda",
			'093462' => "Santa Cruz del Valle Urbión",
			'093478' => "Santa Gadea del Cid",
			'093484' => "Santa Inés",
			'093500' => "Santa María del Campo",
			'093517' => "Santa María del Invierno",
			'093522' => "Santa María del Mercadillo",
			'093538' => "Santa María Rivarredonda",
			'093543' => "Santa Olalla de Bureba",
			'093556' => "Santibáñez de Esgueva",
			'093569' => "Santibáñez del Val",
			'093581' => "Santo Domingo de Silos",
			'093615' => "Sargentes de la Lora",
			'093620' => "Sarracín",
			'093636' => "Sasamón",
			'093654' => "Sequera de Haza (La)",
			'093667' => "Solarana",
			'093689' => "Sordillos",
			'093692' => "Sotillo de la Ribera",
			'093728' => "Sotragero",
			'093734' => "Sotresgudo",
			'093749' => "Susinos del Páramo",
			'093752' => "Tamarón",
			'093771' => "Tardajos",
			'093787' => "Tejada",
			'093804' => "Terradillos de Esgueva",
			'093811' => "Tinieblas de la Sierra",
			'093826' => "Tobar",
			'093847' => "Tordómar",
			'093863' => "Torrecilla del Monte",
			'093879' => "Torregalindo",
			'093885' => "Torrelara",
			'093898' => "Torrepadre",
			'093902' => "Torresandino",
			'093919' => "Tórtoles de Esgueva",
			'093924' => "Tosantos",
			'093945' => "Trespaderne",
			'093958' => "Tubilla del Agua",
			'093961' => "Tubilla del Lago",
			'093983' => "Úrbel del Castillo",
			'094000' => "Vadocondes",
			'094038' => "Valdeande",
			'094056' => "Valdezate",
			'094069' => "Valdorros",
			'094081' => "Vallarta de Bureba",
			'099048' => "Valle de las Navas",
			'099086' => "Valle de Losa",
			'094094' => "Valle de Manzanedo",
			'094108' => "Valle de Mena",
			'094115' => "Valle de Oca",
			'099027' => "Valle de Santibáñez",
			'099051' => "Valle de Sedano",
			'094120' => "Valle de Tobalina",
			'094136' => "Valle de Valdebezana",
			'094141' => "Valle de Valdelaguna",
			'094154' => "Valle de Valdelucio",
			'094167' => "Valle de Zamanzas",
			'094173' => "Vallejera",
			'094189' => "Valles de Palenzuela",
			'094192' => "Valluércanes",
			'094075' => "Valmala",
			'094228' => "Vid de Bureba (La)",
			'094213' => "Vid y Barrios (La)",
			'094234' => "Vileña",
			'094271' => "Villadiego",
			'094287' => "Villaescusa de Roa",
			'094290' => "Villaescusa la Sombría",
			'094304' => "Villaespasa",
			'094311' => "Villafranca Montes de Oca",
			'094326' => "Villafruela",
			'094332' => "Villagalijo",
			'094347' => "Villagonzalo Pedernales",
			'094379' => "Villahoz",
			'094385' => "Villalba de Duero",
			'094398' => "Villalbilla de Burgos",
			'094402' => "Villalbilla de Gumiel",
			'094419' => "Villaldemiro",
			'094424' => "Villalmanzo",
			'094430' => "Villamayor de los Montes",
			'094445' => "Villamayor de Treviño",
			'094458' => "Villambistia",
			'094461' => "Villamedianilla",
			'094477' => "Villamiel de la Sierra",
			'094483' => "Villangómez",
			'094496' => "Villanueva de Argaño",
			'094509' => "Villanueva de Carazo",
			'094516' => "Villanueva de Gumiel",
			'094542' => "Villanueva de Teba",
			'094555' => "Villaquirán de la Puebla",
			'094568' => "Villaquirán de los Infantes",
			'099033' => "Villarcayo de Merindad de Castilla la Vieja",
			'094580' => "Villariezo",
			'094607' => "Villasandino",
			'094635' => "Villasur de Herreros",
			'094640' => "Villatuelda",
			'094666' => "Villaverde del Monte",
			'094672' => "Villaverde-Mogina",
			'094712' => "Villayerno Morquillas",
			'094727' => "Villazopeque",
			'094733' => "Villegas",
			'094764' => "Villoruebo",
			'094249' => "Viloria de Rioja",
			'094252' => "Vilviestre del Pinar",
			'094786' => "Vizcaínos",
			'094803' => "Zael",
			'094825' => "Zarzosa de Río Pisuerga",
			'094831' => "Zazuar",
			'094859' => "Zuñeda",
			'100015' => "Abadía",
			'100020' => "Abertura",
			'100036' => "Acebo",
			'100041' => "Acehúche",
			'100054' => "Aceituna",
			'100067' => "Ahigal",
			'109037' => "Alagón del Río",
			'100073' => "Albalá",
			'100089' => "Alcántara",
			'100092' => "Alcollarín",
			'100106' => "Alcuéscar",
			'100128' => "Aldea del Cano",
			'100134' => "Aldea del Obispo (La)",
			'100113' => "Aldeacentenera",
			'100149' => "Aldeanueva de la Vera",
			'100152' => "Aldeanueva del Camino",
			'100165' => "Aldehuela de Jerte",
			'100171' => "Alía",
			'100187' => "Aliseda",
			'100190' => "Almaraz",
			'100204' => "Almoharín",
			'100211' => "Arroyo de la Luz",
			'100232' => "Arroyomolinos",
			'100226' => "Arroyomolinos de la Vera",
			'100247' => "Baños de Montemayor",
			'100250' => "Barrado",
			'100263' => "Belvís de Monroy",
			'100279' => "Benquerencia",
			'100285' => "Berrocalejo",
			'100298' => "Berzocana",
			'100302' => "Bohonal de Ibor",
			'100319' => "Botija",
			'100324' => "Brozas",
			'100330' => "Cabañas del Castillo",
			'100345' => "Cabezabellosa",
			'100358' => "Cabezuela del Valle",
			'100361' => "Cabrero",
			'100377' => "Cáceres",
			'100383' => "Cachorrilla",
			'100396' => "Cadalso",
			'100400' => "Calzadilla",
			'100417' => "Caminomorisco",
			'100422' => "Campillo de Deleitosa",
			'100438' => "Campo Lugar",
			'100443' => "Cañamero",
			'100456' => "Cañaveral",
			'100469' => "Carbajo",
			'100475' => "Carcaboso",
			'100481' => "Carrascalejo",
			'100494' => "Casar de Cáceres",
			'100507' => "Casar de Palomero",
			'100514' => "Casares de las Hurdes",
			'100529' => "Casas de Don Antonio",
			'100535' => "Casas de Don Gómez",
			'100566' => "Casas de Millán",
			'100572' => "Casas de Miravete",
			'100540' => "Casas del Castañar",
			'100553' => "Casas del Monte",
			'100588' => "Casatejada",
			'100591' => "Casillas de Coria",
			'100605' => "Castañar de Ibor",
			'100612' => "Ceclavín",
			'100627' => "Cedillo",
			'100633' => "Cerezo",
			'100648' => "Cilleros",
			'100651' => "Collado",
			'100664' => "Conquista de la Sierra",
			'100670' => "Coria",
			'100686' => "Cuacos de Yuste",
			'100699' => "Cumbre (La)",
			'100703' => "Deleitosa",
			'100710' => "Descargamaría",
			'100725' => "Eljas",
			'100731' => "Escurial",
			'100759' => "Fresnedoso de Ibor",
			'100762' => "Galisteo",
			'100778' => "Garciaz",
			'100784' => "Garganta (La)",
			'100797' => "Garganta la Olla",
			'100801' => "Gargantilla",
			'100818' => "Gargüera",
			'100823' => "Garrovillas de Alconétar",
			'100839' => "Garvín",
			'100844' => "Gata",
			'100857' => "Gordo (El)",
			'100860' => "Granja (La)",
			'100876' => "Guadalupe",
			'100882' => "Guijo de Coria",
			'100895' => "Guijo de Galisteo",
			'100909' => "Guijo de Granadilla",
			'100916' => "Guijo de Santa Bárbara",
			'100921' => "Herguijuela",
			'100937' => "Hernán-Pérez",
			'100942' => "Herrera de Alcántara",
			'100955' => "Herreruela",
			'100968' => "Hervás",
			'100974' => "Higuera",
			'100980' => "Hinojal",
			'100993' => "Holguera",
			'101007' => "Hoyos",
			'101014' => "Huélaga",
			'101029' => "Ibahernando",
			'101035' => "Jaraicejo",
			'101040' => "Jaraíz de la Vera",
			'101053' => "Jarandilla de la Vera",
			'101066' => "Jarilla",
			'101072' => "Jerte",
			'101088' => "Ladrillar",
			'101091' => "Logrosán",
			'101105' => "Losar de la Vera",
			'101112' => "Madrigal de la Vera",
			'101127' => "Madrigalejo",
			'101133' => "Madroñera",
			'101148' => "Majadas",
			'101151' => "Malpartida de Cáceres",
			'101164' => "Malpartida de Plasencia",
			'101170' => "Marchagaz",
			'101186' => "Mata de Alcántara",
			'101199' => "Membrío",
			'101203' => "Mesas de Ibor",
			'101210' => "Miajadas",
			'101225' => "Millanes",
			'101231' => "Mirabel",
			'101246' => "Mohedas de Granadilla",
			'101259' => "Monroy",
			'101262' => "Montánchez",
			'101278' => "Montehermoso",
			'101284' => "Moraleja",
			'101297' => "Morcillo",
			'101301' => "Navaconcejo",
			'101318' => "Navalmoral de la Mata",
			'101323' => "Navalvillar de Ibor",
			'101339' => "Navas del Madroño",
			'101344' => "Navezuelas",
			'101357' => "Nuñomoral",
			'101360' => "Oliva de Plasencia",
			'101376' => "Palomero",
			'101382' => "Pasarón de la Vera",
			'101395' => "Pedroso de Acim",
			'101409' => "Peraleda de la Mata",
			'101416' => "Peraleda de San Román",
			'101421' => "Perales del Puerto",
			'101437' => "Pescueza",
			'101442' => "Pesga (La)",
			'101455' => "Piedras Albas",
			'101468' => "Pinofranqueado",
			'101474' => "Piornal",
			'101480' => "Plasencia",
			'101493' => "Plasenzuela",
			'101506' => "Portaje",
			'101513' => "Portezuelo",
			'101528' => "Pozuelo de Zarzón",
			'101534' => "Puerto de Santa Cruz",
			'101549' => "Rebollar",
			'101552' => "Riolobos",
			'101565' => "Robledillo de Gata",
			'101571' => "Robledillo de la Vera",
			'101587' => "Robledillo de Trujillo",
			'101590' => "Robledollano",
			'101604' => "Romangordo",
			'109016' => "Rosalejo",
			'101611' => "Ruanes",
			'101626' => "Salorino",
			'101632' => "Salvatierra de Santiago",
			'101647' => "San Martín de Trevejo",
			'101650' => "Santa Ana",
			'101663' => "Santa Cruz de la Sierra",
			'101679' => "Santa Cruz de Paniagua",
			'101685' => "Santa Marta de Magasca",
			'101698' => "Santiago de Alcántara",
			'101702' => "Santiago del Campo",
			'101719' => "Santibáñez el Alto",
			'101724' => "Santibáñez el Bajo",
			'101730' => "Saucedilla",
			'101745' => "Segura de Toro",
			'101758' => "Serradilla",
			'101761' => "Serrejón",
			'101777' => "Sierra de Fuentes",
			'101783' => "Talaván",
			'101796' => "Talaveruela de la Vera",
			'101800' => "Talayuela",
			'101817' => "Tejeda de Tiétar",
			'101822' => "Toril",
			'101838' => "Tornavacas",
			'101843' => "Torno (El)",
			'101875' => "Torre de Don Miguel",
			'101881' => "Torre de Santa María",
			'101856' => "Torrecilla de los Ángeles",
			'101869' => "Torrecillas de la Tiesa",
			'101908' => "Torrejón el Rubio",
			'101894' => "Torrejoncillo",
			'101915' => "Torremenga",
			'101920' => "Torremocha",
			'101936' => "Torreorgaz",
			'101941' => "Torrequemada",
			'101954' => "Trujillo",
			'101967' => "Valdastillas",
			'101973' => "Valdecañas de Tajo",
			'101989' => "Valdefuentes",
			'101992' => "Valdehúncar",
			'102006' => "Valdelacasa de Tajo",
			'102013' => "Valdemorales",
			'102028' => "Valdeobispo",
			'102034' => "Valencia de Alcántara",
			'102049' => "Valverde de la Vera",
			'102052' => "Valverde del Fresno",
			'109021' => "Vegaviana",
			'102065' => "Viandar de la Vera",
			'102071' => "Villa del Campo",
			'102087' => "Villa del Rey",
			'102090' => "Villamesías",
			'102104' => "Villamiel",
			'102111' => "Villanueva de la Sierra",
			'102126' => "Villanueva de la Vera",
			'102147' => "Villar de Plasencia",
			'102132' => "Villar del Pedroso",
			'102150' => "Villasbuenas de Gata",
			'102163' => "Zarza de Granadilla",
			'102179' => "Zarza de Montánchez",
			'102185' => "Zarza la Mayor",
			'102198' => "Zorita",
			'110012' => "Alcalá de los Gazules",
			'110027' => "Alcalá del Valle",
			'110033' => "Algar",
			'110048' => "Algeciras",
			'110051' => "Algodonales",
			'110064' => "Arcos de la Frontera",
			'110070' => "Barbate",
			'110086' => "Barrios (Los)",
			'119013' => "Benalup-Casas Viejas",
			'110099' => "Benaocaz",
			'110103' => "Bornos",
			'110110' => "Bosque (El)",
			'110125' => "Cádiz",
			'110131' => "Castellar de la Frontera",
			'110159' => "Chiclana de la Frontera",
			'110162' => "Chipiona",
			'110146' => "Conil de la Frontera",
			'110178' => "Espera",
			'110184' => "Gastor (El)",
			'110197' => "Grazalema",
			'110201' => "Jerez de la Frontera",
			'110218' => "Jimena de la Frontera",
			'110223' => "Línea de la Concepción (La)",
			'110239' => "Medina-Sidonia",
			'110244' => "Olvera",
			'110257' => "Paterna de Rivera",
			'110260' => "Prado del Rey",
			'110276' => "Puerto de Santa María (El)",
			'110282' => "Puerto Real",
			'110295' => "Puerto Serrano",
			'110309' => "Rota",
			'110316' => "San Fernando",
			'119028' => "San José del Valle",
			'110337' => "San Roque",
			'110321' => "Sanlúcar de Barrameda",
			'110342' => "Setenil de las Bodegas",
			'110355' => "Tarifa",
			'110368' => "Torre Alháquime",
			'110374' => "Trebujena",
			'110380' => "Ubrique",
			'110393' => "Vejer de la Frontera",
			'110407' => "Villaluenga del Rosario",
			'110414' => "Villamartín",
			'110429' => "Zahara",
			'120022' => "Aín",
			'120038' => "Albocàsser",
			'120043' => "Alcalà de Xivert",
			'120056' => "Alcora (l')",
			'120069' => "Alcudia de Veo",
			'120075' => "Alfondeguilla",
			'120081' => "Algimia de Almonacid",
			'120094' => "Almazora/Almassora",
			'120108' => "Almedíjar",
			'120115' => "Almenara",
			'129018' => "Alquerías del Niño Perdido",
			'120120' => "Altura",
			'120136' => "Arañuel",
			'120141' => "Ares del Maestrat",
			'120154' => "Argelita",
			'120167' => "Artana",
			'120017' => "Atzeneta del Maestrat",
			'120173' => "Ayódar",
			'120189' => "Azuébar",
			'120206' => "Barracas",
			'120228' => "Bejís",
			'120249' => "Benafer",
			'120252' => "Benafigos",
			'120265' => "Benasal",
			'120271' => "Benicarló",
			'120287' => "Benicasim/Benicàssim",
			'120290' => "Benlloch",
			'120213' => "Betxí",
			'120326' => "Borriana/Burriana",
			'120311' => "Borriol",
			'120332' => "Cabanes",
			'120347' => "Càlig",
			'120363' => "Canet lo Roig",
			'120379' => "Castell de Cabres",
			'120385' => "Castellfort",
			'120398' => "Castellnovo",
			'120402' => "Castellón de la Plana/Castelló de la Plana",
			'120419' => "Castillo de Villamalefa",
			'120424' => "Catí",
			'120430' => "Caudiel",
			'120445' => "Cervera del Maestre",
			'120521' => "Chert/Xert",
			'120537' => "Chilches/Xilxes",
			'120555' => "Chodos/Xodos",
			'120568' => "Chóvar",
			'120458' => "Cinctorres",
			'120461' => "Cirat",
			'120483' => "Cortes de Arenoso",
			'120496' => "Costur",
			'120509' => "Coves de Vinromà (les)",
			'120516' => "Culla",
			'120574' => "Eslida",
			'120580' => "Espadilla",
			'120593' => "Fanzara",
			'120607' => "Figueroles",
			'120614' => "Forcall",
			'120635' => "Fuente la Reina",
			'120640' => "Fuentes de Ayódar",
			'120653' => "Gaibiel",
			'120672' => "Geldo",
			'120688' => "Herbés",
			'120691' => "Higueras",
			'120705' => "Jana (la)",
			'120712' => "Jérica",
			'120748' => "Llosa (la)",
			'120727' => "Lucena del Cid",
			'120733' => "Ludiente",
			'120751' => "Mata de Morella (la)",
			'120764' => "Matet",
			'120770' => "Moncofa",
			'120786' => "Montán",
			'120799' => "Montanejos",
			'120803' => "Morella",
			'120810' => "Navajas",
			'120825' => "Nules",
			'120831' => "Olocau del Rey",
			'120846' => "Onda",
			'120859' => "Oropesa del Mar/Orpesa",
			'120878' => "Palanques",
			'120884' => "Pavías",
			'120897' => "Peníscola/Peñíscola",
			'120901' => "Pina de Montalgrao",
			'120939' => "Pobla de Benifassà (la)",
			'120944' => "Pobla Tornesa (la)",
			'120918' => "Portell de Morella",
			'120923' => "Puebla de Arenoso",
			'120957' => "Ribesalbes",
			'120960' => "Rossell",
			'120976' => "Sacañet",
			'120982' => "Salzadella (la)",
			'121016' => "San Rafael del Río",
			'129023' => "Sant Joan de Moró",
			'120995' => "Sant Jordi/San Jorge",
			'121009' => "Sant Mateu",
			'121021' => "Santa Magdalena de Pulpis",
			'121037' => "Sarratella",
			'121042' => "Segorbe",
			'121055' => "Sierra Engarcerán",
			'121068' => "Soneja",
			'121074' => "Sot de Ferrer",
			'121080' => "Sueras/Suera",
			'121093' => "Tales",
			'121107' => "Teresa",
			'121114' => "Tírig",
			'121129' => "Todolella",
			'121135' => "Toga",
			'121140' => "Torás",
			'121153' => "Toro (El)",
			'121166' => "Torralba del Pinar",
			'121191' => "Torre d'En Besora (la)",
			'121205' => "Torre d'en Doménec (la)",
			'121172' => "Torreblanca",
			'121188' => "Torrechiva",
			'121212' => "Traiguera",
			'121227' => "Useras/Useres (les)",
			'121248' => "Vall d'Alba",
			'121251' => "Vall de Almonacid",
			'121264' => "Vall d'Uixó (la)",
			'121233' => "Vallat",
			'121270' => "Vallibona",
			'121286' => "Vilafamés",
			'121325' => "Vilanova d'Alcolea",
			'121346' => "Vilar de Canes",
			'121359' => "Vila-real",
			'121362' => "Vilavella (la)",
			'121299' => "Villafranca del Cid/Vilafranca",
			'121303' => "Villahermosa del Río",
			'121310' => "Villamalur",
			'121331' => "Villanueva de Viver",
			'121378' => "Villores",
			'121384' => "Vinaròs",
			'121397' => "Vistabella del Maestrazgo",
			'121401' => "Viver",
			'121418' => "Zorita del Maestrazgo",
			'121423' => "Zucaina",
			'130013' => "Abenójar",
			'130028' => "Agudo",
			'130034' => "Alamillo",
			'130049' => "Albaladejo",
			'130052' => "Alcázar de San Juan",
			'130065' => "Alcoba",
			'130071' => "Alcolea de Calatrava",
			'130087' => "Alcubillas",
			'130090' => "Aldea del Rey",
			'130104' => "Alhambra",
			'130111' => "Almadén",
			'130126' => "Almadenejos",
			'130132' => "Almagro",
			'130147' => "Almedina",
			'130150' => "Almodóvar del Campo",
			'130163' => "Almuradiel",
			'130179' => "Anchuras",
			'139035' => "Arenales de San Gregorio",
			'130185' => "Arenas de San Juan",
			'130198' => "Argamasilla de Alba",
			'130202' => "Argamasilla de Calatrava",
			'130219' => "Arroba de los Montes",
			'130224' => "Ballesteros de Calatrava",
			'130230' => "Bolaños de Calatrava",
			'130245' => "Brazatortas",
			'130258' => "Cabezarados",
			'130261' => "Cabezarrubias del Puerto",
			'130277' => "Calzada de Calatrava",
			'130283' => "Campo de Criptana",
			'130296' => "Cañada de Calatrava",
			'130300' => "Caracuel de Calatrava",
			'130317' => "Carrión de Calatrava",
			'130322' => "Carrizosa",
			'130338' => "Castellar de Santiago",
			'130381' => "Chillón",
			'130343' => "Ciudad Real",
			'130356' => "Corral de Calatrava",
			'130369' => "Cortijos (Los)",
			'130375' => "Cózar",
			'130394' => "Daimiel",
			'130408' => "Fernán Caballero",
			'130415' => "Fontanarejo",
			'130420' => "Fuencaliente",
			'130436' => "Fuenllana",
			'130441' => "Fuente el Fresno",
			'130454' => "Granátula de Calatrava",
			'130467' => "Guadalmez",
			'130473' => "Herencia",
			'130489' => "Hinojosas de Calatrava",
			'130492' => "Horcajo de los Montes",
			'130505' => "Labores (Las)",
			'139040' => "Llanos del Caudillo",
			'130512' => "Luciana",
			'130527' => "Malagón",
			'130533' => "Manzanares",
			'130548' => "Membrilla",
			'130551' => "Mestanza",
			'130564' => "Miguelturra",
			'130570' => "Montiel",
			'130586' => "Moral de Calatrava",
			'130599' => "Navalpino",
			'130603' => "Navas de Estena",
			'130610' => "Pedro Muñoz",
			'130625' => "Picón",
			'130631' => "Piedrabuena",
			'130646' => "Poblete",
			'130659' => "Porzuna",
			'130662' => "Pozuelo de Calatrava",
			'130678' => "Pozuelos de Calatrava (Los)",
			'130684' => "Puebla de Don Rodrigo",
			'130697' => "Puebla del Príncipe",
			'130701' => "Puerto Lápice",
			'130718' => "Puertollano",
			'130723' => "Retuerta del Bullaque",
			'139014' => "Robledo (El)",
			'139029' => "Ruidera",
			'130739' => "Saceruela",
			'130744' => "San Carlos del Valle",
			'130757' => "San Lorenzo de Calatrava",
			'130760' => "Santa Cruz de los Cáñamos",
			'130776' => "Santa Cruz de Mudela",
			'130782' => "Socuéllamos",
			'130795' => "Solana (La)",
			'130809' => "Solana del Pino",
			'130816' => "Terrinches",
			'130821' => "Tomelloso",
			'130837' => "Torralba de Calatrava",
			'130842' => "Torre de Juan Abad",
			'130855' => "Torrenueva",
			'130868' => "Valdemanco del Esteras",
			'130874' => "Valdepeñas",
			'130880' => "Valenzuela de Calatrava",
			'130893' => "Villahermosa",
			'130907' => "Villamanrique",
			'130914' => "Villamayor de Calatrava",
			'130929' => "Villanueva de la Fuente",
			'130935' => "Villanueva de los Infantes",
			'130940' => "Villanueva de San Carlos",
			'130953' => "Villar del Pozo",
			'130966' => "Villarrubia de los Ojos",
			'130972' => "Villarta de San Juan",
			'130988' => "Viso del Marqués",
			'140018' => "Adamuz",
			'140023' => "Aguilar de la Frontera",
			'140039' => "Alcaracejos",
			'140044' => "Almedinilla",
			'140057' => "Almodóvar del Río",
			'140060' => "Añora",
			'140076' => "Baena",
			'140082' => "Belalcázar",
			'140095' => "Belmez",
			'140109' => "Benamejí",
			'140116' => "Blázquez (Los)",
			'140121' => "Bujalance",
			'140137' => "Cabra",
			'140142' => "Cañete de las Torres",
			'140155' => "Carcabuey",
			'140168' => "Cardeña",
			'140174' => "Carlota (La)",
			'140180' => "Carpio (El)",
			'140193' => "Castro del Río",
			'140207' => "Conquista",
			'140214' => "Córdoba",
			'140229' => "Doña Mencía",
			'140235' => "Dos Torres",
			'140240' => "Encinas Reales",
			'140253' => "Espejo",
			'140266' => "Espiel",
			'140272' => "Fernán-Núñez",
			'140288' => "Fuente la Lancha",
			'140291' => "Fuente Obejuna",
			'140305' => "Fuente Palmera",
			'140312' => "Fuente-Tójar",
			'140327' => "Granjuela (La)",
			'140333' => "Guadalcázar",
			'140348' => "Guijo (El)",
			'140351' => "Hinojosa del Duque",
			'140364' => "Hornachuelos",
			'140370' => "Iznájar",
			'140386' => "Lucena",
			'140399' => "Luque",
			'140403' => "Montalbán de Córdoba",
			'140410' => "Montemayor",
			'140425' => "Montilla",
			'140431' => "Montoro",
			'140446' => "Monturque",
			'140459' => "Moriles",
			'140462' => "Nueva Carteya",
			'140478' => "Obejo",
			'140484' => "Palenciana",
			'140497' => "Palma del Río",
			'140500' => "Pedro Abad",
			'140517' => "Pedroche",
			'140522' => "Peñarroya-Pueblonuevo",
			'140538' => "Posadas",
			'140543' => "Pozoblanco",
			'140556' => "Priego de Córdoba",
			'140569' => "Puente Genil",
			'140575' => "Rambla (La)",
			'140581' => "Rute",
			'140594' => "San Sebastián de los Ballesteros",
			'140615' => "Santa Eufemia",
			'140608' => "Santaella",
			'140620' => "Torrecampo",
			'140636' => "Valenzuela",
			'140641' => "Valsequillo",
			'140654' => "Victoria (La)",
			'140667' => "Villa del Río",
			'140673' => "Villafranca de Córdoba",
			'140689' => "Villaharta",
			'140692' => "Villanueva de Córdoba",
			'140706' => "Villanueva del Duque",
			'140713' => "Villanueva del Rey",
			'140728' => "Villaralto",
			'140734' => "Villaviciosa de Córdoba",
			'140749' => "Viso (El)",
			'140752' => "Zuheros",
			'150011' => "Abegondo",
			'150026' => "Ames",
			'150032' => "Aranga",
			'150047' => "Ares",
			'150050' => "Arteixo",
			'150063' => "Arzúa",
			'150079' => "Baña (A)",
			'150085' => "Bergondo",
			'150098' => "Betanzos",
			'150102' => "Boimorto",
			'150119' => "Boiro",
			'150124' => "Boqueixón",
			'150130' => "Brión",
			'150145' => "Cabana de Bergantiños",
			'150158' => "Cabanas",
			'150161' => "Camariñas",
			'150177' => "Cambre",
			'150183' => "Capela (A)",
			'150196' => "Carballo",
			'159012' => "Cariño",
			'150200' => "Carnota",
			'150217' => "Carral",
			'150222' => "Cedeira",
			'150238' => "Cee",
			'150243' => "Cerceda",
			'150256' => "Cerdido",
			'150269' => "Cesuras",
			'150275' => "Coirós",
			'150281' => "Corcubión",
			'150294' => "Coristanco",
			'150308' => "Coruña (A)",
			'150315' => "Culleredo",
			'150320' => "Curtis",
			'150336' => "Dodro",
			'150341' => "Dumbría",
			'150354' => "Fene",
			'150367' => "Ferrol",
			'150373' => "Fisterra",
			'150389' => "Frades",
			'150392' => "Irixoa",
			'150413' => "Laracha (A)",
			'150406' => "Laxe",
			'150428' => "Lousame",
			'150434' => "Malpica de Bergantiños",
			'150449' => "Mañón",
			'150452' => "Mazaricos",
			'150465' => "Melide",
			'150471' => "Mesía",
			'150487' => "Miño",
			'150490' => "Moeche",
			'150503' => "Monfero",
			'150510' => "Mugardos",
			'150531' => "Muros",
			'150525' => "Muxía",
			'150546' => "Narón",
			'150559' => "Neda",
			'150562' => "Negreira",
			'150578' => "Noia",
			'150584' => "Oleiros",
			'150597' => "Ordes",
			'150601' => "Oroso",
			'150618' => "Ortigueira",
			'150623' => "Outes",
			'150639' => "Oza dos Ríos",
			'150644' => "Paderne",
			'150657' => "Padrón",
			'150660' => "Pino (O)",
			'150676' => "Pobra do Caramiñal (A)",
			'150682' => "Ponteceso",
			'150695' => "Pontedeume",
			'150709' => "Pontes de García Rodríguez (As)",
			'150716' => "Porto do Son",
			'150721' => "Rianxo",
			'150737' => "Ribeira",
			'150742' => "Rois",
			'150755' => "Sada",
			'150768' => "San Sadurniño",
			'150774' => "Santa Comba",
			'150780' => "Santiago de Compostela",
			'150793' => "Santiso",
			'150807' => "Sobrado",
			'150814' => "Somozas (As)",
			'150829' => "Teo",
			'150835' => "Toques",
			'150840' => "Tordoia",
			'150853' => "Touro",
			'150866' => "Trazo",
			'150888' => "Val do Dubra",
			'150872' => "Valdoviño",
			'150891' => "Vedra",
			'150912' => "Vilarmaior",
			'150905' => "Vilasantar",
			'150927' => "Vimianzo",
			'150933' => "Zas",
			'160014' => "Abia de la Obispalía",
			'160029' => "Acebrón (El)",
			'160035' => "Alarcón",
			'160040' => "Albaladejo del Cuende",
			'160053' => "Albalate de las Nogueras",
			'160066' => "Albendea",
			'160072' => "Alberca de Záncara (La)",
			'160088' => "Alcalá de la Vega",
			'160091' => "Alcantud",
			'160105' => "Alcázar del Rey",
			'160112' => "Alcohujate",
			'160127' => "Alconchel de la Estrella",
			'160133' => "Algarra",
			'160148' => "Aliaguilla",
			'160151' => "Almarcha (La)",
			'160164' => "Almendros",
			'160170' => "Almodóvar del Pinar",
			'160186' => "Almonacid del Marquesado",
			'160199' => "Altarejos",
			'160203' => "Arandilla del Arroyo",
			'169054' => "Arcas del Villar",
			'160225' => "Arcos de la Sierra",
			'160246' => "Arguisuelas",
			'160259' => "Arrancacepas",
			'160262' => "Atalaya del Cañavate",
			'160278' => "Barajas de Melo",
			'160297' => "Barchín del Hoyo",
			'160301' => "Bascuñana de San Pedro",
			'160318' => "Beamud",
			'160323' => "Belinchón",
			'160339' => "Belmonte",
			'160344' => "Belmontejo",
			'160357' => "Beteta",
			'160360' => "Boniches",
			'160382' => "Buciegas",
			'160395' => "Buenache de Alarcón",
			'160409' => "Buenache de la Sierra",
			'160416' => "Buendía",
			'160421' => "Campillo de Altobuey",
			'160437' => "Campillos-Paravientos",
			'160442' => "Campillos-Sierra",
			'169015' => "Campos del Paraíso",
			'160455' => "Canalejas del Arroyo",
			'160468' => "Cañada del Hoyo",
			'160474' => "Cañada Juncosa",
			'160480' => "Cañamares",
			'160493' => "Cañavate (El)",
			'160506' => "Cañaveras",
			'160513' => "Cañaveruelas",
			'160528' => "Cañete",
			'160534' => "Cañizares",
			'160552' => "Carboneras de Guadazaón",
			'160565' => "Cardenete",
			'160571' => "Carrascosa",
			'160587' => "Carrascosa de Haro",
			'160604' => "Casas de Benítez",
			'160611' => "Casas de Fernando Alonso",
			'160626' => "Casas de Garcimolina",
			'160632' => "Casas de Guijarro",
			'160647' => "Casas de Haro",
			'160650' => "Casas de los Pinos",
			'160663' => "Casasimarro",
			'160679' => "Castejón",
			'160685' => "Castillejo de Iniesta",
			'160702' => "Castillejo-Sierra",
			'160724' => "Castillo de Garcimuñoz",
			'160719' => "Castillo-Albaráñez",
			'160730' => "Cervera del Llano",
			'160231' => "Chillarón de Cuenca",
			'160817' => "Chumillas",
			'160745' => "Cierva (La)",
			'160783' => "Cuenca",
			'160796' => "Cueva del Hierro",
			'160822' => "Enguídanos",
			'160838' => "Fresneda de Altarejos",
			'160843' => "Fresneda de la Sierra",
			'160856' => "Frontera (La)",
			'160869' => "Fuente de Pedro Naharro",
			'160875' => "Fuentelespino de Haro",
			'160881' => "Fuentelespino de Moya",
			'169041' => "Fuentenava de Jábaga",
			'160894' => "Fuentes",
			'160915' => "Fuertescusa",
			'160920' => "Gabaldón",
			'160936' => "Garaballa",
			'160941' => "Gascueña",
			'160954' => "Graja de Campalbo",
			'160967' => "Graja de Iniesta",
			'160973' => "Henarejos",
			'160989' => "Herrumblar (El)",
			'160992' => "Hinojosa (La)",
			'161006' => "Hinojosos (Los)",
			'161013' => "Hito (El)",
			'161028' => "Honrubia",
			'161034' => "Hontanaya",
			'161049' => "Hontecillas",
			'161065' => "Horcajo de Santiago",
			'161071' => "Huélamo",
			'161087' => "Huelves",
			'161090' => "Huérguina",
			'161104' => "Huerta de la Obispalía",
			'161111' => "Huerta del Marquesado",
			'161126' => "Huete",
			'161132' => "Iniesta",
			'161150' => "Laguna del Marquesado",
			'161163' => "Lagunaseca",
			'161179' => "Landete",
			'161185' => "Ledaña",
			'161198' => "Leganiel",
			'161219' => "Majadas (Las)",
			'161224' => "Mariana",
			'161230' => "Masegosa",
			'161245' => "Mesas (Las)",
			'161258' => "Minglanilla",
			'161261' => "Mira",
			'161283' => "Monreal del Llano",
			'161296' => "Montalbanejo",
			'161300' => "Montalbo",
			'161317' => "Monteagudo de las Salinas",
			'161322' => "Mota de Altarejos",
			'161338' => "Mota del Cuervo",
			'161343' => "Motilla del Palancar",
			'161356' => "Moya",
			'161375' => "Narboneta",
			'161394' => "Olivares de Júcar",
			'161408' => "Olmeda de la Cuesta",
			'161415' => "Olmeda del Rey",
			'161420' => "Olmedilla de Alarcón",
			'161436' => "Olmedilla de Eliz",
			'161454' => "Osa de la Vega",
			'161467' => "Pajarón",
			'161473' => "Pajaroncillo",
			'161489' => "Palomares del Campo",
			'161492' => "Palomera",
			'161505' => "Paracuellos",
			'161512' => "Paredes",
			'161527' => "Parra de las Vegas (La)",
			'161533' => "Pedernoso (El)",
			'161548' => "Pedroñeras (Las)",
			'161551' => "Peral (El)",
			'161564' => "Peraleja (La)",
			'161570' => "Pesquera (La)",
			'161586' => "Picazo (El)",
			'161599' => "Pinarejo",
			'161603' => "Pineda de Gigüela",
			'161610' => "Piqueras del Castillo",
			'161625' => "Portalrubio de Guadamejud",
			'161631' => "Portilla",
			'161659' => "Poyatos",
			'161662' => "Pozoamargo",
			'169089' => "Pozorrubielos de la Mancha",
			'161678' => "Pozorrubio",
			'161697' => "Pozuelo (El)",
			'161701' => "Priego",
			'161718' => "Provencio (El)",
			'161723' => "Puebla de Almenara",
			'161739' => "Puebla de Don Francisco",
			'161744' => "Puebla del Salvador",
			'161757' => "Quintanar del Rey",
			'161760' => "Rada de Haro",
			'161776' => "Reíllo",
			'161816' => "Rozalén del Monte",
			'161855' => "Saceda-Trasierra",
			'161868' => "Saelices",
			'161874' => "Salinas del Manzano",
			'161880' => "Salmeroncillos",
			'161893' => "Salvacañete",
			'161907' => "San Clemente",
			'161914' => "San Lorenzo de la Parrilla",
			'161929' => "San Martín de Boniches",
			'161935' => "San Pedro Palmiches",
			'161940' => "Santa Cruz de Moya",
			'161966' => "Santa María de los Llanos",
			'161953' => "Santa María del Campo Rus",
			'161972' => "Santa María del Val",
			'161988' => "Sisante",
			'161991' => "Solera de Gabaldón",
			'169092' => "Sotorribas",
			'162027' => "Talayuelas",
			'162033' => "Tarancón",
			'162048' => "Tébar",
			'162051' => "Tejadillos",
			'162064' => "Tinajas",
			'162099' => "Torralba",
			'162110' => "Torrejoncillo del Rey",
			'162125' => "Torrubia del Campo",
			'162131' => "Torrubia del Castillo",
			'162159' => "Tragacete",
			'162162' => "Tresjuncos",
			'162178' => "Tribaldos",
			'162184' => "Uclés",
			'162197' => "Uña",
			'169067' => "Valdecolmenas (Los)",
			'162244' => "Valdemeca",
			'162257' => "Valdemorillo de la Sierra",
			'162276' => "Valdemoro-Sierra",
			'162282' => "Valdeolivas",
			'169020' => "Valdetórtola",
			'169036' => "Valeras (Las)",
			'162316' => "Valhermoso de la Fuente",
			'162342' => "Valsalobre",
			'162368' => "Valverde de Júcar",
			'162374' => "Valverdejo",
			'162380' => "Vara de Rey",
			'162393' => "Vega del Codorno",
			'162407' => "Vellisca",
			'162429' => "Villaconejos de Trabaque",
			'162435' => "Villaescusa de Haro",
			'162440' => "Villagarcía del Llano",
			'162453' => "Villalba de la Sierra",
			'162466' => "Villalba del Rey",
			'162472' => "Villalgordo del Marquesado",
			'162488' => "Villalpardo",
			'162491' => "Villamayor de Santiago",
			'162504' => "Villanueva de Guadamejud",
			'162511' => "Villanueva de la Jara",
			'162532' => "Villar de Cañas",
			'162547' => "Villar de Domingo García",
			'162550' => "Villar de la Encina",
			'162630' => "Villar de Olalla",
			'162585' => "Villar del Humo",
			'162598' => "Villar del Infantado",
			'169106' => "Villar y Velasco",
			'162645' => "Villarejo de Fuentes",
			'162658' => "Villarejo de la Peñuela",
			'162661' => "Villarejo-Periesteban",
			'162696' => "Villares del Saz",
			'162700' => "Villarrubio",
			'162717' => "Villarta",
			'162722' => "Villas de la Ventosa",
			'162738' => "Villaverde y Pasaconsol",
			'162743' => "Víllora",
			'162756' => "Vindel",
			'162769' => "Yémeda",
			'162775' => "Zafra de Záncara",
			'162781' => "Zafrilla",
			'162794' => "Zarza de Tajo",
			'162808' => "Zarzuela",
			'170010' => "Agullana",
			'170025' => "Aiguaviva",
			'170031' => "Albanyà",
			'170046' => "Albons",
			'170062' => "Alp",
			'170078' => "Amer",
			'170084' => "Anglès",
			'170097' => "Arbúcies",
			'170101' => "Argelaguer",
			'170118' => "Armentera (L')",
			'170123' => "Avinyonet de Puigventós",
			'170157' => "Banyoles",
			'170160' => "Bàscara",
			'170139' => "Begur",
			'170182' => "Bellcaire d'Empordà",
			'170195' => "Besalú",
			'170209' => "Bescanó",
			'170216' => "Beuda",
			'170221' => "Bisbal d'Empordà (La)",
			'172348' => "Biure",
			'170237' => "Blanes",
			'170293' => "Boadella i les Escaules",
			'170242' => "Bolvir",
			'170255' => "Bordils",
			'170268' => "Borrassà",
			'170274' => "Breda",
			'170280' => "Brunyola",
			'170314' => "Cabanelles",
			'170307' => "Cabanes",
			'170329' => "Cadaqués",
			'170335' => "Caldes de Malavella",
			'170340' => "Calonge",
			'170353' => "Camós",
			'170366' => "Campdevànol",
			'170372' => "Campelles",
			'170388' => "Campllong",
			'170391' => "Camprodon",
			'170405' => "Canet d'Adri",
			'170412' => "Cantallops",
			'170427' => "Capmany",
			'170448' => "Cassà de la Selva",
			'170464' => "Castellfollit de la Roca",
			'170470' => "Castelló d'Empúries",
			'170486' => "Castell-Platja d'Aro",
			'171899' => "Cellera de Ter (La)",
			'170499' => "Celrà",
			'170502' => "Cervià de Ter",
			'170519' => "Cistella",
			'170545' => "Colera",
			'170558' => "Colomers",
			'170577' => "Corçà",
			'170561' => "Cornellà del Terri",
			'170583' => "Crespià",
			'179011' => "Cruïlles, Monells i Sant Sadurní de l'Heura",
			'170600' => "Darnius",
			'170617' => "Das",
			'170622' => "Escala (L')",
			'170638' => "Espinelves",
			'170643' => "Espolla",
			'170656' => "Esponellà",
			'170059' => "Far d'Empordà (El)",
			'170669' => "Figueres",
			'170675' => "Flaçà",
			'170681' => "Foixà",
			'170694' => "Fontanals de Cerdanya",
			'170708' => "Fontanilles",
			'170715' => "Fontcoberta",
			'179026' => "Forallac",
			'170736' => "Fornells de la Selva",
			'170741' => "Fortià",
			'170754' => "Garrigàs",
			'170767' => "Garrigoles",
			'170773' => "Garriguella",
			'170789' => "Ger",
			'170792' => "Girona",
			'170806' => "Gombrèn",
			'170813' => "Gualta",
			'170828' => "Guils de Cerdanya",
			'170834' => "Hostalric",
			'170849' => "Isòvol",
			'170852' => "Jafre",
			'170865' => "Jonquera (La)",
			'170871' => "Juià",
			'170887' => "Lladó",
			'170890' => "Llagostera",
			'170904' => "Llambilles",
			'170911' => "Llanars",
			'170926' => "Llançà",
			'170932' => "Llers",
			'170947' => "Llívia",
			'170950' => "Lloret de Mar",
			'170963' => "Llosses (Les)",
			'171024' => "Maçanet de Cabrenys",
			'171030' => "Maçanet de la Selva",
			'170979' => "Madremanya",
			'170985' => "Maià de Montcal",
			'171002' => "Masarac",
			'171019' => "Massanes",
			'170998' => "Meranges",
			'171058' => "Mieres",
			'171061' => "Mollet de Peralada",
			'171077' => "Molló",
			'171096' => "Montagut i Oix",
			'171100' => "Mont-ras",
			'171117' => "Navata",
			'171122' => "Ogassa",
			'171143' => "Olot",
			'171156' => "Ordis",
			'171169' => "Osor",
			'171175' => "Palafrugell",
			'171181' => "Palamós",
			'171194' => "Palau de Santa Eulàlia",
			'171215' => "Palau-sator",
			'171208' => "Palau-saverdera",
			'171236' => "Palol de Revardit",
			'171241' => "Pals",
			'171254' => "Pardines",
			'171267' => "Parlavà",
			'171289' => "Pau",
			'171292' => "Pedret i Marzà",
			'171306' => "Pera (La)",
			'171328' => "Peralada",
			'171334' => "Planes d'Hostoles (Les)",
			'171349' => "Planoles",
			'171352' => "Pont de Molins",
			'171365' => "Pontós",
			'171371' => "Porqueres",
			'171404' => "Port de la Selva (El)",
			'171387' => "Portbou",
			'171390' => "Preses (Les)",
			'171411' => "Puigcerdà",
			'171426' => "Quart",
			'170433' => "Queralbs",
			'171432' => "Rabós",
			'171447' => "Regencós",
			'171450' => "Ribes de Freser",
			'171463' => "Riells i Viabrea",
			'171479' => "Ripoll",
			'171485' => "Riudarenes",
			'171498' => "Riudaura",
			'171501' => "Riudellots de la Selva",
			'171518' => "Riumors",
			'171523' => "Roses",
			'171539' => "Rupià",
			'171544' => "Sales de Llierca",
			'171557' => "Salt",
			'171576' => "Sant Andreu Salou",
			'171833' => "Sant Aniol de Finestres",
			'171582' => "Sant Climent Sescebes",
			'171595' => "Sant Feliu de Buixalleu",
			'171609' => "Sant Feliu de Guíxols",
			'171616' => "Sant Feliu de Pallerols",
			'171621' => "Sant Ferriol",
			'171637' => "Sant Gregori",
			'171642' => "Sant Hilari Sacalm",
			'171655' => "Sant Jaume de Llierca",
			'171674' => "Sant Joan de les Abadesses",
			'171680' => "Sant Joan de Mollet",
			'171851' => "Sant Joan les Fonts",
			'171668' => "Sant Jordi Desvalls",
			'171693' => "Sant Julià de Ramis",
			'179032' => "Sant Julià del Llor i Bonmatí",
			'171714' => "Sant Llorenç de la Muga",
			'171729' => "Sant Martí de Llémena",
			'171735' => "Sant Martí Vell",
			'171740' => "Sant Miquel de Campmajor",
			'171753' => "Sant Miquel de Fluvià",
			'171766' => "Sant Mori",
			'171772' => "Sant Pau de Segúries",
			'171788' => "Sant Pere Pescador",
			'171805' => "Santa Coloma de Farners",
			'171812' => "Santa Cristina d'Aro",
			'171827' => "Santa Llogaia d'Àlguema",
			'171848' => "Santa Pau",
			'171864' => "Sarrià de Ter",
			'171870' => "Saus, Camallera i Llampaies",
			'171886' => "Selva de Mar (La)",
			'171903' => "Serinyà",
			'171910' => "Serra de Daró",
			'171925' => "Setcases",
			'171931' => "Sils",
			'170524' => "Siurana",
			'171946' => "Susqueda",
			'171959' => "Tallada d'Empordà (La)",
			'171962' => "Terrades",
			'171978' => "Torrent",
			'171984' => "Torroella de Fluvià",
			'171997' => "Torroella de Montgrí",
			'172001' => "Tortellà",
			'172018' => "Toses",
			'172023' => "Tossa de Mar",
			'172044' => "Ullà",
			'172057' => "Ullastret",
			'172039' => "Ultramort",
			'172060' => "Urús",
			'170144' => "Vajol (La)",
			'172082' => "Vall de Bianya (La)",
			'172076' => "Vall d'en Bas (La)",
			'171707' => "Vallfogona de Ripollès",
			'172095' => "Vall-llobrega",
			'172109' => "Ventalló",
			'172116' => "Verges",
			'172121' => "Vidrà",
			'172137' => "Vidreres",
			'172142' => "Vilabertran",
			'172155' => "Vilablareix",
			'172174' => "Viladamat",
			'172168' => "Viladasens",
			'172180' => "Vilademuls",
			'172207' => "Viladrau",
			'172214' => "Vilafant",
			'172235' => "Vilajuïga",
			'172240' => "Vilallonga de Ter",
			'172253' => "Vilamacolum",
			'172266' => "Vilamalla",
			'172272' => "Vilamaniscle",
			'172288' => "Vilanant",
			'172305' => "Vila-sacra",
			'172229' => "Vilaür",
			'172333' => "Vilobí d'Onyar",
			'172327' => "Vilopriu",
			'180016' => "Agrón",
			'180021' => "Alamedilla",
			'180037' => "Albolote",
			'180042' => "Albondón",
			'180055' => "Albuñán",
			'180068' => "Albuñol",
			'180074' => "Albuñuelas",
			'180107' => "Aldeire",
			'180114' => "Alfacar",
			'180129' => "Algarinejo",
			'180135' => "Alhama de Granada",
			'180140' => "Alhendín",
			'180153' => "Alicún de Ortega",
			'180166' => "Almegíjar",
			'180172' => "Almuñécar",
			'189043' => "Alpujarra de la Sierra",
			'180188' => "Alquife",
			'180205' => "Arenas del Rey",
			'180212' => "Armilla",
			'180227' => "Atarfe",
			'180233' => "Baza",
			'180248' => "Beas de Granada",
			'180251' => "Beas de Guadix",
			'180270' => "Benalúa",
			'180286' => "Benalúa de las Villas",
			'180299' => "Benamaurel",
			'180303' => "Bérchules",
			'180325' => "Bubión",
			'180331' => "Busquístar",
			'180346' => "Cacín",
			'180359' => "Cádiar",
			'180362' => "Cájar",
			'181149' => "Calahorra (La)",
			'180378' => "Calicasas",
			'180384' => "Campotéjar",
			'180397' => "Caniles",
			'180401' => "Cáñar",
			'180423' => "Capileira",
			'180439' => "Carataunas",
			'180444' => "Cástaras",
			'180457' => "Castilléjar",
			'180460' => "Castril",
			'180476' => "Cenes de la Vega",
			'180592' => "Chauchina",
			'180613' => "Chimeneas",
			'180628' => "Churriana de la Vega",
			'180482' => "Cijuela",
			'180495' => "Cogollos de Guadix",
			'180508' => "Cogollos de la Vega",
			'180515' => "Colomera",
			'180536' => "Cortes de Baza",
			'180541' => "Cortes y Graena",
			'189120' => "Cuevas del Campo",
			'180567' => "Cúllar",
			'180573' => "Cúllar Vega",
			'180634' => "Darro",
			'180649' => "Dehesas de Guadix",
			'180665' => "Deifontes",
			'180671' => "Diezma",
			'180687' => "Dílar",
			'180690' => "Dólar",
			'180704' => "Dúdar",
			'180711' => "Dúrcal",
			'180726' => "Escúzar",
			'180747' => "Ferreira",
			'180763' => "Fonelas",
			'180785' => "Freila",
			'180798' => "Fuente Vaqueros",
			'189056' => "Gabias (Las)",
			'180824' => "Galera",
			'180830' => "Gobernador",
			'180845' => "Gójar",
			'180858' => "Gor",
			'180861' => "Gorafe",
			'180877' => "Granada",
			'180883' => "Guadahortuna",
			'180896' => "Guadix",
			'189069' => "Guajares (Los)",
			'180938' => "Gualchos",
			'180943' => "Güejar Sierra",
			'180956' => "Güevéjar",
			'180969' => "Huélago",
			'180975' => "Huéneja",
			'180981' => "Huéscar",
			'180994' => "Huétor de Santillán",
			'181008' => "Huétor Tájar",
			'181015' => "Huétor Vega",
			'181020' => "Illora",
			'181036' => "Itrabo",
			'181054' => "Iznalloz",
			'181073' => "Jayena",
			'181089' => "Jerez del Marquesado",
			'181092' => "Jete",
			'181113' => "Jun",
			'181128' => "Juviles",
			'181152' => "Láchar",
			'181165' => "Lanjarón",
			'181171' => "Lanteira",
			'181190' => "Lecrín",
			'181204' => "Lentegí",
			'181211' => "Lobras",
			'181226' => "Loja",
			'181232' => "Lugros",
			'181247' => "Lújar",
			'181263' => "Malahá (La)",
			'181279' => "Maracena",
			'181285' => "Marchal",
			'181324' => "Moclín",
			'181330' => "Molvízar",
			'181345' => "Monachil",
			'181358' => "Montefrío",
			'181361' => "Montejícar",
			'181377' => "Montillana",
			'181383' => "Moraleda de Zafayona",
			'189094' => "Morelábor",
			'181400' => "Motril",
			'181417' => "Murtas",
			'189038' => "Nevada",
			'181438' => "Nigüelas",
			'181443' => "Nívar",
			'181456' => "Ogíjares",
			'181469' => "Orce",
			'181475' => "Órgiva",
			'181481' => "Otívar",
			'181494' => "Otura",
			'181507' => "Padul",
			'181514' => "Pampaneira",
			'181529' => "Pedro Martínez",
			'181535' => "Peligros",
			'181540' => "Peza (La)",
			'189108' => "Pinar (El)",
			'181572' => "Pinos Genil",
			'181588' => "Pinos Puente",
			'181591' => "Píñar",
			'181612' => "Polícar",
			'181627' => "Polopos",
			'181633' => "Pórtugos",
			'181648' => "Puebla de Don Fadrique",
			'181651' => "Pulianas",
			'181670' => "Purullena",
			'181686' => "Quéntar",
			'181703' => "Rubite",
			'181710' => "Salar",
			'181731' => "Salobreña",
			'181746' => "Santa Cruz del Comercio",
			'181759' => "Santa Fe",
			'181762' => "Soportújar",
			'181778' => "Sorvilán",
			'189017' => "Taha (La)",
			'181784' => "Torre-Cardela",
			'181797' => "Torvizcón",
			'181801' => "Trevélez",
			'181818' => "Turón",
			'181823' => "Ugíjar",
			'189022' => "Valle (El)",
			'189075' => "Valle del Zalabí",
			'181839' => "Válor",
			'189115' => "Vegas del Genil",
			'181844' => "Vélez de Benaudalla",
			'181857' => "Ventas de Huelma",
			'189081' => "Villamena",
			'181876' => "Villanueva de las Torres",
			'181882' => "Villanueva Mesía",
			'181895' => "Víznar",
			'181921' => "Zafarraya",
			'189136' => "Zagra",
			'181937' => "Zubia (La)",
			'181942' => "Zújar",
			'190019' => "Abánades",
			'190024' => "Ablanque",
			'190030' => "Adobes",
			'190045' => "Alaminos",
			'190058' => "Alarilla",
			'190061' => "Albalate de Zorita",
			'190077' => "Albares",
			'190083' => "Albendiego",
			'190096' => "Alcocer",
			'190100' => "Alcolea de las Peñas",
			'190117' => "Alcolea del Pinar",
			'190138' => "Alcoroches",
			'190156' => "Aldeanueva de Guadalajara",
			'190169' => "Algar de Mesa",
			'190175' => "Algora",
			'190181' => "Alhóndiga",
			'190194' => "Alique",
			'190208' => "Almadrones",
			'190215' => "Almoguera",
			'190220' => "Almonacid de Zorita",
			'190236' => "Alocén",
			'190241' => "Alovera",
			'190273' => "Alustante",
			'190313' => "Angón",
			'190328' => "Anguita",
			'190334' => "Anquela del Ducado",
			'190349' => "Anquela del Pedregal",
			'190365' => "Aranzueque",
			'190371' => "Arbancón",
			'190387' => "Arbeteta",
			'190390' => "Argecilla",
			'190404' => "Armallones",
			'190411' => "Armuña de Tajuña",
			'190426' => "Arroyo de las Fraguas",
			'190432' => "Atanzón",
			'190447' => "Atienza",
			'190450' => "Auñón",
			'190463' => "Azuqueca de Henares",
			'190479' => "Baides",
			'190485' => "Baños de Tajo",
			'190498' => "Bañuelos",
			'190501' => "Barriopedro",
			'190518' => "Berninches",
			'190523' => "Bodera (La)",
			'190539' => "Brihuega",
			'190544' => "Budia",
			'190557' => "Bujalaro",
			'190576' => "Bustares",
			'190582' => "Cabanillas del Campo",
			'190595' => "Campillo de Dueñas",
			'190609' => "Campillo de Ranas",
			'190616' => "Campisábalos",
			'190642' => "Canredondo",
			'190655' => "Cantalojas",
			'190668' => "Cañizar",
			'190674' => "Cardoso de la Sierra (El)",
			'190707' => "Casa de Uceda",
			'190714' => "Casar (El)",
			'190735' => "Casas de San Galindo",
			'190740' => "Caspueñas",
			'190753' => "Castejón de Henares",
			'190766' => "Castellar de la Muela",
			'190788' => "Castilforte",
			'190791' => "Castilnuevo",
			'190805' => "Cendejas de Enmedio",
			'190812' => "Cendejas de la Torre",
			'190827' => "Centenera",
			'191039' => "Checa",
			'191044' => "Chequilla",
			'191060' => "Chillarón del Rey",
			'191057' => "Chiloeches",
			'190864' => "Cifuentes",
			'190870' => "Cincovillas",
			'190886' => "Ciruelas",
			'190899' => "Ciruelos del Pinar",
			'190903' => "Cobeta",
			'190910' => "Cogollor",
			'190925' => "Cogolludo",
			'190959' => "Condemios de Abajo",
			'190962' => "Condemios de Arriba",
			'190978' => "Congostrina",
			'190984' => "Copernal",
			'190997' => "Corduente",
			'191023' => "Cubillo de Uceda (El)",
			'191076' => "Driebes",
			'191082' => "Durón",
			'191095' => "Embid",
			'191109' => "Escamilla",
			'191116' => "Escariche",
			'191121' => "Escopete",
			'191137' => "Espinosa de Henares",
			'191142' => "Esplegares",
			'191155' => "Establés",
			'191168' => "Estriégana",
			'191174' => "Fontanar",
			'191180' => "Fuembellida",
			'191193' => "Fuencemillán",
			'191207' => "Fuentelahiguera de Albatages",
			'191214' => "Fuentelencina",
			'191229' => "Fuentelsaz",
			'191235' => "Fuentelviejo",
			'191240' => "Fuentenovilla",
			'191253' => "Gajanejos",
			'191266' => "Galápagos",
			'191272' => "Galve de Sorbe",
			'191291' => "Gascueña de Bornova",
			'191305' => "Guadalajara",
			'191327' => "Henche",
			'191333' => "Heras de Ayuso",
			'191348' => "Herrería",
			'191351' => "Hiendelaencina",
			'191364' => "Hijes",
			'191386' => "Hita",
			'191399' => "Hombrados",
			'191425' => "Hontoba",
			'191431' => "Horche",
			'191459' => "Hortezuela de Océn",
			'191462' => "Huerce (La)",
			'191478' => "Huérmeces del Cerro",
			'191484' => "Huertahernando",
			'191500' => "Hueva",
			'191517' => "Humanes",
			'191522' => "Illana",
			'191538' => "Iniéstola",
			'191543' => "Inviernas (Las)",
			'191556' => "Irueste",
			'191569' => "Jadraque",
			'191575' => "Jirueque",
			'191594' => "Ledanca",
			'191608' => "Loranca de Tajuña",
			'191615' => "Lupiana",
			'191620' => "Luzaga",
			'191636' => "Luzón",
			'191654' => "Majaelrayo",
			'191667' => "Málaga del Fresno",
			'191673' => "Malaguilla",
			'191689' => "Mandayona",
			'191692' => "Mantiel",
			'191706' => "Maranchón",
			'191713' => "Marchamalo",
			'191728' => "Masegoso de Tajuña",
			'191734' => "Matarrubia",
			'191749' => "Matillas",
			'191752' => "Mazarete",
			'191765' => "Mazuecos",
			'191771' => "Medranda",
			'191787' => "Megina",
			'191790' => "Membrillera",
			'191811' => "Miedes de Atienza",
			'191826' => "Mierla (La)",
			'191847' => "Millana",
			'191832' => "Milmarcos",
			'191850' => "Miñosa (La)",
			'191863' => "Mirabueno",
			'191879' => "Miralrío",
			'191885' => "Mochales",
			'191898' => "Mohernando",
			'191902' => "Molina de Aragón",
			'191919' => "Monasterio",
			'191924' => "Mondéjar",
			'191930' => "Montarrón",
			'191945' => "Moratilla de los Meleros",
			'191958' => "Morenilla",
			'191961' => "Muduex",
			'191977' => "Navas de Jadraque (Las)",
			'191983' => "Negredo",
			'191996' => "Ocentejo",
			'192000' => "Olivar (El)",
			'192017' => "Olmeda de Cobeta",
			'192022' => "Olmeda de Jadraque (La)",
			'192038' => "Ordial (El)",
			'192043' => "Orea",
			'192081' => "Pálmaces de Jadraque",
			'192094' => "Pardos",
			'192108' => "Paredes de Sigüenza",
			'192115' => "Pareja",
			'192120' => "Pastrana",
			'192136' => "Pedregal (El)",
			'192141' => "Peñalén",
			'192154' => "Peñalver",
			'192167' => "Peralejos de las Truchas",
			'192173' => "Peralveche",
			'192189' => "Pinilla de Jadraque",
			'192192' => "Pinilla de Molina",
			'192206' => "Pioz",
			'192213' => "Piqueras",
			'192228' => "Pobo de Dueñas (El)",
			'192234' => "Poveda de la Sierra",
			'192249' => "Pozo de Almoguera",
			'192252' => "Pozo de Guadalajara",
			'192265' => "Prádena de Atienza",
			'192271' => "Prados Redondos",
			'192287' => "Puebla de Beleña",
			'192290' => "Puebla de Valles",
			'192304' => "Quer",
			'192311' => "Rebollosa de Jadraque",
			'192326' => "Recuenco (El)",
			'192332' => "Renera",
			'192347' => "Retiendas",
			'192350' => "Riba de Saelices",
			'192379' => "Rillo de Gallo",
			'192385' => "Riofrío del Llano",
			'192398' => "Robledillo de Mohernando",
			'192402' => "Robledo de Corpes",
			'192419' => "Romanillos de Atienza",
			'192424' => "Romanones",
			'192430' => "Rueda de la Sierra",
			'192445' => "Sacecorbo",
			'192458' => "Sacedón",
			'192461' => "Saelices de la Sal",
			'192477' => "Salmerón",
			'192483' => "San Andrés del Congosto",
			'192496' => "San Andrés del Rey",
			'192509' => "Santiuste",
			'192516' => "Saúca",
			'192521' => "Sayatón",
			'192542' => "Selas",
			'199010' => "Semillas",
			'192555' => "Setiles",
			'192568' => "Sienes",
			'192574' => "Sigüenza",
			'192580' => "Solanillos del Extremo",
			'192593' => "Somolinos",
			'192607' => "Sotillo (El)",
			'192614' => "Sotodosos",
			'192629' => "Tamajón",
			'192635' => "Taragudo",
			'192640' => "Taravilla",
			'192653' => "Tartanedo",
			'192666' => "Tendilla",
			'192672' => "Terzaga",
			'192688' => "Tierzo",
			'192691' => "Toba (La)",
			'192712' => "Tordellego",
			'192705' => "Tordelrábano",
			'192727' => "Tordesilos",
			'192748' => "Torija",
			'192799' => "Torre del Burgo",
			'192770' => "Torrecuadrada de Molina",
			'192786' => "Torrecuadradilla",
			'192803' => "Torrejón del Rey",
			'192810' => "Torremocha de Jadraque",
			'192825' => "Torremocha del Campo",
			'192831' => "Torremocha del Pinar",
			'192846' => "Torremochuela",
			'192859' => "Torrubia",
			'192862' => "Tórtola de Henares",
			'192878' => "Tortuera",
			'192884' => "Tortuero",
			'192897' => "Traíd",
			'192901' => "Trijueque",
			'192918' => "Trillo",
			'192939' => "Uceda",
			'192944' => "Ujados",
			'192960' => "Utande",
			'192976' => "Valdarachas",
			'192982' => "Valdearenas",
			'192995' => "Valdeavellano",
			'193009' => "Valdeaveruelo",
			'193016' => "Valdeconcha",
			'193021' => "Valdegrudas",
			'193037' => "Valdelcubo",
			'193042' => "Valdenuño Fernández",
			'193055' => "Valdepeñas de la Sierra",
			'193068' => "Valderrebollo",
			'193074' => "Valdesotos",
			'193080' => "Valfermoso de Tajuña",
			'193093' => "Valhermoso",
			'193107' => "Valtablado del Río",
			'193114' => "Valverde de los Arroyos",
			'193140' => "Viana de Jadraque",
			'193172' => "Villanueva de Alcorón",
			'193188' => "Villanueva de Argecilla",
			'193191' => "Villanueva de la Torre",
			'193212' => "Villares de Jadraque",
			'193227' => "Villaseca de Henares",
			'193233' => "Villaseca de Uceda",
			'193248' => "Villel de Mesa",
			'193251' => "Viñuelas",
			'193264' => "Yebes",
			'193270' => "Yebra",
			'193299' => "Yélamos de Abajo",
			'193303' => "Yélamos de Arriba",
			'193310' => "Yunquera de Henares",
			'193325' => "Yunta (La)",
			'193331' => "Zaorejas",
			'193346' => "Zarzuela de Jadraque",
			'193359' => "Zorita de los Canes",
			'200013' => "Abaltzisketa",
			'200028' => "Aduna",
			'200163' => "Aia",
			'200034' => "Aizarnazabal",
			'200049' => "Albiztur",
			'200052' => "Alegia",
			'200065' => "Alkiza",
			'209066' => "Altzaga",
			'200071' => "Altzo",
			'200087' => "Amezketa",
			'200090' => "Andoain",
			'200104' => "Anoeta",
			'200111' => "Antzuola",
			'200126' => "Arama",
			'200132' => "Aretxabaleta",
			'200551' => "Arrasate/Mondragón",
			'200147' => "Asteasu",
			'209035' => "Astigarraga",
			'200150' => "Ataun",
			'200179' => "Azkoitia",
			'200185' => "Azpeitia",
			'209040' => "Baliarrain",
			'200198' => "Beasain",
			'200202' => "Beizama",
			'200219' => "Belauntza",
			'200224' => "Berastegi",
			'200744' => "Bergara",
			'200230' => "Berrobi",
			'200245' => "Bidegoian",
			'200296' => "Deba",
			'200697' => "Donostia-San Sebastián",
			'200300' => "Eibar",
			'200317' => "Elduain",
			'200338' => "Elgeta",
			'200322' => "Elgoibar",
			'200678' => "Errenteria",
			'200662' => "Errezil",
			'200343' => "Eskoriatza",
			'200356' => "Ezkio-Itsaso",
			'200381' => "Gabiria",
			'200375' => "Gaintza",
			'209072' => "Gaztelu",
			'200394' => "Getaria",
			'200408' => "Hernani",
			'200415' => "Hernialde",
			'200369' => "Hondarribia",
			'200420' => "Ibarra",
			'200436' => "Idiazabal",
			'200441' => "Ikaztegieta",
			'200454' => "Irun",
			'200467' => "Irura",
			'200473' => "Itsasondo",
			'200489' => "Larraul",
			'209029' => "Lasarte-Oria",
			'200492' => "Lazkao",
			'200505' => "Leaburu",
			'200512' => "Legazpi",
			'200527' => "Legorreta",
			'200684' => "Leintz-Gatzaga",
			'200533' => "Lezo",
			'200548' => "Lizartza",
			'209014' => "Mendaro",
			'200570' => "Mutiloa",
			'200564' => "Mutriku",
			'200631' => "Oiartzun",
			'200586' => "Olaberria",
			'200599' => "Oñati",
			'200760' => "Ordizia",
			'209053' => "Orendain",
			'200603' => "Orexa",
			'200610' => "Orio",
			'200625' => "Ormaiztegi",
			'200646' => "Pasaia",
			'200701' => "Segura",
			'200659' => "Soraluze/Placencia de las Armas",
			'200718' => "Tolosa",
			'200723' => "Urnieta",
			'200776' => "Urretxu",
			'200739' => "Usurbil",
			'200757' => "Villabona",
			'200782' => "Zaldibia",
			'200795' => "Zarautz",
			'200258' => "Zegama",
			'200261' => "Zerain",
			'200277' => "Zestoa",
			'200283' => "Zizurkil",
			'200816' => "Zumaia",
			'200809' => "Zumarraga",
			'210010' => "Alájar",
			'210025' => "Aljaraque",
			'210031' => "Almendro (El)",
			'210046' => "Almonaster la Real",
			'210059' => "Almonte",
			'210062' => "Alosno",
			'210078' => "Aracena",
			'210084' => "Aroche",
			'210097' => "Arroyomolinos de León",
			'210101' => "Ayamonte",
			'210118' => "Beas",
			'210123' => "Berrocal",
			'210139' => "Bollullos Par del Condado",
			'210144' => "Bonares",
			'210157' => "Cabezas Rubias",
			'210160' => "Cala",
			'210176' => "Calañas",
			'210182' => "Campillo (El)",
			'210195' => "Campofrío",
			'210209' => "Cañaveral de León",
			'210216' => "Cartaya",
			'210221' => "Castaño del Robledo",
			'210237' => "Cerro de Andévalo (El)",
			'210307' => "Chucena",
			'210242' => "Corteconcepción",
			'210255' => "Cortegana",
			'210268' => "Cortelazor",
			'210274' => "Cumbres de Enmedio",
			'210280' => "Cumbres de San Bartolomé",
			'210293' => "Cumbres Mayores",
			'210314' => "Encinasola",
			'210329' => "Escacena del Campo",
			'210335' => "Fuenteheridos",
			'210340' => "Galaroza",
			'210353' => "Gibraleón",
			'210366' => "Granada de Río-Tinto (La)",
			'210372' => "Granado (El)",
			'210388' => "Higuera de la Sierra",
			'210391' => "Hinojales",
			'210405' => "Hinojos",
			'210412' => "Huelva",
			'210427' => "Isla Cristina",
			'210433' => "Jabugo",
			'210448' => "Lepe",
			'210451' => "Linares de la Sierra",
			'210464' => "Lucena del Puerto",
			'210470' => "Manzanilla",
			'210486' => "Marines (Los)",
			'210499' => "Minas de Riotinto",
			'210502' => "Moguer",
			'210519' => "Nava (La)",
			'210524' => "Nerva",
			'210530' => "Niebla",
			'210545' => "Palma del Condado (La)",
			'210558' => "Palos de la Frontera",
			'210561' => "Paterna del Campo",
			'210577' => "Paymogo",
			'210583' => "Puebla de Guzmán",
			'210596' => "Puerto Moral",
			'210600' => "Punta Umbría",
			'210617' => "Rociana del Condado",
			'210622' => "Rosal de la Frontera",
			'210638' => "San Bartolomé de la Torre",
			'210643' => "San Juan del Puerto",
			'210669' => "San Silvestre de Guzmán",
			'210656' => "Sanlúcar de Guadiana",
			'210675' => "Santa Ana la Real",
			'210681' => "Santa Bárbara de Casa",
			'210694' => "Santa Olalla del Cala",
			'210708' => "Trigueros",
			'210715' => "Valdelarco",
			'210720' => "Valverde del Camino",
			'210736' => "Villablanca",
			'210741' => "Villalba del Alcor",
			'210754' => "Villanueva de las Cruces",
			'210767' => "Villanueva de los Castillejos",
			'210773' => "Villarrasa",
			'210789' => "Zalamea la Real",
			'210792' => "Zufre",
			'220015' => "Abiego",
			'220020' => "Abizanda",
			'220036' => "Adahuesca",
			'220041' => "Agüero",
			'229074' => "Aínsa-Sobrarbe",
			'220067' => "Aisa",
			'220073' => "Albalate de Cinca",
			'220089' => "Albalatillo",
			'220092' => "Albelda",
			'220113' => "Albero Alto",
			'220128' => "Albero Bajo",
			'220134' => "Alberuela de Tubo",
			'220149' => "Alcalá de Gurrea",
			'220152' => "Alcalá del Obispo",
			'220165' => "Alcampell",
			'220171' => "Alcolea de Cinca",
			'220187' => "Alcubierre",
			'220190' => "Alerre",
			'220204' => "Alfántega",
			'220211' => "Almudévar",
			'220226' => "Almunia de San Juan",
			'220232' => "Almuniente",
			'220247' => "Alquézar",
			'220250' => "Altorricón",
			'220279' => "Angüés",
			'220285' => "Ansó",
			'220298' => "Antillón",
			'220324' => "Aragüés del Puerto",
			'220358' => "Arén",
			'220361' => "Argavieso",
			'220377' => "Arguis",
			'220396' => "Ayerbe",
			'220400' => "Azanuy-Alins",
			'220417' => "Azara",
			'220422' => "Azlor",
			'220438' => "Baélls",
			'220443' => "Bailo",
			'220456' => "Baldellou",
			'220469' => "Ballobar",
			'220475' => "Banastás",
			'220481' => "Barbastro",
			'220494' => "Barbués",
			'220507' => "Barbuñales",
			'220514' => "Bárcabo",
			'220529' => "Belver de Cinca",
			'220535' => "Benabarre",
			'220540' => "Benasque",
			'220553' => "Berbegal",
			'220572' => "Bielsa",
			'220588' => "Bierge",
			'220591' => "Biescas",
			'220605' => "Binaced",
			'220612' => "Binéfar",
			'220627' => "Bisaurri",
			'220633' => "Biscarrués",
			'220648' => "Blecua y Torres",
			'220664' => "Boltaña",
			'220670' => "Bonansa",
			'220686' => "Borau",
			'220699' => "Broto",
			'220725' => "Caldearenas",
			'220746' => "Campo",
			'220759' => "Camporrélls",
			'220762' => "Canal de Berdún",
			'220778' => "Candasnos",
			'220784' => "Canfranc",
			'220797' => "Capdesaso",
			'220801' => "Capella",
			'220818' => "Casbas de Huesca",
			'220839' => "Castejón de Monegros",
			'220844' => "Castejón de Sos",
			'220823' => "Castejón del Puente",
			'220857' => "Castelflorite",
			'220860' => "Castiello de Jaca",
			'220876' => "Castigaleu",
			'220882' => "Castillazuelo",
			'220895' => "Castillonroy",
			'220942' => "Chalamera",
			'220955' => "Chía",
			'220968' => "Chimillas",
			'220909' => "Colungo",
			'220993' => "Esplús",
			'221029' => "Estada",
			'221035' => "Estadilla",
			'221053' => "Estopiñán del Castillo",
			'221066' => "Fago",
			'221072' => "Fanlo",
			'221091' => "Fiscal",
			'221105' => "Fonz",
			'221112' => "Foradada del Toscar",
			'221127' => "Fraga",
			'221133' => "Fueva (La)",
			'221148' => "Gistaín",
			'221151' => "Grado (El)",
			'221164' => "Grañén",
			'221170' => "Graus",
			'221199' => "Gurrea de Gállego",
			'221225' => "Hoz de Jaca",
			'229080' => "Hoz y Costean",
			'221246' => "Huerto",
			'221259' => "Huesca",
			'221262' => "Ibieca",
			'221278' => "Igriés",
			'221284' => "Ilche",
			'221297' => "Isábena",
			'221301' => "Jaca",
			'221318' => "Jasa",
			'221339' => "Labuerda",
			'221357' => "Laluenga",
			'221360' => "Lalueza",
			'221376' => "Lanaja",
			'221395' => "Laperdiguera",
			'221416' => "Lascellas-Ponzano",
			'221421' => "Lascuarre",
			'221437' => "Laspaúles",
			'221442' => "Laspuña",
			'221493' => "Loarre",
			'221506' => "Loporzano",
			'221513' => "Loscorrales",
			'229055' => "Lupiñén-Ortilla",
			'221552' => "Monesma y Cajigar",
			'221565' => "Monflorite-Lascasas",
			'221571' => "Montanuy",
			'221587' => "Monzón",
			'221604' => "Naval",
			'221626' => "Novales",
			'221632' => "Nueno",
			'221647' => "Olvena",
			'221650' => "Ontiñena",
			'221679' => "Osso de Cinca",
			'221685' => "Palo",
			'221702' => "Panticosa",
			'221724' => "Peñalba",
			'221730' => "Peñas de Riglos (Las)",
			'221745' => "Peralta de Alcofea",
			'221758' => "Peralta de Calasanz",
			'221761' => "Peraltilla",
			'221777' => "Perarrúa",
			'221783' => "Pertusa",
			'221817' => "Piracés",
			'221822' => "Plan",
			'221843' => "Poleñino",
			'221869' => "Pozán de Vero",
			'221875' => "Puebla de Castro (La)",
			'221881' => "Puente de Montañana",
			'229021' => "Puente la Reina de Jaca",
			'221894' => "Puértolas",
			'221908' => "Pueyo de Araguás (El)",
			'221936' => "Pueyo de Santa Cruz",
			'221954' => "Quicena",
			'221973' => "Robres",
			'221992' => "Sabiñánigo",
			'222006' => "Sahún",
			'222013' => "Salas Altas",
			'222028' => "Salas Bajas",
			'222034' => "Salillas",
			'222049' => "Sallent de Gállego",
			'222052' => "San Esteban de Litera",
			'222071' => "San Juan de Plan",
			'229037' => "San Miguel del Cinca",
			'222065' => "Sangarrén",
			'222087' => "Santa Cilia",
			'222090' => "Santa Cruz de la Serós",
			'229068' => "Santa María de Dulcis",
			'222126' => "Santaliestra y San Quílez",
			'222132' => "Sariñena",
			'222147' => "Secastilla",
			'222150' => "Seira",
			'222179' => "Sena",
			'222185' => "Senés de Alcubierre",
			'222202' => "Sesa",
			'222219' => "Sesué",
			'222224' => "Siétamo",
			'222230' => "Sopeira",
			'229042' => "Sotonera (La)",
			'222258' => "Tamarite de Litera",
			'222261' => "Tardienta",
			'222277' => "Tella-Sin",
			'222283' => "Tierz",
			'222296' => "Tolva",
			'222300' => "Torla",
			'222322' => "Torralba de Aragón",
			'222338' => "Torre la Ribera",
			'222343' => "Torrente de Cinca",
			'222356' => "Torres de Alcanadre",
			'222369' => "Torres de Barbués",
			'222394' => "Tramaced",
			'222420' => "Valfarta",
			'222436' => "Valle de Bardají",
			'229016' => "Valle de Hecho",
			'222441' => "Valle de Lierp",
			'222454' => "Velilla de Cinca",
			'229093' => "Vencillón",
			'222467' => "Veracruz",
			'222473' => "Viacamp y Litera",
			'222489' => "Vicién",
			'222492' => "Villanova",
			'222505' => "Villanúa",
			'222512' => "Villanueva de Sigena",
			'222527' => "Yebra de Basa",
			'222533' => "Yésero",
			'222548' => "Zaidín",
			'230011' => "Albanchez de Mágina",
			'230026' => "Alcalá la Real",
			'230032' => "Alcaudete",
			'230047' => "Aldeaquemada",
			'230050' => "Andújar",
			'230063' => "Arjona",
			'230079' => "Arjonilla",
			'230085' => "Arquillos",
			'239051' => "Arroyo del Ojanco",
			'230098' => "Baeza",
			'230102' => "Bailén",
			'230119' => "Baños de la Encina",
			'230124' => "Beas de Segura",
			'239027' => "Bedmar y Garcíez",
			'230145' => "Begíjar",
			'230158' => "Bélmez de la Moraleda",
			'230161' => "Benatae",
			'230177' => "Cabra del Santo Cristo",
			'230183' => "Cambil",
			'230196' => "Campillo de Arenas",
			'230200' => "Canena",
			'230217' => "Carboneros",
			'239012' => "Cárcheles",
			'230243' => "Carolina (La)",
			'230256' => "Castellar",
			'230269' => "Castillo de Locubín",
			'230275' => "Cazalilla",
			'230281' => "Cazorla",
			'230294' => "Chiclana de Segura",
			'230308' => "Chilluévar",
			'230315' => "Escañuela",
			'230320' => "Espelúy",
			'230336' => "Frailes",
			'230341' => "Fuensanta de Martos",
			'230354' => "Fuerte del Rey",
			'230373' => "Génave",
			'230389' => "Guardia de Jaén (La)",
			'230392' => "Guarromán",
			'230413' => "Higuera de Calatrava",
			'230428' => "Hinojares",
			'230434' => "Hornos",
			'230449' => "Huelma",
			'230452' => "Huesa",
			'230465' => "Ibros",
			'230471' => "Iruela (La)",
			'230487' => "Iznatoraf",
			'230490' => "Jabalquinto",
			'230503' => "Jaén",
			'230510' => "Jamilena",
			'230525' => "Jimena",
			'230531' => "Jódar",
			'230406' => "Lahiguera",
			'230546' => "Larva",
			'230559' => "Linares",
			'230562' => "Lopera",
			'230578' => "Lupión",
			'230584' => "Mancha Real",
			'230597' => "Marmolejo",
			'230601' => "Martos",
			'230618' => "Mengíbar",
			'230623' => "Montizón",
			'230639' => "Navas de San Juan",
			'230644' => "Noalejo",
			'230657' => "Orcera",
			'230660' => "Peal de Becerro",
			'230676' => "Pegalajar",
			'230695' => "Porcuna",
			'230709' => "Pozo Alcón",
			'230716' => "Puente de Génave",
			'230721' => "Puerta de Segura (La)",
			'230737' => "Quesada",
			'230742' => "Rus",
			'230755' => "Sabiote",
			'230768' => "Santa Elena",
			'230774' => "Santiago de Calatrava",
			'239048' => "Santiago-Pontones",
			'230793' => "Santisteban del Puerto",
			'230807' => "Santo Tomé",
			'230814' => "Segura de la Sierra",
			'230829' => "Siles",
			'230840' => "Sorihuela del Guadalimar",
			'230866' => "Torre del Campo",
			'230853' => "Torreblascopedro",
			'230872' => "Torredonjimeno",
			'230888' => "Torreperogil",
			'230905' => "Torres",
			'230912' => "Torres de Albánchez",
			'230927' => "Úbeda",
			'230933' => "Valdepeñas de Jaén",
			'230948' => "Vilches",
			'230951' => "Villacarrillo",
			'230964' => "Villanueva de la Reina",
			'230970' => "Villanueva del Arzobispo",
			'230986' => "Villardompardo",
			'230999' => "Villares (Los)",
			'231010' => "Villarrodrigo",
			'239033' => "Villatorres",
			'240016' => "Acebedo",
			'240021' => "Algadefe",
			'240037' => "Alija del Infantado",
			'240042' => "Almanza",
			'240055' => "Antigua (La)",
			'240068' => "Ardón",
			'240074' => "Arganza",
			'240080' => "Astorga",
			'240093' => "Balboa",
			'240107' => "Bañeza (La)",
			'240114' => "Barjas",
			'240129' => "Barrios de Luna (Los)",
			'240140' => "Bembibre",
			'240153' => "Benavides",
			'240166' => "Benuza",
			'240172' => "Bercianos del Páramo",
			'240188' => "Bercianos del Real Camino",
			'240191' => "Berlanga del Bierzo",
			'240205' => "Boca de Huérgano",
			'240212' => "Boñar",
			'240227' => "Borrenes",
			'240233' => "Brazuelo",
			'240248' => "Burgo Ranero (El)",
			'240251' => "Burón",
			'240264' => "Bustillo del Páramo",
			'240270' => "Cabañas Raras",
			'240286' => "Cabreros del Río",
			'240299' => "Cabrillanes",
			'240303' => "Cacabelos",
			'240310' => "Calzada del Coto",
			'240325' => "Campazas",
			'240331' => "Campo de Villavidel",
			'240346' => "Camponaraya",
			'240362' => "Candín",
			'240378' => "Cármenes",
			'240384' => "Carracedelo",
			'240397' => "Carrizo",
			'240401' => "Carrocera",
			'240418' => "Carucedo",
			'240423' => "Castilfalé",
			'240439' => "Castrillo de Cabrera",
			'240444' => "Castrillo de la Valduerna",
			'240460' => "Castrocalbón",
			'240476' => "Castrocontrigo",
			'240495' => "Castropodame",
			'240508' => "Castrotierra de Valmadrigal",
			'240515' => "Cea",
			'240520' => "Cebanico",
			'240536' => "Cebrones del Río",
			'240652' => "Chozas de Abajo",
			'240541' => "Cimanes de la Vega",
			'240554' => "Cimanes del Tejar",
			'240567' => "Cistierna",
			'240573' => "Congosto",
			'240589' => "Corbillos de los Oteros",
			'240592' => "Corullón",
			'240606' => "Crémenes",
			'240613' => "Cuadros",
			'240628' => "Cubillas de los Oteros",
			'240634' => "Cubillas de Rueda",
			'240649' => "Cubillos del Sil",
			'240665' => "Destriana",
			'240671' => "Encinedo",
			'240687' => "Ercina (La)",
			'240690' => "Escobar de Campos",
			'240704' => "Fabero",
			'240711' => "Folgoso de la Ribera",
			'240732' => "Fresno de la Vega",
			'240747' => "Fuentes de Carbajal",
			'240763' => "Garrafe de Torío",
			'240779' => "Gordaliza del Pino",
			'240785' => "Gordoncillo",
			'240798' => "Gradefes",
			'240802' => "Grajal de Campos",
			'240819' => "Gusendos de los Oteros",
			'240824' => "Hospital de Órbigo",
			'240830' => "Igüeña",
			'240845' => "Izagre",
			'240861' => "Joarilla de las Matas",
			'240877' => "Laguna Dalga",
			'240883' => "Laguna de Negrillos",
			'240896' => "León",
			'240922' => "Llamas de la Ribera",
			'240900' => "Lucillo",
			'240917' => "Luyego",
			'240938' => "Magaz de Cepeda",
			'240943' => "Mansilla de las Mulas",
			'240956' => "Mansilla Mayor",
			'240969' => "Maraña",
			'240975' => "Matadeón de los Oteros",
			'240981' => "Matallana de Torío",
			'240994' => "Matanza",
			'241008' => "Molinaseca",
			'241015' => "Murias de Paredes",
			'241020' => "Noceda del Bierzo",
			'241036' => "Oencia",
			'241041' => "Omañas (Las)",
			'241054' => "Onzonilla",
			'241067' => "Oseja de Sajambre",
			'241073' => "Pajares de los Oteros",
			'241089' => "Palacios de la Valduerna",
			'241092' => "Palacios del Sil",
			'241106' => "Páramo del Sil",
			'241128' => "Peranzanes",
			'241134' => "Pobladura de Pelayo García",
			'241149' => "Pola de Gordón (La)",
			'241152' => "Ponferrada",
			'241165' => "Posada de Valdeón",
			'241171' => "Pozuelo del Páramo",
			'241187' => "Prado de la Guzpeña",
			'241190' => "Priaranza del Bierzo",
			'241204' => "Prioro",
			'241211' => "Puebla de Lillo",
			'241226' => "Puente de Domingo Flórez",
			'241232' => "Quintana del Castillo",
			'241247' => "Quintana del Marco",
			'241250' => "Quintana y Congosto",
			'241279' => "Regueras de Arriba",
			'241298' => "Reyero",
			'241302' => "Riaño",
			'241319' => "Riego de la Vega",
			'241324' => "Riello",
			'241330' => "Rioseco de Tapia",
			'241345' => "Robla (La)",
			'241361' => "Roperuelos del Páramo",
			'241377' => "Sabero",
			'241396' => "Sahagún",
			'241417' => "San Adrián del Valle",
			'241422' => "San Andrés del Rabanedo",
			'241443' => "San Cristóbal de la Polantera",
			'241456' => "San Emiliano",
			'241469' => "San Esteban de Nogales",
			'241481' => "San Justo de la Vega",
			'241494' => "San Millán de los Caballeros",
			'241507' => "San Pedro Bercianos",
			'241438' => "Sancedo",
			'241514' => "Santa Colomba de Curueño",
			'241529' => "Santa Colomba de Somoza",
			'241535' => "Santa Cristina de Valmadrigal",
			'241540' => "Santa Elena de Jamuz",
			'241553' => "Santa María de la Isla",
			'241588' => "Santa María de Ordás",
			'241566' => "Santa María del Monte de Cea",
			'241572' => "Santa María del Páramo",
			'241591' => "Santa Marina del Rey",
			'241605' => "Santas Martas",
			'241612' => "Santiago Millas",
			'241627' => "Santovenia de la Valdoncina",
			'241633' => "Sariegos",
			'241648' => "Sena de Luna",
			'241651' => "Sobrado",
			'241664' => "Soto de la Vega",
			'241670' => "Soto y Amío",
			'241686' => "Toral de los Guzmanes",
			'241699' => "Toreno",
			'241703' => "Torre del Bierzo",
			'241710' => "Trabadelo",
			'241725' => "Truchas",
			'241731' => "Turcia",
			'241746' => "Urdiales del Páramo",
			'241857' => "Val de San Lorenzo",
			'241759' => "Valdefresno",
			'241762' => "Valdefuentes del Páramo",
			'241778' => "Valdelugueros",
			'241784' => "Valdemora",
			'241797' => "Valdepiélago",
			'241801' => "Valdepolo",
			'241818' => "Valderas",
			'241823' => "Valderrey",
			'241839' => "Valderrueda",
			'241844' => "Valdesamario",
			'241876' => "Valdevimbre",
			'241882' => "Valencia de Don Juan",
			'241916' => "Vallecillo",
			'241895' => "Valverde de la Virgen",
			'241909' => "Valverde-Enrique",
			'241937' => "Vecilla (La)",
			'241968' => "Vega de Espinareda",
			'241974' => "Vega de Infanzones",
			'241980' => "Vega de Valcarce",
			'241942' => "Vegacervera",
			'241993' => "Vegaquemada",
			'242014' => "Vegas del Condado",
			'242029' => "Villablino",
			'242035' => "Villabraz",
			'242053' => "Villadangos del Páramo",
			'242066' => "Villadecanes",
			'242072' => "Villademor de la Vega",
			'242091' => "Villafranca del Bierzo",
			'242105' => "Villagatón",
			'242112' => "Villamandos",
			'249017' => "Villamanín",
			'242127' => "Villamañán",
			'242133' => "Villamartín de Don Sancho",
			'242148' => "Villamejil",
			'242151' => "Villamol",
			'242164' => "Villamontán de la Valduerna",
			'242170' => "Villamoratiel de las Matas",
			'242186' => "Villanueva de las Manzanas",
			'242199' => "Villaobispo de Otero",
			'249022' => "Villaornate y Castro",
			'242210' => "Villaquejida",
			'242225' => "Villaquilambre",
			'242231' => "Villarejo de Órbigo",
			'242246' => "Villares de Órbigo",
			'242259' => "Villasabariego",
			'242262' => "Villaselán",
			'242278' => "Villaturiel",
			'242284' => "Villazala",
			'242297' => "Villazanzo de Valderaduey",
			'242301' => "Zotes del Páramo",
			'250019' => "Abella de la Conca",
			'250024' => "Àger",
			'250030' => "Agramunt",
			'250387' => "Aitona",
			'250045' => "Alamús (Els)",
			'250058' => "Alàs i Cerc",
			'250061' => "Albagés (L')",
			'250077' => "Albatàrrec",
			'250083' => "Albesa",
			'250096' => "Albi (L')",
			'250100' => "Alcanó",
			'250117' => "Alcarràs",
			'250122' => "Alcoletge",
			'250138' => "Alfarràs",
			'250143' => "Alfés",
			'250156' => "Algerri",
			'250169' => "Alguaire",
			'250175' => "Alins",
			'250194' => "Almacelles",
			'250208' => "Almatret",
			'250215' => "Almenar",
			'250220' => "Alòs de Balaguer",
			'250236' => "Alpicat",
			'250241' => "Alt Àneu",
			'250273' => "Anglesola",
			'250292' => "Arbeca",
			'250313' => "Arres",
			'250328' => "Arsèguel",
			'250334' => "Artesa de Lleida",
			'250349' => "Artesa de Segre",
			'250365' => "Aspa",
			'250371' => "Avellanes i Santa Linya (Les)",
			'250390' => "Baix Pallars",
			'250404' => "Balaguer",
			'250411' => "Barbens",
			'250426' => "Baronia de Rialb (La)",
			'250447' => "Bassella",
			'250450' => "Bausen",
			'250463' => "Belianes",
			'251706' => "Bellaguarda",
			'250479' => "Bellcaire d'Urgell",
			'250485' => "Bell-lloc d'Urgell",
			'250498' => "Bellmunt d'Urgell",
			'250501' => "Bellpuig",
			'250518' => "Bellver de Cerdanya",
			'250523' => "Bellvís",
			'250539' => "Benavent de Segrià",
			'250557' => "Biosca",
			'250576' => "Bòrdes (Es)",
			'250582' => "Borges Blanques (Les)",
			'250595' => "Bossòst",
			'250560' => "Bovera",
			'250609' => "Cabanabona",
			'250616' => "Cabó",
			'250621' => "Camarasa",
			'250637' => "Canejan",
			'259046' => "Castell de Mur",
			'250642' => "Castellar de la Ribera",
			'250674' => "Castelldans",
			'250680' => "Castellnou de Seana",
			'250693' => "Castelló de Farfanya",
			'250707' => "Castellserà",
			'250714' => "Cava",
			'250729' => "Cervera",
			'250735' => "Cervià de les Garrigues",
			'250740' => "Ciutadilla",
			'250753' => "Clariana de Cardener",
			'250766' => "Cogul (El)",
			'250772' => "Coll de Nargó",
			'251636' => "Coma i la Pedra (La)",
			'251615' => "Conca de Dalt",
			'250788' => "Corbins",
			'250791' => "Cubells",
			'250812' => "Espluga Calba (L')",
			'250827' => "Espot",
			'250886' => "Estamariu",
			'250851' => "Estaràs",
			'250864' => "Esterri d'Àneu",
			'250870' => "Esterri de Cardós",
			'250899' => "Farrera",
			'259084' => "Fígols i Alinyà",
			'250925' => "Floresta (La)",
			'250931' => "Fondarella",
			'250946' => "Foradada",
			'250962' => "Fuliola (La)",
			'250978' => "Fulleda",
			'250984' => "Gavet de la Conca",
			'259123' => "Gimenells i el Pla de la Font",
			'250997' => "Golmés",
			'251001' => "Gósol",
			'251018' => "Granadella (La)",
			'251023' => "Granja d'Escarp (La)",
			'251039' => "Granyanella",
			'251057' => "Granyena de les Garrigues",
			'251044' => "Granyena de Segarra",
			'251095' => "Guimerà",
			'259031' => "Guingueta d'Àneu (La)",
			'251109' => "Guissona",
			'251116' => "Guixers",
			'251155' => "Isona i Conca Dellà",
			'251121' => "Ivars de Noguera",
			'251137' => "Ivars d'Urgell",
			'251142' => "Ivorra",
			'259101' => "Josa i Tuixén",
			'251180' => "Juncosa",
			'251193' => "Juneda",
			'251214' => "Les",
			'251229' => "Linyola",
			'251235' => "Lladorre",
			'251240' => "Lladurs",
			'251253' => "Llardecans",
			'251266' => "Llavorsí",
			'251207' => "Lleida",
			'251272' => "Lles de Cerdanya",
			'251288' => "Llimiana",
			'251291' => "Llobera",
			'251333' => "Maials",
			'251305' => "Maldà",
			'251312' => "Massalcoreig",
			'251327' => "Massoteres",
			'251348' => "Menàrguens",
			'251351' => "Miralcamp",
			'251370' => "Mollerussa",
			'251364' => "Molsosa (La)",
			'251399' => "Montellà i Martinet",
			'251403' => "Montferrer i Castellbò",
			'251386' => "Montgai",
			'251425' => "Montoliu de Lleida",
			'251410' => "Montoliu de Segarra",
			'251431' => "Montornès de Segarra",
			'251459' => "Nalec",
			'250254' => "Naut Aran",
			'251462' => "Navès",
			'251484' => "Odèn",
			'251497' => "Oliana",
			'251500' => "Oliola",
			'251517' => "Olius",
			'251522' => "Oluges (Les)",
			'251538' => "Omellons (Els)",
			'251543' => "Omells de na Gaia (Els)",
			'251556' => "Organyà",
			'251569' => "Os de Balaguer",
			'251575' => "Ossó de Sió",
			'251581' => "Palau d'Anglesola (El)",
			'251641' => "Penelles",
			'251654' => "Peramola",
			'251667' => "Pinell de Solsonès",
			'251673' => "Pinós",
			'259118' => "Plans de Sió (Els)",
			'251689' => "Poal (El)",
			'251692' => "Pobla de Cérvoles (La)",
			'251713' => "Pobla de Segur (La)",
			'250306' => "Pont de Bar (El)",
			'251734' => "Pont de Suert (El)",
			'251728' => "Ponts",
			'251749' => "Portella (La)",
			'251752' => "Prats i Sansor",
			'251765' => "Preixana",
			'251771' => "Preixens",
			'251790' => "Prullans",
			'251804' => "Puiggròs",
			'251811' => "Puigverd d'Agramunt",
			'251826' => "Puigverd de Lleida",
			'251832' => "Rialp",
			'259059' => "Ribera d'Ondara",
			'251850' => "Ribera d'Urgellet",
			'251863' => "Riner",
			'259139' => "Riu de Cerdanya",
			'251898' => "Rosselló",
			'251902' => "Salàs de Pallars",
			'251919' => "Sanaüja",
			'251961' => "Sant Esteve de la Sarga",
			'251924' => "Sant Guim de Freixenet",
			'251977' => "Sant Guim de la Plana",
			'251930' => "Sant Llorenç de Morunys",
			'259025' => "Sant Martí de Riucorb",
			'251945' => "Sant Ramon",
			'252017' => "Sarroca de Bellera",
			'252000' => "Sarroca de Lleida",
			'252022' => "Senterada",
			'250352' => "Sentiu de Sió (La)",
			'252043' => "Seròs",
			'252038' => "Seu d'Urgell (La)",
			'252056' => "Sidamon",
			'252069' => "Soleràs (El)",
			'252075' => "Solsona",
			'252081' => "Soriguera",
			'252094' => "Sort",
			'252108' => "Soses",
			'252115' => "Sudanell",
			'252120' => "Sunyer",
			'252154' => "Talarn",
			'252167' => "Talavera",
			'252173' => "Tàrrega",
			'252189' => "Tarrés",
			'252192' => "Tarroja de Segarra",
			'252206' => "Térmens",
			'252213' => "Tírvia",
			'252228' => "Tiurana",
			'252234' => "Torà",
			'252249' => "Torms (Els)",
			'252252' => "Tornabous",
			'252271' => "Torre de Cabdella (La)",
			'252265' => "Torrebesses",
			'252287' => "Torrefarrera",
			'259078' => "Torrefeta i Florejacs",
			'252304' => "Torregrossa",
			'252311' => "Torrelameu",
			'252326' => "Torres de Segre",
			'252332' => "Torre-serona",
			'252347' => "Tremp",
			'250432' => "Vall de Boí (La)",
			'259010' => "Vall de Cardós",
			'252385' => "Vallbona de les Monges",
			'252402' => "Vallfogona de Balaguer",
			'259062' => "Valls d'Aguilar (Les)",
			'252398' => "Valls de Valira (Les)",
			'259097' => "Vansa i Fórnols (La)",
			'252424' => "Verdú",
			'252430' => "Vielha e Mijaran",
			'252445' => "Vilagrassa",
			'252458' => "Vilaller",
			'252477' => "Vilamòs",
			'252483' => "Vilanova de Bellpuig",
			'252542' => "Vilanova de la Barca",
			'252496' => "Vilanova de l'Aguda",
			'252509' => "Vilanova de Meià",
			'252516' => "Vilanova de Segrià",
			'252521' => "Vila-sana",
			'252537' => "Vilosell (El)",
			'252555' => "Vinaixa",
			'260012' => "Ábalos",
			'260027' => "Agoncillo",
			'260033' => "Aguilar del Río Alhama",
			'260048' => "Ajamil de Cameros",
			'260051' => "Albelda de Iregua",
			'260064' => "Alberite",
			'260070' => "Alcanadre",
			'260086' => "Aldeanueva de Ebro",
			'260099' => "Alesanco",
			'260103' => "Alesón",
			'260110' => "Alfaro",
			'260125' => "Almarza de Cameros",
			'260131' => "Anguciana",
			'260146' => "Anguiano",
			'260159' => "Arenzana de Abajo",
			'260162' => "Arenzana de Arriba",
			'260178' => "Arnedillo",
			'260184' => "Arnedo",
			'260197' => "Arrúbal",
			'260201' => "Ausejo",
			'260218' => "Autol",
			'260223' => "Azofra",
			'260239' => "Badarán",
			'260244' => "Bañares",
			'260260' => "Baños de Río Tobía",
			'260257' => "Baños de Rioja",
			'260276' => "Berceo",
			'260282' => "Bergasa",
			'260295' => "Bergasillas Bajera",
			'260309' => "Bezares",
			'260316' => "Bobadilla",
			'260321' => "Brieva de Cameros",
			'260337' => "Briñas",
			'260342' => "Briones",
			'260355' => "Cabezón de Cameros",
			'260368' => "Calahorra",
			'260374' => "Camprovín",
			'260380' => "Canales de la Sierra",
			'260393' => "Canillas de Río Tuerto",
			'260407' => "Cañas",
			'260414' => "Cárdenas",
			'260429' => "Casalarreina",
			'260435' => "Castañares de Rioja",
			'260440' => "Castroviejo",
			'260453' => "Cellorigo",
			'260466' => "Cenicero",
			'260472' => "Cervera del Río Alhama",
			'260488' => "Cidamón",
			'260491' => "Cihuri",
			'260504' => "Cirueña",
			'260511' => "Clavijo",
			'260526' => "Cordovín",
			'260532' => "Corera",
			'260547' => "Cornago",
			'260550' => "Corporales",
			'260563' => "Cuzcurrita de Río Tirón",
			'260579' => "Daroca de Rioja",
			'260585' => "Enciso",
			'260598' => "Entrena",
			'260602' => "Estollo",
			'260619' => "Ezcaray",
			'260624' => "Foncea",
			'260630' => "Fonzaleche",
			'260645' => "Fuenmayor",
			'260658' => "Galbárruli",
			'260661' => "Galilea",
			'260677' => "Gallinero de Cameros",
			'260683' => "Gimileo",
			'260696' => "Grañón",
			'260700' => "Grávalos",
			'260717' => "Haro",
			'260722' => "Herce",
			'260738' => "Herramélluri",
			'260743' => "Hervías",
			'260756' => "Hormilla",
			'260769' => "Hormilleja",
			'260775' => "Hornillos de Cameros",
			'260781' => "Hornos de Moncalvillo",
			'260794' => "Huércanos",
			'260808' => "Igea",
			'260815' => "Jalón de Cameros",
			'260820' => "Laguna de Cameros",
			'260836' => "Lagunilla del Jubera",
			'260841' => "Lardero",
			'260867' => "Ledesma de la Cogolla",
			'260873' => "Leiva",
			'260889' => "Leza de Río Leza",
			'260892' => "Logroño",
			'260913' => "Lumbreras",
			'260928' => "Manjarrés",
			'260934' => "Mansilla de la Sierra",
			'260949' => "Manzanares de Rioja",
			'260952' => "Matute",
			'260965' => "Medrano",
			'260987' => "Munilla",
			'260990' => "Murillo de Río Leza",
			'261004' => "Muro de Aguas",
			'261011' => "Muro en Cameros",
			'261026' => "Nájera",
			'261032' => "Nalda",
			'261047' => "Navajún",
			'261050' => "Navarrete",
			'261063' => "Nestares",
			'261079' => "Nieva de Cameros",
			'261098' => "Ochánduri",
			'261085' => "Ocón",
			'261102' => "Ojacastro",
			'261119' => "Ollauri",
			'261124' => "Ortigosa de Cameros",
			'261130' => "Pazuengos",
			'261145' => "Pedroso",
			'261158' => "Pinillos",
			'261177' => "Pradejón",
			'261183' => "Pradillo",
			'261196' => "Préjano",
			'261200' => "Quel",
			'261217' => "Rabanera",
			'261222' => "Rasillo de Cameros (El)",
			'261238' => "Redal (El)",
			'261243' => "Ribafrecha",
			'261256' => "Rincón de Soto",
			'261269' => "Robres del Castillo",
			'261275' => "Rodezno",
			'261281' => "Sajazarra",
			'261294' => "San Asensio",
			'261308' => "San Millán de la Cogolla",
			'261315' => "San Millán de Yécora",
			'261320' => "San Román de Cameros",
			'261392' => "San Torcuato",
			'261428' => "San Vicente de la Sonsierra",
			'261341' => "Santa Coloma",
			'261354' => "Santa Engracia del Jubera",
			'261367' => "Santa Eulalia Bajera",
			'261389' => "Santo Domingo de la Calzada",
			'261406' => "Santurde de Rioja",
			'261413' => "Santurdejo",
			'261434' => "Sojuela",
			'261449' => "Sorzano",
			'261452' => "Sotés",
			'261465' => "Soto en Cameros",
			'261471' => "Terroba",
			'261487' => "Tirgo",
			'261490' => "Tobía",
			'261503' => "Tormantos",
			'261531' => "Torre en Cameros",
			'261510' => "Torrecilla en Cameros",
			'261525' => "Torrecilla sobre Alesanco",
			'261546' => "Torremontalbo",
			'261559' => "Treviana",
			'261578' => "Tricio",
			'261584' => "Tudelilla",
			'261601' => "Uruñuela",
			'261618' => "Valdemadera",
			'261623' => "Valgañón",
			'261639' => "Ventosa",
			'261644' => "Ventrosa",
			'261657' => "Viguera",
			'261660' => "Villalba de Rioja",
			'261676' => "Villalobar de Rioja",
			'261682' => "Villamediana de Iregua",
			'261695' => "Villanueva de Cameros",
			'261709' => "Villar de Arnedo (El)",
			'261716' => "Villar de Torre",
			'261721' => "Villarejo",
			'261737' => "Villarroya",
			'261742' => "Villarta-Quintana",
			'261755' => "Villavelayo",
			'261768' => "Villaverde de Rioja",
			'261774' => "Villoslada de Cameros",
			'261780' => "Viniegra de Abajo",
			'261793' => "Viniegra de Arriba",
			'261807' => "Zarratón",
			'261814' => "Zarzosa",
			'261835' => "Zorraquín",
			'270018' => "Abadín",
			'270023' => "Alfoz",
			'270039' => "Antas de Ulla",
			'270044' => "Baleira",
			'279019' => "Baralla",
			'270057' => "Barreiros",
			'270060' => "Becerreá",
			'270076' => "Begonte",
			'270082' => "Bóveda",
			'279024' => "Burela",
			'270095' => "Carballedo",
			'270109' => "Castro de Rei",
			'270116' => "Castroverde",
			'270121' => "Cervantes",
			'270137' => "Cervo",
			'270168' => "Chantada",
			'270142' => "Corgo (O)",
			'270155' => "Cospeito",
			'270174' => "Folgoso do Courel",
			'270180' => "Fonsagrada (A)",
			'270193' => "Foz",
			'270207' => "Friol",
			'270229' => "Guitiriz",
			'270235' => "Guntín",
			'270240' => "Incio (O)",
			'270266' => "Láncara",
			'270272' => "Lourenzá",
			'270288' => "Lugo",
			'270291' => "Meira",
			'270305' => "Mondoñedo",
			'270312' => "Monforte de Lemos",
			'270327' => "Monterroso",
			'270333' => "Muras",
			'270348' => "Navia de Suarna",
			'270351' => "Negueira de Muñiz",
			'270370' => "Nogais (As)",
			'270386' => "Ourol",
			'270399' => "Outeiro de Rei",
			'270403' => "Palas de Rei",
			'270410' => "Pantón",
			'270425' => "Paradela",
			'270431' => "Páramo (O)",
			'270446' => "Pastoriza (A)",
			'270459' => "Pedrafita do Cebreiro",
			'270478' => "Pobra do Brollón (A)",
			'270462' => "Pol",
			'270484' => "Pontenova (A)",
			'270497' => "Portomarín",
			'270500' => "Quiroga",
			'270569' => "Rábade",
			'270517' => "Ribadeo",
			'270522' => "Ribas de Sil",
			'270538' => "Ribeira de Piquín",
			'270543' => "Riotorto",
			'270556' => "Samos",
			'270575' => "Sarria",
			'270581' => "Saviñao (O)",
			'270594' => "Sober",
			'270608' => "Taboada",
			'270615' => "Trabada",
			'270620' => "Triacastela",
			'270636' => "Valadouro (O)",
			'270641' => "Vicedo (O)",
			'270654' => "Vilalba",
			'270667' => "Viveiro",
			'270214' => "Xermade",
			'270253' => "Xove",
			'280014' => "Acebeda (La)",
			'280029' => "Ajalvir",
			'280035' => "Alameda del Valle",
			'280040' => "Álamo (El)",
			'280053' => "Alcalá de Henares",
			'280066' => "Alcobendas",
			'280072' => "Alcorcón",
			'280088' => "Aldea del Fresno",
			'280091' => "Algete",
			'280105' => "Alpedrete",
			'280112' => "Ambite",
			'280127' => "Anchuelo",
			'280133' => "Aranjuez",
			'280148' => "Arganda del Rey",
			'280151' => "Arroyomolinos",
			'280164' => "Atazar (El)",
			'280170' => "Batres",
			'280186' => "Becerril de la Sierra",
			'280199' => "Belmonte de Tajo",
			'280210' => "Berrueco (El)",
			'280203' => "Berzosa del Lozoya",
			'280225' => "Boadilla del Monte",
			'280231' => "Boalo (El)",
			'280246' => "Braojos",
			'280259' => "Brea de Tajo",
			'280262' => "Brunete",
			'280278' => "Buitrago del Lozoya",
			'280284' => "Bustarviejo",
			'280297' => "Cabanillas de la Sierra",
			'280301' => "Cabrera (La)",
			'280318' => "Cadalso de los Vidrios",
			'280323' => "Camarma de Esteruelas",
			'280339' => "Campo Real",
			'280344' => "Canencia",
			'280357' => "Carabaña",
			'280360' => "Casarrubuelos",
			'280376' => "Cenicientos",
			'280382' => "Cercedilla",
			'280395' => "Cervera de Buitrago",
			'280513' => "Chapinería",
			'280528' => "Chinchón",
			'280409' => "Ciempozuelos",
			'280416' => "Cobeña",
			'280468' => "Collado Mediano",
			'280474' => "Collado Villalba",
			'280437' => "Colmenar de Oreja",
			'280421' => "Colmenar del Arroyo",
			'280455' => "Colmenar Viejo",
			'280442' => "Colmenarejo",
			'280480' => "Corpa",
			'280493' => "Coslada",
			'280506' => "Cubas de la Sagra",
			'280534' => "Daganzo de Arriba",
			'280549' => "Escorial (El)",
			'280552' => "Estremera",
			'280565' => "Fresnedillas de la Oliva",
			'280571' => "Fresno de Torote",
			'280587' => "Fuenlabrada",
			'280590' => "Fuente el Saz de Jarama",
			'280604' => "Fuentidueña de Tajo",
			'280611' => "Galapagar",
			'280626' => "Garganta de los Montes",
			'280632' => "Gargantilla del Lozoya y Pinilla de Buitrago",
			'280647' => "Gascones",
			'280650' => "Getafe",
			'280663' => "Griñón",
			'280679' => "Guadalix de la Sierra",
			'280685' => "Guadarrama",
			'280698' => "Hiruela (La)",
			'280702' => "Horcajo de la Sierra-Aoslos",
			'280719' => "Horcajuelo de la Sierra",
			'280724' => "Hoyo de Manzanares",
			'280730' => "Humanes de Madrid",
			'280745' => "Leganés",
			'280758' => "Loeches",
			'280761' => "Lozoya",
			'289015' => "Lozoyuela-Navas-Sieteiglesias",
			'280783' => "Madarcos",
			'280796' => "Madrid",
			'280800' => "Majadahonda",
			'280822' => "Manzanares el Real",
			'280838' => "Meco",
			'280843' => "Mejorada del Campo",
			'280856' => "Miraflores de la Sierra",
			'280869' => "Molar (El)",
			'280875' => "Molinos (Los)",
			'280881' => "Montejo de la Sierra",
			'280894' => "Moraleja de Enmedio",
			'280908' => "Moralzarzal",
			'280915' => "Morata de Tajuña",
			'280920' => "Móstoles",
			'280936' => "Navacerrada",
			'280941' => "Navalafuente",
			'280954' => "Navalagamella",
			'280967' => "Navalcarnero",
			'280973' => "Navarredonda y San Mamés",
			'280992' => "Navas del Rey",
			'281006' => "Nuevo Baztán",
			'281013' => "Olmeda de las Fuentes",
			'281028' => "Orusco de Tajuña",
			'281049' => "Paracuellos de Jarama",
			'281065' => "Parla",
			'281071' => "Patones",
			'281087' => "Pedrezuela",
			'281090' => "Pelayos de la Presa",
			'281104' => "Perales de Tajuña",
			'281111' => "Pezuela de las Torres",
			'281126' => "Pinilla del Valle",
			'281132' => "Pinto",
			'281147' => "Piñuécar-Gandullas",
			'281150' => "Pozuelo de Alarcón",
			'281163' => "Pozuelo del Rey",
			'281179' => "Prádena del Rincón",
			'281185' => "Puebla de la Sierra",
			'289020' => "Puentes Viejas",
			'281198' => "Quijorna",
			'281202' => "Rascafría",
			'281219' => "Redueña",
			'281224' => "Ribatejada",
			'281230' => "Rivas-Vaciamadrid",
			'281245' => "Robledillo de la Jara",
			'281258' => "Robledo de Chavela",
			'281261' => "Robregordo",
			'281277' => "Rozas de Madrid (Las)",
			'281283' => "Rozas de Puerto Real",
			'281296' => "San Agustín del Guadalix",
			'281300' => "San Fernando de Henares",
			'281317' => "San Lorenzo de El Escorial",
			'281322' => "San Martín de la Vega",
			'281338' => "San Martín de Valdeiglesias",
			'281343' => "San Sebastián de los Reyes",
			'281356' => "Santa María de la Alameda",
			'281369' => "Santorcaz",
			'281375' => "Santos de la Humosa (Los)",
			'281381' => "Serna del Monte (La)",
			'281408' => "Serranillos del Valle",
			'281415' => "Sevilla la Nueva",
			'281436' => "Somosierra",
			'281441' => "Soto del Real",
			'281454' => "Talamanca de Jarama",
			'281467' => "Tielmes",
			'281473' => "Titulcia",
			'281489' => "Torrejón de Ardoz",
			'281492' => "Torrejón de la Calzada",
			'281505' => "Torrejón de Velasco",
			'281512' => "Torrelaguna",
			'281527' => "Torrelodones",
			'281533' => "Torremocha de Jarama",
			'281548' => "Torres de la Alameda",
			'289036' => "Tres Cantos",
			'281551' => "Valdaracete",
			'281564' => "Valdeavero",
			'281570' => "Valdelaguna",
			'281586' => "Valdemanco",
			'281599' => "Valdemaqueda",
			'281603' => "Valdemorillo",
			'281610' => "Valdemoro",
			'281625' => "Valdeolmos-Alalpardo",
			'281631' => "Valdepiélagos",
			'281646' => "Valdetorres de Jarama",
			'281659' => "Valdilecha",
			'281662' => "Valverde de Alcalá",
			'281678' => "Velilla de San Antonio",
			'281684' => "Vellón (El)",
			'281697' => "Venturada",
			'281718' => "Villa del Prado",
			'281701' => "Villaconejos",
			'281723' => "Villalbilla",
			'281739' => "Villamanrique de Tajo",
			'281744' => "Villamanta",
			'281757' => "Villamantilla",
			'281760' => "Villanueva de la Cañada",
			'281782' => "Villanueva de Perales",
			'281776' => "Villanueva del Pardillo",
			'281795' => "Villar del Olmo",
			'281809' => "Villarejo de Salvanés",
			'281816' => "Villaviciosa de Odón",
			'281821' => "Villavieja del Lozoya",
			'281837' => "Zarzalejo",
			'290017' => "Alameda",
			'290022' => "Alcaucín",
			'290038' => "Alfarnate",
			'290043' => "Alfarnatejo",
			'290056' => "Algarrobo",
			'290069' => "Algatocín",
			'290075' => "Alhaurín de la Torre",
			'290081' => "Alhaurín el Grande",
			'290094' => "Almáchar",
			'290108' => "Almargen",
			'290115' => "Almogía",
			'290120' => "Álora",
			'290136' => "Alozaina",
			'290141' => "Alpandeire",
			'290154' => "Antequera",
			'290167' => "Árchez",
			'290173' => "Archidona",
			'290189' => "Ardales",
			'290192' => "Arenas",
			'290206' => "Arriate",
			'290213' => "Atajate",
			'290228' => "Benadalid",
			'290234' => "Benahavís",
			'290249' => "Benalauría",
			'290252' => "Benalmádena",
			'290265' => "Benamargosa",
			'290271' => "Benamocarra",
			'290287' => "Benaoján",
			'290290' => "Benarrabá",
			'290304' => "Borge (El)",
			'290311' => "Burgo (El)",
			'290326' => "Campillos",
			'290332' => "Canillas de Aceituno",
			'290347' => "Canillas de Albaida",
			'290350' => "Cañete la Real",
			'290363' => "Carratraca",
			'290379' => "Cartajima",
			'290385' => "Cártama",
			'290398' => "Casabermeja",
			'290402' => "Casarabonela",
			'290419' => "Casares",
			'290424' => "Coín",
			'290430' => "Colmenar",
			'290445' => "Comares",
			'290458' => "Cómpeta",
			'290461' => "Cortes de la Frontera",
			'290477' => "Cuevas Bajas",
			'290496' => "Cuevas de San Marcos",
			'290483' => "Cuevas del Becerro",
			'290509' => "Cútar",
			'290516' => "Estepona",
			'290521' => "Faraján",
			'290537' => "Frigiliana",
			'290542' => "Fuengirola",
			'290555' => "Fuente de Piedra",
			'290568' => "Gaucín",
			'290574' => "Genalguacil",
			'290580' => "Guaro",
			'290593' => "Humilladero",
			'290607' => "Igualeja",
			'290614' => "Istán",
			'290629' => "Iznate",
			'290635' => "Jimera de Líbar",
			'290640' => "Jubrique",
			'290653' => "Júzcar",
			'290666' => "Macharaviaya",
			'290672' => "Málaga",
			'290688' => "Manilva",
			'290691' => "Marbella",
			'290705' => "Mijas",
			'290712' => "Moclinejo",
			'290727' => "Mollina",
			'290733' => "Monda",
			'290748' => "Montejaque",
			'290751' => "Nerja",
			'290764' => "Ojén",
			'290770' => "Parauta",
			'290799' => "Periana",
			'290803' => "Pizarra",
			'290810' => "Pujerra",
			'290825' => "Rincón de la Victoria",
			'290831' => "Riogordo",
			'290846' => "Ronda",
			'290859' => "Salares",
			'290862' => "Sayalonga",
			'290878' => "Sedella",
			'290884' => "Sierra de Yeguas",
			'290897' => "Teba",
			'290901' => "Tolox",
			'299018' => "Torremolinos",
			'290918' => "Torrox",
			'290923' => "Totalán",
			'290939' => "Valle de Abdalajís",
			'290944' => "Vélez-Málaga",
			'290957' => "Villanueva de Algaidas",
			'290982' => "Villanueva de Tapia",
			'290960' => "Villanueva del Rosario",
			'290976' => "Villanueva del Trabuco",
			'290995' => "Viñuela",
			'291009' => "Yunquera",
			'300011' => "Abanilla",
			'300026' => "Abarán",
			'300032' => "Águilas",
			'300047' => "Albudeite",
			'300050' => "Alcantarilla",
			'309027' => "Alcázares (Los)",
			'300063' => "Aledo",
			'300079' => "Alguazas",
			'300085' => "Alhama de Murcia",
			'300098' => "Archena",
			'300102' => "Beniel",
			'300119' => "Blanca",
			'300124' => "Bullas",
			'300130' => "Calasparra",
			'300145' => "Campos del Río",
			'300158' => "Caravaca de la Cruz",
			'300161' => "Cartagena",
			'300177' => "Cehegín",
			'300183' => "Ceutí",
			'300196' => "Cieza",
			'300200' => "Fortuna",
			'300217' => "Fuente Álamo de Murcia",
			'300222' => "Jumilla",
			'300238' => "Librilla",
			'300243' => "Lorca",
			'300256' => "Lorquí",
			'300269' => "Mazarrón",
			'300275' => "Molina de Segura",
			'300281' => "Moratalla",
			'300294' => "Mula",
			'300308' => "Murcia",
			'300315' => "Ojós",
			'300320' => "Pliego",
			'300336' => "Puerto Lumbreras",
			'300341' => "Ricote",
			'300354' => "San Javier",
			'300367' => "San Pedro del Pinatar",
			'309012' => "Santomera",
			'300373' => "Torre-Pacheco",
			'300389' => "Torres de Cotillas (Las)",
			'300392' => "Totana",
			'300406' => "Ulea",
			'300413' => "Unión (La)",
			'300428' => "Villanueva del Río Segura",
			'300434' => "Yecla",
			'310018' => "Abáigar",
			'310023' => "Abárzuza",
			'310039' => "Abaurregaina/Abaurrea Alta",
			'310044' => "Abaurrepea/Abaurrea Baja",
			'310057' => "Aberin",
			'310060' => "Ablitas",
			'310076' => "Adiós",
			'310082' => "Aguilar de Codés",
			'310095' => "Aibar/Oibar",
			'310116' => "Allín",
			'310121' => "Allo",
			'310109' => "Altsasu/Alsasua",
			'310137' => "Améscoa Baja",
			'310142' => "Ancín/Antzin",
			'310155' => "Andosilla",
			'310168' => "Ansoáin/Antsoain",
			'310174' => "Anue",
			'310180' => "Añorbe",
			'310193' => "Aoiz/Agoitz",
			'310207' => "Araitz",
			'310253' => "Arakil",
			'310214' => "Aranarache/Aranaratxe",
			'310235' => "Aranguren",
			'310240' => "Arano",
			'310229' => "Arantza",
			'310266' => "Aras",
			'310272' => "Arbizu",
			'310288' => "Arce/Artzi",
			'310291' => "Arcos (Los)",
			'310305' => "Arellano",
			'310312' => "Areso",
			'310327' => "Arguedas",
			'310333' => "Aria",
			'310348' => "Aribe",
			'310351' => "Armañanzas",
			'310364' => "Arróniz",
			'310370' => "Arruazu",
			'310386' => "Artajona",
			'310399' => "Artazu",
			'310403' => "Atez",
			'310581' => "Auritz/Burguete",
			'310410' => "Ayegui/Aiegi",
			'310425' => "Azagra",
			'310431' => "Azuelo",
			'310446' => "Bakaiku",
			'319019' => "Barañain",
			'310459' => "Barásoain",
			'310462' => "Barbarin",
			'310478' => "Bargota",
			'310484' => "Barillas",
			'310497' => "Basaburua",
			'310500' => "Baztan",
			'311379' => "Beintza-Labaien",
			'310517' => "Beire",
			'310522' => "Belascoáin",
			'312508' => "Bera",
			'310538' => "Berbinzana",
			'319058' => "Beriáin",
			'319024' => "Berrioplano",
			'319030' => "Berriozar",
			'310543' => "Bertizarana",
			'310556' => "Betelu",
			'312536' => "Bidaurreta",
			'310569' => "Biurrun-Olcoz",
			'310575' => "Buñuel",
			'310594' => "Burgui/Burgi",
			'310608' => "Burlada/Burlata",
			'310615' => "Busto (El)",
			'310620' => "Cabanillas",
			'310636' => "Cabredo",
			'310641' => "Cadreita",
			'310654' => "Caparroso",
			'310667' => "Cárcar",
			'310673' => "Carcastillo",
			'310689' => "Cascante",
			'310692' => "Cáseda",
			'310706' => "Castejón",
			'310713' => "Castillonuevo",
			'311939' => "Cendea de Olza/Oltza Zendea",
			'310728' => "Cintruénigo",
			'310749' => "Cirauqui/Zirauki",
			'310752' => "Ciriza",
			'310765' => "Cizur",
			'310771' => "Corella",
			'310787' => "Cortes",
			'310790' => "Desojo",
			'310804' => "Dicastillo",
			'310811' => "Donamaria",
			'312212' => "Doneztebe/Santesteban",
			'310832' => "Echarri",
			'310863' => "Egüés",
			'310879' => "Elgorriaga",
			'310898' => "Enériz/Eneritz",
			'310902' => "Eratsun",
			'310919' => "Ergoiena",
			'310924' => "Erro",
			'310945' => "Eslava",
			'310958' => "Esparza de Salazar/Espartza Zaraitzu",
			'310961' => "Espronceda",
			'310977' => "Estella/Lizarra",
			'310983' => "Esteribar",
			'310996' => "Etayo",
			'310826' => "Etxalar",
			'310847' => "Etxarri-Aranatz",
			'310850' => "Etxauri",
			'311000' => "Eulate",
			'311017' => "Ezcabarte",
			'310930' => "Ezcároz/Ezkaroze",
			'311022' => "Ezkurra",
			'311038' => "Ezprogui",
			'311043' => "Falces",
			'311056' => "Fitero",
			'311069' => "Fontellas",
			'311075' => "Funes",
			'311081' => "Fustiñana",
			'311094' => "Galar",
			'311108' => "Gallipienzo",
			'311115' => "Gallués/Galoze",
			'311120' => "Garaioa",
			'311136' => "Garde",
			'311141' => "Garínoain",
			'311154' => "Garralda",
			'311167' => "Genevilla",
			'311173' => "Goizueta",
			'311189' => "Goñi",
			'311192' => "Güesa/Gorza",
			'311206' => "Guesálaz/Gesalatz",
			'311213' => "Guirguillano",
			'312567' => "Hiriberri/Villanueva de Aezkoa",
			'311228' => "Huarte/Uharte",
			'311249' => "Ibargoiti",
			'312592' => "Igantzi",
			'311252' => "Igúzquiza",
			'311265' => "Imotz",
			'311271' => "Irañeta",
			'319045' => "Irurtzun",
			'311287' => "Isaba/Izaba",
			'311290' => "Ituren",
			'311304' => "Iturmendi",
			'311311' => "Iza/Itza",
			'311326' => "Izagaondoa",
			'311332' => "Izalzu/Itzaltzu",
			'311347' => "Jaurrieta",
			'311350' => "Javier",
			'311363' => "Juslapeña",
			'311385' => "Lakuntza",
			'311398' => "Lana",
			'311402' => "Lantz",
			'311419' => "Lapoblación",
			'311424' => "Larraga",
			'311430' => "Larraona",
			'311445' => "Larraun",
			'311458' => "Lazagurría",
			'311461' => "Leache",
			'311477' => "Legarda",
			'311483' => "Legaria",
			'311496' => "Leitza",
			'319083' => "Lekunberri",
			'311509' => "Leoz/Leotz",
			'311516' => "Lerga",
			'311521' => "Lerín",
			'311537' => "Lesaka",
			'311542' => "Lezáun",
			'311555' => "Liédena",
			'311568' => "Lizoáin-Arriasgoiti",
			'311574' => "Lodosa",
			'311580' => "Lónguida/Longida",
			'311593' => "Lumbier",
			'311607' => "Luquin",
			'312482' => "Luzaide/Valcarlos",
			'311614' => "Mañeru",
			'311629' => "Marañón",
			'311635' => "Marcilla",
			'311640' => "Mélida",
			'311653' => "Mendavia",
			'311666' => "Mendaza",
			'311672' => "Mendigorría",
			'311688' => "Metauten",
			'311691' => "Milagro",
			'311705' => "Mirafuentes",
			'311712' => "Miranda de Arga",
			'311727' => "Monreal",
			'311733' => "Monteagudo",
			'311748' => "Morentin",
			'311751' => "Mues",
			'311764' => "Murchante",
			'311770' => "Murieta",
			'311786' => "Murillo el Cuende",
			'311799' => "Murillo el Fruto",
			'311803' => "Muruzábal",
			'311810' => "Navascués",
			'311825' => "Nazar",
			'310885' => "Noáin (Valle de Elorz)/Noain (Elortzibar)",
			'311831' => "Obanos",
			'311859' => "Ochagavía/Otsagabia",
			'311846' => "Oco",
			'311862' => "Odieta",
			'311878' => "Oitz",
			'311884' => "Olaibar",
			'311897' => "Olazti/Olazagutía",
			'311901' => "Olejua",
			'311918' => "Olite/Erriberri",
			'311944' => "Ollo",
			'311923' => "Olóriz/Oloritz",
			'311957' => "Orbaitzeta",
			'311960' => "Orbara",
			'311976' => "Orísoain",
			'319061' => "Orkoien",
			'311982' => "Oronz/Orontze",
			'311995' => "Oroz-Betelu/Orotz-Betelu",
			'312114' => "Orreaga/Roncesvalles",
			'312009' => "Oteiza",
			'312016' => "Pamplona/Iruña",
			'312021' => "Peralta/Azkoien",
			'312037' => "Petilla de Aragón",
			'312042' => "Piedramillera",
			'312055' => "Pitillas",
			'312068' => "Puente la Reina/Gares",
			'312074' => "Pueyo",
			'312080' => "Ribaforada",
			'312093' => "Romanzado",
			'312107' => "Roncal/Erronkari",
			'312129' => "Sada",
			'312135' => "Saldías",
			'312140' => "Salinas de Oro/Jaitz",
			'312153' => "San Adrián",
			'312172' => "San Martín de Unx",
			'312166' => "Sangüesa/Zangoza",
			'312191' => "Sansol",
			'312205' => "Santacara",
			'312227' => "Sarriés/Sartze",
			'312233' => "Sartaguda",
			'312248' => "Sesma",
			'312251' => "Sorlada",
			'312264' => "Sunbilla",
			'312270' => "Tafalla",
			'312286' => "Tiebas-Muruarte de Reta",
			'312299' => "Tirapu",
			'312303' => "Torralba del Río",
			'312310' => "Torres del Río",
			'312325' => "Tudela",
			'312331' => "Tulebras",
			'312346' => "Ucar",
			'311234' => "Uharte-Arakil",
			'312359' => "Ujué",
			'312362' => "Ultzama",
			'312378' => "Unciti",
			'312384' => "Unzué",
			'312397' => "Urdazubi/Urdax",
			'312401' => "Urdiain",
			'312418' => "Urraul Alto",
			'312423' => "Urraul Bajo",
			'312444' => "Urrotz",
			'312439' => "Urroz-Villa",
			'312457' => "Urzainqui/Urzainki",
			'312460' => "Uterga",
			'312476' => "Uztárroz/Uztarroze",
			'312606' => "Valle de Yerri/Deierri",
			'312495' => "Valtierra",
			'312515' => "Viana",
			'312520' => "Vidángoz/Bidankoze",
			'312541' => "Villafranca",
			'312554' => "Villamayor de Monjardín",
			'312573' => "Villatuerta",
			'312589' => "Villava/Atarrabia",
			'312613' => "Yesa",
			'312628' => "Zabalza/Zabaltza",
			'310734' => "Ziordia",
			'319077' => "Zizur Mayor/Zizur Nagusia",
			'312634' => "Zubieta",
			'312649' => "Zugarramurdi",
			'312652' => "Zúñiga",
			'320013' => "Allariz",
			'320028' => "Amoeiro",
			'320034' => "Arnoia (A)",
			'320049' => "Avión",
			'320052' => "Baltar",
			'320065' => "Bande",
			'320071' => "Baños de Molgas",
			'320087' => "Barbadás",
			'320090' => "Barco de Valdeorras (O)",
			'320104' => "Beade",
			'320111' => "Beariz",
			'320126' => "Blancos (Os)",
			'320132' => "Boborás",
			'320147' => "Bola (A)",
			'320150' => "Bolo (O)",
			'320163' => "Calvos de Randín",
			'320185' => "Carballeda de Avia",
			'320179' => "Carballeda de Valdeorras",
			'320198' => "Carballiño (O)",
			'320202' => "Cartelle",
			'320224' => "Castrelo de Miño",
			'320219' => "Castrelo do Val",
			'320230' => "Castro Caldelas",
			'320245' => "Celanova",
			'320258' => "Cenlle",
			'320296' => "Chandrexa de Queixa",
			'320261' => "Coles",
			'320277' => "Cortegada",
			'320283' => "Cualedro",
			'320300' => "Entrimo",
			'320317' => "Esgos",
			'320338' => "Gomesende",
			'320343' => "Gudiña (A)",
			'320356' => "Irixo (O)",
			'320381' => "Larouco",
			'320394' => "Laza",
			'320408' => "Leiro",
			'320415' => "Lobeira",
			'320420' => "Lobios",
			'320436' => "Maceda",
			'320441' => "Manzaneda",
			'320454' => "Maside",
			'320467' => "Melón",
			'320473' => "Merca (A)",
			'320489' => "Mezquita (A)",
			'320492' => "Montederramo",
			'320505' => "Monterrei",
			'320512' => "Muíños",
			'320527' => "Nogueira de Ramuín",
			'320533' => "Oímbra",
			'320548' => "Ourense",
			'320551' => "Paderne de Allariz",
			'320564' => "Padrenda",
			'320570' => "Parada de Sil",
			'320586' => "Pereiro de Aguiar (O)",
			'320599' => "Peroxa (A)",
			'320603' => "Petín",
			'320610' => "Piñor",
			'320631' => "Pobra de Trives (A)",
			'320646' => "Pontedeva",
			'320625' => "Porqueira",
			'320659' => "Punxín",
			'320662' => "Quintela de Leirado",
			'320678' => "Rairiz de Veiga",
			'320684' => "Ramirás",
			'320697' => "Ribadavia",
			'320718' => "Riós",
			'320723' => "Rúa (A)",
			'320739' => "Rubiá",
			'320744' => "San Amaro",
			'320757' => "San Cibrao das Viñas",
			'320760' => "San Cristovo de Cea",
			'320701' => "San Xoán de Río",
			'320776' => "Sandiás",
			'320782' => "Sarreaus",
			'320795' => "Taboadela",
			'320809' => "Teixeira (A)",
			'320816' => "Toén",
			'320821' => "Trasmiras",
			'320837' => "Veiga (A)",
			'320842' => "Verea",
			'320855' => "Verín",
			'320868' => "Viana do Bolo",
			'320874' => "Vilamarín",
			'320880' => "Vilamartín de Valdeorras",
			'320893' => "Vilar de Barrio",
			'320907' => "Vilar de Santos",
			'320914' => "Vilardevós",
			'320929' => "Vilariño de Conso",
			'320322' => "Xinzo de Limia",
			'320369' => "Xunqueira de Ambía",
			'320375' => "Xunqueira de Espadanedo",
			'330019' => "Allande",
			'330024' => "Aller",
			'330030' => "Amieva",
			'330045' => "Avilés",
			'330058' => "Belmonte de Miranda",
			'330061' => "Bimenes",
			'330077' => "Boal",
			'330083' => "Cabrales",
			'330096' => "Cabranes",
			'330100' => "Candamo",
			'330122' => "Cangas de Onís",
			'330117' => "Cangas del Narcea",
			'330138' => "Caravia",
			'330143' => "Carreño",
			'330156' => "Caso",
			'330169' => "Castrillón",
			'330175' => "Castropol",
			'330181' => "Coaña",
			'330194' => "Colunga",
			'330208' => "Corvera de Asturias",
			'330215' => "Cudillero",
			'330220' => "Degaña",
			'330236' => "Franco (El)",
			'330241' => "Gijón",
			'330254' => "Gozón",
			'330267' => "Grado",
			'330273' => "Grandas de Salime",
			'330289' => "Ibias",
			'330292' => "Illano",
			'330306' => "Illas",
			'330313' => "Langreo",
			'330328' => "Laviana",
			'330334' => "Lena",
			'330352' => "Llanera",
			'330365' => "Llanes",
			'330371' => "Mieres",
			'330387' => "Morcín",
			'330390' => "Muros de Nalón",
			'330404' => "Nava",
			'330411' => "Navia",
			'330426' => "Noreña",
			'330432' => "Onís",
			'330447' => "Oviedo",
			'330450' => "Parres",
			'330463' => "Peñamellera Alta",
			'330479' => "Peñamellera Baja",
			'330485' => "Pesoz",
			'330498' => "Piloña",
			'330501' => "Ponga",
			'330518' => "Pravia",
			'330523' => "Proaza",
			'330539' => "Quirós",
			'330544' => "Regueras (Las)",
			'330557' => "Ribadedeva",
			'330560' => "Ribadesella",
			'330576' => "Ribera de Arriba",
			'330582' => "Riosa",
			'330595' => "Salas",
			'330616' => "San Martín de Oscos",
			'330609' => "San Martín del Rey Aurelio",
			'330637' => "San Tirso de Abres",
			'330621' => "Santa Eulalia de Oscos",
			'330642' => "Santo Adriano",
			'330655' => "Sariego",
			'330668' => "Siero",
			'330674' => "Sobrescobio",
			'330680' => "Somiedo",
			'330693' => "Soto del Barco",
			'330707' => "Tapia de Casariego",
			'330714' => "Taramundi",
			'330729' => "Teverga",
			'330735' => "Tineo",
			'330349' => "Valdés",
			'330740' => "Vegadeo",
			'330753' => "Villanueva de Oscos",
			'330766' => "Villaviciosa",
			'330772' => "Villayón",
			'330788' => "Yernes y Tameza",
			'340014' => "Abarca de Campos",
			'340035' => "Abia de las Torres",
			'340040' => "Aguilar de Campoo",
			'340053' => "Alar del Rey",
			'340066' => "Alba de Cerrato",
			'340091' => "Amayuelas de Arriba",
			'340105' => "Ampudia",
			'340112' => "Amusco",
			'340127' => "Antigüedad",
			'340151' => "Arconada",
			'340170' => "Astudillo",
			'340186' => "Autilla del Pino",
			'340199' => "Autillo de Campos",
			'340203' => "Ayuela",
			'340225' => "Baltanás",
			'340246' => "Baquerín de Campos",
			'340259' => "Bárcena de Campos",
			'340278' => "Barruelo de Santullán",
			'340284' => "Báscones de Ojeda",
			'340297' => "Becerril de Campos",
			'340318' => "Belmonte de Campos",
			'340323' => "Berzosilla",
			'340339' => "Boada de Campos",
			'340357' => "Boadilla de Rioseco",
			'340344' => "Boadilla del Camino",
			'340360' => "Brañosera",
			'340376' => "Buenavista de Valdavia",
			'340382' => "Bustillo de la Vega",
			'340395' => "Bustillo del Páramo de Carrión",
			'340416' => "Calahorra de Boedo",
			'340421' => "Calzada de los Molinos",
			'340455' => "Capillas",
			'340468' => "Cardeñosa de Volpejera",
			'340474' => "Carrión de los Condes",
			'340480' => "Castil de Vela",
			'340493' => "Castrejón de la Peña",
			'340506' => "Castrillo de Don Juan",
			'340513' => "Castrillo de Onielo",
			'340528' => "Castrillo de Villavega",
			'340534' => "Castromocho",
			'340552' => "Cervatos de la Cueza",
			'340565' => "Cervera de Pisuerga",
			'340571' => "Cevico de la Torre",
			'340587' => "Cevico Navero",
			'340590' => "Cisneros",
			'340604' => "Cobos de Cerrato",
			'340611' => "Collazos de Boedo",
			'340626' => "Congosto de Valdavia",
			'340632' => "Cordovilla la Real",
			'340663' => "Cubillas de Cerrato",
			'340679' => "Dehesa de Montejo",
			'340685' => "Dehesa de Romanos",
			'340698' => "Dueñas",
			'340702' => "Espinosa de Cerrato",
			'340719' => "Espinosa de Villagonzalo",
			'340724' => "Frechilla",
			'340730' => "Fresno del Río",
			'340745' => "Frómista",
			'340761' => "Fuentes de Nava",
			'340777' => "Fuentes de Valdepero",
			'340796' => "Grijota",
			'340800' => "Guardo",
			'340817' => "Guaza de Campos",
			'340822' => "Hérmedes de Cerrato",
			'340838' => "Herrera de Pisuerga",
			'340843' => "Herrera de Valdecañas",
			'340869' => "Hontoria de Cerrato",
			'340875' => "Hornillos de Cerrato",
			'340881' => "Husillos",
			'340894' => "Itero de la Vega",
			'340915' => "Lagartos",
			'340920' => "Lantadilla",
			'340941' => "Ledigos",
			'349036' => "Loma de Ucieza",
			'340967' => "Lomas",
			'340989' => "Magaz de Pisuerga",
			'340992' => "Manquillos",
			'341006' => "Mantinos",
			'341013' => "Marcilla de Campos",
			'341028' => "Mazariegos",
			'341034' => "Mazuecos de Valdeginate",
			'341049' => "Melgar de Yuso",
			'341065' => "Meneses de Campos",
			'341071' => "Micieces de Ojeda",
			'341087' => "Monzón de Campos",
			'341090' => "Moratinos",
			'341104' => "Mudá",
			'341126' => "Nogal de las Huertas",
			'341132' => "Olea de Boedo",
			'341147' => "Olmos de Ojeda",
			'341163' => "Osornillo",
			'349015' => "Osorno la Mayor",
			'341202' => "Palencia",
			'341219' => "Palenzuela",
			'341224' => "Páramo de Boedo",
			'341230' => "Paredes de Nava",
			'341245' => "Payo de Ojeda",
			'341258' => "Pedraza de Campos",
			'341261' => "Pedrosa de la Vega",
			'341277' => "Perales",
			'349041' => "Pernía (La)",
			'341296' => "Pino del Río",
			'341300' => "Piña de Campos",
			'341317' => "Población de Arroyo",
			'341322' => "Población de Campos",
			'341338' => "Población de Cerrato",
			'341343' => "Polentinos",
			'341356' => "Pomar de Valdivia",
			'341369' => "Poza de la Vega",
			'341375' => "Pozo de Urama",
			'341394' => "Prádanos de Ojeda",
			'341408' => "Puebla de Valdavia (La)",
			'341415' => "Quintana del Puente",
			'341436' => "Quintanilla de Onsoña",
			'341467' => "Reinoso de Cerrato",
			'341473' => "Renedo de la Vega",
			'341492' => "Requena de Campos",
			'341512' => "Respenda de la Peña",
			'341527' => "Revenga de Campos",
			'341548' => "Revilla de Collazos",
			'341551' => "Ribas de Campos",
			'341564' => "Riberos de la Cueza",
			'341570' => "Saldaña",
			'341586' => "Salinas de Pisuerga",
			'341599' => "San Cebrián de Campos",
			'341603' => "San Cebrián de Mudá",
			'341610' => "San Cristóbal de Boedo",
			'341631' => "San Mamés de Campos",
			'341659' => "San Román de la Cuba",
			'341678' => "Santa Cecilia del Alcor",
			'341684' => "Santa Cruz de Boedo",
			'341697' => "Santervás de la Vega",
			'341701' => "Santibáñez de Ecla",
			'341718' => "Santibáñez de la Peña",
			'341744' => "Santoyo",
			'341757' => "Serna (La)",
			'341776' => "Soto de Cerrato",
			'341760' => "Sotobañado y Priorato",
			'341782' => "Tabanera de Cerrato",
			'341795' => "Tabanera de Valdavia",
			'341809' => "Támara de Campos",
			'341816' => "Tariego de Cerrato",
			'341821' => "Torquemada",
			'341842' => "Torremormojón",
			'341855' => "Triollo",
			'341868' => "Valbuena de Pisuerga",
			'341893' => "Valdeolmillos",
			'341907' => "Valderrábano",
			'341929' => "Valde-Ucieza",
			'341966' => "Valle de Cerrato",
			'349020' => "Valle del Retortillo",
			'341991' => "Velilla del Río Carrión",
			'340231' => "Venta de Baños",
			'342012' => "Vertavillo",
			'340936' => "Vid de Ojeda (La)",
			'342027' => "Villabasta de Valdavia",
			'342048' => "Villacidaler",
			'342051' => "Villaconancio",
			'342064' => "Villada",
			'342086' => "Villaeles de Valdavia",
			'342103' => "Villahán",
			'342110' => "Villaherreros",
			'342131' => "Villalaco",
			'342146' => "Villalba de Guardo",
			'342159' => "Villalcázar de Sirga",
			'342162' => "Villalcón",
			'342178' => "Villalobón",
			'342184' => "Villaluenga de la Vega",
			'342201' => "Villamartín de Campos",
			'342218' => "Villamediana",
			'342223' => "Villameriel",
			'342239' => "Villamoronta",
			'342244' => "Villamuera de la Cueza",
			'342257' => "Villamuriel de Cerrato",
			'342276' => "Villanueva del Rebollar",
			'342282' => "Villanuño de Valdavia",
			'342295' => "Villaprovedo",
			'342309' => "Villarmentero de Campos",
			'342316' => "Villarrabé",
			'342321' => "Villarramiel",
			'342337' => "Villasarracino",
			'342342' => "Villasila de Valdavia",
			'342368' => "Villaturde",
			'342374' => "Villaumbrales",
			'342380' => "Villaviudas",
			'342407' => "Villerías de Campos",
			'342414' => "Villodre",
			'342429' => "Villodrigo",
			'342435' => "Villoldo",
			'342453' => "Villota del Páramo",
			'342466' => "Villovieco",
			'350017' => "Agaete",
			'350022' => "Agüimes",
			'350206' => "Aldea de San Nicolás (La)",
			'350038' => "Antigua",
			'350043' => "Arrecife",
			'350056' => "Artenara",
			'350069' => "Arucas",
			'350075' => "Betancuria",
			'350081' => "Firgas",
			'350094' => "Gáldar",
			'350108' => "Haría",
			'350115' => "Ingenio",
			'350120' => "Mogán",
			'350136' => "Moya",
			'350141' => "Oliva (La)",
			'350154' => "Pájara",
			'350167' => "Palmas de Gran Canaria (Las)",
			'350173' => "Puerto del Rosario",
			'350189' => "San Bartolomé",
			'350192' => "San Bartolomé de Tirajana",
			'350213' => "Santa Brígida",
			'350228' => "Santa Lucía de Tirajana",
			'350234' => "Santa María de Guía de Gran Canaria",
			'350249' => "Teguise",
			'350252' => "Tejeda",
			'350265' => "Telde",
			'350271' => "Teror",
			'350287' => "Tías",
			'350290' => "Tinajo",
			'350304' => "Tuineje",
			'350326' => "Valleseco",
			'350311' => "Valsequillo de Gran Canaria",
			'350332' => "Vega de San Mateo",
			'350347' => "Yaiza",
			'360209' => "Agolada",
			'360010' => "Arbo",
			'360031' => "Baiona",
			'360025' => "Barro",
			'360046' => "Bueu",
			'360059' => "Caldas de Reis",
			'360062' => "Cambados",
			'360078' => "Campo Lameiro",
			'360084' => "Cangas",
			'360097' => "Cañiza (A)",
			'360101' => "Catoira",
			'360118' => "Cerdedo",
			'360123' => "Cotobade",
			'360139' => "Covelo",
			'360144' => "Crecente",
			'360157' => "Cuntis",
			'360160' => "Dozón",
			'360176' => "Estrada (A)",
			'360182' => "Forcarei",
			'360195' => "Fornelos de Montes",
			'360216' => "Gondomar",
			'360221' => "Grove (O)",
			'360237' => "Guarda (A)",
			'369011' => "Illa de Arousa (A)",
			'360242' => "Lalín",
			'360255' => "Lama (A)",
			'360268' => "Marín",
			'360274' => "Meaño",
			'360280' => "Meis",
			'360293' => "Moaña",
			'360307' => "Mondariz",
			'360314' => "Mondariz-Balneario",
			'360329' => "Moraña",
			'360335' => "Mos",
			'360340' => "Neves (As)",
			'360353' => "Nigrán",
			'360366' => "Oia",
			'360372' => "Pazos de Borbén",
			'360412' => "Poio",
			'360433' => "Ponte Caldelas",
			'360427' => "Ponteareas",
			'360448' => "Pontecesures",
			'360388' => "Pontevedra",
			'360391' => "Porriño (O)",
			'360405' => "Portas",
			'360451' => "Redondela",
			'360464' => "Ribadumia",
			'360470' => "Rodeiro",
			'360486' => "Rosal (O)",
			'360499' => "Salceda de Caselas",
			'360502' => "Salvaterra de Miño",
			'360519' => "Sanxenxo",
			'360524' => "Silleda",
			'360530' => "Soutomaior",
			'360545' => "Tomiño",
			'360558' => "Tui",
			'360561' => "Valga",
			'360577' => "Vigo",
			'360596' => "Vila de Cruces",
			'360583' => "Vilaboa",
			'360600' => "Vilagarcía de Arousa",
			'360617' => "Vilanova de Arousa",
			'370016' => "Abusejo",
			'370021' => "Agallas",
			'370037' => "Ahigal de los Aceiteros",
			'370042' => "Ahigal de Villarino",
			'370055' => "Alameda de Gardón (La)",
			'370068' => "Alamedilla (La)",
			'370074' => "Alaraz",
			'370080' => "Alba de Tormes",
			'370093' => "Alba de Yeltes",
			'370107' => "Alberca (La)",
			'370114' => "Alberguería de Argañán (La)",
			'370129' => "Alconada",
			'370153' => "Aldea del Obispo",
			'370135' => "Aldeacipreste",
			'370140' => "Aldeadávila de la Ribera",
			'370166' => "Aldealengua",
			'370172' => "Aldeanueva de Figueroa",
			'370188' => "Aldeanueva de la Sierra",
			'370191' => "Aldearrodrigo",
			'370205' => "Aldearrubia",
			'370212' => "Aldeaseca de Alba",
			'370227' => "Aldeaseca de la Frontera",
			'370233' => "Aldeatejada",
			'370248' => "Aldeavieja de Tormes",
			'370251' => "Aldehuela de la Bóveda",
			'370264' => "Aldehuela de Yeltes",
			'370270' => "Almenara de Tormes",
			'370286' => "Almendra",
			'370299' => "Anaya de Alba",
			'370303' => "Añover de Tormes",
			'370310' => "Arabayona de Mógica",
			'370325' => "Arapiles",
			'370331' => "Arcediano",
			'370346' => "Arco (El)",
			'370359' => "Armenteros",
			'370378' => "Atalaya (La)",
			'370384' => "Babilafuente",
			'370397' => "Bañobárez",
			'370401' => "Barbadillo",
			'370418' => "Barbalos",
			'370423' => "Barceo",
			'370444' => "Barruecopardo",
			'370457' => "Bastida (La)",
			'370460' => "Béjar",
			'370476' => "Beleña",
			'370495' => "Bermellar",
			'370508' => "Berrocal de Huebra",
			'370515' => "Berrocal de Salvatierra",
			'370520' => "Boada",
			'370541' => "Bodón (El)",
			'370554' => "Bogajo",
			'370567' => "Bouza (La)",
			'370573' => "Bóveda del Río Almar",
			'370589' => "Brincones",
			'370592' => "Buenamadre",
			'370606' => "Buenavista",
			'370613' => "Cabaco (El)",
			'370634' => "Cabeza de Béjar (La)",
			'370652' => "Cabeza del Caballo",
			'370628' => "Cabezabellosa de la Calzada",
			'370671' => "Cabrerizos",
			'370687' => "Cabrillas",
			'370690' => "Calvarrasa de Abajo",
			'370704' => "Calvarrasa de Arriba",
			'370711' => "Calzada de Béjar (La)",
			'370726' => "Calzada de Don Diego",
			'370732' => "Calzada de Valdunciel",
			'370747' => "Campillo de Azaba",
			'370779' => "Campo de Peñaranda (El)",
			'370785' => "Candelario",
			'370798' => "Canillas de Abajo",
			'370802' => "Cantagallo",
			'370819' => "Cantalapiedra",
			'370824' => "Cantalpino",
			'370830' => "Cantaracillo",
			'370858' => "Carbajosa de la Sagrada",
			'370861' => "Carpio de Azaba",
			'370877' => "Carrascal de Barregas",
			'370883' => "Carrascal del Obispo",
			'370896' => "Casafranca",
			'370900' => "Casas del Conde (Las)",
			'370917' => "Casillas de Flores",
			'370922' => "Castellanos de Moriscos",
			'371857' => "Castellanos de Villiquera",
			'370969' => "Castillejo de Martín Viejo",
			'370975' => "Castraz",
			'370981' => "Cepeda",
			'370994' => "Cereceda de la Sierra",
			'371008' => "Cerezal de Peñahorcada",
			'371015' => "Cerralbo",
			'371020' => "Cerro (El)",
			'371036' => "Cespedosa de Tormes",
			'371149' => "Chagarcía Medianero",
			'371041' => "Cilleros de la Bastida",
			'371067' => "Cipérez",
			'371073' => "Ciudad Rodrigo",
			'371089' => "Coca de Alba",
			'371092' => "Colmenar de Montemayor",
			'371106' => "Cordovilla",
			'371128' => "Cristóbal",
			'371134' => "Cubo de Don Sancho (El)",
			'371152' => "Dios le Guarde",
			'371165' => "Doñinos de Ledesma",
			'371171' => "Doñinos de Salamanca",
			'371187' => "Ejeme",
			'371190' => "Encina (La)",
			'371204' => "Encina de San Silvestre",
			'371211' => "Encinas de Abajo",
			'371226' => "Encinas de Arriba",
			'371232' => "Encinasola de los Comendadores",
			'371247' => "Endrinal",
			'371250' => "Escurial de la Sierra",
			'371263' => "Espadaña",
			'371279' => "Espeja",
			'371285' => "Espino de la Orbada",
			'371298' => "Florida de Liébana",
			'371302' => "Forfoleda",
			'371319' => "Frades de la Sierra",
			'371324' => "Fregeneda (La)",
			'371330' => "Fresnedoso",
			'371345' => "Fresno Alhándiga",
			'371358' => "Fuente de San Esteban (La)",
			'371361' => "Fuenteguinaldo",
			'371377' => "Fuenteliante",
			'371383' => "Fuenterroble de Salvatierra",
			'371396' => "Fuentes de Béjar",
			'371400' => "Fuentes de Oñoro",
			'371417' => "Gajates",
			'371422' => "Galindo y Perahuy",
			'371438' => "Galinduste",
			'371443' => "Galisancho",
			'371456' => "Gallegos de Argañán",
			'371469' => "Gallegos de Solmirón",
			'371475' => "Garcibuey",
			'371481' => "Garcihernández",
			'371494' => "Garcirrey",
			'371507' => "Gejuelo del Barro",
			'371514' => "Golpejas",
			'371529' => "Gomecello",
			'371540' => "Guadramiro",
			'371553' => "Guijo de Ávila",
			'371566' => "Guijuelo",
			'371572' => "Herguijuela de Ciudad Rodrigo",
			'371588' => "Herguijuela de la Sierra",
			'371591' => "Herguijuela del Campo",
			'371605' => "Hinojosa de Duero",
			'371612' => "Horcajo de Montemayor",
			'371627' => "Horcajo Medianero",
			'371633' => "Hoya (La)",
			'371648' => "Huerta",
			'371651' => "Iruelos",
			'371664' => "Ituero de Azaba",
			'371670' => "Juzbado",
			'371686' => "Lagunilla",
			'371699' => "Larrodrigo",
			'371703' => "Ledesma",
			'371710' => "Ledrada",
			'371725' => "Linares de Riofrío",
			'371731' => "Lumbrales",
			'371759' => "Machacón",
			'371746' => "Macotera",
			'371762' => "Madroñal",
			'371778' => "Maíllo (El)",
			'371784' => "Malpartida",
			'371797' => "Mancera de Abajo",
			'371801' => "Manzano (El)",
			'371818' => "Martiago",
			'371839' => "Martín de Yeltes",
			'371823' => "Martinamor",
			'371844' => "Masueco",
			'371860' => "Mata de Ledesma (La)",
			'371876' => "Matilla de los Caños del Río",
			'371882' => "Maya (La)",
			'371895' => "Membribe de la Sierra",
			'371909' => "Mieza",
			'371916' => "Milano (El)",
			'371921' => "Miranda de Azán",
			'371937' => "Miranda del Castañar",
			'371942' => "Mogarraz",
			'371955' => "Molinillo",
			'371968' => "Monforte de la Sierra",
			'371974' => "Monleón",
			'371980' => "Monleras",
			'371993' => "Monsagro",
			'372007' => "Montejo",
			'372014' => "Montemayor del Río",
			'372029' => "Monterrubio de Armuña",
			'372035' => "Monterrubio de la Sierra",
			'372040' => "Morasverdes",
			'372053' => "Morille",
			'372066' => "Moríñigo",
			'372072' => "Moriscos",
			'372088' => "Moronta",
			'372091' => "Mozárbez",
			'372112' => "Narros de Matalayegua",
			'372133' => "Nava de Béjar",
			'372148' => "Nava de Francia",
			'372151' => "Nava de Sotrobal",
			'372127' => "Navacarros",
			'372164' => "Navales",
			'372170' => "Navalmoral de Béjar",
			'372186' => "Navamorales",
			'372199' => "Navarredonda de la Rinconada",
			'372210' => "Navasfrías",
			'372225' => "Negrilla de Palencia",
			'372231' => "Olmedo de Camaces",
			'372246' => "Orbada (La)",
			'372259' => "Pajares de la Laguna",
			'372262' => "Palacios del Arzobispo",
			'372284' => "Palaciosrubios",
			'372297' => "Palencia de Negrilla",
			'372301' => "Parada de Arriba",
			'372318' => "Parada de Rubiales",
			'372323' => "Paradinas de San Juan",
			'372339' => "Pastores",
			'372344' => "Payo (El)",
			'372357' => "Pedraza de Alba",
			'372360' => "Pedrosillo de Alba",
			'372376' => "Pedrosillo de los Aires",
			'372382' => "Pedrosillo el Ralo",
			'372395' => "Pedroso de la Armuña (El)",
			'372409' => "Pelabravo",
			'372416' => "Pelarrodríguez",
			'372421' => "Pelayos",
			'372437' => "Peña (La)",
			'372442' => "Peñacaballera",
			'372455' => "Peñaparda",
			'372468' => "Peñaranda de Bracamonte",
			'372474' => "Peñarandilla",
			'372480' => "Peralejos de Abajo",
			'372493' => "Peralejos de Arriba",
			'372506' => "Pereña de la Ribera",
			'372513' => "Peromingo",
			'372528' => "Pinedas",
			'372534' => "Pino de Tormes (El)",
			'372549' => "Pitiegua",
			'372552' => "Pizarral",
			'372565' => "Poveda de las Cintas",
			'372571' => "Pozos de Hinojo",
			'372587' => "Puebla de Azaba",
			'372590' => "Puebla de San Medel",
			'372604' => "Puebla de Yeltes",
			'372611' => "Puente del Congosto",
			'372626' => "Puertas",
			'372632' => "Puerto de Béjar",
			'372647' => "Puerto Seguro",
			'372650' => "Rágama",
			'372663' => "Redonda (La)",
			'372679' => "Retortillo",
			'372685' => "Rinconada de la Sierra (La)",
			'372698' => "Robleda",
			'372702' => "Robliza de Cojos",
			'372719' => "Rollán",
			'372724' => "Saelices el Chico",
			'372730' => "Sagrada (La)",
			'373034' => "Sahugo (El)",
			'372745' => "Salamanca",
			'372758' => "Saldeana",
			'372761' => "Salmoral",
			'372777' => "Salvatierra de Tormes",
			'372783' => "San Cristóbal de la Cuesta",
			'372843' => "San Esteban de la Sierra",
			'372856' => "San Felices de los Gallegos",
			'372869' => "San Martín del Castañar",
			'372875' => "San Miguel de Valero",
			'370362' => "San Miguel del Robledo",
			'372881' => "San Morales",
			'372894' => "San Muñoz",
			'372915' => "San Pedro de Rozados",
			'372908' => "San Pedro del Valle",
			'372920' => "San Pelayo de Guareña",
			'372800' => "Sanchón de la Ribera",
			'372817' => "Sanchón de la Sagrada",
			'372822' => "Sanchotello",
			'372796' => "Sancti-Spíritus",
			'372838' => "Sando",
			'372936' => "Santa María de Sando",
			'372941' => "Santa Marta de Tormes",
			'372967' => "Santiago de la Puebla",
			'372973' => "Santibáñez de Béjar",
			'372989' => "Santibáñez de la Sierra",
			'372992' => "Santiz",
			'373006' => "Santos (Los)",
			'373013' => "Sardón de los Frailes",
			'373028' => "Saucelle",
			'373049' => "Sepulcro-Hilario",
			'373052' => "Sequeros",
			'373065' => "Serradilla del Arroyo",
			'373071' => "Serradilla del Llano",
			'373090' => "Sierpe (La)",
			'373104' => "Sieteiglesias de Tormes",
			'373111' => "Sobradillo",
			'373126' => "Sorihuela",
			'373132' => "Sotoserrano",
			'373147' => "Tabera de Abajo",
			'373150' => "Tala (La)",
			'373163' => "Tamames",
			'373179' => "Tarazona de Guareña",
			'373185' => "Tardáguila",
			'373198' => "Tejado (El)",
			'373202' => "Tejeda y Segoyuela",
			'373219' => "Tenebrón",
			'373224' => "Terradillos",
			'373230' => "Topas",
			'373245' => "Tordillos",
			'373258' => "Tornadizo (El)",
			'373277' => "Torresmenudas",
			'373283' => "Trabanca",
			'373296' => "Tremedal de Tormes",
			'373300' => "Valdecarros",
			'373317' => "Valdefuentes de Sangusín",
			'373322' => "Valdehijaderos",
			'373338' => "Valdelacasa",
			'373343' => "Valdelageve",
			'373356' => "Valdelosa",
			'373369' => "Valdemierque",
			'373375' => "Valderrodrigo",
			'373381' => "Valdunciel",
			'373394' => "Valero",
			'373436' => "Vallejera de Riofrío",
			'373408' => "Valsalabroso",
			'373415' => "Valverde de Valdelacasa",
			'373420' => "Valverdón",
			'373441' => "Vecinos",
			'373454' => "Vega de Tirados",
			'373467' => "Veguillas (Las)",
			'373473' => "Vellés (La)",
			'373489' => "Ventosa del Río Almar",
			'373492' => "Vídola (La)",
			'373512' => "Villaflores",
			'373527' => "Villagonzalo de Tormes",
			'373533' => "Villalba de los Llanos",
			'373548' => "Villamayor",
			'373551' => "Villanueva del Conde",
			'373564' => "Villar de Argañán",
			'373570' => "Villar de Ciervo",
			'373586' => "Villar de Gallimazo",
			'373599' => "Villar de la Yegua",
			'373603' => "Villar de Peralonso",
			'373610' => "Villar de Samaniego",
			'373625' => "Villares de la Reina",
			'373631' => "Villares de Yeltes",
			'373646' => "Villarino de los Aires",
			'373659' => "Villarmayor",
			'373662' => "Villarmuerto",
			'373678' => "Villasbuenas",
			'373684' => "Villasdardo",
			'373697' => "Villaseco de los Gamitos",
			'373701' => "Villaseco de los Reyes",
			'373718' => "Villasrubias",
			'373723' => "Villaverde de Guareña",
			'373739' => "Villavieja de Yeltes",
			'373744' => "Villoria",
			'373757' => "Villoruela",
			'373505' => "Vilvestre",
			'373760' => "Vitigudino",
			'373776' => "Yecla de Yeltes",
			'373782' => "Zamarra",
			'373795' => "Zamayón",
			'373809' => "Zarapicos",
			'373816' => "Zarza de Pumareda (La)",
			'373821' => "Zorita de la Frontera",
			'380012' => "Adeje",
			'380027' => "Agulo",
			'380033' => "Alajeró",
			'380048' => "Arafo",
			'380051' => "Arico",
			'380064' => "Arona",
			'380070' => "Barlovento",
			'380086' => "Breña Alta",
			'380099' => "Breña Baja",
			'380103' => "Buenavista del Norte",
			'380110' => "Candelaria",
			'380125' => "Fasnia",
			'380131' => "Frontera",
			'380146' => "Fuencaliente de la Palma",
			'380159' => "Garachico",
			'380162' => "Garafía",
			'380178' => "Granadilla de Abona",
			'380184' => "Guancha (La)",
			'380197' => "Guía de Isora",
			'380201' => "Güímar",
			'380218' => "Hermigua",
			'380223' => "Icod de los Vinos",
			'380244' => "Llanos de Aridane (Los)",
			'380257' => "Matanza de Acentejo (La)",
			'380260' => "Orotava (La)",
			'380276' => "Paso (El)",
			'389013' => "Pinar de El Hierro (El)",
			'380282' => "Puerto de la Cruz",
			'380295' => "Puntagorda",
			'380309' => "Puntallana",
			'380316' => "Realejos (Los)",
			'380321' => "Rosario (El)",
			'380337' => "San Andrés y Sauces",
			'380239' => "San Cristóbal de La Laguna",
			'380342' => "San Juan de la Rambla",
			'380355' => "San Miguel de Abona",
			'380368' => "San Sebastián de la Gomera",
			'380374' => "Santa Cruz de la Palma",
			'380380' => "Santa Cruz de Tenerife",
			'380393' => "Santa Úrsula",
			'380407' => "Santiago del Teide",
			'380414' => "Sauzal (El)",
			'380429' => "Silos (Los)",
			'380435' => "Tacoronte",
			'380440' => "Tanque (El)",
			'380453' => "Tazacorte",
			'380466' => "Tegueste",
			'380472' => "Tijarafe",
			'380491' => "Valle Gran Rey",
			'380504' => "Vallehermoso",
			'380488' => "Valverde",
			'380511' => "Victoria de Acentejo (La)",
			'380526' => "Vilaflor",
			'380532' => "Villa de Mazo",
			'390015' => "Alfoz de Lloredo",
			'390020' => "Ampuero",
			'390036' => "Anievas",
			'390041' => "Arenas de Iguña",
			'390054' => "Argoños",
			'390067' => "Arnuero",
			'390073' => "Arredondo",
			'390089' => "Astillero (El)",
			'390092' => "Bárcena de Cicero",
			'390106' => "Bárcena de Pie de Concha",
			'390113' => "Bareyo",
			'390128' => "Cabezón de la Sal",
			'390134' => "Cabezón de Liébana",
			'390149' => "Cabuérniga",
			'390152' => "Camaleño",
			'390165' => "Camargo",
			'390279' => "Campoo de Enmedio",
			'390171' => "Campoo de Yuso",
			'390187' => "Cartes",
			'390190' => "Castañeda",
			'390204' => "Castro-Urdiales",
			'390211' => "Cieza",
			'390226' => "Cillorigo de Liébana",
			'390232' => "Colindres",
			'390247' => "Comillas",
			'390250' => "Corrales de Buelna (Los)",
			'390263' => "Corvera de Toranzo",
			'390285' => "Entrambasaguas",
			'390298' => "Escalante",
			'390302' => "Guriezo",
			'390319' => "Hazas de Cesto",
			'390324' => "Hermandad de Campoo de Suso",
			'390330' => "Herrerías",
			'390345' => "Lamasón",
			'390358' => "Laredo",
			'390361' => "Liendo",
			'390377' => "Liérganes",
			'390383' => "Limpias",
			'390396' => "Luena",
			'390400' => "Marina de Cudeyo",
			'390417' => "Mazcuerras",
			'390422' => "Medio Cudeyo",
			'390438' => "Meruelo",
			'390443' => "Miengo",
			'390456' => "Miera",
			'390469' => "Molledo",
			'390475' => "Noja",
			'390481' => "Penagos",
			'390494' => "Peñarrubia",
			'390507' => "Pesaguero",
			'390514' => "Pesquera",
			'390529' => "Piélagos",
			'390535' => "Polaciones",
			'390540' => "Polanco",
			'390553' => "Potes",
			'390566' => "Puente Viesgo",
			'390572' => "Ramales de la Victoria",
			'390588' => "Rasines",
			'390591' => "Reinosa",
			'390605' => "Reocín",
			'390612' => "Ribamontán al Mar",
			'390627' => "Ribamontán al Monte",
			'390633' => "Rionansa",
			'390648' => "Riotuerto",
			'390651' => "Rozas de Valdearroyo (Las)",
			'390664' => "Ruente",
			'390670' => "Ruesga",
			'390686' => "Ruiloba",
			'390699' => "San Felices de Buelna",
			'390703' => "San Miguel de Aguayo",
			'390710' => "San Pedro del Romeral",
			'390725' => "San Roque de Riomiera",
			'390801' => "San Vicente de la Barquera",
			'390731' => "Santa Cruz de Bezana",
			'390746' => "Santa María de Cayón",
			'390759' => "Santander",
			'390762' => "Santillana del Mar",
			'390778' => "Santiurde de Reinosa",
			'390784' => "Santiurde de Toranzo",
			'390797' => "Santoña",
			'390818' => "Saro",
			'390823' => "Selaya",
			'390839' => "Soba",
			'390844' => "Solórzano",
			'390857' => "Suances",
			'390860' => "Tojos (Los)",
			'390876' => "Torrelavega",
			'390882' => "Tresviso",
			'390895' => "Tudanca",
			'390909' => "Udías",
			'390955' => "Val de San Vicente",
			'390916' => "Valdáliga",
			'390921' => "Valdeolea",
			'390937' => "Valdeprado del Río",
			'390942' => "Valderredible",
			'391014' => "Valle de Villaverde",
			'390968' => "Vega de Liébana",
			'390974' => "Vega de Pas",
			'390980' => "Villacarriedo",
			'390993' => "Villaescusa",
			'391007' => "Villafufre",
			'391029' => "Voto",
			'400019' => "Abades",
			'400024' => "Adrada de Pirón",
			'400030' => "Adrados",
			'400045' => "Aguilafuente",
			'400058' => "Alconada de Maderuelo",
			'400122' => "Aldea Real",
			'400061' => "Aldealcorvo",
			'400077' => "Aldealengua de Pedraza",
			'400083' => "Aldealengua de Santa María",
			'400096' => "Aldeanueva de la Serrezuela",
			'400100' => "Aldeanueva del Codonal",
			'400138' => "Aldeasoña",
			'400143' => "Aldehorno",
			'400156' => "Aldehuela del Codonal",
			'400169' => "Aldeonte",
			'400175' => "Anaya",
			'400181' => "Añe",
			'400194' => "Arahuetes",
			'400208' => "Arcones",
			'400215' => "Arevalillo de Cega",
			'400220' => "Armuña",
			'400241' => "Ayllón",
			'400254' => "Barbolla",
			'400267' => "Basardilla",
			'400289' => "Bercial",
			'400292' => "Bercimuel",
			'400306' => "Bernardos",
			'400313' => "Bernuy de Porreros",
			'400328' => "Boceguillas",
			'400334' => "Brieva",
			'400349' => "Caballar",
			'400352' => "Cabañas de Polendos",
			'400365' => "Cabezuela",
			'400371' => "Calabazas de Fuentidueña",
			'400390' => "Campo de San Pedro",
			'400404' => "Cantalejo",
			'400411' => "Cantimpalos",
			'400432' => "Carbonero el Mayor",
			'400447' => "Carrascal del Río",
			'400450' => "Casla",
			'400463' => "Castillejo de Mesleón",
			'400479' => "Castro de Fuentidueña",
			'400485' => "Castrojimeno",
			'400498' => "Castroserna de Abajo",
			'400518' => "Castroserracín",
			'400523' => "Cedillo de la Torre",
			'400539' => "Cerezo de Abajo",
			'400544' => "Cerezo de Arriba",
			'400655' => "Chañe",
			'400557' => "Cilleruelo de San Mamés",
			'400560' => "Cobos de Fuentidueña",
			'400576' => "Coca",
			'400582' => "Codorniz",
			'400595' => "Collado Hermoso",
			'400609' => "Condado de Castilnovo",
			'400616' => "Corral de Ayllón",
			'409025' => "Cozuelos de Fuentidueña",
			'400621' => "Cubillo",
			'400637' => "Cuéllar",
			'409059' => "Cuevas de Provanco",
			'400680' => "Domingo García",
			'400693' => "Donhierro",
			'400707' => "Duruelo",
			'400714' => "Encinas",
			'400729' => "Encinillas",
			'400735' => "Escalona del Prado",
			'400740' => "Escarabajosa de Cabezas",
			'400753' => "Escobar de Polendos",
			'400766' => "Espinar (El)",
			'400772' => "Espirdo",
			'400788' => "Fresneda de Cuéllar",
			'400791' => "Fresno de Cantespino",
			'400805' => "Fresno de la Fuente",
			'400812' => "Frumales",
			'400827' => "Fuente de Santa Cruz",
			'400833' => "Fuente el Olmo de Fuentidueña",
			'400848' => "Fuente el Olmo de Íscar",
			'400864' => "Fuentepelayo",
			'400870' => "Fuentepiñel",
			'400886' => "Fuenterrebollo",
			'400899' => "Fuentesaúco de Fuentidueña",
			'400910' => "Fuentesoto",
			'400925' => "Fuentidueña",
			'400931' => "Gallegos",
			'400946' => "Garcillán",
			'400959' => "Gomezserracín",
			'400978' => "Grajera",
			'400997' => "Honrubia de la Cuesta",
			'401001' => "Hontalbilla",
			'401018' => "Hontanares de Eresma",
			'401039' => "Huertos (Los)",
			'401044' => "Ituero y Lama",
			'401057' => "Juarros de Riomoros",
			'401060' => "Juarros de Voltoya",
			'401076' => "Labajos",
			'401082' => "Laguna de Contreras",
			'401095' => "Languilla",
			'401109' => "Lastras de Cuéllar",
			'401116' => "Lastras del Pozo",
			'401121' => "Lastrilla (La)",
			'401137' => "Losa (La)",
			'401155' => "Maderuelo",
			'409031' => "Marazoleja",
			'401180' => "Marazuela",
			'401193' => "Martín Miguel",
			'401207' => "Martín Muñoz de la Dehesa",
			'401214' => "Martín Muñoz de las Posadas",
			'401229' => "Marugán",
			'401240' => "Mata de Cuéllar",
			'401235' => "Matabuena",
			'401253' => "Matilla (La)",
			'401266' => "Melque de Cercos",
			'401272' => "Membibre de la Hoz",
			'401288' => "Migueláñez",
			'401291' => "Montejo de Arévalo",
			'401305' => "Montejo de la Vega de la Serrezuela",
			'401312' => "Monterrubio",
			'401327' => "Moral de Hornuez",
			'401348' => "Mozoncillo",
			'401351' => "Muñopedro",
			'401364' => "Muñoveros",
			'401386' => "Nava de la Asunción",
			'401399' => "Navafría",
			'401403' => "Navalilla",
			'401410' => "Navalmanzano",
			'401425' => "Navares de Ayuso",
			'401431' => "Navares de Enmedio",
			'401446' => "Navares de las Cuevas",
			'401459' => "Navas de Oro",
			'409046' => "Navas de Riofrío",
			'401462' => "Navas de San Antonio",
			'401484' => "Nieva",
			'401497' => "Olombrada",
			'401500' => "Orejana",
			'401517' => "Ortigosa de Pestaño",
			'409010' => "Ortigosa del Monte",
			'401522' => "Otero de Herreros",
			'401543' => "Pajarejos",
			'401556' => "Palazuelos de Eresma",
			'401569' => "Pedraza",
			'401575' => "Pelayos del Arroyo",
			'401581' => "Perosillo",
			'401594' => "Pinarejos",
			'401608' => "Pinarnegrillo",
			'401615' => "Pradales",
			'401620' => "Prádena",
			'401636' => "Puebla de Pedraza",
			'401641' => "Rapariegos",
			'401654' => "Rebollo",
			'401667' => "Remondo",
			'401689' => "Riaguas de San Bartolomé",
			'401706' => "Riaza",
			'401713' => "Ribota",
			'401728' => "Riofrío de Riaza",
			'401734' => "Roda de Eresma",
			'401749' => "Sacramenia",
			'401765' => "Samboal",
			'401771' => "San Cristóbal de Cuéllar",
			'401787' => "San Cristóbal de la Vega",
			'409062' => "San Cristóbal de Segovia",
			'401811' => "San Ildefonso",
			'401826' => "San Martín y Mudrián",
			'401832' => "San Miguel de Bernuy",
			'401847' => "San Pedro de Gaíllos",
			'401790' => "Sanchonuño",
			'401804' => "Sangarcía",
			'401850' => "Santa María la Real de Nieva",
			'401863' => "Santa Marta del Cerro",
			'401885' => "Santiuste de Pedraza",
			'401898' => "Santiuste de San Juan Bautista",
			'401902' => "Santo Domingo de Pirón",
			'401919' => "Santo Tomé del Puerto",
			'401924' => "Sauquillo de Cabezas",
			'401930' => "Sebúlcor",
			'401945' => "Segovia",
			'401958' => "Sepúlveda",
			'401961' => "Sequera de Fresno",
			'401983' => "Sotillo",
			'401996' => "Sotosalbos",
			'402000' => "Tabanera la Luenga",
			'402017' => "Tolocirio",
			'402069' => "Torre Val de San Pedro",
			'402022' => "Torreadrada",
			'402038' => "Torrecaballeros",
			'402043' => "Torrecilla del Pinar",
			'402056' => "Torreiglesias",
			'402075' => "Trescasas",
			'402081' => "Turégano",
			'402108' => "Urueñas",
			'402115' => "Valdeprados",
			'402120' => "Valdevacas de Montejo",
			'402136' => "Valdevacas y Guijar",
			'402189' => "Valle de Tabladillo",
			'402192' => "Vallelado",
			'402206' => "Valleruela de Pedraza",
			'402213' => "Valleruela de Sepúlveda",
			'402141' => "Valseca",
			'402154' => "Valtiendas",
			'402167' => "Valverde del Majano",
			'402228' => "Veganzones",
			'402234' => "Vegas de Matute",
			'402249' => "Ventosilla y Tejadilla",
			'402252' => "Villacastín",
			'402287' => "Villaverde de Íscar",
			'402290' => "Villaverde de Montejo",
			'402304' => "Villeguillo",
			'402311' => "Yanguas de Eresma",
			'402332' => "Zarzuela del Monte",
			'402347' => "Zarzuela del Pinar",
			'410016' => "Aguadulce",
			'410021' => "Alanís",
			'410037' => "Albaida del Aljarafe",
			'410042' => "Alcalá de Guadaíra",
			'410055' => "Alcalá del Río",
			'410068' => "Alcolea del Río",
			'410074' => "Algaba (La)",
			'410080' => "Algámitas",
			'410093' => "Almadén de la Plata",
			'410107' => "Almensilla",
			'410114' => "Arahal",
			'410129' => "Aznalcázar",
			'410135' => "Aznalcóllar",
			'410140' => "Badolatosa",
			'410153' => "Benacazón",
			'410166' => "Bollullos de la Mitación",
			'410172' => "Bormujos",
			'410188' => "Brenes",
			'410191' => "Burguillos",
			'410205' => "Cabezas de San Juan (Las)",
			'410212' => "Camas",
			'410227' => "Campana (La)",
			'410233' => "Cantillana",
			'419017' => "Cañada Rosal",
			'410248' => "Carmona",
			'410251' => "Carrión de los Céspedes",
			'410264' => "Casariche",
			'410270' => "Castilblanco de los Arroyos",
			'410286' => "Castilleja de Guzmán",
			'410299' => "Castilleja de la Cuesta",
			'410303' => "Castilleja del Campo",
			'410310' => "Castillo de las Guardas (El)",
			'410325' => "Cazalla de la Sierra",
			'410331' => "Constantina",
			'410346' => "Coria del Río",
			'410359' => "Coripe",
			'410362' => "Coronil (El)",
			'410378' => "Corrales (Los)",
			'419038' => "Cuervo de Sevilla (El)",
			'410384' => "Dos Hermanas",
			'410397' => "Écija",
			'410401' => "Espartinas",
			'410418' => "Estepa",
			'410423' => "Fuentes de Andalucía",
			'410439' => "Garrobo (El)",
			'410444' => "Gelves",
			'410457' => "Gerena",
			'410460' => "Gilena",
			'410476' => "Gines",
			'410482' => "Guadalcanal",
			'410495' => "Guillena",
			'410508' => "Herrera",
			'410515' => "Huévar del Aljarafe",
			'419022' => "Isla Mayor",
			'410520' => "Lantejuela (La)",
			'410536' => "Lebrija",
			'410541' => "Lora de Estepa",
			'410554' => "Lora del Río",
			'410567' => "Luisiana (La)",
			'410573' => "Madroño (El)",
			'410589' => "Mairena del Alcor",
			'410592' => "Mairena del Aljarafe",
			'410606' => "Marchena",
			'410613' => "Marinaleda",
			'410628' => "Martín de la Jara",
			'410634' => "Molares (Los)",
			'410649' => "Montellano",
			'410652' => "Morón de la Frontera",
			'410665' => "Navas de la Concepción (Las)",
			'410671' => "Olivares",
			'410687' => "Osuna",
			'410690' => "Palacios y Villafranca (Los)",
			'410704' => "Palomares del Río",
			'410711' => "Paradas",
			'410726' => "Pedrera",
			'410732' => "Pedroso (El)",
			'410747' => "Peñaflor",
			'410750' => "Pilas",
			'410763' => "Pruna",
			'410779' => "Puebla de Cazalla (La)",
			'410785' => "Puebla de los Infantes (La)",
			'410798' => "Puebla del Río (La)",
			'410802' => "Real de la Jara (El)",
			'410819' => "Rinconada (La)",
			'410824' => "Roda de Andalucía (La)",
			'410830' => "Ronquillo (El)",
			'410845' => "Rubio (El)",
			'410858' => "Salteras",
			'410861' => "San Juan de Aznalfarache",
			'410883' => "San Nicolás del Puerto",
			'410877' => "Sanlúcar la Mayor",
			'410896' => "Santiponce",
			'410900' => "Saucejo (El)",
			'410917' => "Sevilla",
			'410922' => "Tocina",
			'410938' => "Tomares",
			'410943' => "Umbrete",
			'410956' => "Utrera",
			'410969' => "Valencina de la Concepción",
			'410975' => "Villamanrique de la Condesa",
			'411008' => "Villanueva de San Juan",
			'410981' => "Villanueva del Ariscal",
			'410994' => "Villanueva del Río y Minas",
			'411015' => "Villaverde del Río",
			'411020' => "Viso del Alcor (El)",
			'420011' => "Abejar",
			'420032' => "Adradas",
			'420047' => "Ágreda",
			'420063' => "Alconaba",
			'420079' => "Alcubilla de Avellaneda",
			'420085' => "Alcubilla de las Peñas",
			'420098' => "Aldealafuente",
			'420102' => "Aldealices",
			'420119' => "Aldealpozo",
			'420124' => "Aldealseñor",
			'420130' => "Aldehuela de Periáñez",
			'420145' => "Aldehuelas (Las)",
			'420158' => "Alentisque",
			'420161' => "Aliud",
			'420177' => "Almajano",
			'420183' => "Almaluez",
			'420196' => "Almarza",
			'420200' => "Almazán",
			'420217' => "Almazul",
			'420222' => "Almenar de Soria",
			'420238' => "Alpanseque",
			'420243' => "Arancón",
			'420256' => "Arcos de Jalón",
			'420269' => "Arenillas",
			'420275' => "Arévalo de la Sierra",
			'420281' => "Ausejo de la Sierra",
			'420294' => "Baraona",
			'420308' => "Barca",
			'420315' => "Barcones",
			'420320' => "Bayubas de Abajo",
			'420336' => "Bayubas de Arriba",
			'420341' => "Beratón",
			'420354' => "Berlanga de Duero",
			'420367' => "Blacos",
			'420373' => "Bliecos",
			'420389' => "Borjabad",
			'420392' => "Borobia",
			'420413' => "Buberos",
			'420428' => "Buitrago",
			'420434' => "Burgo de Osma-Ciudad de Osma",
			'420449' => "Cabrejas del Campo",
			'420452' => "Cabrejas del Pinar",
			'420465' => "Calatañazor",
			'420487' => "Caltojar",
			'420490' => "Candilichera",
			'420503' => "Cañamaque",
			'420510' => "Carabantes",
			'420525' => "Caracena",
			'420531' => "Carrascosa de Abajo",
			'420546' => "Carrascosa de la Sierra",
			'420559' => "Casarejos",
			'420562' => "Castilfrío de la Sierra",
			'420584' => "Castillejo de Robledo",
			'420578' => "Castilruiz",
			'420597' => "Centenera de Andaluz",
			'420601' => "Cerbón",
			'420618' => "Cidones",
			'420623' => "Cigudosa",
			'420639' => "Cihuela",
			'420644' => "Ciria",
			'420657' => "Cirujales del Río",
			'420682' => "Coscurita",
			'420695' => "Covaleda",
			'420709' => "Cubilla",
			'420716' => "Cubo de la Solana",
			'420737' => "Cueva de Ágreda",
			'420755' => "Dévanos",
			'420768' => "Deza",
			'420780' => "Duruelo de la Sierra",
			'420793' => "Escobosa de Almazán",
			'420807' => "Espeja de San Marcelino",
			'420814' => "Espejón",
			'420829' => "Estepa de San Juan",
			'420835' => "Frechilla de Almazán",
			'420840' => "Fresno de Caracena",
			'420853' => "Fuentearmegil",
			'420866' => "Fuentecambrón",
			'420872' => "Fuentecantos",
			'420888' => "Fuentelmonge",
			'420891' => "Fuentelsaz de Soria",
			'420905' => "Fuentepinilla",
			'420927' => "Fuentes de Magaña",
			'420933' => "Fuentestrún",
			'420948' => "Garray",
			'420951' => "Golmayo",
			'420964' => "Gómara",
			'420970' => "Gormaz",
			'420986' => "Herrera de Soria",
			'421003' => "Hinojosa del Campo",
			'421031' => "Langa de Duero",
			'421059' => "Liceras",
			'421062' => "Losilla (La)",
			'421078' => "Magaña",
			'421084' => "Maján",
			'421101' => "Matalebreras",
			'421118' => "Matamala de Almazán",
			'421139' => "Medinaceli",
			'421157' => "Miño de Medinaceli",
			'421160' => "Miño de San Esteban",
			'421176' => "Molinos de Duero",
			'421182' => "Momblona",
			'421195' => "Monteagudo de las Vicarías",
			'421209' => "Montejo de Tiermes",
			'421216' => "Montenegro de Cameros",
			'421237' => "Morón de Almazán",
			'421242' => "Muriel de la Fuente",
			'421255' => "Muriel Viejo",
			'421274' => "Nafría de Ucero",
			'421280' => "Narros",
			'421293' => "Navaleno",
			'421307' => "Nepas",
			'421314' => "Nolay",
			'421329' => "Noviercas",
			'421340' => "Ólvega",
			'421353' => "Oncala",
			'421391' => "Pinilla del Campo",
			'421405' => "Portillo de Soria",
			'421412' => "Póveda de Soria (La)",
			'421427' => "Pozalmuro",
			'421448' => "Quintana Redonda",
			'421451' => "Quintanas de Gormaz",
			'421486' => "Quiñonería",
			'421499' => "Rábanos (Los)",
			'421519' => "Rebollar",
			'421524' => "Recuerda",
			'421530' => "Rello",
			'421545' => "Renieblas",
			'421558' => "Retortillo de Soria",
			'421561' => "Reznos",
			'421577' => "Riba de Escalote (La)",
			'421583' => "Rioseco de Soria",
			'421596' => "Rollamienta",
			'421600' => "Royo (El)",
			'421617' => "Salduero",
			'421622' => "San Esteban de Gormaz",
			'421638' => "San Felices",
			'421643' => "San Leonardo de Yagüe",
			'421656' => "San Pedro Manrique",
			'421669' => "Santa Cruz de Yanguas",
			'421675' => "Santa María de Huerta",
			'421681' => "Santa María de las Hoyas",
			'421715' => "Serón de Nágima",
			'421720' => "Soliedra",
			'421736' => "Soria",
			'421741' => "Sotillo del Rincón",
			'421754' => "Suellacabras",
			'421767' => "Tajahuerce",
			'421773' => "Tajueco",
			'421789' => "Talveila",
			'421813' => "Tardelcuende",
			'421828' => "Taroda",
			'421834' => "Tejado",
			'421849' => "Torlengua",
			'421852' => "Torreblacos",
			'421871' => "Torrubia de Soria",
			'421887' => "Trévago",
			'421890' => "Ucero",
			'421904' => "Vadillo",
			'421911' => "Valdeavellano de Tera",
			'421926' => "Valdegeña",
			'421932' => "Valdelagua del Cerro",
			'421947' => "Valdemaluque",
			'421950' => "Valdenebro",
			'421963' => "Valdeprado",
			'421979' => "Valderrodilla",
			'421985' => "Valtajeros",
			'422002' => "Velamazán",
			'422019' => "Velilla de la Sierra",
			'422024' => "Velilla de los Ajos",
			'422045' => "Viana de Duero",
			'422058' => "Villaciervos",
			'422061' => "Villanueva de Gormaz",
			'422077' => "Villar del Ala",
			'422083' => "Villar del Campo",
			'422096' => "Villar del Río",
			'422117' => "Villares de Soria (Los)",
			'422122' => "Villasayas",
			'422138' => "Villaseca de Arciel",
			'422156' => "Vinuesa",
			'422169' => "Vizmanos",
			'422175' => "Vozmediano",
			'422181' => "Yanguas",
			'422194' => "Yelo",
			'430017' => "Aiguamúrcia",
			'430022' => "Albinyana",
			'430038' => "Albiol (L')",
			'430043' => "Alcanar",
			'430056' => "Alcover",
			'439044' => "Aldea (L')",
			'430069' => "Aldover",
			'430075' => "Aleixar (L')",
			'430081' => "Alfara de Carles",
			'430094' => "Alforja",
			'430108' => "Alió",
			'430115' => "Almoster",
			'430120' => "Altafulla",
			'430136' => "Ametlla de Mar (L')",
			'439060' => "Ampolla (L')",
			'430141' => "Amposta",
			'430167' => "Arboç (L')",
			'430154' => "Arbolí",
			'430173' => "Argentera (L')",
			'430189' => "Arnes",
			'430192' => "Ascó",
			'430206' => "Banyeres del Penedès",
			'430213' => "Barberà de la Conca",
			'430228' => "Batea",
			'430234' => "Bellmunt del Priorat",
			'430249' => "Bellvei",
			'430252' => "Benifallet",
			'430265' => "Benissanet",
			'430271' => "Bisbal de Falset (La)",
			'430287' => "Bisbal del Penedès (La)",
			'430290' => "Blancafort",
			'430304' => "Bonastre",
			'430311' => "Borges del Camp (Les)",
			'430326' => "Bot",
			'430332' => "Botarell",
			'430347' => "Bràfim",
			'430350' => "Cabacés",
			'430363' => "Cabra del Camp",
			'430379' => "Calafell",
			'439039' => "Camarles",
			'430385' => "Cambrils",
			'430398' => "Capafonts",
			'430402' => "Capçanes",
			'430419' => "Caseres",
			'430424' => "Castellvell del Camp",
			'430430' => "Catllar (El)",
			'430458' => "Colldejou",
			'430461' => "Conesa",
			'430477' => "Constantí",
			'430483' => "Corbera d'Ebre",
			'430496' => "Cornudella de Montsant",
			'430509' => "Creixell",
			'430516' => "Cunit",
			'439018' => "Deltebre",
			'430537' => "Duesaigües",
			'430542' => "Espluga de Francolí (L')",
			'430555' => "Falset",
			'430568' => "Fatarella (La)",
			'430574' => "Febró (La)",
			'430580' => "Figuera (La)",
			'430593' => "Figuerola del Camp",
			'430607' => "Flix",
			'430614' => "Forès",
			'430629' => "Freginals",
			'430635' => "Galera (La)",
			'430640' => "Gandesa",
			'430653' => "Garcia",
			'430666' => "Garidells (Els)",
			'430672' => "Ginestar",
			'430688' => "Godall",
			'430691' => "Gratallops",
			'430705' => "Guiamets (Els)",
			'430712' => "Horta de Sant Joan",
			'430727' => "Lloar (El)",
			'430733' => "Llorac",
			'430748' => "Llorenç del Penedès",
			'430764' => "Marçà",
			'430751' => "Margalef",
			'430770' => "Mas de Barberans",
			'430786' => "Masdenverge",
			'430799' => "Masllorenç",
			'430803' => "Masó (La)",
			'430810' => "Maspujols",
			'430825' => "Masroig (El)",
			'430831' => "Milà (El)",
			'430846' => "Miravet",
			'430859' => "Molar (El)",
			'430862' => "Montblanc",
			'430884' => "Montbrió del Camp",
			'430897' => "Montferri",
			'430901' => "Montmell (El)",
			'430918' => "Mont-ral",
			'430923' => "Mont-roig del Camp",
			'430939' => "Móra d'Ebre",
			'430944' => "Móra la Nova",
			'430957' => "Morell (El)",
			'430960' => "Morera de Montsant (La)",
			'430976' => "Nou de Gaià (La)",
			'430982' => "Nulles",
			'431009' => "Pallaresos (Els)",
			'430995' => "Palma d'Ebre (La)",
			'431016' => "Passanant i Belltall",
			'431021' => "Paüls",
			'431037' => "Perafort",
			'431042' => "Perelló (El)",
			'431055' => "Piles (Les)",
			'431068' => "Pinell de Brai (El)",
			'431074' => "Pira",
			'431080' => "Pla de Santa Maria (El)",
			'431093' => "Pobla de Mafumet (La)",
			'431107' => "Pobla de Massaluca (La)",
			'431114' => "Pobla de Montornès (La)",
			'431129' => "Poboleda",
			'431135' => "Pont d'Armentera (El)",
			'431418' => "Pontils",
			'431140' => "Porrera",
			'431153' => "Pradell de la Teixeta",
			'431166' => "Prades",
			'431172' => "Prat de Comte",
			'431188' => "Pratdip",
			'431191' => "Puigpelat",
			'431205' => "Querol",
			'431212' => "Rasquera",
			'431227' => "Renau",
			'431233' => "Reus",
			'431248' => "Riba (La)",
			'431251' => "Riba-roja d'Ebre",
			'431264' => "Riera de Gaià (La)",
			'431270' => "Riudecanyes",
			'431286' => "Riudecols",
			'431299' => "Riudoms",
			'431303' => "Rocafort de Queralt",
			'431310' => "Roda de Barà",
			'431325' => "Rodonyà",
			'431331' => "Roquetes",
			'431346' => "Rourell (El)",
			'431359' => "Salomó",
			'439057' => "Salou",
			'431362' => "Sant Carles de la Ràpita",
			'431378' => "Sant Jaume dels Domenys",
			'439023' => "Sant Jaume d'Enveja",
			'431384' => "Santa Bàrbara",
			'431397' => "Santa Coloma de Queralt",
			'431401' => "Santa Oliva",
			'431423' => "Sarral",
			'431439' => "Savallà del Comtat",
			'431444' => "Secuita (La)",
			'431457' => "Selva del Camp (La)",
			'431460' => "Senan",
			'430445' => "Sénia (La)",
			'431476' => "Solivella",
			'431482' => "Tarragona",
			'431495' => "Tivenys",
			'431508' => "Tivissa",
			'431515' => "Torre de Fontaubella (La)",
			'431520' => "Torre de l'Espanyol (La)",
			'431536' => "Torredembarra",
			'431541' => "Torroja del Priorat",
			'431554' => "Tortosa",
			'431567' => "Ulldecona",
			'431573' => "Ulldemolins",
			'431589' => "Vallclara",
			'431592' => "Vallfogona de Riucorb",
			'431606' => "Vallmoll",
			'431613' => "Valls",
			'431628' => "Vandellòs i l'Hospitalet de l'Infant",
			'431634' => "Vendrell (El)",
			'431649' => "Vespella de Gaià",
			'431652' => "Vilabella",
			'431750' => "Vilalba dels Arcs",
			'431665' => "Vilallonga del Camp",
			'431687' => "Vilanova de Prades",
			'431671' => "Vilanova d'Escornalbou",
			'431690' => "Vilaplana",
			'431704' => "Vila-rodona",
			'431711' => "Vila-seca",
			'431726' => "Vilaverd",
			'431732' => "Vilella Alta (La)",
			'431747' => "Vilella Baixa (La)",
			'431763' => "Vimbodí i Poblet",
			'431779' => "Vinebre",
			'431785' => "Vinyols i els Arcs",
			'430521' => "Xerta",
			'440012' => "Ababuj",
			'440027' => "Abejuela",
			'440033' => "Aguatón",
			'440048' => "Aguaviva",
			'440051' => "Aguilar del Alfambra",
			'440064' => "Alacón",
			'440070' => "Alba",
			'440086' => "Albalate del Arzobispo",
			'440099' => "Albarracín",
			'440103' => "Albentosa",
			'440110' => "Alcaine",
			'440125' => "Alcalá de la Selva",
			'440131' => "Alcañiz",
			'440146' => "Alcorisa",
			'440162' => "Alfambra",
			'440178' => "Aliaga",
			'440218' => "Allepuz",
			'440223' => "Alloza",
			'440239' => "Allueva",
			'440184' => "Almohaja",
			'440197' => "Alobras",
			'440201' => "Alpeñés",
			'440244' => "Anadón",
			'440257' => "Andorra",
			'440260' => "Arcos de las Salinas",
			'440276' => "Arens de Lledó",
			'440282' => "Argente",
			'440295' => "Ariño",
			'440316' => "Azaila",
			'440321' => "Bádenas",
			'440337' => "Báguena",
			'440342' => "Bañón",
			'440355' => "Barrachina",
			'440368' => "Bea",
			'440374' => "Beceite",
			'440393' => "Bello",
			'440380' => "Belmonte de San José",
			'440407' => "Berge",
			'440414' => "Bezas",
			'440429' => "Blancas",
			'440435' => "Blesa",
			'440440' => "Bordón",
			'440453' => "Bronchales",
			'440466' => "Bueña",
			'440472' => "Burbáguena",
			'440488' => "Cabra de Mora",
			'440491' => "Calaceite",
			'440504' => "Calamocha",
			'440511' => "Calanda",
			'440526' => "Calomarde",
			'440532' => "Camañas",
			'440547' => "Camarena de la Sierra",
			'440550' => "Camarillas",
			'440563' => "Caminreal",
			'440598' => "Cantavieja",
			'440602' => "Cañada de Benatanduz",
			'440619' => "Cañada de Verich (La)",
			'440624' => "Cañada Vellida",
			'440630' => "Cañizar del Olivar",
			'440645' => "Cascante del Río",
			'440658' => "Castejón de Tornos",
			'440661' => "Castel de Cabra",
			'440700' => "Castellar (El)",
			'440717' => "Castellote",
			'440677' => "Castelnou",
			'440683' => "Castelserás",
			'440743' => "Cedrillas",
			'440756' => "Celadas",
			'440769' => "Cella",
			'440775' => "Cerollera (La)",
			'440808' => "Codoñera (La)",
			'440820' => "Corbalán",
			'440841' => "Cortes de Aragón",
			'440854' => "Cosa",
			'440867' => "Cretas",
			'440873' => "Crivillén",
			'440889' => "Cuba (La)",
			'440892' => "Cubla",
			'440906' => "Cucalón",
			'440928' => "Cuervo (El)",
			'440934' => "Cuevas de Almudén",
			'440949' => "Cuevas Labradas",
			'440965' => "Ejulve",
			'440971' => "Escorihuela",
			'440990' => "Escucha",
			'441004' => "Estercuel",
			'441011' => "Ferreruela de Huerva",
			'441026' => "Fonfría",
			'441032' => "Formiche Alto",
			'441050' => "Fórnoles",
			'441063' => "Fortanete",
			'441079' => "Foz-Calanda",
			'441085' => "Fresneda (La)",
			'441098' => "Frías de Albarracín",
			'441102' => "Fuenferrada",
			'441119' => "Fuentes Calientes",
			'441124' => "Fuentes Claras",
			'441130' => "Fuentes de Rubielos",
			'441145' => "Fuentespalda",
			'441158' => "Galve",
			'441161' => "Gargallo",
			'441177' => "Gea de Albarracín",
			'441183' => "Ginebrosa (La)",
			'441196' => "Griegos",
			'441200' => "Guadalaviar",
			'441217' => "Gúdar",
			'441222' => "Híjar",
			'441238' => "Hinojosa de Jarque",
			'441243' => "Hoz de la Vieja (La)",
			'441256' => "Huesa del Común",
			'441269' => "Iglesuela del Cid (La)",
			'441275' => "Jabaloyas",
			'441281' => "Jarque de la Val",
			'441294' => "Jatiel",
			'441308' => "Jorcas",
			'441315' => "Josa",
			'441320' => "Lagueruela",
			'441336' => "Lanzuela",
			'441354' => "Libros",
			'441367' => "Lidón",
			'441373' => "Linares de Mora",
			'441413' => "Lledó",
			'441389' => "Loscos",
			'441428' => "Maicas",
			'441434' => "Manzanera",
			'441449' => "Martín del Río",
			'441452' => "Mas de las Matas",
			'441465' => "Mata de los Olmos (La)",
			'441471' => "Mazaleón",
			'441487' => "Mezquita de Jarque",
			'441490' => "Mirambel",
			'441503' => "Miravete de la Sierra",
			'441510' => "Molinos",
			'441525' => "Monforte de Moyuela",
			'441531' => "Monreal del Campo",
			'441546' => "Monroyo",
			'441559' => "Montalbán",
			'441562' => "Monteagudo del Castillo",
			'441578' => "Monterde de Albarracín",
			'441584' => "Mora de Rubielos",
			'441597' => "Moscardón",
			'441601' => "Mosqueruela",
			'441618' => "Muniesa",
			'441639' => "Noguera de Albarracín",
			'441644' => "Nogueras",
			'441657' => "Nogueruelas",
			'441676' => "Obón",
			'441682' => "Odón",
			'441695' => "Ojos Negros",
			'441716' => "Olba",
			'441721' => "Oliete",
			'441737' => "Olmos (Los)",
			'441742' => "Orihuela del Tremedal",
			'441755' => "Orrios",
			'441768' => "Palomar de Arroyos",
			'441774' => "Pancrudo",
			'441780' => "Parras de Castellote (Las)",
			'441793' => "Peñarroya de Tastavins",
			'441807' => "Peracense",
			'441814' => "Peralejos",
			'441829' => "Perales del Alfambra",
			'441835' => "Pitarque",
			'441840' => "Plou",
			'441853' => "Pobo (El)",
			'441872' => "Portellada (La)",
			'441891' => "Pozondón",
			'441905' => "Pozuel del Campo",
			'441912' => "Puebla de Híjar (La)",
			'441927' => "Puebla de Valverde (La)",
			'441933' => "Puertomingalvo",
			'441948' => "Ráfales",
			'441951' => "Rillo",
			'441964' => "Riodeva",
			'441970' => "Ródenas",
			'441986' => "Royuela",
			'441999' => "Rubiales",
			'442003' => "Rubielos de la Cérida",
			'442010' => "Rubielos de Mora",
			'442031' => "Salcedillo",
			'442046' => "Saldón",
			'442059' => "Samper de Calanda",
			'442062' => "San Agustín",
			'442078' => "San Martín del Río",
			'442084' => "Santa Cruz de Nogueras",
			'442097' => "Santa Eulalia",
			'442101' => "Sarrión",
			'442118' => "Segura de los Baños",
			'442123' => "Seno",
			'442139' => "Singra",
			'442157' => "Terriente",
			'442160' => "Teruel",
			'442176' => "Toril y Masegoso",
			'442182' => "Tormón",
			'442195' => "Tornos",
			'442209' => "Torralba de los Sisones",
			'442237' => "Torre de Arcas",
			'442242' => "Torre de las Arcas",
			'442255' => "Torre del Compte",
			'442274' => "Torre los Negros",
			'442216' => "Torrecilla de Alcañiz",
			'442221' => "Torrecilla del Rebollar",
			'442268' => "Torrelacárcel",
			'442280' => "Torremocha de Jiloca",
			'442293' => "Torres de Albarracín",
			'442307' => "Torrevelilla",
			'442314' => "Torrijas",
			'442329' => "Torrijo del Campo",
			'442340' => "Tramacastiel",
			'442353' => "Tramacastilla",
			'442366' => "Tronchón",
			'442372' => "Urrea de Gaén",
			'442388' => "Utrillas",
			'442391' => "Valacloche",
			'442405' => "Valbona",
			'442412' => "Valdealgorfa",
			'442433' => "Valdecuenca",
			'442448' => "Valdelinares",
			'442451' => "Valdeltormo",
			'442464' => "Valderrobres",
			'442470' => "Valjunquera",
			'442499' => "Vallecillo (El)",
			'442502' => "Veguillas de la Sierra",
			'442519' => "Villafranca del Campo",
			'442524' => "Villahermosa del Campo",
			'442561' => "Villanueva del Rebollar de la Sierra",
			'442577' => "Villar del Cobo",
			'442583' => "Villar del Salz",
			'442600' => "Villarluengo",
			'442617' => "Villarquemado",
			'442622' => "Villarroya de los Pinares",
			'442638' => "Villastar",
			'442643' => "Villel",
			'442656' => "Vinaceite",
			'442669' => "Visiedo",
			'442675' => "Vivel del Río Martín",
			'442681' => "Zoma (La)",
			'450015' => "Ajofrín",
			'450020' => "Alameda de la Sagra",
			'450036' => "Albarreal de Tajo",
			'450041' => "Alcabón",
			'450054' => "Alcañizo",
			'450067' => "Alcaudete de la Jara",
			'450073' => "Alcolea de Tajo",
			'450089' => "Aldea en Cabo",
			'450092' => "Aldeanueva de Barbarroya",
			'450106' => "Aldeanueva de San Bartolomé",
			'450113' => "Almendral de la Cañada",
			'450128' => "Almonacid de Toledo",
			'450134' => "Almorox",
			'450149' => "Añover de Tajo",
			'450152' => "Arcicóllar",
			'450165' => "Argés",
			'450171' => "Azután",
			'450187' => "Barcience",
			'450190' => "Bargas",
			'450204' => "Belvís de la Jara",
			'450211' => "Borox",
			'450226' => "Buenaventura",
			'450232' => "Burguillos de Toledo",
			'450247' => "Burujón",
			'450250' => "Cabañas de la Sagra",
			'450263' => "Cabañas de Yepes",
			'450279' => "Cabezamesada",
			'450285' => "Calera y Chozas",
			'450298' => "Caleruela",
			'450302' => "Calzada de Oropesa",
			'450319' => "Camarena",
			'450324' => "Camarenilla",
			'450330' => "Campillo de la Jara (El)",
			'450345' => "Camuñas",
			'450358' => "Cardiel de los Montes",
			'450361' => "Carmena",
			'450377' => "Carpio de Tajo (El)",
			'450383' => "Carranque",
			'450396' => "Carriches",
			'450400' => "Casar de Escalona (El)",
			'450417' => "Casarrubios del Monte",
			'450422' => "Casasbuenas",
			'450438' => "Castillo de Bayuela",
			'450456' => "Cazalegas",
			'450469' => "Cebolla",
			'450475' => "Cedillo del Condado",
			'450481' => "Cerralbos (Los)",
			'450494' => "Cervera de los Montes",
			'450566' => "Chozas de Canales",
			'450572' => "Chueca",
			'450507' => "Ciruelos",
			'450514' => "Cobeja",
			'450529' => "Cobisa",
			'450535' => "Consuegra",
			'450540' => "Corral de Almaguer",
			'450553' => "Cuerva",
			'450588' => "Domingo Pérez",
			'450591' => "Dosbarrios",
			'450605' => "Erustes",
			'450612' => "Escalona",
			'450627' => "Escalonilla",
			'450633' => "Espinoso del Rey",
			'450648' => "Esquivias",
			'450651' => "Estrella (La)",
			'450664' => "Fuensalida",
			'450670' => "Gálvez",
			'450686' => "Garciotum",
			'450699' => "Gerindote",
			'450703' => "Guadamur",
			'450710' => "Guardia (La)",
			'450725' => "Herencias (Las)",
			'450731' => "Herreruela de Oropesa",
			'450746' => "Hinojosa de San Vicente",
			'450759' => "Hontanar",
			'450762' => "Hormigos",
			'450778' => "Huecas",
			'450784' => "Huerta de Valdecarábanos",
			'450797' => "Iglesuela (La)",
			'450801' => "Illán de Vacas",
			'450818' => "Illescas",
			'450823' => "Lagartera",
			'450839' => "Layos",
			'450844' => "Lillo",
			'450857' => "Lominchar",
			'450860' => "Lucillos",
			'450876' => "Madridejos",
			'450882' => "Magán",
			'450895' => "Malpica de Tajo",
			'450909' => "Manzaneque",
			'450916' => "Maqueda",
			'450921' => "Marjaliza",
			'450937' => "Marrupe",
			'450942' => "Mascaraque",
			'450955' => "Mata (La)",
			'450968' => "Mazarambroz",
			'450974' => "Mejorada",
			'450980' => "Menasalbas",
			'450993' => "Méntrida",
			'451007' => "Mesegar de Tajo",
			'451014' => "Miguel Esteban",
			'451029' => "Mocejón",
			'451035' => "Mohedas de la Jara",
			'451040' => "Montearagón",
			'451053' => "Montesclaros",
			'451066' => "Mora",
			'451072' => "Nambroca",
			'451088' => "Nava de Ricomalillo (La)",
			'451091' => "Navahermosa",
			'451105' => "Navalcán",
			'451112' => "Navalmoralejo",
			'451127' => "Navalmorales (Los)",
			'451133' => "Navalucillos (Los)",
			'451148' => "Navamorcuende",
			'451151' => "Noblejas",
			'451164' => "Noez",
			'451170' => "Nombela",
			'451186' => "Novés",
			'451199' => "Numancia de la Sagra",
			'451203' => "Nuño Gómez",
			'451210' => "Ocaña",
			'451225' => "Olías del Rey",
			'451231' => "Ontígola",
			'451246' => "Orgaz",
			'451259' => "Oropesa",
			'451262' => "Otero",
			'451278' => "Palomeque",
			'451284' => "Pantoja",
			'451297' => "Paredes de Escalona",
			'451301' => "Parrillas",
			'451318' => "Pelahustán",
			'451323' => "Pepino",
			'451339' => "Polán",
			'451344' => "Portillo de Toledo",
			'451357' => "Puebla de Almoradiel (La)",
			'451360' => "Puebla de Montalbán (La)",
			'451376' => "Pueblanueva (La)",
			'451382' => "Puente del Arzobispo (El)",
			'451395' => "Puerto de San Vicente",
			'451409' => "Pulgar",
			'451416' => "Quero",
			'451421' => "Quintanar de la Orden",
			'451437' => "Quismondo",
			'451442' => "Real de San Vicente (El)",
			'451455' => "Recas",
			'451468' => "Retamoso de la Jara",
			'451474' => "Rielves",
			'451480' => "Robledo del Mazo",
			'451493' => "Romeral (El)",
			'451506' => "San Bartolomé de las Abiertas",
			'451513' => "San Martín de Montalbán",
			'451528' => "San Martín de Pusa",
			'451534' => "San Pablo de los Montes",
			'451549' => "San Román de los Montes",
			'451552' => "Santa Ana de Pusa",
			'451565' => "Santa Cruz de la Zarza",
			'451571' => "Santa Cruz del Retamar",
			'451587' => "Santa Olalla",
			'459016' => "Santo Domingo-Caudilla",
			'451590' => "Sartajada",
			'451604' => "Segurilla",
			'451611' => "Seseña",
			'451626' => "Sevilleja de la Jara",
			'451632' => "Sonseca",
			'451647' => "Sotillo de las Palomas",
			'451650' => "Talavera de la Reina",
			'451663' => "Tembleque",
			'451679' => "Toboso (El)",
			'451685' => "Toledo",
			'451698' => "Torralba de Oropesa",
			'451719' => "Torre de Esteban Hambrán (La)",
			'451702' => "Torrecilla de la Jara",
			'451724' => "Torrico",
			'451730' => "Torrijos",
			'451745' => "Totanés",
			'451758' => "Turleque",
			'451761' => "Ugena",
			'451777' => "Urda",
			'451796' => "Valdeverdeja",
			'451800' => "Valmojado",
			'451817' => "Velada",
			'451822' => "Ventas con Peña Aguilera (Las)",
			'451838' => "Ventas de Retamosa (Las)",
			'451843' => "Ventas de San Julián (Las)",
			'451869' => "Villa de Don Fadrique (La)",
			'451856' => "Villacañas",
			'451875' => "Villafranca de los Caballeros",
			'451881' => "Villaluenga de la Sagra",
			'451894' => "Villamiel de Toledo",
			'451908' => "Villaminaya",
			'451915' => "Villamuelas",
			'451920' => "Villanueva de Alcardete",
			'451936' => "Villanueva de Bogas",
			'451941' => "Villarejo de Montalbán",
			'451954' => "Villarrubia de Santiago",
			'451967' => "Villaseca de la Sagra",
			'451973' => "Villasequilla",
			'451989' => "Villatobas",
			'451992' => "Viso de San Juan (El)",
			'452006' => "Yébenes (Los)",
			'452013' => "Yeles",
			'452028' => "Yepes",
			'452034' => "Yuncler",
			'452049' => "Yunclillos",
			'452052' => "Yuncos",
			'460018' => "Ademuz",
			'460023' => "Ador",
			'460044' => "Agullent",
			'460425' => "Aielo de Malferit",
			'460431' => "Aielo de Rugat",
			'460057' => "Alaquàs",
			'460060' => "Albaida",
			'460076' => "Albal",
			'460082' => "Albalat de la Ribera",
			'460095' => "Albalat dels Sorells",
			'460109' => "Albalat dels Tarongers",
			'460116' => "Alberic",
			'460121' => "Alborache",
			'460137' => "Alboraya",
			'460142' => "Albuixech",
			'460168' => "Alcàntera de Xúquer",
			'460155' => "Alcàsser",
			'460180' => "Alcublas",
			'460193' => "Alcúdia (l')",
			'460207' => "Alcúdia de Crespins (l')",
			'460214' => "Aldaia",
			'460229' => "Alfafar",
			'460240' => "Alfara de Algimia",
			'460253' => "Alfara del Patriarca",
			'460266' => "Alfarp",
			'460272' => "Alfarrasí",
			'460235' => "Alfauir",
			'460288' => "Algar de Palancia",
			'460291' => "Algemesí",
			'460305' => "Algimia de Alfara",
			'460312' => "Alginet",
			'460327' => "Almàssera",
			'460333' => "Almiserà",
			'460348' => "Almoines",
			'460351' => "Almussafes",
			'460364' => "Alpuente",
			'460370' => "Alqueria de la Comtessa (l')",
			'460174' => "Alzira",
			'460386' => "Andilla",
			'460399' => "Anna",
			'460403' => "Antella",
			'460410' => "Aras de los Olmos",
			'460039' => "Atzeneta d'Albaida",
			'460446' => "Ayora",
			'460462' => "Barx",
			'460459' => "Barxeta",
			'460478' => "Bèlgida",
			'460484' => "Bellreguard",
			'460497' => "Bellús",
			'460500' => "Benagéber",
			'460517' => "Benaguasil",
			'460522' => "Benavites",
			'460538' => "Beneixida",
			'460543' => "Benetússer",
			'460556' => "Beniarjó",
			'460569' => "Beniatjar",
			'460575' => "Benicolet",
			'469045' => "Benicull de Xúquer",
			'460608' => "Benifaió",
			'460594' => "Benifairó de la Valldigna",
			'460581' => "Benifairó de les Valls",
			'460615' => "Beniflá",
			'460620' => "Benigánim",
			'460636' => "Benimodo",
			'460641' => "Benimuslem",
			'460654' => "Beniparrell",
			'460667' => "Benirredrà",
			'460673' => "Benisanó",
			'460689' => "Benissoda",
			'460692' => "Benisuera",
			'460706' => "Bétera",
			'460713' => "Bicorp",
			'460728' => "Bocairent",
			'460734' => "Bolbaite",
			'460749' => "Bonrepòs i Mirambell",
			'460752' => "Bufali",
			'460765' => "Bugarra",
			'460771' => "Buñol",
			'460787' => "Burjassot",
			'460790' => "Calles",
			'460804' => "Camporrobles",
			'460811' => "Canals",
			'460826' => "Canet d'En Berenguer",
			'460832' => "Carcaixent",
			'460847' => "Càrcer",
			'460850' => "Carlet",
			'460863' => "Carrícola",
			'460879' => "Casas Altas",
			'460885' => "Casas Bajas",
			'460898' => "Casinos",
			'460902' => "Castelló de Rugat",
			'460919' => "Castellonet de la Conquesta",
			'460924' => "Castielfabib",
			'460930' => "Catadau",
			'460945' => "Catarroja",
			'460958' => "Caudete de las Fuentes",
			'460961' => "Cerdà",
			'461075' => "Chella",
			'461069' => "Chelva",
			'461081' => "Chera",
			'461094' => "Cheste",
			'461115' => "Chiva",
			'461120' => "Chulilla",
			'460977' => "Cofrentes",
			'460983' => "Corbera",
			'460996' => "Cortes de Pallás",
			'461000' => "Cotes",
			'461056' => "Cullera",
			'461136' => "Daimús",
			'461141' => "Domeño",
			'461154' => "Dos Aguas",
			'461167' => "Eliana (l')",
			'461173' => "Emperador",
			'461189' => "Enguera",
			'461192' => "Ènova (l')",
			'461206' => "Estivella",
			'461213' => "Estubeny",
			'461228' => "Faura",
			'461234' => "Favara",
			'461265' => "Foios",
			'461287' => "Font de la Figuera (la)",
			'461271' => "Font d'En Carròs (la)",
			'461249' => "Fontanars dels Alforins",
			'461252' => "Fortaleny",
			'461290' => "Fuenterrobles",
			'461311' => "Gandia",
			'469024' => "Gátova",
			'461304' => "Gavarda",
			'461326' => "Genovés",
			'461332' => "Gestalgar",
			'461347' => "Gilet",
			'461350' => "Godella",
			'461363' => "Godelleta",
			'461379' => "Granja de la Costera (la)",
			'461385' => "Guadasequies",
			'461398' => "Guadassuar",
			'461402' => "Guardamar de la Safor",
			'461419' => "Higueruelas",
			'461424' => "Jalance",
			'461445' => "Jarafuel",
			'461542' => "Llanera de Ranes",
			'461555' => "Llaurí",
			'461477' => "Llíria",
			'461521' => "Llocnou de la Corona",
			'461537' => "Llocnou de Sant Jeroni",
			'461516' => "Llocnou d'En Fenollet",
			'461568' => "Llombai",
			'461574' => "Llosa de Ranes (la)",
			'461509' => "Llutxent",
			'461483' => "Loriguilla",
			'461496' => "Losa del Obispo",
			'461580' => "Macastre",
			'461593' => "Manises",
			'461607' => "Manuel",
			'461614' => "Marines",
			'461629' => "Masalavés",
			'461635' => "Massalfassar",
			'461640' => "Massamagrell",
			'461653' => "Massanassa",
			'461666' => "Meliana",
			'461672' => "Millares",
			'461688' => "Miramar",
			'461691' => "Mislata",
			'461705' => "Mogente/Moixent",
			'461712' => "Moncada",
			'461733' => "Montaverner",
			'461748' => "Montesa",
			'461751' => "Montitxelvo/Montichelvo",
			'461764' => "Montroy",
			'461727' => "Montserrat",
			'461770' => "Museros",
			'461786' => "Náquera",
			'461799' => "Navarrés",
			'461803' => "Novelé/Novetlè",
			'461810' => "Oliva",
			'461831' => "Olleria (l')",
			'461825' => "Olocau",
			'461846' => "Ontinyent",
			'461859' => "Otos",
			'461862' => "Paiporta",
			'461878' => "Palma de Gandía",
			'461884' => "Palmera",
			'461897' => "Palomar (el)",
			'461901' => "Paterna",
			'461918' => "Pedralba",
			'461923' => "Petrés",
			'461939' => "Picanya",
			'461944' => "Picassent",
			'461957' => "Piles",
			'461960' => "Pinet",
			'461995' => "Pobla de Farnals (la)",
			'462021' => "Pobla de Vallbona (la)",
			'462009' => "Pobla del Duc (la)",
			'462037' => "Pobla Llarga (la)",
			'461976' => "Polinyà de Xúquer",
			'461982' => "Potríes",
			'462055' => "Puçol",
			'462016' => "Puebla de San Miguel",
			'462042' => "Puig",
			'461017' => "Quart de les Valls",
			'461022' => "Quart de Poblet",
			'461038' => "Quartell",
			'461043' => "Quatretonda",
			'462068' => "Quesa",
			'462074' => "Rafelbuñol/Rafelbunyol",
			'462080' => "Rafelcofer",
			'462093' => "Rafelguaraf",
			'462107' => "Ráfol de Salem",
			'462129' => "Real",
			'462114' => "Real de Gandía",
			'462135' => "Requena",
			'462140' => "Riba-roja de Túria",
			'462153' => "Riola",
			'462166' => "Rocafort",
			'462172' => "Rotglà i Corberà",
			'462188' => "Rótova",
			'462191' => "Rugat",
			'462205' => "Sagunto/Sagunt",
			'462212' => "Salem",
			'469030' => "San Antonio de Benagéber",
			'462227' => "Sant Joan de l'Ènova",
			'462233' => "Sedaví",
			'462248' => "Segart",
			'462251' => "Sellent",
			'462264' => "Sempere",
			'462270' => "Senyera",
			'462286' => "Serra",
			'462299' => "Siete Aguas",
			'462303' => "Silla",
			'462310' => "Simat de la Valldigna",
			'462325' => "Sinarcas",
			'462331' => "Sollana",
			'462346' => "Sot de Chera",
			'462359' => "Sueca",
			'462362' => "Sumacàrcer",
			'462378' => "Tavernes Blanques",
			'462384' => "Tavernes de la Valldigna",
			'462397' => "Teresa de Cofrentes",
			'462401' => "Terrateig",
			'462418' => "Titaguas",
			'462423' => "Torrebaja",
			'462439' => "Torrella",
			'462444' => "Torrent",
			'462457' => "Torres Torres",
			'462460' => "Tous",
			'462476' => "Tuéjar",
			'462482' => "Turís",
			'462495' => "Utiel",
			'462508' => "Valencia",
			'462515' => "Vallada",
			'462520' => "Vallanca",
			'462536' => "Vallés",
			'462541' => "Venta del Moro",
			'462567' => "Vilamarxant",
			'462554' => "Villalonga",
			'462573' => "Villanueva de Castellón",
			'462589' => "Villar del Arzobispo",
			'462592' => "Villargordo del Cabriel",
			'462606' => "Vinalesa",
			'461458' => "Xàtiva",
			'461430' => "Xeraco",
			'461461' => "Xeresa",
			'461108' => "Xirivella",
			'462613' => "Yátova",
			'462628' => "Yesa (La)",
			'462634' => "Zarra",
			'470014' => "Adalia",
			'470029' => "Aguasal",
			'470035' => "Aguilar de Campos",
			'470040' => "Alaejos",
			'470053' => "Alcazarén",
			'470066' => "Aldea de San Miguel",
			'470072' => "Aldeamayor de San Martín",
			'470088' => "Almenara de Adaja",
			'470091' => "Amusquillo",
			'470105' => "Arroyo de la Encomienda",
			'470112' => "Ataquines",
			'470127' => "Bahabón",
			'470133' => "Barcial de la Loma",
			'470148' => "Barruelo del Valle",
			'470151' => "Becilla de Valderaduey",
			'470164' => "Benafarces",
			'470170' => "Bercero",
			'470186' => "Berceruelo",
			'470199' => "Berrueces",
			'470203' => "Bobadilla del Campo",
			'470210' => "Bocigas",
			'470225' => "Bocos de Duero",
			'470231' => "Boecillo",
			'470246' => "Bolaños de Campos",
			'470259' => "Brahojos de Medina",
			'470262' => "Bustillo de Chaves",
			'470278' => "Cabezón de Pisuerga",
			'470284' => "Cabezón de Valderaduey",
			'470297' => "Cabreros del Monte",
			'470301' => "Campaspero",
			'470318' => "Campillo (El)",
			'470323' => "Camporredondo",
			'470339' => "Canalejas de Peñafiel",
			'470344' => "Canillas de Esgueva",
			'470357' => "Carpio",
			'470360' => "Casasola de Arión",
			'470376' => "Castrejón de Trabancos",
			'470382' => "Castrillo de Duero",
			'470395' => "Castrillo-Tejeriego",
			'470409' => "Castrobol",
			'470416' => "Castrodeza",
			'470421' => "Castromembibre",
			'470437' => "Castromonte",
			'470442' => "Castronuevo de Esgueva",
			'470455' => "Castronuño",
			'470468' => "Castroponce",
			'470474' => "Castroverde de Cerrato",
			'470480' => "Ceinos de Campos",
			'470493' => "Cervillego de la Cruz",
			'470506' => "Cigales",
			'470513' => "Ciguñuela",
			'470528' => "Cistérniga",
			'470534' => "Cogeces de Íscar",
			'470549' => "Cogeces del Monte",
			'470552' => "Corcos",
			'470565' => "Corrales de Duero",
			'470571' => "Cubillas de Santa Marta",
			'470587' => "Cuenca de Campos",
			'470590' => "Curiel de Duero",
			'470604' => "Encinas de Esgueva",
			'470611' => "Esguevillas de Esgueva",
			'470626' => "Fombellida",
			'470632' => "Fompedraza",
			'470647' => "Fontihoyuelo",
			'470650' => "Fresno el Viejo",
			'470663' => "Fuensaldaña",
			'470679' => "Fuente el Sol",
			'470685' => "Fuente-Olmedo",
			'470698' => "Gallegos de Hornija",
			'470702' => "Gatón de Campos",
			'470719' => "Geria",
			'470730' => "Herrín de Campos",
			'470745' => "Hornillos de Eresma",
			'470758' => "Íscar",
			'470761' => "Laguna de Duero",
			'470777' => "Langayo",
			'470796' => "Llano de Olmedo",
			'470783' => "Lomoviejo",
			'470800' => "Manzanillo",
			'470817' => "Marzales",
			'470822' => "Matapozuelos",
			'470838' => "Matilla de los Caños",
			'470843' => "Mayorga",
			'470869' => "Medina de Rioseco",
			'470856' => "Medina del Campo",
			'470875' => "Megeces",
			'470881' => "Melgar de Abajo",
			'470894' => "Melgar de Arriba",
			'470908' => "Mojados",
			'470915' => "Monasterio de Vega",
			'470920' => "Montealegre de Campos",
			'470936' => "Montemayor de Pililla",
			'470941' => "Moral de la Reina",
			'470954' => "Moraleja de las Panaderas",
			'470967' => "Morales de Campos",
			'470973' => "Mota del Marqués",
			'470989' => "Mucientes",
			'470992' => "Mudarra (La)",
			'471006' => "Muriel",
			'471013' => "Nava del Rey",
			'471028' => "Nueva Villa de las Torres",
			'471034' => "Olivares de Duero",
			'471049' => "Olmedo",
			'471052' => "Olmos de Esgueva",
			'471065' => "Olmos de Peñafiel",
			'471090' => "Palazuelo de Vedija",
			'471104' => "Parrilla (La)",
			'471111' => "Pedraja de Portillo (La)",
			'471126' => "Pedrajas de San Esteban",
			'471132' => "Pedrosa del Rey",
			'471147' => "Peñafiel",
			'471150' => "Peñaflor de Hornija",
			'471163' => "Pesquera de Duero",
			'471179' => "Piña de Esgueva",
			'471185' => "Piñel de Abajo",
			'471198' => "Piñel de Arriba",
			'471219' => "Pollos",
			'471224' => "Portillo",
			'471230' => "Pozal de Gallinas",
			'471245' => "Pozaldez",
			'471258' => "Pozuelo de la Orden",
			'471261' => "Puras",
			'471277' => "Quintanilla de Arriba",
			'471296' => "Quintanilla de Onésimo",
			'471300' => "Quintanilla de Trigueros",
			'471283' => "Quintanilla del Molar",
			'471317' => "Rábano",
			'471322' => "Ramiro",
			'471338' => "Renedo de Esgueva",
			'471343' => "Roales de Campos",
			'471356' => "Robladillo",
			'471375' => "Roturas",
			'471381' => "Rubí de Bracamonte",
			'471394' => "Rueda",
			'471408' => "Saelices de Mayorga",
			'471415' => "Salvador de Zapardiel",
			'471420' => "San Cebrián de Mazote",
			'471436' => "San Llorente",
			'471441' => "San Martín de Valvení",
			'471454' => "San Miguel del Arroyo",
			'471467' => "San Miguel del Pino",
			'471473' => "San Pablo de la Moraleja",
			'471489' => "San Pedro de Latarce",
			'471492' => "San Pelayo",
			'471505' => "San Román de Hornija",
			'471512' => "San Salvador",
			'471564' => "San Vicente del Palacio",
			'471527' => "Santa Eufemia del Arroyo",
			'471533' => "Santervás de Campos",
			'471548' => "Santibáñez de Valcorba",
			'471551' => "Santovenia de Pisuerga",
			'471570' => "Sardón de Duero",
			'471586' => "Seca (La)",
			'471599' => "Serrada",
			'471603' => "Siete Iglesias de Trabancos",
			'471610' => "Simancas",
			'471625' => "Tamariz de Campos",
			'471631' => "Tiedra",
			'471646' => "Tordehumos",
			'471659' => "Tordesillas",
			'471697' => "Torre de Esgueva",
			'471701' => "Torre de Peñafiel",
			'471662' => "Torrecilla de la Abadesa",
			'471678' => "Torrecilla de la Orden",
			'471684' => "Torrecilla de la Torre",
			'471718' => "Torrelobatón",
			'471723' => "Torrescárcela",
			'471739' => "Traspinedo",
			'471744' => "Trigueros del Valle",
			'471757' => "Tudela de Duero",
			'471760' => "Unión de Campos (La)",
			'471776' => "Urones de Castroponce",
			'471782' => "Urueña",
			'471795' => "Valbuena de Duero",
			'471809' => "Valdearcos de la Vega",
			'471816' => "Valdenebro de los Valles",
			'471821' => "Valdestillas",
			'471837' => "Valdunquillo",
			'471868' => "Valladolid",
			'471842' => "Valoria la Buena",
			'471855' => "Valverde de Campos",
			'471874' => "Vega de Ruiponce",
			'471880' => "Vega de Valdetronco",
			'471893' => "Velascálvaro",
			'471907' => "Velilla",
			'471914' => "Velliza",
			'471929' => "Ventosa de la Cuesta",
			'471935' => "Viana de Cega",
			'471953' => "Villabáñez",
			'471966' => "Villabaruz de Campos",
			'471972' => "Villabrágima",
			'471988' => "Villacarralón",
			'471991' => "Villacid de Campos",
			'472005' => "Villaco",
			'472033' => "Villafrades de Campos",
			'472048' => "Villafranca de Duero",
			'472051' => "Villafrechós",
			'472064' => "Villafuerte",
			'472070' => "Villagarcía de Campos",
			'472086' => "Villagómez la Nueva",
			'472099' => "Villalán de Campos",
			'472103' => "Villalar de los Comuneros",
			'472110' => "Villalba de la Loma",
			'472125' => "Villalba de los Alcores",
			'472131' => "Villalbarba",
			'472146' => "Villalón de Campos",
			'472159' => "Villamuriel de Campos",
			'472162' => "Villán de Tordesillas",
			'472178' => "Villanubla",
			'472184' => "Villanueva de Duero",
			'472197' => "Villanueva de la Condesa",
			'472201' => "Villanueva de los Caballeros",
			'472218' => "Villanueva de los Infantes",
			'472223' => "Villanueva de San Mancio",
			'472239' => "Villardefrades",
			'472244' => "Villarmentero de Esgueva",
			'472257' => "Villasexmir",
			'472260' => "Villavaquerín",
			'472276' => "Villavellid",
			'472282' => "Villaverde de Medina",
			'472295' => "Villavicencio de los Caballeros",
			'471940' => "Viloria",
			'472309' => "Wamba",
			'472316' => "Zaratán",
			'472321' => "Zarza (La)",
			'480010' => "Abadiño",
			'480025' => "Abanto y Ciérvana-Abanto Zierbena",
			'489119' => "Ajangiz",
			'489124' => "Alonsotegi",
			'480031' => "Amorebieta-Etxano",
			'480046' => "Amoroto",
			'480059' => "Arakaldo",
			'480062' => "Arantzazu",
			'480932' => "Areatza",
			'480097' => "Arrankudiaga",
			'489145' => "Arratzu",
			'480101' => "Arrieta",
			'480118' => "Arrigorriaga",
			'480237' => "Artea",
			'480084' => "Artzentales",
			'480911' => "Atxondo",
			'480708' => "Aulesti",
			'480123' => "Bakio",
			'480904' => "Balmaseda",
			'480139' => "Barakaldo",
			'480144' => "Barrika",
			'480157' => "Basauri",
			'480926' => "Bedia",
			'480160' => "Berango",
			'480176' => "Bermeo",
			'480182' => "Berriatua",
			'480195' => "Berriz",
			'480209' => "Bilbao",
			'480216' => "Busturia",
			'489011' => "Derio",
			'480268' => "Dima",
			'480274' => "Durango",
			'480280' => "Ea",
			'480314' => "Elantxobe",
			'480329' => "Elorrio",
			'489026' => "Erandio",
			'480335' => "Ereño",
			'480340' => "Ermua",
			'480792' => "Errigoiti",
			'480293' => "Etxebarri",
			'480307' => "Etxebarria",
			'489063' => "Forua",
			'480353' => "Fruiz",
			'480366' => "Galdakao",
			'480372' => "Galdames",
			'480388' => "Gamiz-Fika",
			'480391' => "Garai",
			'480405' => "Gatika",
			'480412' => "Gautegiz Arteaga",
			'480464' => "Gernika-Lumo",
			'480448' => "Getxo",
			'480470' => "Gizaburuaga",
			'480427' => "Gordexola",
			'480433' => "Gorliz",
			'480451' => "Güeñes",
			'480486' => "Ibarrangelu",
			'480947' => "Igorre",
			'480499' => "Ispaster",
			'489102' => "Iurreta",
			'480502' => "Izurtza",
			'480221' => "Karrantza Harana/Valle de Carranza",
			'489079' => "Kortezubi",
			'480519' => "Lanestosa",
			'480524' => "Larrabetzu",
			'480530' => "Laukiz",
			'480545' => "Leioa",
			'480577' => "Lekeitio",
			'480558' => "Lemoa",
			'480561' => "Lemoiz",
			'480813' => "Lezama",
			'489032' => "Loiu",
			'480583' => "Mallabia",
			'480596' => "Mañaria",
			'480600' => "Markina-Xemein",
			'480617' => "Maruri-Jatabe",
			'480622' => "Mendata",
			'480638' => "Mendexa",
			'480643' => "Meñaka",
			'480669' => "Morga",
			'480681' => "Mundaka",
			'480694' => "Mungia",
			'480078' => "Munitibar-Arbatzegi Gerrikaitz",
			'489085' => "Murueta",
			'480715' => "Muskiz",
			'480675' => "Muxika",
			'489098' => "Nabarniz",
			'480736' => "Ondarroa",
			'480754' => "Orozko",
			'480834' => "Ortuella",
			'480720' => "Otxandio",
			'480773' => "Plentzia",
			'480789' => "Portugalete",
			'480828' => "Santurtzi",
			'480849' => "Sestao",
			'489047' => "Sondika",
			'480852' => "Sopelana",
			'480865' => "Sopuerta",
			'480767' => "Sukarrieta",
			'480871' => "Trucios-Turtzioz",
			'480887' => "Ubide",
			'480656' => "Ugao-Miraballes",
			'480890' => "Urduliz",
			'480741' => "Urduña/Orduña",
			'480806' => "Valle de Trápaga-Trapagaran",
			'480950' => "Zaldibar",
			'480963' => "Zalla",
			'489050' => "Zamudio",
			'480979' => "Zaratamo",
			'480242' => "Zeanuri",
			'480255' => "Zeberio",
			'489130' => "Zierbena",
			'489158' => "Ziortza-Bolibar",
			'490028' => "Abezames",
			'490034' => "Alcañices",
			'490049' => "Alcubilla de Nogales",
			'490052' => "Alfaraz de Sayago",
			'490065' => "Algodre",
			'490071' => "Almaraz de Duero",
			'490087' => "Almeida de Sayago",
			'490090' => "Andavías",
			'490104' => "Arcenillas",
			'490111' => "Arcos de la Polvorosa",
			'490126' => "Argañín",
			'490132' => "Argujillo",
			'490147' => "Arquillinos",
			'490150' => "Arrabalde",
			'490163' => "Aspariegos",
			'490179' => "Asturianos",
			'490185' => "Ayoó de Vidriales",
			'490198' => "Barcial del Barco",
			'490202' => "Belver de los Montes",
			'490219' => "Benavente",
			'490224' => "Benegiles",
			'490230' => "Bermillo de Sayago",
			'490245' => "Bóveda de Toro (La)",
			'490258' => "Bretó",
			'490261' => "Bretocino",
			'490277' => "Brime de Sog",
			'490283' => "Brime de Urz",
			'490296' => "Burganes de Valverde",
			'490300' => "Bustillo del Oro",
			'490317' => "Cabañas de Sayago",
			'490322' => "Calzadilla de Tera",
			'490338' => "Camarzana de Tera",
			'490343' => "Cañizal",
			'490356' => "Cañizo",
			'490369' => "Carbajales de Alba",
			'490375' => "Carbellino",
			'490381' => "Casaseca de Campeán",
			'490394' => "Casaseca de las Chanas",
			'490408' => "Castrillo de la Guareña",
			'490415' => "Castrogonzalo",
			'490420' => "Castronuevo",
			'490436' => "Castroverde de Campos",
			'490441' => "Cazurra",
			'490467' => "Cerecinos de Campos",
			'490473' => "Cerecinos del Carrizal",
			'490489' => "Cernadilla",
			'490505' => "Cobreros",
			'490527' => "Coomonte",
			'490533' => "Coreses",
			'490548' => "Corrales",
			'490551' => "Cotanes del Monte",
			'490564' => "Cubillos",
			'490570' => "Cubo de Benavente",
			'490586' => "Cubo de Tierra del Vino (El)",
			'490599' => "Cuelgamures",
			'490610' => "Entrala",
			'490625' => "Espadañedo",
			'490631' => "Faramontanos de Tábara",
			'490646' => "Fariza",
			'490659' => "Fermoselle",
			'490662' => "Ferreras de Abajo",
			'490678' => "Ferreras de Arriba",
			'490684' => "Ferreruela",
			'490697' => "Figueruela de Arriba",
			'490718' => "Fonfría",
			'490757' => "Fresno de la Polvorosa",
			'490760' => "Fresno de la Ribera",
			'490776' => "Fresno de Sayago",
			'490782' => "Friera de Valverde",
			'490795' => "Fuente Encalada",
			'490809' => "Fuentelapeña",
			'490821' => "Fuentes de Ropel",
			'490816' => "Fuentesaúco",
			'490837' => "Fuentesecas",
			'490842' => "Fuentespreadas",
			'490855' => "Galende",
			'490868' => "Gallegos del Pan",
			'490874' => "Gallegos del Río",
			'490880' => "Gamones",
			'490907' => "Gema",
			'490914' => "Granja de Moreruela",
			'490929' => "Granucillo",
			'490935' => "Guarrate",
			'490940' => "Hermisende",
			'490953' => "Hiniesta (La)",
			'490966' => "Jambrina",
			'490972' => "Justel",
			'490988' => "Losacino",
			'490991' => "Losacio",
			'491005' => "Lubián",
			'491012' => "Luelmo",
			'491027' => "Maderal (El)",
			'491033' => "Madridanos",
			'491048' => "Mahide",
			'491051' => "Maire de Castroponce",
			'491070' => "Malva",
			'491086' => "Manganeses de la Lampreana",
			'491099' => "Manganeses de la Polvorosa",
			'491103' => "Manzanal de Arriba",
			'491125' => "Manzanal de los Infantes",
			'491110' => "Manzanal del Barco",
			'491131' => "Matilla de Arzón",
			'491146' => "Matilla la Seca",
			'491159' => "Mayalde",
			'491162' => "Melgar de Tera",
			'491178' => "Micereces de Tera",
			'491184' => "Milles de la Polvorosa",
			'491197' => "Molacillos",
			'491201' => "Molezuelas de la Carballeda",
			'491218' => "Mombuey",
			'491223' => "Monfarracinos",
			'491239' => "Montamarta",
			'491244' => "Moral de Sayago",
			'491260' => "Moraleja de Sayago",
			'491257' => "Moraleja del Vino",
			'491282' => "Morales de Rey",
			'491295' => "Morales de Toro",
			'491309' => "Morales de Valverde",
			'491276' => "Morales del Vino",
			'491316' => "Moralina",
			'491321' => "Moreruela de los Infanzones",
			'491337' => "Moreruela de Tábara",
			'491342' => "Muelas de los Caballeros",
			'491355' => "Muelas del Pan",
			'491368' => "Muga de Sayago",
			'491374' => "Navianos de Valverde",
			'491380' => "Olmillos de Castro",
			'491393' => "Otero de Bodas",
			'491414' => "Pajares de la Lampreana",
			'491435' => "Palacios de Sanabria",
			'491429' => "Palacios del Pan",
			'491453' => "Pedralba de la Pradería",
			'491466' => "Pego (El)",
			'491472' => "Peleagonzalo",
			'491488' => "Peleas de Abajo",
			'491491' => "Peñausende",
			'491504' => "Peque",
			'491511' => "Perdigón (El)",
			'491526' => "Pereruela",
			'491532' => "Perilla de Castro",
			'491547' => "Pías",
			'491550' => "Piedrahita de Castro",
			'491563' => "Pinilla de Toro",
			'491579' => "Pino del Oro",
			'491585' => "Piñero (El)",
			'491602' => "Pobladura de Valderaduey",
			'491598' => "Pobladura del Valle",
			'491624' => "Porto",
			'491630' => "Pozoantiguo",
			'491645' => "Pozuelo de Tábara",
			'491658' => "Prado",
			'491661' => "Puebla de Sanabria",
			'491677' => "Pueblica de Valverde",
			'491700' => "Quintanilla de Urz",
			'491683' => "Quintanilla del Monte",
			'491696' => "Quintanilla del Olmo",
			'491717' => "Quiruelas de Vidriales",
			'491722' => "Rabanales",
			'491738' => "Rábano de Aliste",
			'491743' => "Requejo",
			'491756' => "Revellinos",
			'491769' => "Riofrío de Aliste",
			'491775' => "Rionegro del Puente",
			'491781' => "Roales",
			'491794' => "Robleda-Cervantes",
			'491808' => "Roelos de Sayago",
			'491815' => "Rosinos de la Requejada",
			'491836' => "Salce",
			'491841' => "Samir de los Caños",
			'491854' => "San Agustín del Pozo",
			'491867' => "San Cebrián de Castro",
			'491873' => "San Cristóbal de Entreviñas",
			'491889' => "San Esteban del Molar",
			'491892' => "San Justo",
			'491906' => "San Martín de Valderaduey",
			'491913' => "San Miguel de la Ribera",
			'491928' => "San Miguel del Valle",
			'491934' => "San Pedro de Ceque",
			'491949' => "San Pedro de la Nave-Almendra",
			'492085' => "San Vicente de la Cabeza",
			'492098' => "San Vitero",
			'491971' => "Santa Clara de Avedillo",
			'491990' => "Santa Colomba de las Monjas",
			'492004' => "Santa Cristina de la Polvorosa",
			'492011' => "Santa Croya de Tera",
			'492026' => "Santa Eufemia del Barco",
			'492032' => "Santa María de la Vega",
			'492047' => "Santa María de Valverde",
			'492050' => "Santibáñez de Tera",
			'492063' => "Santibáñez de Vidriales",
			'492079' => "Santovenia",
			'492102' => "Sanzoles",
			'492145' => "Tábara",
			'492161' => "Tapioles",
			'492196' => "Toro",
			'492200' => "Torre del Valle (La)",
			'492217' => "Torregamones",
			'492222' => "Torres del Carrizal",
			'492238' => "Trabazos",
			'492243' => "Trefacio",
			'492256' => "Uña de Quintana",
			'492269' => "Vadillo de la Guareña",
			'492275' => "Valcabado",
			'492281' => "Valdefinjas",
			'492294' => "Valdescorriel",
			'492308' => "Vallesa de la Guareña",
			'492315' => "Vega de Tera",
			'492320' => "Vega de Villalobos",
			'492336' => "Vegalatrave",
			'492341' => "Venialbo",
			'492354' => "Vezdemarbán",
			'492367' => "Vidayanes",
			'492373' => "Videmala",
			'492389' => "Villabrázaro",
			'492392' => "Villabuena del Puente",
			'492406' => "Villadepera",
			'492413' => "Villaescusa",
			'492428' => "Villafáfila",
			'492434' => "Villaferrueña",
			'492449' => "Villageriz",
			'492452' => "Villalazán",
			'492465' => "Villalba de la Lampreana",
			'492471' => "Villalcampo",
			'492487' => "Villalobos",
			'492490' => "Villalonso",
			'492503' => "Villalpando",
			'492510' => "Villalube",
			'492525' => "Villamayor de Campos",
			'492559' => "Villamor de los Escuderos",
			'492562' => "Villanázar",
			'492578' => "Villanueva de Azoague",
			'492584' => "Villanueva de Campeán",
			'492597' => "Villanueva de las Peras",
			'492601' => "Villanueva del Campo",
			'492639' => "Villar de Fallaves",
			'492644' => "Villar del Buey",
			'492618' => "Villaralbo",
			'492623' => "Villardeciervos",
			'492657' => "Villardiegua de la Ribera",
			'492660' => "Villárdiga",
			'492676' => "Villardondiego",
			'492682' => "Villarrín de Campos",
			'492695' => "Villaseco del Pan",
			'492709' => "Villavendimio",
			'492721' => "Villaveza de Valverde",
			'492716' => "Villaveza del Agua",
			'492737' => "Viñas",
			'492755' => "Zamora",
			'500016' => "Abanto",
			'500021' => "Acered",
			'500037' => "Agón",
			'500042' => "Aguarón",
			'500055' => "Aguilón",
			'500068' => "Ainzón",
			'500074' => "Aladrén",
			'500080' => "Alagón",
			'500093' => "Alarba",
			'500107' => "Alberite de San Juan",
			'500114' => "Albeta",
			'500129' => "Alborge",
			'500135' => "Alcalá de Ebro",
			'500140' => "Alcalá de Moncayo",
			'500153' => "Alconchel de Ariza",
			'500166' => "Aldehuela de Liestos",
			'500172' => "Alfajarín",
			'500188' => "Alfamén",
			'500191' => "Alforque",
			'500205' => "Alhama de Aragón",
			'500212' => "Almochuel",
			'500227' => "Almolda (La)",
			'500233' => "Almonacid de la Cuba",
			'500248' => "Almonacid de la Sierra",
			'500251' => "Almunia de Doña Godina (La)",
			'500264' => "Alpartir",
			'500270' => "Ambel",
			'500286' => "Anento",
			'500299' => "Aniñón",
			'500303' => "Añón de Moncayo",
			'500310' => "Aranda de Moncayo",
			'500325' => "Arándiga",
			'500331' => "Ardisa",
			'500346' => "Ariza",
			'500359' => "Artieda",
			'500362' => "Asín",
			'500378' => "Atea",
			'500384' => "Ateca",
			'500397' => "Azuara",
			'500401' => "Badules",
			'500418' => "Bagüés",
			'500423' => "Balconchán",
			'500439' => "Bárboles",
			'500444' => "Bardallur",
			'500457' => "Belchite",
			'500460' => "Belmonte de Gracián",
			'500476' => "Berdejo",
			'500482' => "Berrueco",
			'509017' => "Biel",
			'500508' => "Bijuesca",
			'500515' => "Biota",
			'500520' => "Bisimbre",
			'500536' => "Boquiñeni",
			'500541' => "Bordalba",
			'500554' => "Borja",
			'500567' => "Botorrita",
			'500573' => "Brea de Aragón",
			'500589' => "Bubierca",
			'500592' => "Bujaraloz",
			'500606' => "Bulbuente",
			'500613' => "Bureta",
			'500628' => "Burgo de Ebro (El)",
			'500634' => "Buste (El)",
			'500649' => "Cabañas de Ebro",
			'500652' => "Cabolafuente",
			'500665' => "Cadrete",
			'500671' => "Calatayud",
			'500687' => "Calatorao",
			'500690' => "Calcena",
			'500704' => "Calmarza",
			'500711' => "Campillo de Aragón",
			'500726' => "Carenas",
			'500732' => "Cariñena",
			'500747' => "Caspe",
			'500750' => "Castejón de Alarba",
			'500763' => "Castejón de las Armas",
			'500779' => "Castejón de Valdejasa",
			'500785' => "Castiliscar",
			'500798' => "Cervera de la Cañada",
			'500802' => "Cerveruela",
			'500819' => "Cetina",
			'500922' => "Chiprana",
			'500938' => "Chodes",
			'500824' => "Cimballa",
			'500830' => "Cinco Olivas",
			'500845' => "Clarés de Ribota",
			'500858' => "Codo",
			'500861' => "Codos",
			'500877' => "Contamina",
			'500883' => "Cosuenda",
			'500896' => "Cuarte de Huerva",
			'500900' => "Cubel",
			'500917' => "Cuerlas (Las)",
			'500943' => "Daroca",
			'500956' => "Ejea de los Caballeros",
			'500969' => "Embid de Ariza",
			'500981' => "Encinacorba",
			'500994' => "Épila",
			'501008' => "Erla",
			'501015' => "Escatrón",
			'501020' => "Fabara",
			'501041' => "Farlete",
			'501054' => "Fayón",
			'501067' => "Fayos (Los)",
			'501073' => "Figueruelas",
			'501089' => "Fombuena",
			'501092' => "Frago (El)",
			'501106' => "Frasno (El)",
			'501113' => "Fréscano",
			'501134' => "Fuendejalón",
			'501149' => "Fuendetodos",
			'501152' => "Fuentes de Ebro",
			'501165' => "Fuentes de Jiloca",
			'501171' => "Gallocanta",
			'501187' => "Gallur",
			'501190' => "Gelsa",
			'501204' => "Godojos",
			'501211' => "Gotor",
			'501226' => "Grisel",
			'501232' => "Grisén",
			'501247' => "Herrera de los Navarros",
			'501250' => "Ibdes",
			'501263' => "Illueca",
			'501285' => "Isuerre",
			'501298' => "Jaraba",
			'501302' => "Jarque",
			'501319' => "Jaulín",
			'501324' => "Joyosa (La)",
			'501330' => "Lagata",
			'501345' => "Langa del Castillo",
			'501358' => "Layana",
			'501361' => "Lécera",
			'501383' => "Lechón",
			'501377' => "Leciñena",
			'501396' => "Letux",
			'501400' => "Litago",
			'501417' => "Lituénigo",
			'501422' => "Lobera de Onsella",
			'501438' => "Longares",
			'501443' => "Longás",
			'501469' => "Lucena de Jalón",
			'501475' => "Luceni",
			'501481' => "Luesia",
			'501494' => "Luesma",
			'501507' => "Lumpiaque",
			'501514' => "Luna",
			'501529' => "Maella",
			'501535' => "Magallón",
			'501540' => "Mainar",
			'501553' => "Malanquilla",
			'501566' => "Maleján",
			'501605' => "Mallén",
			'501572' => "Malón",
			'501591' => "Maluenda",
			'501612' => "Manchones",
			'501627' => "Mara",
			'501633' => "María de Huerva",
			'509022' => "Marracos",
			'501648' => "Mediana de Aragón",
			'501651' => "Mequinenza",
			'501664' => "Mesones de Isuela",
			'501670' => "Mezalocha",
			'501686' => "Mianos",
			'501699' => "Miedes de Aragón",
			'501703' => "Monegrillo",
			'501710' => "Moneva",
			'501725' => "Monreal de Ariza",
			'501731' => "Monterde",
			'501746' => "Montón",
			'501759' => "Morata de Jalón",
			'501762' => "Morata de Jiloca",
			'501778' => "Morés",
			'501784' => "Moros",
			'501797' => "Moyuela",
			'501801' => "Mozota",
			'501818' => "Muel",
			'501823' => "Muela (La)",
			'501839' => "Munébrega",
			'501844' => "Murero",
			'501857' => "Murillo de Gállego",
			'501860' => "Navardún",
			'501876' => "Nigüella",
			'501882' => "Nombrevilla",
			'501895' => "Nonaspe",
			'501909' => "Novallas",
			'501916' => "Novillas",
			'501921' => "Nuévalos",
			'501937' => "Nuez de Ebro",
			'501942' => "Olvés",
			'501955' => "Orcajo",
			'501968' => "Orera",
			'501974' => "Orés",
			'501980' => "Oseja",
			'501993' => "Osera de Ebro",
			'502007' => "Paniza",
			'502014' => "Paracuellos de Jiloca",
			'502029' => "Paracuellos de la Ribera",
			'502035' => "Pastriz",
			'502040' => "Pedrola",
			'502053' => "Pedrosas (Las)",
			'502066' => "Perdiguera",
			'502072' => "Piedratajada",
			'502088' => "Pina de Ebro",
			'502091' => "Pinseque",
			'502105' => "Pintanos (Los)",
			'502112' => "Plasencia de Jalón",
			'502127' => "Pleitas",
			'502133' => "Plenas",
			'502148' => "Pomer",
			'502151' => "Pozuel de Ariza",
			'502164' => "Pozuelo de Aragón",
			'502170' => "Pradilla de Ebro",
			'502186' => "Puebla de Albortón",
			'502199' => "Puebla de Alfindén (La)",
			'502203' => "Puendeluna",
			'502210' => "Purujosa",
			'502225' => "Quinto",
			'502231' => "Remolinos",
			'502246' => "Retascón",
			'502259' => "Ricla",
			'502278' => "Romanos",
			'502284' => "Rueda de Jalón",
			'502297' => "Ruesca",
			'502416' => "Sabiñán",
			'502301' => "Sádaba",
			'502318' => "Salillas de Jalón",
			'502323' => "Salvatierra de Esca",
			'502339' => "Samper del Salz",
			'502344' => "San Martín de la Virgen de Moncayo",
			'502357' => "San Mateo de Gállego",
			'502360' => "Santa Cruz de Grío",
			'502376' => "Santa Cruz de Moncayo",
			'502382' => "Santa Eulalia de Gállego",
			'502395' => "Santed",
			'502409' => "Sástago",
			'502421' => "Sediles",
			'502437' => "Sestrica",
			'502442' => "Sierra de Luna",
			'502455' => "Sigüés",
			'502468' => "Sisamón",
			'502474' => "Sobradiel",
			'502480' => "Sos del Rey Católico",
			'502493' => "Tabuenca",
			'502506' => "Talamantes",
			'502513' => "Tarazona",
			'502528' => "Tauste",
			'502534' => "Terrer",
			'502549' => "Tierga",
			'502552' => "Tobed",
			'502565' => "Torralba de los Frailes",
			'502571' => "Torralba de Ribota",
			'502587' => "Torralbilla",
			'502590' => "Torrehermosa",
			'502604' => "Torrelapaja",
			'502611' => "Torrellas",
			'502626' => "Torres de Berrellén",
			'502632' => "Torrijo de la Cañada",
			'502647' => "Tosos",
			'502650' => "Trasmoz",
			'502663' => "Trasobares",
			'502679' => "Uncastillo",
			'502685' => "Undués de Lerda",
			'502698' => "Urrea de Jalón",
			'502702' => "Urriés",
			'502719' => "Used",
			'502724' => "Utebo",
			'502745' => "Val de San Martín",
			'502730' => "Valdehorna",
			'502758' => "Valmadrid",
			'502761' => "Valpalmas",
			'502777' => "Valtorres",
			'502783' => "Velilla de Ebro",
			'502796' => "Velilla de Jiloca",
			'502800' => "Vera de Moncayo",
			'502817' => "Vierlas",
			'502838' => "Villadoz",
			'502843' => "Villafeliche",
			'502856' => "Villafranca de Ebro",
			'502869' => "Villalba de Perejil",
			'502875' => "Villalengua",
			'509038' => "Villamayor de Gállego",
			'502881' => "Villanueva de Gállego",
			'502908' => "Villanueva de Huerva",
			'502894' => "Villanueva de Jiloca",
			'502915' => "Villar de los Navarros",
			'502920' => "Villarreal de Huerva",
			'502936' => "Villarroya de la Sierra",
			'502941' => "Villarroya del Campo",
			'502822' => "Vilueña (La)",
			'502954' => "Vistabella",
			'502967' => "Zaida (La)",
			'502973' => "Zaragoza",
			'502989' => "Zuera",
			'510013' => "Ceuta",
			'520018' => "Melilla",
		];
	}
}
