<?php
/**
 * Front Style.
 *
 * @package twentig
 */

/**
 * Enqueue scripts and styles.
 */
function twentig_twentyone_theme_scripts() {

	wp_enqueue_style(
		'twentig-twentyone',
		TWENTIG_ASSETS_URI . '/css/twentytwentyone/style.css',
		array(),
		TWENTIG_VERSION
	);

	twentig_twentyone_print_customizer_css();

	$fonts_url = twentig_twentyone_fonts_url();

	if ( $fonts_url ) {
		if ( get_theme_mod( 'twentig_local_fonts', false ) ) {
			require_once TWENTIG_PATH . 'inc/classic/theme-tools/wptt-webfont-loader.php';
			wp_register_style( 'twentig-webfonts', '' );
			wp_enqueue_style( 'twentig-webfonts' );
			wp_add_inline_style( 'twentig-webfonts', twentig_minify_css( wptt_get_webfont_styles( $fonts_url ) ) );
		} else {
			wp_enqueue_style( // phpcs:ignore WordPress.WP.EnqueuedResourceParameters.MissingVersion
				'twentig-twentyone-fonts',
				$fonts_url,
				array(),
				null
			);
		}
	}

	// Enqueue JavaScript only if header is sticky and this is not an AMP response.
	$header_sticky = get_theme_mod( 'twentig_header_sticky', false );
	if ( $header_sticky && ! twentig_is_amp_endpoint() ) {
		wp_enqueue_script( 'twentig-twentyone', TWENTIG_ASSETS_URI . '/js/classic/twentig-twentytwentyone.js', array(), '1.0', true );
	}

	if ( is_page_template( 'tw-no-header-footer.php' ) ) {
		wp_dequeue_script( 'twenty-twenty-one-primary-navigation-script' );
		wp_dequeue_script( 'twenty-twenty-one-ie11-polyfills-js' );
		wp_dequeue_script( 'twenty-twenty-one-responsive-embeds-script' );
		wp_dequeue_script( 'twentig-twentyone' );
	}
}
add_action( 'wp_enqueue_scripts', 'twentig_twentyone_theme_scripts', 12 );

/**
 * Add preconnect for Google Fonts.
 *
 * @param array  $urls          URLs to print for resource hints.
 * @param string $relation_type The relation type the URLs are printed.
 */
function twentig_twentyone_resource_hints( $urls, $relation_type ) {
	if ( wp_style_is( 'twentig-twentyone-fonts', 'queue' ) && 'preconnect' === $relation_type ) {
		$urls[] = array(
			'href' => 'https://fonts.gstatic.com',
			'crossorigin',
		);
	}
	return $urls;
}
add_filter( 'wp_resource_hints', 'twentig_twentyone_resource_hints', 10, 2 );

/**
 * Add custom classes generated by Customizer settings to the array of body classes.
 *
 * @param array $classes Classes added to the body tag.
 */
function twentig_twentyone_body_class( $classes ) {

	if ( is_page_template( 'tw-no-header-footer.php' ) ) {
		return twentig_twentyone_canvas_body_class( $classes );
	}

	$site_width         = get_theme_mod( 'twentig_site_width' );
	$page_background    = get_theme_mod( 'twentig_inner_background_color' );
	$header_layout      = get_theme_mod( 'twentig_header_layout' );
	$header_background  = get_theme_mod( 'twentig_header_background_color' );
	$header_decoration  = get_theme_mod( 'twentig_header_decoration' );
	$header_sticky      = get_theme_mod( 'twentig_header_sticky', false );
	$header_breakpoint  = get_theme_mod( 'twentig_header_breakpoint', 'mobile' );
	$header_width       = get_theme_mod( 'twentig_header_width' );
	$header_padding     = get_theme_mod( 'twentig_header_padding', 'large' );
	$menu_spacing       = get_theme_mod( 'twentig_menu_spacing' );
	$menu_hover         = get_theme_mod( 'twentig_menu_hover' );
	$footer_layout      = get_theme_mod( 'twentig_footer_layout' );
	$footer_background  = get_theme_mod( 'twentig_footer_background_color' );
	$footer_width       = get_theme_mod( 'twentig_footer_width' );
	$widget_background  = get_theme_mod( 'twentig_widgets_background_color' );
	$widgets_width      = get_theme_mod( 'twentig_footer_widgets_width' );
	$widgets_columns    = get_theme_mod( 'twentig_footer_widgets_columns' );
	$search_layout      = get_theme_mod( 'twentig_page_search_layout' );
	$link_style         = get_theme_mod( 'twentig_links_style' );
	$cpt_archive_layout = get_theme_mod( 'twentig_cpt_archive_layout' );
	$cpt_single_layout  = get_theme_mod( 'twentig_cpt_single_layout' );

	if ( $site_width && $page_background ) {
		$luminance = Twenty_Twenty_One_Custom_Colors::get_relative_luminance_from_hex( $page_background );
		if ( 127 > $luminance && in_array( 'is-light-theme', $classes, true ) ) {
			$classes[] = 'is-dark-theme';
			$classes   = array_diff( $classes, array( 'is-light-theme' ) );
		} elseif ( 127 <= $luminance && in_array( 'is-dark-theme', $classes, true ) ) {
			$classes[] = 'is-light-theme';
			$classes   = array_diff( $classes, array( 'is-dark-theme' ) );
		}
	}

	if ( is_page_template() ) {
		$classes[] = basename( get_page_template_slug(), '.php' );
	}

	if ( $header_layout ) {
		$classes[] = 'tw-header-layout-' . $header_layout;
	} else {
		$classes[] = 'tw-header-layout-menu-right';
	}

	if ( 'full' === $header_width ) {
		$classes[] = 'tw-header-full';
	}

	if ( $header_background || $header_decoration ) {
		$classes[] = 'tw-header-bg';
	}

	if ( $header_sticky ) {
		$classes[] = 'tw-header-sticky';
		$classes[] = 'tw-header-bg';
	}

	if ( $header_breakpoint ) {
		$classes[] = 'tw-header-break-' . $header_breakpoint;
	}

	$classes[] = 'tw-header-padding-' . $header_padding;

	if ( $menu_spacing ) {
		$classes[] = 'tw-nav-spacing-' . $menu_spacing;
	}

	if ( $menu_hover ) {
		$classes[] = 'tw-nav-hover-' . $menu_hover;
	}

	if ( twentig_twentyone_has_sidebar() ) {
		$classes[] = 'tw-has-sidebar';
	}

	if ( $footer_background || twentig_twentyone_has_footer_block_background() ) {
		$classes[] = 'tw-footer-bg';
	}

	if ( 'full' === $footer_width ) {
		$classes[] = 'tw-footer-full';
	}

	if ( is_active_sidebar( 'sidebar-1' ) ) {
		if ( 'full' === $widgets_width ) {
			$classes[] = 'tw-footer-widgets-full';
		}
		if ( $widget_background || in_array( 'tw-footer-bg', $classes, true ) ) {
			$classes[] = 'tw-footer-widgets-bg';
			if ( $footer_background === $widget_background || ( $footer_background && empty( $widget_background ) ) && 'hidden' !== $footer_layout ) {
				$classes[] = 'tw-footer-monocolor';
			}
		}
	} elseif ( 'hidden' === $footer_layout ) {
		$classes[] = 'tw-no-footer';
	}

	if ( $link_style ) {
		$classes[] = 'tw-link-' . $link_style;
		$classes   = array_diff( $classes, array( 'has-background-white' ) );

		if ( in_array( 'is-dark-theme', $classes, true ) ) {
			$classes   = array_diff( $classes, array( 'is-dark-theme' ) );
			$classes[] = 'dark-theme';
		}
	}

	if ( ! get_theme_mod( 'twentig_blog_meta_label', true ) ) {
		$classes[] = 'tw-no-meta-label';
	}

	if ( in_array( 'blog', $classes, true ) && ! get_theme_mod( 'twentig_blog_show_title', false ) ) {
		$classes[] = 'tw-hide-blog-title';
	}

	if ( twentig_twentyone_is_blog_page()
		|| ( is_search() && 'blog-layout' === $search_layout )
		|| ( 'blog' === $cpt_archive_layout && twentig_twentyone_is_cpt_archive() )
	) {

		$blog_layout     = get_theme_mod( 'twentig_blog_layout', 'stack' );
		$blog_style      = get_theme_mod( 'twentig_blog_style', 'separator' );
		$blog_image      = get_theme_mod( 'twentig_blog_image', true );
		$image_width     = get_theme_mod( 'twentig_blog_image_width', 'wide' );
		$image_placement = get_theme_mod( 'twentig_blog_image_placement', 'below' );
		$image_ratio     = get_theme_mod( 'twentig_blog_image_ratio' );
		$blog_pagination = get_theme_mod( 'twentig_blog_pagination' );

		$classes[] = 'tw-blog-' . $blog_layout;

		if ( 'grid' === $blog_layout ) {
			$classes[] = 'tw-blog-columns-' . get_theme_mod( 'twentig_blog_columns', '3' );
			add_filter(
				'post_thumbnail_size',
				function() {
					return 'medium_large';
				}
			);
			twentig_twentyone_remove_image_inline_size_style();
		}

		if ( $blog_style ) {
			if ( in_array( $blog_style, array( 'card-shadow', 'card-border' ), true ) ) {
				$classes[] = 'tw-blog-card';
			}
			$classes[] = 'tw-blog-' . $blog_style;
		}

		if ( $blog_image ) {
			$classes[] = 'tw-blog-image-' . $image_width;
			$classes[] = 'tw-blog-image-' . $image_placement;
			if ( $image_ratio ) {
				$classes[] = 'tw-blog-img-ratio';
				twentig_twentyone_remove_image_inline_size_style();
			}
		}

		if ( 'center' === get_theme_mod( 'twentig_blog_text_align' ) ) {
			$classes[] = 'tw-blog-text-center';
		}

		if ( ! get_theme_mod( 'twentig_blog_content', true ) ) {
			$classes[] = 'tw-blog-no-content';
		}

		if ( empty( get_theme_mod( 'twentig_blog_meta', array( 'author', 'categories', 'tags' ) ) ) ) {
			$classes[] = 'tw-no-meta';
		}

		if ( $blog_pagination ) {
			$classes[] = 'tw-pagination-' . $blog_pagination;
		}
	} elseif ( is_search() ) {
		$blog_pagination = get_theme_mod( 'twentig_blog_pagination' );
		if ( $blog_pagination ) {
			$classes[] = 'tw-pagination-' . $blog_pagination;
		}
	} elseif ( is_page() || ( 'page' === $cpt_single_layout && twentig_twentyone_is_cpt_single() ) ) {

		$hero_layout     = get_theme_mod( 'twentig_page_hero_layout', '' );
		$title_center    = get_theme_mod( 'twentig_page_title_text_align', false );
		$title_width     = get_theme_mod( 'twentig_page_title_width', 'wide' );
		$title_border    = get_theme_mod( 'twentig_page_title_border', true );
		$image_placement = get_theme_mod( 'twentig_page_image_placement', 'below' );

		if ( has_post_thumbnail() && ! is_page_template() ) {

			if ( $hero_layout ) {
				$classes[] = 'tw-hero-' . $hero_layout;
			}

			if ( in_array( $hero_layout, array( '', 'narrow-image', 'full-image' ), true ) ) {
				$classes[] = 'tw-hero-' . $image_placement;
			}

			if ( 'cover-full' === $hero_layout ) {
				$classes[] = 'tw-hero-cover';
			}

			if ( in_array( $hero_layout, array( 'cover', 'cover-full' ), true ) ) {
				twentig_twentyone_remove_image_inline_size_style();
			}
		}

		if ( 'text-width' === $title_width ) {
			$classes[] = 'tw-title-text-width';
		}

		if ( $title_center ) {
			$classes[] = 'tw-title-center';
		}

		if ( ! $title_border ) {
			$classes[] = 'tw-title-no-border';
		}

		if ( comments_open() || get_comments_number() && ! post_password_required() ) {
			$classes[] = 'tw-showing-comments';
		}
	} elseif ( is_singular( 'post' ) || ( 'post' === $cpt_single_layout && twentig_twentyone_is_cpt_single() ) ) {
		$hero_layout     = get_theme_mod( 'twentig_post_hero_layout', '' );
		$title_center    = get_theme_mod( 'twentig_post_title_text_align', false );
		$title_width     = get_theme_mod( 'twentig_post_title_width', 'wide' );
		$title_border    = get_theme_mod( 'twentig_post_title_border', true );
		$image_placement = get_theme_mod( 'twentig_post_image_placement', 'below' );

		if ( has_post_thumbnail() ) {

			if ( $hero_layout ) {
				$classes[] = 'tw-hero-' . $hero_layout;
			}

			if ( in_array( $hero_layout, array( '', 'narrow-image', 'full-image' ), true ) ) {
				$classes[] = 'tw-hero-' . $image_placement;
			}

			if ( 'cover-full' === $hero_layout ) {
				$classes[] = 'tw-hero-cover';
			}

			if ( in_array( $hero_layout, array( 'cover', 'cover-full' ), true ) ) {
				twentig_twentyone_remove_image_inline_size_style();
			}
		}

		if ( 'text-width' === $title_width ) {
			$classes[] = 'tw-title-text-width';
		}

		if ( $title_center ) {
			$classes[] = 'tw-title-center';
		}

		if ( ! $title_border ) {
			$classes[] = 'tw-title-no-border';
		}

		if ( empty( get_theme_mod( 'twentig_post_bottom_meta', array( 'date', 'author', 'categories', 'tags' ) ) ) ) {
			$classes[] = 'has-no-footer-meta';
		}
	}

	if ( ! in_array( 'tw-header-bg', $classes, true ) && ( in_array( 'tw-hero-cover', $classes, true ) || in_array( 'tw-header-transparent', $classes, true ) || in_array( 'tw-header-transparent-light', $classes, true ) ) ) {
		$classes[] = 'tw-header-bg';
	}

	if ( in_array( 'tw-hero-cover', $classes, true ) || is_page_template( 'tw-header-transparent-light.php', $classes, true ) ) {
		$classes[] = 'tw-header-light';
	}

	if ( twentig_is_amp_endpoint() && in_array( 'tw-header-light', $classes, true ) && in_array( 'tw-header-sticky', $classes, true ) ) {
		$classes[] = 'tw-header-opaque';
	}

	return $classes;
}
add_filter( 'body_class', 'twentig_twentyone_body_class', 11 );

/**
 * Add custom classes generated by Customizer settings to the array of body classes for the No header no footer page.
 *
 * @param array $classes Classes added to the body tag.
 */
function twentig_twentyone_canvas_body_class( $classes ) {
	$site_width      = get_theme_mod( 'twentig_site_width' );
	$page_background = get_theme_mod( 'twentig_inner_background_color' );
	$link_style      = get_theme_mod( 'twentig_links_style' );
	$classes         = array_diff( $classes, array( 'wp-custom-logo', 'has-main-navigation' ) );

	if ( $site_width && $page_background ) {
		$luminance = Twenty_Twenty_One_Custom_Colors::get_relative_luminance_from_hex( $page_background );
		if ( 127 > $luminance && in_array( 'is-light-theme', $classes, true ) ) {
			$classes[] = 'is-dark-theme';
			$classes   = array_diff( $classes, array( 'is-light-theme' ) );
		} elseif ( 127 <= $luminance && in_array( 'is-dark-theme', $classes, true ) ) {
			$classes[] = 'is-light-theme';
			$classes   = array_diff( $classes, array( 'is-dark-theme' ) );
		}
	}

	if ( $link_style ) {
		$classes[] = 'tw-link-' . $link_style;
		$classes   = array_diff( $classes, array( 'has-background-white' ) );

		if ( in_array( 'is-dark-theme', $classes, true ) ) {
			$classes   = array_diff( $classes, array( 'is-dark-theme' ) );
			$classes[] = 'dark-theme';
		}
	}

	if ( is_page_template() ) {
		$classes[] = basename( get_page_template_slug(), '.php' );
	}

	if ( comments_open() || get_comments_number() && ! post_password_required() ) {
		$classes[] = 'tw-showing-comments';
	}

	return $classes;
}

/**
 * Display custom CSS generated by the Customizer settings.
 */
function twentig_twentyone_print_customizer_css() {
	$css  = twentig_twentyone_generate_dynamic_css();
	$css .= twentig_twentyone_generate_customizer_css();
	$css  = apply_filters( 'twentig_twentyone_custom_css', $css );
	if ( $css ) :
		wp_add_inline_style( 'twentig-twentyone', twentig_minify_css( $css ) );
	endif;
}

/**
 * Get custom frontend CSS based on the Customizer settings.
 */
function twentig_twentyone_generate_customizer_css() {

	$site_width              = get_theme_mod( 'twentig_site_width' );
	$wide_width              = get_theme_mod( 'twentig_wide_width' );
	$default_width           = get_theme_mod( 'twentig_default_width' );
	$body_font               = get_theme_mod( 'twentig_body_font' );
	$body_font_size          = get_theme_mod( 'twentig_body_font_size', 20 );
	$body_line_height        = get_theme_mod( 'twentig_body_line_height', 1.7 );
	$heading_font            = get_theme_mod( 'twentig_heading_font' );
	$heading_font_weight     = get_theme_mod( 'twentig_heading_font_weight', '400' );
	$heading_letter_spacing  = get_theme_mod( 'twentig_heading_letter_spacing', 0 );
	$tertiary_font           = get_theme_mod( 'twentig_secondary_elements_font', 'body' );
	$menu_font               = get_theme_mod( 'twentig_menu_font', 'body' );
	$menu_letter_spacing     = get_theme_mod( 'twentig_menu_letter_spacing', 0 );
	$show_site_title         = get_theme_mod( 'display_title_and_tagline', true );
	$h1_font_size            = get_theme_mod( 'twentig_h1_font_size', 96 );
	$post_h1_font_size       = get_theme_mod( 'twentig_post_h1_font_size' );
	$menu_font_size          = get_theme_mod( 'twentig_menu_font_size' );
	$menu_font_weight        = get_theme_mod( 'twentig_menu_font_weight', 400 );
	$menu_transform          = get_theme_mod( 'twentig_menu_text_transform' );
	$logo_font_size          = get_theme_mod( 'twentig_logo_font_size' );
	$logo_mobile_font_size   = get_theme_mod( 'twentig_logo_font_size_mobile' );
	$header_decoration       = get_theme_mod( 'twentig_header_decoration' );
	$header_button           = get_theme_mod( 'twentig_header_button', false );
	$button_size             = get_theme_mod( 'twentig_button_size' );
	$button_shape            = get_theme_mod( 'twentig_button_shape', 'square' );
	$button_text_transform   = get_theme_mod( 'twentig_button_text_transform' );
	$button_text_color       = get_theme_mod( 'twentig_button_text_color' );
	$button_background       = get_theme_mod( 'twentig_button_background_color' );
	$button_hover_background = get_theme_mod( 'twentig_button_hover_background_color' );
	$footer_link_color       = get_theme_mod( 'twentig_footer_link_color' );
	$border_size             = get_theme_mod( 'twentig_border_thickness' );
	$subtle_background       = get_theme_mod( 'twentig_subtle_background_color' );
	$archive_image_ratio     = get_theme_mod( 'twentig_blog_image_ratio' );
	$archive_heading_size    = get_theme_mod( 'twentig_blog_title_size' );
	$css                     = '';
	$css_mobile              = '';
	$css_var                 = ':root{';

	if ( $site_width ) {
		$site_width_value = $site_width . 'px';
		$page_background  = get_theme_mod( 'twentig_inner_background_color' );

		if ( $page_background ) {
			$body_background = get_theme_mod( 'background_color', 'd1e4dd' );
			$page_luminance  = Twenty_Twenty_One_Custom_Colors::get_relative_luminance_from_hex( $page_background );
			$body_luminance  = Twenty_Twenty_One_Custom_Colors::get_relative_luminance_from_hex( $body_background );

			$css_var .= '--global--color-background: ' . $page_background . ';';

			if ( 127 < $page_luminance && 127 >= $body_luminance ) {
				$css_var .= '
					--global--color-primary: #000;
					--global--color-secondary: #000;
					--button--color-background: #000;
					--button--color-text-hover: #000;
				';
			} elseif ( 127 >= $page_luminance && 127 < $body_luminance ) {
				$css_var .= '
					--global--color-primary: #fff;
					--global--color-secondary: #fff;
					--button--color-background: #fff;
					--button--color-text-hover: #fff;
				';
			}

			if ( 'd1e4dd' === $body_background ) {
				$css .= 'body { background-color: #d1e4dd; }';
			}
		} else {
			$page_background = 'var(--global--color-background)';
		}

		$css .= '.site { background-color: ' . $page_background . '; }

		@media only screen and (min-width:' . $site_width_value . ') {
			:root {
				--responsive--alignfull-padding: var(--site--padding);
				--responsive--aligndefault-width: min(calc(' . $site_width_value . ' - 2 * var(--site--padding)), var(--max--aligndefault-width));
				--responsive--alignwide-width: min(calc(' . $site_width_value . ' - 2 * var(--site--padding)), var(--max--alignwide-width));
			}
			
			.site { 
				max-width:' . $site_width_value . ';
				margin: 0 auto;	
			}
			
			body:not(.tw-header-full) .site-header,
			body:not(.tw-footer-widgets-full) .widget-area,
			body:not(.tw-footer-full) .site-footer {
				max-width: inherit;
				padding-left: max( calc((' . $site_width_value . ' - var(--max--alignwide-width) ) /2), var(--site--padding));
				padding-right: max( calc((' . $site_width_value . ' - var(--max--alignwide-width) ) /2), var(--site--padding));
			}

			.tw-header-full .site-header {
				max-width: inherit;
			}
		}	
		';

		if ( get_background_image() ) {
			$css .= '@media only screen and (max-width:' . $site_width_value . ') { body.custom-background.custom-background{ background-image: none; } }';
		}
	}

	if ( $wide_width ) {
		$css_var .= '--max--alignwide-width:' . twentig_twentyone_generate_value( $wide_width );
	}

	if ( $default_width ) {
		$css_var .= '--max--aligndefault-width:' . twentig_twentyone_generate_value( $default_width );
	}

	if ( $body_font ) {
		$css_var .= '--font-base:' . $body_font . ';';
	}

	if ( $body_font_size ) {
		$css_var .= '--global--font-size-base:' . twentig_twentyone_generate_value( $body_font_size, 'rem' );
		if ( $body_font_size > 20 ) {
			$css_mobile .= ':root { --global--font-size-base: 1.25rem; }';
		}
	}

	if ( $body_line_height && 1.7 !== $body_line_height ) {
		$css_var .= '--global--line-height-body:' . $body_line_height . ';';
	}

	if ( $heading_font ) {
		$css_var .= '--font-headings:' . $heading_font . ';';
		if ( $heading_font_weight ) {
			$css_var .= '--heading--font-weight:' . $heading_font_weight . ';';
			$css_var .= '--heading--font-weight-page-title:' . $heading_font_weight . ';';
			$css_var .= '--heading--font-weight-strong:' . $heading_font_weight . ';';
			$css_var .= '--widget--font-weight-title:' . $heading_font_weight . ';';
		}
	} elseif ( '400' !== $heading_font_weight ) {
		$css_var .= '--heading--font-weight:' . $heading_font_weight . ';';
		$css_var .= '--heading--font-weight-page-title:' . $heading_font_weight . ';';
		$css_var .= '--heading--font-weight-strong:' . $heading_font_weight . ';';
	}

	if ( ! empty( $heading_letter_spacing ) ) {
		$css_var .= '--global--letter-spacing:' . twentig_twentyone_generate_value( $heading_letter_spacing, 'em' );
		$css_var .= '--heading--letter-spacing-h5:' . twentig_twentyone_generate_value( $heading_letter_spacing, 'em' );
		$css_var .= '--heading--letter-spacing-h6:' . twentig_twentyone_generate_value( $heading_letter_spacing, 'em' );
	}

	if ( $h1_font_size && 96 !== $h1_font_size ) {
		$css_var .= '--global--font-size-xxl:' . twentig_twentyone_generate_value( $h1_font_size, 'rem' );
	}

	if ( 'heading' === $tertiary_font ) {
		$css_var .= '--global--font-tertiary: var(--font-headings);';
	}

	if ( 'heading' === $menu_font ) {
		$css_var .= '--primary-nav--font-family: var(--font-headings);';
	}

	if ( $menu_font_size ) {
		$css_var .= '--primary-nav--font-size:' . twentig_twentyone_generate_value( $menu_font_size, 'rem' );
	}

	if ( $menu_font_weight ) {
		$css_var .= '--primary-nav--font-weight:' . $menu_font_weight . ';';
	}

	if ( ! empty( $menu_letter_spacing ) ) {
		$css_var .= '--primary-nav--letter-spacing:' . twentig_twentyone_generate_value( $menu_letter_spacing, 'em' );
	}

	if ( $show_site_title ) {
		if ( $logo_font_size ) {
			$css_var .= '--branding--title--font-size:' . twentig_twentyone_generate_value( $logo_font_size );
		}

		if ( $logo_mobile_font_size ) {
			$css_var .= '--branding--title--font-size-mobile:' . twentig_twentyone_generate_value( $logo_mobile_font_size );
		} elseif ( $logo_font_size ) {
			$css_var .= '--branding--title--font-size-mobile:' . twentig_twentyone_generate_value( $logo_font_size );
		}
	}

	if ( 'small' === $button_size ) {
		$css_var .= '
			--button--padding-vertical: 8px;
			--button--padding-horizontal: 16px;
			--button--font-size: var(--global--font-size-sm);
			';
	} elseif ( 'medium' === $button_size ) {
		$css_var .= '
			--button--padding-vertical: 12px;
			--button--padding-horizontal: 24px;
			--button--font-size: var(--global--font-size-sm);
			';
	}

	if ( 'rounded' === $button_shape ) {
		$css_var .= '--button--border-radius: 6px;';
	} elseif ( 'pill' === $button_shape ) {
		$css_var .= '--button--border-radius: 50px;';
	}

	if ( 'uppercase' === $button_text_transform ) {
		$css_var .= '--button--font-size: var(--global--font-size-xs);';
	}

	if ( 'thin' === $border_size ) {
		$css_var .= '
			--button--border-width: 1px;
			--form--border-width: 1px;
		';
	}

	if ( $archive_image_ratio ) {
		$ratio    = array(
			'20-9' => '45%',
			'16-9' => '56.25%',
			'3-2'  => '66.66667%',
			'4-3'  => '75%',
			'1-1'  => '100%',
			'3-4'  => '133.33333%',
			'2-3'  => '150%',
		);
		$css_var .= '--archive-img-ratio:' . $ratio[ $archive_image_ratio ] . ';';
	}

	if ( $archive_heading_size ) {
		$sizes    = array(
			'h2'          => 'var(--heading--font-size-h2)',
			'h3'          => 'var(--heading--font-size-h3)',
			'h4'          => 'var(--heading--font-size-h4)',
			'extra-large' => 'var(--global--font-size-xl)',
			'medium'      => 'min(calc(1.125 * var(--global--font-size-base)), 23px)',
		);
		$css_var .= '--archive-heading-size:' . $sizes[ $archive_heading_size ] . ';';
	}

	$css_var .= twentig_twentyone_generate_color_variables();

	$css_var .= '}';

	$css .= $css_var;

	if ( 'shadow' === $header_decoration ) {
		$css .= '.site-header { box-shadow: 0 2px 10px 0 rgba(0, 0, 0, 0.16); }';
	} elseif ( 'border' === $header_decoration ) {
		$css .= '.site-header { border-bottom: var(--form--border-width) solid var(--global--color-border); }';
	}

	if ( $post_h1_font_size ) {
		$css .= '.single-post { --global--font-size-page-title:' . twentig_twentyone_generate_value( $post_h1_font_size, 'rem' ) . '}';
	}

	/* Site title */
	if ( $show_site_title ) {
		$css_logo            = '';
		$logo_font           = get_theme_mod( 'twentig_logo_font' );
		$logo_font_weight    = get_theme_mod( 'twentig_logo_font_weight', '400' );
		$logo_letter_spacing = get_theme_mod( 'twentig_logo_letter_spacing', false );
		$logo_transform      = get_theme_mod( 'twentig_logo_text_transform', 'uppercase' );

		if ( $logo_font ) {
			$css_logo .= 'font-family: ' . $logo_font . ';';
		}

		if ( $logo_font_weight ) {
			$css_logo .= 'font-weight: ' . $logo_font_weight . ';';
		}

		if ( $logo_letter_spacing ) {
			$css_logo .= 'letter-spacing:' . twentig_twentyone_generate_value( $logo_letter_spacing, 'em' );
		}

		if ( $logo_transform ) {
			$css_logo .= 'text-transform: ' . esc_attr( $logo_transform ) . ';';
		}

		if ( $css_logo ) {
			$css .= '.site-header .site-title, .site-footer > .site-info .site-name {' . $css_logo . '}';
		}
	}

	/* Logo */
	if ( has_custom_logo() ) {
		$logo = wp_get_attachment_image_src( get_theme_mod( 'custom_logo' ), 'full' );

		if ( $logo ) {
			$logo_width                = $logo[1];
			$logo_height               = $logo[2];
			$logo_maxwidth             = get_theme_mod( 'twentig_logo_width' );
			$logo_mobile_maxwidth      = get_theme_mod( 'twentig_logo_width_mobile' );
			$logo_visual_height        = 30;
			$logo_visual_height_mobile = 30;
			$logo_ratio                = $logo_width && $logo_height ? $logo_height / $logo_width : 0;

			if ( empty( $logo_maxwidth ) && empty( $logo_mobile_maxwidth ) ) {
				$logo_mobile_maxwidth = 96;
			}

			if ( $logo_maxwidth ) {
				$css .= '.site-logo .custom-logo { max-width:' . twentig_twentyone_generate_value( $logo_maxwidth ) . 'max-height: none !important; }';
				if ( $logo_ratio ) {
					$logo_visual_height = min( $logo_maxwidth, $logo_width ) * $logo_ratio;
				}
			} else {
				if ( $logo_ratio ) {
					$logo_visual_height = min( 100, min( 300, $logo_width ) * $logo_ratio );
				}
			}
			if ( $logo_mobile_maxwidth ) {
				$css_mobile .= '.site-logo .custom-logo { max-width:' . twentig_twentyone_generate_value( $logo_mobile_maxwidth ) . 'max-height: none !important; }';
				if ( $logo_ratio ) {
					$logo_visual_height_mobile = min( $logo_mobile_maxwidth, $logo_width ) * $logo_ratio;
				}
			} else {
				$logo_visual_height_mobile = $logo_visual_height;
			}

			$css .= ':root{ --logo--height:' . esc_attr( round( $logo_visual_height, 2 ) ) . 'px; --logo--height-mobile:' . esc_attr( round( $logo_visual_height_mobile, 2 ) ) . 'px; }';
		}
	}

	/* Menu */
	if ( $menu_transform ) {
		$css .= '.primary-navigation, .menu-button-container .button { text-transform:' . esc_attr( $menu_transform ) . '; }';
	}

	if ( 'rounded' === $button_shape ) {
		$css .= '.search-form .search-submit, .wp-block-search .wp-block-search__button { border-radius: 0; }';
	} elseif ( 'pill' === $button_shape ) {
		$css .=
		'.search-form .search-submit, .wp-block-search .wp-block-search__button { 
			border-radius: 0;
		}
		.wp-block-button a.wp-block-button__link:focus,
		input[type=submit]:focus {
			outline-offset: 2px;
			outline: 1px dotted var(--button--color-background);
		}';
	}

	/* Minimal Link */
	if ( 'minimal' === get_theme_mod( 'twentig_footer_link_style' ) ) {
		$css .= '
		.site-footer a,
		.widget-area a {
			text-decoration: none;
		}
		.site-footer a:hover,
		.widget-area a:hover,
		.site-footer a:focus,
		.widget-area a:focus {
			text-decoration: underline;
		}
		.footer-navigation-wrapper li a:hover{
			text-decoration-style: solid;
		}';
	}

	/* Subtle Color */

	if ( $subtle_background ) {
		$css .= ':root { --wp--preset--color--subtle:' . $subtle_background . '; }';
		$css .= ':root .has-subtle-background-color, :root .has-subtle-background-background-color{ background-color: ' . $subtle_background . '; }';
		$css .= ':root .has-subtle-color{ color: ' . $subtle_background . '; }';
	}

	if ( twentig_twentyone_is_light_theme() && 36 > Twenty_Twenty_One_Custom_Colors::get_relative_luminance_from_hex( get_theme_mod( 'twentig_primary_color' ) ) ) {
		$css .= ':not(.has-text-color).has-green-background-color[class],
		:not(.has-text-color).has-blue-background-color[class],
		:not(.has-text-color).has-purple-background-color[class],
		:not(.has-text-color).has-red-background-color[class],
		:not(.has-text-color).has-orange-background-color[class],
		:not(.has-text-color).has-yellow-background-color[class] { 
			color: var(--global--color-primary); 
		}';
	}

	/* Footer */
	if ( $footer_link_color ) {
		$css .= '.footer-custom a { color: var(--footer--color-link); }';
	}

	/* Border */
	if ( 'thin' === $border_size ) {
		$css .= '
		.site-footer > .site-info,
		.single .site-main > article > .entry-footer,
		.page-header,
		.pagination,
		.comments-pagination,
		.wp-block-image.is-style-twentytwentyone-border img,
		.wp-block-image.is-style-twentytwentyone-image-frame img,
		.wp-block-latest-posts.is-style-twentytwentyone-latest-posts-borders li,
		.wp-block-media-text.is-style-twentytwentyone-border,
		.wp-block-group.is-style-twentytwentyone-border {
			border-width: 1px;
		}

		input[type=checkbox]:after {
			left: 7px;
			top: 4px;
		}';
	}

	/* Button */

	if ( 'uppercase' === $button_text_transform ) {
		$css .= 'input[type="submit"],
		.wp-block-button__link,
		.wp-block-file__button,
		.wp-block-search__button,
		.primary-navigation .menu-button a { 
			text-transform:uppercase; letter-spacing: 0.05em;
		}';
	}

	if ( $button_background ) {
		$css .= '
		#page input[type=submit],
		#page .wp-block-button:not(.is-style-outline) .wp-block-button__link:not(.has-background),
		.wp-block-file .wp-block-file__button { 
			background-color:' . $button_background . ';
			border-color:' . $button_background . ';';

		if ( $button_text_color ) {
			$css .= 'color:' . $button_text_color . ';';
		}
		$css .= '}';

		$css .= '#page .wp-block-button.is-style-outline .wp-block-button__link:not(.has-background):not(.has-text-color) { 
			border-color:' . $button_background . ';
			color:' . $button_background . ';
		}';

		$header_background = get_theme_mod( 'twentig_header_background_color' );
		if ( $header_button && $header_background !== $button_background ) {
			$css .= '
			#site-navigation .menu-button a,
			#site-navigation .menu-button a:focus { 
				background-color:' . $button_background . ';';
			if ( $button_text_color ) {
				$css .= 'color:' . $button_text_color . ';';
			}
			$css .= '}';
		}

		if ( $button_hover_background ) {
			$css .= '
			input[type=submit]:hover,
			input[type=submit]:focus,
			input[type=submit]:active,
			.wp-block-button:not(.is-style-outline) .wp-block-button__link:not(.has-background):hover,
			.wp-block-button:not(.is-style-outline) .wp-block-button__link:not(.has-background):focus,
			.wp-block-button:not(.is-style-outline) .wp-block-button__link:not(.has-background):active,
			.wp-block-button.is-style-outline .wp-block-button__link:not(.has-text-color):hover,
			.wp-block-button.is-style-outline .wp-block-button__link:not(.has-text-color):focus,
			.wp-block-button.is-style-outline .wp-block-button__link:not(.has-text-color):active,
			.wp-block-file .wp-block-file__button:hover,
			.wp-block-file .wp-block-file__button:focus,
			.wp-block-file .wp-block-file__button:active { 
				background-color:' . $button_hover_background . ' !important;
				border-color:' . $button_hover_background . ' !important;';
			if ( $button_text_color ) {
				$css .= 'color:' . $button_text_color . ' !important;';
			}
			$css .= '}';

			if ( $header_button && $header_background !== $button_background ) {
				$css .= '
				#site-navigation .menu-button a:hover,
				#site-navigation .menu-button a:focus { 
					opacity: 1; 
					background-color:' . $button_hover_background . ' !important;';
				if ( $button_text_color ) {
					$css .= 'color:' . $button_text_color . ' !important;';
				}
				$css .= '}';
			}
		} else {
			$css .= '
			input[type=submit]:hover,
			input[type=submit]:focus,
			input[type=submit]:active,
			.wp-block-button:not(.is-style-outline) .wp-block-button__link:not(.has-background):hover,
			.wp-block-button:not(.is-style-outline) .wp-block-button__link:not(.has-background):focus,
			.wp-block-button:not(.is-style-outline) .wp-block-button__link:not(.has-background):active,
			.wp-block-file .wp-block-file__button:hover,
			.wp-block-file .wp-block-file__button:focus,
			.wp-block-file .wp-block-file__button:active { 
				background-color: transparent !important;
				border-color:' . $button_background . ' !important;
				color:' . $button_background . ' !important;';
			$css .= '}';

			$css .= '
			.wp-block-button.is-style-outline .wp-block-button__link:not(.has-text-color):hover,
			.wp-block-button.is-style-outline .wp-block-button__link:not(.has-text-color):focus,
			.wp-block-button.is-style-outline .wp-block-button__link:not(.has-text-color):active {
				border-color: transparent;
				background-color:' . $button_background . ' !important;';
			if ( $button_text_color ) {
				$css .= 'color:' . $button_text_color . ' !important;';
			}
			$css .= '}';
		}
	}

	if ( $css_mobile ) {
		$css .= '@media( max-width:651px ) {' . $css_mobile . '}';
	}

	return $css;
}

/**
 * Generates CSS.
 */
function twentig_twentyone_generate_dynamic_css() {

	$css           = '';
	$header_layout = get_theme_mod( 'twentig_header_layout' );
	$header_break  = get_theme_mod( 'twentig_header_breakpoint', 'mobile' );
	$header_search = get_theme_mod( 'twentig_header_search', false );
	$header_button = get_theme_mod( 'twentig_header_button', false );
	$header_social = get_theme_mod( 'twentig_header_social_icons', false );
	$footer_layout = get_theme_mod( 'twentig_footer_layout' );
	$body_classes  = apply_filters( 'body_class', array(), array() ); // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
	$is_preview    = is_customize_preview();

	/* Header Layout */
	if ( $header_layout ) {
		$breakpoint = '482px';
		if ( 'tablet' === $header_break ) {
			$breakpoint = '1024px';
		} elseif ( 'tablet-landscape' === $header_break ) {
			$breakpoint = '1280px';
		}
		$css .= '@media (min-width:' . $breakpoint . ') {' . twentig_twentyone_load_component_styles( 'header-' . $header_layout ) . '}';
	}

	if ( $header_search || $is_preview ) {
		$css .= twentig_twentyone_load_component_styles( 'header-search' );
	}

	if ( $header_button || $header_social || $is_preview ) {
		$css .= twentig_twentyone_load_component_styles( 'header-actions' );
	}

	if ( 'mobile' !== $header_break ) {
		$breakpoint = 'tablet-landscape' === $header_break ? '1279px' : '1023px';
		$css       .= '@media (max-width:' . $breakpoint . ') {' .
			twentig_twentyone_load_component_styles( 'header-burger' );
		if ( has_custom_logo() && true === get_theme_mod( 'display_title_and_tagline', true ) ) {
			$css .= twentig_twentyone_load_component_styles( 'header-logo-and-title' );
		}
		$css .= '}';
	} else {
		$css .= twentig_twentyone_load_component_styles( 'header-break-mobile' );
	}

	/* Footer Layout */
	if ( in_array( $footer_layout, array( 'inline', 'stack', 'custom' ), true ) ) {
		$css .= twentig_twentyone_load_component_styles( 'footer-' . $footer_layout );
	}

	if ( in_array( 'tw-header-light', $body_classes, true ) ) {
		$css .= twentig_twentyone_load_component_styles( 'header-light' );
	}

	if ( is_page_template( 'tw-header-transparent.php' ) || is_page_template( 'tw-header-transparent-light.php' ) ) {
		$css .= twentig_twentyone_load_component_styles( 'template-transparent' );
	}

	if ( in_array( 'tw-hero-cover', $body_classes, true ) ) {
		$css .= twentig_twentyone_load_component_styles( 'hero-cover' );
	} elseif ( in_array( 'tw-hero-full-image', $body_classes, true ) ) {
		$css .= twentig_twentyone_load_component_styles( 'hero-full-image' );
	}

	if ( in_array( 'tw-hero-above', $body_classes, true ) ) {
		$css .= twentig_twentyone_load_component_styles( 'hero-above' );
	}

	if ( in_array( 'tw-blog-card', $body_classes, true ) ) {
		$css .= twentig_twentyone_load_component_styles( 'blog-card' );
	}

	if ( in_array( 'tw-blog-grid', $body_classes, true ) ) {
		$css .= twentig_twentyone_load_component_styles( 'blog-grid' );
	} elseif ( in_array( 'tw-blog-stack', $body_classes, true ) ) {
		$css .= twentig_twentyone_load_component_styles( 'blog-stack' );
		if ( in_array( 'tw-blog-card', $body_classes, true ) ) {
			$css .= twentig_twentyone_load_component_styles( 'blog-stack-card' );
		}
	}

	if ( in_array( 'tw-has-sidebar', $body_classes, true ) ) {
		$css .= twentig_twentyone_load_component_styles( 'sidebar' );
	}

	if ( ! in_array( 'no-widgets', $body_classes, true ) ) {
		$css .= twentig_twentyone_load_component_styles( 'footer-widgets' );

		$widgets_columns = get_theme_mod( 'twentig_footer_widgets_columns' );
		if ( '4' === $widgets_columns ) {
			$css .= '
			@media only screen and (min-width: 1024px) {
				.widget-area {
					column-gap: 40px;
					grid-template-columns: repeat(4, minmax(0, 1fr));
				}
			}';
		} elseif ( '2' === $widgets_columns || '1' === $widgets_columns ) {
			$css .= '
			@media only screen and (min-width: 822px) {			
				.widget-area {
					grid-template-columns: repeat(' . $widgets_columns . ', minmax(0, 1fr));
				}			
			}';
		}
	}

	if ( in_array( 'is-dark-theme', $body_classes, true ) || in_array( 'dark-theme', $body_classes, true ) || get_theme_mod( 'respect_user_color_preference', false ) ) {
		$css .= twentig_twentyone_load_component_styles( 'dark-theme' );
	}

	if ( is_admin_bar_showing() ) {
		$css .= twentig_twentyone_load_component_styles( 'admin-bar' );
	}

	if ( $is_preview ) {
		$css .= '.site-footer .customize-partial-edit-shortcut { display: none; }';
	}

	return $css;
}

/**
 * Load component styles.
 *
 * @param string $component_name The component name.
 */
function twentig_twentyone_load_component_styles( $component_name ) {
	$style_path = TWENTIG_PATH . "dist/css/twentytwentyone/components/$component_name.css";
	$styles     = '';
	if ( file_exists( $style_path ) ) {
		$styles = file_get_contents( $style_path );
	}
	return $styles;
}

/**
 * Generate CSS value with its unit.
 *
 * @param string $value The CSS value.
 * @param string $unit The CSS unit.
 */
function twentig_twentyone_generate_value( $value, $unit = 'px' ) {
	if ( 'rem' === $unit ) {
		return intval( $value ) / 16 . 'rem;';
	} elseif ( 'em' === $unit ) {
		return floatval( $value ) . 'em;';
	}
	return intval( $value ) . 'px;';
}

/**
 * Remove image inline size style.
 */
function twentig_twentyone_remove_image_inline_size_style() {
	remove_filter( 'wp_get_attachment_image_attributes', 'twenty_twenty_one_get_attachment_image_attributes' );
}

/**
 * Detects if a sidebar is displayed.
 */
function twentig_twentyone_has_sidebar() {
	if (
		is_active_sidebar( 'sidebar-blog' ) && (
		( get_theme_mod( 'twentig_blog_sidebar', false ) && ( twentig_twentyone_is_blog_page() || ( is_search() && 'blog-layout' === get_theme_mod( 'twentig_page_search_layout' ) ) ) )
		|| ( get_theme_mod( 'twentig_post_sidebar', false ) && is_singular( 'post' ) )
		)
	) {
		return true;
	}
	return false;
}

/**
 * Add custom image sizes attribute to enhance responsive image functionality.
 *
 * @param string $sizes A source size value for use in a 'sizes' attribute.
 */
function twentig_twentyone_calculate_image_sizes( $sizes ) {

	if ( ! in_the_loop() ) {
		return $sizes;
	}

	$wide_width         = get_theme_mod( 'twentig_wide_width', 1240 );
	$wide_width         = $wide_width ? $wide_width : 1240;
	$default_width      = get_theme_mod( 'twentig_default_width', 610 );
	$default_width      = $default_width ? $default_width : 610;
	$search_layout      = get_theme_mod( 'twentig_page_search_layout' );
	$has_sidebar        = twentig_twentyone_has_sidebar();
	$cpt_archive_layout = get_theme_mod( 'twentig_cpt_archive_layout' );

	if ( twentig_twentyone_is_blog_page()
		|| ( is_search() && 'blog-layout' === $search_layout )
		|| ( 'blog' === $cpt_archive_layout && twentig_twentyone_is_cpt_archive() )
	) {

		$blog_layout = get_theme_mod( 'twentig_blog_layout' );
		$image_width = get_theme_mod( 'twentig_blog_image_width', 'wide' );
		$blog_style  = get_theme_mod( 'twentig_blog_style', 'separator' );

		if ( $has_sidebar ) {
			if ( 'grid' === $blog_layout ) {
				$blog_columns = get_theme_mod( 'twentig_blog_columns', '3' );
				if ( '2' === $blog_columns ) {
					$sizes = '(min-width: 1280px) ' . intval( ( $wide_width - 440 ) / 2 ) . 'px, (min-width: 1024px) calc(50vw - 242px), (min-width: 652px) ' . intval( $default_width / 2 - 12 ) . 'px, (min-width: 482px) calc(100vw - 80px), calc(100vw - 40px)';
				} else {
					$sizes = '(min-width: 1280px) ' . intval( ( $wide_width - 480 ) / 3 ) . 'px, (min-width: 1024px) calc(50vw - 242px), (min-width: 652px) ' . intval( $default_width / 2 - 12 ) . 'px, (min-width: 482px) calc(100vw - 80px), calc(100vw - 40px)';
				}
			} else {
				$sizes = '(min-width: 1280px) ' . intval( $wide_width - 400 ) . 'px, (min-width: 1024px) calc(100vw - 460px), (min-width: 652px) ' . intval( $default_width ) . 'px, (min-width: 482px) calc(100vw - 80px), calc(100vw - 40px)';
			}
		} else {
			if ( 'grid' === $blog_layout ) {
				$blog_columns = get_theme_mod( 'twentig_blog_columns', '3' );
				if ( '2' === $blog_columns ) {
					$sizes = '(min-width: 1280px) ' . intval( $wide_width / 2 - 20 ) . 'px, (min-width: 822px) calc(50vw - 80px), (min-width: 652px) calc(50vw - 52px), (min-width: 482px) calc(100vw - 80px), calc(100vw - 40px)';
				} else {
					$sizes = '(min-width: 1280px) ' . intval( $wide_width / 3 - 27 ) . 'px, (min-width: 822px) calc(50vw - 80px), (min-width: 652px) calc(50vw - 52px), (min-width: 482px) calc(100vw - 80px), calc(100vw - 40px)';
				}
			} else {
				if ( 'text-width' === $image_width || 'card-shadow' === $blog_style || 'card-border' === $blog_style ) {
					$sizes = '(min-width: 652px) ' . intval( $default_width ) . 'px, (min-width: 482px) calc(100vw - 80px), calc(100vw - 40px)';
				} else {
					$sizes = '(min-width: 1280px) ' . intval( $wide_width ) . 'px, (min-width: 822px) calc(100vw - 120px), (min-width: 482px) calc(100vw - 80px), calc(100vw - 40px)';
				}
			}
		}
	} elseif ( has_post_thumbnail() && ! is_page_template() && ( is_singular( array( 'post', 'page' ) ) || twentig_twentyone_is_cpt_single() ) ) {
		$hero_layout = '';
		if ( is_page() ) {
			$hero_layout = get_theme_mod( 'twentig_page_hero_layout' );
		} elseif ( is_singular( 'post' ) ) {
			$hero_layout = get_theme_mod( 'twentig_post_hero_layout' );
		} elseif ( twentig_twentyone_is_cpt_single() ) {
			$layout_option = get_theme_mod( 'twentig_cpt_single_layout' );
			$hero_layout   = get_theme_mod( 'twentig_' . $layout_option . '_hero_layout' );
		}

		if ( $has_sidebar ) {
			$sizes = '(min-width: 1280px) ' . intval( $wide_width - 400 ) . 'px, (min-width: 1024px) calc(100vw - 460px), (min-width: 652px) ' . intval( $default_width ) . 'px, (min-width: 482px) calc(100vw - 80px), calc(100vw - 40px)';
		} elseif ( in_array( $hero_layout, array( 'full-image', 'cover', 'cover-full' ), true ) ) {
			$sizes = '100vw';
		} elseif ( 'narrow-image' === $hero_layout ) {
			$sizes = '(min-width: 652px) ' . intval( $default_width ) . 'px, (min-width: 482px) calc(100vw - 80px), calc(100vw - 40px)';
		} else {
			$sizes = '(min-width: 1280px) ' . intval( $wide_width ) . 'px, (min-width: 822px) calc(100vw - 120px), (min-width: 482px) calc(100vw - 80px), calc(100vw - 40px)';
		}
	}
	return $sizes;
}
add_filter( 'wp_calculate_image_sizes', 'twentig_twentyone_calculate_image_sizes' );

/**
 * Determines whether the query is for a blog page.
 */
function twentig_twentyone_is_blog_page() {
	if ( is_home() || is_author() || is_category() || is_tag() || is_date() || is_tax( get_object_taxonomies( 'post' ) ) ) {
		return true;
	}
	return false;
}

/**
 * Get all Custom Post Types excerpt Core and special ones that shouldn't be modified.
 */
function twentig_twentyone_get_cpt() {
	$args = array(
		'public'   => true,
		'_builtin' => false,
	);

	$post_types     = get_post_types( $args, 'names', 'and' );
	$cpt_to_exclude = apply_filters(
		'twentig_cpt_not_in',
		array(
			'product',
			'tribe_events',
			'course',
			'lesson',
			'forum',
			'topic',
			'reply',
		)
	);
	$results        = array_diff( $post_types, $cpt_to_exclude );

	return $results;
}

/**
 * Determines whether the query is for an existing singular CPT.
 */
function twentig_twentyone_is_cpt_single() {
	$cpts = twentig_twentyone_get_cpt();
	if ( empty( $cpts ) ) {
		return false;
	}

	if ( is_singular( $cpts ) ) {
		return true;
	}
	return false;
}

/**
 * Determines whether the query is for an archive or taxonomy CPT.
 */
function twentig_twentyone_is_cpt_archive() {
	$cpts    = twentig_twentyone_get_cpt();
	$cpt_tax = array();

	if ( empty( $cpts ) ) {
		return false;
	}

	if ( is_post_type_archive( $cpts ) ) {
		return true;
	}

	foreach ( $cpts as $post_type ) {
		$taxonomies = get_object_taxonomies( $post_type );
		$cpt_tax    = array_merge( $cpt_tax, $taxonomies );
	}

	if ( ! empty( $cpt_tax ) && is_tax( $cpt_tax ) ) {
		return true;
	}
	return false;
}

/**
 * Adds social icons svg sources.
 */
function twentig_twentyone_social_svg( $icons ) {
	$icons['patreon']  = '<svg width="24" height="24" viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M21 9.847a6.471 6.471 0 1 1-6.46-6.487A6.472 6.472 0 0 1 21 9.847ZM3 20.64h3.164V3.36H3Z"></path></svg>';
	$icons['telegram'] = '<svg width="24" height="24" viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M3.375 10.848q8.053-3.509 10.74-4.626c5.114-2.127 6.177-2.497 6.87-2.51a1.211 1.211 0 0 1 .713.215.776.776 0 0 1 .262.499 3.248 3.248 0 0 1 .031.725c-.277 2.912-1.476 9.978-2.086 13.24-.259 1.38-.767 1.843-1.259 1.888-1.07.099-1.881-.707-2.917-1.386-1.621-1.062-2.537-1.724-4.11-2.76-1.818-1.199-.64-1.857.396-2.933.272-.282 4.984-4.568 5.075-4.957a.376.376 0 0 0-.086-.325.424.424 0 0 0-.381-.037q-.244.055-7.768 5.134a3.496 3.496 0 0 1-1.997.738 13.143 13.143 0 0 1-2.863-.678c-1.153-.375-2.07-.573-1.99-1.21q.062-.497 1.37-1.017Z"></path></svg>';
	$icons['tiktok']   = '<svg width="24" height="24" viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M12.404 2.808c1.005-.016 2.003-.007 3-.016a4.78 4.78 0 0 0 1.343 3.202A5.418 5.418 0 0 0 20 7.366v3.09a8.151 8.151 0 0 1-3.223-.743A9.325 9.325 0 0 1 15.533 9c-.005 2.242.01 4.481-.014 6.714a5.841 5.841 0 0 1-1.038 3.025 5.713 5.713 0 0 1-4.535 2.46 5.584 5.584 0 0 1-3.129-.787 5.782 5.782 0 0 1-2.799-4.383 12.832 12.832 0 0 1-.01-1.142 5.782 5.782 0 0 1 6.7-5.129c.015 1.137-.03 2.272-.03 3.409a2.64 2.64 0 0 0-3.363 1.628 3.05 3.05 0 0 0-.105 1.234 2.62 2.62 0 0 0 4.811.969 1.81 1.81 0 0 0 .315-.816c.075-1.372.045-2.74.055-4.112.006-3.093-.01-6.178.014-9.262Z""></path></svg>';
	$icons['whatsapp'] = '<svg width="24" height="24" viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M18.732 5.229a9.462 9.462 0 0 0-14.89 11.414L2.5 21.546l5.015-1.316a9.452 9.452 0 0 0 4.52 1.152h.005a9.464 9.464 0 0 0 6.692-16.153ZM12.04 19.784h-.003a7.851 7.851 0 0 1-4.003-1.096l-.287-.17-2.976.78.795-2.901-.187-.298a7.863 7.863 0 1 1 6.66 3.685Zm4.766-5.613a1.977 1.977 0 0 1-.138 1.124 2.434 2.434 0 0 1-1.596 1.124 3.24 3.24 0 0 1-1.488-.094 13.59 13.59 0 0 1-1.348-.498 10.537 10.537 0 0 1-4.038-3.569 4.598 4.598 0 0 1-.965-2.445A2.651 2.651 0 0 1 8.06 7.84a.868.868 0 0 1 .63-.296c.158 0 .316.002.453.008.146.008.34-.055.532.406.197.474.67 1.637.73 1.755a.435.435 0 0 1 .019.415 1.614 1.614 0 0 1-.237.394c-.118.138-.248.308-.354.414-.118.118-.242.246-.104.482a7.127 7.127 0 0 0 1.315 1.638 6.454 6.454 0 0 0 1.901 1.173c.236.119.374.099.512-.059s.591-.69.749-.927.315-.197.532-.118 1.378.65 1.615.769.394.177.453.276Z"/></path></svg>';
	$icons['x']        = '<svg width="24" height="24" viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M13.982 10.622 20.54 3h-1.554l-5.693 6.618L8.745 3H3.5l6.876 10.007L3.5 21h1.554l6.012-6.989L15.868 21h5.245l-7.131-10.378Zm-2.128 2.474-.697-.997-5.543-7.93H8l4.474 6.4.697.996 5.815 8.318h-2.387l-4.745-6.787Z"></path></svg>';
	$icons['bluesky']  = '<svg width="24" height="24" viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M6.3,4.2c2.3,1.7,4.8,5.3,5.7,7.2.9-1.9,3.4-5.4,5.7-7.2,1.7-1.3,4.3-2.2,4.3.9s-.4,5.2-.6,5.9c-.7,2.6-3.3,3.2-5.6,2.8,4,.7,5.1,3,2.9,5.3-5,5.2-6.7-2.8-6.7-2.8,0,0-1.7,8-6.7,2.8-2.2-2.3-1.2-4.6,2.9-5.3-2.3.4-4.9-.3-5.6-2.8-.2-.7-.6-5.3-.6-5.9,0-3.1,2.7-2.1,4.3-.9h0Z"></path></svg>';

	return $icons;
}
add_filter( 'twenty_twenty_one_svg_icons_social', 'twentig_twentyone_social_svg' );

/**
 * Adds social icons domain mappings.
 */
function twentig_twentyone_social_map( $icons ) {
	$icons['mastodon'] = array(
		'mastodon.social',
	);
	$icons['telegram'] = array(
		't.me',
		'telegram.me',
	);
	$icons['whatsapp'] = array(
		'wa.me',
		'whatsapp.com',
	);
	$icons['bluesky'] = array(
		'bsky.app',
	);
	return $icons;
}
add_filter( 'twenty_twenty_one_social_icons_map', 'twentig_twentyone_social_map' );

/**
 * Updates margin sizes.
 */
function twentig_twentyone_spacing_sizes( $sizes ) {
	$sizes[9]['size']  = $sizes[9]['name'] = '90px';
	$sizes[10]['size'] = $sizes[10]['name'] = '120px';
	return $sizes;
}
add_filter( 'twentig_spacing_sizes', 'twentig_twentyone_spacing_sizes' );

/**
 * Updates breakpoints.
 */
add_filter( 'twentig_breakpoints', function() {
	return array(
		'mobile' => 652,
		'tablet' => 1024
	);
} );
