<?php

class SyncBasalamJobsRunner
{
    private static $instance = null;

    private $jobTypesPriority = [
        'sync_basalam_bulk_update_products' => 1,
        'sync_basalam_update_all_products' => 2,
        'sync_basalam_update_single_product' => 3,
        'sync_basalam_create_single_product' => 4,
        'sync_basalam_create_all_products' => 5
    ];

    private $jobManager;

    public static function getInstance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    function __construct()
    {
        add_action('init', [$this, 'checkAndRunJobs']);
        $this->jobManager = SyncBasalamJobManager::getInstance();
    }

    public function checkAndRunJobs()
    {
        $this->jobManager->deleteStaleProcessingJobs(120);

        $tasksPerMinute = max(1, intval(SyncBasalamAdminSettings::getEffectiveTasksPerMinute()));
        $thresholdSeconds = 60.0 / $tasksPerMinute;

        asort($this->jobTypesPriority);

        foreach ($this->jobTypesPriority as $jobType => $priority) {
            if (!$this->acquireLock($jobType, 0)) {
                continue;
            }

            try {
                $lastRun = floatval(get_option($jobType . '_last_run', 0));
                $now = microtime(true);

                if (($now - $lastRun) >= $thresholdSeconds) {

                    if ($jobType === 'sync_basalam_create_all_products') {
                        if (!$this->areAllCreateSingleJobsCompleted()) {
                            continue;
                        }
                    }

                    if ($jobType === 'sync_basalam_update_all_products') {
                        if (!$this->areAllUpdateSingleJobsCompleted()) {
                            continue;
                        }
                    }

                    $job = $this->jobManager->getJob(['job_type' => $jobType, 'status' => 'pending']);
                    $proccesing_job = $this->jobManager->getJob(['job_type' => $jobType, 'status' => 'processing']);

                    if ($job && !$proccesing_job) {
                        update_option($jobType . '_last_run', microtime(true), false);

                        $this->jobManager->updateJob(['status' => 'processing', 'started_at' => time()], ['id' => $job->id]);

                        $this->releaseLock($jobType);

                        $this->executeJob($job);

                        break;
                    }
                }
            } finally {
                $this->releaseLock($jobType);
            }
        }
    }

    /**
     * Acquire MySQL named lock for atomic job execution
     *
     * @param string $jobType
     * @param int $timeout Timeout in seconds (0 = non-blocking)
     * @return bool True if lock acquired, false otherwise
     */
    private function acquireLock($jobType, $timeout = 0)
    {
        global $wpdb;

        $lockName = 'sync_basalam_job_' . $jobType;
        $result = $wpdb->get_var($wpdb->prepare("SELECT GET_LOCK(%s, %d)", $lockName, $timeout));

        return $result === '1';
    }

    /**
     * Release MySQL named lock
     *
     * @param string $jobType
     * @return bool True if lock released, false if lock was not held
     */
    private function releaseLock($jobType)
    {
        global $wpdb;

        $lockName = 'sync_basalam_job_' . $jobType;
        $result = $wpdb->get_var($wpdb->prepare("SELECT RELEASE_LOCK(%s)", $lockName));

        return $result === '1';
    }

    private function executeJob($job)
    {
        $jobType = $job->job_type;
        $payload = json_decode($job->payload, true);

        if ($jobType === 'sync_basalam_bulk_update_products') {
            $this->bulkUpdateProducts($payload);
            $this->jobManager->deleteJob(['id' => $job->id]);
        } elseif ($jobType === 'sync_basalam_update_all_products') {
            $this->UpdateAllProducts($payload);
            $this->jobManager->deleteJob(['id' => $job->id]);
        } elseif ($jobType === 'sync_basalam_update_single_product') {
            $this->updateSingleProduct($payload);
            $this->jobManager->deleteJob(['id' => $job->id]);
        } elseif ($jobType === 'sync_basalam_create_single_product') {
            $this->createSingleProduct($payload);
            $this->jobManager->deleteJob(['id' => $job->id]);
        } elseif ($jobType === 'sync_basalam_create_all_products') {
            $this->createAllProducts($payload);
            $this->jobManager->deleteJob(['id' => $job->id]);
        }
    }

    private function bulkUpdateProducts($offset)
    {
        SyncBasalamProductService::enqueueBulkUpdateProducts($offset);
    }
    private function updateSingleProduct($payload)
    {
        $productId = $payload['product_id'] ?? $payload;
        if ($productId) {
            $productOperations = new SyncBasalamAdminProductOperations();
            $productOperations->updateExistProduct($productId, null);
        }
    }
    private function createSingleProduct($payload)
    {
        $productId = $payload['product_id'] ?? $payload;
        if ($productId) {
            $productOperations = new SyncBasalamAdminProductOperations();
            $productOperations->createNewProduct($productId, null);
        }
    }

    private function createAllProducts($payload)
    {
        $postsPerPage = 100;
        $includeOutOfStock = $payload['include_out_of_stock'] ?? false;

        $batch_data = [
            'posts_per_page'        => $postsPerPage,
            'include_out_of_stock'  => $includeOutOfStock
        ];

        $productIds = SyncBasalamProductService::getCreatableProducts($batch_data);

        if (!$productIds) {
            delete_option('sync_basalam_last_creatable_product_id');
            return false;
        }

        foreach ($productIds as $product_id) {
            if (!$this->hasProductJobInProgress($product_id, 'sync_basalam_create_single_product')) {
                $this->jobManager->createJob(
                    'sync_basalam_create_single_product',
                    'pending',
                    json_encode(['product_id' => $product_id])
                );
            }
        }

        $next_batch_data = json_encode([
            'posts_per_page'       => $postsPerPage,
            'include_out_of_stock' => $includeOutOfStock
        ]);

        $this->jobManager->createJob(
            'sync_basalam_create_all_products',
            'pending',
            $next_batch_data
        );
    }

    private function UpdateAllProducts($payload)
    {
        $batch_data = [
            'posts_per_page'        => 100,
        ];

        $productIds = SyncBasalamProductService::getUpdatableProducts($batch_data);

        if (!$productIds) {
            delete_option('sync_basalam_last_updatable_product_id');
            return false;
        }

        foreach ($productIds as $product_id) {
            if (!$this->hasProductJobInProgress($product_id, 'sync_basalam_update_single_product')) {
                $this->jobManager->createJob(
                    'sync_basalam_update_single_product',
                    'pending',
                    json_encode(['product_id' => $product_id])
                );
            }
        }

        $this->jobManager->createJob(
            'sync_basalam_update_all_products',
            'pending',
            []
        );
    }

    private function areAllCreateSingleJobsCompleted()
    {
        $pendingJob = $this->jobManager->getJob([
            'job_type' => 'sync_basalam_create_single_product',
            'status' => 'pending'
        ]);

        if ($pendingJob) {
            return false;
        }

        return true;
    }

    private function areAllUpdateSingleJobsCompleted()
    {

        $pendingJob = $this->jobManager->getJob([
            'job_type' => 'sync_basalam_update_single_product',
            'status' => 'pending'
        ]);

        if ($pendingJob) {
            return false;
        }

        return true;
    }

    public function hasProductJobInProgress($productId, $jobType)
    {
        global $wpdb;

        $tableName = $wpdb->prefix . 'sync_basalam_job_manager';

        $jobs = $wpdb->get_results($wpdb->prepare(
            "SELECT payload FROM {$tableName}
            WHERE job_type = %s
            AND (status = %s OR status = %s)",
            $jobType,
            'pending',
            'processing'
        ));

        if (empty($jobs)) {
            return false;
        }

        foreach ($jobs as $job) {
            $payload = json_decode($job->payload, true);
            $jobProductId = $payload['product_id'] ?? $payload;

            if (intval($jobProductId) === intval($productId)) {
                return true;
            }
        }

        return false;
    }
}
