<?php
if (!defined('ABSPATH')) exit;
class SyncBasalamProductDiscountHandler
{
    private const PRICE_FIELD_SALE = 'sale_strikethrough_price';

    private $discountService;
    private $priceField;
    private $syncPriceMode;
    private $syncFieldsMode;
    private $taskScheduler;

    public function __construct()
    {
        require_once SYNC_BASALAM_PLUGIN_INCLUDES_DIR . 'services/class-sync-basalam-discount-task-scheduler.php';

        $this->discountService = new SyncBasalamDiscountManager();
        $this->priceField = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::PRODUCT_PRICE_FIELD);
        $this->syncPriceMode = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELD_PRICE);
        $this->syncFieldsMode = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELDS);
        $this->taskScheduler = new SyncBasalamDiscountTaskScheduler();
    }

    public function handle(int $productId)
    {
        try {
            if ($this->syncFieldsMode == 'custom') {
                if (!$this->syncPriceMode) {
                    return false;
                }
            }
            $product = wc_get_product($productId);
            if (!$product) {
                throw new Exception('محصول یافت نشد.');
            }

            if ($product->is_type('variable')) {
                return $this->handleVariableProduct($product);
            } else {
                return $this->handleSimpleProduct($product);
            }
        } catch (Throwable $th) {
            return $this->error('عملیات تخفیف ناموفق بود', $th);
        }
    }

    private function scheduleDiscountTasks(WC_Product $product): array
    {
        $active_days = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::DISCOUNT_DURATION);
        $tasks_created = 0;

        if ($product->is_type('simple')) {
            $sale_price = $product->get_sale_price();
            $regular_price = $product->get_regular_price();

            if ($sale_price && $regular_price) {
                $discount_percent = SyncBasalamDiscountManager::calculateDiscountPercent($regular_price, $sale_price);
                $basalam_p_id = get_post_meta($product->get_id(), 'sync_basalam_product_id', true);
                if ($discount_percent > 0 && $basalam_p_id) {

                    $this->taskScheduler->scheduleDiscountTask(
                        $basalam_p_id,
                        $discount_percent,
                        $active_days,
                        null,
                        0,
                        'apply'
                    );
                    $tasks_created++;
                }
            }
        } elseif ($product->is_type('variable')) {
            foreach ($product->get_children() as $variation_id) {
                $variation = wc_get_product($variation_id);
                if ($variation && $variation->get_sale_price() && $variation->get_regular_price()) {
                    $discount_percent = SyncBasalamDiscountManager::calculateDiscountPercent(
                        $variation->get_regular_price(),
                        $variation->get_sale_price()
                    );

                    $basalam_p_id = get_post_meta($product->get_id(), 'sync_basalam_product_id', true);
                    $basalam_v_id = get_post_meta($variation_id, 'sync_basalam_variation_id', true);

                    if ($discount_percent > 0) {

                        $this->taskScheduler->scheduleDiscountTask(
                            null,
                            $discount_percent,
                            $active_days,
                            $basalam_v_id,
                            0,
                            'apply'
                        );
                        $tasks_created++;
                    }
                }
            }
        }

        if ($tasks_created > 0) {

            $this->taskScheduler->startProcessor();

            $this->scheduleDiscountProcessor();

            return $this->success("تخفیف برای پردازش در صف قرار گرفت. {$tasks_created} وظیفه تخفیف ایجاد شد.");
        }

        return $this->success('هیچ تخفیف معتبری برای اعمال یافت نشد.');
    }

    private function resolveStrategy(WC_Product $product): SyncBasalamDiscountInterface
    {
        if ($product->is_type('variable')) {
            return new SyncBasalamVariableProductDiscount($this->discountService);
        }
        return new SyncBasalamSimpleProductDiscount($this->discountService);
    }

    private function shouldApplySalePriceDiscount(WC_Product $product): bool
    {

        if ($this->priceField !== self::PRICE_FIELD_SALE) {
            return false;
        }

        if ($product->is_type('simple') && $product->get_sale_price()) {
            return true;
        }

        if ($product->is_type('variable')) {
            foreach ($product->get_children() as $variation_id) {
                $variation = wc_get_product($variation_id);
                if ($variation && $variation->get_sale_price()) {
                    return true;
                }
            }
        }

        return false;
    }

    private function handleSimpleProduct(WC_Product $product): array
    {

        if ($this->shouldApplySalePriceDiscount($product)) {
            return $this->scheduleDiscountTasks($product);
        } else {
            return $this->checkAndScheduleRemoveDiscount($product);
        }
    }

    private function handleVariableProduct(WC_Product $product): array
    {
        $tasks_created = 0;
        $apply_tasks = 0;
        $remove_tasks = 0;

        require_once SYNC_BASALAM_PLUGIN_INCLUDES_DIR . 'services/class-sync-basalam-discount-task-processor.php';
        $processor = new SyncBasalamDiscountTaskProcessor();

        foreach ($product->get_children() as $variation_id) {
            $variation = wc_get_product($variation_id);
            if (!$variation) continue;

            $result = $this->handleSingleVariation($variation, $product->get_id(), $processor);
            if ($result['action'] === 'apply') {
                $apply_tasks++;
                $tasks_created++;
            } elseif ($result['action'] === 'remove') {
                $remove_tasks++;
                $tasks_created++;
            }
        }

        if ($tasks_created > 0) {

            $this->taskScheduler->startProcessor();
            $this->scheduleDiscountProcessor();

            $message = "پردازش محصول متغیر انجام شد. ";
            if ($apply_tasks > 0) {
                $message .= "{$apply_tasks} تسک اعمال تخفیف ایجاد شد. ";
            }
            if ($remove_tasks > 0) {
                $message .= "{$remove_tasks} تسک حذف تخفیف ایجاد شد. ";
            }

            return $this->success($message);
        }

        return $this->success('هیچ تغییری در تخفیف وریشن‌ها لازم نیست.');
    }

    private function handleSingleVariation(WC_Product_Variation $variation, int $parent_product_id, $processor): array
    {
        $variation_id = $variation->get_id();
        $has_sale_price = !empty($variation->get_sale_price());
        $has_discount_meta = $this->variationHasDiscountMeta($variation_id);

        if ($this->priceField !== self::PRICE_FIELD_SALE) {
            if ($has_discount_meta) {
                $result = $processor->createRemoveDiscountTask($parent_product_id, $variation_id);
                if ($result['success']) {
                    return ['action' => 'remove', 'success' => true];
                }
            }
            return ['action' => 'none', 'success' => true];
        }

        if ($has_sale_price) {

            $discount_percent = $this->calculateVariationDiscountPercent($variation);
            if ($discount_percent > 0) {
                $active_days = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::DISCOUNT_DURATION);
                $basalam_product_id = get_post_meta($parent_product_id, 'sync_basalam_product_id', true);
                $basalam_variation_id = get_post_meta($variation_id, 'sync_basalam_variation_id', true);

                if ($basalam_variation_id) {
                    $this->taskScheduler->scheduleDiscountTask(
                        null,
                        $discount_percent,
                        $active_days,
                        $basalam_variation_id,
                        0,
                        'apply'
                    );
                    return ['action' => 'apply', 'success' => true];
                }
            }
            return ['action' => 'none', 'success' => true];
        } else {

            if ($has_discount_meta) {
                $result = $processor->createRemoveDiscountTask($parent_product_id, $variation_id);
                if ($result['success']) {
                    return ['action' => 'remove', 'success' => true];
                }
            }
            return ['action' => 'none', 'success' => true];
        }
    }

    private function variationHasDiscountMeta(int $variation_id): bool
    {
        $had_discount_applied = get_post_meta($variation_id, '_sync_basalam_discount_applied', true);
        $had_discount_percent = get_post_meta($variation_id, '_sync_basalam_discount_percent', true);
        return !empty($had_discount_applied) || !empty($had_discount_percent);
    }

    private function calculateVariationDiscountPercent(WC_Product_Variation $variation): float
    {
        $regular_price = $variation->get_regular_price();
        $sale_price = $variation->get_sale_price();

        if ($regular_price && $sale_price && $regular_price > $sale_price) {
            return SyncBasalamDiscountManager::calculateDiscountPercent($regular_price, $sale_price);
        }

        return 0;
    }

    private function checkAndScheduleRemoveDiscount(WC_Product $product): array
    {
        $tasks_created = 0;
        require_once SYNC_BASALAM_PLUGIN_INCLUDES_DIR . 'services/class-sync-basalam-discount-task-processor.php';
        $processor = new SyncBasalamDiscountTaskProcessor();

        $should_remove_discount = $this->shouldRemoveDiscount($product);

        if (!$should_remove_discount) {
            return $this->success('شرایط حذف تخفیف برقرار نیست.');
        }

        if ($product->is_type('simple')) {

            $had_discount_applied = get_post_meta($product->get_id(), '_sync_basalam_discount_applied', true);
            $had_discount_percent = get_post_meta($product->get_id(), '_sync_basalam_discount_percent', true);

            if ($had_discount_applied || $had_discount_percent) {
                $result = $processor->createRemoveDiscountTask($product->get_id());
                if ($result['success']) {
                    $tasks_created++;
                }
            }
        } elseif ($product->is_type('variable')) {

            foreach ($product->get_children() as $variation_id) {
                $had_discount_applied = get_post_meta($variation_id, '_sync_basalam_discount_applied', true);
                $had_discount_percent = get_post_meta($variation_id, '_sync_basalam_discount_percent', true);

                if ($had_discount_applied || $had_discount_percent) {

                    $variation = wc_get_product($variation_id);
                    if ($variation) {
                        $should_remove = $this->shouldRemoveDiscountForVariation($variation);

                        if ($should_remove) {
                            $result = $processor->createRemoveDiscountTask($product->get_id(), $variation_id);
                            if ($result['success']) {
                                $tasks_created++;
                            } else {
                            }
                        }
                    }
                }
            }
        }

        if ($tasks_created > 0) {

            $this->taskScheduler->startProcessor();

            $this->scheduleDiscountProcessor();

            return $this->success("تسک حذف تخفیف برای پردازش در صف قرار گرفت. {$tasks_created} وظیفه حذف تخفیف ایجاد شد.");
        }

        return $this->success('هیچ تخفیف قبلی برای حذف یافت نشد.');
    }

    private function shouldRemoveDiscount(WC_Product $product): bool
    {

        $product_has_discount_meta = $this->productHasDiscountMeta($product);

        if (!$product_has_discount_meta) {
            return false;
        }

        if ($this->priceField !== self::PRICE_FIELD_SALE) {
            return true;
        }

        if ($product->is_type('simple') && !$product->get_sale_price()) {
            return true;
        }

        if ($product->is_type('variable')) {

            foreach ($product->get_children() as $variation_id) {
                $variation = wc_get_product($variation_id);
                if (!$variation) continue;

                $had_discount_applied = get_post_meta($variation_id, '_sync_basalam_discount_applied', true);
                $had_discount_percent = get_post_meta($variation_id, '_sync_basalam_discount_percent', true);

                if ($had_discount_applied || $had_discount_percent) {

                    if ($this->priceField !== self::PRICE_FIELD_SALE) {
                        return true;
                    }

                    if (!$variation->get_sale_price()) {
                        return true;
                    }
                }
            }
        }

        return false;
    }

    private function shouldRemoveDiscountForVariation(WC_Product_Variation $variation): bool
    {

        if ($this->priceField !== self::PRICE_FIELD_SALE) {
            return true;
        }

        if (!$variation->get_sale_price()) {
            return true;
        }

        return false;
    }

    private function productHasDiscountMeta(WC_Product $product): bool
    {
        if ($product->is_type('simple')) {
            $had_discount_applied = get_post_meta($product->get_id(), '_sync_basalam_discount_applied', true);
            $had_discount_percent = get_post_meta($product->get_id(), '_sync_basalam_discount_percent', true);
            return !empty($had_discount_applied) || !empty($had_discount_percent);
        }

        if ($product->is_type('variable')) {
            foreach ($product->get_children() as $variation_id) {
                $had_discount_applied = get_post_meta($variation_id, '_sync_basalam_discount_applied', true);
                $had_discount_percent = get_post_meta($variation_id, '_sync_basalam_discount_percent', true);
                if (!empty($had_discount_applied) || !empty($had_discount_percent)) {
                    return true;
                }
            }
        }

        return false;
    }

    private function scheduleDiscountProcessor()
    {
        require_once SYNC_BASALAM_PLUGIN_INCLUDES_DIR . 'services/class-sync-basalam-discount-task-processor.php';
        SyncBasalamDiscountTaskProcessor::scheduleRecurringProcessor();
    }

    private function success(string $message): array
    {
        return [
            'success' => true,
            'message' => $message,
            'status_code' => 200
        ];
    }

    private function error(string $message, Throwable $exception): array
    {
        return [
            'success' => false,
            'message' => $message . ' : ' . $exception->getMessage(),
            'error' => $exception->getMessage(),
            'status_code' => 400
        ];
    }
}
