<?php
if (!defined('ABSPATH')) exit;
class SyncBasalamWebhookService
{
    private ?string $token;
    private SyncBasalamApiServiceManager $apiService;

    const TARGET_EVENT_IDS = [3, 5, 7];

    public function __construct()
    {
        $this->token = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::TOKEN);
        $this->apiService = new SyncBasalamApiServiceManager();
    }

    public function setupWebhook()
    {
        if (!$this->token) {
            return false;
        }

        $existingWebhooks = $this->fetchWebhooks();

        if (!$existingWebhooks || !isset($existingWebhooks['data'])) {
            return false;
        }

        $webhookUrl = SyncBasalamAdminSettings::getStaticSettings("site_url_webhook");

        $correctWebhook = null;
        $webhooksMarkedForDeletion = [];

        foreach ($existingWebhooks['data'] as $webhook) {
            if (isset($webhook['events']) && is_array($webhook['events'])) {
                $webhookEventIds = [];
                foreach ($webhook['events'] as $event) {
                    if (isset($event['id'])) {
                        $webhookEventIds[] = $event['id'];
                    }
                }

                $hasTargetEvents = !empty(array_intersect($webhookEventIds, self::TARGET_EVENT_IDS));

                if ($hasTargetEvents) {
                    $hasAllEvents = count(array_intersect($webhookEventIds, self::TARGET_EVENT_IDS)) == count(self::TARGET_EVENT_IDS);

                    if ($hasAllEvents) {
                        $correctWebhook = $webhook['id'];
                    } else {
                        $webhooksMarkedForDeletion[] = $webhook['id'];
                    }
                }
            }
        }

        foreach (array_unique($webhooksMarkedForDeletion) as $webhookId) {
            $this->removeCurrentWebhook($webhookId);
        }

        if ($correctWebhook) {
            $this->updateCurrentWebhook($correctWebhook, self::TARGET_EVENT_IDS, $webhookUrl);
        } else {
            $this->createNewWebhook(self::TARGET_EVENT_IDS, $webhookUrl);
        }

        return true;
    }

    private function fetchWebhooks()
    {
        $headers = [
            'authorization' => $this->token
        ];

        $response = $this->apiService->sendGetRequest('https://webhook.basalam.com/v1/webhooks', $headers);

        if ($response && $response['status_code'] == 200) {
            return $response['data'];
        }

        return null;
    }

    private function createNewWebhook($eventIds, $webhookUrl)
    {
        $webhookToken = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::WEBHOOK_HEADER_TOKEN);
        $headers = [
            'authorization' => $this->token
        ];

        $data = json_encode([
            'event_ids' => array_values($eventIds),
            'request_headers' => json_encode(['token' => $webhookToken]),
            'request_method' => 'POST',
            'url' => $webhookUrl,
            'is_active' => true,
            'register_me' => true
        ]);

        $response = $this->apiService->sendPostRequest('https://webhook.basalam.com/v1/webhooks', $data, $headers);

        if ($response && $response['status_code'] == 200) {
            return true;
        } else {
            return false;
        }
    }

    private function updateCurrentWebhook($webhookId, $eventIds, $webhookUrl)
    {
        $webhookToken = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::WEBHOOK_HEADER_TOKEN);

        $headers = [
            'authorization' => $this->token
        ];

        $data = json_encode([
            'event_ids' => array_values($eventIds),
            'request_headers' => json_encode(['token' => $webhookToken]),
            'request_method' => 'POST',
            'url' => $webhookUrl,
            'is_active' => true
        ]);

        $updateUrl = 'https://webhook.basalam.com/v1/webhooks/' . $webhookId;

        $response = $this->apiService->sendPatchRequest($updateUrl, $data, $headers);

        if ($response && $response['status_code'] == 200) {
            return true;
        } else {
            return false;
        }
    }

    private function removeCurrentWebhook($webhookId)
    {
        $headers = [
            'authorization' => $this->token
        ];

        $deleteUrl = 'https://webhook.basalam.com/v1/webhooks/' . $webhookId;

        $response = $this->apiService->sendDeleteRequest($deleteUrl, $headers);

        if ($response && ($response['status_code'] == 200 || $response['status_code'] == 204)) {
            return true;
        } else {
            return false;
        }
    }
}
