<?php
/**
 * Sync Basalam Get Category ID Service
 *
 * This service handles automatic category detection and retrieval from Basalam's
 * machine learning-powered category detection API. It analyzes product titles
 * to suggest appropriate Basalam categories, enabling automatic categorization
 * of products during synchronization.
 *
 * The service supports multiple modes:
 * - Single category detection (returns best match)
 * - Multiple category suggestions (returns all possibilities)
 * - Full category tree extraction with hierarchical data
 *
 * @package    Sync_Basalam_Plugin
 * @subpackage Services
 * @since      1.0.0
 * @author     Basalam Team
 */

// Prevent direct access to this file
if (! defined('ABSPATH')) exit;

/**
 * Get Category ID Service Class
 *
 * Interfaces with Basalam's AI-powered category detection system to:
 * - Analyze product titles for category suggestions
 * - Extract hierarchical category structures
 * - Provide multiple categorization options
 * - Handle category tree traversal and data extraction
 *
 * @class SyncBasalamGetCategoryId
 * @since 1.0.0
 */
class SyncBasalamGetCategoryId
{
    /**
     * Get category ID suggestions from Basalam's AI detection service
     *
     * Uses Basalam's machine learning API to analyze product titles and suggest
     * appropriate categories. Supports different modes of operation:
     *
     * - 'single': Returns the best matching category ID only
     * - 'all': Returns all suggested categories with full hierarchy
     * - 'multiple': Returns array of category IDs for the best match
     *
     * The API analyzes the product title using natural language processing
     * to determine the most suitable category placement on Basalam.
     *
     * @since 1.0.0
     * @param string $productTitle The product title to analyze
     * @param string $mode Detection mode: 'single', 'all', or 'multiple'
     * @return mixed Category ID(s) or false on failure
     *   - 'single' mode: int|false
     *   - 'all' mode: array|false
     *   - 'multiple' mode: array|false
     *
     * @example
     * // Get single best category
     * $category_id = SyncBasalamGetCategoryId::getCategoryIdFromBasalam('iPhone 13 Pro');
     *
     * // Get all suggested categories
     * $all_categories = SyncBasalamGetCategoryId::getCategoryIdFromBasalam('iPhone 13 Pro', 'all');
     */
    static function getCategoryIdFromBasalam($productTitle, $mode = 'single')
    {
        // Initialize API service for external communication
        $apiservice = new SyncBasalamApiServiceManager;

        // Build category detection API endpoint URL
        $url = "https://categorydetection.basalam.com/category_detection/api_v1.0/predict/?title=" . $productTitle;

        // Send request to Basalam's category detection service
        $result = $apiservice->sendGetRequest($url, []);

        // Handle different return modes
        if ($mode != 'single') {
            // Return all category suggestions with full hierarchy
            if ($mode == 'all') {
                if (isset($result['data']['result']) && count($result['data']['result']) > 0) {
                    $categories = [];

                    // Process each suggested category
                    foreach ($result['data']['result'] as $category) {
                        $catIds = [];

                        // Extract all category IDs in the hierarchy
                        self::extractCategoryIds([$category], $catIds);

                        // Build category data with IDs and full title path
                        $categoryData = [
                            'cat_id' => $catIds,
                            'cat_title' => self::getCombinedTitles($category['cat_parent'], $category['cat_title'])
                        ];
                        $categories[] = $categoryData;
                    }
                    return $categories;
                }
            }
            // Return multiple category IDs for the best match (hierarchical array)
            if (isset($result['data']['result']) && count($result['data']['result']) > 0) {
                $categoryIds = [];

                // Extract all category IDs from the best matching category
                self::extractCategoryIds([$result['data']['result'][0]], $categoryIds);

                return $categoryIds;
            }
        }
        // Single mode: return only the best matching category ID
        if (isset($result['data']['result'][0]['cat_id'])) {
            return $result['data']['result'][0]['cat_id'];
        }

        // Return false if no categories were found
        return false;
    }

    /**
     * Extract category IDs from hierarchical category structure
     *
     * Recursively traverses a category tree to extract all category IDs
     * from the current category up to the root parent. This creates a
     * complete category path for proper categorization on Basalam.
     *
     * The method modifies the passed array by reference, adding category IDs
     * in hierarchical order (child to parent).
     *
     * @since 1.0.0
     * @param array $categories Array of category objects to process
     * @param array $categoryIds Reference to array where IDs will be stored
     * @return void Modifies $categoryIds array by reference
     */
    static function extractCategoryIds($categories, &$categoryIds)
    {
        foreach ($categories as $category) {
            // Add current category ID to the list
            $categoryIds[] = $category['cat_id'];

            // Recursively process parent categories if they exist
            if (isset($category['cat_parent']) && $category['cat_parent'] !== null) {
                self::extractCategoryIds([$category['cat_parent']], $categoryIds);
            }
        }
    }
    /**
     * Build complete category title path from hierarchical structure
     *
     * Recursively constructs a full category path string by traversing
     * from the root parent category down to the current category.
     * Creates a breadcrumb-style path separated by ' > '.
     *
     * @since 1.0.0
     * @param array|null $categoryParent Parent category object or null for root
     * @param string $currentTitle Current category title
     * @return string Complete category path (e.g., 'Electronics > Mobile > Smartphones')
     *
     * @example
     * getCombinedTitles($parent, 'iPhone')
     * // Returns: 'Electronics > Mobile Phones > Smartphones > iPhone'
     */
    static function getCombinedTitles($categoryParent, $currentTitle)
    {
        // Base case: no parent category, return current title
        if ($categoryParent === null) {
            return $currentTitle;
        }

        // Recursively build path from root to current category
        return self::getCombinedTitles($categoryParent['cat_parent'], $categoryParent['cat_title']) . ' > ' . $currentTitle;
    }
}
