<?php
defined('ABSPATH') || exit;

class SyncBasalamDiscountTaskProcessor
{
    private $discountManager;
    private $taskModel;
    private $logger;

    public function __construct()
    {
        $this->discountManager = new SyncBasalamDiscountManager();
        $this->taskModel = new SyncBasalamDiscountTaskModel();
        $this->logger = SyncBasalamLogger::getInstance();
    }

    public function processDiscountTasks($productIds = [], $variationIds = [], $discountPercent = null, $activeDays = null)
    {
        try {
            if (empty($productIds) && empty($variationIds)) {

                $firstGroup = $this->taskModel->getFirstPendingTaskGroup();

                if (!$firstGroup) {

                    return;
                }
                $this->processTaskGroup($firstGroup);
            } else {
                $this->processDirectDiscount($productIds, $variationIds, $discountPercent, $activeDays);
            }

            $this->cleanupOldTasks();
        } catch (Exception $e) {
        }
    }

    private function processDirectDiscount($productIds, $variationIds, $discountPercent, $activeDays)
    {
        try {

            $productIds = array_filter($productIds);
            $variationIds = array_filter($variationIds);

            if (empty($productIds) && empty($variationIds)) {

                return;
            }

            $result = $this->discountManager->apply(
                $discountPercent,
                $productIds,
                $variationIds,
                $activeDays
            );
            if ($result && isset($result['status_code']) && $result['status_code'] === 200) {
            } else {
                $errorMessage = 'Unknown error occurred';

                if ($result) {
                    if (isset($result['body']['message'])) {
                        $errorMessage = $result['body']['message'];
                    } elseif (isset($result['body']['error'])) {
                        $errorMessage = $result['body']['error'];
                    } elseif (isset($result['status_code'])) {
                        $errorMessage = sprintf('API returned status code: %d', $result['status_code']);
                    }
                }

                $detailedErrorLog = $errorMessage;
                if ($result && isset($result['body'])) {
                    $detailedErrorLog .= ' | Response Body: ' . json_encode($result['body']);
                }
                if ($result && isset($result['status_code'])) {
                    $detailedErrorLog .= ' | Status Code: ' . $result['status_code'];
                }
            }
        } catch (Exception $e) {
        }
    }

    private function processTaskGroup($group)
    {
        try {
            $taskIds = !empty($group->task_ids) ? explode(',', $group->task_ids) : [];
            $rawProductIds = !empty($group->product_ids)
                ? array_filter(explode(',', $group->product_ids))
                : [];

            $rawVariationIds = !empty($group->variation_ids)
                ? array_filter(explode(',', $group->variation_ids))
                : [];

            $productIds = array_filter($rawProductIds, function ($id) {
                return !empty($id) && $id !== 'NULL' && $id !== null;
            });

            $variationIds = array_filter($rawVariationIds, function ($id) {
                return !empty($id) && $id !== 'NULL' && $id !== null;
            });

            if (empty($productIds) && empty($variationIds)) {
                $this->taskModel->updateMultipleStatus(
                    $taskIds,
                    SyncBasalamDiscountTaskModel::STATUS_FAILED,
                    'No valid Basalam product or variation IDs found'
                );
                return;
            }

            $this->taskModel->updateMultipleStatus($taskIds, SyncBasalamDiscountTaskModel::STATUS_PROCESSING);

            if ($group->action === SyncBasalamDiscountTaskModel::ACTION_REMOVE) {
                $result = $this->discountManager->remove($productIds, $variationIds);
            } else {
                $result = $this->discountManager->apply(
                    $group->discount_percent,
                    $productIds,
                    $variationIds,
                    $group->active_days
                );
            }

            if ($result && isset($result['status_code']) && $result['status_code'] === 202) {

                $this->taskModel->updateMultipleStatus($taskIds, SyncBasalamDiscountTaskModel::STATUS_COMPLETED);
                $this->trackDiscountStatus($group, $productIds, $variationIds);

                $actionText = $group->action === SyncBasalamDiscountTaskModel::ACTION_REMOVE ? 'removed' : 'applied';
            } else {
                $errorMessage = 'Unknown error occurred';

                if ($result) {
                    if (isset($result['body']['message'])) {
                        $errorMessage = $result['body']['message'];
                    } elseif (isset($result['body']['error'])) {
                        $errorMessage = $result['body']['error'];
                    } elseif (isset($result['status_code'])) {
                        $errorMessage = sprintf('API returned status code: %d', $result['status_code']);
                    }
                }

                $detailedError = $errorMessage;
                if ($result && isset($result['body'])) {
                    $detailedError .= ' | Response Body: ' . json_encode($result['body']);
                }
                if ($result && isset($result['status_code'])) {
                    $detailedError .= ' | Status Code: ' . $result['status_code'];
                }

                $this->taskModel->updateMultipleStatus(
                    $taskIds,
                    SyncBasalamDiscountTaskModel::STATUS_FAILED,
                    $detailedError
                );
            }
        } catch (Exception $e) {
            $taskIds = !empty($group->task_ids) ? explode(',', $group->task_ids) : [];
            $this->taskModel->updateMultipleStatus(
                $taskIds,
                SyncBasalamDiscountTaskModel::STATUS_FAILED,
                $e->getMessage()
            );
        }
    }

    private function cleanupOldTasks()
    {
        $deletedCount = $this->taskModel->deleteOldCompletedTasks(30);
        if ($deletedCount > 0) {
        }
    }

    public static function scheduleRecurringProcessor()
    {
        $hook = 'sync_basalam_process_discount_tasks';

        if (!wp_next_scheduled($hook)) {
            wp_schedule_event(time(), 'every_minute', $hook);
        }
    }

    public function processSingleDiscountGroup()
    {
        try {

            $firstGroup = $this->taskModel->getFirstPendingTaskGroup();

            if (!$firstGroup) {

                return ['success' => true, 'message' => 'No pending tasks'];
            }

            $this->processTaskGroup($firstGroup);

            return [
                'success' => true,
                'message' => sprintf(
                    'اعمال تخفیف گروهی روی: %s%% (%d tasks)',
                    $firstGroup->discount_percent,
                    $firstGroup->count
                )
            ];
        } catch (Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    public static function unscheduleProcessor()
    {
        $hook = 'sync_basalam_process_discount_tasks';
        wp_clear_scheduled_hook($hook);
    }

    private function convertToBasalamIds($productIds = [], $variationIds = [])
    {
        $basalamProductIds = [];
        $basalamVariationIds = [];

        foreach ($productIds as $wcProductId) {
            $basalamId = get_post_meta($wcProductId, 'sync_basalam_product_id', true);
            if ($basalamId) {
                $basalamProductIds[] = $basalamId;
            }
        }

        foreach ($variationIds as $wcVariationId) {
            $basalamId = get_post_meta($wcVariationId, 'sync_basalam_variation_id', true);
            if ($basalamId) {
                $basalamVariationIds[] = $basalamId;
            }
        }

        return [
            'product_ids' => $basalamProductIds,
            'variation_ids' => $basalamVariationIds
        ];
    }

    public function processGroupedDiscounts($items)
    {
        if (empty($items)) {
            return;
        }

        $groupedItems = [];
        foreach ($items as $item) {
            $discountPercent = $item['discount_percent'] ?? 0;
            $activeDays = $item['active_days'] ?? null;

            $groupKey = $discountPercent . '_' . ($activeDays ?: 'default');

            if (!isset($groupedItems[$groupKey])) {
                $groupedItems[$groupKey] = [
                    'discount_percent' => $discountPercent,
                    'active_days' => $activeDays,
                    'product_ids' => [],
                    'variation_ids' => []
                ];
            }

            if (!empty($item['product_id'])) {
                $basalamProductId = get_post_meta($item['product_id'], 'sync_basalam_product_id', true);
                if ($basalamProductId) {
                    $groupedItems[$groupKey]['product_ids'][] = $basalamProductId;
                }
            }
            if (!empty($item['variation_id'])) {
                $basalamVariationId = get_post_meta($item['variation_id'], 'sync_basalam_variation_id', true);
                if ($basalamVariationId) {
                    $groupedItems[$groupKey]['variation_ids'][] = $basalamVariationId;
                }
            }
        }

        foreach ($groupedItems as $group) {
            $group['product_ids'] = array_unique($group['product_ids']);
            $group['variation_ids'] = array_unique($group['variation_ids']);

            $this->processDirectDiscount(
                $group['product_ids'],
                $group['variation_ids'],
                $group['discount_percent'],
                $group['active_days']
            );
        }
    }

    public function createDiscountTasks($items)
    {
        if (empty($items)) {
            return ['success' => false, 'message' => 'No items provided'];
        }

        $createdCount = 0;
        $failedCount = 0;

        foreach ($items as $item) {
            try {
                $basalamProductId = $item['basalam_product_id'] ?? null;
                $basalamVariationId = $item['basalam_variation_id'] ?? null;
                $discountPercent = $item['discount_percent'] ?? 0;
                $activeDays = $item['active_days'] ?? null;

                if (!$basalamProductId && !$basalamVariationId) {
                    $failedCount++;
                    continue;
                }

                $taskId = $this->taskModel->create([
                    'product_id' => $basalamProductId,
                    'variation_id' => $basalamVariationId,
                    'discount_percent' => $discountPercent,
                    'active_days' => $activeDays,
                    'status' => SyncBasalamDiscountTaskModel::STATUS_PENDING
                ]);

                if ($taskId) {
                    $createdCount++;
                } else {
                    $failedCount++;
                }
            } catch (Exception $e) {
                $failedCount++;
            }
        }

        $message = sprintf(
            'Created %d discount tasks successfully. %d failed.',
            $createdCount,
            $failedCount
        );

        return [
            'success' => $createdCount > 0,
            'message' => $message,
            'created' => $createdCount,
            'failed' => $failedCount
        ];
    }

    public function createDiscountTasksFromWcIds($items)
    {
        $basalamItems = [];

        foreach ($items as $item) {
            $wcProductId = $item['product_id'] ?? null;
            $wcVariationId = $item['variation_id'] ?? null;

            $basalamProductId = null;
            $basalamVariationId = null;

            if ($wcProductId) {
                $basalamProductId = get_post_meta($wcProductId, 'sync_basalam_product_id', true);
            }
            if ($wcVariationId) {
                $basalamVariationId = get_post_meta($wcVariationId, 'sync_basalam_variation_id', true);
            }

            if ($basalamProductId || $basalamVariationId) {
                $basalamItems[] = [
                    'basalam_product_id' => $basalamProductId,
                    'basalam_variation_id' => $basalamVariationId,
                    'discount_percent' => $item['discount_percent'] ?? 0,
                    'active_days' => $item['active_days'] ?? null
                ];
            }
        }

        return $this->createDiscountTasks($basalamItems);
    }

    public function applyDiscountWithWcIds($wcProductIds, $wcVariationIds, $discountPercent, $activeDays = null)
    {
        $converted = $this->convertToBasalamIds($wcProductIds, $wcVariationIds);

        if (empty($converted['product_ids']) && empty($converted['variation_ids'])) {
            return [
                'success' => false,
                'message' => 'No valid Basalam IDs found for provided WooCommerce IDs'
            ];
        }

        $this->processDirectDiscount(
            $converted['product_ids'],
            $converted['variation_ids'],
            $discountPercent,
            $activeDays
        );

        return [
            'success' => true,
            'message' => sprintf(
                'Processing discount for %d products and %d variations',
                count($converted['product_ids']),
                count($converted['variation_ids'])
            )
        ];
    }

    private function trackDiscountStatus($group, $basalamProductIds, $basalamVariationIds)
    {
        try {
            $action = $group->action;
            $timestamp = current_time('mysql');

            foreach ($basalamProductIds as $basalamProductId) {
                $wcProductId = $this->getWcProductIdByBasalamId($basalamProductId);
                if ($wcProductId) {
                    if ($action === SyncBasalamDiscountTaskModel::ACTION_APPLY) {
                        update_post_meta($wcProductId, '_sync_basalam_discount_applied', $timestamp);
                        update_post_meta($wcProductId, '_sync_basalam_discount_percent', $group->discount_percent);
                    } else {
                        delete_post_meta($wcProductId, '_sync_basalam_discount_applied');
                        delete_post_meta($wcProductId, '_sync_basalam_discount_percent');
                    }
                }
            }

            foreach ($basalamVariationIds as $basalamVariationId) {
                $wcVariationId = $this->getWcVariationIdByBasalamId($basalamVariationId);
                if ($wcVariationId) {
                    if ($action === SyncBasalamDiscountTaskModel::ACTION_APPLY) {
                        update_post_meta($wcVariationId, '_sync_basalam_discount_applied', $timestamp);
                        update_post_meta($wcVariationId, '_sync_basalam_discount_percent', $group->discount_percent);
                    } else {
                        delete_post_meta($wcVariationId, '_sync_basalam_discount_applied');
                        delete_post_meta($wcVariationId, '_sync_basalam_discount_percent');
                    }
                }
            }
        } catch (Exception $e) {
        }
    }

    private function getWcProductIdByBasalamId($basalamProductId)
    {
        global $wpdb;

        $result = $wpdb->get_var($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta}
             WHERE meta_key = 'sync_basalam_product_id'
             AND meta_value = %s
             LIMIT 1",
            $basalamProductId
        ));

        return $result ? (int)$result : null;
    }

    private function getWcVariationIdByBasalamId($basalamVariationId)
    {
        global $wpdb;

        $result = $wpdb->get_var($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta}
             WHERE meta_key = 'sync_basalam_variation_id'
             AND meta_value = %s
             LIMIT 1",
            $basalamVariationId
        ));

        return $result ? (int)$result : null;
    }

    public function createRemoveDiscountTask($wcProductId, $wcVariationId = null)
    {
        try {
            $basalamProductId = null;
            $basalamVariationId = null;

            if ($wcProductId) {
                $basalamProductId = get_post_meta($wcProductId, 'sync_basalam_product_id', true);
            }
            if ($wcVariationId) {
                $basalamVariationId = get_post_meta($wcVariationId, 'sync_basalam_variation_id', true);
            }

            if (!$basalamProductId && !$basalamVariationId) {
                return ['success' => false, 'message' => 'No Basalam IDs found'];
            }

            $taskId = $this->taskModel->create([
                'product_id' => $basalamProductId,
                'variation_id' => $basalamVariationId,
                'discount_percent' => 0,
                'active_days' => 0,
                'action' => SyncBasalamDiscountTaskModel::ACTION_REMOVE,
                'status' => SyncBasalamDiscountTaskModel::STATUS_PENDING
            ]);

            if ($taskId) {
                return ['success' => true, 'message' => 'Remove discount task created', 'task_id' => $taskId];
            } else {
                return ['success' => false, 'message' => 'Failed to create remove discount task'];
            }
        } catch (Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
}
