<?php
if (! defined('ABSPATH')) exit;

class SyncBasalamApiServiceManager
{
    private array $headers;

    public function __construct()
    {
        $this->headers = [
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
            'user-agent' => 'Wp-Basalam',
            'referer' => get_site_url(),
        ];
    }

    public function sendPostRequest($url, $data, $headers = [])
    {
        $headers = array_merge($this->headers, $headers);

        $response = wp_remote_post($url, array(
            'body'    => $data,
            'headers' => $headers,
        ));

        if (is_wp_error($response)) {
            $errorCode = $response->get_error_code();
            $errorMessage = $response->get_error_message();

            if ($errorCode === 'timeout') {
                SyncBasalamLogger::error("ارسال درخواست به آدرس $url به دلیل timeout ناموفق بود ، این خطا به علت اختلالات api های باسلام یا هاست سایت شماست.");
            } else {
                SyncBasalamLogger::error("درخواست API برای آدرس $url با خطا مواجه شد. پاسخ: $errorMessage");
            }

            return [
                'body' => null,
                'status_code' => 500
            ];
        }

        if (is_wp_error($response)) {
            SyncBasalamLogger::error("درخواست API برای آدرس " . $url . " با خطا مواجه شد. پاسخ: " . $response->get_error_message());
            return [
                'body' => null,
                'status_code' => 500
            ];
        }

        $body = wp_remote_retrieve_body($response);
        $statusCode = wp_remote_retrieve_response_code($response);

        if ($statusCode == 401) {
            $data = [
                SyncBasalamAdminSettings::TOKEN => '',
                SyncBasalamAdminSettings::REFRESH_TOKEN => '',
            ];
            SyncBasalamAdminSettings::updateSettings($data);

            SyncBasalamQueueManager::cancelAllTasksGroup('sync_basalam_plugin_create_product');
            SyncBasalamQueueManager::cancelAllTasksGroup('sync_basalam_plugin_update_product');
            SyncBasalamQueueManager::cancelAllTasksGroup('sync_basalam_plugin_connect_auto_product');
        }

        return [
            'body' => json_decode($body, true),
            'status_code' => $statusCode
        ];
    }

    public function sendGetRequest($url, $headers = [], $maxRetries = 3)
    {
        $headers = array_merge($this->headers, $headers);
        $attempt = 0;
        $response = null;

        while ($attempt < $maxRetries) {
            $response = wp_remote_get($url, array(
                'timeout'   => 30,
                'headers'   => $headers,
            ));

            if (!is_wp_error($response)) {
                break;
            }

            $errorCode = $response->get_error_code();
            $errorMessage = $response->get_error_message();

            if (
                $errorCode === 'http_request_failed' &&
                (strpos($errorMessage, 'cURL error 28') !== false ||
                    strpos($errorMessage, 'Operation timed out') !== false ||
                    strpos($errorMessage, 'Connection timed out') !== false)
            ) {
                $attempt++;
                if ($attempt < $maxRetries) {
                    SyncBasalamLogger::debug("تلاش $attempt از $maxRetries برای درخواست به $url به دلیل timeout ناموفق بود. در حال تلاش مجدد...");
                    sleep(2 * $attempt);
                } else {
                    SyncBasalamLogger::error("درخواست به $url پس از $maxRetries تلاش به دلیل timeout ناموفق بود.");
                    return [
                        'data' => null,
                        'status_code' => 500,
                        'timeout_error' => true
                    ];
                }
            } else {
                SyncBasalamLogger::error("درخواست API برای آدرس " . $url . " با خطا مواجه شد. پاسخ: " . $errorMessage);
                return false;
            }
        }

        $body = wp_remote_retrieve_body($response);
        $statusCode = wp_remote_retrieve_response_code($response);

        if ($statusCode == 401) {
            $data = [
                SyncBasalamAdminSettings::TOKEN => '',
                SyncBasalamAdminSettings::REFRESH_TOKEN => '',
            ];
            SyncBasalamAdminSettings::updateSettings($data);

            SyncBasalamQueueManager::cancelAllTasksGroup('sync_basalam_plugin_create_product');
            SyncBasalamQueueManager::cancelAllTasksGroup('sync_basalam_plugin_update_product');
            SyncBasalamQueueManager::cancelAllTasksGroup('sync_basalam_plugin_connect_auto_product');
        }

        return [
            'data' => json_decode($body, true),
            'status_code' => $statusCode,
        ];
    }

    public function sendPatchRequest($url, $data, $headers = [])
    {
        $headers = array_merge($this->headers, $headers);

        $response = wp_remote_request($url, array(
            'method' => 'PATCH',
            'body'      => $data,
            'headers'   => $headers,
        ));

        if (is_wp_error($response)) {
            SyncBasalamLogger::error("درخواست API برای آدرس " . $url . " با خطا مواجه شد. پاسخ: " . $response->get_error_message());
            return [
                'body' => null,
                'status_code' => 500
            ];
        }

        $body = wp_remote_retrieve_body($response);
        $statusCode = wp_remote_retrieve_response_code($response);

        if ($statusCode == 401) {
            $data = [
                SyncBasalamAdminSettings::TOKEN => '',
                SyncBasalamAdminSettings::REFRESH_TOKEN => '',
            ];
            SyncBasalamAdminSettings::updateSettings($data);

            SyncBasalamQueueManager::cancelAllTasksGroup('sync_basalam_plugin_create_product');
            SyncBasalamQueueManager::cancelAllTasksGroup('sync_basalam_plugin_update_product');
            SyncBasalamQueueManager::cancelAllTasksGroup('sync_basalam_plugin_connect_auto_product');
        }

        return [
            'body' => json_decode($body, true),
            'status_code' => $statusCode
        ];
    }

    public function sendPutRequest($url, $data, $headers = [])
    {
        $headers = array_merge($this->headers, $headers);
        $response = wp_remote_request($url, array(
            'method' => 'PUT',
            'body'      => $data,
            'headers'   => $headers,
        ));

        if (is_wp_error($response)) {
            SyncBasalamLogger::error("درخواست API برای آدرس " . $url . " با خطا مواجه شد. پاسخ: " . $response->get_error_message());
            return [
                'body' => null,
                'status_code' => 500
            ];
        }

        $body = wp_remote_retrieve_body($response);
        $statusCode = wp_remote_retrieve_response_code($response);
        if ($statusCode == 401) {
            $data = [
                SyncBasalamAdminSettings::TOKEN => '',
                SyncBasalamAdminSettings::REFRESH_TOKEN => '',
            ];
            SyncBasalamAdminSettings::updateSettings($data);
            SyncBasalamQueueManager::cancelAllTasksGroup('sync_basalam_plugin_create_product');
            SyncBasalamQueueManager::cancelAllTasksGroup('sync_basalam_plugin_update_product');
            SyncBasalamQueueManager::cancelAllTasksGroup('sync_basalam_plugin_connect_auto_product');
        }
        return [
            'body' => json_decode($body, true),
            'status_code' => $statusCode
        ];
    }

    public function sendDeleteRequest($url, $headers = [], $data = null)
    {
        $headers = array_merge($this->headers, $headers);
        $response = wp_remote_request($url, array(
            'method' => 'DELETE',
            'headers'   => $headers,
            'body'      => $data,
        ));

        if (is_wp_error($response)) {
            SyncBasalamLogger::error("درخواست API برای آدرس " . $url . " با خطا مواجه شد. پاسخ: " . $response->get_error_message());
            return [
                'body' => null,
                'status_code' => 500
            ];
        }

        $body = wp_remote_retrieve_body($response);
        $statusCode = wp_remote_retrieve_response_code($response);
        if ($statusCode == 401) {
            $data = [
                SyncBasalamAdminSettings::TOKEN => '',
                SyncBasalamAdminSettings::REFRESH_TOKEN => '',
            ];
            SyncBasalamAdminSettings::updateSettings($data);
            SyncBasalamQueueManager::cancelAllTasksGroup('sync_basalam_plugin_create_product');
            SyncBasalamQueueManager::cancelAllTasksGroup('sync_basalam_plugin_update_product');
            SyncBasalamQueueManager::cancelAllTasksGroup('sync_basalam_plugin_connect_auto_product');
        }
        return [
            'body' => json_decode($body, true),
            'status_code' => $statusCode
        ];
    }
    public function uploadFileRequest($url, $localFile, $data = [], $headers = [])
    {
        if (!file_exists($localFile)) {
            SyncBasalamLogger::error("فایل وجود ندارد: " . $localFile);
            return false;
        } elseif (!is_readable($localFile)) {
            SyncBasalamLogger::error("فایل قابل خواندن نیست: " . $localFile);
            return false;
        } else {
            $fileContent = file_get_contents($localFile);
            if ($fileContent === false) {
                SyncBasalamLogger::error("file_get_contents شکست خورد: " . $localFile);
                return false;
            }
        }

        $boundary = wp_generate_password(24);

        $payload = $this->makePayloadUploadFileRequest($data, $localFile, $boundary);

        $headers = array_merge(
            $headers,
            array(
                'content-type' => 'multipart/form-data; boundary=' . $boundary,
                'user-agent' => 'Wp-Basalam',
                'Accept' => 'application/json',
                'referer' => get_site_url(),
            )
        );

        $response = wp_remote_post(
            $url,
            array(
                'headers'    => $headers,
                'body'       => $payload,
            )
        );

        if (is_wp_error($response)) {
            SyncBasalamLogger::error("درخواست API برای آدرس " . $url  . " با خطا مواجه شد. پاسخ: " . $response->get_error_message());
            return false;
        }

        $body = wp_remote_retrieve_body($response);
        $statusCode = wp_remote_retrieve_response_code($response);

        if ($statusCode == 401) {
            $data = [
                SyncBasalamAdminSettings::TOKEN => '',
                SyncBasalamAdminSettings::REFRESH_TOKEN => '',
            ];
            SyncBasalamAdminSettings::updateSettings($data);

            SyncBasalamQueueManager::cancelAllTasksGroup('sync_basalam_plugin_create_product');
            SyncBasalamQueueManager::cancelAllTasksGroup('sync_basalam_plugin_update_product');
            SyncBasalamQueueManager::cancelAllTasksGroup('sync_basalam_plugin_connect_auto_product');
        }

        return [
            'body' => json_decode($body, true),
            'status_code' => $statusCode
        ];
    }

    private function makePayloadUploadFileRequest($data, $localFile, $boundary)
    {
        $payload = '';
        $eol = "\r\n";

        foreach ($data as $name => $value) {
            $payload .= '--' . $boundary . $eol;
            $payload .= 'Content-Disposition: form-data; name="' . $name . '"' . $eol . $eol;
            $payload .= $value . $eol;
        }

        if ($localFile) {
            $filename = basename($localFile);
            $fileContent = file_get_contents($localFile);

            $filetypeInfo = wp_check_filetype($localFile);
            $mimeType = $filetypeInfo['type'] ?? 'application/octet-stream';

            $payload .= '--' . $boundary . $eol;
            $payload .= 'Content-Disposition: form-data; name="file"; filename="' . $filename . '"' . $eol;
            $payload .= 'Content-Type: ' . $mimeType . $eol;
            $payload .= 'Content-Transfer-Encoding: binary' . $eol . $eol;
            $payload .= $fileContent . $eol;
        }

        $payload .= '--' . $boundary . '--' . $eol;

        return $payload;
    }
}
