<?php
if (! defined('ABSPATH')) exit;

class SyncBasalamWooLogger implements SyncBasalamLoggerInterface
{
    public static function getLogs()
    {
        if (!current_user_can('manage_options')) {
            return [
                'error' => 'شما دسترسی لازم برای مشاهده لاگ‌ها را ندارید.'
            ];
        }

        $uploadDir = wp_upload_dir();
        $logDir = trailingslashit($uploadDir['basedir']) . 'wc-logs/';

        $logFiles = glob($logDir . 'basalam-sync-plugin*.log');
        if (empty($logFiles)) {
            return [
                'error' => 'هیچ لاگی یافت نشد.'
            ];
        }

        rsort($logFiles);
        $allLogs = [];

        foreach ($logFiles as $file) {
            $lines = file($file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            foreach (array_reverse($lines) as $line) {
                // Match ISO 8601 date format more strictly: YYYY-MM-DDTHH:MM:SS+TZ
                if (preg_match('/^(\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}[+-]\d{2}:\d{2}) (INFO|WARNING|ERROR|DEBUG|ALERT) (.*?)( CONTEXT: (.*))?$/', $line, $matches)) {
                    $tehranDatetime = SyncBasalamDateConverter::utcToTehran($matches[1]);

                    // Skip this log entry if date conversion failed
                    if ($tehranDatetime === null) {
                        continue;
                    }

                    $jalaliDate = SyncBasalamDateConverter::gregorianToJalali(
                        $tehranDatetime->format('Y'),
                        $tehranDatetime->format('m'),
                        $tehranDatetime->format('d')
                    ) . ' - ' . $tehranDatetime->format('H:i:s');

                    $context = isset($matches[5]) ? json_decode($matches[5], true) : null;

                    $allLogs[] = [
                        'date' => $jalaliDate,
                        'level' => $matches[2],
                        'message' => $matches[3],
                        'context' => $context,
                    ];
                }
            }
        }

        $logsByLevel = ['info' => [], 'warning' => [], 'error' => [], 'debug' => [], 'alert' => []];
        foreach ($allLogs as $log) {
            $logsByLevel[strtolower($log['level'])][] = $log;
        }

        return [
            'logs_by_level' => $logsByLevel,
            'current_tab' => sanitize_text_field(isset($_GET['tab'])) ? sanitize_text_field(wp_unslash($_GET['tab'])) : 'info',
            'current_page' => sanitize_text_field(isset($_GET['paged'])) ? max(1, intval(sanitize_text_field(wp_unslash($_GET['paged'])))) : 1,
            'per_page' => 10
        ];
    }

    public function log($level, $message, $context = [])
    {
        $logger = wc_get_logger();
        $context = array_merge(is_array($context) ? $context : [], ['source' => 'basalam-sync-plugin']);
        $logger->log($level, $message, $context);
    }

    public function info($message, $context)
    {
        $this->log("info", $message, $context);
    }

    public function debug($message, $context)
    {
        $this->log("debug", $message, $context);
    }

    public function warning($message, $context)
    {
        $this->log("warning", $message, $context);
    }

    public function error($message, $context)
    {
        $this->log("error", $message, $context);
    }

    public function alert($message, $context)
    {
        $this->log("alert", $message, $context);
    }
}
