<?php
if (! defined('ABSPATH')) exit;

class SyncBasalamGetProductData
{
    private $currency;

    public function __construct()
    {
        $this->currency = get_woocommerce_currency();
    }

    public function getProductData($product_id, $is_update, $category_ids)
    {
        $product = wc_get_product($product_id);
        $product_data = [];
        $description = $this->getDescription($product);

        if (!$category_ids) {
            $mapped_category_ids = $this->getMappedCategory($product_id);

            if ($mapped_category_ids) {
                $category_ids = $mapped_category_ids;
            } else {
                $product_title = mb_substr($product->get_name(), 0, 120);
                $category_ids = SyncBasalamGetCategoryId::getCategoryIdFromBasalam(urlencode($product_title), 'multi');
            }
        }
        $category_id = isset($category_ids[0]) ? $category_ids[0] : null;

        if (!$category_id) {
            throw new \Exception('دسته بندی محصول یافت نشد.');
        }

        $stock_quantity = $this->getStockQuantity($product);

        $product_name = $this->getProductName($product);
        if ($is_update) {
            $sync_fields = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELDS);
            if ($sync_fields == 'price_stock') {
                $product_data['status'] = 2976;
                $product_data['stock'] = $stock_quantity;
                if (!$product->is_type('variable')) {
                    $price = $this->calculateFinalPrice($this->getPrice($product), $category_ids);
                    if (!$price) {
                        throw new \Exception(' قیمت محصول ' . esc_html($product->get_name()) . ' ' . 'کمتر از 1000 تومان است.');
                    }
                    $product_data['primary_price'] = intval($price);
                }
                if ($product->is_type('variable')) {
                    $variants = $this->getVariants($product, $category_ids);
                    if (!$variants) {
                        throw new \Exception('دریافت متغیر ها با مشکل مواجه شد.');
                    }
                    $product_data['variants'] = $variants;
                } else {
                    $product_data['variants'] = [];
                }
                return $product_data;
            } elseif ($sync_fields == 'custom') {
                $sync_field_name = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELD_NAME);

                if ($sync_field_name == true) {
                    $product_data['name'] = $product_name;
                }
                $sync_field_photos = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELD_PHOTOS);
                if ($sync_field_photos == true) {
                    $photos = $this->getProductPhotos($product);
                    $product_data['photo'] = $photos['main'];
                    $product_data['photos'] = $photos['gallery'];
                }

                $sync_field_price = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELD_PRICE);
                if ($sync_field_price == true) {
                    if (!$product->is_type('variable')) {
                        $price = $this->calculateFinalPrice($this->getPrice($product), $category_ids);
                        if (!$price) {
                            throw new \Exception(' قیمت محصول ' . esc_html($product->get_name()) . ' ' . 'کمتر از 1000 تومان است.');
                        }
                        $product_data['primary_price'] = intval($price);
                    }
                    if ($product->is_type('variable')) {
                        $variants = $this->getVariants($product, $category_ids);
                        if ($variants) {
                            $product_data['variants'] = $variants;
                        }
                    } else {
                        $product_data['variants'] = [];
                    }
                }
                $sync_field_stock = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELD_STOCK);
                if ($sync_field_stock == true) {
                    if (!$product->is_type('variable')) {
                        $product_data['stock'] = $stock_quantity;
                    }
                    if ($product->is_type('variable')) {
                        $variants = $this->getVariants($product, $category_ids);
                        if ($variants) {
                            $product_data['variants'] = $variants;
                        }
                    } else {
                        $product_data['variants'] = [];
                    }
                }
                $sync_field_weight = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELD_WEIGHT);
                if ($sync_field_weight == true) {
                    $product_data['weight'] = $this->getWeight($product);
                    $product_data['package_weight'] = $this->getPackageWeight($product_data['weight']);
                }
                $sync_field_description = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELD_DESCRIPTION);
                if ($sync_field_description  === "1") {
                    $product_data['description'] = $description;
                }
                $sync_field_attr = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELD_ATTR);
                if ($sync_field_attr == true) {
                    $mobile_attrs = [];
                    $woo_attrs = [];
                    $woo_attrs = $this->mapAttributesToBasalam($product, $category_id);

                    if ($this->isMobileProduct($product)) {
                        $mobile_attrs = $this->getMobileAttributes($product);
                    }
                    if (!empty($woo_attrs) || !empty($mobile_attrs)) {
                        if (!empty($woo_attrs) && !empty($mobile_attrs)) {
                            $attrs = array_merge($woo_attrs, $mobile_attrs);
                        } elseif ($woo_attrs && !$mobile_attrs) {
                            $attrs = $woo_attrs;
                        } elseif (!$woo_attrs && $mobile_attrs) {
                            $attrs = $mobile_attrs;
                        }
                        $product_data['product_attribute'] = $attrs;
                    }
                }
                return $product_data;
            }
        }

        $photos = $this->getProductPhotos($product);
        if (!$photos) {
            throw new \Exception('دریافت تصاویر محصول با خطا مواجه شد.');
        }

        $photo = $photos['main'];
        $gallery_photos = $photos['gallery'];
        $product_data['name'] = $product_name;
        $product_data['photo'] = $photo;
        $product_data['photos'] = $gallery_photos;
        $product_data['status'] = 2976;
        $product_data['stock'] = $stock_quantity;

        if (!$is_update) {
            $product_data['category_id'] = $category_id;
        }
        $product_data['description'] = $description;
        $product_data['weight'] = $this->getWeight($product);
        $product_data['package_weight'] = $this->getPackageWeight($product_data['weight']);
        $product_data['preparation_days'] = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::DEFAULT_PREPARATION);
        if (!$product->is_type('variable')) {
            $price = $this->calculateFinalPrice($this->getPrice($product), $category_ids);
            if (!$price) {
                throw new \Exception(' قیمت محصول ' . esc_html($product->get_name()) . ' ' . 'کمتر از 1000 تومان است.');
            }
            $product_data['primary_price'] = intval($price);
        }
        if ($product->is_type('variable')) {
            $variants = $this->getVariants($product, $category_ids);
            $product_data['variants'] = $variants;
        } else {
            $product_data['variants'] = [];
        }

        if ($this->isMobileProduct($product)) {
            $product_data['product_attribute'] = $this->getMobileAttributes($product);
        }
        if ($this->isProductType($product)) {
            $unit_type = get_post_meta($product->get_id(), '_sync_basalam_product_unit', true);

            if ($unit_type != 'none' && is_numeric($unit_type)) {
                $product_data['unit_type'] = $unit_type;
                $product_data['unit_quantity'] = get_post_meta($product->get_id(), '_sync_basalam_product_value', true);
                if (!is_numeric($product_data['unit_quantity'])) {
                    $product_data['unit_quantity'] = 1;
                }
            } else {
                $product_data['unit_type'] = 6304;
                $product_data['unit_quantity'] = 1;
            }
        } else {
            $product_data['unit_type'] = 6304;
            $product_data['unit_quantity'] = 1;
        }
        if (SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::ALL_PRODUCTS_WHOLESALE) == 'all') {
            $product_data['is_wholesale'] = true;
        } else {
            if ($this->isWholesale($product)) {
                $product_data['is_wholesale'] = true;
            } else {
                $product_data['is_wholesale'] = false;
            }
        }
        return $product_data;
    }

    private function getAttributeSuffixFromProduct($product)
    {
        $is_enabled = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::PRODUCT_ATTRIBUTE_SUFFIX_ENABLED);

        if ($is_enabled !== 'yes') {
            return null;
        }

        $attribute_name = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::PRODUCT_ATTRIBUTE_SUFFIX_PRIORITY);

        if (empty($attribute_name)) {
            return null;
        }

        $attribute_name = trim($attribute_name);

        $product_attributes = $this->getWooAttributes($product);

        if (empty($product_attributes)) {
            return null;
        }

        foreach ($product_attributes as $product_attr) {
            if (trim($product_attr['title']) === $attribute_name && !empty($product_attr['value'])) {
                return $product_attr['value'];
            }
        }

        return null;
    }

    private function getProductName($product)
    {
        $product_name = $product->get_name();

        $prefix_title = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::PRODUCT_PREFIX_TITLE);
        $suffix_title = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::PRODUCT_SUFFIX_TITLE);

        if ($prefix_title) {
            $product_name = "{$prefix_title} {$product_name}";
        }

        $attribute_suffix = $this->getAttributeSuffixFromProduct($product);
        if ($attribute_suffix) {
            $product_name = "{$product_name} ({$attribute_suffix})";
        }

        if ($suffix_title) {
            $product_name = "{$product_name} {$suffix_title}";
        }
        $word_count = count(explode(' ', trim($product_name)));

        if (mb_strlen($product_name) < 6 || $word_count < 2) {
            $product_name .= " ویژه";
        }

        $product_name = mb_substr($product_name, 0, 120);
        return $product_name;
    }

    private function getProductPhotos($product)
    {
        $photoFiles = [];
        $photos = [];
        $photoFiles = [];

        $photoFiles[] = [
            'id'   => $product->get_image_id(),
            'file_path' => $this->getPathOrUrl($product->get_image_id())
        ];

        foreach ($product->get_gallery_image_ids() as $photo_id) {
            $photoFiles[] = [
                'id'   => $photo_id,
                'file_path' => $this->getPathOrUrl($photo_id)
            ];
        }
        if (count($photoFiles) > 10) {
            $photoFiles = array_slice($photoFiles, 0, 10);
        }

        foreach ($photoFiles as $photoFile) {
            if ($photoFile) {
                $check_exist = $this->getExistingPhotoFromDb($photoFile['id']);

                if (!$check_exist) {
                    $uploaded = SyncBasalamFileUploader::upload($photoFile['file_path']);

                    if ($uploaded) {
                        $photos[] = $uploaded;
                        $this->storePhotoInDb($photoFile['id'], $uploaded);
                    }
                } else {
                    $photos[] = $check_exist;
                }
            }
        }

        $main_photo_id = null;
        $gallery_photo_ids = [];
        foreach ($photos as $photo) {
            if (is_null($main_photo_id)) {
                $main_photo_id = $photo['file_id'];
            } else {
                $gallery_photo_ids[] = $photo['file_id'];
            }
        }
        return [
            'main' => $main_photo_id,
            'gallery' => $gallery_photo_ids
        ];
    }

    private function storePhotoInDb($woo_photo_id, $sync_basalam_photo)
    {
        global $wpdb;
        $table_name_uploaded_photo = $wpdb->prefix . 'sync_basalam_uploaded_photo';

        $wpdb->insert(
            $table_name_uploaded_photo,
            array(
                'woo_photo_id'              => (int) $woo_photo_id,
                'sync_basalam_photo_id'     => (int) $sync_basalam_photo['file_id'],
                'sync_basalam_photo_url'    => $sync_basalam_photo['url'],
                'created_at'                => current_time('mysql')
            ),
            array('%d', '%d', '%s', '%s')
        );
    }

    private function getPathOrUrl($attachment_id)
    {
        $url = wp_get_attachment_url($attachment_id);
        $path = get_attached_file($attachment_id);

        if ($path && file_exists($path)) {
            return $path;
        }

        return $url;
    }
    private function getExistingPhotoFromDb($woo_photo_id)
    {
        global $wpdb;
        $table_name_uploaded_photo = $wpdb->prefix . 'sync_basalam_uploaded_photo';

        $result = (array) $wpdb->get_row(
            $wpdb->prepare(
                "SELECT sync_basalam_photo_id AS file_id, sync_basalam_photo_url AS url, created_at
                FROM $table_name_uploaded_photo
                WHERE woo_photo_id = %d",
                $woo_photo_id
            )
        );

        if ($result) {
            $created_at_ts = strtotime($result['created_at']);
            $now_wp = current_time('timestamp');
            $fourteen_days = 14 * DAY_IN_SECONDS;

            if (($now_wp - $created_at_ts) >= $fourteen_days) {
                $wpdb->delete(
                    $table_name_uploaded_photo,
                    array('woo_photo_id' => $woo_photo_id),
                    array('%d')
                );
                return false;
            }
        }

        return $result ?: null;
    }

    private function getPrice($product)
    {
        $price_field = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::PRODUCT_PRICE_FIELD);

        $regular_price = $product->get_regular_price();
        $sale_price = $product->get_sale_price();

        $regular_price = !empty($regular_price) && is_numeric($regular_price) ? $regular_price : null;
        $sale_price = !empty($sale_price) && is_numeric($sale_price) ? $sale_price : null;

        if ($price_field == 'original_price' || $price_field == 'sale_strikethrough_price') {
            if ($regular_price !== null && $this->calculateFinalPrice($regular_price)) {
                return $regular_price;
            }
        } elseif ($price_field == 'sale_price') {
            if ($sale_price) {
                if ($this->calculateFinalPrice($sale_price)) {
                    return $sale_price;
                }
            }
            if ($this->calculateFinalPrice($regular_price)) {
                return $regular_price;
            }
            return null;
        }
    }

    private function calculateFinalPrice($price, $category_ids = null)
    {
        $increase_value = intval(SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::INCREASE_PRICE_VALUE));
        $round_mode = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::ROUND_PRICE);
        $currency = $this->currency;
        $converted_price = $this->convertPriceToRial($price, $currency);

        if (empty($converted_price) || !is_numeric($converted_price)) {
            return false;
        }

        $converted_price = floatval($converted_price);

        if ($increase_value == -1) {
            SyncBasalamLogger::debug('Fetching category commission for categories: ' . json_encode($category_ids, JSON_UNESCAPED_UNICODE));
            $category_data = SyncBasalamFetchCommission::fetchCategoryCommission($category_ids);

            if (is_array($category_data)) {
                list($category_percent, $max_amount) = $category_data;
            } else {
                $category_percent = $category_data;
                $max_amount = null;
            }

            $category_percent = floatval($category_percent);

            if ($category_percent > 0 && $category_percent < 100) {
                $finalPrice = $converted_price / (1 - ($category_percent / 100));
            } else {
                $finalPrice = $converted_price;
            }
            if ($max_amount !== null) {
                $max_amount = floatval($max_amount);
                $actual_commission = $finalPrice - $converted_price;

                if ($actual_commission > $max_amount) {
                    $finalPrice = $converted_price + $max_amount;
                }
            }
        } elseif ($increase_value <= 100) {
            $finalPrice = $converted_price + ($converted_price * ($increase_value / 100));
        } else {
            $finalPrice = $converted_price + ($increase_value * 10);
        }

        if ($round_mode === 'up') {
            return ceil($finalPrice / 10000) * 10000;
        } elseif ($round_mode === 'down') {
            return floor($finalPrice / 10000) * 10000;
        } else {
            return floor($finalPrice / 10) * 10;
        }
    }

    private function convertPriceToRial($price, $currency)
    {

        if (empty($price) || !is_numeric($price)) {
            return 0;
        }

        $price = floatval($price);

        if ($currency == 'IRT') {
            return $price * 10;
        } elseif ($currency == 'IRHT') {
            return $price * 10000;
        } elseif ($currency == 'IRHR') {
            return $price * 1000;
        }
        return $price;
    }

    function getWooAttributes($product)
    {
        $attributes = [];
        $product_attrs = $product->get_attributes();

        if (is_array($product_attrs) & !empty($product_attrs)) {
            foreach ($product_attrs as $attribute) {
                if ($attribute->is_taxonomy()) {
                    $taxonomy = $attribute->get_name();
                    $label = wc_attribute_label($taxonomy);

                    $terms = wc_get_product_terms($product->get_id(), $taxonomy, ['fields' => 'names']);
                    $value = implode(', ', $terms);

                    $attributes[] = [
                        'title' => $label,
                        'value' => $value,
                    ];
                } else {
                    $label = wc_attribute_label($attribute->get_name());
                    $value = $attribute->get_options();
                    $value = implode(', ', $value);

                    $attributes[] = [
                        'title' => $label,
                        'value' => $value,
                    ];
                }
            }
        }

        return $attributes;
    }

    function mapAttributesToBasalam($product, $category_id)
    {
        $woo_attrs = $this->getWooAttributes($product);
        $mapped_attrs = $this->getMappedCategoryOptions();
        if (!$woo_attrs) {
            return null;
        }
        if ($mapped_attrs) {
            foreach ($woo_attrs as &$woo_attr) {
                foreach ($mapped_attrs as $mapped_attr) {
                    if (trim($mapped_attr['woo_name']) == trim($woo_attr['title'])) {
                        $woo_attr['title'] = $mapped_attr['sync_basalam_name'];
                        break;
                    }
                }
            }
            unset($woo_attr);
        }
        $response = SyncBasalamGetCategoryAttr::getAttr($category_id);
        $sync_basalam_attrs = [];
        foreach ($response['data'] as $group) {
            foreach ($group['attributes'] as $attr) {
                $sync_basalam_attrs[] = [
                    'id' => $attr['id'],
                    'title' => $attr['title']
                ];
            }
        }
        $matched_attrs = [];

        foreach ($woo_attrs as $woo_attr) {
            foreach ($sync_basalam_attrs as $sync_basalam_attr) {
                if (trim($woo_attr['title']) === trim($sync_basalam_attr['title'])) {
                    $matched_attrs[] = [
                        'attribute_id' => $sync_basalam_attr['id'],
                        'value' => $woo_attr['value'],
                    ];
                    break;
                }
            }
        }

        return ($matched_attrs);
    }

    function getMappedCategoryOptions()
    {
        global $wpdb;
        $categoryOptionsManager = new SyncBasalamCategoryOptions($wpdb);
        $category_map_options = $categoryOptionsManager->getAll();
        return $category_map_options;
    }

    private function getStockQuantity($product)
    {
        $default_stock_quantity = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::DEFAULT_STOCK_QUANTITY);
        $stock_quantity = $product->get_stock_quantity();
        $stock_status =  $product->get_stock_status();
        if ($stock_status == 'instock') {
            return $stock_quantity === null ? $default_stock_quantity : $stock_quantity;
        } else {
            return 0;
        }
    }

    private function getWeight($product)
    {
        if (empty($product->get_weight())) {
            return SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::DEFAULT_WEIGHT);
        }
        $weight = $product->get_weight();
        $weight = str_replace(',', '.', $weight);
        $weight_unit = get_option('woocommerce_weight_unit');
        return ($weight_unit === 'kg') ? floatval($weight) * 1000 : floatval($weight);
    }

    private function getPackageWeight($weight)
    {
        $package_weight =  SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::DEFAULT_PACKAGE_WEIGHT);
        return intval($weight + $package_weight);
    }

    private function getVariants($product, $category_ids)
    {
        $variants = [];
        $variation_ids = $product->get_children();

        $price_field = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::PRODUCT_PRICE_FIELD);

        foreach ($variation_ids as $variation_id) {
            $variant_product = wc_get_product($variation_id);
            if (!$variant_product) {
                continue;
            }

            $regular_price = $variant_product->get_regular_price();
            $sale_price    = $variant_product->get_sale_price();

            $regular_price = !empty($regular_price) && is_numeric($regular_price) && $regular_price > 1000 ? $regular_price : null;
            $sale_price = !empty($sale_price) && is_numeric($sale_price) && $sale_price > 1000 ? $sale_price : null;

            if ($price_field === 'original_price' || $price_field == 'sale_strikethrough_price') {
                $variant_price = $regular_price !== null ? $this->calculateFinalPrice($regular_price, $category_ids) : false;
            } elseif ($price_field == 'sale_price') {
                if ($sale_price && $this->calculateFinalPrice($sale_price, $category_ids)) {
                    $variant_price = $this->calculateFinalPrice($sale_price, $category_ids);
                } else {
                    $variant_price = $this->calculateFinalPrice($regular_price, $category_ids);
                }
            }

            if (!$variant_price) {
                continue;
            }

            $attributes = [];

            $variation_data = $variant_product->get_variation_attributes();
            foreach ($variation_data as $attribute_name => $attribute_value) {
                $taxonomy_name   = str_replace('attribute_', '', $attribute_name);
                $attribute_label = str_replace(['pa_', '-'], ' ', wc_attribute_label($taxonomy_name, $product));

                $value_name = rawurldecode($attribute_value);
                if (taxonomy_exists($taxonomy_name)) {
                    $term = get_term_by('slug', $attribute_value, $taxonomy_name);
                    if ($term && !is_wp_error($term)) {
                        $value_name = $term->name;
                    }
                }

                $attributes[] = [
                    'property' => $attribute_label,
                    'value'    => str_replace('-', ' ', mb_convert_encoding($value_name, 'UTF-8', 'auto')),
                ];
            }

            $variants[] = [
                'primary_price' => intval($variant_price),
                'stock'         => $this->getStockQuantity($variant_product),
                'properties'    => $attributes,
            ];
        }

        return $variants;
    }

    private function isMobileProduct($product)
    {
        return get_post_meta($product->get_id(), '_sync_basalam_is_mobile_product_checkbox', true) === 'yes';
    }

    private function getMobileAttributes($product)
    {
        return [
            [
                "attribute_id" => 1707,
                "value" => get_post_meta($product->get_id(), '_sync_basalam_mobile_storage', true),
            ],
            [
                "attribute_id" => 1708,
                "value" => get_post_meta($product->get_id(), '_sync_basalam_cpu_type', true),
            ],
            [
                "attribute_id" => 1709,
                "value" => get_post_meta($product->get_id(), '_sync_basalam_mobile_ram', true),
            ],
            [
                "attribute_id" => 1710,
                "value" => get_post_meta($product->get_id(), '_sync_basalam_screen_size', true),
            ],
            [
                "attribute_id" => 1711,
                "value" => get_post_meta($product->get_id(), '_sync_basalam_rear_camera', true),
            ],
            [
                "attribute_id" => 1712,
                "value" => get_post_meta($product->get_id(), '_sync_basalam_battery_capacity', true),
            ],
        ];
    }

    private function isProductType($product)
    {
        return get_post_meta($product->get_id(), '_sync_basalam_is_product_type_checkbox', true) === 'yes';
    }
    private function isWholesale($product)
    {
        return get_post_meta($product->get_id(), '_sync_basalam_is_wholesale', true) === 'yes';
    }
    private function getDescription($product)
    {
        $add_attrs_to_desc = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::ADD_ATTR_TO_DESC_PRODUCT);
        $add_short_desc_to_desc = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::ADD_SHORT_DESC_TO_DESC_PRODUCT);

        $description_parts = [];

        if ($add_short_desc_to_desc == 'yes') {
            $short_desc = $product->get_short_description();
            $short_desc_clean = SyncBasalamTextConverter::convertHtmlToPlainText($short_desc);

            if (!empty($short_desc_clean)) {
                $description_parts[] = trim($short_desc_clean);
            }
        }

        $main_desc = $product->get_description();
        $main_desc_clean = SyncBasalamTextConverter::convertHtmlToPlainText($main_desc);
        if (!empty($main_desc_clean)) {
            $description_parts[] = trim($main_desc_clean);
        }

        if ($add_attrs_to_desc == 'yes') {
            $attrs = $this->getWooAttributes($product);

            if (!empty($attrs) && is_array($attrs)) {
                $attrs_text = [];

                foreach ($attrs as $attr) {
                    if (!empty($attr['title']) && !empty($attr['value'])) {
                        $attrs_text[] = trim($attr['title']) . ' : ' . trim($attr['value']);
                    }
                }

                if (!empty($attrs_text)) {
                    $description_parts[] = implode("\n", $attrs_text);
                }
            }
        }

        $full_description = implode("\n\n", $description_parts);

        return mb_substr($full_description, 0, 5000);
    }

    private function getMappedCategory($product_id)
    {
        $product = wc_get_product($product_id);
        if (!$product) {
            return false;
        }

        $woo_categories = $product->get_category_ids();

        if (empty($woo_categories)) {
            return false;
        }

        foreach ($woo_categories as $woo_category_id) {
            $mapped_category = SyncBasalamCategoryMapping::getBasalamCategoryForWooCategory($woo_category_id);

            if ($mapped_category && isset($mapped_category->basalam_category_ids) && !empty($mapped_category->basalam_category_ids)) {
                return $mapped_category->basalam_category_ids;
            }
        }

        return false;
    }
}
