<?php
if (! defined('ABSPATH')) exit;

class SyncBasalamCreateProductDataV2
{
    private $currency;

    public function __construct()
    {
        $this->currency = get_woocommerce_currency();
    }

    public function getProductData($product_id)
    {
        $product_data = [];
        $basalam_product_id = get_post_meta($product_id, 'sync_basalam_product_id', true);

        if (!$basalam_product_id)  return false;

        $product = wc_get_product($product_id);

        $stock_quantity = $this->getStockQuantity($product);

        $mapped_category_ids = $this->getMappedCategory($product);

        if ($mapped_category_ids) {
            $category_ids = $mapped_category_ids;
        } else {
            $product_title = mb_substr($product->get_name(), 0, 120);
            $category_ids = SyncBasalamGetCategoryId::getCategoryIdFromBasalam(urlencode($product_title), 'multi');
        }

        $product_data['id'] = $basalam_product_id;
        $product_data['status'] = 2976;

        if (!$product->is_type('variable')) {
            $price = $this->calculateFinalPrice($this->getPrice($product), $category_ids);
            if (!$price) {
                return false;
            }
            $product_data['primary_price'] = $price;
        }

        if ($product->is_type('variable')) {
            $variants = $this->getVariants($product, $category_ids);
            $product_data['variants'] = $variants;
        } else {
            $product_data['stock'] = $stock_quantity;
        }

        if ($this->isMobileProduct($product)) {
            $product_data['product_attribute'] = $this->getMobileAttributes($product);
        }

        return $product_data;
    }

    private function getPrice($product)
    {
        $price_field = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::PRODUCT_PRICE_FIELD);

        $regular_price = $product->get_regular_price();
        $sale_price = $product->get_sale_price();

        $regular_price = !empty($regular_price) && is_numeric($regular_price) ? $regular_price : null;
        $sale_price = !empty($sale_price) && is_numeric($sale_price) ? $sale_price : null;

        if ($price_field == 'original_price' || $price_field == 'sale_strikethrough_price') {
            if ($regular_price !== null && $this->calculateFinalPrice($regular_price)) {
                return $regular_price;
            }
        } elseif ($price_field == 'sale_price') {
            if ($sale_price) {
                if ($this->calculateFinalPrice($sale_price)) {
                    return $sale_price;
                }
            }
            if ($this->calculateFinalPrice($regular_price)) {
                return $regular_price;
            }
            return null;
        }
    }

    private function calculateFinalPrice($price, $category_ids = null)
    {
        $increase_value = intval(SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::INCREASE_PRICE_VALUE));
        $round_mode = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::ROUND_PRICE);
        $currency = $this->currency;
        $converted_price = $this->convertPriceToRial($price, $currency);

        if (empty($converted_price) || !is_numeric($converted_price)) {
            return false;
        }

        $converted_price = floatval($converted_price);

        if ($increase_value == -1) {
            $category_data = SyncBasalamFetchCommission::fetchCategoryCommission($category_ids);

            if (is_array($category_data)) {
                list($category_percent, $max_amount) = $category_data;
            } else {
                $category_percent = $category_data;
                $max_amount = null;
            }

            $category_percent = floatval($category_percent);

            if ($category_percent > 0 && $category_percent < 100) {
                $finalPrice = $converted_price / (1 - ($category_percent / 100));
            } else {
                $finalPrice = $converted_price;
            }

            if ($max_amount !== null) {
                $max_amount = floatval($max_amount);
                $actual_commission = $finalPrice - $converted_price;

                if ($actual_commission > $max_amount) {
                    $finalPrice = $converted_price + $max_amount;
                }
            }
        } elseif ($increase_value <= 100) {
            $finalPrice = $converted_price + ($converted_price * ($increase_value / 100));
        } else {
            $finalPrice = $converted_price + ($increase_value * 10);
        }

        if ($round_mode === 'up') {
            return ceil($finalPrice / 10000) * 10000;
        } elseif ($round_mode === 'down') {
            return floor($finalPrice / 10000) * 10000;
        } else {
            return floor($finalPrice / 10) * 10;
        }
    }

    private function convertPriceToRial($price, $currency)
    {

        if (empty($price) || !is_numeric($price)) {
            return 0;
        }

        $price = floatval($price);

        if ($currency == 'IRT') {
            return $price * 10;
        } elseif ($currency == 'IRHT') {
            return $price * 10000;
        } elseif ($currency == 'IRHR') {
            return $price * 1000;
        }
        return $price;
    }

    private function getStockQuantity($product)
    {
        $stock_quantity = $product->get_stock_quantity();
        $stock_status =  $product->get_stock_status();
        if ($stock_status == 'instock' && $product->get_status() === 'publish') {
            return $stock_quantity === null ? SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::DEFAULT_STOCK_QUANTITY) : $stock_quantity;
        } else {
            return 0;
        }
    }

    private function getVariants($product, $category_ids)
    {
        $variants = [];
        $variation_ids = $product->get_children();

        $price_field = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::PRODUCT_PRICE_FIELD);

        foreach ($variation_ids as $variation_id) {
            $variant_product = wc_get_product($variation_id);
            $basalam_variant_id = get_post_meta($variation_id, 'sync_basalam_variation_id', true);
            if (!$variant_product || !$basalam_variant_id) {
                continue;
            }

            $regular_price = $variant_product->get_regular_price();
            $sale_price    = $variant_product->get_sale_price();

            $regular_price = !empty($regular_price) && is_numeric($regular_price) ? $regular_price : null;
            $sale_price = !empty($sale_price) && is_numeric($sale_price) ? $sale_price : null;

            if ($price_field === 'original_price' || $price_field == 'sale_strikethrough_price') {
                $variant_price = $regular_price !== null ? $this->calculateFinalPrice($regular_price, $category_ids) : false;
            } elseif ($price_field == 'sale_price') {
                if ($sale_price && $this->calculateFinalPrice($sale_price, $category_ids)) {
                    $variant_price = $this->calculateFinalPrice($sale_price, $category_ids);
                } else {
                    $variant_price = $this->calculateFinalPrice($regular_price, $category_ids);
                }
            }

            if (!$variant_price) {
                continue;
            }

            $attributes = [];

            $variation_data = $variant_product->get_variation_attributes();
            foreach ($variation_data as $attribute_name => $attribute_value) {
                $taxonomy_name   = str_replace('attribute_', '', $attribute_name);
                $attribute_label = str_replace(['pa_', '-'], ' ', wc_attribute_label($taxonomy_name, $product));

                $value_name = rawurldecode($attribute_value);
                if (taxonomy_exists($taxonomy_name)) {
                    $term = get_term_by('slug', $attribute_value, $taxonomy_name);
                    if ($term && !is_wp_error($term)) {
                        $value_name = $term->name;
                    }
                }

                $attributes[] = [
                    'property' => $attribute_label,
                    'value'    => str_replace('-', ' ', mb_convert_encoding($value_name, 'UTF-8', 'auto')),
                ];
            }

            $variants[] = [
                'id' => $basalam_variant_id,
                'primary_price' => $variant_price,
                'stock'         => $this->getStockQuantity($variant_product),
            ];
        }

        return $variants;
    }

    private function isMobileProduct($product)
    {
        return get_post_meta($product->get_id(), '_sync_basalam_is_mobile_product_checkbox', true) === 'yes';
    }

    private function getMobileAttributes($product)
    {
        return [
            [
                "attribute_id" => 1707,
                "value" => get_post_meta($product->get_id(), '_sync_basalam_mobile_storage', true),
            ],
            [
                "attribute_id" => 1708,
                "value" => get_post_meta($product->get_id(), '_sync_basalam_cpu_type', true),
            ],
            [
                "attribute_id" => 1709,
                "value" => get_post_meta($product->get_id(), '_sync_basalam_mobile_ram', true),
            ],
            [
                "attribute_id" => 1710,
                "value" => get_post_meta($product->get_id(), '_sync_basalam_screen_size', true),
            ],
            [
                "attribute_id" => 1711,
                "value" => get_post_meta($product->get_id(), '_sync_basalam_rear_camera', true),
            ],
            [
                "attribute_id" => 1712,
                "value" => get_post_meta($product->get_id(), '_sync_basalam_battery_capacity', true),
            ],
        ];
    }

    private function getMappedCategory($product)
    {
        $woo_categories = $product->get_category_ids();

        if (empty($woo_categories)) {
            return false;
        }

        foreach ($woo_categories as $woo_category_id) {
            $mapped_category = SyncBasalamCategoryMapping::getBasalamCategoryForWooCategory($woo_category_id);

            if ($mapped_category && isset($mapped_category->basalam_category_ids) && !empty($mapped_category->basalam_category_ids)) {
                return $mapped_category->basalam_category_ids;
            }
        }

        return false;
    }
}
