<?php
if (! defined('ABSPATH')) exit;
class SyncBasalamProductService
{

    public static function enqueueAllProductsForCreation($includeOutOfStock = false, $postsPerPage = 100)
    {
        $jobManager = new SyncBasalamJobManager();

        $existingJob = $jobManager->getJob([
            'job_type' => 'sync_basalam_create_all_products',
            'status' => 'pending'
        ]);

        if ($existingJob) {
            return [
                'success' => false,
                'message' => 'در حال حاضر یک عملیات در صف انتظار است.',
                'status_code' => 409
            ];
        }

        $initialData = json_encode([
            'posts_per_page'        => $postsPerPage,
            'include_out_of_stock'  => $includeOutOfStock
        ]);

        $jobManager->createJob(
            'sync_basalam_create_all_products',
            'pending',
            $initialData
        );

        return [
            'success' => true,
            'message' => 'محصولات با موفقیت به صف ایجاد افزوده شدند.',
            'status_code' => 200
        ];
    }

    public static function enqueueSelectedProductsForCreation($productIds)
    {
        if (is_array($productIds)) {

            $operationType = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::PRODUCT_OPERATION_TYPE);

            foreach ($productIds as $productId) {
                $basalamProductId = get_post_meta($productId, 'sync_basalam_product_id', true);
                if (empty($basalamProductId)) {
                    update_post_meta($productId, 'sync_basalam_product_sync_status', 'pending');
                    if ($operationType === 'immediate') {
                        try {
                            $class = new SyncBasalamCreateProduct();
                            $class->push(['type' => 'create_product', 'id' => $productId]);
                            $class->save();
                        } catch (\Throwable $th) {
                            update_post_meta($productId, 'sync_basalam_product_sync_status', 'no');
                            SyncBasalamLogger::error("خطا در ایجاد محصول فوری: " . $th->getMessage(), [
                                'product_id' => $productId,
                                'عملیات' => 'ایجاد فوری محصولات انتخابی',
                            ]);
                        }
                    } else {
                        $jobManager = new SyncBasalamJobManager();
                        $jobManager->createJob(
                            'sync_basalam_create_single_product',
                            'pending',
                            $productId,
                        );
                    }
                }
            }
            if ($operationType === 'immediate') {
                $class->dispatch();
            }
        }
    }

    public static function getCreatableProducts($args = [])
    {
        global $wpdb;

        $postsTable = $wpdb->posts;
        $metaTable  = $wpdb->postmeta;

        $postsPerPage = $args['posts_per_page'] ?? 100;
        $includeOutOfStock = $args['include_out_of_stock'] ?? false;
        $optionName   = 'sync_basalam_last_creatable_product_id';
        $lastId       = intval(get_option($optionName, 0));


        $stockCondition = $includeOutOfStock
            ? ""
            : "AND stock.meta_value = 'instock'";

        $query = $wpdb->prepare("
            SELECT p.ID
            FROM {$postsTable} AS p
            LEFT JOIN {$metaTable} AS thumb
                ON thumb.post_id = p.ID
                AND thumb.meta_key = '_thumbnail_id'
            LEFT JOIN {$metaTable} AS basalam
                ON basalam.post_id = p.ID
                AND basalam.meta_key = 'sync_basalam_product_id'
            LEFT JOIN {$metaTable} AS stock
                ON stock.post_id = p.ID
                AND stock.meta_key = '_stock_status'
            WHERE p.post_type = 'product'
            AND p.post_status = 'publish'
            AND thumb.meta_value IS NOT NULL
            AND basalam.post_id IS NULL
            AND p.ID > %d
            {$stockCondition}
            ORDER BY p.ID ASC
            LIMIT %d
        ", $lastId, $postsPerPage);

        $ids = $wpdb->get_col($query);

        if (!empty($ids)) {
            $newLastId = max($ids);
            update_option($optionName, $newLastId);
        }

        return array_map('intval', $ids);
    }

    public static function enqueueBulkUpdateProducts($offset)
    {
        $class = new SyncBasalamBulkUpdateProducts();
        $class->push($offset);
        $class->save();
        $class->dispatch();

        return [
            'success' => true,
            'message' => 'محصولات با موفقیت به صف بروزرسانی فوری افزوده شدند.',
            'status_code' => 200
        ];
    }

    public static function enqueueSelectedProductsForUpdate($productIds)
    {
        if (is_array($productIds)) {
            $operationType = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::PRODUCT_OPERATION_TYPE);

            foreach ($productIds as $productId) {
                $basalamProductId = get_post_meta($productId, 'sync_basalam_product_id', true);
                $jobsRunner = new SyncBasalamJobsRunner;
                if (!$jobsRunner->hasProductJobInProgress($productId, 'sync_basalam_update_single_product')) {
                    if (!empty($basalamProductId)) {
                        update_post_meta($productId, 'sync_basalam_product_sync_status', 'pending');
                        if ($operationType === 'immediate') {
                            try {
                                $class = new SyncBasalamUpdateProduct();
                                $class->push(['type' => 'update_product', 'id' => $productId]);
                                $class->save();
                            } catch (\Throwable $th) {
                                update_post_meta($productId, 'sync_basalam_product_sync_status', 'no');
                                SyncBasalamLogger::error("خطا در بروزرسانی محصول فوری: " . $th->getMessage(), [
                                    'product_id' => $productId,
                                    'عملیات' => 'بروزرسانی فوری محصولات انتخابی',
                                ]);
                            }
                        } else {
                            $jobManager = new SyncBasalamJobManager();
                            $jobManager->createJob(
                                'sync_basalam_update_single_product',
                                'pending',
                                $productId,
                            );
                        }
                    }
                }
            }
            if ($operationType === 'immediate') {
                $class->dispatch();
            }
        }
    }

    public static function disconnectSelectedProducts($productIds)
    {
        if (is_array($productIds)) {
            foreach ($productIds as $productId) {
                $operator = new SyncBasalamAdminProductOperations;
                $operator->disconnectProduct($productId);
            }
        }
    }

    public static function getUpdatableProducts($args = [])
    {
        global $wpdb;

        $postsTable = $wpdb->posts;
        $metaTable  = $wpdb->postmeta;

        $postsPerPage = $args['posts_per_page'] ?? 100;
        $optionName   = 'sync_basalam_last_updatable_product_id';
        $lastId       = intval(get_option($optionName, 0));

        $query = $wpdb->prepare("
            SELECT p.ID
            FROM {$postsTable} AS p
            INNER JOIN {$metaTable} AS basalam
                ON basalam.post_id = p.ID
                AND basalam.meta_key = 'sync_basalam_product_id'
            WHERE p.post_type = 'product'
            AND p.post_status = 'publish'
            AND p.ID > %d
            ORDER BY p.ID ASC
            LIMIT %d
        ", $lastId, $postsPerPage);

        $ids = $wpdb->get_col($query);

        if (!empty($ids)) {
            $newLastId = max($ids);
            update_option($optionName, $newLastId);
        }

        return array_map('intval', $ids);
    }

    public static function autoConnectAllProducts($page = 1)
    {
        $class = new SyncBasalamProductsConnectDispatcher();

        $class->push($page);

        $class->save()->dispatch();
    }

    public static function addToSchedule($createTaskClass, $args, $delay = null)
    {
        ($createTaskClass)->schedule($args, $delay);
    }
}
