<?php
/**
 * Sync Basalam Admin UI Components
 *
 * This class provides a comprehensive set of UI components and form elements
 * for the plugin's admin interface. It handles rendering of various form inputs,
 * buttons, modals, tables, and other UI elements with consistent styling and
 * Persian (Farsi) localization.
 *
 * The class includes:
 * - Form input elements (text, select, checkbox, etc.)
 * - Modal popups and tooltips
 * - Data tables for products and mappings
 * - Status indicators and icons
 * - Action buttons with AJAX functionality
 * - Settings form components
 *
 * @package    Sync_Basalam_Plugin
 * @subpackage Admin
 * @since      1.0.0
 * @author     Basalam Team
 */

// Prevent direct access to this file
if (! defined('ABSPATH')) exit;

/**
 * Admin UI Components Class
 *
 * Handles rendering of all admin interface elements for the plugin.
 * Provides consistent styling, localization, and security features
 * for all admin UI components.
 *
 * @class SyncBasalamAdminUI
 * @since 1.0.0
 */
class SyncBasalamAdminUI
{
    /**
     * Render WordPress Dashicon
     *
     * Creates a properly styled dashicon element for use in admin menus
     * and UI components.
     *
     * @since 1.0.0
     * @param string $icon Dashicon class name (e.g., 'dashicons-admin-home')
     * @return string HTML span element with dashicon
     */
    public static function renderIcon($icon)
    {
        return sprintf(
            '<span class="dashicons %s" style="font-size: 17px; vertical-align: middle;"></span>',
            esc_attr($icon)
        );
    }

    public static function renderInfoPopup($content, $unique_id = '')
    {
        $info_icon_url = syncBasalamPlugin()->assetsUrl() . "/icons/info-black.svg";
        $modal_id = 'basalam-info-modal-' . $unique_id;

        return sprintf(
            '<div class="basalam-info-trigger" data-modal-id="%s">' .
                '<img src="%s" alt="اطلاعات" class="basalam-info-icon" title="برای مشاهده توضیحات کلیک کنید">' .
                '</div>' .
                '<div id="%s" class="basalam-info-modal" style="display: none;">' .
                '<div class="basalam-info-modal-overlay"></div>' .
                '<div class="basalam-info-modal-content">' .
                '<div class="basalam-info-modal-header">' .
                '<h3 style="color: #fff;">راهنما</h3>' .
                '<span class="basalam-info-modal-close">&times;</span>' .
                '</div>' .
                '<div class="basalam-info-modal-body">%s</div>' .
                '</div>' .
                '</div>',
            esc_attr($modal_id),
            esc_url($info_icon_url),
            esc_attr($modal_id),
            esc_html($content)
        );
    }

    public static function renderLabelWithTooltip($label_text, $tooltip_content, $position = 'top')
    {

        $unique_id = sanitize_title($label_text);

        return sprintf(
            '<div class="basalam-label-container">' .
                '<label class="basalam-label">' .
                '<span class="basalam-label-text">%s</span>' .
                '%s' .
                '</label>' .
                '</div>',
            esc_html($label_text),
            self::renderInfoPopup($tooltip_content, $unique_id)
        );
    }

    /**
     * Render hidden fields to delete access credentials
     *
     * Outputs hidden form fields that will clear authentication tokens
     * when the form is submitted, effectively logging out from Basalam.
     *
     * @since 1.0.0
     */
    public static function renderSyncBasalamDeleteAccess()
    {
        echo '<input type="hidden" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::TOKEN) . ']" value="">' .
            '<input type="hidden" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::REFRESH_TOKEN) . ']" value="">';
    }

    public static function syncStatusProduct()
    {
        $value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::SYNC_STATUS_PRODUCT) == true ? false : true;
        echo '<input type="hidden" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::SYNC_STATUS_PRODUCT) . ']" value="' . esc_attr($value) . '">';
    }

    public static function syncStatusOrder()
    {
        $value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::SYNC_STATUS_ORDER) == true ? false : true;
        echo '<input type="hidden" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::SYNC_STATUS_ORDER) . ']" value="' . esc_attr($value) . '">';
    }
    public static function renderAutoConfirmOrderButton()
    {
        $value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::AUTO_CONFIRM_ORDER) == true ? false : true;
        echo '<input type="hidden" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::AUTO_CONFIRM_ORDER) . ']" value="' . esc_attr($value) . '">';
    }

    public static function renderDefaultWeight()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::DEFAULT_WEIGHT);
        echo '<input type="number" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::DEFAULT_WEIGHT) . ']" min="50" value="' . esc_attr($current_value) . '" class="basalam-input basalam-p" required>';
    }

    public static function renderPackageWeight()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::DEFAULT_PACKAGE_WEIGHT);
        echo '<input type="number" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::DEFAULT_PACKAGE_WEIGHT) . ']" min="10" value="' . esc_attr($current_value) . '" class="basalam-input basalam-p" required>';
    }

    public static function renderDefaultPreparation()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::DEFAULT_PREPARATION);
        echo '<input type="number" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::DEFAULT_PREPARATION) . ']" min="0" value="' . esc_attr($current_value) . '" class="basalam-input basalam-p" required>';
    }

    public static function renderDefaultStockQuantity()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::DEFAULT_STOCK_QUANTITY);
        echo '<input type="number" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::DEFAULT_STOCK_QUANTITY) . ']" min="0" value="' . esc_attr($current_value) . '" class="basalam-input basalam-p" required>';
    }

    public static function renderDefaultPercentage()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::INCREASE_PRICE_VALUE);
        $is_checked = ($current_value == -1) ? 'checked' : '';
        $input_disabled = ($current_value == -1) ? 'disabled' : '';
        $input_value = ($current_value == -1) ? '' : number_format($current_value);

        echo '<div class="basalam-input-container" style="display: flex; align-items: center; gap: 4px;justify-content: center;">';
        echo '<input type="text" id="percentage-input" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::INCREASE_PRICE_VALUE) . ']" min="0" value="' . esc_attr($input_value) . '" class="basalam-input basalam-p percentage-input" ' . esc_attr($input_disabled) . ' required>';
        echo '<span class="percentage-unit basalam-p" style="min-width: 0px;font-size: 13px;">' . ($current_value <= 100 ? 'درصد' : 'تومان') . '</span>';
        echo '</div>';

        echo '<div style="margin-top: 8px; display: flex; align-items: center; gap: 4px;">';
        echo '<input type="checkbox" id="toggle-percentage" name="toggle_percentage" class="Basalam-checkbox" ' . esc_attr($is_checked) . '>';
        echo '<label style="font-size:10px;" for="toggle-percentage">کارمزد دسته‌بندی</label>';
        echo '</div>';

        echo '<input type="hidden" id="final-value" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::INCREASE_PRICE_VALUE) . ']" value="' . esc_attr($current_value) . '">';
    }

    public static function renderDefaultRound()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::ROUND_PRICE);
        echo '<select style="text-align: center; font-size:12px;" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::ROUND_PRICE) . ']" class="basalam-select"> ' .
            '<option value="none"' . selected($current_value, "none", false) . '>رند نکردن</option>' .
            '<option value="up"' . selected($current_value, "up", false) . '>بالا</option>' .
            '<option value="down"' . selected($current_value, "down", false) . '>پایین</option>' .
            '</select>';
    }

    public static function renderSyncProduct()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELDS);
        echo '<select style="text-align: center;font-size:12px;" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELDS) . ']" class="basalam-select" onchange="BasalamToggleCustomFields(this.value)" id="basalam-sync-type">' .
            '<option value="all"' . selected($current_value, "all", false) . '>همه اطلاعات</option>' .
            '<option value="price_stock"' . selected($current_value, "price_stock", false) . '>فقط قیمت و موجودی</option>' .
            '<option value="custom"' . selected($current_value, "custom", false) . '>سفارشی</option>' .
            '</select>';
    }

    public static function renderWholesaleProducts()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::ALL_PRODUCTS_WHOLESALE);
        echo '<select style="text-align: center; font-size:12px;" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::ALL_PRODUCTS_WHOLESALE) . ']" class="basalam-select">' .
            '<option value="none"' . selected($current_value, "none", false) . '>هیچ یا برخی محصولات عمده</option>' .
            '<option value="all"' . selected($current_value, "all", false) . '>همه محصولات عمده</option>' .
            '</select>';
    }

    public static function renderAttrAddToDesc()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::ADD_ATTR_TO_DESC_PRODUCT);
        echo '<select style="text-align: center; font-size:12px;" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::ADD_ATTR_TO_DESC_PRODUCT) . ']" class="basalam-select">' .
            '<option value="no"' . selected($current_value, 'no', false) . '>اضافه نشود</option>' .
            '<option value="yes"' . selected($current_value, 'yes', false) . '>اضافه شود</option>' .
            '</select>';
    }

    public static function renderOrderStatus()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::ORDER_STATUES_TYPE);
        echo '<select style="text-align: center; font-size:12px;" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::ORDER_STATUES_TYPE) . ']" class="basalam-select">' .
            '<option value="woosalam_statuses"' . selected($current_value, 'woosalam_statuses', false) . '>وضعیت های ووسلام</option>' .
            '<option value="woocommerce_statuses"' . selected($current_value, 'woocommerce_statuses', false) . '>وضعیت های ووکامرس</option>' .
            '</select>';
    }

    public static function renderShortAttrAddToDesc()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::ADD_SHORT_DESC_TO_DESC_PRODUCT);
        echo '<select style="text-align: center; font-size:12px;" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::ADD_SHORT_DESC_TO_DESC_PRODUCT) . ']" class="basalam-select">' .
            '<option value="no"' . selected($current_value, 'no', false) . '>اضافه نشود</option>' .
            '<option value="yes"' . selected($current_value, 'yes', false) . '>اضافه شود</option>' .
            '</select>';
    }

    public static function renderProductPrice()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::PRODUCT_PRICE_FIELD);
        echo '<select style="text-align: center; font-size:12px;" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::PRODUCT_PRICE_FIELD) . ']" class="basalam-select">' .
            '<option value="original_price"' . selected($current_value, 'original_price', false) . '>قیمت اصلی</option>' .
            '<option value="sale_price"' . selected($current_value, 'sale_price', false) . '>قیمت حراجی (تک قیمت)</option>' .
            '<option value="sale_strikethrough_price"' . selected($current_value, 'sale_strikethrough_price', false) . '>قیمت حراجی (خط خورده)</option>' .
            '</select>';
    }
    public static function renderProductOperationType()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::PRODUCT_OPERATION_TYPE);
        echo '<select style="text-align: center; font-size:12px;" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::PRODUCT_OPERATION_TYPE) . ']" class="basalam-select">' .
            '<option value="optimized"' . selected($current_value, 'optimized', false) . '>بهینه (استفاده از صف)</option>' .
            '<option value="immediate"' . selected($current_value, 'immediate', false) . '>در لحظه</option>' .
            '</select>';
    }

    public static function renderProductDiscountDuration()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::DISCOUNT_DURATION);

        echo '<input type="number" id="percentage-input" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::DISCOUNT_DURATION) . ']" min="1" max="90" value="' . esc_attr($current_value) . '" class="basalam-input basalam-p percentage-input" required>';
    }

    public static function renderTasksPerMinute()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::TASKS_PER_MINUTE);
        $is_auto = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::TASKS_PER_MINUTE_AUTO) == 'true';
        $disabled = $is_auto ? 'disabled' : '';

        echo '<input type="number" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::TASKS_PER_MINUTE) . ']" min="1" max="60" value="' . esc_attr($current_value) . '" class="basalam-input basalam-p basalam-tasks-manual-input" ' . $disabled . ' required>';
    }

    public static function renderTasksPerMinuteAutoToggle()
    {
        $is_auto = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::TASKS_PER_MINUTE_AUTO) == 'true';
        $checked = $is_auto ? 'checked' : '';

        echo '<label class="basalam-switch">';
        echo '<input type="checkbox" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::TASKS_PER_MINUTE_AUTO) . ']" value="true" ' . $checked . ' class="basalam-tasks-auto-toggle">';
        echo '<span class="basalam-slider"></span>';
        echo '</label>';
        echo '<input type="hidden" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::TASKS_PER_MINUTE_AUTO) . ']" value="false" class="basalam-tasks-auto-hidden">';
    }

    public static function renderTasksPerMinuteInfo()
    {
        $is_auto = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::TASKS_PER_MINUTE_AUTO) == 'true';

        if ($is_auto) {
            $monitor = SyncBasalamSystemResourceMonitor::getInstance();
            $optimal = $monitor->calculateOptimalTasksPerMinute();
            $batch_size = $monitor->calculateOptimalBatchSize(20, 120);

            echo '<div class="basalam-form-group basalam-p" style="grid-column: 1 / -1;">';
            echo '<div class="basalam-tasks-info basalam-p" style="margin-top: 10px; padding: 15px; background:linear-gradient(135deg, #f5f7fa 0%, #c3cfe266 100%); border-radius: 8px; border: 1px solid #ddd;">';

            echo '<div style="display: flex; flex-direction: column; gap: 8px;">';

            echo '<div style="display: flex; align-items: center; gap: 10px;">';
            echo '<strong style="color: #2d3748; font-size: 14px;">🚀 تعداد تسک های اجرایی در دقیقه: </strong>';
            echo '<span style="color: #ff5c35; font-size: 16px; font-weight: bold;">' . esc_html($optimal) . ' تسک در دقیقه</span>';
            echo '</div>';

            echo '<div style="display: flex; align-items: center; gap: 10px;">';
            echo '<strong style="color: #2d3748; font-size: 14px;">📦 تعداد آیتم در هر بروزرسانی: </strong>';
            echo '<span style="color: #00a884; font-size: 16px; font-weight: bold;">' . esc_html($batch_size) . ' محصول</span>';
            echo '<span style="color: #718096; font-size: 12px;">(بروزرسانی فوری)</span>';
            echo '</div>';

            echo '</div>';

            echo '</div>';
            echo '</div>';
        }
    }
    public static function renderPrefixProductTitle()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::PRODUCT_PREFIX_TITLE);
        echo '<input type="text" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::PRODUCT_PREFIX_TITLE) . ']" value="' . esc_attr($current_value) . '" class="basalam-input basalam-p" style="max-width:80% !important;font-size:12px;">';
    }

    public static function renderSuffixProductTitle()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::PRODUCT_SUFFIX_TITLE);
        echo '<input type="text" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::PRODUCT_SUFFIX_TITLE) . ']" value="' . esc_attr($current_value) . '" class="basalam-input basalam-p"style="max-width:80% !important;font-size:12px;">';
    }

    public static function renderAttributeSuffixEnabled()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::PRODUCT_ATTRIBUTE_SUFFIX_ENABLED);
        $checked = $current_value == 'yes' ? 'checked' : '';

        echo '<label class="basalam-switch">';
        echo '<input type="checkbox" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::PRODUCT_ATTRIBUTE_SUFFIX_ENABLED) . ']" value="yes" ' . $checked . ' class="basalam-attribute-suffix-toggle">';
        echo '<span class="basalam-slider"></span>';
        echo '</label>';
        echo '<input type="hidden" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::PRODUCT_ATTRIBUTE_SUFFIX_ENABLED) . ']" value="no" class="basalam-attribute-suffix-hidden">';
    }

    public static function renderAttributeSuffixPriority()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::PRODUCT_ATTRIBUTE_SUFFIX_PRIORITY);
        $is_enabled = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::PRODUCT_ATTRIBUTE_SUFFIX_ENABLED) == 'yes';
        $disabled = $is_enabled ? '' : 'disabled';

        echo '<input type="text" name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::PRODUCT_ATTRIBUTE_SUFFIX_PRIORITY) . ']" value="' . esc_attr($current_value) . '" class="basalam-input basalam-p basalam-attribute-suffix-priority" placeholder="مثال: ناشر" ' . $disabled . ' style="max-width:80% !important;font-size:12px;">';
    }

    public static function renderMapOptionsProduct()
    {
?>

        <form id="Basalam-map-option-form" method="post" style="display: flex; align-items: center; gap: 10px;">
            <?php wp_nonce_field('basalam_add_map_option_nonce', 'basalam_add_map_option_nonce'); ?>
            <label for="woo-option-name" class="basalam-p__small">نام ویژگی در ووکامرس</label>
            <input type="text" class="basalam-input" style="width: auto;" id="woo-option-name" name="woo-option-name" required>
            <label for="Basalam-option-name" class="basalam-p__small">نام ویژگی در باسلام</label>
            <input type="text" class="basalam-input" style="width: auto;" id="Basalam-option-name" name="Basalam-option-name" required>
            <button type="submit" class="basalam-primary-button basalam-p" style="width: auto;">ذخیره</button>
        </form>

    <?php
    }

    public static function renderSyncProductFields()
    {
        echo '<div>';
        echo wp_kses(self::renderSingleCheckbox(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELD_NAME, 'نام'), self::allowedHtml());
        echo wp_kses(self::renderSingleCheckbox(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELD_PHOTOS, 'عکس'), self::allowedHtml());
        echo wp_kses(self::renderSingleCheckbox(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELD_PRICE, 'قیمت'), self::allowedHtml());
        echo wp_kses(self::renderSingleCheckbox(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELD_STOCK, 'موجودی'), self::allowedHtml());
        echo wp_kses(self::renderSingleCheckbox(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELD_WEIGHT, 'وزن'), self::allowedHtml());
        echo wp_kses(self::renderSingleCheckbox(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELD_DESCRIPTION, 'توضیحات'), self::allowedHtml());
        echo wp_kses(self::renderSingleCheckbox(SyncBasalamAdminSettings::SYNC_PRODUCT_FIELD_ATTR, 'ویژگی ها'), self::allowedHtml());
        echo '</div>';
    }

    private static function renderSingleCheckbox($field_key, $label)
    {
        return '<label class="basalam-p sync-checkbox-label" style="width: 32%; text-align:right;  margin-bottom: 10px;">' .
            '<input type="hidden" name="sync_basalam_settings[' . esc_attr($field_key) . ']" value="">' .
            '<input type="checkbox" name="sync_basalam_settings[' . esc_attr($field_key) . ']" value="1" ' .
            checked(SyncBasalamAdminSettings::getSettings($field_key), true, false) . '>' .
            esc_html($label) .
            '</label>';
    }

    /**
     * Get allowed HTML tags for wp_kses
     *
     * Returns an array of allowed HTML tags and attributes for use with
     * WordPress's wp_kses function to safely output HTML content.
     *
     * @since 1.0.0
     * @return array Allowed HTML tags and attributes
     */
    public static function allowedHtml()
    {
        return [
            'form' => [
                'action' => [],
                'method' => [],
                'id' => [],
                'class' => [],
            ],
            'input' => [
                'type' => [],
                'name' => [],
                'value' => [],
                'class' => [],
                'required' => [],
                'style' => [],
                'id' => [],
                'checked' => [],
            ],
            'button' => [
                'type' => [],
                'class' => [],
                'style' => [],
            ],
            'div' => [
                'class' => [],
                'id' => [],
                'style' => [],
            ],
            'p' => [
                'class' => [],
                'style' => [],
            ],
            'strong' => [],
            'label' => [
                'class' => [],
                'style' => [],
            ],
            'svg' => [
                'width' => [],
                'height' => [],
                'viewBox' => [],
                'xmlns' => [],
                'fill' => [],
            ],
            'path' => [
                'fill' => [],
                'fill-rule' => [],
                'clip-rule' => [],
                'd' => [],
            ],
            'span' => [
                'class' => [],
            ],
            'code' => [
                'class' => [],
            ],
            'a' => [
                'href' => [],
                'target' => [],
                'class' => [],
            ],
            'ul' => [
                'class' => [],
                'style' => [],
            ],
            'li' => [
                'class' => [],
                'style' => [],
            ],
            'img' => [
                'src' => [],
                'alt' => [],
                'class' => [],
            ],
        ];
    }

    public static function renderDeveloperMode()
    {
        $current_value = SyncBasalamAdminSettings::getSettings(SyncBasalamAdminSettings::DEVELOPER_MODE);
        echo '<select name="sync_basalam_settings[' . esc_attr(SyncBasalamAdminSettings::DEVELOPER_MODE) . ']" class="basalam-select" onchange="this.form.submit()">' .
            '<option value="false"' . selected($current_value, "false", false) . '>غیرفعال</option>' .
            '<option value="true"' . selected($current_value, "true", false) . '>فعال</option>' .
            '</select>';
    }

    public static function renderFaqByCategory($categories)
    {
        foreach ($categories as $category) {
            $is_active = $category === 'عمومی' ? ' active' : '';
            $nonce = wp_create_nonce('sync_basalam_faq_nonce');
            echo '<div class="basalam-faq-section' . esc_attr($is_active) . '" data-category="' . esc_attr($category) . '" data-nonce="' . esc_attr($nonce) . '">';

            $faqs = SyncBasalamFaq::getFaqByCategory($category);
            $faqs_html = array_map(function ($faq) {
                return '
                    <div class="basalam-faq-item">
                        <div class="basalam-faq-question">
                            <h3>' . esc_html($faq['question']) . '</h3>
                            <span class="basalam-faq-toggle">+</span>
                        </div>
                        <div class="basalam-faq-answer">
                            <p>' . esc_html($faq['answer']) . '</p>
                        </div>
                    </div>
                ';
            }, $faqs);

            $escaped_faqs_html = implode('', $faqs_html);
            echo wp_kses_post($escaped_faqs_html);

            echo '</div>';
        }
    }

    public static function renderSyncProductStatusOk()
    {
        echo '<span class="dashicons dashicons-yes-alt" style="color: #00b67a;" title="محصول با باسلام سینک شده است."></span>';
    }

    public static function renderSyncProductStatusPending()
    {
        echo '<span class="dashicons dashicons-update" style="color: #f7a700;" title="در حال سینک با باسلام"></span>';
    }

    public static function renderSyncProductStatusFail()
    {
        echo '<span class="dashicons dashicons-no-alt" style="color: #dc3232;" title="محصول در باسلام ثبت نشده است یا فرایند سینک موفق نبود"></span>';
    }

    /**
     * Render action button or link
     *
     * Creates either an AJAX action button or a regular link button
     * based on the parameters provided. Used throughout the admin interface
     * for various actions.
     *
     * @since 1.0.0
     * @param string $text Button text to display
     * @param string|null $link URL for link buttons
     * @param string|null $name Action name for AJAX buttons
     * @param int|null $product_id Product ID for product-specific actions
     * @param string|null $nonce_name Nonce action name for security
     */
    public static function renderBtn($text, $link = null, $name = null, $product_id = null, $nonce_name = null)
    {
        if ($name && $link == null) {
            if ($nonce_name) {
                $nonce = wp_create_nonce($nonce_name);
            }
            echo '<button class="basalam-button basalam-action-button  basalam-p basalam-a" 
            data-url="' . esc_url(admin_url('admin-post.php')) . '" 
            data-action="' . esc_attr($name) . '" 
            data-product-id="' . esc_attr($product_id) . '" 
            data-_wpnonce="' . esc_attr($nonce) . '">' . esc_html($text) . '</button>';
        } else {
            echo '<a href="' . esc_url($link) . '" target="_blank" class="basalam-button basalam-btn basalam-p basalam-a">' . esc_html($text) . '</a>';
        }
    }
    public static function renderCheckSyncBasalamOrdersButton()
    {
    ?>
        <div class="alignleft actions custom">
            <button type="button" class="basalam-button basalam-p basalam_add_unsync_orders" style="height:32px;"
                title="تمامی سفارشات جدیدی باسلامی که همگام سازی نشده اند ، اضافه میشوند."
                data-nonce="<?php echo esc_attr(wp_create_nonce('add_unsync_orders_from_basalam_nonce')); ?>">
                بررسی سفارشات باسلام
            </button>
        </div>
<?php
    }
    public static function renderCategoryOptionsMapping($data)
    {
        $delete_nonce = wp_create_nonce('basalam_delete_mapped_option_nonce');

        echo '<div class="options_mapping_section">';
        echo '<p class="basalam-p">لیست ویژگی ها : </p>';
        echo "<table class='basalam-table basalam-p'>";
        echo '<thead><tr><th>نام ویژگی در ووکامرس</th><th>نام ویژگی در باسلام</th><th>عملیات</th></tr></thead>';
        echo '<tbody>';

        if (!empty($data)) {
            foreach ($data as $item) {
                echo '<tr data-woo="' . esc_attr($item['woo_name']) . '" data-Basalam="' . esc_attr($item['sync_basalam_name']) . '">';
                echo '<td>' . esc_html($item['woo_name']) . '</td>';
                echo '<td>' . esc_html($item['sync_basalam_name']) . '</td>';
                echo '<td>
                    <button 
                        class="Basalam-delete-option basalam-primary-button" 
                        data-_wpnonce="' . esc_attr($delete_nonce) . '" 
                        onclick="return confirm(\'آیا مطمئن هستید؟\')" 
                        style="width:auto">
                        حذف
                    </button>
                </td>';
                echo '</tr>';
            }
        }

        echo '</tbody></table>';
        echo '</div>';
    }

    public static function renderUnsyncBasalamProductsTable($unsync_products)
    {
        echo "<div style='display: flex;flex-direction: column;justify-content: center;align-items: center;'>";
        if (empty($unsync_products)) {
            return null;
        }

        echo "<h3 style='margin-bottom: 15px !important;font-family: PelakFA !important;font-weight: bold;'>📦 محصولات سینک‌نشده باسلام:</h3>";
        echo "<table class='basalam-p' style='border-collapse: collapse;max-width: 1005px;width: 100%;'>";

        echo "<thead style='background-color: #f4f4f4; color: #333;'>
                <tr>
                    <th style='padding: 6px 8px;  border: 1px solid #ddd;'>#</th>
                    <th style='padding: 6px 8px;  border: 1px solid #ddd;'>تصویر</th>
                    <th style='padding: 6px 8px; border: 1px solid #ddd;'>عنوان</th>
                    <th style='padding: 6px 8px; border: 1px solid #ddd;'>قیمت (تومان)</th>
                    <th style='padding: 6px 8px;  border: 1px solid #ddd;'>آیدی باسلام</th>
                    <th style='padding: 6px 20px;  border: 1px solid #ddd;'>محصول در باسلام</th>
                </tr>
              </thead>";

        echo "<tbody>";

        foreach ($unsync_products as $index => $product) {
            echo "<tr style='background-color: #fff; border-bottom: 1px solid #ddd;'>
            <td style='padding: 6px 8px; '>" . esc_html($index + 1) . "</td>
            <td style='padding: 6px 8px; '><img src='" . esc_url($product['photo']) . "' alt='Product Image' style='border-radius: 4px; width: 60px; height: 60px; object-fit: cover;'></td>
            <td style='padding: 6px 8px;'>" . esc_html($product['title']) . "</td>
            <td style='padding: 6px 8px;'>" . esc_html(number_format($product['price'])) . "</td>
            <td style='padding: 6px 8px; '>" . esc_html($product['id']) . "</td>
            <td style='padding: 6px 8px; '>
            <button class='basalam-button basalam-p' style='width: unset;margin:auto !important;height: 35px;'>
                <a class='basalam-a' style='font-size:10px' href='https://basalam.com/p/" . esc_attr($product['id']) . "' target='_blank'>مشاهده محصول</a>
            </button>

            </td>
          </tr>";
        }

        echo "</tbody></table>";
        echo "</div>";
    }
}
