<?php
/**
 * The template used to generate a sample nginx.conf configuration.
 *
 * This assumes the user has a correct "root "/app/wordpress"; definition in their server block, the cache path is relative to that.
 *
 * @var string $cache_path The relative path from WP_CONTENT_DIR without slashes, e.g. wp-content/cache/solid-performance/page
 * @var int $expiration The expiration time in seconds.
 *
 * @package SolidWP\Performance
 */

declare( strict_types=1 );

$nginx = <<<RULES
# BEGIN SolidPerformanceNginxCacheRules
# This should be included in the server{} block of your vhost.
# This file is automatically generated. Do no edit. Use the `solidwp/performance/nginx/rules` filter.

# Deny direct access to the Solid Performance swpsp-nginx.conf.
location ~ swpsp-nginx\.conf$ {
	deny all;
}

# Headers to always send with each request.
add_header Referrer-Policy "no-referrer-when-downgrade" always;

# Copy the \$uri variable to ensure rewrite processing is always using the original.
set \$swpsp_cache_uri \$uri;
# Assume http for the default scheme.
set \$swpsp_scheme http;
# Don't bypass the cache by default.
set \$swpsp_bypass_cache 0;
# The default location block to use with try_files in the / location.
set \$swpsp_cache_try "@html_cache";
# The relative cache directory to the configured Nginx root.
# This supports sub directories automatically, e.g. root "/app/wordpress"; will look in /app/wordpress/wp-content/...
set \$swpsp_cache_dir "/{$cache_path}";

# Extra checks for scheme.
if (\$https = on) {
	set \$swpsp_scheme https;
}

if (\$http_x_forwarded_proto = https) {
	set \$swpsp_scheme https;
}

# Bypass all post requests.
if (\$request_method = POST) {
	set \$swpsp_bypass_cache 1;
}

# Bypass URLs with query strings.
if (\$query_string != "") {
	set \$swpsp_bypass_cache 1;
}

# Bypass URLs with the following segments.
if (\$request_uri ~* "/wp-admin/|/wp-json/|/xmlrpc.php|wp-.*.php|/feed/|index.php|sitemap(_index)?.xml|/cart/|/basket/|/checkout/|/my-account/") {
	set \$swpsp_bypass_cache 1;
}

# Bypass cache if the following cookies are set.
if (\$http_cookie ~* "comment_author|wordpress_[a-f0-9]+|wp-postpass|wordpress_no_cache|wordpress_logged_in|edd_items_in_cart|woocommerce_items_in_cart") {
	set \$swpsp_bypass_cache 1;
}

# If the request wants a gzip response, try that first.
if (\$http_accept_encoding ~ gzip) {
	set \$swpsp_cache_try "@gz_cache";
}

# If we need to bypass the cache, use named @wordpress location directly.
if (\$swpsp_bypass_cache = 1) {
	set \$swpsp_cache_try "@wordpress";
}

# All requests. If you have this in your configuration already, ensure to replace it with ours.
location / {
	try_files \$uri \$swpsp_cache_try;
}

# Attempt to serve our already compressed gzip file without having Nginx compress it again.
# This will automatically try the @html_cache location if no file is found.
location @gz_cache {
  internal;
  log_not_found off;
  recursive_error_pages on;
  error_page 404 = @html_cache;

  gzip off;
  types {
    text/html gz html;
  }
  default_type text/html;

  expires {$expiration}s;
  etag on;
  if_modified_since exact;

  add_header Content-Encoding gzip;
  add_header X-Cached-By "Solid Performance (nginx)";
  add_header X-Cache "HIT";
  add_header X-Cache-Encoding "GZIP";
  add_header Cache-Control "public";
  add_header Vary "Accept-Encoding";

  rewrite ^ "\${swpsp_cache_dir}/\${host}\${swpsp_cache_uri}/index-\${swpsp_scheme}.gz" break;
}

# Attempt to serve a fallback html cache file, if it exits.
location @html_cache {
  internal;
  log_not_found off;
  recursive_error_pages on;
  error_page 404 = @wordpress;

  # Have Nginx compress this if the request allows it and our gz file is missing.
  gzip on;

  expires {$expiration}s;
  etag on;
  if_modified_since exact;

  add_header X-Cached-By "Solid Performance (nginx)";
  add_header X-Cache "HIT";
  add_header X-Cache-Encoding "HTML";
  add_header Cache-Control "public";
  add_header Vary "Accept-Encoding";

  rewrite ^ "\${swpsp_cache_dir}/\${host}\${swpsp_cache_uri}/index-\${swpsp_scheme}.html" break;
}

# The default WordPress location.
location @wordpress {
	try_files \$uri \$uri/ /index.php?\$args;
}
# END SolidPerformanceNginxCacheRules

RULES;

// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped, StellarWP.XSS.EscapeOutput.OutputNotEscaped
echo apply_filters( 'solidwp/performance/nginx/rules', $nginx, $cache_path );
