import React from 'react';
import { __ } from '@wordpress/i18n';
import SideTabs from '../../../components/side-tabs';
import SettingsRow from '../../../components/settings-row';
import NoticeUtil from '../../../utils/notice-util';
import { createInterpolateElement } from '@wordpress/element';
import FloatingNoticePlaceholder from '../../../components/floating-notice-placeholder';
import CodeSnippet from '../../../components/code-snippet';

export default class CodeType extends React.Component {
	constructor(props) {
		super(props);

		this.state = {
			codeType: '0',
		};
	}

	render() {
		const { codeType } = this.state;

		return (
			<SettingsRow
				id="wds-breadcrumb-code"
				label={__('Add Breadcrumbs to your Webpage', 'smartcrawl-seo')}
				description={__(
					'You can add breadcrumbs to any page on your website using the ‘shortcode’ on the post editor, or the ‘PHP code’ on the template page.',
					'wds'
				)}
			>
				<FloatingNoticePlaceholder id="smartcrawl-breadcrumb-copied" />

				<SideTabs
					tabs={{
						0: __('Shortcode', 'smartcrawl-seo'),
						1: __('PHP Code', 'smartcrawl-seo'),
					}}
					value={codeType}
					onChange={(checked) => this.handleChange(checked)}
				>
					{codeType === '0' && (
						<CodeSnippet
							snippet="[smartcrawl_breadcrumbs]"
							description={createInterpolateElement(
								__(
									'Copy the shortcode and paste it to the desired location in the post editor to display the breadcrumbs on your page or post. <a>Learn more</a>',
									'smartcrawl-seo'
								),
								{
									a: (
										<a
											href="https://wpmudev.com/docs/wpmu-dev-plugins/smartcrawl/#add-breadcrumbs"
											target="_blank"
											rel="noreferrer"
											className="learn-more"
										/>
									),
								}
							)}
							onCopy={() => this.handleCopy(codeType)}
						/>
					)}
					{codeType === '1' && (
						<CodeSnippet
							snippet={
								"<?php\n if ( function_exists( 'smartcrawl_breadcrumbs' ) ) {\n    smartcrawl_breadcrumbs();\n } \n?>"
							}
							description={createInterpolateElement(
								__(
									'Copy the PHP code and paste it in the desired location within template editor to display the breadcrumbs on your page or post. <a>Learn more</a>',
									'smartcrawl-seo'
								),
								{
									a: (
										<a
											href="https://wpmudev.com/docs/wpmu-dev-plugins/smartcrawl/#add-breadcrumbs"
											target="_blank"
											rel="noreferrer"
											className="learn-more"
										/>
									),
								}
							)}
							onCopy={() => this.handleCopy(codeType)}
						/>
					)}
				</SideTabs>
			</SettingsRow>
		);
	}

	handleChange(value) {
		this.setState({
			codeType: value,
		});
	}

	handleCopy(codeType) {
		if (codeType === '0') {
			navigator.clipboard
				.writeText('[smartcrawl_breadcrumbs]')
				.then(() => {
					NoticeUtil.showSuccessNotice(
						'smartcrawl-breadcrumb-copied',
						__('Shortcode copied successfully.', 'smartcrawl-seo'),
						false
					);
				})
				.catch(() => {
					NoticeUtil.showErrorNotice(
						'smartcrawl-breadcrumb-copied',
						__(
							'Shortcode could not be copied to clipboard.',
							'smartcrawl-seo'
						),
						false
					);
				});
		}
		if (codeType === '1') {
			navigator.clipboard
				.writeText(
					"<?php if ( function_exists( 'smartcrawl_breadcrumbs' ) ) {\n    smartcrawl_breadcrumbs();\n} ?>"
				)
				.then(() => {
					NoticeUtil.showSuccessNotice(
						'smartcrawl-breadcrumb-copied',
						__(
							'The PHP Code copied successfully.',
							'smartcrawl-seo'
						),
						false
					);
				})
				.catch(() => {
					NoticeUtil.showErrorNotice(
						'smartcrawl-breadcrumb-copied',
						__(
							'The PHP Code could not be copied to clipboard.',
							'smartcrawl-seo'
						),
						false
					);
				});
		}
	}
}
