<?php

/**
 * Plugin Name: Sky Login Redirect
 * Plugin URI: https://utopique.net/products/sky-login-redirect-premium/
 * Description: Redirects users to the page they were prior to logging in or out. Features an awesome login customizer.
 * Version: 3.8.0
 * Author: Utopique
 * Author URI: https://utopique.net/
 * Developer: Utopique
 * Developer URI: https://utopique.net/
 * Copyright: (c) 2009-2025 Utopique
 * Text Domain: sky-login-redirect
 * Domain Path: /languages
 * License: GPLv3 or later
 * Requires at least: 3.0
 * Tested up to: 6.9
 * Requires PHP: 7.1
 * WC requires at least: 3.3
 * WC tested up to: 11
 * PHP version 7
 *
 * @category        Login_Redirect
 * @package         Sky_Login_Redirect
 * @author          Utopique <support@utopique.net>
 * @license         GPL https://utopique.net
 * @link            https://utopique.net
 */
namespace SkyLoginRedirect;

if ( !defined( 'ABSPATH' ) ) {
    exit;
    // Exit if accessed directly
}
// current version
define( 'SLR_VERSION', '3.8.0' );
// Plugin root path
define( "SLR_ROOT", trailingslashit( plugin_dir_path( __FILE__ ) ) );
/**
 * FS
 */
if ( function_exists( __NAMESPACE__ . '\\Sky_Login_Redirect_fs' ) ) {
    Sky_Login_Redirect_fs()->set_basename( false, __FILE__ );
} else {
    if ( !function_exists( __NAMESPACE__ . '\\Sky_Login_Redirect_fs' ) ) {
        /**
         * Create a helper function for easy SDK access.
         *
         * @return $Sky_Login_Redirect_fs
         */
        function Sky_Login_Redirect_fs() {
            global $Sky_Login_Redirect_fs;
            if ( !isset( $Sky_Login_Redirect_fs ) ) {
                /**
                 * Include Freemius SDK.
                 */
                include_once SLR_ROOT . '/vendor/freemius/wordpress-sdk/start.php';
                $Sky_Login_Redirect_fs = fs_dynamic_init( array(
                    'id'             => '3088',
                    'slug'           => 'sky-login-redirect',
                    'type'           => 'plugin',
                    'public_key'     => 'pk_f0e9c9d4e383120cf38d5b44b586b',
                    'is_premium'     => false,
                    'premium_suffix' => 'Pro',
                    'has_addons'     => false,
                    'has_paid_plans' => true,
                    'menu'           => array(
                        'slug' => 'sky-login-redirect',
                    ),
                    'is_live'        => true,
                ) );
            }
            return $Sky_Login_Redirect_fs;
        }

        // Init Freemius.
        Sky_Login_Redirect_fs();
        // Signal that SDK was initiated.
        do_action( 'Sky_Login_Redirect_fs_loaded' );
    }
    /**
     * Register the plugin text domain.
     *
     * WordPress 6.7+ expects translations to load on or after `init`.
     * If you bundle .mo files in /languages, this makes them available.
     *
     * @return void
     */
    function slr_load_plugin_textdomain() : void {
        load_plugin_textdomain( 'sky-login-redirect', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
    }

    add_action( 'init', __NAMESPACE__ . '\\slr_load_plugin_textdomain', 0 );
    /**
     * Charge notre dépendance Carbon Fields via Composer
     *
     * @return void
     */
    function Sky_Load_carbonfields() {
        include_once plugin_dir_path( __FILE__ ) . 'vendor/autoload.php';
        \Carbon_Fields\Carbon_Fields::boot();
        /**
         * Remove sidebar creation
         */
        if ( is_admin() ) {
            $sidebar_manager = \Carbon_Fields\Carbon_Fields::resolve( 'sidebar_manager' );
            remove_action( 'admin_enqueue_scripts', array($sidebar_manager, 'enqueue_scripts') );
        }
    }

    add_action( 'after_setup_theme', __NAMESPACE__ . '\\Sky_Load_carbonfields' );
    /**
     * Charge notre fichier de plugin
     *
     * @return void
     */
    function Sky_Load_plugin() {
        include_once plugin_dir_path( __FILE__ ) . 'includes/options.php';
    }

    add_action( 'plugins_loaded', __NAMESPACE__ . '\\Sky_Load_plugin' );
    include_once plugin_dir_path( __FILE__ ) . 'lib/admin/meta.php';
    //include_once plugin_dir_path(__FILE__) . 'lib/admin/core.php';
    //include_once plugin_dir_path(__FILE__) . 'lib/admin/settings.php';
    /**
     * Sky_Maybe_Is_ssl
     *
     * @return true if ssl is enabled
     */
    function Sky_Maybe_Is_ssl() : bool {
        if ( isset( $_SERVER['HTTPS'] ) && $_SERVER['HTTPS'] === 'on' ) {
            return true;
        }
        if ( isset( $_SERVER['HTTP_X_FORWARDED_PROTO'] ) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https' ) {
            return true;
        }
        return function_exists( 'is_ssl' ) && is_ssl();
    }

    /**
     * Determines if the current page is the WordPress login page.
     *
     * @return bool True if inside WordPress login page.
     */
    function is_login_page() : bool {
        global $pagenow;
        // Check if current page is wp-login.php
        if ( 'wp-login.php' === $pagenow ) {
            return true;
        }
        // Check if current page contains the [login-form] shortcode
        if ( function_exists( 'is_singular' ) && is_singular() ) {
            $post = get_post();
            if ( $post && function_exists( 'has_shortcode' ) && has_shortcode( (string) $post->post_content, 'login-form' ) ) {
                return true;
            }
        }
        return false;
    }

    /**
     * Set a cookie with the current page on all pages but the login page
     * If a login page, change the value of redirect_to with the last
     * cookie value
     *
     * @return void
     */
    function Sky_set_current_page_cookie() : void {
        if ( is_admin() || is_user_logged_in() || is_login_page() ) {
            return;
        }
        $req = filter_input( INPUT_SERVER, 'REQUEST_URI', FILTER_UNSAFE_RAW );
        $req = ( $req ? wp_unslash( (string) $req ) : '/' );
        $current_url = esc_url_raw( home_url( $req ) );
        $cookie_options = [
            'expires'  => time() + HOUR_IN_SECONDS,
            'path'     => '/',
            'secure'   => Sky_Maybe_Is_ssl(),
            'httponly' => true,
            'samesite' => 'Lax',
        ];
        // Only set a domain if WP defined a usable one.
        if ( defined( 'COOKIE_DOMAIN' ) && COOKIE_DOMAIN ) {
            $cookie_options['domain'] = COOKIE_DOMAIN;
        }
        setcookie( 'last_page_visited', $current_url, $cookie_options );
    }

    add_action( 'template_redirect', __NAMESPACE__ . '\\Sky_set_current_page_cookie' );
    /**
     * Set redirect field on login page to last visited page URL
     *
     * @return void
     */
    function Sky_set_login_redirect_field() : void {
        if ( !is_login_page() ) {
            return;
        }
        $redirect_to = Sky_get_last_page_visited_cookie();
        wp_register_script(
            'slr-inline',
            false,
            array(),
            ( defined( 'SLR_VERSION' ) ? SLR_VERSION : null ),
            true
        );
        wp_enqueue_script( 'slr-inline' );
        wp_add_inline_script( 'slr-inline', 'document.addEventListener("DOMContentLoaded",function(){var f=document.getElementById("redirect_to");if(f){f.value=' . wp_json_encode( esc_url_raw( $redirect_to ) ) . ';}});', 'after' );
    }

    add_action( 'login_enqueue_scripts', __NAMESPACE__ . '\\Sky_set_login_redirect_field' );
    /**
     * Clear login cookies
     *
     * @return void
     */
    function Sky_clear_cookies_on_logout() : void {
        global $wpdb;
        // Set the cookie expiration time to a year ago
        $expiration_time = time() - YEAR_IN_SECONDS;
        // Define the cookie prefixes to clear
        $cookie_prefixes = array(
            'wordpress_',
            'woocommerce_',
            $wpdb->prefix . 'woocommerce_',
            'comment_',
            'wp-postpass_',
            'wp-settings-',
            'wp-lang'
        );
        // Loop through the cookie prefixes and clear any matching cookies
        foreach ( $cookie_prefixes as $prefix ) {
            foreach ( $_COOKIE as $cookie_key => $cookie_value ) {
                if ( strpos( $cookie_key, $prefix ) === 0 ) {
                    $opts = array(
                        'expires'  => $expiration_time,
                        'path'     => '/',
                        'secure'   => Sky_Maybe_Is_ssl(),
                        'httponly' => true,
                        'samesite' => 'Lax',
                    );
                    if ( defined( 'COOKIE_DOMAIN' ) && COOKIE_DOMAIN ) {
                        $opts['domain'] = COOKIE_DOMAIN;
                    }
                    setcookie( $cookie_key, '', $opts );
                }
            }
        }
        // Clear-Site-Data header
        // https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Clear-Site-Data
        if ( !headers_sent() && apply_filters( 'slr_use_clear_site_data_on_logout', false ) ) {
            header( 'Clear-Site-Data: "cookies"' );
        }
    }

    add_action( 'wp_logout', __NAMESPACE__ . '\\Sky_clear_cookies_on_logout', PHP_INT_MAX );
    /**
     * Get last_page_visited cookie: this is our referer URL
     *
     * @return string The referer URL if it exists, otherwise an empty string.
     */
    function Sky_get_last_page_visited_cookie() : string {
        $raw_cookie = filter_input( INPUT_COOKIE, 'last_page_visited', FILTER_UNSAFE_RAW );
        if ( $raw_cookie ) {
            $raw_cookie = wp_unslash( (string) $raw_cookie );
            $referer_url = sanitize_url( $raw_cookie );
            return esc_url_raw( $referer_url );
        } else {
            return wp_get_referer();
        }
        return '';
    }

    /**
     * Removes the 'redirect_to' parameter from the logout URL.
     *
     * This function filters the WordPress logout URL to remove the 'redirect_to'
     * GET parameter, resulting in a cleaner URL.
     *
     * @param string $logout_url The original logout URL.
     * @param string $redirect   The redirect URL (not used in this function).
     * @return string The modified logout URL without the 'redirect_to' parameter.
     */
    function Slr_clean_logout_url(  $logout_url, $redirect  ) {
        return remove_query_arg( 'redirect_to', $logout_url );
    }

    add_filter(
        'logout_url',
        __NAMESPACE__ . '\\Slr_clean_logout_url',
        10,
        2
    );
    /**
     * Redirection for login and logout
     *
     * @param mixed $redirect_to           the redirect
     * @param mixed $requested_redirect_to the requested redirect
     * @param array $user                  the current user
     *
     * @return $redirect
     */
    function Slr_redirection(  $redirect_to, $requested_redirect_to, $user  ) {
        $slr_xselect_redirect = carbon_get_theme_option( 'slr_xlogin_logout' );
        // If a redirect_to parameter exists in the URL, honour it safely (internal/allowed hosts only)
        $default_redirect = admin_url( '/' );
        $requested_raw = filter_input( INPUT_GET, 'redirect_to', FILTER_UNSAFE_RAW );
        if ( $requested_raw ) {
            $requested = wp_unslash( (string) $requested_raw );
            return wp_validate_redirect( $requested, $default_redirect );
        }
        // bypass wp-admin for login
        // disabled in 3.6.0 as it was taking precedence over rules
        /*
        if (preg_match(
            "/wp-admin/",
            $requested_redirect_to
        ) && current_filter() === 'login_redirect') {
            $redirect_to = esc_url_raw($requested_redirect_to);
            return $redirect_to;
        }
        */
        // in case no rule were created, redirect all users to homepage
        if ( empty( $slr_xselect_redirect ) ) {
            return esc_url_raw( home_url( '/' ) );
        }
        foreach ( $slr_xselect_redirect as $rule ) {
            // user
            if ( $rule['slr_xselect_redirect'] == 'user' ) {
                //$current_user = wp_get_current_user();
                $current_user = $user;
                if ( $current_user && is_object( $current_user ) && is_a( $current_user, 'WP_User' ) ) {
                    foreach ( $rule['slr_xuser'] as $usr ) {
                        $usrID = $usr;
                        $str = $usrID;
                        $start = strpos( $str, '(' );
                        $end = strpos( $str, ')', $start + 1 );
                        $length = $end - $start;
                        $result = substr( $str, $start + 1, $length - 1 );
                        $result = str_replace( 'ID=', '', $result );
                        $arr[$result] = array(
                            'name'   => $usrID,
                            'login'  => $rule['slr_xlogin_url'],
                            'logout' => $rule['slr_xlogout_url'],
                        );
                        foreach ( $arr as $key => $value ) {
                            if ( $key == $current_user->ID ) {
                                $diff = str_replace( ['_redirect', 'wp_ajax_nopriv_ajax'], '', current_filter() );
                                // prior
                                if ( $rule["slr_xselect_{$diff}"] == 'prior' ) {
                                    $link = Slr_Get_Prior_url( $redirect_to, $requested_redirect_to, $user );
                                    return $link;
                                    break;
                                } elseif ( $rule["slr_xselect_{$diff}"] == 'page' ) {
                                    // page
                                    $link = get_permalink( $rule["slr_x{$diff}_page"] );
                                    return $link;
                                    break;
                                } elseif ( $rule["slr_xselect_{$diff}"] == 'custom' ) {
                                    // custom
                                    $link = $rule["slr_x{$diff}_url"];
                                    return $link;
                                    break;
                                } else {
                                    // WP defaults
                                    return admin_url( '/' );
                                    break;
                                }
                            }
                        }
                    }
                }
            }
            /* role */
            if ( $rule['slr_xselect_redirect'] == 'role' ) {
                foreach ( $rule['slr_xrole'] as $role ) {
                    //retrieve current user info
                    global $wp_roles;
                    //$current_user = wp_get_current_user();
                    $current_user = $user;
                    $roles = $wp_roles->roles;
                    //is there a user to check?
                    if ( isset( $current_user->roles ) && is_array( $current_user->roles ) ) {
                        if ( in_array( $current_user->roles[0], $rule['slr_xrole'] ) ) {
                            $diff = str_replace( ['_redirect', 'wp_ajax_nopriv_ajax'], '', current_filter() );
                            if ( $rule["slr_xselect_{$diff}"] == 'prior' ) {
                                // prior
                                $link = Slr_Get_Prior_url( $redirect_to, $requested_redirect_to, $user );
                                return $link;
                                break;
                            } elseif ( $rule["slr_xselect_{$diff}"] == 'page' ) {
                                // page
                                $link = get_permalink( $rule["slr_x{$diff}_page"] );
                                return $link;
                                break;
                            } elseif ( $rule["slr_xselect_{$diff}"] == 'custom' ) {
                                // custom
                                $link = $rule["slr_x{$diff}_url"];
                                return $link;
                                break;
                            } else {
                                // WP defaults
                                return admin_url( '/' );
                                break;
                            }
                        }
                    }
                }
            }
            // all
            if ( $rule['slr_xselect_redirect'] == 'all' ) {
                //$diff = str_replace( ['_redirect'], '', current_filter());
                $diff = str_replace( ['_redirect', 'wp_ajax_nopriv_ajax'], '', current_filter() );
                if ( $rule["slr_xselect_{$diff}"] == 'prior' ) {
                    // prior
                    $link = Slr_Get_Prior_url( $redirect_to, $requested_redirect_to, $user );
                    return $link;
                    break;
                } elseif ( $rule["slr_xselect_{$diff}"] == 'page' ) {
                    // page
                    $link = get_permalink( $rule["slr_x{$diff}_page"] );
                    return $link;
                    break;
                } elseif ( $rule["slr_xselect_{$diff}"] == 'custom' ) {
                    // custom
                    $link = $rule["slr_x{$diff}_url"];
                    return $link;
                    break;
                } else {
                    // WP defaults
                    return admin_url( '/' );
                    break;
                }
            }
        }
    }

    add_filter(
        'login_redirect',
        __NAMESPACE__ . '\\Slr_redirection',
        100,
        3
    );
    add_filter(
        'logout_redirect',
        __NAMESPACE__ . '\\Slr_redirection',
        100,
        3
    );
    /**
     * Get prior URL
     *
     * @param string $redirect_to           the redirect
     * @param string $requested_redirect_to the requested redirect
     * @param array  $user                  the current user
     *
     * @return $redirect_to
     */
    function Slr_Get_Prior_url(  ?string $redirect_to, ?string $requested_redirect_to, $user  ) : ?string {
        $referer = Sky_get_last_page_visited_cookie();
        /**
         * If the login page calls itself in $redirect_to,
         * avoid the loop and redirect to the homepage.
         * This would happen w/ password recovery and registration links.
         * It only concerns login, not logout
         * (otherwise it breaks the logout referer)
         */
        if ( preg_match( "/wp-login\\.php/", $redirect_to ) && current_filter() === 'login_redirect' ) {
            $redirect_to = home_url( '/' );
            return wp_validate_redirect( (string) $redirect_to, home_url( '/' ) );
            //return '';
            //this will blank out if user has clicked on register link
            //or password recovery links before landing on the login page
        }
        // bypass wp-admin for login
        if ( preg_match( "/wp-admin/", $requested_redirect_to ) && current_filter() === 'login_redirect' ) {
            $redirect_to = esc_url_raw( $requested_redirect_to );
            //return $redirect_to;
            return wp_validate_redirect( (string) $redirect_to, home_url( '/' ) );
        }
        // logout: if we log out from wp-admin, redirect to homepage
        if ( preg_match( "/wp-admin/", $referer ) && current_filter() === 'logout_redirect' ) {
            $redirect_to = home_url( '/' );
            return wp_validate_redirect( (string) $redirect_to, home_url( '/' ) );
        }
        // this bit is for when WP is installed in a subdirectory
        // if $referer equals the login page, get value from $redirect_to
        if ( preg_match( "/wp-login\\.php/", $referer ) && current_filter() === 'login_redirect' ) {
            $redirect_to = esc_url_raw( $redirect_to );
            return wp_validate_redirect( (string) $redirect_to, home_url( '/' ) );
        }
        // If the referer is empty, use $redirect_to as it is set via the cookie.
        if ( !$referer ) {
            return wp_validate_redirect( (string) $redirect_to, home_url( '/' ) );
        }
        // Otherwise, go back to the referring page.
        return wp_validate_redirect( (string) $referer, home_url( '/' ) );
    }

    /**
     * Shortcode : [login-logout]
     *
     * @return mixed
     */
    function Slr_Login_Logout_shortcode() {
        if ( is_user_logged_in() ) {
            // user is logged in, therefore display the logout link
            // If you want to redirect the user to the same page after logout,
            // then use this line below instead
            return '<a class="logout-btn slr-lilo-shortcode" href="' . esc_url( wp_logout_url() ) . '">' . esc_html__( 'Logout', 'sky-login-redirect' ) . '</a>';
        } else {
            // user is logged out, therefore display the login link
            return '<a class="login-btn slr-lilo-shortcode" href="' . esc_url( wp_login_url() ) . '">' . esc_html__( 'Login', 'sky-login-redirect' ) . '</a>';
        }
    }

    add_shortcode( 'login-logout', __NAMESPACE__ . '\\Slr_Login_Logout_shortcode' );
    // Execute shortcodes in widget_text
    add_filter( 'widget_text', 'do_shortcode' );
    /**
     * Register Login Customizer
     *
     * @return void
     */
    function Slr_Register_Login_customizer() {
        include_once plugin_dir_path( __FILE__ ) . 'includes/login-customizer.php';
    }

    add_action( 'carbon_fields_register_fields', __NAMESPACE__ . '\\Slr_Register_Login_customizer' );
    /**
     * Get cached options from transient
     *
     * @return void
     */
    function Slr_Get_Cached_options() : array {
        global $wpdb;
        // Try object cache first to avoid direct DB queries when possible.
        $cached = wp_cache_get( 'sky_login_redirect', 'slr' );
        if ( false !== $cached && is_array( $cached ) && $cached ) {
            return $cached;
        }
        $rows = $wpdb->get_results( $wpdb->prepare( "SELECT option_name, option_value FROM {$wpdb->options} WHERE option_name LIKE %s", $wpdb->esc_like( '_slr_' ) . '%' ), ARRAY_A );
        $cache = array();
        foreach ( $rows as $row ) {
            $cache[$row['option_name']] = maybe_unserialize( $row['option_value'] );
        }
        // Create with autoload=no, then update thereafter.
        if ( false === get_option( 'sky_login_redirect', false ) ) {
            add_option(
                'sky_login_redirect',
                $cache,
                '',
                false
            );
            // autoload=no
        } else {
            // Bust object cache before updating the option value.
            wp_cache_delete( 'sky_login_redirect', 'slr' );
            update_option( 'sky_login_redirect', $cache );
        }
        // Populate object cache for subsequent calls.
        wp_cache_set(
            'sky_login_redirect',
            $cache,
            'slr',
            HOUR_IN_SECONDS
        );
        return $cache;
    }

    add_action( 'carbon_fields_theme_options_container_saved', __NAMESPACE__ . '\\Slr_Get_Cached_options' );
    /**
     * Getter : retrieve cached options from transient
     *
     * @param mixed $key     the key to retrieve
     * @param bool  $default false by default
     *
     * @return mixed the value for a given key
     */
    function carbonade(  $key, $default = 'false'  ) {
        static $cache;
        if ( null === $cache ) {
            $cache = (array) get_option( 'sky_login_redirect', array() );
        }
        $k = '_' . $key;
        return ( array_key_exists( $k, $cache ) ? esc_html( $cache[$k] ) : $default );
    }

    /**
     * Declare HPOS (High-Performance Order Storage) compatibility for Sky Login Redirect plugin.
     *
     * This function declares that the plugin is compatible with WooCommerce's custom order tables (HPOS).
     * It's important to declare compatibility even if the plugin doesn't directly interact with WooCommerce tables,
     * as it informs WooCommerce and store owners that this plugin won't interfere with HPOS functionality.
     *
     * @since 3.7.5
     *
     * @return void
     */
    add_action( 'before_woocommerce_init', function () {
        if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
            \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
        }
    } );
    /**
     * Enqueue scripts
     *
     * @param mixed $hook the current admin hook
     *
     * @return mixed styles and scripts
     */
    function Slr_Add_scripts(  $hook  ) {
        $svg = plugins_url( 'lib/img/sky-login-redirect.svg', __FILE__ );
        $handle = 'slr-admin-css';
        wp_register_style(
            $handle,
            false,
            array(),
            SLR_VERSION
        );
        // empty style as a target for inline CSS
        wp_enqueue_style( $handle );
        wp_add_inline_style( $handle, "\n:root{\n\t--slr-accent:#a2ff37;\n\t--slr-icon-size:16px;\n}\n\n#toplevel_page_sky-login-redirect .wp-menu-image{\n\tbackground-color:var(--slr-accent);\n\tmask-image:url('{$svg}');\n\tmask-repeat:no-repeat;\n\tmask-position:center;\n\tmask-size:var(--slr-icon-size);\n\t-webkit-mask-image:url('{$svg}');\n\t-webkit-mask-repeat:no-repeat;\n\t-webkit-mask-position:center;\n\t-webkit-mask-size:var(--slr-icon-size);\n\ttransition:transform .9s ease;\n\twill-change:transform;\n}\n\n/* Hover / focus rotates the icon. */\n#toplevel_page_sky-login-redirect:hover .wp-menu-image,\n#toplevel_page_sky-login-redirect .wp-menu-image:hover,\na.toplevel_page_sky-login-redirect:hover .wp-menu-image{\n\ttransform:rotate(180deg);\n}\n\n/* Respect reduced motion: no transition. */\n@media (prefers-reduced-motion: reduce){\n\t#toplevel_page_sky-login-redirect .wp-menu-image{\n\t\ttransition:none;\n\t}\n}\n\n/* Fallback when CSS masks aren't supported. */\n@supports not ((mask-image:url('')) or (-webkit-mask-image:url(''))){\n\t#toplevel_page_sky-login-redirect .wp-menu-image{\n\t\tbackground-color:transparent;\n\t\tbackground-image:url('{$svg}');\n\t\tbackground-repeat:no-repeat;\n\t\tbackground-position:center;\n\t\tbackground-size:var(--slr-icon-size);\n\t}\n}" );
        $array = [
            'toplevel_page_sky-login-redirect',
            'login-redirect_page_sky-login-redirect-account',
            'login-redirect_page_sky-login-redirect-contact',
            'login-redirect_page_sky-login-redirect-pricing'
        ];
        if ( in_array( $hook, $array ) ) {
            // Only on our plugin page
            if ( $hook === $array[0] ) {
                wp_enqueue_style(
                    'utopique-elements',
                    plugins_url( 'lib/css/elements.css', __FILE__ ),
                    false,
                    SLR_VERSION,
                    'all'
                );
                wp_enqueue_style(
                    'slr',
                    plugins_url( 'lib/css/slr.css', __FILE__ ),
                    false,
                    SLR_VERSION,
                    'all'
                );
                wp_enqueue_script(
                    'slr-js',
                    plugins_url( 'lib/js/slr.js', __FILE__ ),
                    ['jquery', 'underscore', 'code-editor'],
                    SLR_VERSION,
                    true
                );
                wp_localize_script( 'slr-js', 'SLR', array(
                    'upgrade_url'      => Sky_Login_Redirect_fs()->get_upgrade_url(),
                    'pro_feature'      => __( 'unlock with Pro version', 'sky-login-redirect' ),
                    'business_feature' => __( 'unlock with Business version', 'sky-login-redirect' ),
                    'iconsBase'        => trailingslashit( plugins_url( 'lib/img/icons', __FILE__ ) ),
                ) );
                // Codemirror editor
                $cm_css['codeEditor'] = wp_enqueue_code_editor( [
                    'type' => 'text/css',
                ] );
                wp_localize_script( 'jquery', 'cm_css', $cm_css );
                $cm_js['codeEditor'] = wp_enqueue_code_editor( [
                    'type' => 'text/html',
                ] );
                wp_localize_script( 'jquery', 'cm_js', $cm_js );
                wp_enqueue_script( 'wp-theme-plugin-editor' );
                wp_enqueue_style( 'wp-codemirror' );
                // remove WP emoji on our pages
                remove_action( 'admin_print_scripts', 'print_emoji_detection_script' );
                remove_action( 'admin_print_styles', 'print_emoji_styles' );
            }
            return;
        }
    }

    add_action( 'admin_enqueue_scripts', __NAMESPACE__ . '\\Slr_Add_scripts' );
}
// FS endif