<?php
/**
 * Notices
 * PHP version 7
 *
 * @category Notices
 * @package  Sky_Login_Redirect
 * @author   Utopique <support@utopique.net>
 * @license  GPL https://utopique.net
 * @link     https://utopique.net
 */

namespace SkyLoginRedirect\Notices;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

/**
 * Add plugin activation time
 *
 * @return null
 */
function Slr_Activation_time()
{
    $get_activation_time = strtotime("now");
    add_option('SLR_activation_time', $get_activation_time);
}
register_activation_hook(dirname(__DIR__) . '/sky-login-redirect.php', __NAMESPACE__ . '\\Slr_Activation_time');

/**
 * Slr_Add_Notices_script
 *
 * Add JS script to record dismiss action
 *
 * @return file admin-notices.js
 */
function Slr_Add_Notices_script()
{
    wp_register_script(
        'slr-notice-update',
        plugins_url('lib/js/admin-notices.js', dirname(__DIR__) . '/sky-login-redirect.php'),
        ['jquery'],
        SLR_VERSION,
        true
    );
    wp_localize_script(
        'slr-notice-update',
        'notice_params',
        array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce('slr_notice'),
        )
    );
    wp_enqueue_script('slr-notice-update');
}
add_action('admin_enqueue_scripts', __NAMESPACE__ . '\\Slr_Add_Notices_script');

/**
 * Record dismiss
 *
 * Action hook (wp_ajax_SLR_EOY_2020) must correspond to (wp_ajax_)
 * And also to the action name in the JS file (SLR_EOY_2020)
 *
 * @return null
 */
function Slr_Dismiss_notice()
{
    check_ajax_referer('slr_notice');

    if (! current_user_can('manage_options')) {
        wp_send_json_error(['message' => 'forbidden'], 403);
    }

    update_option('SLR_EOY_2020', true);

    wp_send_json_success();
}
add_action('wp_ajax_SLR_EOY_2020', __NAMESPACE__ . '\\Slr_Dismiss_notice');

/**
 * Check if notice should be shown or not
 *
 * Wait at least 48 hours before displaying the notice
 *
 * @return null
 */
function Slr_Check_Installation_time()
{
    $install_date = get_option('SLR_activation_time');
    $past_date = strtotime('-2 days');
    if ($past_date >= $install_date) {
        // if no record set, display the notice
        if (get_option('SLR_EOY_2020') != true) {
            add_action('admin_notices', __NAMESPACE__ . '\\Slr_EOY_Admin_notice');
        }
    }
}
add_action('admin_init', __NAMESPACE__ . '\\Slr_Check_Installation_time');

/**
 * Slr_EOY_Admin_notice
 *
 * @return void
 */
function Slr_EOY_Admin_notice()
{
    $current_screen = get_current_screen();
    if ( ! $current_screen ) {
        return;
    }
    $hook = $current_screen->id;
    $array = [
        'toplevel_page_sky-login-redirect',
        'login-redirect_page_sky-login-redirect-account',
        'login-redirect_page_sky-login-redirect-pricing',
        'plugins',
        'dashboard',
    ];
    $SLR_fs = Sky_Login_Redirect_fs();

    if ( in_array( $hook, $array, true )
        && gmdate( 'Ymd' ) < '20210101' // end of year sale
        && ( $SLR_fs->is_not_paying() || $SLR_fs->is_free_plan() )
    ) {
        // class name is important as it's detected by the JS script: EOY-2020
        $notice_html = sprintf(
            '<div class="EOY-2020 notice notice-info is-dismissible"><p>%1$s <a href="%2$s" target="_blank" rel="noopener noreferrer">%3$s</a>.</p></div>',
            // translators: %s: the discount amount to save (e.g., "20%")
            sprintf( esc_html__( 'End of year sale: save %s on', 'sky-login-redirect' ), '20%' ),
            esc_url( 'https://utopique.net/products/sky-login-redirect-premium/' ),
            esc_html__( 'Sky Login Redirect Pro', 'sky-login-redirect' )
        );
        echo wp_kses_post( $notice_html );
    }
}
