<?php

if ( ! class_exists( 'SKSocialMedia' ) ) :

	/**
	 * SKSocialMedia class.
	 *
	 * @since 1.4
	*/
	class SKSocialMedia {

		/**
		 * List of social media profiles.
		 *
		 * @since 1.4
		 * @var array
		 */
		protected $social_media_profiles = array();

		/**
		 * The single instance of the class.
		 *
		 * @since 1.4
		 * @var SKSocialMedia
		*/
		protected static $_instance = null;

		/**
		 * SKSocialMedia constructor.
		 *
		 * @since 1.4
		*/
		public function __construct() {

			$this->set_profiles();

			if( !get_option( 'social_media_options_import', false ) ) {
				$done_import = $this->import_options();
				update_option( 'social_media_options_import', true );
			}

			if( !get_option( 'social_media_repeater_option_import', false ) ) {
				$this->set_repeater_default();
				update_option( 'social_media_repeater_option_import', true );
			}

			$this->enqueue_styles();
			$this->customizer_options();
			$this->create_shortcode();

			if ( defined(  'WPB_VC_VERSION' ) ) {
				$this->create_wb_element();
			}

			add_action( 'footer_socials', function() {
				if( sk_string_to_bool( get_option( 'sk_footer_social_icons', 'yes' ) ) ) {
					echo '<div class="footer_socials_wrapper">' . do_shortcode('[social-media items_align="center"]') . '</div>';
				}
			} );

			add_action( 'header_socials', function() {
				if( sk_string_to_bool( get_option( 'sk_top_bar_social_icons', 'no' ) ) ) {
					echo '<div class="site-top-bar-social-icons-wrapper">' . do_shortcode('[social-media items_align="right"]') . '</div>';
				}
			} );
		}

		/**
		 * Ensures only one instance of SKSocialMedia is loaded or can be loaded.
		 *
		 * @since 1.4
		 *
		 * @return SKSocialMedia
		*/
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Enqueue styles.
		 *
		 * @since 1.4
		 * @return void
		*/
		protected function enqueue_styles() {
			add_action( 'wp_enqueue_scripts', function() {
				wp_enqueue_style('sk-social-media-styles', plugins_url( 'assets/css/social-media'.SK_EXT_ENQUEUE_SUFFIX.'.css', __FILE__ ), NULL );
			});
		}

		/**
		 * Imports social media links stored as theme mods into the options WP table.
		 *
		 * @since 1.4
		 * @return void
		 */
		private function import_options() {

			foreach( $this->social_media_profiles as $social) {
				if( get_theme_mod( $social['link'] ) ) {
					update_option( 'sk_' . $social['link'], get_theme_mod( $social['link'], '' ) );
				}

				/*if( get_theme_mod( $social['link_title'] ) ) {
					update_option( 'sk_' . $social['link_title'], get_theme_mod( $social['link_title'], '' ) );
				}*/
			}

			update_option( 'sk_top_bar_social_icons', get_theme_mod( 'top_bar_social_icons', false ) );
			update_option( 'sk_footer_social_icons', get_theme_mod( 'footer_social_icons', true ) );
		}

		/**
		 * Set new repeater option default based on old options.
		 *
		 * @since 1.5.9
		 * @return void
		 */
		protected function set_repeater_default() {

			$repeater_default = array();
			foreach( $this->social_media_profiles as $social) {
				if( !empty( get_option( 'sk_' . $social['link'] ) ) ) {
					$repeater_default[] = array(
						'icon_slug' => $social['slug'],
						'link' => get_option( 'sk_' . $social['link'] ),
						'link_title' => get_option( 'sk_' . $social['link_title'] ),
						'title' => $social['name'],
					);
				}
			}

			update_option( 'sk_social_media_repeater', json_encode( $repeater_default ) );

			return;
		}

		/**
		 * Sets the social media profile array.
		 *
		 * @since 1.4
		 * @return void
		 */
		protected function set_profiles() {

			$this->social_media_profiles = array(
			    array(
			        'link' 		=> 'facebook_link',
			        'slug' 		=> 'facebook',
			        'svg_path'  => 'M32,11h5c0.552,0,1-0.448,1-1V3.263c0-0.524-0.403-0.96-0.925-0.997C35.484,2.153,32.376,2,30.141,2C24,2,20,5.68,20,12.368 V19h-7c-0.552,0-1,0.448-1,1v7c0,0.552,0.448,1,1,1h7v19c0,0.552,0.448,1,1,1h7c0.552,0,1-0.448,1-1V28h7.222 c0.51,0,0.938-0.383,0.994-0.89l0.778-7C38.06,19.518,37.596,19,37,19h-8v-5C29,12.343,30.343,11,32,11z',
			        'name' 		=> 'Facebook'
			    ),
			    array(
			        'link' 		=> 'facebook_messenger_link',
			        'slug' 		=> 'facebook-messenger',
			        'svg_path'	=> 'M 25 2 C 12.300781 2 2 11.601563 2 23.5 C 2 29.800781 4.898438 35.699219 10 39.800781 L 10 48.601563 L 18.601563 44.101563 C 20.699219 44.699219 22.800781 44.898438 25 44.898438 C 37.699219 44.898438 48 35.300781 48 23.398438 C 48 11.601563 37.699219 2 25 2 Z M 27.300781 30.601563 L 21.5 24.398438 L 10.699219 30.5 L 22.699219 17.800781 L 28.601563 23.699219 L 39.101563 17.800781 Z ',
			        'name' 		=> 'Facebook Messenger'
			    ),
			    array(
			        'link' 		=> 'twitter_link',
			        'slug' 		=> 'twitter',
			        'svg_path'	=> 'm28.34828,22.30743l13.09073,-14.9014l-3.10099,0l-11.37148,12.93604l-9.07562,-12.93604l-10.47024,0l13.72711,19.56344l-13.72711,15.6245l3.10099,0l12.00082,-13.66382l9.58659,13.66382l10.47024,0m-30.93843,-32.89794l4.764,0l21.93079,30.72045l-4.76517,0',
			        'name' 		=> 'Twitter'
			    ),
			    array(
			        'link' 		=> 'pinterest_link',
			        'slug' 		=> 'pinterest',
			        'svg_path'	=> 'M25,2C12.318,2,2,12.317,2,25s10.318,23,23,23s23-10.317,23-23S37.682,2,25,2z M27.542,32.719 c-3.297,0-4.516-2.138-4.516-2.138s-0.588,2.309-1.021,3.95s-0.507,1.665-0.927,2.591c-0.471,1.039-1.626,2.674-1.966,3.177 c-0.271,0.401-0.607,0.735-0.804,0.696c-0.197-0.038-0.197-0.245-0.245-0.678c-0.066-0.595-0.258-2.594-0.166-3.946 c0.06-0.88,0.367-2.371,0.367-2.371l2.225-9.108c-1.368-2.807-0.246-7.192,2.871-7.192c2.211,0,2.79,2.001,2.113,4.406 c-0.301,1.073-1.246,4.082-1.275,4.224c-0.029,0.142-0.099,0.442-0.083,0.738c0,0.878,0.671,2.672,2.995,2.672 c3.744,0,5.517-5.535,5.517-9.237c0-2.977-1.892-6.573-7.416-6.573c-5.628,0-8.732,4.283-8.732,8.214 c0,2.205,0.87,3.091,1.273,3.577c0.328,0.395,0.162,0.774,0.162,0.774l-0.355,1.425c-0.131,0.471-0.552,0.713-1.143,0.368 C15.824,27.948,13,26.752,13,21.649C13,16.42,17.926,11,25.571,11C31.64,11,37,14.817,37,21.001 C37,28.635,32.232,32.719,27.542,32.719z',
			        'name' 		=> 'Pinterest'
			    ),
				array(
			        'link' 		=> 'instagram_link',
			        'slug' 		=> 'instagram',
			        'svg_path'	=> 'M 16 3 C 8.83 3 3 8.83 3 16 L 3 34 C 3 41.17 8.83 47 16 47 L 34 47 C 41.17 47 47 41.17 47 34 L 47 16 C 47 8.83 41.17 3 34 3 L 16 3 z M 37 11 C 38.1 11 39 11.9 39 13 C 39 14.1 38.1 15 37 15 C 35.9 15 35 14.1 35 13 C 35 11.9 35.9 11 37 11 z M 25 14 C 31.07 14 36 18.93 36 25 C 36 31.07 31.07 36 25 36 C 18.93 36 14 31.07 14 25 C 14 18.93 18.93 14 25 14 z M 25 16 C 20.04 16 16 20.04 16 25 C 16 29.96 20.04 34 25 34 C 29.96 34 34 29.96 34 25 C 34 20.04 29.96 16 25 16 z',
			        'name' 		=> 'Instagram'
			    ),
			    array(
			        'link' 		=> 'youtube_link',
			        'slug' 		=> 'youtube',
			        'svg_path'	=> 'M 44.898438 14.5 C 44.5 12.300781 42.601563 10.699219 40.398438 10.199219 C 37.101563 9.5 31 9 24.398438 9 C 17.800781 9 11.601563 9.5 8.300781 10.199219 C 6.101563 10.699219 4.199219 12.199219 3.800781 14.5 C 3.398438 17 3 20.5 3 25 C 3 29.5 3.398438 33 3.898438 35.5 C 4.300781 37.699219 6.199219 39.300781 8.398438 39.800781 C 11.898438 40.5 17.898438 41 24.5 41 C 31.101563 41 37.101563 40.5 40.601563 39.800781 C 42.800781 39.300781 44.699219 37.800781 45.101563 35.5 C 45.5 33 46 29.398438 46.101563 25 C 45.898438 20.5 45.398438 17 44.898438 14.5 Z M 19 32 L 19 18 L 31.199219 25 Z ',
			        'name' 		=> 'YouTube'
			    ),
			    array(
			        'link' 		=> 'linkedin_link',
			        'slug' 		=> 'linkedin',
			        'svg_path'	=> 'M41,4H9C6.24,4,4,6.24,4,9v32c0,2.76,2.24,5,5,5h32c2.76,0,5-2.24,5-5V9C46,6.24,43.76,4,41,4z M17,20v19h-6V20H17z M11,14.47c0-1.4,1.2-2.47,3-2.47s2.93,1.07,3,2.47c0,1.4-1.12,2.53-3,2.53C12.2,17,11,15.87,11,14.47z M39,39h-6c0,0,0-9.26,0-10 c0-2-1-4-3.5-4.04h-0.08C27,24.96,26,27.02,26,29c0,0.91,0,10,0,10h-6V20h6v2.56c0,0,1.93-2.56,5.81-2.56 c3.97,0,7.19,2.73,7.19,8.26V39z',
			        'name' 		=> 'LinkedIn'
			    ),
			    array(
			        'link' 		=> 'tiktok_link',
			        'slug' 		=> 'tiktok',
			        'svg_path'	=> 'm37.80194,14.3998a8.94064,8.94064 0 0 1 -0.77321,-0.45002a10.87163,10.87163 0 0 1 -1.98581,-1.68716c-1.48097,-1.69452 -2.03408,-3.41359 -2.23782,-4.61719l0.00818,0c-0.17019,-0.99904 -0.09982,-1.64543 -0.08919,-1.64543l-6.74537,0l0,26.08306c0,0.3502 0,0.6963 -0.01473,1.03832c0,0.04255 -0.00409,0.08182 -0.00655,0.12764c0,0.01882 0,0.03846 -0.00409,0.05809c0,0.00491 0,0.00982 0,0.01473a5.72751,5.72751 0 0 1 -2.88175,4.546a5.62932,5.62932 0 0 1 -2.79093,0.73639c-3.14276,0 -5.68987,-2.56265 -5.68987,-5.72751s2.5471,-5.72751 5.68987,-5.72751a5.6375,5.6375 0 0 1 1.7518,0.27737l0.00818,-6.8681a12.53015,12.53015 0 0 0 -9.65494,2.82448a13.2379,13.2379 0 0 0 -2.8883,3.56169c-0.28474,0.49093 -1.35906,2.46365 -1.48915,5.66532c-0.08182,1.81726 0.46393,3.69997 0.72412,4.47809l0,0.01636c0.16364,0.4582 0.79776,2.02181 1.83117,3.33995a13.70756,13.70756 0 0 0 2.92185,2.75657l0,-0.01636l0.01636,0.01636c3.2655,2.219 6.8861,2.07336 6.8861,2.07336c0.62675,-0.02536 2.72629,0 5.11057,-1.12996c2.64447,-1.25269 4.14999,-3.11904 4.14999,-3.11904a12.96544,12.96544 0 0 0 2.26155,-3.75806c0.61039,-1.60452 0.81412,-3.52896 0.81412,-4.29808l0,-13.83766c0.08182,0.04909 1.17168,0.76994 1.17168,0.76994s1.57015,1.0064 4.01989,1.6618c1.75753,0.46638 4.12544,0.56457 4.12544,0.56457l0,-6.69627c-0.82967,0.09 -2.51438,-0.17183 -4.23917,-1.03177z',
			        'name' 		=> 'TikTok'
			    ),
			    array(
			        'link' 		=> 'whatsapp_link',
			        'slug' 		=> 'whatsapp',
			        'svg_path'	=> 'M25,2C12.318,2,2,12.318,2,25c0,3.96,1.023,7.854,2.963,11.29L2.037,46.73c-0.096,0.343-0.003,0.711,0.245,0.966 C2.473,47.893,2.733,48,3,48c0.08,0,0.161-0.01,0.24-0.029l10.896-2.699C17.463,47.058,21.21,48,25,48c12.682,0,23-10.318,23-23 S37.682,2,25,2z M36.57,33.116c-0.492,1.362-2.852,2.605-3.986,2.772c-1.018,0.149-2.306,0.213-3.72-0.231 c-0.857-0.27-1.957-0.628-3.366-1.229c-5.923-2.526-9.791-8.415-10.087-8.804C15.116,25.235,13,22.463,13,19.594 s1.525-4.28,2.067-4.864c0.542-0.584,1.181-0.73,1.575-0.73s0.787,0.005,1.132,0.021c0.363,0.018,0.85-0.137,1.329,1.001 c0.492,1.168,1.673,4.037,1.819,4.33c0.148,0.292,0.246,0.633,0.05,1.022c-0.196,0.389-0.294,0.632-0.59,0.973 s-0.62,0.76-0.886,1.022c-0.296,0.291-0.603,0.606-0.259,1.19c0.344,0.584,1.529,2.493,3.285,4.039 c2.255,1.986,4.158,2.602,4.748,2.894c0.59,0.292,0.935,0.243,1.279-0.146c0.344-0.39,1.476-1.703,1.869-2.286 s0.787-0.487,1.329-0.292c0.542,0.194,3.445,1.604,4.035,1.896c0.59,0.292,0.984,0.438,1.132,0.681 C37.062,30.587,37.062,31.755,36.57,33.116z',
			        'name' 		=> 'WhatsApp'
			    ),
			    array(
			        'link' 		=> 'googleplus_link',
			        'slug' 		=> 'googleplus',
			        'svg_path'	=> 'M 25.996094 48 C 13.3125 48 2.992188 37.683594 2.992188 25 C 2.992188 12.316406 13.3125 2 25.996094 2 C 31.742188 2 37.242188 4.128906 41.488281 7.996094 L 42.261719 8.703125 L 34.675781 16.289063 L 33.972656 15.6875 C 31.746094 13.78125 28.914063 12.730469 25.996094 12.730469 C 19.230469 12.730469 13.722656 18.234375 13.722656 25 C 13.722656 31.765625 19.230469 37.269531 25.996094 37.269531 C 30.875 37.269531 34.730469 34.777344 36.546875 30.53125 L 24.996094 30.53125 L 24.996094 20.175781 L 47.546875 20.207031 L 47.714844 21 C 48.890625 26.582031 47.949219 34.792969 43.183594 40.667969 C 39.238281 45.53125 33.457031 48 25.996094 48 Z ',
			        'name' 		=> 'Google'
			    ),
			    array(
			        'link' 		=> 'tumblr_link',
			        'slug' 		=> 'tumblr',
			        'svg_path'	=> 'M 40 0 L 10 0 C 4.484375 0 0 4.484375 0 10 L 0 40 C 0 45.515625 4.484375 50 10 50 L 40 50 C 45.515625 50 50 45.515625 50 40 L 50 10 C 50 4.484375 45.515625 0 40 0 Z M 34 40.238281 C 34 40.363281 33.945313 40.480469 33.855469 40.5625 C 33.738281 40.664063 31.011719 43 24.742188 43 C 17.230469 43 17 34.617188 17 33.664063 L 17 23.011719 L 13.429688 23 C 13.191406 23 13 22.816406 13 22.578125 L 13 18.808594 C 13 18.632813 13.109375 18.472656 13.273438 18.40625 C 13.34375 18.382813 20.058594 15.773438 20.058594 9.429688 C 20.058594 9.191406 20.253906 9 20.492188 9 L 24.578125 9 C 24.816406 9 25.007813 9.191406 25.007813 9.429688 L 25 17 L 31.5625 17 C 31.800781 17 31.992188 17.207031 31.992188 17.445313 L 31.992188 22.554688 C 31.992188 22.789063 31.800781 23 31.5625 23 L 25 23 C 25 23 25 33.253906 25 33.503906 C 25 33.75 25.226563 36.765625 28.433594 36.765625 C 31.089844 36.765625 33.320313 35.398438 33.34375 35.382813 C 33.476563 35.296875 33.640625 35.292969 33.777344 35.371094 C 33.914063 35.445313 34 35.589844 34 35.746094 Z ',
			        'name' 		=> 'Tumblr'
			    ),
			    array(
			        'link' 		=> 'vimeo_link',
			        'slug' 		=> 'vimeo',
			        'svg_path'	=> 'M 41 5 C 34.210938 4.992188 30.46875 8.796875 28.167969 16.210938 C 29.371094 15.765625 30.578125 15.214844 31.671875 15.214844 C 33.972656 15.214844 34.738281 16.070313 34.410156 18.726563 C 34.300781 20.386719 33.644531 23.066406 31.671875 26.164063 C 29.699219 29.152344 27.984375 30 27 30 C 25.796875 30 24.882813 28.269531 23.898438 23.621094 C 23.570313 22.292969 22.804688 19.304688 21.925781 13.664063 C 21.160156 8.464844 18.613281 5.667969 15 6 C 13.46875 6.109375 11.636719 7.535156 8.570313 10.191406 C 6.378906 12.183594 4.300781 13.621094 2 15.613281 L 4.191406 18.421875 C 6.269531 16.984375 7.476563 16.429688 7.804688 16.429688 C 9.335938 16.429688 10.757813 18.863281 12.183594 23.84375 C 13.386719 28.378906 14.699219 32.914063 15.90625 37.449219 C 17.765625 42.429688 20.066406 44.863281 22.695313 44.863281 C 27.074219 44.863281 32.328125 40.882813 38.570313 32.695313 C 44.699219 24.949219 47.78125 18.535156 48 14 C 48.21875 8.027344 45.816406 5.109375 41 5 Z ',
			        'name' 		=> 'Vimeo'
			    ),
			    array(
			        'link' 		=> 'behance_link',
			        'slug' 		=> 'behace',
			        'svg_path'	=> 'M 9 4 C 6.24 4 4 6.24 4 9 L 4 41 C 4 43.76 6.24 46 9 46 L 41 46 C 43.76 46 46 43.76 46 41 L 46 9 C 46 6.24 43.76 4 41 4 L 9 4 z M 12 18 L 18.730469 18 C 19.460469 18 23.410156 17.950312 23.410156 21.570312 C 23.410156 23.490313 22.099766 24.139688 21.509766 24.429688 C 22.389766 24.709688 24 25.52 24 28 C 24 31.83 19.609531 32 19.019531 32 L 12 32 L 12 18 z M 29 18 L 36 18 L 36 20 L 29 20 L 29 18 z M 15 20.429688 L 15 23.710938 L 18.220703 23.710938 C 18.660703 23.710938 20.119141 23.47 20.119141 22 C 20.119141 20.53 18.219687 20.429688 17.929688 20.429688 L 15 20.429688 z M 32.730469 21 C 36.630469 21 37.689609 24.039766 37.849609 24.759766 C 37.999609 25.489766 38 26.13 38 27 L 30.099609 27 C 30.099609 27.87 30.560625 29.830078 32.890625 29.830078 C 33.510625 29.830078 33.969453 29.680625 34.439453 29.390625 C 34.899453 29.100625 35.060938 28.819297 35.210938 28.529297 L 37.839844 28.529297 C 37.379844 29.679297 36.760078 30.550859 35.830078 31.130859 C 34.900078 31.710859 33.820078 32 32.580078 32 C 31.800078 32 31.03 31.850547 30.25 31.560547 C 29.63 31.270547 29.010781 30.840156 28.550781 30.410156 C 28.090781 29.970156 27.780703 29.389922 27.470703 28.669922 C 27.160703 28.089922 27 27.22 27 26.5 C 27 25.78 27.290469 21 32.730469 21 z M 32.730469 23.029297 C 30.470469 23.029297 30.099609 25.199844 30.099609 25.339844 L 35.060547 25.339844 C 34.900547 24.619844 34.250469 23.029297 32.730469 23.029297 z M 15 25.710938 L 15 29.570312 L 18.351562 29.570312 C 18.640563 29.570312 20.679688 29.480937 20.679688 27.710938 C 20.679687 25.950937 19.077562 25.710938 18.351562 25.710938 L 15 25.710938 z',
			        'name' 		=> 'Behance'
			    ),
			    array(
			        'link' 		=> 'dribbble_link',
			        'slug' 		=> 'dribbble',
			        'svg_path'	=> 'M 25 2 C 12.317 2 2 12.318 2 25 C 2 37.682 12.317 48 25 48 C 37.683 48 48 37.682 48 25 C 48 12.318 37.683 2 25 2 z M 26 11 C 28.504 11 30 16.593 30 22 C 30 25.983 29.075875 30.095344 27.296875 33.152344 C 27.912875 33.728344 28.516 34 29 34 C 30.946 34 32.212781 30.200547 33.050781 27.685547 C 33.225781 27.160547 33.797406 26.881734 34.316406 27.052734 C 34.840406 27.227734 35.122266 27.792406 34.947266 28.316406 C 33.866266 31.559406 32.388 36 29 36 C 28.121 36 27.133594 35.60225 26.183594 34.78125 C 24.593594 36.75625 22.522 38 20 38 C 17.11 38 14 35.81 14 31 C 14 25.474 17.771 24 20 24 C 20.542 24 21.321703 24.145422 22.095703 24.482422 C 22.032703 23.697422 22 22.871 22 22 C 22 20.874 22.112 11 26 11 z M 25.939453 13.007812 C 25.239453 13.200813 24 16.661 24 22 C 24 23.706 24.131469 25.211344 24.355469 26.527344 C 24.375469 26.594344 24.387578 26.662469 24.392578 26.730469 C 24.755578 28.743469 25.338094 30.302172 25.996094 31.451172 C 27.315094 28.812172 28 25.354 28 22 C 28 16.516 26.565453 13.424812 25.939453 13.007812 z M 20 26 C 19.332 26 16 26.241 16 31 C 16 34.69 18.154 36 20 36 C 21.971 36 23.598422 34.961734 24.857422 33.302734 C 23.834422 31.867734 22.956125 29.844547 22.453125 27.185547 C 21.909125 26.421547 20.686 26 20 26 z',
			        'name' 		=> 'Dribbble'
			    ),
			    array(
			        'link' 		=> 'flickr_link',
			        'slug' 		=> 'flickr',
			        'svg_path'	=> 'M 9 4 C 6.2504839 4 4 6.2504839 4 9 L 4 41 C 4 43.749516 6.2504839 46 9 46 L 41 46 C 43.749516 46 46 43.749516 46 41 L 46 9 C 46 6.2504839 43.749516 4 41 4 L 9 4 z M 9 6 L 41 6 C 42.668484 6 44 7.3315161 44 9 L 44 41 C 44 42.668484 42.668484 44 41 44 L 9 44 C 7.3315161 44 6 42.668484 6 41 L 6 9 C 6 7.3315161 7.3315161 6 9 6 z M 16 17 C 11.59 17 8 20.59 8 25 C 8 29.41 11.59 33 16 33 C 20.41 33 24 29.41 24 25 C 24 20.59 20.41 17 16 17 z M 34 17 C 29.59 17 26 20.59 26 25 C 26 29.41 29.59 33 34 33 C 38.41 33 42 29.41 42 25 C 42 20.59 38.41 17 34 17 z',
			        'name' 		=> 'Flickr'
			    ),
			    array(
			        'link' 		=> 'git_link',
			        'slug' 		=> 'github',
			        'svg_path'	=> 'M17.791,46.836C18.502,46.53,19,45.823,19,45v-5.4c0-0.197,0.016-0.402,0.041-0.61C19.027,38.994,19.014,38.997,19,39 c0,0-3,0-3.6,0c-1.5,0-2.8-0.6-3.4-1.8c-0.7-1.3-1-3.5-2.8-4.7C8.9,32.3,9.1,32,9.7,32c0.6,0.1,1.9,0.9,2.7,2c0.9,1.1,1.8,2,3.4,2 c2.487,0,3.82-0.125,4.622-0.555C21.356,34.056,22.649,33,24,33v-0.025c-5.668-0.182-9.289-2.066-10.975-4.975 c-3.665,0.042-6.856,0.405-8.677,0.707c-0.058-0.327-0.108-0.656-0.151-0.987c1.797-0.296,4.843-0.647,8.345-0.714 c-0.112-0.276-0.209-0.559-0.291-0.849c-3.511-0.178-6.541-0.039-8.187,0.097c-0.02-0.332-0.047-0.663-0.051-0.999 c1.649-0.135,4.597-0.27,8.018-0.111c-0.079-0.5-0.13-1.011-0.13-1.543c0-1.7,0.6-3.5,1.7-5c-0.5-1.7-1.2-5.3,0.2-6.6 c2.7,0,4.6,1.3,5.5,2.1C21,13.4,22.9,13,25,13s4,0.4,5.6,1.1c0.9-0.8,2.8-2.1,5.5-2.1c1.5,1.4,0.7,5,0.2,6.6c1.1,1.5,1.7,3.2,1.6,5 c0,0.484-0.045,0.951-0.11,1.409c3.499-0.172,6.527-0.034,8.204,0.102c-0.002,0.337-0.033,0.666-0.051,0.999 c-1.671-0.138-4.775-0.28-8.359-0.089c-0.089,0.336-0.197,0.663-0.325,0.98c3.546,0.046,6.665,0.389,8.548,0.689 c-0.043,0.332-0.093,0.661-0.151,0.987c-1.912-0.306-5.171-0.664-8.879-0.682C35.112,30.873,31.557,32.75,26,32.969V33 c2.6,0,5,3.9,5,6.6V45c0,0.823,0.498,1.53,1.209,1.836C41.37,43.804,48,35.164,48,25C48,12.318,37.683,2,25,2S2,12.318,2,25 C2,35.164,8.63,43.804,17.791,46.836z',
			        'name' 		=> 'GitHub'
			    ),
			    array(
			        'link' 		=> 'skype_link',
			        'slug' 		=> 'skype',
			        'svg_path'	=> 'M 14.28125 0 C 6.410156 0 0 6.324219 0 14.09375 C 0 16.476563 0.621094 18.800781 1.78125 20.875 C 1.507813 22.3125 1.375 23.792969 1.375 25.25 C 1.375 38.25 12.074219 48.8125 25.25 48.8125 C 26.601563 48.8125 27.953125 48.722656 29.28125 48.5 C 31.257813 49.488281 33.480469 50 35.71875 50 C 43.589844 50 50 43.675781 50 35.90625 C 50 33.820313 49.539063 31.828125 48.65625 29.96875 C 48.976563 28.433594 49.15625 26.847656 49.15625 25.25 C 49.15625 12.253906 38.425781 1.6875 25.25 1.6875 C 24.011719 1.6875 22.761719 1.78125 21.53125 1.96875 C 19.335938 0.683594 16.847656 0 14.28125 0 Z M 25.09375 9.375 C 27.140625 9.375 28.933594 9.597656 30.4375 10.0625 C 31.945313 10.523438 33.222656 11.136719 34.21875 11.90625 C 35.226563 12.683594 35.996094 13.511719 36.46875 14.375 C 36.945313 15.246094 37.1875 16.132813 37.1875 16.96875 C 37.1875 17.777344 36.84375 18.484375 36.21875 19.125 C 35.59375 19.765625 34.808594 20.09375 33.875 20.09375 C 33.027344 20.09375 32.367188 19.898438 31.90625 19.5 C 31.476563 19.125 31.023438 18.554688 30.53125 17.71875 C 29.960938 16.648438 29.292969 15.777344 28.5 15.1875 C 27.730469 14.613281 26.421875 14.34375 24.65625 14.34375 C 23.015625 14.34375 21.675781 14.640625 20.6875 15.28125 C 19.730469 15.898438 19.28125 16.628906 19.28125 17.46875 C 19.28125 17.984375 19.441406 18.410156 19.75 18.78125 C 20.078125 19.171875 20.511719 19.496094 21.09375 19.78125 C 21.695313 20.078125 22.316406 20.332031 22.9375 20.5 C 23.574219 20.671875 24.660156 20.945313 26.125 21.28125 C 27.976563 21.675781 29.679688 22.089844 31.1875 22.5625 C 32.710938 23.046875 34.019531 23.652344 35.09375 24.34375 C 36.1875 25.046875 37.070313 25.949219 37.6875 27.03125 C 38.304688 28.113281 38.59375 29.441406 38.59375 31 C 38.589844 32.859375 38.066406 34.546875 37 36.03125 C 35.9375 37.511719 34.371094 38.703125 32.34375 39.53125 C 30.335938 40.351563 27.925781 40.75 25.1875 40.75 C 21.898438 40.75 19.148438 40.191406 17 39.0625 C 15.460938 38.246094 14.175781 37.117188 13.21875 35.75 C 12.242188 34.367188 11.75 33.011719 11.75 31.6875 C 11.75 30.863281 12.085938 30.152344 12.71875 29.5625 C 13.347656 28.984375 14.136719 28.6875 15.09375 28.6875 C 15.878906 28.6875 16.574219 28.914063 17.125 29.375 C 17.65625 29.816406 18.105469 30.472656 18.46875 31.3125 C 18.875 32.230469 19.308594 32.980469 19.78125 33.59375 C 20.226563 34.175781 20.863281 34.679688 21.6875 35.0625 C 22.515625 35.453125 23.652344 35.65625 25.03125 35.65625 C 26.925781 35.65625 28.460938 35.253906 29.625 34.46875 C 30.765625 33.703125 31.34375 32.785156 31.34375 31.65625 C 31.34375 30.765625 31.03125 30.046875 30.4375 29.5 C 29.8125 28.925781 28.992188 28.492188 28 28.1875 C 26.957031 27.871094 25.558594 27.519531 23.8125 27.15625 C 21.4375 26.652344 19.417969 26.046875 17.8125 25.375 C 16.167969 24.683594 14.84375 23.722656 13.875 22.53125 C 12.890625 21.316406 12.375 19.820313 12.375 18.03125 C 12.375 16.324219 12.902344 14.761719 13.9375 13.4375 C 14.957031 12.125 16.449219 11.105469 18.375 10.40625 C 20.273438 9.710938 22.539063 9.375 25.09375 9.375 Z ',
			        'name' 		=> 'Skype'
			    ),
			    array(
			        'link' 		=> 'weibo_link',
			        'slug' 		=> 'weibo',
			        'svg_path'	=> 'M 35 6 C 34.222656 6 33.472656 6.078125 32.75 6.207031 C 32.207031 6.300781 31.84375 6.820313 31.9375 7.363281 C 32.03125 7.910156 32.550781 8.273438 33.09375 8.179688 C 33.726563 8.066406 34.359375 8 35 8 C 41.085938 8 46 12.914063 46 19 C 46 20.316406 45.757813 21.574219 45.328125 22.753906 C 45.195313 23.09375 45.253906 23.476563 45.484375 23.757813 C 45.71875 24.039063 46.082031 24.171875 46.441406 24.105469 C 46.800781 24.039063 47.09375 23.78125 47.207031 23.4375 C 47.710938 22.054688 48 20.566406 48 19 C 48 11.832031 42.167969 6 35 6 Z M 35 12 C 34.574219 12 34.171875 12.042969 33.789063 12.109375 C 33.246094 12.207031 32.878906 12.722656 32.976563 13.269531 C 33.070313 13.8125 33.589844 14.175781 34.132813 14.082031 C 34.425781 14.03125 34.714844 14 35 14 C 37.773438 14 40 16.226563 40 19 C 40 19.597656 39.890625 20.167969 39.691406 20.707031 C 39.503906 21.226563 39.773438 21.800781 40.292969 21.988281 C 40.8125 22.175781 41.386719 21.910156 41.574219 21.390625 C 41.84375 20.648438 42 19.84375 42 19 C 42 15.144531 38.855469 12 35 12 Z M 21.175781 12.40625 C 17.964844 12.34375 13.121094 14.878906 8.804688 19.113281 C 4.511719 23.40625 2 27.90625 2 31.78125 C 2 39.3125 11.628906 43.8125 21.152344 43.8125 C 33.5 43.8125 41.765625 36.699219 41.765625 31.046875 C 41.765625 27.59375 38.835938 25.707031 36.21875 24.871094 C 35.59375 24.660156 35.175781 24.558594 35.488281 23.71875 C 35.695313 23.21875 36 22.265625 36 21 C 36 19.5625 35 18.316406 33 18.09375 C 32.007813 17.984375 28 18 25.339844 19.113281 C 25.339844 19.113281 23.871094 19.746094 24.289063 18.59375 C 25.023438 16.292969 24.917969 14.40625 23.765625 13.359375 C 23.140625 12.730469 22.25 12.425781 21.175781 12.40625 Z M 20.3125 23.933594 C 28.117188 23.933594 34.441406 27.914063 34.441406 32.828125 C 34.441406 37.738281 28.117188 41.71875 20.3125 41.71875 C 12.511719 41.71875 6.1875 37.738281 6.1875 32.828125 C 6.1875 27.914063 12.511719 23.933594 20.3125 23.933594 Z M 19.265625 26.023438 C 16.246094 26.046875 13.3125 27.699219 12.039063 30.246094 C 10.46875 33.484375 11.933594 37.042969 15.699219 38.191406 C 19.464844 39.445313 23.960938 37.5625 25.53125 34.113281 C 27.097656 30.769531 25.113281 27.214844 21.347656 26.277344 C 20.660156 26.097656 19.960938 26.019531 19.265625 26.023438 Z M 20.824219 30.25 C 21.402344 30.25 21.871094 30.714844 21.871094 31.292969 C 21.871094 31.871094 21.402344 32.339844 20.824219 32.339844 C 20.246094 32.339844 19.777344 31.871094 19.777344 31.292969 C 19.777344 30.714844 20.246094 30.25 20.824219 30.25 Z M 16.417969 31.292969 C 16.746094 31.296875 17.074219 31.347656 17.382813 31.453125 C 18.722656 31.878906 19.132813 33.148438 18.308594 34.207031 C 17.589844 35.265625 15.945313 35.792969 14.707031 35.265625 C 13.476563 34.738281 13.167969 33.464844 13.886719 32.515625 C 14.425781 31.71875 15.429688 31.28125 16.417969 31.292969 Z ',
			        'name' 		=> 'Weibo'
			    ),
			    array(
			        'link' 		=> 'foursquare_link',
			        'slug' 		=> 'foursquare',
			        'svg_path'	=> 'M 9 4 C 6.24 4 4 6.24 4 9 L 4 41 C 4 43.76 6.24 46 9 46 L 41 46 C 42.44 46 43.740391 45.390156 44.650391 44.410156 C 45.490391 43.520156 46 42.32 46 41 L 46 14.570312 L 36.349609 23.849609 L 37.210938 24.710938 C 38.400937 25.890938 38.400937 27.810234 37.210938 28.990234 L 27.279297 39.125 C 26.689297 39.725 25.910625 40.015625 25.140625 40.015625 C 24.370625 40.015625 23.59 39.725 23 39.125 L 12.990234 28.990234 C 12.670234 28.670234 12.429297 28.280859 12.279297 27.880859 C 11.969297 27.710859 11.670625 27.510781 11.390625 27.300781 C 10.910625 26.940781 10.459062 26.509297 10.039062 26.029297 C 6.9990625 22.469297 7.4207031 17.099063 10.970703 14.039062 C 12.510703 12.729063 14.47 12 16.5 12 C 18.94 12 21.249375 13.18 22.859375 15 L 22.970703 14.890625 C 23.560703 14.300625 24.339375 14 25.109375 14 C 25.879375 14 26.66 14.300625 27.25 14.890625 L 29.830078 17.330078 L 40.910156 4 L 9 4 z M 43.119141 4.4707031 L 27.039062 23.820312 L 25.460938 25.720703 C 25.270938 25.930703 25.089453 26 24.939453 26 C 24.729453 26 24.560469 25.869531 24.480469 25.769531 C 24.450469 25.739531 24.189297 25.420391 23.779297 24.900391 C 23.589297 25.210391 23.389922 25.499297 23.169922 25.779297 L 22.560547 26.539062 L 22.480469 26.539062 C 22.340469 26.679062 22.189297 26.820937 22.029297 26.960938 C 21.379297 27.520938 20.649141 27.970781 19.869141 28.300781 L 24.390625 31.849609 C 24.390625 31.849609 24.539766 32 24.759766 32 C 24.869766 32 24.989141 31.969375 25.119141 31.859375 L 32.070312 25.179688 L 34.5 22.849609 L 46 11.800781 L 46 9 C 46 7 44.819141 5.2707031 43.119141 4.4707031 z M 16.5 14 A 6.5 6.5 0 0 0 10 20.5 A 6.5 6.5 0 0 0 16.5 27 A 6.5 6.5 0 0 0 23 20.5 A 6.5 6.5 0 0 0 16.5 14 z',
			        'name' 		=> 'Foursquare'
			    ),
			    array(
			        'link' 		=> 'soundcloud_link',
			        'slug' 		=> 'soundcloud',
			        'svg_path'	=> 'M 30 11 C 26.398438 11 23 12.789063 21 15.6875 L 21 36 L 42.5 36 C 46.601563 36 50 32.601563 50 28.5 C 50 24.398438 46.601563 21 42.5 21 C 42 21 41.5 21.085938 41 21.1875 C 40.5 15.488281 35.800781 11 30 11 Z M 17 16 C 16.300781 16 15.601563 16.085938 15 16.1875 L 15 36 L 17 36 Z M 18 16 L 18 36 L 20 36 L 20 16.5 C 19.398438 16.300781 18.699219 16.101563 18 16 Z M 14 16.5 C 13.300781 16.800781 12.601563 17.101563 12 17.5 L 12 36 L 14 36 Z M 11 18.3125 C 10.199219 19.011719 9.5 19.90625 9 20.90625 L 9 36 L 11 36 Z M 6.5 22 C 6.324219 22.011719 6.148438 22.042969 6 22.09375 L 6 35.90625 C 6.300781 36.007813 6.699219 36 7 36 L 8 36 L 8 22.09375 C 7.699219 21.992188 7.300781 22 7 22 C 6.851563 22 6.675781 21.988281 6.5 22 Z M 5 22.3125 C 4.300781 22.511719 3.601563 22.8125 3 23.3125 L 3 34.6875 C 3.601563 35.085938 4.300781 35.488281 5 35.6875 Z M 2 24.09375 C 0.800781 25.394531 0 27.101563 0 29 C 0 30.898438 0.800781 32.605469 2 33.90625 Z ',
			        'name' 		=> 'SoundCloud'
			    ),
			    array(
			        'link' 		=> 'vk_link',
			        'slug' 		=> 'vk',
			        'svg_path'	=> 'M25,2C12.318,2,2,12.318,2,25s10.318,23,23,23c12.683,0,23-10.318,23-23S37.683,2,25,2z M34.973,29.535 c2.237,1.986,2.702,2.695,2.778,2.816C38.678,33.821,36.723,34,36.723,34h-4.12c0,0-1.003,0.011-1.86-0.557 c-1.397-0.915-2.86-2.689-3.888-2.381C25.992,31.32,26,32.486,26,33.483C26,33.84,25.693,34,25,34s-0.981,0-1.288,0 c-2.257,0-4.706-0.76-7.149-3.313c-3.456-3.609-6.487-10.879-6.487-10.879s-0.179-0.366,0.016-0.589 c0.221-0.25,0.822-0.218,0.822-0.218L14.909,19c0,0,0.376,0.071,0.646,0.261c0.223,0.156,0.347,0.454,0.347,0.454 s0.671,2.216,1.526,3.629c1.67,2.758,2.447,2.828,3.014,2.531C21.27,25.445,21,22.513,21,22.513s0.037-1.259-0.395-1.82 c-0.333-0.434-0.97-0.665-1.248-0.701c-0.225-0.029,0.151-0.423,0.63-0.648C20.627,19.059,21.498,18.986,23,19 c1.169,0.011,1.506,0.081,1.962,0.186C26.341,19.504,26,20.343,26,23.289c0,0.944-0.13,2.271,0.582,2.711 c0.307,0.19,1.359,0.422,3.231-2.618c0.889-1.442,1.596-3.834,1.596-3.834s0.146-0.263,0.373-0.393 c0.232-0.133,0.225-0.13,0.543-0.13S35.832,19,36.532,19c0.699,0,1.355-0.008,1.468,0.402c0.162,0.589-0.516,2.607-2.234,4.797 C32.943,27.793,32.63,27.457,34.973,29.535z',
			        'name' 		=> 'VK'
			    ),
			    array(
			        'link' 		=> 'houzz_link',
			        'slug' 		=> 'houzz',
			        'svg_path'	=> 'M 24 4.589844 L 13 10.949219 L 13 23.808594 L 13.984375 23.242188 L 23 18.03125 L 24 17.453125 Z M 37 11.105469 L 26 17.449219 L 37 23.871094 Z M 26 19.765625 L 26 30.296875 L 35.027344 25.035156 Z M 13 26.113281 L 13 37.859375 L 23.011719 32.03125 Z M 37 26.203125 L 36.023438 26.777344 L 27.003906 32.03125 L 26 32.613281 L 26 45.4375 L 37 39.09375 Z ',
			        'name' 		=> 'Houzz'
			    ),
			    array(
			        'link' 		=> 'naver_line_link',
			        'slug' 		=> 'naver',
			        'svg_path'	=> 'M 9 4 C 6.24 4 4 6.24 4 9 L 4 41 C 4 43.76 6.24 46 9 46 L 41 46 C 43.76 46 46 43.76 46 41 L 46 9 C 46 6.24 43.76 4 41 4 L 9 4 z M 25 11 C 33.27 11 40 16.359219 40 22.949219 C 40 25.579219 38.959297 27.960781 36.779297 30.300781 C 35.209297 32.080781 32.660547 34.040156 30.310547 35.660156 C 27.960547 37.260156 25.8 38.519609 25 38.849609 C 24.68 38.979609 24.44 39.039062 24.25 39.039062 C 23.59 39.039062 23.649219 38.340781 23.699219 38.050781 C 23.739219 37.830781 23.919922 36.789063 23.919922 36.789062 C 23.969922 36.419063 24.019141 35.830937 23.869141 35.460938 C 23.699141 35.050938 23.029062 34.840234 22.539062 34.740234 C 15.339063 33.800234 10 28.849219 10 22.949219 C 10 16.359219 16.73 11 25 11 z M 23.992188 18.998047 C 23.488379 19.007393 23 19.391875 23 20 L 23 26 C 23 26.552 23.448 27 24 27 C 24.552 27 25 26.552 25 26 L 25 23.121094 L 27.185547 26.580078 C 27.751547 27.372078 29 26.973 29 26 L 29 20 C 29 19.448 28.552 19 28 19 C 27.448 19 27 19.448 27 20 L 27 23 L 24.814453 19.419922 C 24.602203 19.122922 24.294473 18.992439 23.992188 18.998047 z M 15 19 C 14.448 19 14 19.448 14 20 L 14 26 C 14 26.552 14.448 27 15 27 L 18 27 C 18.552 27 19 26.552 19 26 C 19 25.448 18.552 25 18 25 L 16 25 L 16 20 C 16 19.448 15.552 19 15 19 z M 21 19 C 20.448 19 20 19.448 20 20 L 20 26 C 20 26.552 20.448 27 21 27 C 21.552 27 22 26.552 22 26 L 22 20 C 22 19.448 21.552 19 21 19 z M 31 19 C 30.448 19 30 19.448 30 20 L 30 26 C 30 26.552 30.448 27 31 27 L 34 27 C 34.552 27 35 26.552 35 26 C 35 25.448 34.552 25 34 25 L 32 25 L 32 24 L 34 24 C 34.553 24 35 23.552 35 23 C 35 22.448 34.553 22 34 22 L 32 22 L 32 21 L 34 21 C 34.552 21 35 20.552 35 20 C 35 19.448 34.552 19 34 19 L 31 19 z',
			        'name' 		=> 'Naver Line'
			    ),
			    array(
			        'link' 		=> 'tripadvisor_link',
			        'slug' 		=> 'tripadvisor',
			        'svg_path'	=> 'M 25 11 C 19.167969 11 13.84375 12.511719 9.789063 15 L 2 15 C 2 15 3.753906 17.152344 4.578125 19.578125 C 2.96875 21.621094 2 24.195313 2 27 C 2 33.628906 7.371094 39 14 39 C 17.496094 39 20.636719 37.492188 22.828125 35.105469 L 25 38 L 27.171875 35.105469 C 29.363281 37.492188 32.503906 39 36 39 C 42.628906 39 48 33.628906 48 27 C 48 24.195313 47.03125 21.621094 45.421875 19.578125 C 46.246094 17.152344 48 15 48 15 L 40.203125 15 C 36.148438 12.511719 30.828125 11 25 11 Z M 14 18 C 18.972656 18 23 22.027344 23 27 C 23 31.972656 18.972656 36 14 36 C 9.027344 36 5 31.972656 5 27 C 5 22.027344 9.027344 18 14 18 Z M 36 18 C 40.972656 18 45 22.027344 45 27 C 45 31.972656 40.972656 36 36 36 C 31.027344 36 27 31.972656 27 27 C 27 22.027344 31.027344 18 36 18 Z M 14 21 C 10.6875 21 8 23.6875 8 27 C 8 30.3125 10.6875 33 14 33 C 17.3125 33 20 30.3125 20 27 C 20 23.6875 17.3125 21 14 21 Z M 36 21 C 32.6875 21 30 23.6875 30 27 C 30 30.3125 32.6875 33 36 33 C 39.3125 33 42 30.3125 42 27 C 42 23.6875 39.3125 21 36 21 Z M 14 23 C 16.210938 23 18 24.789063 18 27 C 18 29.210938 16.210938 31 14 31 C 11.789063 31 10 29.210938 10 27 C 10 24.789063 11.789063 23 14 23 Z M 36 23 C 38.210938 23 40 24.789063 40 27 C 40 29.210938 38.210938 31 36 31 C 33.789063 31 32 29.210938 32 27 C 32 24.789063 33.789063 23 36 23 Z M 14 25 C 12.894531 25 12 25.894531 12 27 C 12 28.105469 12.894531 29 14 29 C 15.105469 29 16 28.105469 16 27 C 16 25.894531 15.105469 25 14 25 Z M 36 25 C 34.894531 25 34 25.894531 34 27 C 34 28.105469 34.894531 29 36 29 C 37.105469 29 38 28.105469 38 27 C 38 25.894531 37.105469 25 36 25 Z ',
			        'name' 		=> 'TripAdvisor'
			    ),
			    array(
			        'link' 		=> 'wechat_link',
			        'slug' 		=> 'wechat',
			        'svg_path'	=> 'M 19 6 C 9.625 6 2 12.503906 2 20.5 C 2 24.769531 4.058594 28.609375 7.816406 31.390625 L 5.179688 39.304688 L 13.425781 34.199219 C 15.714844 34.917969 18.507813 35.171875 21.203125 34.875 C 23.390625 39.109375 28.332031 42 34 42 C 35.722656 42 37.316406 41.675781 38.796875 41.234375 L 45.644531 45.066406 L 43.734375 38.515625 C 46.3125 36.375 48 33.394531 48 30 C 48 23.789063 42.597656 18.835938 35.75 18.105469 C 34.40625 11.152344 27.367188 6 19 6 Z M 13 14 C 14.101563 14 15 14.898438 15 16 C 15 17.101563 14.101563 18 13 18 C 11.898438 18 11 17.101563 11 16 C 11 14.898438 11.898438 14 13 14 Z M 25 14 C 26.101563 14 27 14.898438 27 16 C 27 17.101563 26.101563 18 25 18 C 23.898438 18 23 17.101563 23 16 C 23 14.898438 23.898438 14 25 14 Z M 34 20 C 40.746094 20 46 24.535156 46 30 C 46 32.957031 44.492188 35.550781 42.003906 37.394531 L 41.445313 37.8125 L 42.355469 40.933594 L 39.105469 39.109375 L 38.683594 39.25 C 37.285156 39.71875 35.6875 40 34 40 C 27.253906 40 22 35.464844 22 30 C 22 24.535156 27.253906 20 34 20 Z M 29.5 26 C 28.699219 26 28 26.699219 28 27.5 C 28 28.300781 28.699219 29 29.5 29 C 30.300781 29 31 28.300781 31 27.5 C 31 26.699219 30.300781 26 29.5 26 Z M 38.5 26 C 37.699219 26 37 26.699219 37 27.5 C 37 28.300781 37.699219 29 38.5 29 C 39.300781 29 40 28.300781 40 27.5 C 40 26.699219 39.300781 26 38.5 26 Z ',
			        'name' 		=> 'WeChat'
			    ),
			    array(
			        'link' 		=> 'telegram_link',
			        'slug' 		=> 'telegram',
			        'svg_path'	=> 'M 44.4375 4.15625 L 4.59375 21.28125 C 4.238281 21.433594 4.011719 21.796875 4 22.1875 C 3.988281 22.578125 4.214844 22.917969 4.5625 23.09375 L 18.5 30.0625 Z M 45.84375 5.5625 L 19.9375 31.5 L 26.90625 45.4375 C 27.078125 45.777344 27.433594 46 27.8125 46 L 27.84375 46 C 28.234375 45.988281 28.5625 45.761719 28.71875 45.40625 Z ',
			        'name' 		=> 'Telegram'
			    ),
			    array(
			        'link' 		=> 'viber_link',
			        'slug' 		=> 'viber',
			        'svg_path'	=> 'M 44.78125 13.15625 C 44 10.367188 42.453125 8.164063 40.1875 6.605469 C 37.328125 4.632813 34.039063 3.9375 31.199219 3.511719 C 27.269531 2.925781 23.710938 2.84375 20.316406 3.257813 C 17.136719 3.648438 14.742188 4.269531 12.558594 5.273438 C 8.277344 7.242188 5.707031 10.425781 4.921875 14.734375 C 4.539063 16.828125 4.28125 18.71875 4.132813 20.523438 C 3.789063 24.695313 4.101563 28.386719 5.085938 31.808594 C 6.046875 35.144531 7.722656 37.527344 10.210938 39.09375 C 10.84375 39.492188 11.65625 39.78125 12.441406 40.058594 C 12.886719 40.214844 13.320313 40.367188 13.675781 40.535156 C 14.003906 40.6875 14.003906 40.714844 14 40.988281 C 13.972656 43.359375 14 48.007813 14 48.007813 L 14.007813 49 L 15.789063 49 L 16.078125 48.71875 C 16.269531 48.539063 20.683594 44.273438 22.257813 42.554688 L 22.472656 42.316406 C 22.742188 42.003906 22.742188 42.003906 23.019531 42 C 25.144531 41.957031 27.316406 41.875 29.472656 41.757813 C 32.085938 41.617188 35.113281 41.363281 37.964844 40.175781 C 40.574219 39.085938 42.480469 37.355469 43.625 35.035156 C 44.820313 32.613281 45.527344 29.992188 45.792969 27.019531 C 46.261719 21.792969 45.929688 17.257813 44.78125 13.15625 Z M 35.382813 33.480469 C 34.726563 34.546875 33.75 35.289063 32.597656 35.769531 C 31.753906 36.121094 30.894531 36.046875 30.0625 35.695313 C 23.097656 32.746094 17.632813 28.101563 14.023438 21.421875 C 13.277344 20.046875 12.761719 18.546875 12.167969 17.09375 C 12.046875 16.796875 12.054688 16.445313 12 16.117188 C 12.050781 13.769531 13.851563 12.445313 15.671875 12.046875 C 16.367188 11.890625 16.984375 12.136719 17.5 12.632813 C 18.929688 13.992188 20.058594 15.574219 20.910156 17.347656 C 21.28125 18.125 21.113281 18.8125 20.480469 19.390625 C 20.347656 19.511719 20.210938 19.621094 20.066406 19.730469 C 18.621094 20.816406 18.410156 21.640625 19.179688 23.277344 C 20.492188 26.0625 22.671875 27.933594 25.488281 29.09375 C 26.230469 29.398438 26.929688 29.246094 27.496094 28.644531 C 27.574219 28.566406 27.660156 28.488281 27.714844 28.394531 C 28.824219 26.542969 30.4375 26.726563 31.925781 27.78125 C 32.902344 28.476563 33.851563 29.210938 34.816406 29.917969 C 36.289063 31 36.277344 32.015625 35.382813 33.480469 Z M 26.144531 15 C 25.816406 15 25.488281 15.027344 25.164063 15.082031 C 24.617188 15.171875 24.105469 14.804688 24.011719 14.257813 C 23.921875 13.714844 24.289063 13.199219 24.835938 13.109375 C 25.265625 13.035156 25.707031 13 26.144531 13 C 30.476563 13 34 16.523438 34 20.855469 C 34 21.296875 33.964844 21.738281 33.890625 22.164063 C 33.808594 22.652344 33.386719 23 32.90625 23 C 32.851563 23 32.796875 22.996094 32.738281 22.984375 C 32.195313 22.894531 31.828125 22.378906 31.917969 21.835938 C 31.972656 21.515625 32 21.1875 32 20.855469 C 32 17.628906 29.371094 15 26.144531 15 Z M 31 21 C 31 21.550781 30.550781 22 30 22 C 29.449219 22 29 21.550781 29 21 C 29 19.347656 27.652344 18 26 18 C 25.449219 18 25 17.550781 25 17 C 25 16.449219 25.449219 16 26 16 C 28.757813 16 31 18.242188 31 21 Z M 36.710938 23.222656 C 36.605469 23.6875 36.191406 24 35.734375 24 C 35.660156 24 35.585938 23.992188 35.511719 23.976563 C 34.972656 23.851563 34.636719 23.316406 34.757813 22.777344 C 34.902344 22.140625 34.976563 21.480469 34.976563 20.816406 C 34.976563 15.957031 31.019531 12 26.160156 12 C 25.496094 12 24.835938 12.074219 24.199219 12.21875 C 23.660156 12.34375 23.125 12.003906 23.003906 11.464844 C 22.878906 10.925781 23.21875 10.390625 23.757813 10.269531 C 24.539063 10.089844 25.347656 10 26.160156 10 C 32.125 10 36.976563 14.851563 36.976563 20.816406 C 36.976563 21.628906 36.886719 22.4375 36.710938 23.222656 Z ',
			        'name' 		=> 'Viber'
			    ),
			    array(
			        'link' 		=> 'spotify_link',
			        'slug' 		=> 'spotify',
			        'svg_path'	=> 'M25.009,1.982C12.322,1.982,2,12.304,2,24.991S12.322,48,25.009,48s23.009-10.321,23.009-23.009S37.696,1.982,25.009,1.982z M34.748,35.333c-0.289,0.434-0.765,0.668-1.25,0.668c-0.286,0-0.575-0.081-0.831-0.252C30.194,34.1,26,33,22.5,33.001 c-3.714,0.002-6.498,0.914-6.526,0.923c-0.784,0.266-1.635-0.162-1.897-0.948s0.163-1.636,0.949-1.897 c0.132-0.044,3.279-1.075,7.474-1.077C26,30,30.868,30.944,34.332,33.253C35.022,33.713,35.208,34.644,34.748,35.333z M37.74,29.193 c-0.325,0.522-0.886,0.809-1.459,0.809c-0.31,0-0.624-0.083-0.906-0.26c-4.484-2.794-9.092-3.385-13.062-3.35 c-4.482,0.04-8.066,0.895-8.127,0.913c-0.907,0.258-1.861-0.272-2.12-1.183c-0.259-0.913,0.272-1.862,1.184-2.12 c0.277-0.079,3.854-0.959,8.751-1c4.465-0.037,10.029,0.61,15.191,3.826C37.995,27.328,38.242,28.388,37.74,29.193z M40.725,22.013 C40.352,22.647,39.684,23,38.998,23c-0.344,0-0.692-0.089-1.011-0.275c-5.226-3.068-11.58-3.719-15.99-3.725 c-0.021,0-0.042,0-0.063,0c-5.333,0-9.44,0.938-9.481,0.948c-1.078,0.247-2.151-0.419-2.401-1.495 c-0.25-1.075,0.417-2.149,1.492-2.4C11.729,16.01,16.117,15,21.934,15c0.023,0,0.046,0,0.069,0 c4.905,0.007,12.011,0.753,18.01,4.275C40.965,19.835,41.284,21.061,40.725,22.013z',
			        'name' 		=> 'Spotify'
			    ),
			    array(
			        'link' 		=> 'bandcamp_link',
			        'slug' 		=> 'bandcamp',
			        'svg_path'	=> 'M25,2C12.318,2,2,12.317,2,25s10.318,23,23,23s23-10.317,23-23S37.682,2,25,2z M29.36,32H13l7.64-14H37L29.36,32z',
			        'name' 		=> 'Bandcamp'
			    ),
			    array(
			        'link' 		=> 'snapchat_link',
			        'slug' 		=> 'snapchat',
			        'svg_path'	=> 'M 46.773438 35.078125 C 40.96875 34.121094 38.316406 28.109375 38.230469 27.914063 C 38.21875 27.878906 38.191406 27.816406 38.175781 27.78125 C 38 27.429688 37.824219 26.933594 37.972656 26.582031 C 38.226563 25.980469 39.433594 25.597656 40.15625 25.367188 C 40.410156 25.285156 40.652344 25.210938 40.839844 25.136719 C 42.59375 24.441406 43.46875 23.535156 43.449219 22.433594 C 43.433594 21.546875 42.753906 20.734375 41.753906 20.382813 C 41.40625 20.234375 41.007813 20.160156 40.605469 20.160156 C 40.332031 20.160156 39.917969 20.199219 39.519531 20.382813 C 38.851563 20.695313 38.265625 20.863281 37.847656 20.882813 C 37.757813 20.878906 37.679688 20.871094 37.613281 20.859375 L 37.65625 20.171875 C 37.851563 17.0625 38.097656 13.1875 37.046875 10.832031 C 33.945313 3.890625 27.375 3.351563 25.433594 3.351563 L 24.550781 3.359375 C 22.613281 3.359375 16.054688 3.898438 12.960938 10.835938 C 11.910156 13.191406 12.152344 17.0625 12.351563 20.175781 L 12.359375 20.292969 C 12.371094 20.484375 12.382813 20.675781 12.394531 20.859375 C 11.960938 20.9375 11.113281 20.792969 10.234375 20.382813 C 9.039063 19.824219 6.886719 20.5625 6.589844 22.125 C 6.457031 22.816406 6.617188 24.128906 9.164063 25.132813 C 9.355469 25.210938 9.59375 25.285156 9.851563 25.367188 C 10.570313 25.597656 11.777344 25.976563 12.03125 26.582031 C 12.179688 26.933594 12.003906 27.429688 11.796875 27.855469 C 11.6875 28.109375 9.050781 34.121094 3.234375 35.078125 C 2.492188 35.199219 1.964844 35.855469 2.003906 36.613281 C 2.015625 36.8125 2.066406 37.011719 2.148438 37.207031 C 2.675781 38.445313 4.59375 39.296875 8.171875 39.878906 C 8.234375 40.089844 8.304688 40.402344 8.34375 40.574219 C 8.417969 40.929688 8.5 41.289063 8.609375 41.664063 C 8.714844 42.019531 9.078125 42.84375 10.210938 42.84375 C 10.554688 42.84375 10.929688 42.769531 11.332031 42.691406 C 11.925781 42.574219 12.667969 42.429688 13.621094 42.429688 C 14.152344 42.429688 14.699219 42.476563 15.25 42.566406 C 16.265625 42.734375 17.183594 43.386719 18.25 44.136719 C 19.914063 45.316406 21.800781 46.648438 24.726563 46.648438 C 24.804688 46.648438 24.882813 46.644531 24.957031 46.640625 C 25.0625 46.644531 25.171875 46.648438 25.28125 46.648438 C 28.207031 46.648438 30.09375 45.3125 31.761719 44.136719 C 32.777344 43.414063 33.738281 42.738281 34.757813 42.566406 C 35.308594 42.476563 35.855469 42.429688 36.386719 42.429688 C 37.304688 42.429688 38.03125 42.546875 38.679688 42.675781 C 39.140625 42.765625 39.507813 42.808594 39.847656 42.808594 C 40.605469 42.808594 41.1875 42.375 41.398438 41.648438 C 41.507813 41.28125 41.585938 40.925781 41.664063 40.566406 C 41.695313 40.433594 41.769531 40.097656 41.835938 39.875 C 45.414063 39.292969 47.332031 38.441406 47.855469 37.214844 C 47.941406 37.019531 47.988281 36.816406 48.003906 36.605469 C 48.042969 35.859375 47.515625 35.203125 46.773438 35.078125 Z ',
			        'name' 		=> 'Snapchat'
			    ),
			    array(
			        'link' 		=> 'medium_link',
			        'slug' 		=> 'medium',
			        'svg_path'	=> 'M 18 41.578125 C 18 42.46875 17.523438 43 16.847656 43 C 16.609375 43 16.34375 42.933594 16.0625 42.792969 L 5.164063 37.273438 C 4.523438 36.949219 4 36.089844 4 35.367188 L 4 8.3125 C 4 7.597656 4.378906 7.175781 4.921875 7.175781 C 5.113281 7.175781 5.328125 7.226563 5.550781 7.339844 L 5.933594 7.535156 C 5.933594 7.535156 5.9375 7.535156 5.9375 7.535156 L 17.949219 13.617188 C 17.96875 13.628906 17.984375 13.648438 18 13.664063 Z M 30.585938 8.882813 L 31.320313 7.675781 C 31.582031 7.246094 32.0625 7 32.550781 7 C 32.664063 7 32.78125 7.015625 32.894531 7.046875 C 32.980469 7.066406 33.070313 7.097656 33.164063 7.144531 L 45.832031 13.558594 C 45.835938 13.558594 45.835938 13.5625 45.835938 13.5625 L 45.847656 13.566406 C 45.855469 13.570313 45.855469 13.582031 45.863281 13.585938 C 45.925781 13.636719 45.953125 13.722656 45.910156 13.792969 L 33.292969 34.558594 L 32.003906 36.675781 L 23.644531 20.304688 Z M 20 30.605469 L 20 17.5625 L 28.980469 35.15625 L 20.902344 31.066406 Z M 46 41.578125 C 46 42.414063 45.503906 42.890625 44.773438 42.890625 C 44.445313 42.890625 44.070313 42.796875 43.675781 42.597656 L 41.867188 41.679688 L 33.789063 37.589844 L 46 17.496094 Z ',
			        'name' 		=> 'Medium'
			    ),
				array(
			        'link' 		=> 'rss_link',
			        'slug' 		=> 'rss',
			        'svg_path'	=> 'M41,4H9C6.24,4,4,6.24,4,9v32c0,2.76,2.24,5,5,5h32c2.76,0,5-2.24,5-5V9C46,6.24,43.76,4,41,4z M14.5,39 c-1.94,0-3.5-1.57-3.5-3.5c0-1.94,1.56-3.5,3.5-3.5s3.5,1.56,3.5,3.5C18,37.43,16.44,39,14.5,39z M28.53,39.25 C28.42,39.7,28.01,40,27.57,40c-0.09,0-0.17-0.01-0.25-0.03c-0.54-0.14-0.86-0.69-0.72-1.22c0.27-1.03,0.4-2.1,0.4-3.18 C27,28.64,21.39,23,14.5,23c-1.1,0-2.19,0.14-3.24,0.43c-0.52,0.14-1.08-0.17-1.23-0.71c-0.14-0.53,0.18-1.08,0.71-1.22 c1.22-0.33,2.49-0.5,3.76-0.5c8,0,14.5,6.54,14.5,14.57C29,36.82,28.84,38.06,28.53,39.25z M38.73,39.15 c-0.07,0.49-0.5,0.85-0.99,0.85c-0.05,0-0.1,0-0.15-0.01c-0.54-0.08-0.92-0.59-0.84-1.14c0.17-1.1,0.25-2.22,0.25-3.34 C37,23.1,26.9,13,14.49,13c-1.12,0-2.24,0.08-3.34,0.25c-0.55,0.08-1.06-0.3-1.14-0.84c-0.08-0.55,0.3-1.06,0.84-1.14 c1.2-0.18,2.42-0.27,3.64-0.27C28.01,11,39,21.99,39,35.51C39,36.73,38.91,37.95,38.73,39.15z',
			        'name' 		=> 'RSS'
			    ),
			);
		}

		/**
		 * Registers customizer options.
		 *
		 * @since 1.4
		 * @return void
		 */
		protected function customizer_options() {
			add_action( 'customize_register', array( $this, 'sk_social_media_customizer' ) );
		}

		/**
		 * Creates customizer options.
		 *
		 * @since 1.4
		 * @return void
		 */
		public function sk_social_media_customizer( $wp_customize ) {

			global $social_media_profiles;

			$theme = wp_get_theme();
			if ( $theme->template == 'shopkeeper') {

				$wp_customize->add_setting( 'sk_top_bar_social_icons', array(
					'type'		 			=> 'option',
					'capability' 			=> 'manage_options',
					'sanitize_callback'    	=> 'sk_sanitize_checkbox',
					'sanitize_js_callback'  => 'sk_string_to_bool',
					'transport'				=> 'refresh',
					'default'	 			=> 'no',
				) );

				$wp_customize->add_control(
					new WP_Customize_Control(
						$wp_customize,
						'sk_top_bar_social_icons',
						array(
							'type'		  => 'checkbox',
							'label'       => esc_attr__( 'Top Bar Social Icons', 'shopkeeper-extender' ),
							'section'     => 'top_bar',
							'priority'    => 20,
							'active_callback' => 'sk_ext_is_topbar_enabled'
						)
					)
				);

				$wp_customize->add_setting( 'sk_footer_social_icons', array(
					'type'		 			=> 'option',
					'capability' 			=> 'manage_options',
					'sanitize_callback'    	=> 'sk_sanitize_checkbox',
					'sanitize_js_callback'  => 'sk_string_to_bool',
					'transport'				=> 'refresh',
					'default'	 			=> 'yes',
				) );

				$wp_customize->add_control(
					new WP_Customize_Control(
						$wp_customize,
						'sk_footer_social_icons',
						array(
							'type'			=> 'checkbox',
							'label'       	=> esc_attr__( 'Social Networking Icons', 'shopkeeper-extender' ),
							'section'     	=> 'footer',
							'priority'    	=> 20,
						)
					)
				);
			}

			// Section
			$wp_customize->add_section( 'social_media', array(
		 		'title'       => esc_attr__( 'Social Media', 'shopkeeper-extender' ),
		 		'priority'    => 10,
		 	) );

			// Fields
			$wp_customize->add_setting( 'sk_social_media_repeater', array(
				'type'		 		=> 'option',
				'sanitize_callback' => 'sk_sanitize_repeater',
				'capability' 		=> 'manage_options',
				'transport'  		=> 'refresh',
				'default' 			=> json_encode( array() ),
			) );

			$wp_customize->add_control(
				new SK_Ext_Customize_Repeater_Control(
					$wp_customize,
					'sk_social_media_repeater',
					array(
						'section' => 'social_media',
						'profiles' => $this->social_media_profiles,
						'priority' => 10,
					)
				)
			);

			// Abort if selective refresh is not available.
			if ( ! isset( $wp_customize->selective_refresh ) ) {
				return;
			}

			$wp_customize->selective_refresh->add_partial( 'sk_social_media_repeater', array(
				'selector' => '.site-top-bar-social-icons-wrapper ul.sk_social_icons_list',
				'settings' => 'sk_social_media_repeater',
				'render_callback' => function() {
					echo sk_social_media_shortcode();
				},
			) );
		}

		/**
		 * Adds social media shortcode.
		 *
		 * @since 1.4
		 * @return void
		 */
		protected function create_shortcode() {
			add_shortcode( 'social-media', array( $this, 'sk_social_media_shortcode' ) );
		}

		/**
		 * Creates social media shortcode.
		 *
		 * @since 1.4
		 * @return string
		 */
		public function sk_social_media_shortcode( $atts, $content = null ) {
			global $social_media_profiles;

		    extract( shortcode_atts( array(
		        'items_align' => 'left',
		        'type'		  => 'shortcode',
		        'fontsize'    => '24',
                'fontcolor'   => '#000',
		        ),
		    $atts ) );

		    $color = '';
		    if( $type == 'block') {
			    if( !empty($fontcolor) ) {
			    	$color = 'fill=' . $fontcolor;
				}
		    }

		    ob_start();

		    ?>

	        <ul class="sk_social_icons_list <?php echo esc_html($items_align); ?>">

				<?php

				$social_media_encoded_list = get_option( 'sk_social_media_repeater', json_encode( array() ) );
				$social_media_profiles 	   = json_decode( $social_media_encoded_list );

				foreach( $social_media_profiles as $social ) {

					if( !empty( $social->link ) && !empty($social->icon_slug) ) {

						if( 'custom' === $social->icon_slug && !empty( $social->image_url ) ) {
						?>

							<li class="sk_social_icon custom_icon">
								<a class="sk_social_icon_link" target="_blank" href="<?php echo esc_url( $social->link ); ?>" alt="<?php echo esc_attr( $social->link_title ); ?>" title="<?php echo esc_attr( $social->link_title ); ?>">
									<img src="<?php echo esc_url( $social->image_url ); ?>" alt="Social Media Profile"
										width="<?php echo esc_attr( $fontsize ); ?>" height="<?php echo esc_attr( $fontsize ); ?>" />
								</a>
							</li>

						<?php } else if( 'custom' !== $social->icon_slug ) {

							$svg_path = '';
							foreach( $this->social_media_profiles as $social_profile => $val ) {
								if( $val['slug'] === $social->icon_slug && isset( $val['svg_path'] ) ) {
									$svg_path = $val['svg_path'];
								}
							}

							if( !empty($svg_path) ) {
								?>

								<li class="sk_social_icon default_icon">
									<a class="sk_social_icon_link" target="_blank"
										href="<?php echo esc_url( $social->link ); ?>" alt="<?php echo esc_attr( $social->link_title ); ?>" title="<?php echo esc_attr( $social->link_title ); ?>">
										<svg
											class="<?php echo !empty($color) ? 'has-color' : ''; ?>"
											xmlns="http://www.w3.org/2000/svg" x="0px" y="0px"
											width="<?php echo esc_attr( $fontsize ); ?>" height="<?php echo esc_attr( $fontsize ); ?>"
											viewBox="0 0 50 50"
											<?php echo esc_attr( $color ); ?>>
											<path d="<?php echo esc_attr( $svg_path ); ?>"></path>
										</svg>
									</a>
								</li>

								<?php
							}
						}
					}
				}
				?>

	        </ul>

		    <?php

		    $content = ob_get_contents();

		    ob_end_clean();

		    return $content;
		}

		/**
		 * Adds social media as a WPBakery element.
		 *
		 * @since 1.4
		 * @return void
		 */
		protected function create_wb_element() {

			vc_map(
				array(
				   "name"			=> "Social Media Profiles",
				   "category"		=> 'Social',
				   "description"	=> "Links to your social media profiles",
				   "base"			=> "social-media",
				   "class"			=> "",
				   "icon"			=> "social-media",
				   "params" 		=> array(
						array(
							"type"			=> "dropdown",
							"holder"		=> "div",
							"class" 		=> "hide_in_vc_editor",
							"admin_label" 	=> true,
							"heading"		=> "Align",
							"param_name"	=> "items_align",
							"value"			=> array(
								"Left"		=> "left",
								"Center"	=> "center",
								"Right"		=> "right"
							)
						)
				   )
				)
			);
		}
	}

endif;

$sk_social_media = new SKSocialMedia;
