<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://ays-pro.com/
 * @since      1.0.0
 *
 * @package    Secure_Copy_Content_Protection
 * @subpackage Secure_Copy_Content_Protection/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Secure_Copy_Content_Protection
 * @subpackage Secure_Copy_Content_Protection/admin
 * @author     Security Team <info@ays-pro.com>
 */
class Secure_Copy_Content_Protection_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string $plugin_name The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string $version The current version of this plugin.
	 */
	private $version;
	private $results_obj;
	private $settings_obj;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @param string $plugin_name The name of this plugin.
	 * @param string $version The version of this plugin.
	 *
	 * @since    1.0.0
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version     = $version;
		
		add_filter('set-screen-option', array(__CLASS__, 'set_screen'), 10, 3);

        $per_page_array = array(
            'sccp_results_per_page'
        );
        foreach($per_page_array as $option_name){
            add_filter('set_screen_option_'.$option_name, array(__CLASS__, 'set_screen'), 10, 3);
        }

        add_action('admin_post_sccp_download_export', array( $this, 'handle_sccp_download') );

	}

	/**
	 * Register the styles for the admin menu area.
	 *
	 * @since    1.5.0
	 */
	public function admin_menu_styles() {
		echo "
        <style>
        	.ays_menu_badge_new{
                padding: 2px 2px !important;
            }

        	.ays_menu_badge{
                color: #fff;
                display: inline-block;
                font-size: 10px;
                line-height: 14px;
                text-align: center;
                background: #ca4a1f;
                margin-left: 5px;
                border-radius: 20px;
                padding: 2px 5px;
            }            

            #adminmenu a.toplevel_page_secure-copy-content-protection div.wp-menu-image img {
                padding: 0;
                opacity: .6;
                width: 32px;
                transition: all .3s ease-in;
            }

            #adminmenu a.toplevel_page_secure-copy-content-protection + ul.wp-submenu.wp-submenu-wrap li:last-child a {
                color: #68A615;
            }
        </style>
        ";
	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles( $hook_suffix ) {

		wp_enqueue_style($this->plugin_name . '-admin', plugin_dir_url(__FILE__) . 'css/admin.css', array(), $this->version, 'all');
		wp_enqueue_style($this->plugin_name . '-sweetalert2', plugin_dir_url(__FILE__) . 'css/sweetalert2.min.css', array(), $this->version, 'all');		


		if (false === strpos($hook_suffix, $this->plugin_name)) {
			return;
		}

		wp_enqueue_style($this->plugin_name . "-banner", plugin_dir_url(__FILE__) . 'css/secure-copy-content-protection-banner.css', array(), $this->version, 'all');
		
		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Secure_Copy_Content_Protection_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Secure_Copy_Content_Protection_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		// You need styling for the datepicker. For simplicity I've linked to the jQuery UI CSS on a CDN.
        wp_register_style( 'jquery-ui', 'https://code.jquery.com/ui/1.12.1/themes/smoothness/jquery-ui.css' );
        wp_enqueue_style( 'jquery-ui' );

		wp_enqueue_style('wp-color-picker');		
		wp_enqueue_style($this->plugin_name.'-select2', plugin_dir_url(__FILE__) . 'css/select2.min.css', array(), $this->version, 'all');
		wp_enqueue_style($this->plugin_name . "-codemirror", plugin_dir_url( __FILE__ ) . 'css/codemirror.css', array(), $this->version, 'all');
		wp_enqueue_style($this->plugin_name.'-bootstrap', plugin_dir_url(__FILE__) . 'css/bootstrap.min.css', array(), $this->version, 'all');
		wp_enqueue_style($this->plugin_name . '-sweetalert2', plugin_dir_url(__FILE__) . 'css/sweetalert2.min.css', array(), $this->version, 'all');
		wp_enqueue_style($this->plugin_name.'-jquery-datetimepicker', plugin_dir_url(__FILE__) . 'css/jquery-ui-timepicker-addon.css', array(), $this->version, 'all');
		wp_enqueue_style($this->plugin_name.'-dataTables-bootstrap4', plugin_dir_url(__FILE__) . 'css/dataTables.bootstrap4.min.css', array(), $this->version, 'all');
		wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/secure-copy-content-protection-admin.css', array(), $this->version, 'all');
		wp_enqueue_style($this->plugin_name . "-font-awesome", plugin_dir_url( __FILE__ ) . 'css/fontawesome.min.css', array(), $this->version, 'all');	
		wp_enqueue_style('animate.css', plugin_dir_url(__FILE__) . 'css/animate.css', array(), $this->version, 'all');
	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts( $hook_suffix ) {
		global $wp_version;

        $version1 = $wp_version;
        $operator = '>=';
        $version2 = '5.5';
        $versionCompare = $this->versionCompare($version1, $operator, $version2);
        if ($versionCompare) {	
            wp_enqueue_script( $this->plugin_name.'-wp-load-scripts', plugin_dir_url(__FILE__) . 'js/ays-wp-load-scripts.js', array(), $this->version, true);
        }

        wp_enqueue_script( $this->plugin_name . "-banner", plugin_dir_url(__FILE__) . 'js/secure-copy-content-protection-banner.js', array('jquery'), $this->version, true);

        $sccp_banner_date = $this->ays_sccp_update_banner_time();
        wp_localize_script($this->plugin_name . '-banner', 'sccpBannerLangObj', array(
            'sccpBannerDate'  	     => $sccp_banner_date,
            'copied'                 => esc_html__( 'Copied!', 'secure-copy-content-protection'),
            'somethingWentWrong'     => __( "Maybe something went wrong.", 'secure-copy-content-protection' ),
            'errorMsg'               => __( "Error", 'secure-copy-content-protection' ),
            'successCopyCoupon'      => __( "Coupon code copied!", 'secure-copy-content-protection' ),
            'failedCopyCoupon'       => __( "Failed to copy coupon code", 'secure-copy-content-protection' ),
        ) );	

		if (false !== strpos($hook_suffix, "plugins.php")){			
			wp_enqueue_script($this->plugin_name . '-sweetalert2', plugin_dir_url(__FILE__) . 'js/sweetalert2.all.min.js', array('jquery'), $this->version, true);
			wp_enqueue_script($this->plugin_name . '-admin', plugin_dir_url(__FILE__) . 'js/admin.js', array('jquery'), $this->version, true);
			wp_localize_script($this->plugin_name . '-admin', 'sccp_admin_ajax', array(
                'ajax_url'              => admin_url('admin-ajax.php'),
                'errorMsg'              => __( "Error", 'secure-copy-content-protection' ),
                'loadResource'          => __( "Can't load resource.", 'secure-copy-content-protection' ),
                'somethingWentWrong'    => __( "Maybe something went wrong.", 'secure-copy-content-protection' ),
            ));
		}

		if (false === strpos($hook_suffix, $this->plugin_name)) {
			return;
		}

		global $wp_roles;
		$ays_users_roles = $wp_roles->roles;

		wp_enqueue_script('jquery-ui-datepicker');
		wp_enqueue_media();

		wp_enqueue_script( $this->plugin_name.'-wp-color-picker-alpha', plugin_dir_url(__FILE__) . 'js/wp-color-picker-alpha.min.js', array('wp-color-picker'), $this->version, true);
		wp_enqueue_script( $this->plugin_name.'-select2', plugin_dir_url(__FILE__) . 'js/select2.min.js', array('jquery'), $this->version, true);		
		wp_enqueue_script( $this->plugin_name.'-dataTables', plugin_dir_url(__FILE__) . 'js/jquery.dataTables.min.js', array('jquery'), $this->version, true);
		wp_enqueue_script( $this->plugin_name.'-dataTables-bootstrap4', plugin_dir_url(__FILE__) . 'js/dataTables.bootstrap4.min.js', array('jquery'), $this->version, true);
		wp_enqueue_script($this->plugin_name . '-sweetalert2', plugin_dir_url(__FILE__) . 'js/sweetalert2.all.min.js', array('jquery'), $this->version, true);		
		wp_enqueue_script('cpy_content_protection_popper', plugin_dir_url(__FILE__) . 'js/popper.min.js', array('jquery'), $this->version, true);
		wp_enqueue_script('cpy_content_protection_bootstrap', plugin_dir_url(__FILE__) . 'js/bootstrap.min.js', array('jquery'), $this->version, true);
		wp_enqueue_script( $this->plugin_name."-jquery.datetimepicker.js", plugin_dir_url( __FILE__ ) . 'js/jquery-ui-timepicker-addon.js', array( 'jquery' ), $this->version, true );
		wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/secure-copy-content-protection-admin.js', array('jquery', 'wp-color-picker'), $this->version, true);
		wp_localize_script($this->plugin_name, 'sccp', array(
			'ajax'           	=> admin_url('admin-ajax.php'),
            'ajaxurl'           => admin_url('admin-post.php'),
			'loader_message' 	=> __('Just a moment...', 'secure-copy-content-protection'),
			'loader_url'     	=> SCCP_ADMIN_URL . '/images/rocket.svg',
			'bc_user_role'    	=> $ays_users_roles,
            "emptyEmailError"               => __( 'Email field is empty', 'secure-copy-content-protection'),
            "invalidEmailError"             => __( 'Invalid Email address', 'secure-copy-content-protection'),
            'selectUser'                    => __( 'Select user', 'secure-copy-content-protection'),
            'pleaseEnterMore'               => __( "Please enter 1 or more characters", 'secure-copy-content-protection' ),
            'searching'                     => __( "Searching...", 'secure-copy-content-protection' ),
            'activated'                     => __( "Activated", 'secure-copy-content-protection' ),
            'errorMsg'                      => __( "Error", 'secure-copy-content-protection' ),
            'loadResource'                  => __( "Can't load resource.", 'secure-copy-content-protection' ),
            'somethingWentWrong'            => __( "Maybe something went wrong.", 'secure-copy-content-protection' ),            
            'greateJob'                     => __( 'Great job', 'secure-copy-content-protection'),
            'formMoreDetailed'              => __( 'For more detailed configuration visit', 'secure-copy-content-protection'),
            'greate'                        => __( 'Great!', 'secure-copy-content-protection'),
		));

		$color_picker_strings = array(
			'clear'            => __( 'Clear', 'secure-copy-content-protection' ),
			'clearAriaLabel'   => __( 'Clear color', 'secure-copy-content-protection' ),
			'defaultString'    => __( 'Default', 'secure-copy-content-protection' ),
			'defaultAriaLabel' => __( 'Select default color', 'secure-copy-content-protection' ),
			'pick'             => __( 'Select Color', 'secure-copy-content-protection' ),
			'defaultLabel'     => __( 'Color value', 'secure-copy-content-protection' ),
		);
		wp_localize_script( $this->plugin_name.'-wp-color-picker-alpha', 'wpColorPickerL10n', $color_picker_strings );

		wp_localize_script($this->plugin_name, 'sccpLangObj', array(
            // 'sccpBannerDate'  => $sccp_banner_date,
            'nameField'       => __( 'Name field', 'secure-copy-content-protection' ),
            'title'           => __( 'Tick the checkbox to show the Name field', 'secure-copy-content-protection' ),
            'descField'       => __( 'Description field', 'secure-copy-content-protection' ),
            'descTitle'       => __( 'Tick the checkbox to show the Description field', 'secure-copy-content-protection' ),
            'adminUrl'        => SCCP_ADMIN_URL,
        ) );		
		
		wp_enqueue_script( $this->plugin_name."-xlsx.core.min.js", plugin_dir_url( __FILE__ ) . 'js/xlsx.core.min.js', array( 'jquery' ), $this->version, true );
		wp_enqueue_script( $this->plugin_name."-fileSaver.js", plugin_dir_url( __FILE__ ) . 'js/FileSaver.js', array( 'jquery' ), $this->version, true );	
		wp_enqueue_script( $this->plugin_name."-jhxlsx.js", plugin_dir_url( __FILE__ ) . 'js/jhxlsx.js', array( 'jquery' ), $this->version, true );

		wp_enqueue_script( $this->plugin_name."-codemirror", plugin_dir_url( __FILE__ ) . 'js/codemirror.min.js', array( 'jquery' ), $this->version, true );

	}

	/**
     * De-register JavaScript files for the admin area.
     *
     * @since    1.0.0
     */
    public function disable_scripts($hook_suffix) {
        if (false !== strpos($hook_suffix, $this->plugin_name)) {
            if (is_plugin_active('ai-engine/ai-engine.php')) {
                wp_deregister_script('mwai');
                wp_deregister_script('mwai-vendor');
                wp_dequeue_script('mwai');
                wp_dequeue_script('mwai-vendor');
            }

            if (is_plugin_active('html5-video-player/html5-video-player.php')) {
                wp_dequeue_style('h5vp-admin');
                wp_dequeue_style('fs_common');
            }

            if (is_plugin_active('panorama/panorama.php')) {
                wp_dequeue_style('bppiv_admin_custom_css');
                wp_dequeue_style('bppiv-custom-style');
            }

            if (is_plugin_active('wp-social/wp-social.php')) {
                wp_dequeue_style('wp_social_select2_css');
                wp_deregister_script('wp_social_select2_js');
                wp_dequeue_script('wp_social_select2_js');
            }

            if (is_plugin_active('real-media-library-lite/index.php')) {
                wp_dequeue_style('real-media-library-lite-rml');
            }

            // Theme | Pixel Ebook Store
            wp_dequeue_style('pixel-ebook-store-free-demo-content-style');

            // Theme | Interactive Education
            wp_dequeue_style('interactive-education-free-demo-content-style');

            // Theme | Phlox 2.17.6
            wp_dequeue_style('auxin-admin-style');
        }
    }

    public function ays_sccp_disable_all_notice_from_plugin() {
        if (!function_exists('get_current_screen')) {
            return;
        }

        $screen = get_current_screen();

        if (empty($screen) || strpos($screen->id, $this->plugin_name) === false) {
            return;
        }

        global $wp_filter;

        // Keep plugin-specific notices
        $our_plugin_notices = array();

        $exclude_functions = [
            'sccp_admin_notice',
        ];

        if (!empty($wp_filter['admin_notices'])) {
            foreach ($wp_filter['admin_notices']->callbacks as $priority => $callbacks) {
                foreach ($callbacks as $key => $callback) {
                    // For class-based methods
                    if (
                        is_array($callback['function']) &&
                        is_object($callback['function'][0]) &&
                        get_class($callback['function'][0]) === __CLASS__
                    ) {
                        $our_plugin_notices[$priority][$key] = $callback;
                    }                    
                    elseif (
                        is_array($callback['function']) &&
                        is_object($callback['function'][0]) &&
                        get_class($callback['function'][0]) === 'Ays_Sccp_Data'
                    ) {
                        $our_plugin_notices[$priority][$key] = $callback;
                    }
                    // For standalone functions
                    elseif (
                        is_string($callback['function']) &&
                        in_array($callback['function'], $exclude_functions)
                    ) {
                        $our_plugin_notices[$priority][$key] = $callback;
                    }
                }
            }
        }

        // Remove all notices
        remove_all_actions('admin_notices');
        remove_all_actions('all_admin_notices');

        // Re-add only your plugin's notices
        foreach ($our_plugin_notices as $priority => $callbacks) {
            foreach ($callbacks as $callback) {
                add_action('admin_notices', $callback['function'], $priority);
            }
        }
    }

	function codemirror_enqueue_scripts($hook) {
		if (strpos($hook, $this->plugin_name) !== false) {
			if(function_exists('wp_enqueue_code_editor')){
	            $cm_settings['codeEditor'] = wp_enqueue_code_editor(array(
	                'type' => 'text/css',
	                'codemirror' => array(
	                    'inputStyle' => 'contenteditable',
	                    'theme' => 'cobalt',
	                )
	            ));
	        
		        wp_localize_script('wp-theme-plugin-editor', 'cm_settings', $cm_settings);
		       
		        wp_enqueue_script('wp-theme-plugin-editor');
	            wp_enqueue_style('wp-codemirror');
	            
	        }
		}
        
	}

	function versionCompare($version1, $operator, $version2) {
   
        $_fv = intval ( trim ( str_replace ( '.', '', $version1 ) ) );
        $_sv = intval ( trim ( str_replace ( '.', '', $version2 ) ) );
       
        if (strlen ( $_fv ) > strlen ( $_sv )) {
            $_sv = str_pad ( $_sv, strlen ( $_fv ), 0 );
        }
       
        if (strlen ( $_fv ) < strlen ( $_sv )) {
            $_fv = str_pad ( $_fv, strlen ( $_sv ), 0 );
        }
       
        return version_compare ( ( string ) $_fv, ( string ) $_sv, $operator );
    }

	/**
	 * Register the administration menu for this plugin into the WordPress Dashboard menu.
	 *
	 * @since    1.0.0
	 */
	public function add_plugin_admin_menu() {
		$hook_sccp = add_menu_page(
			'Copy Protection', 
			'Copy Protection', 
			'manage_options', $this->plugin_name, 
			array(
			$this,
			'display_plugin_setup_page'
		), SCCP_ADMIN_URL . '/images/icons/icon-sccp-128x128.svg', 6);
		add_action( "load-$hook_sccp", array( $this, 'add_tabs' ));

		$hook_subscribe_to_view = add_submenu_page( $this->plugin_name,
            __('Subscribe to view', 'secure-copy-content-protection'),
            __('Subscribe to view', 'secure-copy-content-protection'),
            'manage_options',
            $this->plugin_name . '-subscribe-to-view',
            array($this, 'display_plugin_sccp_subscribe_to_view_page') 
        );
		add_action( "load-$hook_subscribe_to_view", array( $this, 'add_tabs' ));

        global $wpdb;
        $sql = "SELECT COUNT(*) FROM {$wpdb->prefix}ays_sccp_reports WHERE `unread` = 1";
        $unread_results_count = $wpdb->get_var($sql);
        $results_text = __('Results', 'secure-copy-content-protection');
        $menu_item = ($unread_results_count == 0) ? $results_text : $results_text . '<span class="ays_menu_badge ays_results_bage">' . $unread_results_count . '</span>';
		$hook_results = add_submenu_page( $this->plugin_name,
			$results_text,
            $menu_item,
            'manage_options',
            $this->plugin_name . '-results-to-view',
            array($this, 'display_plugin_sccp_results_to_view_page') 
        );
        add_action("load-$hook_results", array($this, 'screen_option_results'));
		add_action( "load-$hook_results", array( $this, 'add_tabs' ));

		$hook_settings = add_submenu_page( $this->plugin_name,
            __('General Settings', 'secure-copy-content-protection'),
            __('General Settings', 'secure-copy-content-protection'),
            'manage_options',
            $this->plugin_name . '-settings',
            array($this, 'display_plugin_sccp_settings_page') 
        );
        add_action("load-$hook_settings", array($this, 'screen_option_settings'));
		add_action( "load-$hook_settings", array( $this, 'add_tabs' ));

		$hook_featured_plugins = add_submenu_page( $this->plugin_name,
            __('Our Products', 'secure-copy-content-protection'),
            __('Our Products', 'secure-copy-content-protection'),
            'manage_options',
            $this->plugin_name . '-featured-plugins',
            array($this, 'display_plugin_sccp_featured_plugins_page') 
        );
		add_action( "load-$hook_featured_plugins", array( $this, 'add_tabs' ));

		$hook_pro_features = add_submenu_page(
			$this->plugin_name,
			__('PRO Features', 'secure-copy-content-protection'),
			__('PRO Features', 'secure-copy-content-protection'),
			'manage_options',
			$this->plugin_name . '-pro-features',
			array($this, 'display_plugin_sccp_pro_features_page')
		);
		add_action( "load-$hook_pro_features", array( $this, 'add_tabs' ));
	}

	public function add_tabs() {
		$screen = get_current_screen();
	
		if ( ! $screen) {
			return;
		}
	
		$screen->add_help_tab(
			array(
				'id'      => 'sccp_help_tab',
				'title'   => __( 'General Information:
					', 'secure-copy-content-protection'),
				'content' =>
					'<h2>' . __( 'SCCP Information', 'secure-copy-content-protection') . '</h2>' .
					'<p>' .
						__( 'Copy Content Protection is a must-have WordPress plugin which prevents the risk of plagiarism on your website. After the activation of the plugin the Copy and Paste, right-click option, inspect elements (F12 key), content-selection, copy the image, save image as features will be automatically disabled. In addition, the user has an option to enable or disable the features via the checkbox.',  'secure-copy-content-protection' ).'</p>'
			)
		);
	
		$screen->set_help_sidebar(
			'<p><strong>' . __( 'For more information:', 'secure-copy-content-protection') . '</strong></p>' .
			'<p>
				<a href="https://www.youtube.com/watch?v=whYBGV703SM" target="_blank">' . __( 'Youtube video tutorials' , 'secure-copy-content-protection' ) . '</a>
			</p>' .
			'<p>
				<a href="https://ays-pro.com/wordpress-copy-content-protection-user-manual" target="_blank">' . __( 'Documentation', 'secure-copy-content-protection' ) . '</a>
			</p>' .
			'<p>
				<a href="https://ays-pro.com/wordpress/secure-copy-content-protection" target="_blank">' . __( 'Copy Protection plugin Premium version', 'secure-copy-content-protection' ) . '</a>
			</p>'
		);
	}

	public function ays_sccp_get_user_roles_by_userId($id){
        $user_meta = get_userdata($id);
        return $user_meta->roles;
	}

	public function ays_sccp_results_export_xlsx($results){
        
		global $wpdb;
		error_reporting(0);        

        $results_array = array();
		$results_headers = array(
            array( 'text' => "Shortcode ID" ),
            array( 'text' => "User Email" ),
            array( 'text' => "User Name" ),
            array( 'text' => "User IP" ),
            array( 'text' => "Date" ),
            array( 'text' => "WP User" ),
            array( 'text' => "User Roles" ),
            array( 'text' => "City, Country" )
		);

        $results_array[] = $results_headers;
        foreach ($results as $key => $result){
        	
        	$user_roles = $this->ays_sccp_get_user_roles_by_userId($result['user_id']);

        	$role = "";
        	if ( $user_roles && !is_null( $user_roles ) && is_array($user_roles) ) {
        		$role = count($user_roles) > 1 ? implode(", ", $user_roles) : implode("", $user_roles);
        	}
        	
            $result['user_id'] = $result['user_id'] > 0 ? get_user_by('ID', $result['user_id'])->display_name : "Guest";
            $res_array = array(
                array( 'text' => $result['subscribe_id'] ),
                array( 'text' => $result['subscribe_email'] ),
                array( 'text' => $result['user_name'] ),
                array( 'text' => $result['user_ip'] ),
                array( 'text' => $result['vote_date'] ),
                array( 'text' => $result['user_id'] ),
                array( 'text' => $role ),
                array( 'text' => $result['user_address'] )
            );            
      
            $results_array[] = $res_array;
        }
        
		$response = array(
			'status' => true,
			'data'   => $results_array,
			"type"   => 'xlsx'
		);
		return $response;
    }

    public function ays_sccp_results_export_csv( $results ){
        
        if ( ! current_user_can('manage_options') ) {
            return array('status' => false, 'error' => 'no_permission');
        }

        global $wpdb;
        error_reporting(0);

        $upload = wp_upload_dir();
        $private_dir = trailingslashit($upload['basedir']) . 'sccp-private/';

        if ( ! file_exists($private_dir) ) {
            wp_mkdir_p($private_dir);
            
            @file_put_contents($private_dir . '.htaccess', "Require all denied\n");
            
            @file_put_contents($private_dir . 'index.php', "<?php http_response_code(403); exit('Access denied');");
        }

        $filename = 'export-' . wp_generate_password(20, false, false) . '.csv';
        $file_path = $private_dir . $filename;

        $export_file = fopen($file_path, 'w');

        if (!$export_file) {
            return array('status' => false, 'error' => 'cannot_create_file');
        }

        fputs($export_file, chr(0xEF) . chr(0xBB) . chr(0xBF));

        $export_file_fields = array('Shortcode ID', 'User Email', 'User Name', 'User IP', 'Date', 'WP User', 'User Roles', 'City, Country');
        fputcsv($export_file, $export_file_fields);

        foreach ($results as $f_value) {
            $user_roles = $this->ays_sccp_get_user_roles_by_userId($f_value['user_id']);
            $role = "";
            if ( $user_roles && !is_null($user_roles) && is_array($user_roles) ) {
                $role = count($user_roles) > 1 ? implode(", ", $user_roles) : implode("", $user_roles);
            }
            
            $row = $f_value;
            array_splice($row, 5, 0, $role);
            $row['user_id'] = $row['user_id'] > 0 ? get_user_by('ID', $row['user_id'])->display_name : "Guest";
            
            fputcsv($export_file, $row);
        }
        
        fclose($export_file);

        $token = wp_generate_password(24, false, false);
        $user_id = get_current_user_id();
        $transient_key = 'sccp_export_' . $user_id . '_' . $token;
        set_transient($transient_key, $file_path, HOUR_IN_SECONDS);

        $response = array(
            'status' => true,
            'token'  => $token,
            'type'   => 'csv'
        );

        return $response;
    }

    public function handle_sccp_download() {
        if ( ! is_user_logged_in() || ! current_user_can('manage_options') ) {
            wp_die('Access denied', '', 403);
        }

        $token = isset($_GET['token']) ? sanitize_text_field($_GET['token']) : '';
        if ( empty($token) ) {
            wp_die('Token not found', '', 400);
        }

        $user_id = get_current_user_id();
        $transient_key = 'sccp_export_' . $user_id . '_' . $token;
        $file_path = get_transient($transient_key);

        if ( ! $file_path || ! file_exists($file_path) ) {
            wp_die('File not found or expired', '', 404);
        }

        header('Content-Description: File Transfer');
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="sccp-export.csv"');
        header('Content-Length: ' . filesize($file_path));
        header('Cache-Control: must-revalidate');
        header('Pragma: public');
        
        if (ob_get_level()) {
            ob_end_clean();
        }
        
        readfile($file_path);

        delete_transient($transient_key);
        unlink($file_path);
        exit;
    }

    public function ays_sccp_results_export_json($results){
        
		global $wpdb;
		error_reporting(0);
        $results_array = array();
        foreach ($results as $key => $result){

            $user_roles = $this->ays_sccp_get_user_roles_by_userId($result['user_id']);
        	$role = "";
        	if ( $user_roles && !is_null( $user_roles ) && is_array($user_roles) ) {
        		$role = count($user_roles) > 1 ? implode(", ", $user_roles) : implode("", $user_roles);
        	}
        	
            $user_id = $result['user_id'] > 0 ? get_user_by('ID', $result['user_id'])->display_name : "Guest";
            $res_array = array(
                'subscribe_id'	  => $result['subscribe_id'],
                'subscribe_email' => $result['subscribe_email'],
                'subscribe_email' => $result['user_name'],
                'user_ip' 		  => $result['user_ip'],
                'vote_date' 	  => $result['vote_date'],
                'user_id' 		  => $user_id,
                'user_roles' 	  => $role,
                'user_address'    => $result['user_address']            
            );

            $results_array[] = $res_array;
        }
        
		$response = array(
			'status' => true,
			'data'   => $results_array,
			"type"   => 'json'
		);
		return $response;
    } 

	public function ays_sccp_results_export_file(){

        // Run a security check.
        check_ajax_referer( 'sccp-ajax-export-results-nonce', sanitize_key( $_REQUEST['_ajax_nonce'] ) );
        
    	global $wpdb;
		error_reporting(0);

		if ( current_user_can('administrator') ) {
			$reports_table = esc_sql($wpdb->prefix . "ays_sccp_reports");

			$type = isset($_REQUEST['type']) ? sanitize_text_field( $_REQUEST['type'] ) : '';

			$shortcode_ids = "SELECT DISTINCT subscribe_id FROM {$reports_table}";
			$short_id = (isset($_REQUEST['sccp_id']) && $_REQUEST['sccp_id'] != null) ? implode(',', array_map('intval', $_REQUEST['sccp_id'])) : esc_sql($shortcode_ids);
			$date_from = isset($_REQUEST['date_from']) && $_REQUEST['date_from'] != '' ? esc_sql($_REQUEST['date_from'])  : esc_sql('2000-01-01');
			$date_to = isset($_REQUEST['date_to']) && $_REQUEST['date_to'] != '' ? esc_sql($_REQUEST['date_to'])  : esc_sql(current_time('Y-m-d'));
			
			$formfields = array();
			if ( ! empty( $short_id ) && $short_id != "") {
				$sql = "SELECT subscribe_id, subscribe_email, user_name, user_ip, vote_date, user_id, user_address 
						FROM {$reports_table}
						WHERE subscribe_id IN ($short_id) 
						AND vote_date BETWEEN '$date_from' AND '$date_to 23:59:59'";

				$formfields = $wpdb->get_results($sql, 'ARRAY_A');
			}

			switch($type){
				case 'csv':
					$export_data = $this->ays_sccp_results_export_csv($formfields);
				break;
				case 'xlsx':
					$export_data = $this->ays_sccp_results_export_xlsx($formfields);
				break;
				case 'json':
					$export_data = $this->ays_sccp_results_export_json($formfields);
				break;
			}

			ob_end_clean();
			$ob_get_clean = ob_get_clean();
			echo json_encode($export_data);
			wp_die();
		}else{
			ob_end_clean();
	        $ob_get_clean = ob_get_clean();
	        echo json_encode(array(
				'status' => false
			));
	        wp_die();
		}

    }

    // EXPORT FILTERS AV
    public function ays_sccp_show_filters(){
        error_reporting(0);
        global $wpdb;

        if (isset($_REQUEST['action']) && $_REQUEST['action'] == 'ays_sccp_show_filters' && current_user_can('administrator')) {
        	$reports_table = esc_sql($wpdb->prefix . "ays_sccp_reports");          

			$shortcode_ids = $wpdb->get_results("SELECT DISTINCT subscribe_id FROM {$reports_table}", "ARRAY_A");

            $sql = "SELECT COUNT(subscribe_id) FROM {$reports_table} ORDER BY subscribe_id DESC";
            $short_count = $wpdb->get_var($sql);

            ob_end_clean();
	        $ob_get_clean = ob_get_clean();
            echo json_encode(array(
                "shortcode" => $shortcode_ids,
                "count" => $short_count
            ));
            wp_die();
        } else {
        	ob_end_clean();
	        $ob_get_clean = ob_get_clean();
        	echo json_encode(array(
                "shortcode" => array(),
                "count" => 0
            ));
            wp_die();
        }
    }

    public function ays_sccp_results_export_filter(){
        global $wpdb;
        error_reporting(0);
		if (isset($_REQUEST['action']) && $_REQUEST['action'] == 'ays_sccp_results_export_filter') {
			$reports_table = esc_sql($wpdb->prefix . "ays_sccp_reports");

			if ( current_user_can('administrator') ) {
				$shortcode_ids = "SELECT DISTINCT subscribe_id FROM {$reports_table}";
				$short_id = (isset($_REQUEST['sccp_id']) && $_REQUEST['sccp_id'] != null) ? implode(',', array_map('intval', $_REQUEST['sccp_id'])) : esc_sql($shortcode_ids);

				$date_from = isset($_REQUEST['date_from']) && $_REQUEST['date_from'] != '' ? esc_sql($_REQUEST['date_from']) : esc_sql('2000-01-01');
				$date_to = isset($_REQUEST['date_to']) && $_REQUEST['date_to'] != '' ? esc_sql($_REQUEST['date_to']) : esc_sql(current_time('Y-m-d'));

				$sql = "SELECT COUNT(subscribe_id) AS qanak FROM {$reports_table}
				WHERE subscribe_id IN ($short_id)
				AND vote_date BETWEEN '$date_from' AND '$date_to 23:59:59'";
				$results = $wpdb->get_var($sql);

				ob_end_clean();
				$ob_get_clean = ob_get_clean();
				$res = array(
					'results' => $results
				);
				echo json_encode($res);
				wp_die();
		 	}else{
				ob_end_clean();
				$ob_get_clean = ob_get_clean();
				echo json_encode(array(
					'status'  => false,
					'results' => 0
				));
				wp_die();
		 	}
		}else{
			ob_end_clean();
			$ob_get_clean = ob_get_clean();
			echo json_encode(array(
				'status'  => false,
				'results' => 0
			));
			wp_die();
	 	}
    }

    public static function set_screen($status, $option, $value){
        return $value;
    }
	/**
	 * Add settings action link to the plugins page.
	 *
	 * @since    1.0.0
	 */

	public function add_action_links( $links ) {
		/*
		*  Documentation : https://codex.wordpress.org/Plugin_API/Filter_Reference/plugin_action_links_(plugin_file_name)
		*/
		$settings_link = array(
			'<a href="' . admin_url('options-general.php?page=' . $this->plugin_name) . '">' . __('Settings', 'secure-copy-content-protection') . '</a>',
			'<a href="https://ays-demo.com/secure-copy-content-protection-free-demo/" target="_blank">' . __('Demo', 'secure-copy-content-protection') . '</a>',
            '<a href="https://ays-pro.com/wordpress/secure-copy-content-protection?utm_source=dashboard-sccp&utm_medium=free-sccp&utm_campaign=buy-now-sccp" class="ays-sccp-upgrade-plugin-btn" target="_blank" style="color:#01A32A; font-weight:bold;">' . __('Upgrade 30% Sale', 'secure-copy-content-protection') . '</a>',
		);

		return array_merge($settings_link, $links);

	}

 	public function add_plugin_row_meta($meta, $file) {
		if ($file == SCCP_BASENAME) {
			$meta[] = '<a href="https://wordpress.org/support/plugin/secure-copy-content-protection/" target="_blank">' . esc_html__( 'Free Support', 'secure-copy-content-protection' ) . '</a>';
		}

		return $meta;
	}


	public function display_plugin_setup_page() {
		$this->settings_obj = new Sccp_Settings_Actions($this->plugin_name);
		require_once('partials/secure-copy-content-protection-admin-display.php');
	}

	public function screen_option_settings() {
        $this->settings_obj = new Sccp_Settings_Actions($this->plugin_name);
    }

	public function display_plugin_sccp_settings_page(){
        include_once('partials/settings/secure-copy-content-protection-settings.php');
    }		

	public function display_plugin_sccp_featured_plugins_page(){
        include_once('partials/features/secure-copy-content-protection-featured-display.php');
    }

    public function display_plugin_sccp_pro_features_page() {
		include_once('partials/features/secure-copy-content-protection-pro-features.php');
	}

	public function display_plugin_sccp_subscribe_to_view_page() {
		include_once('partials/subscribe/secure-copy-content-protection-subscribe-display.php');
    }

	public function display_plugin_sccp_results_to_view_page() {
		include_once('partials/results/secure-copy-content-protection-results-display.php');
    }

	public function deactivate_sccp_option() {		

		if( is_user_logged_in() ) {
            $request_value = esc_sql( sanitize_text_field( $_REQUEST['upgrade_plugin'] ) );
            $upgrade_option = get_option('sccp_upgrade_plugin','');
            if($upgrade_option === ''){
                add_option('sccp_upgrade_plugin', $request_value);
            }else{
                update_option('sccp_upgrade_plugin', $request_value);
            }
            ob_end_clean();
            $ob_get_clean = ob_get_clean();
            echo json_encode(array(
                'option' => get_option('sccp_upgrade_plugin', '')
            ));
            wp_die();
        } else {
            ob_end_clean();
            $ob_get_clean = ob_get_clean();
            echo json_encode(array(
                'option' => ''
            ));
            wp_die();
        }
	}

	public function screen_option_results() {
		$option = 'per_page';
		$args   = array(
			'label'   => __('Results', 'secure-copy-content-protection'),
			'default' => 7,
			'option'  => 'sccp_results_per_page',
		);

		add_screen_option($option, $args);
		$this->results_obj = new Sccp_Results_List_Table($this->plugin_name);
		$this->settings_obj = new Sccp_Settings_Actions($this->plugin_name);

	}

	// Mailchimp - Get mailchimp lists
    public function ays_get_mailchimp_lists($username, $api_key){
        error_reporting(0);
        if($username == ""){
            return array(
                'total_items' => 0
            );
        }
        if($api_key == ""){
            return array(
                'total_items' => 0
            );
        }
        
        $api_prefix = explode("-",$api_key);
        $api_prefix = isset($api_prefix[1]) && $api_prefix[1] != "" ? $api_prefix[1] : '';
        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => "https://".$api_prefix.".api.mailchimp.com/3.0/lists/?count=100",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "GET",
            CURLOPT_USERPWD => "$username:$api_key",
            CURLOPT_HTTPAUTH => CURLAUTH_BASIC,
            CURLOPT_HTTPHEADER => array(
                "Content-Type: application/json",
                "cache-control: no-cache"
            ),
        ));

        $response = curl_exec($curl);
        $err = curl_error($curl);

        curl_close($curl);

        if ($err) {
      		//echo "cURL Error #:" . $err;
        } else {
            return json_decode($response, true);
        }
    }
    public function sccp_admin_footer($a){
        if(isset($_REQUEST['page'])){
            if(false !== strpos($_REQUEST['page'], $this->plugin_name)){
                ?>
                <div class="ays-sccp-footer-support-box">
                    <span class="ays-sccp-footer-link-row"><a href="https://wordpress.org/support/plugin/secure-copy-content-protection/" target="_blank"><?php echo esc_html(__( "Support", 'secure-copy-content-protection') ); ?></a></span>
                    <span class="ays-sccp-footer-slash-row">/</span>
                    <span class="ays-sccp-footer-link-row"><a href="https://ays-pro.com/wordpress-copy-content-protection-user-manual" target="_blank"><?php echo esc_html(__( "Docs", 'secure-copy-content-protection')); ?></a></span>
                    <span class="ays-sccp-footer-slash-row">/</span>
                    <span class="ays-sccp-footer-link-row"><a href="https://ays-demo.com/copy-protection-plugin-survey/" target="_blank"><?php echo esc_html(__( "Suggest a Feature", 'secure-copy-content-protection')); ?></a></span>
                </div>
                <p style="font-size:13px;text-align:center;font-style:italic;">
                    <span style="margin-left:0px;margin-right:10px;" class="ays_heart_beat"><i class="ays_fa ays_fa_heart animated"></i></span>
                    <span><?php echo esc_html(__( "If you love our plugin, please do big favor and rate us on", 'secure-copy-content-protection')); ?></span> 
                    <a target="_blank" href='https://wordpress.org/support/plugin/secure-copy-content-protection/reviews/?rate=5#new-post'>WordPress.org</a>
                    <a target="_blank" class="ays-rated-link" href='https://wordpress.org/support/plugin/secure-copy-content-protection/reviews/'>
                        <span class="ays-dashicons ays-dashicons-star-empty"></span>
                        <span class="ays-dashicons ays-dashicons-star-empty"></span>
                        <span class="ays-dashicons ays-dashicons-star-empty"></span>
                        <span class="ays-dashicons ays-dashicons-star-empty"></span>
                        <span class="ays-dashicons ays-dashicons-star-empty"></span>
                    </a>
                    <span class="ays_heart_beat"><i class="ays_fa ays_fa_heart animated"></i></span>
                </p>
            <?php
            }
        }
    }

	// Mailchimp update list
	public static function ays_add_mailchimp_update_list($username, $api_key, $list_id, $args){
		if($username == "" || $api_key == ""){
			return false;
		}

		if( $list_id == '' ){
			return false;
		}

		if( ! isset( $args['double_optin'] ) || ! array_key_exists( 'double_optin', $args ) ){
			return false;
		}

		$list_data = self::ays_get_mailchimp_list( $username, $api_key, $list_id );

		if( empty( $list_data ) ){
			return false;
		}

		$double_optin = isset( $args['double_optin'] ) && $args['double_optin'] == 'on' ? true : false;

		$fields = array(
			"name"                   => $list_data['name'],
			"contact"                => $list_data['contact'],
			"permission_reminder"    => $list_data['permission_reminder'],
			"use_archive_bar"        => $list_data['use_archive_bar'],
			"campaign_defaults"      => $list_data['campaign_defaults'],
			"email_type_option"      => $list_data['email_type_option'],
			"double_optin"           => $double_optin,
		);

		$api_prefix = explode("-",$api_key)[1];

		$curl = curl_init();
		curl_setopt_array($curl, array(
			CURLOPT_URL => "https://".$api_prefix.".api.mailchimp.com/3.0/lists/".$list_id."/",
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING => "",
			CURLOPT_MAXREDIRS => 10,
			CURLOPT_TIMEOUT => 30,
			CURLOPT_SSL_VERIFYPEER => false,
			CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
			CURLOPT_USERPWD => "$username:$api_key",
			CURLOPT_CUSTOMREQUEST => "PATCH",
			CURLOPT_POSTFIELDS => json_encode($fields),
			CURLOPT_HTTPHEADER => array(
				"Content-Type: application/json",
				"cache-control: no-cache"
			),
		));

		$response = curl_exec($curl);

		$err = curl_error($curl);

		curl_close($curl);

		if ($err) {
			return "cURL Error #: " . $err;
		} else {
			return json_decode( $response, true );
		}
	}

	// Mailchimp - Get mailchimp list
	public static function ays_get_mailchimp_list($username, $api_key, $list_id){
		error_reporting(0);
		if($username == ""){
			return array();
		}
		if($api_key == ""){
			return array();
		}
		if($list_id == ""){
			return array();
		}

		$api_prefix = explode("-",$api_key)[1];

		$curl = curl_init();

		curl_setopt_array($curl, array(
			CURLOPT_URL => "https://".$api_prefix.".api.mailchimp.com/3.0/lists/".$list_id,
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING => "",
			CURLOPT_MAXREDIRS => 10,
			CURLOPT_TIMEOUT => 30,
			CURLOPT_SSL_VERIFYPEER => false,
			CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
			CURLOPT_CUSTOMREQUEST => "GET",
			CURLOPT_USERPWD => "$username:$api_key",
			CURLOPT_HTTPAUTH => CURLAUTH_BASIC,
			CURLOPT_HTTPHEADER => array(
				"Content-Type: application/json",
				"cache-control: no-cache"
			),
		));

		$response = curl_exec($curl);
		$err = curl_error($curl);

		curl_close($curl);

		if ($err) {
//            echo "cURL Error #:" . $err;
		} else {
			return json_decode($response, true);
		}
	}
	
	public function ays_sccp_reports_user_search() {
        error_reporting(0);
        global $wpdb;

        check_ajax_referer( 'sccp-reports-user-search-nonce', sanitize_key( $_REQUEST['_ajax_nonce'] ) );

        $response = array(
            'results' => array()
        );

        if( current_user_can( 'manage_options' ) ){

	        $search = isset($_REQUEST['search']) && $_REQUEST['search'] != '' ? sanitize_text_field( $_REQUEST['search'] ) : null;
	        $checked = isset($_REQUEST['val']) && $_REQUEST['val'] !='' ? sanitize_text_field( $_REQUEST['val'] ) : null;
	        $users_sql = "SELECT user_id
	                       FROM {$wpdb->prefix}ays_sccp_reports
	                       GROUP BY user_id";
	        $users = $wpdb->get_results($users_sql,"ARRAY_A");
	        $args = array();
	        $arg = '';

	        if($search !== null){
	             $arg .= $search;
	             $arg .= '*';
	             $args['search'] = $arg;
	        }
	        $guest = false;
	        foreach ($users as $key => $value ) {
	            $args['include'][] = $value['user_id'];
	            if ( $value['user_id'] == '0' && strpos('guest', strtolower($search)) !== false ) {
	            	$guest = true;
	            }
	        }

	        $reports_users = get_users($args);
	        
	        if(empty($args)){
	            $reports_users = '';
	        }

	        foreach ($reports_users as $key => $user) {
	            if ($checked !== null) {
	                if (in_array($user->ID, $checked)) {
	                    continue;
	                }else{
	                    $response['results'][] = array(
	                        'id' => $user->ID,
	                        'text' => $user->data->display_name
	                    );
	                }
	            }else{
	                $response['results'][] = array(
	                    'id' => $user->ID,
	                    'text' => $user->data->display_name,
	                );
	            }
	        }
	        if ($guest) {
	        	$response['results'][] = array(
	                'id' => 0,
	                'text' => 'Guest',
	            );
	        }        
        }        

        ob_end_clean();
        echo json_encode($response);
        wp_die();
    }

    public function ays_sccp_dismiss_button(){

        $data = array(
            'status' => false,
        );

        if (isset($_REQUEST['action']) && $_REQUEST['action'] == 'ays_sccp_dismiss_button') { 
            if( (isset( $_REQUEST['_ajax_nonce'] ) && wp_verify_nonce( $_REQUEST['_ajax_nonce'], SCCP_NAME . '-sale-banner' )) && current_user_can( 'manage_options' )){
                update_option('ays_sccp_sale_btn', 1);
                update_option('ays_sccp_sale_date', current_time( 'mysql' ));
                $data['status'] = true;
            }
        }

        ob_end_clean();
        $ob_get_clean = ob_get_clean();
        echo json_encode($data);
        wp_die();

    }

    public function ays_sccp_update_banner_time(){

        $date = time() + ( 3 * 24 * 60 * 60 ) + (int) ( get_option( 'gmt_offset' ) * HOUR_IN_SECONDS);
        // $date = time() + ( 60 ) + (int) ( get_option( 'gmt_offset' ) * HOUR_IN_SECONDS); // for testing | 1 min
        $next_3_days = date('M d, Y H:i:s', $date);

        $ays_sccp_banner_time = get_option('ays_new_sccp_banner_time');

        if ( !$ays_sccp_banner_time || is_null( $ays_sccp_banner_time ) ) {
            update_option('ays_new_sccp_banner_time', $next_3_days ); 
        }

        $get_ays_sccp_banner_time = get_option('ays_new_sccp_banner_time');

        $val = 60*60*24*0.5; // half day
        // $val = 60; // for testing | 1 min

        $current_date = current_time( 'mysql' );
        $date_diff = strtotime($current_date) - intval(strtotime($get_ays_sccp_banner_time));

        $days_diff = $date_diff / $val;
        if(intval($days_diff) > 0 ){
            update_option('ays_new_sccp_banner_time', $next_3_days);
        }

        return $get_ays_sccp_banner_time;
    }

    public function ays_sccp_generate_message_vars_html( $sccp_message_vars ) {
        $content = array();
        $var_counter = 0; 

        $content[] = '<div class="ays-sccp-message-vars-box">';
            $content[] = '<div class="ays-sccp-message-vars-icon">';
                $content[] = '<div>';
                    $content[] = '<i class="ays_fa ays_fa_link"></i>';
                $content[] = '</div>';
                $content[] = '<div>';
                    $content[] = '<span>'. __("Message Variables" , 'secure-copy-content-protection') .'</span>';
                    $content[] = '<a class="ays_help" data-toggle="tooltip" data-html="true" title="'. __("Insert your preferred message variable into the editor by clicking." , 'secure-copy-content-protection') .'">';
                        $content[] = '<i class="ays_fa ays_fa_info_circle"></i>';
                    $content[] = '</a>';
                $content[] = '</div>';
            $content[] = '</div>';
            $content[] = '<div class="ays-sccp-message-vars-data">';
                foreach($sccp_message_vars as $var => $var_name){
                    $var_counter++;
                    $content[] = '<label class="ays-sccp-message-vars-each-data-label">';
                        $content[] = '<input type="radio" class="ays-sccp-message-vars-each-data-checker" hidden id="ays_sccp_message_var_count_'. $var_counter .'" name="ays_sccp_message_var_count">';
                        $content[] = '<div class="ays-sccp-message-vars-each-data">';
                            $content[] = '<input type="hidden" class="ays-sccp-message-vars-each-var" value="'. $var .'">';
                            $content[] = '<span>'. $var_name .'</span>';
                        $content[] = '</div>';
                    $content[] = '</label>';
                }
            $content[] = '</div>';
        $content[] = '</div>';

        $content = implode( '', $content );

        return $content;
    }

    /**
     * Determine if the plugin/addon installations are allowed.
     *
     * @since 1.3.9
     *
     * @param string $type Should be `plugin` or `addon`.
     *
     * @return bool
     */
    public static function ays_sccp_can_install( $type ) {

        return self::ays_sccp_can_do( 'install', $type );
    }

    /**
     * Determine if the plugin/addon activations are allowed.
     *
     * @since 1.3.9
     *
     * @param string $type Should be `plugin` or `addon`.
     *
     * @return bool
     */
    public static function ays_sccp_can_activate( $type ) {

        return self::ays_sccp_can_do( 'activate', $type );
    }

    /**
     * Determine if the plugin/addon installations/activations are allowed.
     *
     * @since 1.3.9
     *
     * @param string $what Should be 'activate' or 'install'.
     * @param string $type Should be `plugin` or `addon`.
     *
     * @return bool
     */
    public static function ays_sccp_can_do( $what, $type ) {

        if ( ! in_array( $what, array( 'install', 'activate' ), true ) ) {
            return false;
        }

        if ( ! in_array( $type, array( 'plugin', 'addon' ), true ) ) {
            return false;
        }

        $capability = $what . '_plugins';

        if ( ! current_user_can( $capability ) ) {
            return false;
        }

        // Determine whether file modifications are allowed and it is activation permissions checking.
        if ( $what === 'install' && ! wp_is_file_mod_allowed( 'ays_sccp_can_install' ) ) {
            return false;
        }

        // All plugin checks are done.
        if ( $type === 'plugin' ) {
            return true;
        }
        return false;
    }

    /**
     * Activate plugin.
     *
     * @since 1.0.0
     * @since 1.3.9 Updated the permissions checking.
     */
    public function ays_sccp_activate_plugin() {

        // Run a security check.
        check_ajax_referer( $this->plugin_name . '-install-plugin-nonce', sanitize_key( $_REQUEST['_ajax_nonce'] ) );

        // Check for permissions.
        if ( ! current_user_can( 'activate_plugins' ) ) {
            wp_send_json_error( esc_html__( 'Plugin activation is disabled for you on this site.', 'secure-copy-content-protection' ) );
        }

        $type = 'addon';

        if ( isset( $_POST['plugin'] ) ) {

            if ( ! empty( $_POST['type'] ) ) {
                $type = sanitize_key( $_POST['type'] );
            }

            $plugin   = sanitize_text_field( wp_unslash( $_POST['plugin'] ) );
            $activate = activate_plugins( $plugin );

            if ( ! is_wp_error( $activate ) ) {
                if ( $type === 'plugin' ) {
                    wp_send_json_success( esc_html__( 'Plugin activated.', 'secure-copy-content-protection' ) );
                } else {
                        ( esc_html__( 'Addon activated.', 'secure-copy-content-protection' ) );
                }
            }
        }

        if ( $type === 'plugin' ) {
            wp_send_json_error( esc_html__( 'Could not activate the plugin. Please activate it on the Plugins page.', 'secure-copy-content-protection' ) );
        }

        wp_send_json_error( esc_html__( 'Could not activate the addon. Please activate it on the Plugins page.', 'secure-copy-content-protection' ) );
    }

    /**
     * Install addon.
     *
     * @since 1.0.0
     * @since 1.3.9 Updated the permissions checking.
     */
    public function ays_sccp_install_plugin() {

        // Run a security check.
        check_ajax_referer( $this->plugin_name . '-install-plugin-nonce', sanitize_key( $_REQUEST['_ajax_nonce'] ) );

        $generic_error = esc_html__( 'There was an error while performing your request.', 'secure-copy-content-protection' );
        $type          = ! empty( $_POST['type'] ) ? sanitize_key( $_POST['type'] ) : '';

        // Check if new installations are allowed.
        if ( ! self::ays_sccp_can_install( $type ) ) {
            wp_send_json_error( $generic_error );
        }

        $error = $type === 'plugin'
            ? esc_html__( 'Could not install the plugin. Please download and install it manually.', 'secure-copy-content-protection' )
            : "";

        $plugin_url = ! empty( $_POST['plugin'] ) ? esc_url_raw( wp_unslash( $_POST['plugin'] ) ) : '';

        if ( empty( $plugin_url ) ) {
            wp_send_json_error( $error );
        }

        // Prepare variables.
        $url = esc_url_raw(
            add_query_arg(
                [
                    'page' => 'secure-copy-content-protection-featured-plugins',
                ],
                admin_url( 'admin.php' )
            )
        );

        ob_start();
        $creds = request_filesystem_credentials( $url, '', false, false, null );

        // Hide the filesystem credentials form.
        ob_end_clean();

        // Check for file system permissions.
        if ( $creds === false ) {
            wp_send_json_error( $error );
        }
        
        if ( ! WP_Filesystem( $creds ) ) {
            wp_send_json_error( $error );
        }

        /*
         * We do not need any extra credentials if we have gotten this far, so let's install the plugin.
         */
        require_once SCCP_DIR . 'includes/admin/class-secure-copy-content-protection-upgrader.php';
        require_once SCCP_DIR . 'includes/admin/class-secure-copy-content-protection-install-skin.php';
        require_once SCCP_DIR . 'includes/admin/class-secure-copy-content-protection-skin.php';


        // Do not allow WordPress to search/download translations, as this will break JS output.
        remove_action( 'upgrader_process_complete', array( 'Language_Pack_Upgrader', 'async_upgrade' ), 20 );

        // Create the plugin upgrader with our custom skin.
        $installer = new SecureCopyContentProtection\Helpers\SecureCopyContentProtectionPluginSilentUpgrader( new Secure_Copy_Content_Protection_Install_Skin() );

        // Error check.
        if ( ! method_exists( $installer, 'install' ) ) {
            wp_send_json_error( $error );
        }

        $installer->install( $plugin_url );

        // Flush the cache and return the newly installed plugin basename.
        wp_cache_flush();

        $plugin_basename = $installer->plugin_info();

        if ( empty( $plugin_basename ) ) {
            wp_send_json_error( $error );
        }

        $result = array(
            'msg'          => $generic_error,
            'is_activated' => false,
            'basename'     => $plugin_basename,
        );

        // Check for permissions.
        if ( ! current_user_can( 'activate_plugins' ) ) {
            $result['msg'] = $type === 'plugin' ? esc_html__( 'Plugin installed.', 'secure-copy-content-protection' ) : "";

            wp_send_json_success( $result );
        }

        // Activate the plugin silently.
        $activated = activate_plugin( $plugin_basename );
        remove_action( 'activated_plugin', array( 'gallery_p_gallery_activation_redirect_method', 'poll_maker_activation_redirect_method' ), 100 );

        if ( ! is_wp_error( $activated ) ) {

            $result['is_activated'] = true;
            $result['msg']          = $type === 'plugin' ? esc_html__( 'Plugin installed and activated.', 'secure-copy-content-protection' ) : esc_html__( 'Addon installed and activated.', 'secure-copy-content-protection' );

            wp_send_json_success( $result );
        }

        // Fallback error just in case.
        wp_send_json_error( $result );
    }

    /**
     * List of AM plugins that we propose to install.
     *
     * @since 1.3.9
     *
     * @return array
     */
    protected function sccp_get_am_plugins() {
        if ( !isset( $_SESSION ) ) {
            session_start();
        }

        $images_url = SCCP_ADMIN_URL . '/images/icons/';

        $plugin_slug = array(
        	'fox-lms',
            'quiz-maker',
            'survey-maker',
            'poll-maker',
            'ays-popup-box',
            'gallery-photo-gallery',
            'personal-dictionary',
            'chart-builder',
            'easy-form',
        );

        $plugin_url_arr = array();
        foreach ($plugin_slug as $key => $slug) {
            if ( isset( $_SESSION['ays_sccp_our_product_links'] ) && !empty( $_SESSION['ays_sccp_our_product_links'] ) 
                && isset( $_SESSION['ays_sccp_our_product_links'][$slug] ) && !empty( $_SESSION['ays_sccp_our_product_links'][$slug] ) ) {
                $plugin_url = (isset( $_SESSION['ays_sccp_our_product_links'][$slug] ) && $_SESSION['ays_sccp_our_product_links'][$slug] != "") ? esc_url( $_SESSION['ays_sccp_our_product_links'][$slug] ) : "";
            } else {
                $latest_version = $this->ays_sccp_get_latest_plugin_version($slug);
                $plugin_url = 'https://downloads.wordpress.org/plugin/'. $slug .'.zip';
                if ( $latest_version != '' ) {
                    $plugin_url = 'https://downloads.wordpress.org/plugin/'. $slug .'.'. $latest_version .'.zip';
                    $_SESSION['ays_sccp_our_product_links'][$slug] = $plugin_url;
                }
            }

            $plugin_url_arr[$slug] = $plugin_url;
        }

        $plugins_array = array(
        	'fox-lms/fox-lms.php'        => array(
                'icon'        => $images_url . 'icon-fox-lms-128x128.png',
                'name'        => __( 'Fox LMS', 'secure-copy-content-protection' ),
                'desc'        => __( 'Build and manage online courses directly on your WordPress site.', 'secure-copy-content-protection' ),
                'desc_hidden' => __( 'With the FoxLMS plugin, you can create, sell, and organize courses, lessons, and quizzes, transforming your website into a dynamic e-learning platform.', 'secure-copy-content-protection' ),
                'wporg'       => 'https://wordpress.org/plugins/fox-lms/',
                'buy_now'     => 'https://foxlms.com/pricing/?utm_source=dashboard&utm_medium=sccp-free&utm_campaign=fox-lms-our-products-page',
                'url'         => $plugin_url_arr['fox-lms'],
            ),
           	'quiz-maker/quiz-maker.php'        => array(
                'icon'        => $images_url . 'icon-quiz-128x128.png',
                'name'        => __( 'Quiz Maker', 'secure-copy-content-protection' ),
                'desc'        => __( 'With our Quiz Maker plugin it’s easy to make a quiz in a short time.', 'secure-copy-content-protection' ),
                'desc_hidden' => __( 'You to add images to your quiz, order unlimited questions. Also you can style your quiz to satisfy your visitors.', 'secure-copy-content-protection' ),
                'wporg'       => 'https://wordpress.org/plugins/quiz-maker/',
                'buy_now'     => 'https://ays-pro.com/wordpress/quiz-maker/',
                'url'         => $plugin_url_arr['quiz-maker'],
            ),
            'survey-maker/survey-maker.php'        => array(
                'icon'        => $images_url . 'icon-survey-128x128.png',
                'name'        => __( 'Survey Maker', 'secure-copy-content-protection' ),
                'desc'        => __( 'Make amazing online surveys and get real-time feedback quickly and easily.', 'secure-copy-content-protection' ),
                'desc_hidden' => __( 'Learn what your website visitors want, need, and expect with the help of Survey Maker. Build surveys without limiting your needs.', 'secure-copy-content-protection' ),
                'wporg'       => 'https://wordpress.org/plugins/survey-maker/',
                'buy_now'     => 'https://ays-pro.com/wordpress/survey-maker',
                'url'         => $plugin_url_arr['survey-maker'],
            ),
            'poll-maker/poll-maker-ays.php'        => array(
                'icon'        => $images_url . 'icon-poll-128x128.png',
                'name'        => __( 'Poll Maker', 'secure-copy-content-protection' ),
                'desc'        => __( 'Create amazing online polls for your WordPress website super easily.', 'secure-copy-content-protection' ),
                'desc_hidden' => __( 'Build up various types of polls in a minute and get instant feedback on any topic or product.', 'secure-copy-content-protection' ),
                'wporg'       => 'https://wordpress.org/plugins/poll-maker/',
                'buy_now'     => 'https://ays-pro.com/wordpress/poll-maker/',
                'url'         => $plugin_url_arr['poll-maker'],
            ),
            'ays-popup-box/ays-pb.php'        => array(
                'icon'        => $images_url . 'icon-popup-128x128.png',
                'name'        => __( 'Popup Box', 'secure-copy-content-protection' ),
                'desc'        => __( 'Popup everything you want! Create informative and promotional popups all in one plugin.', 'secure-copy-content-protection' ),
                'desc_hidden' => __( 'Attract your visitors and convert them into email subscribers and paying customers.', 'secure-copy-content-protection' ),
                'wporg'       => 'https://wordpress.org/plugins/ays-popup-box/',
                'buy_now'     => 'https://ays-pro.com/wordpress/popup-box/',
                'url'         => $plugin_url_arr['ays-popup-box'],
            ),
            'gallery-photo-gallery/gallery-photo-gallery.php'        => array(
                'icon'        => $images_url . 'icon-gallery-128x128.png',
                'name'        => __( 'Gallery Photo Gallery', 'secure-copy-content-protection' ),
                'desc'        => __( 'Create unlimited galleries and include unlimited images in those galleries.', 'secure-copy-content-protection' ),
                'desc_hidden' => __( 'Represent images in an attractive way. Attract people with your own single and multiple free galleries from your photo library.', 'secure-copy-content-protection' ),
                'wporg'       => 'https://wordpress.org/plugins/gallery-photo-gallery/',
                'buy_now'     => 'https://ays-pro.com/wordpress/photo-gallery/',
                'url'         => $plugin_url_arr['gallery-photo-gallery'],
            ),
            'personal-dictionary/personal-dictionary.php'        => array(
                'icon'        => $images_url . 'pd-logo-128x128.png',
                'name'        => __( 'Personal Dictionary', 'secure-copy-content-protection' ),
                'desc'        => __( 'Allow your students to create personal dictionary, study and memorize the words.', 'secure-copy-content-protection' ),
                'desc_hidden' => __( 'Allow your users to create their own digital dictionaries and learn new words and terms as fastest as possible.', 'secure-copy-content-protection' ),
                'wporg'       => 'https://wordpress.org/plugins/personal-dictionary/',
                'buy_now'     => 'https://ays-pro.com/wordpress/personal-dictionary/',
                'url'         => $plugin_url_arr['personal-dictionary'],
            ),
            'chart-builder/chart-builder.php'        => array(
                'icon'        => $images_url . 'chartify-150x150.png',
                'name'        => __( 'Chart Builder', 'secure-copy-content-protection' ),
                'desc'        => __( 'Chart Builder plugin allows you to create beautiful charts', 'secure-copy-content-protection' ),
                'desc_hidden' => __( ' and graphs easily and quickly.', 'secure-copy-content-protection' ),
                'wporg'       => 'https://wordpress.org/plugins/chart-builder/',
                'buy_now'     => 'https://ays-pro.com/wordpress/chart-builder/',
                'url'         => $plugin_url_arr['chart-builder'],
            ),
            'easy-form/easy-form.php'        => array(
                'icon'        => $images_url . 'easyform-150x150.png',
                'name'        => __( 'Easy Form', 'secure-copy-content-protection' ),
                'desc'        => __( 'Choose the best WordPress form builder plugin. ', 'secure-copy-content-protection' ),
                'desc_hidden' => __( 'Create contact forms, payment forms, surveys, and many more custom forms. Build forms easily with us.', 'secure-copy-content-protection' ),
                'wporg'       => 'https://wordpress.org/plugins/easy-form/',
                'buy_now'     => 'https://ays-pro.com/wordpress/easy-form',
                'url'         => $plugin_url_arr['easy-form'],
            ),
        );

        return $plugins_array;
    }

    protected function ays_sccp_get_latest_plugin_version( $slug ){

        if ( is_null( $slug ) || empty($slug) ) {
            return "";
        }

        $version_latest = "";

        if ( ! function_exists( 'plugins_api' ) ) {
              require_once( ABSPATH . 'wp-admin/includes/plugin-install.php' );
        }

        // set the arguments to get latest info from repository via API ##
        $args = array(
            'slug' => $slug,
            'fields' => array(
                'version' => true,
            )
        );

        /** Prepare our query */
        $call_api = plugins_api( 'plugin_information', $args );

        /** Check for Errors & Display the results */
        if ( is_wp_error( $call_api ) ) {
            $api_error = $call_api->get_error_message();
        } else {

            //echo $call_api; // everything ##
            if ( ! empty( $call_api->version ) ) {
                $version_latest = $call_api->version;
            }
        }

        return $version_latest;
    }

    /**
     * Get AM plugin data to display in the Addons section of About tab.
     *
     * @since 6.4.0.4
     *
     * @param string $plugin      Plugin slug.
     * @param array  $details     Plugin details.
     * @param array  $all_plugins List of all plugins.
     *
     * @return array
     */
    protected function sccp_get_plugin_data( $plugin, $details, $all_plugins ) {

        $have_pro = ( ! empty( $details['pro'] ) && ! empty( $details['pro']['plug'] ) );
        $show_pro = false;

        $plugin_data = array();

        if ( $have_pro ) {
            if ( array_key_exists( $plugin, $all_plugins ) ) {
                if ( is_plugin_active( $plugin ) ) {
                    $show_pro = true;
                }
            }
            if ( array_key_exists( $details['pro']['plug'], $all_plugins ) ) {
                $show_pro = true;
            }
            if ( $show_pro ) {
                $plugin  = $details['pro']['plug'];
                $details = $details['pro'];
            }
        }

        if ( array_key_exists( $plugin, $all_plugins ) ) {
            if ( is_plugin_active( $plugin ) ) {
                // Status text/status.
                $plugin_data['status_class'] = 'status-active';
                $plugin_data['status_text']  = esc_html__( 'Active', 'secure-copy-content-protection' );
                // Button text/status.
                $plugin_data['action_class'] = $plugin_data['status_class'] . ' ays-sccp-card__btn-info disabled';
                $plugin_data['action_text']  = esc_html__( 'Activated', 'secure-copy-content-protection' );
                $plugin_data['plugin_src']   = esc_attr( $plugin );
            } else {
                // Status text/status.
                $plugin_data['status_class'] = 'status-installed';
                $plugin_data['status_text']  = esc_html__( 'Inactive', 'secure-copy-content-protection' );
                // Button text/status.
                $plugin_data['action_class'] = $plugin_data['status_class'] . ' ays-sccp-card__btn-info';
                $plugin_data['action_text']  = esc_html__( 'Activate', 'secure-copy-content-protection' );
                $plugin_data['plugin_src']   = esc_attr( $plugin );
            }
        } else {
            // Doesn't exist, install.
            // Status text/status.
            $plugin_data['status_class'] = 'status-missing';

            if ( isset( $details['act'] ) && 'go-to-url' === $details['act'] ) {
                $plugin_data['status_class'] = 'status-go-to-url';
            }
            $plugin_data['status_text'] = esc_html__( 'Not Installed', 'secure-copy-content-protection' );
            // Button text/status.
            $plugin_data['action_class'] = $plugin_data['status_class'] . ' ays-sccp-card__btn-info';
            $plugin_data['action_text']  = esc_html__( 'Install Plugin', 'secure-copy-content-protection' );
            $plugin_data['plugin_src']   = esc_url( $details['url'] );
        }

        $plugin_data['details'] = $details;

        return $plugin_data;
    }

    /**
     * Display the Addons section of About tab.
     *
     * @since 1.3.9
     */
    public function sccp_output_about_addons() {

        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $all_plugins          = get_plugins();
        $am_plugins           = $this->sccp_get_am_plugins();
        $can_install_plugins  = self::ays_sccp_can_install( 'plugin' );
        $can_activate_plugins = self::ays_sccp_can_activate( 'plugin' );

        $content = '';
        $content.= '<div class="ays-sccp-cards-block">';
        foreach ( $am_plugins as $plugin => $details ){

            $plugin_data = $this->sccp_get_plugin_data( $plugin, $details, $all_plugins );
            $plugin_ready_to_activate = $can_activate_plugins
                && isset( $plugin_data['status_class'] )
                && $plugin_data['status_class'] === 'status-installed';
            $plugin_not_activated     = ! isset( $plugin_data['status_class'] )
                || $plugin_data['status_class'] !== 'status-active';

            $plugin_action_class = ( isset( $plugin_data['action_class'] ) && esc_attr( $plugin_data['action_class'] ) != "" ) ? esc_attr( $plugin_data['action_class'] ) : "";

            $plugin_action_class_disbaled = "";
            if ( strpos($plugin_action_class, 'status-active') !== false ) {
                $plugin_action_class_disbaled = "disbaled='true'";
            }

            $content .= '
                <div class="ays-sccp-card">
                    <div class="ays-sccp-card__content flexible">
                        <div class="ays-sccp-card__content-img-box">
                            <img class="ays-sccp-card__img" src="'. esc_url( $plugin_data['details']['icon'] ) .'" alt="'. esc_attr( $plugin_data['details']['name'] ) .'">
                        </div>
                        <div class="ays-sccp-card__text-block">
                            <h5 class="ays-sccp-card__title">'. esc_html( $plugin_data['details']['name'] ) .'</h5>
                            <p class="ays-sccp-card__text">'. wp_kses_post( $plugin_data['details']['desc'] ) .'
                                <span class="ays-sccp-card__text-hidden">
                                    '. wp_kses_post( $plugin_data['details']['desc_hidden'] ) .'
                                </span>
                            </p>
                        </div>
                    </div>
                    <div class="ays-sccp-card__footer">';
                        if ( $can_install_plugins || $plugin_ready_to_activate || ! $details['wporg'] ) {
                            $content .= '<button class="'. esc_attr( $plugin_data['action_class'] ) .'" data-plugin="'. esc_attr( $plugin_data['plugin_src'] ) .'" data-type="plugin" '. $plugin_action_class_disbaled .'>
                                '. wp_kses_post( $plugin_data['action_text'] ) .'
                            </button>';
                        }
                        elseif ( $plugin_not_activated ) {
                            $content .= '<a href="'. esc_url( $details['wporg'] ) .'" target="_blank" rel="noopener noreferrer">
                                '. esc_html_e( 'WordPress.org', 'secure-copy-content-protection' ) .'
                                <span aria-hidden="true" class="dashicons dashicons-external"></span>
                            </a>';
                        }
            $content .='
                        <a target="_blank" href="'. esc_url( $plugin_data['details']['buy_now'] ) .'" class="ays-sccp-card__btn-primary">'. __('Buy Now', 'secure-copy-content-protection') .'</a>
                    </div>
                </div>';
        }
        $install_plugin_nonce = wp_create_nonce( $this->plugin_name . '-install-plugin-nonce' );
        $content .= '<input type="hidden" id="ays_sccp_ajax_install_plugin_nonce" name="ays_sccp_ajax_install_plugin_nonce" value="'. $install_plugin_nonce .'">';
        $content .= '</div>';

        echo $content;
    }

    public function ays_sccp_black_friady_popup_box(){
        if(!empty($_REQUEST['page']) && sanitize_text_field( $_REQUEST['page'] ) != $this->plugin_name . "-admin-dashboard"){
            if(false !== strpos( sanitize_text_field( $_REQUEST['page'] ), $this->plugin_name)){

                $flag = true;

                if( isset($_COOKIE['aysSccpBlackFridayPopupCount']) && intval($_COOKIE['aysSccpBlackFridayPopupCount']) >= 2 ){
                    $flag = false;
                }

                $ays_sccp_cta_button_link = esc_url('https://ays-pro.com/photography-bundle?utm_source=dashboard&utm_medium=sccp-free&utm_campaign=mega-bundle-popup-black-friday-sale-' . SCCP_NAME_VERSION);

                if( $flag ){
                ?>
                <div class="ays-sccp-black-friday-popup-overlay" style="opacity: 0; visibility: hidden; display: none;">
                  <div class="ays-sccp-black-friday-popup-dialog">
                    <div class="ays-sccp-black-friday-popup-content">
                      <div class="ays-sccp-black-friday-popup-background-pattern">
                        <div class="ays-sccp-black-friday-popup-pattern-row">
                          <div class="ays-sccp-black-friday-popup-pattern-text">SALE SALE SALE</div>
                          <div class="ays-sccp-black-friday-popup-pattern-text">SALE SALE SALE</div>
                        </div>
                        <div class="ays-sccp-black-friday-popup-pattern-row">
                          <div class="ays-sccp-black-friday-popup-pattern-text">SALE SALE SALE</div>
                          <div class="ays-sccp-black-friday-popup-pattern-text">SALE SALE SALE</div>
                        </div>
                        <div class="ays-sccp-black-friday-popup-pattern-row">
                          <div class="ays-sccp-black-friday-popup-pattern-text">SALE SALE SALE</div>
                          <div class="ays-sccp-black-friday-popup-pattern-text">SALE SALE SALE</div>
                        </div>
                        <div class="ays-sccp-black-friday-popup-pattern-row">
                          <div class="ays-sccp-black-friday-popup-pattern-text">SALE SALE SALE</div>
                          <div class="ays-sccp-black-friday-popup-pattern-text">SALE SALE SALE</div>
                        </div>
                      </div>
                      
                      <button class="ays-sccp-black-friday-popup-close" aria-label="Close">
                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                          <path d="M18 6 6 18"></path>
                          <path d="m6 6 12 12"></path>
                        </svg>
                      </button>
                      
                      <div class="ays-sccp-black-friday-popup-badge">
                        <div class="ays-sccp-black-friday-popup-badge-content">
                          <div class="ays-sccp-black-friday-popup-badge-text-sm"><?php echo esc_html__( 'Up to', 'secure-copy-content-protection' ); ?></div>
                          <div class="ays-sccp-black-friday-popup-badge-text-lg">50%</div>
                          <div class="ays-sccp-black-friday-popup-badge-text-md"><?php echo esc_html__( 'OFF', 'secure-copy-content-protection' ); ?></div>
                        </div>
                      </div>
                      
                      <div class="ays-sccp-black-friday-popup-main-content">
                        <div class="ays-sccp-black-friday-popup-hashtag"><?php echo esc_html__( '#BLACKFRIDAY', 'secure-copy-content-protection' ); ?></div>
                        <h1 class="ays-sccp-black-friday-popup-title-mega"><?php echo esc_html__( 'PHOTOGRAPHY', 'secure-copy-content-protection' ); ?></h1>
                        <h1 class="ays-sccp-black-friday-popup-title-bundle"><?php echo esc_html__( 'BUNDLE', 'secure-copy-content-protection' ); ?></h1>
                        <div class="ays-sccp-black-friday-popup-offer-label">
                          <h2 class="ays-sccp-black-friday-popup-offer-text"><?php echo esc_html__( 'BLACK FRIDAY OFFER', 'secure-copy-content-protection' ); ?></h2>
                        </div>
                        <p class="ays-sccp-black-friday-popup-description"><?php echo esc_html__( 'Get our exclusive plugins in one bundle', 'secure-copy-content-protection' ); ?></p>
                        <a href="<?php echo esc_url($ays_sccp_cta_button_link); ?>" target="_blank" class="ays-sccp-black-friday-popup-cta-btn"><?php echo esc_html__( 'Get Photography Bundle', 'secure-copy-content-protection' ); ?></a>
                      </div>
                    </div>
                  </div>
                </div>
                <script type="text/javascript">
                    (function() {
                      var overlay = document.querySelector('.ays-sccp-black-friday-popup-overlay');
                      var closeBtn = document.querySelector('.ays-sccp-black-friday-popup-close');
                      var learnMoreBtn = document.querySelector('.ays-sccp-black-friday-popup-learn-more');
                      var ctaBtn = document.querySelector('.ays-sccp-black-friday-popup-cta-btn');

                      // Cookie helper functions
                      function setCookie(name, value, days) {
                        var expires = "";
                        if (days) {
                          var date = new Date();
                          date.setTime(date.getTime() + (days * 24 * 60 * 60 * 1000));
                          expires = "; expires=" + date.toUTCString();
                        }
                        document.cookie = name + "=" + (value || "") + expires + "; path=/";
                      }

                      function getCookie(name) {
                        var nameEQ = name + "=";
                        var ca = document.cookie.split(';');
                        for (var i = 0; i < ca.length; i++) {
                          var c = ca[i];
                          while (c.charAt(0) == ' ') c = c.substring(1, c.length);
                          if (c.indexOf(nameEQ) == 0) return c.substring(nameEQ.length, c.length);
                        }
                        return null;
                      }

                      // Get current show count from cookie
                      var showCount = parseInt(getCookie('aysSccpBlackFridayPopupCount') || '0', 10);
                      var maxShows = 2;

                      // Show popup function
                      function showPopup() {
                        if (overlay && showCount < maxShows) {
                          overlay.classList.add('ays-sccp-black-friday-popup-active');
                          showCount++;
                          // Update cookie with new count (expires in 30 days)
                          setCookie('aysSccpBlackFridayPopupCount', showCount.toString(), 30);
                        }
                      }

                      // Close popup function
                      function closePopup(e) {
                        if (e) {
                          e.preventDefault();
                          e.stopPropagation();
                        }
                        if (overlay) {
                          overlay.classList.remove('ays-sccp-black-friday-popup-active');
                        }
                      }

                      // Determine timing based on show count
                      if (showCount === 0) {
                        // First time - show after 30 seconds
                        setTimeout(function() {
                          showPopup();
                        }, 30000);
                      } else if (showCount === 1) {
                        // Second time - show after 200 seconds
                        setTimeout(function() {
                          showPopup();
                        }, 200000);
                      }
                      // If showCount >= 2, don't show popup at all

                      // Close button
                      if (closeBtn) {
                        closeBtn.addEventListener('click', function(e) {
                          closePopup(e);
                        });
                      }

                      // Learn more button
                      if (learnMoreBtn) {
                        learnMoreBtn.addEventListener('click', function(e) {
                          closePopup(e);
                        });
                      }

                      // CTA button (optional - if you want it to close popup too)
                      if (ctaBtn) {
                        ctaBtn.addEventListener('click', function(e) {
                          // You can add redirect logic here if needed
                          // window.location.href = 'your-url';
                        });
                      }

                      // Close on overlay click
                      if (overlay) {
                        overlay.addEventListener('click', function(e) {
                          if (e.target === overlay) {
                            closePopup(e);
                          }
                        });
                      }

                      // Close on Escape key
                      document.addEventListener('keydown', function(e) {
                        if (e.key === 'Escape' && overlay && overlay.classList.contains('ays-sccp-black-friday-popup-active')) {
                          closePopup();
                        }
                      });
                    })();
                </script>
                <style>
                    .ays-sccp-black-friday-popup-overlay{position:fixed;top:0;left:0;right:0;bottom:0;z-index:9999;background-color:rgba(0,0,0,.8);display:flex;align-items:center;justify-content:center;opacity:0;visibility:hidden;transition:opacity .2s,visibility .2s}.ays-sccp-black-friday-popup-overlay.ays-sccp-black-friday-popup-active{display:flex!important;opacity:1!important;visibility:visible!important}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-dialog{position:relative;max-width:470px;width:100%;border-radius:8px;overflow:hidden;background:0 0;box-shadow:0 25px 50px -12px rgba(0,0,0,.25);transform:scale(.95);transition:transform .2s}.ays-sccp-black-friday-popup-overlay.ays-sccp-black-friday-popup-active .ays-sccp-black-friday-popup-dialog{transform:scale(1)}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-content{position:relative;width:470px;height:410px;background:linear-gradient(to right bottom,#c056f5,#f042f0,#7d7de8);overflow:hidden}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-background-pattern{position:absolute;top:0;left:0;right:0;bottom:0;opacity:.07;pointer-events:none;transform:rotate(-12deg) translateY(32px);overflow:hidden}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-pattern-row{display:flex;gap:16px;margin-bottom:16px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-pattern-text{color:#fff;font-weight:900;font-size:96px;white-space:nowrap;line-height:1}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-close{position:absolute;top:16px;right:16px;z-index:9999;background:0 0;border:none;color:rgba(255,255,255,.8);cursor:pointer;padding:4px;transition:color .2s;line-height:0}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-close:hover,.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-learn-more:hover{color:#fff}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-badge{position:absolute;top:32px;right:32px;width:96px;height:96px;background-color:#d4fc79;border-radius:50%;display:flex;align-items:center;justify-content:center;box-shadow:0 25px 50px -12px rgba(0,0,0,.25);animation:3s ease-in-out infinite ays-sccp-black-friday-popup-float}@keyframes ays-sccp-black-friday-popup-float{0%,100%{transform:translateY(0)}50%{transform:translateY(-10px)}}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-badge-content{text-align:center}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-badge-text-sm{color:#1a1a1a;font-weight:900;font-size:24px;line-height:1}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-badge-text-lg{color:#1a1a1a;font-weight:900;font-size:30px;line-height:1;margin-top:4px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-badge-text-md{color:#1a1a1a;font-weight:900;font-size:20px;line-height:1}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-main-content{position:relative;z-index:10;height:100%;display:flex;flex-direction:column;align-items:center;justify-content:center;padding:0 48px;text-align:center}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-hashtag{color:rgba(255,255,255,.9);font-weight:700;font-size:14px;margin-bottom:16px;letter-spacing:.1em}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-title-mega{color:#fff;font-weight:900;font-size:27px;line-height:1;margin:0 0 12px;text-shadow:0 4px 6px rgba(0,0,0,.1)}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-title-bundle{color:#fff;font-weight:900;font-size:27px;line-height:1;margin:0 0 24px;text-shadow:0 4px 6px rgba(0,0,0,.1)}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-offer-label{background-color:#000;padding:12px 32px;margin-bottom:24px;display:inline-block}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-offer-text{color:#fff;font-weight:700;font-size:20px;letter-spacing:.05em;margin:0}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-description{color:rgba(255,255,255,.95);font-size:18px;font-weight:500;margin:0 0 32px!important}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-cta-btn{display:inline-flex;align-items:center;justify-content:center;height:48px;background-color:#fff;color:#a855f7;font-size:18px;font-weight:700;border:none;border-radius:24px;padding:0 40px;cursor:pointer;box-shadow:0 20px 25px -5px rgba(0,0,0,.1);transition:.2s;text-decoration:none}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-cta-btn:hover{background-color:rgba(255,255,255,.9);box-shadow:0 25px 50px -12px rgba(0,0,0,.25);transform:scale(1.05)}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-learn-more{background:0 0;border:none;color:rgba(255,255,255,.9);font-size:14px;text-decoration:underline;text-underline-offset:4px;cursor:pointer;padding:8px;margin-top:16px;transition:color .2s}@media (max-width:768px){.ays-sccp-black-friday-popup-overlay{display:none!important}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-content{width:90vw;max-width:400px;height:380px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-main-content{padding:0 32px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-badge{width:80px;height:80px;top:24px;right:24px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-badge-text-sm{font-size:20px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-badge-text-lg{font-size:26px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-badge-text-md,.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-offer-text{font-size:18px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-title-bundle,.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-title-mega{font-size:48px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-description{font-size:16px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-pattern-text{font-size:72px}}@media (max-width:480px){.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-content{width:95vw;max-width:340px;height:360px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-main-content{padding:0 24px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-badge{width:70px;height:70px;top:20px;right:20px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-badge-text-sm,.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-offer-text{font-size:16px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-badge-text-lg{font-size:22px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-badge-text-md{font-size:14px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-hashtag{font-size:12px;margin-bottom:12px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-title-bundle,.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-title-mega{font-size:40px;margin-bottom:8px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-offer-label{padding:10px 24px;margin-bottom:20px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-description{font-size:15px;margin-bottom:24px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-cta-btn{font-size:16px;height:44px;padding:0 32px}.ays-sccp-black-friday-popup-overlay .ays-sccp-black-friday-popup-pattern-text{font-size:60px}}
                </style>
                <?php
                }
            }
        }

    }

}