/* global _SEARCHWP_LIVE_SEARCH */

( function() {

    'use strict';

    const app = {

        /**
         * Init.
         *
         * @since 1.8.0
         */
        init: () => {

            if ( document.readyState === 'loading' ) {
                document.addEventListener( 'DOMContentLoaded', app.ready );
                return;
            }

            app.ready();
        },

        /**
         * Document ready
         *
         * @since 1.8.0
         */
        ready: () => {

            app.events();
        },

        /**
         * Events.
         *
         * @since 1.8.0
         */
        events: () => {

            document.querySelectorAll( 'select.swp-choicesjs-single' ).forEach( app.initChoicesSingle );
            document.querySelectorAll( 'select.swp-choicesjs-multiple' ).forEach( app.initChoicesMultiple );
            document.querySelectorAll( 'select.swp-choicesjs-hybrid' ).forEach( app.initChoicesHybrid );
        },

        /**
         * Init single choice select.
         *
         * @since 1.8.0
         */
        initChoicesSingle: ( el ) => {

            if ( typeof Choices === 'undefined' ) {
                return;
            }

            const args = {
                searchEnabled: false,
                shouldSort: false,
                allowHTML: true,
            };

            // Attach the Choices object to an element for easy access.
            el.data = el.data || {};
            el.data.choicesjs = new Choices( el, args );
        },

        /**
         * Init searchable multiple choice select.
         *
         * @since 1.8.0
         */
        initChoicesMultiple: ( el ) => {

            if ( typeof Choices === 'undefined' ) {
                return;
            }

            const args = {
                removeItemButton: true,
                duplicateItemsAllowed: false,
                allowHTML: true,
            };

            const choices = new Choices( el, args );

            // This makes the input element take as little space as possible.
            choices.clearInput();

            // Attach the Choices object to an element for easy access.
            el.data = el.data || {};
            el.data.choicesjs = choices;
        },

        /**
         * Init hybrid searchable multiple choice select with an ability to add new items by pressing Enter.
         *
         * @since 1.8.0
         */
        initChoicesHybrid: ( el ) => {

            if ( typeof Choices === 'undefined' ) {
                return;
            }

            const args = {
                removeItemButton: true,
                duplicateItemsAllowed: false,
                noResultsText: 'Press Enter to add item',
                noChoicesText: 'Type to add new items',
                allowHTML: true,
                fuseOptions: {
                    threshold: 0,
                },
            };

            const choices = new Choices( el, args );

            // This makes the input element take as little space as possible.
            choices.clearInput();

            const keyupChoiceHybridCallback = ( e ) => {
                if ( e.keyCode === 13 ) {
                    app.addChoiceToHybrid( choices );
                }
            };

            choices.input.element.addEventListener( 'keyup', keyupChoiceHybridCallback, false );

            // Attach the Choices object to an element for easy access.
            el.data = el.data || {};
            el.data.choicesjs = choices;
        },

        /**
         * Add new item to a hybrid select.
         *
         * @since 1.8.0
         *
         * @param choices Choices.js object.
         */
        addChoiceToHybrid: ( choices ) => {

            if ( ! choices.input.value ) {
                return;
            }

            const canAddItem = choices._canAddItem( choices._store.activeItems, choices.input.value );

            if ( ! canAddItem.response ) {
                const notice = choices._getTemplate( 'notice', canAddItem.notice );
                choices.choiceList.clear();
                choices.choiceList.append( notice );
                return;
            }

            const choice = {
                "value": choices.input.value,
                "isSelected": true,
            }

            choices._addChoice( choice );

            choices.clearInput();
        },
    };

    app.init();

    window.searchwp = window.searchwp || {};

    window.searchwp.ChoicesJs = window.searchwp.ChoicesJs || app;

})();
