<?php
if (!defined('ABSPATH')) {
	die("What you are doing here man.");
}

require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-mvsp-encryption.php';

class MVSP_Export_DB {
    private $backup_dir;
    private $batch_size = 1000;
    private $max_execution_time;
    private $current_table;
    private $current_offset;
    private static $instance = null;

    // Make constructor private for singleton
    private function __construct() {
        $this->backup_dir = self::create_backup_directory();
        $this->max_execution_time = ini_get('max_execution_time');
        
        // Increase memory limit if possible
        ini_set('memory_limit', '512M');
        
        // Set timeout to 0 if allowed
        set_time_limit(0);
    }

    // Singleton instance
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public static function export_db() {
        global $wpdb;
        
        try {
            $instance = self::get_instance();
            
            // Make sure wpdb doesn't cache results
            $wpdb->hide_errors();
            $wpdb->flush();

            // Get and sanitize filename from auth token
            $exported_db_filename = get_option('mvsp_auth_token');
            if (!$exported_db_filename) {
                throw new Exception('Auth token not found');
            }
            
            $exported_db_filename = sanitize_file_name(substr($exported_db_filename, 0, 40));
            $filename = $exported_db_filename . '.sql';
            $filepath = $instance->backup_dir . '/' . $filename;
            
            // Create or clear the backup file
            if (!file_put_contents($filepath, self::get_header())) {
                throw new Exception('Unable to create backup file');
            }
            
            $tables = $wpdb->get_results('SHOW TABLES', ARRAY_N);
            if (!$tables) {
                throw new Exception('No tables found to export');
            }
            
            foreach ($tables as $table) {
                $instance->current_table = $table[0];
                $instance->export_table($filepath);
            }
            
            return $instance->backup_dir . '/' . $filename;
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Export failed: ' . $e->getMessage(),
                'table' => isset($instance->current_table) ? $instance->current_table : '',
                'offset' => isset($instance->current_offset) ? $instance->current_offset : 0
            ];
        }
    }

    private static function get_header() {
        return "-- WordPress Database Backup\n"
             . "-- Generated: " . date('Y-m-d H:i:s') . "\n"
             . "-- Server: " . esc_html($_SERVER['SERVER_NAME']) . "\n"
             . "-- PHP Version: " . phpversion() . "\n\n"
             . "SET FOREIGN_KEY_CHECKS=0;\n"
             . "SET SQL_MODE = 'NO_AUTO_VALUE_ON_ZERO';\n"
             . "SET UNIQUE_CHECKS=0;\n"
             . "SET AUTOCOMMIT=0;\n\n";
    }
    
    private function export_table($filepath) {
        global $wpdb;
        
        // Sanitize table name
        $table_name = esc_sql($this->current_table);
        
        // Get create table syntax
        $create_table = $wpdb->get_row("SHOW CREATE TABLE `{$table_name}`", ARRAY_N);
        if (!$create_table) {
            throw new Exception("Failed to get table structure for {$table_name}");
        }
        
        if (!file_put_contents($filepath, "\n\n" . $create_table[1] . ";\n\n", FILE_APPEND)) {
            throw new Exception("Failed to write table structure for {$table_name}");
        }
        
        // Get total rows
        $total_rows = $wpdb->get_var("SELECT COUNT(*) FROM `{$table_name}`");
        
        // Export in batches
        for ($this->current_offset = 0; $this->current_offset < $total_rows; $this->current_offset += $this->batch_size) {
            $rows = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT * FROM `{$table_name}` LIMIT %d OFFSET %d",
                    $this->batch_size,
                    $this->current_offset
                ),
                ARRAY_A
            );
            
            if ($rows) {
                $insert_queries = [];
                foreach ($rows as $row) {
                    $values = array_map(function($value) use ($wpdb) {
                        if ($value === null) {
                            return 'NULL';
                        }
                        return "'" . $wpdb->_real_escape($value) . "'";
                    }, $row);
                    
                    $insert_queries[] = "INSERT INTO `{$table_name}` VALUES (" . implode(", ", $values) . ");";
                }
                
                // Write batch to file
                if (!file_put_contents($filepath, implode("\n", $insert_queries) . "\n", FILE_APPEND)) {
                    throw new Exception("Failed to write data for {$table_name}");
                }
                
                // Free up memory
                unset($rows, $insert_queries);
                if (function_exists('gc_collect_cycles')) {
                    gc_collect_cycles();
                }
            }
            
            // Give the server a tiny break
            usleep(10000);
        }
    }
    
    public static function create_backup_directory() {
		$upload_dir = wp_upload_dir();
		$backup_dir = $upload_dir['basedir'] . '/db-backups';
		$token = get_option('mvsp_auth_token');
		
		if (!file_exists($backup_dir)) {
			if (!wp_mkdir_p($backup_dir)) {
				throw new Exception('Unable to create backup directory');
			}
		}
		
		return $backup_dir;
    }

    public static function get_db_config() {
        if (!defined('ABSPATH')) {
            throw new Exception('WordPress is not loaded');
        }

        $exported_db_filename = get_option('mvsp_auth_token');
        if (!$exported_db_filename) {
            throw new Exception('Auth token not found');
        }

        $exported_db_filename = sanitize_file_name($exported_db_filename);
        $config_file_path = $exported_db_filename . '.txt';

        $db_config = array(
            'db_user' => DB_USER,
            'db_pass' => DB_PASSWORD,
            'db_name' => DB_NAME,
            'db_host' => DB_HOST,
            'db_table_prefix' => $GLOBALS['table_prefix'],
            'secure_auth_key' => defined('SECURE_AUTH_KEY') ? SECURE_AUTH_KEY : '',
            'logged_in_key' => defined('LOGGED_IN_KEY') ? LOGGED_IN_KEY : '',
            'nonce_key' => defined('NONCE_KEY') ? NONCE_KEY : '',
            'auth_salt_key' => defined('AUTH_SALT') ? AUTH_SALT : '',
            'secure_auth_salt_key' => defined('SECURE_AUTH_SALT') ? SECURE_AUTH_SALT : '',
            'logged_in_salt_key' => defined('LOGGED_IN_SALT') ? LOGGED_IN_SALT : '',
            'nonce_salt_key' => defined('NONCE_SALT') ? NONCE_SALT : '',
            'wp_debug' => defined('WP_DEBUG') ? WP_DEBUG : false
        );

        $result = file_put_contents($config_file_path, json_encode($db_config));
        if ($result === false) {
            throw new Exception('Unable to write configuration file');
        }

        return $config_file_path;
    }
}