<?php
/**
 * The Relevanssi_Live_Search_Form class.
 *
 * @package Relevanssi Live Ajax Search
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Relevanssi_Live_Search_Form
 *
 * The Relevanssi Live Ajax Search search form and it's configuration.
 *
 * @since 1.0
 */
class Relevanssi_Live_Search_Form extends Relevanssi_Live_Search {
	/**
	 * The default configuration
	 *
	 * Developers can add their own configs using the
	 * relevanssi_live_search_configs filter which is applied at runtime.
	 * You are responsible for keeping the $configs array in tact, and either
	 * substituting your own customizations in the existing data, or adding
	 * your own by appending your own array key with values based on the
	 * default.
	 *
	 * To use: set the data-rlvconfig attribute value on your search form input
	 * to be the config you want to use.
	 *
	 * @since 1.0
	 *
	 * @var array All configurations available for use at runtime
	 */
	public $configs = array(
		'default' => array(                // 'default' config
			'input'   => array(
				'delay'     => 300,        // Impose delay (in milliseconds) before firing a search.
				'min_chars' => 3,          // Wait for at least 3 characters before triggering a search.
			),
			'results' => array(
				'position'      => 'bottom',   // Where to position the results (bottom|top).
				'width'         => 'auto',     // Whether the width should automatically match the input (auto|css).
				'offset'        => array(
					'x' => 0,              // X offset (in pixels).
					'y' => 5,              // Y offset (in pixels).
				),
				'static_offset' => true,
			),
		),
	);

	/**
	 * The form counter.
	 *
	 * @var int The search form counter.
	 */
	private $form_counter = 0;

	/**
	 * Returns the next form number and increments the counter.
	 *
	 * @return int The search form counter.
	 */
	public function get_form_number() {
		$this->form_counter++;
		return $this->form_counter;
	}

	/**
	 * Equivalent of __construct() - implement our hooks
	 *
	 * @since 1.0
	 *
	 * @uses add_action() to trigger asset enqueue and output base styles in
	 * the footer.
	 * @uses add_filter() to filter search forms generated by get_search_form().
	 * @uses apply_filters() to ensure developer can filter the configs array
	 * via relevanssi_live_search_configs filter.
	 */
	public function setup() {
		add_action( 'wp_enqueue_scripts', array( $this, 'assets' ), 12 );
		add_filter( 'get_search_form', array( $this, 'get_search_form' ), 999, 1 );
		add_filter( 'get_product_search_form', array( $this, 'get_search_form' ), 999, 1 );
		add_filter( 'render_block', array( $this, 'render_block' ), 999, 2 );
		add_action( 'wp_footer', array( $this, 'base_styles' ) );
		add_filter( 'relevanssi_live_search_configs', array( $this, 'filter_configs' ) );

		// The configs store all of the various configuration arrays that can
		// be used at runtime.
		$this->configs = apply_filters( 'relevanssi_live_search_configs', $this->configs );
	}

	/**
	 * Register, localize, and enqueue all necessary JavaScript and CSS.
	 *
	 * @since 1.0
	 *
	 * @uses wp_enqueue_style() to enqueue CSS.
	 * @uses wp_enqueue_script() to enqueue JavaScript.
	 * @uses wp_register_script() to register JavaScript.
	 * @uses wp_localize_script() to pass PHP variables to JavaScript at
	 * runtime.
	 * @uses json_encode() to prepare the (potentially filtered) configs array.
	 */
	public function assets() {
		wp_enqueue_style( 'relevanssi-live-search', $this->url . '/assets/styles/style.css', null, $this->version );

		wp_enqueue_script( 'jquery' );

		// If WP is in script debug, or we pass ?script_debug in a URL, set
		// $debug to true.
		$debug = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG === true )
			|| ( isset( $_GET['script_debug'] ) ) ? '' : '.min'; // phpcs:ignore WordPress.Security.NonceVerification

		wp_register_script(
			'relevanssi-live-search-client',
			$this->url . "/assets/javascript/dist/script{$debug}.js",
			array( 'jquery' ),
			$this->version,
			true
		);

		$ajaxurl = admin_url( 'admin-ajax.php' );

		/**
		 * Filters whether to enable the AJAX messages template.
		 *
		 * If enabled, the messages template will be loaded live via AJAX. If
		 * disabled, the messages template will be loaded via JS localize
		 * script mechanism.
		 *
		 * @param bool $enable_ajax_messages Whether to enable the AJAX messages
		 * template.
		 */
		if ( apply_filters( 'relevanssi_live_ajax_search_ajax_messages', false ) ) {
			$message_template = '';
		} else {
			require_once dirname( __FILE__ ) . '/class-relevanssi-live-search-client.php';

			$message_tpl_client = new Relevanssi_Live_Search_Client();
			$message_template   = $message_tpl_client->get_ajax_message_template_string();
		}

		// Set up our parameters.
		$params = array(
			'ajaxurl'             => esc_url( $ajaxurl ),
			'config'              => $this->configs,
			'msg_no_config_found' => __( 'No valid Relevanssi Live Search configuration found!', 'relevanssi-live-ajax-search' ),
			'msg_loading_results' => __( 'Loading search results.', 'relevanssi-live-ajax-search' ),
			'messages_template'   => $message_template,
		);

		// We need to JSON encode the configs.
		$encoded_data = array(
			'l10n_print_after' => 'relevanssi_live_search_params = ' . wp_json_encode( $params ) . ';',
		);

		// Localize and enqueue the script with all of the variable goodness.
		wp_localize_script( 'relevanssi-live-search-client', 'relevanssi_live_search_params', $encoded_data );
		wp_enqueue_script( 'relevanssi-live-search-client' );
	}

	/**
	 * Callback to the get_search_form filter, allows us to automagically enable
	 * live search on form fields generated using get_search_form().
	 *
	 * @since 1.0
	 *
	 * @param string $html The generated markup for the search form.
	 *
	 * @return string Markup for the search form.
	 */
	public function get_search_form( string $html ) : string {
		$html = $this->modify_search_form( $html );
		return $html;
	}

	/**
	 * Adds the search parameters to the core/search block.
	 *
	 * @param string $block_content The block HTML.
	 * @param array  $block         The block data.
	 *
	 * @return string The block HTML with the search parameters added.
	 */
	public function render_block( $block_content, $block ) {
		if ( ! $block_content || is_admin() || 'core/search' !== $block['blockName'] ) {
			return $block_content;
		}
		$block_content = $this->modify_search_form( $block_content );
		return $block_content;
	}

	/**
	 * Modifies the search form to add the necessary attributes.
	 *
	 * @param string $html The generated markup for the search form.
	 */
	private function modify_search_form( string $html ) : string {
		/**
		 * Prevents hijacking the search form.
		 *
		 * If this filter hook returns `false`, Relevanssi Live Ajax Search
		 * will not add the parameters to the search form.
		 *
		 * @param bool If true, take over the search form.
		 */
		if ( ! apply_filters( 'relevanssi_live_search_hijack_get_search_form', true ) ) {
			return $html;
		}

		/**
		 * Filters the default config name.
		 *
		 * @param string The form config name. Default 'default'.
		 */
		$config = apply_filters( 'relevanssi_live_search_get_search_form_config', 'default' );

		/**
		 * Controls whether a separate result div is added.
		 *
		 * By default Relevanssi Live Ajax Search adds a separate result div
		 * inside the form so that the search results are accessible with a
		 * tab press. If this filter returns false, the result div will not be
		 * added and the rlvparentel data attribute will not be set.
		 *
		 * @param bool If true, add the result div. Default true.
		 */
		$add_result_div = apply_filters( 'relevanssi_live_search_add_result_div', true );

		$instructions = '<span class="relevanssi-live-search-instructions">'
			. __( 'When autocomplete results are available use up and down arrows to review and enter to go to the desired page. Touch device users, explore by touch or with swipe gestures.', 'relevanssi-live-ajax-search' )
			. '</span>';

		$form_number = $this->get_form_number();
		$form_id     = 'rlvlive_' . $form_number;
		$parentel    = $add_result_div ? 'data-rlvparentel="#' . $form_id . '"' : '';
		$html        = str_replace( 'name="s"', 'name="s" data-rlvlive="true" ' . $parentel . ' data-rlvconfig="' . esc_attr( $config ) . '"', $html );
		$html        = str_replace( '</form>', $instructions . '</form>', $html );
		if ( $add_result_div ) {
			$html = str_replace( '</form>', '<div id="' . $form_id . '"></div></form>', $html );
		}

		return $html;
	}

	/**
	 * Output the base styles (absolutely minimal) necessary to properly set up
	 * the results wrapper.
	 *
	 * @since 1.0
	 *
	 * @uses apply_filters() to allow devs to disable this functionality.
	 */
	public function base_styles() {
		/**
		 * Allows developers to disable the base styles.
		 *
		 * @param boolean Whether or not to enable the base styles.
		 */
		if ( false === apply_filters( 'relevanssi_live_search_base_styles', true ) ) {
			return;
		}
		?>
		<style type="text/css">
			.relevanssi-live-search-results {
				opacity: 0;
				transition: opacity .25s ease-in-out;
				-moz-transition: opacity .25s ease-in-out;
				-webkit-transition: opacity .25s ease-in-out;
				height: 0;
				overflow: hidden;
				z-index: 9999995; /* Exceed SearchWP Modal Search Form overlay. */
				position: absolute;
				display: none;
			}

			.relevanssi-live-search-results-showing {
				display: block;
				opacity: 1;
				height: auto;
				overflow: auto;
			}

			.relevanssi-live-search-no-results, .relevanssi-live-search-didyoumean {
				padding: 0 1em;
			}
		</style>
		<?php
	}

	/**
	 * Checks that the min_chars setting is not lower than the Relevanssi
	 * minimum word length setting.
	 *
	 * @param array $configs The configs.
	 *
	 * @return array The configs.
	 */
	public function filter_configs( array $configs ) : array {
		$relevanssi_min_word_length = get_option( 'relevanssi_min_word_length', 0 );
		if ( ! $relevanssi_min_word_length ) {
			return $configs;
		}
		foreach ( $configs as $key => $config ) {
			if ( $config['input']['min_chars'] < $relevanssi_min_word_length ) {
				$configs[ $key ]['input']['min_chars'] = $relevanssi_min_word_length;
			}
		}
		return $configs;
	}

}
