<?php
/**
 * Settings handler.
 *
 * @package RSFV
 */

namespace RSFV\Settings;

use RSFV\Options;

/**
 * Register Settings.
 */
class Register {
	/**
	 * Class instance
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Class constructor.
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'register_menu' ) );

		// Handle saving settings earlier than load-{page} hook to avoid race conditions in conditional menus.
		add_action( 'wp_loaded', array( $this, 'save_settings' ) );

		add_action( 'init', array( $this, 'create_options' ) );

		add_action( 'load-settings_page_rsfv-settings', array( $this, 'cleanup_plugin_settings_page' ) );
	}

	/**
	 * Get a class instance.
	 *
	 * @return Register
	 */
	public static function get_instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Register plugin menu.
	 *
	 * @return void
	 */
	public function register_menu() {
		/**
		 * Default framework menu.
		 */
		global $admin_page_hooks;
		$primary_slug = 'jetixwp';
		$menu_icon    = 'data:image/svg+xml;base64,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';

		if ( ! isset( $admin_page_hooks['jetixwp'] ) ) {
			add_menu_page(
				_x( 'JetixWP Plugins', 'Page title', 'rsfv' ),
				_x( 'JetixWP', 'Menu title', 'rsfv' ),
				'manage_options',
				$primary_slug,
				'__return_null',
				$menu_icon,
				30
			);

			add_action( 'admin_enqueue_scripts', array( $this, 'hide_freemius_submenus' ) );
		}

		add_submenu_page(
			$primary_slug,
			__( 'Really Simple Featured Video Settings', 'rsfv' ),
			__( 'Featured Video', 'rsfv' ),
			'manage_options',
			'rsfv-settings',
			array( $this, 'settings_page' )
		);

		// Remove duplicate menu hack.
		// Note: It needs to go after the above add_submenu_page call.
		remove_submenu_page( $primary_slug, $primary_slug );
	}

	/**
	 * Hide Freemius submenus if they exist.
	 *
	 * @return void
	 */
	public function hide_freemius_submenus() {
		// Enqueue a core admin style as a handle for inline CSS.
		wp_enqueue_style( 'wp-admin' );

		$custom_css = '.toplevel_page_jetixwp .wp-submenu li a[href*="-addons"] { display: none !important; }';
		wp_add_inline_style( 'wp-admin', $custom_css );
	}

	/**
	 * Add settings page.
	 *
	 * @return void
	 */
	public function settings_page() {
		Admin_Settings::output();
	}

	/**
	 * Default options.
	 *
	 * Sets up the default options used on the settings page.
	 */
	public function create_options() {
		if ( ! is_admin() ) {
			return false;
		}

		// Include settings so that we can run through defaults.
		include RSFV_PLUGIN_DIR . 'includes/Settings/class-admin-settings.php';

		$settings = Admin_Settings::get_settings_pages();

		foreach ( $settings as $section ) {
			if ( 'object' !== gettype( $section ) || ! method_exists( $section, 'get_settings' ) ) {
				continue;
			}
			$subsections = array_unique( array_merge( array( '' ), array_keys( $section->get_sections() ) ) );

			foreach ( $subsections as $subsection ) {
				foreach ( $section->get_settings( $subsection ) as $value ) {
					if ( isset( $value['default'], $value['id'] ) ) {
						$autoload = isset( $value['autoload'] ) ? (bool) $value['autoload'] : true;
						add_option( $value['id'], $value['default'], '', ( $autoload ? 'yes' : 'no' ) );
					}
				}
			}
		}
	}

	/**
	 * Handle saving of settings.
	 *
	 * @return void
	 */
	public function save_settings() {
		global $current_tab, $current_section;

		// We should only save on the settings page.
		if ( ! is_admin() || ! isset( $_GET['page'] ) || 'rsfv-settings' !== $_GET['page'] ) {
			return;
		}

		// Include settings pages.
		Admin_Settings::get_settings_pages();

		// Get current tab/section.
		$current_tab     = empty( $_GET['tab'] ) ? 'general' : sanitize_title( wp_unslash( $_GET['tab'] ) );
		$current_section = empty( $_REQUEST['section'] ) ? '' : sanitize_title( wp_unslash( $_REQUEST['section'] ) );
		$nonce           = isset( $_POST['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ) : '';

		// Save settings if data has been posted.
		if ( wp_verify_nonce( $nonce, 'rsfv-settings' ) ) {
			if ( '' !== $current_section && apply_filters( "rsfv_save_settings_{$current_tab}_{$current_section}", ! empty( $_POST['save'] ) ) ) {
				Admin_Settings::save();
			} elseif ( '' === $current_section && apply_filters( "rsfv_save_settings_{$current_tab}", ! empty( $_POST['save'] ) || isset( $_POST['rsfv-license_activate'] ) ) ) {
				Admin_Settings::save();
			}
		}
	}

	/**
	 * Remove all notices from settings page for a clean and minimal look.
	 *
	 * @return void
	 */
	public function cleanup_plugin_settings_page() {
		remove_all_actions( 'admin_notices' );
	}
}

/**
 * Clean variables using sanitize_text_field. Arrays are cleaned recursively.
 * Non-scalar values are ignored.
 *
 * @param string|array $var Data to sanitize.
 * @return string|array
 */
function rsfv_clean( $var ) {
	if ( is_array( $var ) ) {
		return array_map( __NAMESPACE__ . '\rsfv_clean', $var );
	}

	return is_scalar( $var ) ? sanitize_text_field( $var ) : $var;
}

/**
 * Output admin fields.
 *
 * Loops though the RSFV options array and outputs each field.
 *
 * @param array $options Opens array to output.
 */
function rsfv_admin_fields( $options ) {

	if ( ! class_exists( 'Admin_Settings', false ) ) {
		include __DIR__ . '/class-admin-settings.php';
	}

	Admin_Settings::output_fields( $options );
}

/**
 * Update all settings which are passed.
 *
 * @param array $options Option fields to save.
 * @param array $data Passed data.
 */
function rsfv_update_options( $options, $data = null ) {

	if ( ! class_exists( 'Admin_Settings', false ) ) {
		include __DIR__ . '/class-admin-settings.php';
	}

	Admin_Settings::save_fields( $options, $data );
}

/**
 * Get a setting from the settings API.
 *
 * @param mixed $option_name Option name to save.
 * @param mixed $default Default value to save.
 * @return string
 */
function rsfv_settings_get_option( $option_name, $default = '' ) {

	if ( ! class_exists( 'Admin_Settings', false ) ) {
		include __DIR__ . '/class-admin-settings.php';
	}

	return Admin_Settings::get_option( $option_name, $default );
}

/**
 * Get enabled post types.
 *
 * @return array
 */
function get_post_types() {
	$post_types = Options::get_instance()->get( 'post_types' );
	$post_types = is_array( $post_types ) ? array_keys( $post_types ) : '';

	if ( ! is_array( $post_types ) && empty( $post_types ) ) {
		$post_types = array( 'post' );
	}

	return apply_filters( 'rsfv_get_enabled_post_types', $post_types );
}

/**
 * Get default video controls.
 *
 * @return array
 */
function get_default_video_controls() {
	return array(
		'controls' => true,
	);
}

/**
 * Get enabled video controls.
 *
 * @param string $type Type of video.
 *
 * @return array
 */
function get_video_controls( $type = 'self' ) {
	$defaults = get_default_video_controls();

	if ( 'self' === $type ) {
		$controls = Options::get_instance()->get( 'self_video_controls' );
	} else {
		$controls = Options::get_instance()->get( 'embed_video_controls' );
	}

	return is_array( $controls ) && ! empty( $controls ) ? $controls : $defaults;
}
