<?php
/**
 * Rollbacker.
 *
 * @package RSFV
 */

namespace RSFV\Featuresets\Rollback;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Rollback.
 *
 * Rollback handler class is responsible for rolling back to
 * previous version.
 *
 * @since 0.50.0
 */
class Rollbacker {

	/**
	 * Package URL.
	 *
	 * Holds the package URL.
	 *
	 * @access protected
	 *
	 * @var string Package URL.
	 */
	protected $package_url;

	/**
	 * Version.
	 *
	 * Holds the version.
	 *
	 * @access protected
	 *
	 * @var string Package URL.
	 */
	protected $version;

	/**
	 * Plugin name.
	 *
	 * Holds the plugin name.
	 *
	 * @access protected
	 *
	 * @var string Plugin name.
	 */
	protected $plugin_name;

	/**
	 * Plugin slug.
	 *
	 * Holds the plugin slug.
	 *
	 * @access protected
	 *
	 * @var string Plugin slug.
	 */
	protected $plugin_slug;

	/**
	 * Rollback constructor.
	 *
	 * Initializing rollback.
	 *
	 * @access public
	 *
	 * @param array $args Optional. Rollback arguments. Default is an empty array.
	 */
	public function __construct( $args = array() ) {
		foreach ( $args as $key => $value ) {
			$this->{$key} = $value;
		}
	}

	/**
	 * Print inline style.
	 *
	 * Add an inline CSS to the rollback page.
	 *
	 * @access private
	 */
	private function print_inline_style() {
		?>
		<style>
			.wrap {
				overflow: hidden;
				max-width: 850px;
				margin: auto;
				font-family: Courier, monospace;
			}

			h1 {
				background: #252422;
				text-align: center;
				color: #fff !important;
				padding: 70px !important;
				text-transform: uppercase;
				letter-spacing: 1px;
			}

			h1 img {
				max-width: 300px;
				display: block;
				margin: auto auto 50px;
			}

			.wrap h1 {
				position: relative;
				padding-top: 140px !important;
			}

			.wrap h1:before {
				content: '';
				position: absolute;
				width: 340px;
				height: 104px;
				color: #fff;
				top: 35px;
				background-image: url("data:image/svg+xml;base64,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");
				background-repeat: no-repeat;
				left: 25%;
			}
		</style>
		<?php
	}

	/**
	 * Upgrade.
	 *
	 * Run WordPress upgrade to rollback plugin to previous version.
	 *
	 * @access protected
	 */
	protected function upgrade() {
		if ( ! class_exists( 'WP_Upgrader_Skin' ) ) {
			require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		}

		require_once __DIR__ . '/class-rollback-downgrader-skin.php';

		$skin_args = array(
			'url'    => 'update.php?action=upgrade-plugin&plugin=' . rawurlencode( $this->plugin_name ),
			'plugin' => $this->plugin_name,
			'nonce'  => 'upgrade-plugin_' . $this->plugin_name,
			'title'  => esc_html__( 'Rollback to Previous Version', 'rsfv' ),
		);

		$this->print_inline_style();

		$skin     = new Rollback_Downgrader_Skin( $skin_args );
		$upgrader = new \Plugin_Upgrader( $skin );

		// IMPORTANT: initialize + install strings so feedback() can map keys.
		$upgrader->init();
		$upgrader->upgrade_strings();
		$upgrader->run(
			array(
				'package'                     => $this->package_url,
				'destination'                 => WP_PLUGIN_DIR,
				'abort_if_destination_exists' => false,
				'clear_working'               => true,
				'hook_extra'                  => array(
					'plugin' => $this->plugin_name,
					'type'   => 'plugin',
					'action' => 'update',
				),
			)
		);
	}

	/**
	 * Run.
	 *
	 * Rollback plugin to previous versions.
	 *
	 * @access public
	 */
	public function run() {
		$this->upgrade();
	}
}
