<?php

/**
 * CacheInterface.php
 *
 * @package Embera
 * @author Michael Pratt <yo@michael-pratt.com>
 * @link   http://www.michael-pratt.com/
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace DevOwl\RealCookieBanner\Vendor\Embera\Cache;

use InvalidArgumentException;
/**
 * The Cache Interface. It is based on PSR-16
 * @link https://www.php-fig.org/psr/psr-16/
 * @internal
 */
interface CacheInterface
{
    /**
     * Fetches a value from the cache.
     *
     * @param string $key     The unique key of this item in the cache.
     * @param mixed  $default Default value to return if the key does not exist.
     * @return mixed The value of the item from the cache, or $default in case of cache miss.
     *
     * @throws InvalidArgumentException
     */
    public function get($key, $default = null);
    /**
     * Set data in cache.
     *
     * @param string                 $key   The key of the item to store.
     * @param mixed                  $value The value of the item to store. Must be serializable.
     * @param null|int|\DateInterval $ttl   Optional. The TTL value of this item. If no value is sent and
     * @return bool
     *
     * @throws InvalidArgumentException
     */
    public function set($key, $value, $ttl = null);
    /**
     * Delete an item from the cache by its unique key.
     *
     * @param string $key The unique cache key of the item to delete.
     * @return bool
     *
     * @throws InvalidArgumentException
     */
    public function delete($key);
    /**
     * Wipes clean the entire cache's keys.
     *
     * @return bool
     */
    public function clear();
    /**
     * Obtains multiple cache items by their unique keys.
     *
     * @param iterable $keys    A list of keys that can obtained in a single operation.
     * @param mixed    $default Default value to return for keys that do not exist.
     *
     * @return iterable A list of key => value pairs.
     *
     * @throws InvalidArgumentException
     */
    public function getMultiple($keys, $default = null);
    /**
     * Persists a set of key => value pairs in the cache, with an optional TTL.
     *
     * @param iterable               $values A list of key => value pairs for a multiple-set operation.
     * @param null|int|\DateInterval $ttl    Optional. The TTL value of this item
     * @return bool
     *
     * @throws InvalidArgumentException
     */
    public function setMultiple($values, $ttl = null);
    /**
     * Deletes multiple cache items in a single operation.
     *
     * @param iterable $keys A list of string-based keys to be deleted.
     * @return bool
     *
     * @throws InvalidArgumentException
     */
    public function deleteMultiple($keys);
    /**
     * Determines whether an item is present in the cache.
     *
     * @param string $key The cache item key.
     * @return bool
     *
     * @throws InvalidArgumentException
     */
    public function has($key);
}
