import {Tooltip} from "react-tooltip";

const {FormToggle, TextControl} = wp.components;
import AppContext from "../../../contexts/AppContext";
import {showProModal} from "../../../includes/functions";

const {useContext} = React;

export default function Controls() {

    const isPro = !!radioPlayer.isPro;
    const context = useContext(AppContext);
    const {editData, setEditData} = context;
    const {
        autoplay,
        popup_icon,
        show_progressbar = true,
        playlist_icon,
        volume_control,
        player_status,
        skin,
        player_text,
        show_title = true,
        fallback_title,
        show_artist = isPro,
        fallback_artist,
        show_artwork = isPro,
    } = editData;

    return (
        <div className="settings-body">

            <div className="settings-header">
                <h3 className="header-title">{wp.i18n.__('Player Controls', 'radio-player')}</h3>
                <p>{wp.i18n.__('Customize the different controls and elements of the player.', 'radio-player')}</p>
            </div>

            {/* Autoplay */}
            <div className="form-group">
                <h4 className={`form-label`}>

                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none">
                        <path
                            d="M18.8 7.86671C18.5333 7.46671 18.2667 7.06671 17.8667 6.66671C13.8667 2.26671 6.66666 3.60004 4.53332 9.20004C4.39999 9.60004 4.13332 9.86671 3.73332 10C2.93332 10.1334 2.39999 9.46671 2.66666 8.66671C3.06666 7.33338 3.86666 6.26671 4.79999 5.20004C6.39999 3.46671 8.53332 2.40004 10.9333 2.13338C15.0667 1.73338 18.2667 3.33338 20.6667 6.80004C20.6667 6.80004 20.6667 6.93338 20.8 6.93338C20.8 6.80004 20.9333 6.80004 20.9333 6.66671C21.2 6.13338 21.7333 6.00004 22.2667 6.13338C22.8 6.26671 23.0667 6.80004 22.8 7.33338C22.5333 8.26671 22.1333 9.20004 21.8667 10C21.7333 10.5334 21.2 10.6667 20.6667 10.5334C19.7333 10.2667 18.9333 10 18 9.60004C17.4667 9.46671 17.2 8.80004 17.3333 8.26671C17.4667 7.73338 18 7.46671 18.6667 7.60004C18.5333 7.86671 18.6667 7.86671 18.8 7.86671Z"
                            fill="#5C637E"/>
                        <path
                            d="M3.33331 17.0667C3.33331 17.2 3.19998 17.2 3.19998 17.3333C2.93331 17.8667 2.39998 18.1333 1.86665 17.8667C1.33331 17.7333 1.06665 17.0667 1.33331 16.5333C1.59998 15.8667 1.86665 15.0667 2.13331 14.4C2.13331 14.2667 2.26665 14.1333 2.26665 14C2.53331 13.4667 2.93331 13.2 3.46665 13.4667C4.39998 13.7333 5.33331 14 6.13331 14.4C6.66665 14.5333 6.93331 15.0667 6.79998 15.6C6.66665 16.1333 6.13331 16.4 5.59998 16.2667C5.46665 16.2667 5.33331 16.1333 5.06665 16.1333C5.06665 16.2667 5.19998 16.2667 5.19998 16.4C7.06665 19.0667 9.59998 20.4 12.8 20C16 19.6 18.2666 17.8667 19.4666 14.9333C19.7333 14.2667 20.1333 14 20.6666 14.1333C21.3333 14.2667 21.7333 14.8 21.4666 15.4667C21.2 16.1333 20.9333 16.8 20.5333 17.4667C18.9333 20 16.5333 21.6 13.6 22C9.33331 22.5333 5.99998 20.9333 3.59998 17.3333C3.46665 17.2 3.46665 17.2 3.33331 17.0667Z"
                            fill="#5C637E"/>
                        <path
                            d="M9.06664 12.0002C9.06664 11.2002 9.06664 10.5335 9.06664 9.7335C9.06664 8.1335 10.6666 7.20016 12 8.00016C13.3333 8.66683 14.5333 9.46683 15.8666 10.1335C16.4 10.4002 16.6666 10.8002 16.9333 11.3335C17.2 12.2668 16.9333 13.2002 16.1333 13.7335C14.8 14.5335 13.4666 15.3335 12 16.1335C10.6666 16.9335 9.06664 15.8668 9.06664 14.4002C8.93331 13.4668 9.06664 12.8002 9.06664 12.0002Z"
                            fill="#5C637E"/>
                    </svg>

                    <span>{wp.i18n.__('Autoplay', 'radio-player')}</span>
                    {!isPro &&
                        <span className="radio-player-pro-icon">
                            <i className={`radio-player-icon icon-pro`}/>
                        </span>
                    }

                </h4>

                <div className="form-field">
                    <div className={`input-group ${!isPro ? 'disabled' : ''}`}>
                        <FormToggle
                            checked={isPro && autoplay}
                            onChange={() => {

                                if (!isPro) {
                                    showProModal(wp.i18n.__('Upgrade to PRO to enable autoplay.', 'radio-player'));
                                    return;
                                }

                                setEditData(data => ({...data, autoplay: !autoplay}))
                            }}
                        />

                    </div>

                    <p className="description">{wp.i18n.__('Enable to start playing the radio automatically when the page loads.', 'radio-player')}</p>

                    <div className="rp-notice rp-notice-warning">
                        <p className="rp-notice-content">
                            {wp.i18n.__(`Autoplay may not work as expected, as modern browsers block it for user experience and security.`, 'radio-player')}
                        </p>
                    </div>

                </div>
            </div>

            {!isPro &&
                <Tooltip
                    anchorSelect={`.input-group.disabled`}
                    className={`radio-player-tooltip`}
                    place="right"
                    variant="warning"
                    content={wp.i18n.__('PRO Feature', 'radio-player')}
                />
            }

            {/* Progressbar */}
            <div className="form-group">
                <h4 className={`form-label`}>

                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none">
                        <g clip-path="url(#clip0_2076_482)">
                            <path
                                d="M4 8.79999C4 4.37332 7.62667 0.799988 12.0533 0.799988C16.48 0.799988 20.0533 4.42665 20.0533 8.79999C20 13.2267 16.4267 16.8 12.0533 16.8C7.57333 16.7467 4 13.1733 4 8.79999ZM8.37333 8.79999C8.37333 10.1333 8.37333 11.52 8.37333 12.8533C8.37333 13.44 8.64 13.92 9.17333 14.1867C9.70667 14.4533 10.24 14.4 10.72 14.0267C12.64 12.6933 14.56 11.3067 16.48 9.97332C17.44 9.27999 17.44 8.15999 16.48 7.46665C15.6267 6.87999 14.8267 6.29332 13.9733 5.70665C12.9067 4.95999 11.84 4.21332 10.72 3.41332C10.4 3.19999 10.0267 3.03999 9.65333 3.09332C8.85333 3.19999 8.37333 3.78665 8.37333 4.58665C8.37333 6.07999 8.37333 7.41332 8.37333 8.79999Z"
                                fill="black"/>
                            <path
                                d="M8.80078 21.9733C8.80078 21.7067 8.80078 21.44 8.80078 21.1733C8.85411 21.1733 8.96078 21.1733 9.01411 21.1733C13.5474 21.1733 18.1341 21.1733 22.6674 21.1733C22.7741 21.1733 22.8808 21.1733 22.9341 21.2267C23.1474 21.28 23.2541 21.44 23.2008 21.6533C23.2008 21.8667 23.0408 21.9733 22.8808 22.0267C22.8274 22.0267 22.7741 22.0267 22.7208 22.0267C18.1341 22.0267 13.5474 22.0267 8.96078 22.0267C8.90745 21.9733 8.85411 21.9733 8.80078 21.9733Z"
                                fill="black"/>
                            <path
                                d="M6.40078 20C7.25411 20 8.00078 20.6933 8.00078 21.6C8.00078 22.4533 7.25411 23.2 6.40078 23.2C5.54745 23.2 4.80078 22.5067 4.80078 21.6C4.80078 20.7467 5.49411 20 6.40078 20Z"
                                fill="black"/>
                            <path
                                d="M4.00078 21.1733C4.00078 21.44 4.00078 21.7067 4.00078 21.9733C3.94745 21.9733 3.89411 21.9733 3.84078 21.9733C2.98745 21.9733 2.13411 21.9733 1.28078 21.9733C0.960781 21.9733 0.800781 21.8133 0.800781 21.5467C0.800781 21.28 1.01411 21.1733 1.28078 21.1733C2.18745 21.1733 3.09411 21.1733 4.00078 21.1733Z"
                                fill="black"/>
                            <path
                                d="M9.17383 8.8C9.17383 7.46667 9.17383 6.08 9.17383 4.74667C9.17383 4.53333 9.22716 4.32 9.38716 4.16C9.65383 3.94667 9.92049 3.94667 10.2405 4.16C11.3072 4.90667 12.3738 5.65333 13.4405 6.4C14.2938 6.98667 15.1472 7.57333 16.0005 8.21333C16.4805 8.58667 16.4805 9.06667 16.0005 9.38667C14.0805 10.72 12.1605 12.1067 10.2405 13.44C10.0272 13.6 9.76049 13.6533 9.49383 13.4933C9.22716 13.3333 9.17383 13.12 9.17383 12.8533C9.17383 11.52 9.17383 10.1333 9.17383 8.8Z"
                                fill="black"/>
                        </g>
                        <defs>
                            <clipPath id="clip0_2076_482">
                                <rect width="24" height="24" fill="white"/>
                            </clipPath>
                        </defs>
                    </svg>

                    <span>{wp.i18n.__('Progressbar', 'radio-player')}</span>
                </h4>

                <div className="form-field">
                    <div className={`input-group`}>
                        <FormToggle
                            checked={show_progressbar}
                            onChange={() => {
                                setEditData(data => ({...data, show_progressbar: !show_progressbar}))
                            }}
                        />

                    </div>

                    <p className="description">{wp.i18n.__('Show/hide the playback progress bar in the player.', 'radio-player')}</p>

                    <div className="rp-notice rp-notice-info">
                        <p className="rp-notice-content">
                            {wp.i18n.__('The progress bar is available only for local files and not for live streams.', 'radio-player')}
                        </p>
                    </div>

                </div>
            </div>

            {/* Popup Icon */}
            <div className="form-group">
                <h4 className={`form-label`}>

                    <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <g clipPath="url(#clip0_2235_6384)">
                            <path
                                d="M13.375 3.3125C12.6094 4.07813 11.8203 4.85937 11.0469 5.64062C10.5469 6.14062 10.3359 6.73438 10.5547 7.42188C10.6328 7.67188 10.7812 7.92188 10.9531 8.125C11.2266 8.45312 11.5391 8.75 11.8594 9.03906C12.5938 9.71094 13.5937 9.70312 14.3125 9C15.1094 8.21875 15.8906 7.41406 16.6641 6.64062C16.9844 6.86719 17.2812 7.07031 17.5547 7.29688C17.6172 7.35156 17.6484 7.47656 17.6484 7.57031C17.6562 10.4375 17.6562 13.2969 17.6484 16.1641C17.6484 16.5859 17.4219 16.8047 16.9844 16.8047C12.6094 16.8047 8.23438 16.8047 3.85938 16.8047C3.41406 16.8047 3.19531 16.5859 3.19531 16.125C3.19531 11.7656 3.19531 7.40625 3.19531 3.03906C3.19531 2.5625 3.41406 2.35156 3.89062 2.35156C6.71875 2.35156 9.53906 2.35156 12.3672 2.34375C12.5781 2.34375 12.6797 2.40625 12.7891 2.58594C12.9531 2.83594 13.1641 3.04688 13.375 3.3125Z"
                                fill="#5C637E"/>
                            <path
                                d="M2.03125 4.76562C2.03125 4.97656 2.03125 5.13281 2.03125 5.28906C2.03125 8.86719 2.03125 12.4453 2.03125 16.0234C2.03125 17.25 2.75 17.9688 3.96875 17.9688C7.54688 17.9688 11.125 17.9688 14.7031 17.9688C14.8594 17.9688 15.0156 17.9688 15.2188 17.9688C15.2188 18.5078 15.2578 19.0391 15.1953 19.5547C15.1797 19.7188 14.9219 19.9141 14.8125 19.9531C14.7344 19.9844 14.5156 19.9922 14.4062 19.9922C9.89062 19.9922 5.375 19.9922 0.859375 19.9922C0.164063 20 0 19.8359 0 19.1562C0 14.6406 0 10.125 0 5.60938C0 4.92969 0.164063 4.76562 0.859375 4.76562C1.23438 4.76562 1.60938 4.76562 2.03125 4.76562Z"
                                fill="#5C637E"/>
                            <path
                                d="M17.6563 4.03125C17.4922 4.17969 17.3906 4.27344 17.2969 4.36719C16.0781 5.57812 14.8672 6.79688 13.6563 8.00781C13.1875 8.47656 12.9531 8.46875 12.4844 8C12.3203 7.83594 12.1563 7.67188 11.9844 7.5C11.5313 7.04687 11.5313 6.80469 11.9766 6.35938C13.1875 5.14062 14.4063 3.92969 15.625 2.71875C15.7266 2.61719 15.8516 2.54688 15.9609 2.46094C15.9453 2.42188 15.9219 2.38281 15.9063 2.34375C15.4375 2.34375 14.9609 2.34375 14.4922 2.34375C13.9609 2.34375 13.7578 2.14062 13.75 1.61719C13.75 1.28906 13.7422 0.96875 13.75 0.640625C13.7578 0.234375 13.9688 0 14.375 0C16.0391 -0.0078125 17.7031 -0.0078125 19.375 0C19.7734 0 20 0.226562 20 0.640625C20.0078 2.29688 20.0078 3.94531 20 5.60156C20 6.03125 19.7734 6.24219 19.3438 6.25C18.9922 6.25781 18.6406 6.25781 18.2891 6.25C17.8828 6.24219 17.6719 6.02344 17.6641 5.61719C17.6484 5.10938 17.6563 4.60938 17.6563 4.03125Z"
                                fill="#5C637E"/>
                        </g>
                        <defs>
                            <clipPath id="clip0_2235_6384">
                                <rect width="20" height="20" fill="white"/>
                            </clipPath>
                        </defs>
                    </svg>

                    <span>{wp.i18n.__('Popup Icon', 'radio-player')}</span>
                    {!isPro &&
                        <span className="radio-player-pro-icon">
                            <i className={`radio-player-icon icon-pro`}/>
                        </span>
                    }

                </h4>

                <div className="form-field">
                    <div className={`input-group ${!isPro ? 'disabled' : ''}`}>
                        <FormToggle
                            checked={isPro && popup_icon}
                            onChange={() => {

                                if (!isPro) {
                                    showProModal(wp.i18n.__('Upgrade to PRO to use the popup player.', 'radio-player'));
                                    return;
                                }

                                setEditData(data => ({...data, popup_icon: !popup_icon}))
                            }}
                        />

                    </div>

                    <p className="description">{wp.i18n.__('Show/ hide the popup player icon in the player.', 'radio-player')}</p>
                </div>
            </div>

            {/* Playlist Icon */}
            {!['skin11', 'skin12'].includes(skin) &&
                <div className="form-group">

                    <h4 className={`form-label`}>
                        <svg width="20" height="18" viewBox="0 0 20 18" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <g clipPath="url(#clip0_2235_6398)">
                                <path
                                    d="M12.8918 11.0511C12.8918 10.9992 12.8918 10.8435 12.8918 10.6877C12.8918 7.64153 12.8918 4.60403 12.8918 1.55787C12.8918 0.675183 13.4289 0.0521057 14.2257 0.00883645C15.0583 -0.0430866 15.6402 0.519413 15.7745 1.41076C15.8103 1.64441 16.0073 1.92134 16.2132 2.05114C16.6787 2.35403 17.198 2.58768 17.6904 2.85595C18.711 3.40114 19.5973 4.07614 19.8927 5.21845C20.0091 5.67711 20.0449 6.19634 19.9554 6.65499C19.83 7.29537 19.1586 7.66749 18.4961 7.61557C17.8694 7.56364 17.2517 7.08768 17.2248 6.47326C17.198 5.85884 16.8488 5.57326 16.3654 5.33095C16.1863 5.23576 15.9983 5.14922 15.7566 5.02807C15.7566 5.23576 15.7566 5.38288 15.7566 5.52999C15.7566 8.45499 15.7118 11.3886 15.7745 14.3136C15.8282 16.6069 13.6886 18.2598 11.3161 17.9742C9.5704 17.7665 8.28123 16.6675 7.96789 15.231C7.58293 13.4915 8.67514 11.7781 10.582 11.2848C11.2893 11.0858 12.0592 11.1204 12.8918 11.0511Z"
                                    fill="#5C637E"/>
                                <path
                                    d="M5.72972 2.76923C4.34208 2.76923 2.96339 2.76923 1.57574 2.76923C0.644675 2.76923 -0.0088617 2.18077 9.08544e-05 1.36731C0.00904341 0.5625 0.653627 0 1.57574 0C4.34208 0 7.11737 0 9.88371 0C10.8148 0 11.4683 0.588462 11.4594 1.40192C11.4504 2.19808 10.8058 2.76058 9.88371 2.76058C8.49606 2.76923 7.11737 2.76923 5.72972 2.76923Z"
                                    fill="#5C637E"/>
                                <path
                                    d="M5.72972 6.23077C7.11737 6.23077 8.49606 6.23077 9.88371 6.23077C10.7969 6.23077 11.4415 6.79327 11.4504 7.59807C11.4594 8.42019 10.8058 9 9.87476 9C7.10842 9 4.33313 9 1.56679 9C0.653627 9 0.00904341 8.42884 9.08544e-05 7.63269C-0.0088617 6.81057 0.644675 6.23077 1.57574 6.23077C2.96339 6.23077 4.35103 6.23077 5.72972 6.23077Z"
                                    fill="#5C637E"/>
                                <path
                                    d="M3.22301 15.2308C2.63214 15.2308 2.03232 15.2394 1.44146 15.2308C0.617821 15.2135 -0.00885783 14.599 9.4723e-05 13.8288C0.00904727 13.076 0.626773 12.4788 1.4325 12.4702C2.62319 12.4615 3.81388 12.4615 5.00457 12.4702C5.82821 12.4788 6.45488 13.1019 6.44593 13.8721C6.43698 14.625 5.81925 15.2221 5.01352 15.2308C4.4137 15.2394 3.82283 15.2308 3.22301 15.2308Z"
                                    fill="#5C637E"/>
                            </g>
                            <defs>
                                <clipPath id="clip0_2235_6398">
                                    <rect width="20" height="18" fill="white"/>
                                </clipPath>
                            </defs>
                        </svg>

                        <span>{wp.i18n.__('Stations Playlist', 'radio-player')}</span>
                        {!isPro &&
                            <span className="radio-player-pro-icon">
                                <i className={`radio-player-icon icon-pro`}/>
                            </span>
                        }
                    </h4>

                    <div className="form-field">

                        <div className={`input-group ${!isPro ? 'disabled' : ''}`}>
                            <FormToggle
                                checked={isPro && playlist_icon}
                                onChange={() => {

                                    if (!isPro) {
                                        showProModal(wp.i18n.__('Upgrade to PRO to use the playlist.', 'radio-player'));
                                        return;
                                    }

                                    setEditData(data => ({...data, playlist_icon: !playlist_icon}))
                                }}
                            />


                        </div>

                        <p className="description">{wp.i18n.__('If player has multiple radio stations, playlist icon could be show/hide.', 'radio-player')}</p>

                    </div>

                </div>
            }

            {/* Volume Control */}
            {!['skin11'].includes(skin) &&
                <div className="form-group">
                    <h4 className={`form-label`}>
                        <svg width="20" height="17" viewBox="0 0 20 17" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <g clipPath="url(#clip0_2235_6413)">
                                <path
                                    d="M11.1863 0C11.1863 5.70773 11.1863 11.3025 11.1863 17C10.2663 16.2917 9.41947 15.6244 8.56218 14.9571C7.47488 14.1051 6.39804 13.2325 5.28983 12.4112C5.08074 12.247 4.75664 12.1546 4.48481 12.1546C3.1466 12.1238 1.80839 12.1238 0.459723 12.1546C0.0833509 12.1649 -0.0107422 12.0417 -0.0107422 11.6926C-0.000287403 9.5779 -0.000287403 7.4529 -0.0107422 5.33816C-0.0107422 4.98913 0.0833509 4.86594 0.459723 4.87621C1.76657 4.89674 3.07342 4.907 4.38027 4.86594C4.72527 4.85568 5.13301 4.72222 5.40484 4.50664C7.32852 3.03865 9.21038 1.55012 11.1863 0Z"
                                    fill="#5C637E"/>
                                <path
                                    d="M20.0002 7.94565C20.0002 11.3025 18.8188 13.4686 16.3828 14.9571C16.226 15.0495 16.0482 15.1419 15.8601 15.183C15.5464 15.2548 15.2119 15.2651 15.076 14.8853C14.94 14.5157 15.1073 14.2796 15.4523 14.1051C16.5605 13.5713 17.4387 12.7808 18.0137 11.7132C19.5088 8.92089 19.0279 4.65036 15.421 2.88466C15.1073 2.73068 14.961 2.52536 15.055 2.19686C15.1596 1.84782 15.4732 1.70411 15.7451 1.83756C16.3723 2.14553 17.0101 2.47403 17.5433 2.92572C19.2683 4.38345 20.0211 6.29287 20.0002 7.94565Z"
                                    fill="#5C637E"/>
                                <path
                                    d="M17.5116 8.51027C17.4698 10.4094 16.5184 11.7337 14.8666 12.6268C14.5216 12.8116 14.1661 12.9143 13.9257 12.4831C13.7166 12.1135 13.8316 11.8569 14.3125 11.6002C16.5184 10.4094 16.9994 7.70954 15.2639 6.08756C14.9607 5.81039 14.6052 5.57428 14.2498 5.37923C13.8838 5.18418 13.7061 4.94807 13.9047 4.55797C14.1034 4.15761 14.4484 4.15761 14.7934 4.34239C16.4871 5.23551 17.4803 6.57005 17.5116 8.51027Z"
                                    fill="#5C637E"/>
                            </g>
                            <defs>
                                <clipPath id="clip0_2235_6413">
                                    <rect width="20" height="17" fill="white"/>
                                </clipPath>
                            </defs>
                        </svg>
                        <span>{wp.i18n.__('Volume Control', 'radio-player')}</span>
                    </h4>

                    <div className="form-field">
                        <FormToggle
                            checked={volume_control}
                            onChange={() => setEditData(data => ({...data, volume_control: !volume_control}))}
                        />

                        <p className="description">{wp.i18n.__('Show/hide the volume control button.', 'radio-player')}</p>
                    </div>
                </div>
            }

            {/* Player Status */}
            {!['skin11'].includes(skin) &&
                <div className="form-group">
                    <h4 className={`form-label`}>
                        <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <g clipPath="url(#clip0_2235_6428)">
                                <path
                                    d="M10.0357 -0.000139192C4.50894 -0.00906776 0.00894199 4.44629 1.34231e-05 9.92843C-0.00891515 15.5088 4.43751 19.9909 9.98216 19.9999C15.4911 20.0088 19.9911 15.5356 20 10.0534C20 4.49986 15.5447 0.00878938 10.0357 -0.000139192ZM10 16.5802C6.37501 16.5802 3.41966 13.6249 3.41966 9.99986C3.41966 6.37486 6.37501 3.4195 10 3.4195C13.625 3.4195 16.5804 6.37486 16.5804 9.99986C16.5804 13.6249 13.625 16.5802 10 16.5802Z"
                                    fill="#5C637E"/>
                                <path
                                    d="M9.99979 13.6429C12.0117 13.6429 13.6426 12.0119 13.6426 10C13.6426 7.98811 12.0117 6.35715 9.99979 6.35715C7.9879 6.35715 6.35693 7.98811 6.35693 10C6.35693 12.0119 7.9879 13.6429 9.99979 13.6429Z"
                                    fill="#5C637E"/>
                            </g>
                            <defs>
                                <clipPath id="clip0_2235_6428">
                                    <rect width="20" height="20" fill="white"/>
                                </clipPath>
                            </defs>
                        </svg>
                        <span>{wp.i18n.__('Player Status', 'radio-player')}</span>
                    </h4>

                    <div className="form-field">
                        <FormToggle
                            checked={player_status}
                            onChange={() => setEditData(data => ({...data, player_status: !player_status}))}
                        />
                        <p className="description">{wp.i18n.__('Show/hide the player status.', 'radio-player')}</p>
                    </div>
                </div>
            }

            {/* Song Title */}
            {!['skin11'].includes(skin) &&
                <div className="form-group">
                    <h4 className={`form-label`}>

                        <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20"
                             viewBox="0 0 512.000000 512.000000" preserveAspectRatio="xMidYMid meet">
                            <g transform="translate(0.000000,512.000000) scale(0.100000,-0.100000)" fill="#000000"
                               stroke="none">
                                <path
                                    d="M765 4676 c-44 -20 -83 -60 -105 -108 -18 -40 -20 -68 -20 -373 0 -300 2 -334 19 -370 36 -77 111 -125 196 -125 83 0 148 39 188 115 20 37 22 58 25 248 l3 207 637 -2 637 -3 0 -1705 0 -1705 -252 -5 -251 -5 -43 -30 c-64 -45 -91 -104 -87 -187 4 -82 39 -139 107 -175 l44 -23 697 0 697 0 44 23 c68 36 103 93 107 175 4 83 -23 142 -87 187 l-43 30 -251 5 -252 5 0 1705 0 1705 637 3 637 2 3 -207 c3 -190 5 -211 25 -248 11 -22 32 -50 45 -62 80 -75 224 -68 300 13 56 60 58 74 58 432 l0 328 -25 51 c-16 32 -41 61 -67 79 l-41 29 -1776 2 c-1462 2 -1781 0 -1806 -11z"/>
                            </g>
                        </svg>

                        <span>{wp.i18n.__('Show Track Title', 'radio-player')}</span>
                    </h4>

                    <div className="form-field">

                        <div className={`input-group`}>
                            <FormToggle
                                checked={show_title}
                                onChange={() => {
                                    setEditData(data => ({...data, show_title: !show_title}))
                                }}
                            />
                        </div>

                        <p className="description">
                            {wp.i18n.__('Show/ hide the currently playing track title in the player.', 'radio-player')}
                        </p>


                        {show_title &&
                            <div className="fallback-field">
                                <h5 className={`form-label`}>{wp.i18n.__('Fallback Title', 'radio-player')}</h5>

                                <div className={`input-group`}>
                                    <input
                                        type="text"
                                        value={fallback_title}
                                        onChange={(e) => setEditData(data => ({
                                            ...data,
                                            fallback_title: e.target.value
                                        }))}
                                    />

                                    <p className="description">
                                        {wp.i18n.__('Enter a default title when the stream does not provide metadata.', 'radio-player')}
                                    </p>
                                </div>

                            </div>
                        }

                        <div className="rp-notice rp-notice-warning">
                            <p className="rp-notice-content">
                                {wp.i18n.__('Not all streams may provide metadata for the currently playing content.', 'radio-player')}
                            </p>
                        </div>

                        <div className="rp-notice rp-notice-info">
                            <p className="rp-notice-content">
                                {wp.i18n.__('If metadata isn\'t visible, ensure "allow_url_fopen" is enabled on your server. We advise contacting your web host to activate this setting.', 'radio-player')}
                            </p>
                        </div>

                        <div className="rp-notice rp-notice-info">
                            <p className="rp-notice-content">
                                {wp.i18n.__('If the "allow_url_fopen" is enabled on the server, but still the metadata not displaying, try to enable the "Metadata Proxy" option in the plugin settings page. ', 'radio-player')}
                            </p>
                        </div>

                    </div>
                </div>
            }

            {/* Artist Name */}
            {
                !['skin11'].includes(skin) &&
                <div className="form-group">
                    <h4 className={`form-label`}>

                        <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 512 512"
                             preserveAspectRatio="xMidYMid meet">
                            <g transform="translate(0.000000,512.000000) scale(0.100000,-0.100000)" fill="currentColor"
                               stroke="none">
                                <path
                                    d="M2250 4949 c-135 -24 -286 -85 -397 -160 -82 -55 -224 -208 -274 -295 -102 -175 -133 -293 -133 -494 0 -160 12 -226 65 -359 120 -302 404 -532 721 -585 l71 -12 -138 -22 c-689 -114 -1295 -524 -1661 -1123 -220 -361 -355 -838 -342 -1211 3 -115 12 -138 72 -182 27 -21 34 -21 1173 -24 l1146 -2 -65 69 c-162 171 -191 385 -81 594 50 94 144 187 237 232 71 35 170 65 213 65 l22 0 3 387 3 388 26 56 c31 65 82 120 141 150 26 13 200 55 442 105 221 47 409 87 419 90 23 7 -33 47 -190 134 -324 180 -739 298 -1061 301 l-77 0 85 28 c177 57 288 126 415 255 126 129 198 256 247 436 32 118 32 336 0 457 -47 180 -126 320 -256 449 -128 128 -270 210 -445 255 -89 23 -297 33 -381 18z m208 -1906 c-21 -2 -55 -2 -75 0 -21 2 -4 4 37 4 41 0 58 -2 38 -4z"/>
                                <path
                                    d="M3795 2421 c-358 -76 -665 -145 -682 -154 -17 -9 -42 -33 -55 -54 l-23 -38 -3 -453 -3 -454 -68 7 c-240 24 -445 -140 -459 -368 -4 -74 -3 -82 32 -152 80 -162 234 -245 433 -232 152 9 286 97 359 238 l29 54 3 593 c1 325 5 592 7 592 6 0 897 188 933 197 l22 5 0 -385 0 -386 -82 6 c-188 13 -346 -74 -420 -233 -74 -158 -32 -319 113 -428 141 -106 323 -121 486 -41 87 43 152 105 190 183 l28 57 0 742 c0 733 0 742 -21 769 -34 46 -69 67 -121 70 -34 3 -239 -37 -698 -135z"/>
                            </g>
                        </svg>

                        <span>{wp.i18n.__('Show Artist Name', 'radio-player')}</span>
                        {!isPro && <span className="radio-player-pro-icon"><i className={`radio-player-icon icon-pro`}/></span>}

                    </h4>

                    <div className="form-field">

                        <div className={`input-group ${!isPro ? 'disabled' : ''}`}>
                            <FormToggle
                                checked={show_artist}
                                onChange={() => {

                                    if (!isPro) {
                                        showProModal(wp.i18n.__('Upgrade to PRO to show the artist name.', 'radio-player'));
                                        return;
                                    }

                                    setEditData(data => ({...data, show_artist: !show_artist}))
                                }}
                            />
                        </div>

                        <p className="description">{wp.i18n.__('Show/ hide the currently playing track artist name in the player.', 'radio-player')}</p>

                        {show_artist &&
                            <div className="fallback-field">
                                <h5 className={`form-label`}>{wp.i18n.__('Fallback Artist', 'radio-player')}</h5>

                                <div className={`input-group`}>
                                    <input
                                        type="text"
                                        value={fallback_artist}
                                        onChange={(e) => setEditData(data => ({
                                            ...data,
                                            fallback_artist: e.target.value
                                        }))}
                                    />

                                    <p className="description">
                                        {wp.i18n.__('Enter a default artist when the stream does not provide metadata.', 'radio-player')}
                                    </p>
                                </div>

                            </div>
                        }

                    </div>
                </div>
            }

            {/* Artwork */}
            {
                !['skin11'].includes(skin) &&
                <div className="form-group">
                    <h4 className={`form-label`}>

                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 512 512"
                             preserveAspectRatio="xMidYMid meet">
                            <g transform="translate(0,512) scale(0.1,-0.1)" fill="currentColor" stroke="none">
                                <path
                                    d="M961 4464 c-155 -41 -277 -169 -310 -322 -16 -75 -16 -3089 0 -3164 24 -112 109 -228 207 -281 110 -60 25 -57 1699 -57 1083 0 1548 3 1585 11 157 33 293 170 327 327 8 38 11 489 11 1585 0 1674 3 1589 -57 1699 -53 98 -169 183 -281 207 -38 8 -488 11 -1590 10 -1310 0 -1545 -2 -1591 -15z m2555 -638 c200 -53 338 -244 321 -445 -11 -133 -81 -252 -190 -322 -86 -55 -150 -72 -258 -67 -76 4 -97 9 -162 41 -91 45 -156 112 -200 205 -30 63 -32 74 -32 177 0 104 2 114 33 178 91 186 297 284 488 233z m-1620 -1071 c109 -19 216 -55 314 -106 156 -81 119 -46 1398 -1301 l283 -278 -1410 0 -1411 0 0 750 0 750 43 27 c116 74 267 134 400 158 100 18 277 18 383 0z"/>
                            </g>
                        </svg>

                        <span>{wp.i18n.__('Artwork Image', 'radio-player')}</span>
                        {!isPro && <span className="radio-player-pro-icon"><i className={`radio-player-icon icon-pro`}/></span>}

                    </h4>

                    <div className="form-field">

                        <div className={`input-group ${!isPro ? 'disabled' : ''}`}>
                            <FormToggle
                                checked={show_artwork}
                                onChange={() => {

                                    if (!isPro) {
                                        showProModal(wp.i18n.__('Upgrade to PRO to show the artwork image.', 'radio-player'));
                                        return;
                                    }

                                    setEditData(data => ({...data, show_artwork: !show_artwork}))
                                }}
                            />
                        </div>

                        <p className="description">
                            {wp.i18n.__('Show/ hide the currently playing track artwork image in the player.', 'radio-player')}
                            <br/>
                            {wp.i18n.__('Note: The artwork image will replace the station logo image.', 'radio-player')}
                        </p>
                    </div>
                </div>
            }

            {/* Player Text */
            }
            {
                'skin11' === skin &&
                <div className="form-group">
                    <h4 className={`form-label`}>
                        <svg width="20" height="17" viewBox="0 0 20 17" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <g clipPath="url(#clip0_2235_6378)">
                                <path
                                    d="M10 16.7031C8.08594 16.7031 6.17188 16.7109 4.25781 16.7031C2.24219 16.6953 0.617188 15.4453 0.132812 13.5391C0.046875 13.2031 0.0078125 12.8438 0.0078125 12.5C0 9.75 0 7 0 4.25781C0 1.89844 1.77344 0.046875 4.14062 0.0234375C8.04688 -0.0078125 11.9531 -0.0078125 15.8516 0.0234375C18.2344 0.046875 19.9922 1.89844 19.9922 4.29688C19.9922 7.00781 19.9922 9.71094 19.9922 12.4219C19.9922 14.8594 18.1875 16.6875 15.7344 16.7031C13.8281 16.7109 11.9141 16.7031 10 16.7031ZM5.82812 10.8594C5.53906 10.8594 5.28125 10.8516 5.02344 10.8594C4.53125 10.875 4.16406 11.2266 4.16406 11.6875C4.15625 12.1406 4.52344 12.5078 5.02344 12.5156C6.11719 12.5234 7.21094 12.5234 8.30469 12.5156C8.82812 12.5156 9.15625 12.1953 9.16406 11.7109C9.17188 11.2187 8.82031 10.875 8.29688 10.8672C8.03906 10.8594 7.78906 10.8672 7.53125 10.8672C7.53125 9.17969 7.53125 7.53906 7.53125 5.89062C8.07812 5.89062 8.60156 5.89062 9.17188 5.89062C9.17188 6.17969 9.15625 6.45312 9.17969 6.71875C9.21875 7.21094 9.55469 7.52344 10.0156 7.52344C10.4766 7.52344 10.8281 7.21094 10.8438 6.71875C10.8672 6.14844 10.8672 5.57031 10.8438 5C10.8203 4.52344 10.4922 4.21875 10.0312 4.21875C7.78906 4.21094 5.55469 4.21094 3.3125 4.21875C2.85156 4.21875 2.52344 4.55469 2.51562 5.00781C2.5 5.57812 2.5 6.15625 2.51562 6.72656C2.53125 7.19531 2.89062 7.52344 3.35156 7.52344C3.82031 7.52344 4.14062 7.20312 4.16406 6.71094C4.17188 6.44531 4.16406 6.17188 4.16406 5.89844C4.75 5.89844 5.28906 5.89844 5.84375 5.89844C5.82812 7.54688 5.82812 9.16406 5.82812 10.8594ZM14.2109 5.01562C13.6797 5.01562 13.1484 5.00781 12.6094 5.01562C12.0469 5.02344 11.6719 5.36719 11.6641 5.84375C11.6562 6.32031 12.0313 6.6875 12.5859 6.69531C13.6406 6.70312 14.6875 6.70312 15.7422 6.69531C16.3047 6.6875 16.6562 6.35156 16.6562 5.85156C16.6562 5.35156 16.3047 5.03125 15.7344 5.01562C15.2188 5.00781 14.7109 5.01562 14.2109 5.01562ZM14.1641 9.20312C14.6719 9.20312 15.1797 9.21094 15.6875 9.20312C16.2891 9.19531 16.6484 8.88281 16.6562 8.36719C16.6641 7.85156 16.2969 7.52344 15.7109 7.51562C14.6875 7.50781 13.6562 7.50781 12.6328 7.51562C12.0547 7.52344 11.6641 7.875 11.6719 8.36719C11.6797 8.85156 12.0547 9.1875 12.6172 9.19531C13.125 9.21094 13.6484 9.20312 14.1641 9.20312ZM14.1719 11.7031C14.6953 11.7031 15.2109 11.7109 15.7344 11.7031C16.3047 11.6953 16.6562 11.3594 16.6562 10.8594C16.6562 10.3594 16.3047 10.0234 15.7344 10.0234C14.6953 10.0156 13.6562 10.0156 12.6172 10.0234C12.0547 10.0312 11.6797 10.3672 11.6641 10.8438C11.6484 11.3359 12.0391 11.6953 12.6172 11.7031C13.1328 11.7109 13.6562 11.7031 14.1719 11.7031Z"
                                    fill="#5C637E"/>
                            </g>
                            <defs>
                                <clipPath id="clip0_2235_6378">
                                    <rect width="20" height="16.7031" fill="white"/>
                                </clipPath>
                            </defs>
                        </svg>
                        <span>{wp.i18n.__('Player Button Text', 'radio-player')}</span>
                    </h4>

                    <div className="form-field">
                        <TextControl
                            value={player_text}
                            onChange={player_text => setEditData(data => ({...data, player_text}))}
                            placeholder={wp.i18n.__('Enter the text for the player button.', 'radio-player')}
                        />
                        <p className="description">{wp.i18n.__('Enter the text for the player button.', 'radio-player')}</p>
                    </div>
                </div>
            }

        </div>
    )

}