import {Tooltip} from "react-tooltip";
import Uploader from "./Uploader";
import AppContext from "../../../contexts/AppContext";
import {getSkins, showProModal, useMounted} from "../../../includes/functions";
import ColorPickerWrapper from "../../../includes/ColorPickerWrapper";

let {

    ButtonGroup,
    Button,
    RangeControl,
    FormToggle,
    SelectControl,

    __experimentalRadioGroup: RadioGroup,
    __experimentalRadio: Radio,

    GradientPicker,
    __experimentalGradientPicker: GradientPicker2,

} = wp.components;

if (!GradientPicker) {
    GradientPicker = GradientPicker2;
}

const {useContext, useEffect} = React;

export default function Appearance() {
    const isPro = !!radioPlayer.isPro;

    const context = useContext(AppContext);
    const {editData, setEditData} = context;
    const {
        bg_type = 'color',
        color_type = 'solid',
        bg_color,
        station_list_title = wp.i18n.__('Stations', 'radio-player'),
        width,
        width_mobile,
        border_radius,
        bg_image,
        text_color,

        box_shadow = false,
        box_shadow_h = 0,
        box_shadow_v = 0,
        box_shadow_blur = 0,
        box_shadow_spread = 0,
        box_shadow_color = '#000000',
        box_shadow_type = '',

        width_device = 'desktop',
        skin,
    } = editData;

    const isMounted = useMounted();

    useEffect(() => {
        if (!isMounted) return;

        const skins = getSkins();
        const defaultBgColors = {
            skin1: {
                solid: '#F2F1FC',
                gradient: skins.skin1.config.bg_color,
            },
            skin2: {
                solid: skins.skin2.config.bg_color,
                gradient: 'linear-gradient(135deg, #5581A4 0%, #A3C7E0 100%)',
            },
            skin3: {
                solid: skins.skin3.config.bg_color,
                gradient: 'linear-gradient(135deg, #f4f7f9 0%, #cfdce5 100%)',
            },
            skin4: {
                solid: '#7FA2FF',
                gradient: skins.skin4.config.bg_color,
            },
            skin5: {
                solid: '#3B5F7C',
                gradient: skins.skin5.config.bg_color,
            },
            skin6: {
                solid: '#2A2A2A',
                gradient: skins.skin6.config.bg_color,
            },
            skin7: {
                solid: skins.skin7.config.bg_color,
                gradient: 'linear-gradient(135deg, #235984 0%, #5A8BB0 100%)',
            },
            skin8: {
                solid: '#226BC7',
                gradient: skins.skin8.config.bg_color,
            },
            skin9: {
                solid: skins.skin9.config.bg_color,
                gradient: 'linear-gradient(135deg, #070044 0%, #2F1C7A 100%)',
            },
            skin10: {
                solid: skins.skin10.config.bg_color,
                gradient: 'linear-gradient(135deg, #FFFFFF 0%, #E6EAF0 100%)',
            },
            skin11: {
                solid: '#F03E78',
                gradient: skins.skin11.config.bg_color,
            },
            skin12: {
                solid: skins.skin12.config.bg_color,
                gradient: 'linear-gradient(135deg, #F9FAFE 0%, #DDE4F4 100%)',
            },
        };

        const newBgColor = defaultBgColors[skin]?.[color_type === 'solid' ? 'solid' : 'gradient'];

        if (newBgColor) {
            setEditData(prev => ({...prev, bg_color: newBgColor}));
        }

    }, [skin, color_type]);

    return (
        <div className="settings-body">

            <div className="settings-header">
                <h3 className="header-title">{wp.i18n.__('Player Appearance', 'radio-player')}</h3>
                <p>{wp.i18n.__('Customize the player appearance.', 'radio-player')}</p>
            </div>

            {/* Station list title */}
            {'skin12' === skin &&
                <div className="form-group">
                    <label className={`form-label`}>
                        <span>{wp.i18n.__('Station List Title', 'radio-player')}</span>
                    </label>

                    <div className="form-field">
                        <input
                            type="text"
                            value={station_list_title}
                            onChange={e => setEditData(data => ({...data, station_list_title: e.target.value}))}
                        />

                        <p className="description">{wp.i18n.__(`Enter the station list title.`, 'radio-player')}</p>
                    </div>
                </div>
            }

            {/* Width */}
            <div className="form-group">
                <h4 className={`form-label`}>
                    <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <g clipPath="url(#clip0_2235_6338)">
                            <path
                                d="M10.0156 0C12.3828 0 14.7578 0 17.125 0C18.7969 0 20 1.20313 20 2.88281C20 7.63281 20 12.3828 20 17.1406C20 18.7969 18.7969 20 17.1484 20C12.3828 20 7.61719 20 2.85156 20C1.20312 20 0 18.7891 0 17.1406C0 12.375 0 7.60938 0 2.84375C0 1.21875 1.21094 0 2.82812 0C5.22656 0 7.625 0 10.0156 0ZM7.78125 10.5703C7.78125 10.7656 7.78125 10.9219 7.78125 11.0703C7.78125 12.3438 7.78125 13.6172 7.78125 14.8984C7.78125 15.3359 7.98438 15.5547 8.33594 15.5469C8.67188 15.5391 8.875 15.2969 8.89062 14.8828C8.89062 14.8281 8.89062 14.7812 8.89062 14.7266C8.89062 11.5781 8.89062 8.42969 8.89062 5.28125C8.89062 5.11719 8.90625 4.89844 8.82031 4.79688C8.67969 4.64062 8.4375 4.53125 8.26562 4.54688C8.09375 4.5625 7.88281 4.71875 7.82031 4.83594C7.77344 4.9375 7.78125 5.16406 7.78125 5.28906C7.78125 6.52344 7.78125 7.75781 7.78125 9C7.78125 9.14062 7.78125 9.28125 7.78125 9.41406C6.92188 9.41406 6.11719 9.41406 5.19531 9.41406C5.38281 9.25781 5.48438 9.17969 5.57812 9.09375C5.67969 9 5.78125 8.89062 5.88281 8.78906C6.14844 8.50781 6.17187 8.1875 5.95312 7.96094C5.73438 7.73438 5.38281 7.73438 5.11719 7.99219C4.58594 8.50781 4.0625 9.03125 3.54688 9.5625C3.27344 9.84375 3.27344 10.1484 3.53906 10.4219C4.0625 10.9609 4.60156 11.5 5.14062 12.0234C5.40625 12.2812 5.75 12.2812 5.96875 12.0391C6.17969 11.8125 6.16406 11.5 5.91406 11.2344C5.71094 11.0234 5.49219 10.8281 5.20312 10.5547C6.125 10.5703 6.92969 10.5703 7.78125 10.5703ZM12.2188 9.4375C12.2188 9.24219 12.2188 9.07812 12.2188 8.90625C12.2188 7.63281 12.2266 6.35156 12.2109 5.07812C12.2109 4.66406 12.0078 4.45312 11.6797 4.44531C11.3438 4.4375 11.125 4.6875 11.1016 5.09375C11.1016 5.15625 11.1016 5.22656 11.1016 5.28906C11.1016 8.42188 11.1016 11.5625 11.1016 14.6953C11.1016 14.8594 11.1094 15.0391 11.1641 15.1953C11.2578 15.4609 11.5391 15.5469 11.7422 15.5234C11.8672 15.5078 12.0547 15.4141 12.1719 15.2344C12.2344 15.1406 12.2109 14.8125 12.2109 14.6797C12.2109 13.4453 12.2109 12.2031 12.2109 10.9688C12.2109 10.8359 12.2266 10.6953 12.2344 10.5703C13.0781 10.5703 13.9453 10.5469 14.7422 10.5469C14.7109 10.5859 14.6719 10.6328 14.6641 10.6406C14.4844 10.8281 14.3047 11.0156 14.125 11.2031C13.8438 11.5 13.8203 11.8281 14.0625 12.0625C14.2891 12.2734 14.6328 12.2578 14.8984 12C15.4219 11.4922 15.9375 10.9766 16.4453 10.4531C16.7344 10.1562 16.7344 9.85156 16.4453 9.5625C15.9375 9.03906 15.4219 8.52344 14.8984 8.01562C14.6328 7.75781 14.2891 7.73437 14.0625 7.94531C13.8203 8.17188 13.8438 8.50781 14.125 8.80469C14.3125 9.00781 14.5156 9.20312 14.7578 9.44531C13.875 9.4375 13.0859 9.4375 12.2188 9.4375Z"
                                fill="#5C637E"/>
                        </g>
                        <defs>
                            <clipPath id="clip0_2235_6338">
                                <rect width="20" height="20" fill="white"/>
                            </clipPath>
                        </defs>
                    </svg>
                    <span>{wp.i18n.__('Player Width', 'radio-player')}</span>
                </h4>

                <div className="form-field">

                    <RadioGroup
                        label={wp.i18n.__('Devices', 'radio-player')}
                        checked={width_device}
                        onChange={(value) => setEditData(data => ({...data, width_device: value}))}
                        className="width-devices"
                    >
                        <Radio value="desktop">
                            <i className="dashicons dashicons-desktop"/>
                            <span>{wp.i18n.__('Desktop', 'radio-player')}</span>
                        </Radio>
                        <Radio value="mobile">
                            <i className="dashicons dashicons-smartphone"/>
                            <span>{wp.i18n.__('Mobile', 'radio-player')}</span>
                        </Radio>
                    </RadioGroup>


                    <RangeControl
                        value={width_device === 'mobile' ? width_mobile : width}
                        onChange={width => {
                            if (width_device === 'mobile') {
                                setEditData(data => ({...data, width_mobile: width}))
                            } else {
                                setEditData(data => ({...data, width}))
                            }
                        }}
                        min={100}
                        max={'mobile' === width_device ? 500 : 1000}
                        allowReset={true}
                        step={5}
                        resetFallbackValue={350}
                    />
                    <p className="description">{wp.i18n.__('Set the player width on ', 'radio-player')} {width_device} </p>
                </div>
            </div>

            {/* Background Type */}
            <div className={`form-group`}>
                <h4 className={`form-label`}>
                    <span>{wp.i18n.__('Background Type', 'radio-player')}</span>
                    {!isPro && <span className="radio-player-pro-icon"><i className={`radio-player-icon icon-pro`}/></span>}
                </h4>

                <div className="form-field">
                    <div
                        className={`input-group ${!isPro ? 'disabled' : ''}`}
                        onClick={() => {
                            if (!isPro) {
                                showProModal(wp.i18n.__('Upgrade to PRO to customize the player background.', 'radio-player'));
                            }
                        }}
                    >

                        <ButtonGroup>
                            <Button
                                onClick={() => setEditData(data => ({...data, bg_type: 'color'}))}
                                isPrimary={bg_type === 'color'}
                            >
                                <svg width="18" height="18" viewBox="0 0 18 18" fill="none"
                                     xmlns="http://www.w3.org/2000/svg">
                                    <path fillRule="evenodd" clipRule="evenodd"
                                          d="M3.5998 1.79999C2.60569 1.79999 1.7998 2.60588 1.7998 3.59999V13.5C1.7998 14.9912 3.00864 16.2 4.4998 16.2C5.99097 16.2 7.1998 14.9912 7.1998 13.5V3.59999C7.1998 2.60588 6.39392 1.79999 5.3998 1.79999H3.5998ZM4.4998 14.4C4.99686 14.4 5.3998 13.997 5.3998 13.5C5.3998 13.0029 4.99686 12.6 4.4998 12.6C4.00275 12.6 3.5998 13.0029 3.5998 13.5C3.5998 13.997 4.00275 14.4 4.4998 14.4Z"
                                          fill="#5C637E"/>
                                    <path
                                        d="M8.9998 12.8183L13.4093 8.40876C14.1123 7.70581 14.1123 6.56612 13.4093 5.86317L12.1365 4.59038C11.4336 3.88744 10.2939 3.88744 9.59096 4.59038L8.9998 5.18154V12.8183Z"
                                        fill="#5C637E"/>
                                    <path
                                        d="M14.3998 16.2H8.16374L13.5637 10.8H14.3998C15.3939 10.8 16.1998 11.6059 16.1998 12.6V14.4C16.1998 15.3941 15.3939 16.2 14.3998 16.2Z"
                                        fill="#5C637E"/>
                                </svg>
                                <span>{wp.i18n.__('Color', 'radio-player')}</span>
                            </Button>
                            <Button
                                onClick={() => setEditData(data => ({...data, bg_type: 'image'}))}
                                isPrimary={bg_type === 'image'}
                            >
                                <svg width="18" height="18" viewBox="0 0 18 18" fill="none"
                                     xmlns="http://www.w3.org/2000/svg">
                                    <path fillRule="evenodd" clipRule="evenodd"
                                          d="M5.74964 16.5H12.2496C14.7923 16.5 16.5 14.7164 16.5 12.0625V5.93754C16.5 3.28358 14.7923 1.5 12.2504 1.5H5.74964C3.20767 1.5 1.5 3.28358 1.5 5.93754V12.0625C1.5 14.7164 3.20767 16.5 5.74964 16.5ZM6.37414 8.25C5.34016 8.25 4.5 7.4087 4.5 6.375C4.5 5.3413 5.34016 4.5 6.37414 4.5C7.40737 4.5 8.24829 5.3413 8.24829 6.375C8.24829 7.4087 7.40737 8.25 6.37414 8.25ZM14.8656 11.2005C15.1168 11.8444 14.9863 12.6184 14.7178 13.2562C14.3995 14.0147 13.7901 14.589 13.0224 14.8398C12.6815 14.9512 12.324 15 11.9673 15H5.64648C5.0175 15 4.46091 14.8491 4.00463 14.5681C3.71879 14.3916 3.66826 13.9845 3.88019 13.7206C4.23465 13.2794 4.58459 12.8367 4.93755 12.3901C5.61028 11.5356 6.06355 11.2879 6.56734 11.5054C6.77172 11.5952 6.97686 11.7299 7.18803 11.8723C7.75065 12.2546 8.53274 12.7802 9.56295 12.2098C10.268 11.8149 10.6769 11.1376 11.0329 10.5478L11.0389 10.5379C11.0641 10.4965 11.0891 10.4552 11.1141 10.4139L11.1141 10.4139C11.2337 10.2159 11.3518 10.0205 11.4854 9.84056C11.6528 9.61533 12.2735 8.91099 13.0774 9.41254C13.5895 9.72833 14.0202 10.1556 14.481 10.613C14.6567 10.7879 14.7819 10.9868 14.8656 11.2005Z"
                                          fill="#5C637E"/>
                                </svg>
                                <span>{wp.i18n.__('Image', 'radio-player')}</span>
                            </Button>
                            <Button
                                onClick={() => setEditData(data => ({...data, bg_type: 'blur'}))}
                                isPrimary={bg_type === 'blur'}
                            >
                                <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18"
                                     viewBox="0 0 18 18" fill="none">
                                    <g clipPath="url(#clip0_817_464)">
                                        <path
                                            d="M18.0047 18C12.0063 18 6.00313 18 0 18C0 12.0063 0 6.00313 0 0C5.99844 0 12.0016 0 18.0047 0C18.0047 5.99844 18.0047 11.9969 18.0047 18ZM9 2.40594C8.94841 2.4716 8.91089 2.51381 8.87337 2.56071C8.13705 3.42366 7.38197 4.27254 6.67379 5.15894C5.92809 6.09224 5.26681 7.0865 4.76498 8.17926C4.41793 8.92965 4.18812 9.6988 4.27723 10.5524C4.51641 12.8504 6.45805 14.6983 8.7655 14.7968C10.7587 14.8812 12.569 13.7603 13.357 11.9359C13.704 11.1292 13.8588 10.2897 13.6759 9.41741C13.5399 8.77488 13.2678 8.18864 12.9583 7.61647C12.1845 6.19541 11.1996 4.92444 10.1537 3.69567C9.78322 3.2642 9.39864 2.85149 9 2.40594ZM3.18916 5.29964C3.18916 4.60552 3.18916 3.90672 3.18916 3.19385C3.90203 3.19385 4.59614 3.19385 5.29964 3.19385C5.29964 2.83742 5.29964 2.49505 5.29964 2.14799C4.2444 2.14799 3.19854 2.14799 2.14799 2.14799C2.14799 3.20323 2.14799 4.24909 2.14799 5.29964C2.49505 5.29964 2.83273 5.29964 3.18916 5.29964ZM15.8473 5.29964C15.8473 4.25378 15.8473 3.20323 15.8473 2.14799C14.8015 2.14799 13.7556 2.14799 12.7004 2.14799C12.7004 2.49505 12.7004 2.84211 12.7004 3.19854C13.3992 3.19854 14.098 3.19854 14.8061 3.19854C14.8061 3.90672 14.8061 4.60083 14.8061 5.30433C15.1579 5.29964 15.4956 5.29964 15.8473 5.29964ZM3.18916 14.8015C3.18916 14.0886 3.18916 13.3945 3.18916 12.7004C2.83273 12.7004 2.49036 12.7004 2.14799 12.7004C2.14799 13.7556 2.14799 14.8015 2.14799 15.8473C3.20323 15.8473 4.24909 15.8473 5.29495 15.8473C5.29495 15.4956 5.29495 15.1532 5.29495 14.8015C4.59614 14.8015 3.90672 14.8015 3.18916 14.8015ZM15.8473 15.852C15.8473 14.7968 15.8473 13.7462 15.8473 12.7004C15.4956 12.7004 15.1579 12.7004 14.8061 12.7004C14.8061 13.3992 14.8061 14.0933 14.8061 14.8015C14.098 14.8015 13.4039 14.8015 12.7004 14.8015C12.7004 15.1579 12.7004 15.5003 12.7004 15.852C13.7556 15.852 14.8015 15.852 15.8473 15.852Z"
                                            fill="#5C637E"/>
                                        <path
                                            d="M9.00927 3.98193C9.28129 4.30554 9.54862 4.61508 9.80656 4.9293C10.6836 6.00799 11.509 7.11951 12.1562 8.35296C12.3813 8.78444 12.5877 9.22529 12.658 9.71304C12.7847 10.5994 12.5408 11.4108 12.0437 12.1377C11.1948 13.3853 9.65649 13.9903 8.21667 13.662C6.7112 13.3196 5.59499 12.1237 5.35112 10.6229C5.25732 10.0413 5.34643 9.49262 5.57155 8.95796C5.99833 7.94025 6.59864 7.02571 7.25992 6.15338C7.78989 5.44989 8.36206 4.76984 8.91547 4.08511C8.93892 4.05228 8.96706 4.02414 9.00927 3.98193Z"
                                            fill="#5C637E"/>
                                    </g>
                                    <defs>
                                        <clipPath id="clip0_817_464">
                                            <rect width="18.0047" height="18" fill="none"/>
                                        </clipPath>
                                    </defs>
                                </svg>
                                <span>{wp.i18n.__('Blur', 'radio-player')}</span>
                            </Button>
                        </ButtonGroup>

                    </div>

                    <p className="description">
                        {wp.i18n.__('Select the player background type color/ image.', 'radio-player')}
                        <br/>
                        {'blur' === bg_type &&
                            <div>
                                <strong>{wp.i18n.__('Note:', 'radio-player')} </strong> {wp.i18n.__('Blur background will be automatically applied based on the station thumbnail/ artwork.', 'radio-player')}
                            </div>
                        }
                    </p>
                </div>

            </div>

            {!isPro &&
                <Tooltip
                    anchorSelect={`.input-group.disabled`}
                    className={`radio-player-tooltip`}
                    place="right"
                    variant="warning"
                    content={wp.i18n.__('PRO Feature', 'radio-player')}
                />
            }

            {/* Background Image */}
            {'image' === bg_type &&
                <div className="form-group">
                    <h4 className={`form-label`}><span>{wp.i18n.__('Background Image', 'radio-player')}</span></h4>

                    <div className="form-field">

                        <div className={`input-group ${!isPro ? 'disabled' : ''}`}
                             onClick={() => {
                                 if (!isPro) {
                                     showProModal(wp.i18n.__('Upgrade to PRO to customize the player background.', 'radio-player'));
                                 }
                             }}
                        >
                            <Uploader
                                media={bg_image}
                                onChange={media => setEditData(data => ({...data, bg_image: media}))}
                            />
                        </div>

                        <p className="description">{wp.i18n.__('Set the player background image.', 'radio-player')}</p>
                    </div>
                </div>
            }

            {/* Background Color */}
            {'color' === bg_type &&
                <div className="form-group bg-color">
                    <h4 className={`form-label`}>
                        <span>{wp.i18n.__('Background Color', 'radio-player')}</span>
                        {!isPro && <span className="radio-player-pro-icon"><i className={`radio-player-icon icon-pro`}/></span>}
                    </h4>

                    <div className="form-field">

                        <div className={`input-group ${!isPro ? 'disabled' : ''}`}
                             onClick={() => {
                                 if (!isPro) {
                                     showProModal(wp.i18n.__('Upgrade to PRO to customize the player background.', 'radio-player'));
                                 }
                             }}
                        >

                            <RadioGroup
                                label={wp.i18n.__('Background Color Type', 'radio-player')}
                                checked={color_type}
                                onChange={(value) => setEditData(data => ({...data, color_type: value}))}
                                className="radio-player-radio-group"
                            >
                                <Radio value="solid">
                                    <span>{wp.i18n.__('Solid', 'radio-player')}</span>
                                </Radio>
                                <Radio value="gradient">
                                    <span>{wp.i18n.__('Gradient', 'radio-player')}</span>
                                </Radio>
                            </RadioGroup>

                            <p className={`description`}>{wp.i18n.__('Select the background color type solid/ gradient.', 'radio-player')}</p>

                            <div className="form-field-sub">
                                {'solid' === color_type &&
                                    <ColorPickerWrapper
                                        value={bg_color}
                                        onChange={bg_color => {

                                            if (!isPro) {
                                                showProModal(wp.i18n.__('Upgrade to PRO to customize the player background color.', 'radio-player'));
                                                return;
                                            }

                                            setEditData(data => ({...data, bg_color}))
                                        }}
                                        allowReset
                                    />
                                }

                                {'gradient' === color_type &&
                                    <GradientPicker
                                        value={bg_color}
                                        className={`radio-player-gradient-picker ${isPro ? '' : 'disabled'}`}
                                        onChange={bg_color => {
                                            if (!isPro) {
                                                showProModal(wp.i18n.__('Upgrade to PRO to customize the player background color.', 'radio-player'));
                                                return;
                                            }

                                            setEditData(data => ({...data, bg_color}));
                                        }}
                                        gradients={[
                                            {
                                                gradient: 'linear-gradient(-225deg, #E3FDF5 0%, #FFE6FA 100%)',
                                            },
                                            {
                                                gradient: 'linear-gradient(153.63deg, #61E89D 3.43%, #2DCA74 90.47%)',
                                            },
                                            {
                                                gradient: 'linear-gradient(153.63deg, #CE80FF 3.43%, #8A75FF 90.47%)',
                                            },
                                            {
                                                gradient: 'linear-gradient(143.53deg, #F7CCDC 11.16%, #FD80D2 94.07%)',
                                            },
                                            {
                                                gradient: 'linear-gradient(143.53deg, #8D7DFE 11.16%, #6FB9FF 94.07%)',
                                            },
                                            {
                                                gradient: 'linear-gradient(143.38deg, #FFCDA3 4.99%, #FD6C87 94.05%)',
                                            },
                                        ]}
                                    />
                                }

                                <p className="description">{wp.i18n.__('Customize the player background color.', 'radio-player')}</p>
                            </div>

                        </div>
                    </div>

                </div>
            }

            {/* Text Color */}
            <div className="form-group">
                <h4 className={`form-label`}>
                    <span>{wp.i18n.__('Text & Button Color', 'radio-player')}</span>
                    {!isPro && <span className="radio-player-pro-icon"><i className={`radio-player-icon icon-pro`}/></span>}
                </h4>

                <div className="form-field">

                    <div className={`input-group ${!isPro ? 'disabled' : ''}`}
                         onClick={() => {
                             if (!isPro) {
                                 showProModal(wp.i18n.__('Upgrade to PRO to customize the player colors.', 'radio-player'));
                             }
                         }}
                    >
                        <ColorPickerWrapper
                            value={text_color}
                            onChange={text_color => {
                                setEditData(data => ({...data, text_color}))
                            }}
                            allowReset
                        />

                    </div>

                    <p className="description">{wp.i18n.__('Customize the player text and button color.', 'radio-player')}</p>
                </div>
            </div>

            {/* Border Radius */}
            <div className="form-group">
                <h4 className={`form-label`}>

                    <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path
                            d="M19.5207 6.93741L19.4608 6.75765C19.4142 6.6245 19.3676 6.48468 19.3143 6.35153C19.1346 6.47803 18.9947 6.58455 18.8549 6.68442C18.5087 6.89081 18.4821 6.83755 17.7031 6.87083C17.7431 7.03062 17.7497 7.02396 17.803 7.13715C17.8429 7.24367 17.8563 7.28362 17.8829 7.39014C17.9162 7.50332 17.9495 7.61651 17.9894 7.72303C18.0027 7.75632 18.0027 7.76964 18.0161 7.80293C18.3157 8.93475 18.3955 10.0799 18.2158 11.245C17.9961 12.6565 17.4235 13.9148 16.5913 15.0599C16.1452 15.5859 15.6659 16.0852 15.1399 16.5246C15.1133 16.5379 15.08 16.5513 15.0533 16.5712C13.3822 17.8296 11.5047 18.4021 9.40753 18.2623C8.82164 18.2224 8.25573 18.0826 7.68316 17.976C7.4568 17.8961 7.23043 17.8096 7.00407 17.7297C5.79901 17.2437 4.76705 16.518 3.87491 15.5859C3.72178 15.4061 3.56865 15.2264 3.42218 15.0466C2.98276 14.4208 2.58995 13.775 2.30367 13.0626C2.06399 12.4634 1.92417 11.8375 1.79102 11.2117C1.76439 11.0985 1.74441 10.9787 1.7311 10.8589C1.71113 10.7057 1.70447 10.6391 1.70447 10.5193C1.69781 10.3861 1.69781 10.2597 1.70447 10.1265C1.71778 9.84687 1.7311 9.56724 1.75773 9.29427C2.03736 6.39148 3.46878 4.20772 5.96545 2.73635C7.27038 1.9707 8.69515 1.66444 10.1798 1.65779C10.4062 1.65779 10.6392 1.67776 10.8589 1.71105H10.8656C10.8989 1.71771 10.9322 1.71771 10.9655 1.71771C11.5247 1.73768 11.9042 1.78428 12.337 1.98402C12.7631 2.19041 12.7897 2.10386 12.783 1.85086C12.7764 1.5446 12.7697 1.23834 12.8496 0.932087C12.8962 0.758984 13.1359 0.592539 13.0227 0.459384C12.9162 0.332886 12.6632 0.326228 12.4768 0.272965C12.4368 0.25965 12.3902 0.246334 12.3503 0.246334C12.0174 0.159783 12.1905 0.19973 11.7444 0.119836C11.4848 0.0732318 11.558 0.0798896 11.4581 0.0732318C10.2664 -0.0665818 9.0813 -0.059924 7.90287 0.206388C6.06532 0.625828 4.47411 1.44474 3.13589 2.74301C2.94947 2.90279 2.76306 3.06258 2.61658 3.26231C2.59661 3.28894 2.55666 3.33555 2.53669 3.36218C2.52338 3.38215 2.5034 3.39547 2.49009 3.41544C2.43682 3.47536 2.38356 3.54194 2.33696 3.60186C2.19714 3.71504 2.09728 3.86151 2.00407 4.01464C1.98409 4.02796 1.96412 4.04127 1.95081 4.05459C0.2797 6.3715 -0.319501 8.95472 0.15986 11.7643C0.452803 13.4687 1.15187 15.0067 2.28369 16.3249C2.39022 16.5047 2.53003 16.6445 2.68316 16.7776C2.90287 17.0772 3.1825 17.3102 3.45547 17.5433C4.18117 18.1758 4.97344 18.7084 5.85893 19.0879C6.08529 19.1811 6.31166 19.2743 6.53136 19.3742C7.41019 19.7137 8.32231 19.8935 9.26106 19.9667C10.4928 20.0599 11.6978 19.9467 12.8762 19.5539C12.8762 19.5406 12.8696 19.5273 12.8696 19.514C12.8696 19.5273 12.8762 19.5406 12.8762 19.5539C12.9628 19.534 13.0493 19.5206 13.1292 19.494C14.6139 19.0146 15.9122 18.2157 17.0174 17.1172C18.1492 15.992 19.0014 14.6937 19.4674 13.1691C19.4941 13.0826 19.5274 13.0027 19.5607 12.9161C19.6272 12.7497 19.6472 12.5766 19.7005 12.4035C20.1532 10.5459 20.0866 8.72836 19.5207 6.93741ZM13.056 18.9947C13.0493 18.9814 13.0427 18.968 13.036 18.9547C13.0493 18.968 13.0493 18.9814 13.056 18.9947Z"
                            fill="#5C637E"/>
                        <path
                            d="M13.0028 5.67244C12.7032 5.97869 12.4036 6.28495 12.104 6.59121C10.5461 6.59121 8.98154 6.58455 7.42362 6.59121C6.92429 6.59121 6.60471 6.87084 6.59806 7.34354C6.5914 8.8016 6.60471 10.2663 6.60471 11.7244C6.60471 11.9707 6.60471 12.2104 6.60471 12.4567C6.58474 13.036 6.85771 13.3822 7.37702 13.3822C8.6886 13.3888 10.0068 13.3755 11.3184 13.3755C11.365 13.3755 11.405 13.3755 11.4516 13.3755C11.5381 13.3755 11.6313 13.3755 11.7179 13.3755C11.9975 13.3755 12.2772 13.3755 12.5501 13.3755C13.0961 13.3688 13.389 13.0892 13.389 12.5433C13.389 11.012 13.3957 9.48735 13.389 7.95606C13.389 7.70306 13.4822 7.54993 13.7419 7.35686C14.1147 7.07723 14.4542 6.73768 14.7072 6.35153C14.8937 6.0719 15.0601 5.96538 15.3331 5.96538C16.2385 5.97869 17.1507 5.97204 18.0561 5.97204C18.5355 5.97204 18.7485 5.7723 18.7552 5.32623C18.7618 4.04128 18.7618 2.75632 18.7552 1.47137C18.7552 1.06524 18.5488 0.83888 18.1493 0.83888C16.8444 0.825564 15.5328 0.832222 14.2279 0.83888C13.9616 0.83888 13.8484 0.905458 13.6886 1.1984C13.6553 1.25166 13.6353 1.31824 13.622 1.37816C13.6021 1.47137 13.6021 1.53795 13.6021 1.6245C13.6021 2.62983 13.5954 3.64181 13.6021 4.64714C13.6021 4.89347 13.5754 5.09987 13.3823 5.26631C13.2625 5.39947 13.1293 5.53928 13.0028 5.67244ZM8.8617 10.253C8.8617 10.213 8.8617 10.1731 8.8617 10.1265C8.8617 10.1731 8.8617 10.2197 8.8617 10.253ZM9.97356 9.34753C9.98021 9.3209 9.98687 9.28762 10.0002 9.26098C9.97355 9.25433 9.95358 9.25433 9.93361 9.26764C9.95358 9.26098 9.98021 9.25433 10.0002 9.26098C9.98687 9.28762 9.98021 9.3209 9.97356 9.34753ZM9.99353 9.26098C10.0335 9.24101 10.0668 9.20772 10.1001 9.17443C10.0734 9.20772 10.0401 9.24101 9.99353 9.26098ZM13.5954 5.39947C13.6021 5.51931 13.602 5.63915 13.5754 5.73236C13.602 5.63249 13.5954 5.51931 13.5954 5.39947ZM13.5288 5.31957C13.5488 5.3462 13.5754 5.37283 13.5954 5.39281C13.5688 5.37283 13.5488 5.3462 13.5288 5.31957C13.5022 5.27297 13.4556 5.26631 13.409 5.26631C13.4556 5.26631 13.4955 5.27963 13.5288 5.31957Z"
                            fill="#5C637E"/>
                    </svg>

                    <span>{wp.i18n.__('Border Radius', 'radio-player')}</span>
                </h4>

                <div className="form-field">
                    <RangeControl
                        value={parseInt(border_radius)}
                        onChange={border_radius => setEditData(data => ({...data, border_radius}))}
                        min={2}
                        max={100}
                        allowReset={true}
                        resetFallbackValue={5}
                    />
                    <p className="description">{wp.i18n.__('Set border radius in pixels.', 'radio-player')}</p>
                </div>
            </div>

            {/* Box Shadow */}
            <div className="form-group box-shadow">
                <h4 className={`form-label`}>
                    <span>{wp.i18n.__('Box Shadow', 'radio-player')}</span>
                </h4>

                <div className="form-field">

                    <FormToggle
                        checked={box_shadow}
                        onChange={() => setEditData(data => ({...data, box_shadow: !box_shadow}))}
                    />

                    {box_shadow &&
                        <>
                            <RangeControl
                                label={wp.i18n.__('Horizontal', 'radio-player')}
                                value={parseInt(box_shadow_h)}
                                onChange={box_shadow_h => setEditData(data => ({...data, box_shadow_h}))}
                                min={-50}
                                max={50}
                                step={1}
                                allowReset={true}
                                resetFallbackValue={0}
                            />

                            <RangeControl
                                label={wp.i18n.__('Vertical', 'radio-player')}
                                value={parseInt(box_shadow_v)}
                                onChange={box_shadow_v => setEditData(data => ({...data, box_shadow_v}))}
                                min={-50}
                                max={50}
                                step={1}
                                allowReset={true}
                                resetFallbackValue={0}
                            />

                            <RangeControl
                                label={wp.i18n.__('Blur', 'radio-player')}
                                value={parseInt(box_shadow_blur)}
                                onChange={box_shadow_blur => setEditData(data => ({...data, box_shadow_blur}))}
                                min={0}
                                max={50}
                                step={1}
                                allowReset={true}
                                resetFallbackValue={0}
                            />

                            <RangeControl
                                label={wp.i18n.__('Spread', 'radio-player')}
                                value={parseInt(box_shadow_spread)}
                                onChange={box_shadow_spread => setEditData(data => ({...data, box_shadow_spread}))}
                                min={-50}
                                max={50}
                                step={1}
                                allowReset={true}
                                resetFallbackValue={0}
                            />

                            <h4 className={`shadow-color-label`}>{wp.i18n.__('Shadow Color', 'radio-player')}</h4>

                            <ColorPickerWrapper
                                value={box_shadow_color}
                                onChange={box_shadow_color => setEditData(data => ({...data, box_shadow_color}))}
                                allowReset
                            />

                            <SelectControl
                                label={wp.i18n.__('Shadow Type', 'radio-player')}
                                value={box_shadow_type}
                                onChange={box_shadow_type => setEditData(data => ({...data, box_shadow_type}))}
                                options={[
                                    {value: 'inset', label: wp.i18n.__('Inset', 'radio-player')},
                                    {value: '', label: wp.i18n.__('Outset', 'radio-player')},
                                ]}
                                style={{marginTop: 10}}
                                __next40pxDefaultSize={true}
                                __nextHasNoMarginBottom={true}
                            />

                        </>
                    }

                    <p className="description">{wp.i18n.__('Set Player Box Shadow.', 'radio-player')}</p>
                </div>
            </div>


        </div>
    )
}