import {Tooltip} from "react-tooltip";

import AppContext from "../../contexts/AppContext";
import Stations from "./Form/Stations";
import Skins from "./Form/Skins";
import Controls from "./Form/Controls";
import Appearance from "./Form/Appearance";
import Schedules from "./Form/Schedules";
import {base64Encode, useMounted} from "../../includes/functions";
import metadataHandler from "../../includes/metadata-handler";
import Player from "../../player/Player";

const {Spinner} = wp.components;

const {useContext, useState, useEffect} = React;

export default function Form() {
    const context = useContext(AppContext);

    const {
        editData,
        updatePlayer,
        updating,
        isDirty
    } = context;

    const [tab, setTab] = useState('stations');

    const [previewDevice, setPreviewDevice] = useState(editData.width_device || 'desktop');

    const tabs = {
        stations: wp.i18n.__('Stations', 'radio-player'),
        skins: wp.i18n.__('Skins', 'radio-player'),
        controls: wp.i18n.__('Controls', 'radio-player'),
        appearance: wp.i18n.__('Appearance', 'radio-player'),
        schedules: wp.i18n.__('Schedules', 'radio-player'),
    }

    useEffect(() => {
        if (editData.id > 0) return;
        document.getElementById('player-title').focus();


    }, []);

    useEffect(() => {
        setPreviewDevice(editData.width_device || 'desktop');
    }, [editData.width_device]);

    const isMounted = useMounted();

    useEffect(() => {

        if (!isMounted) return;

        // Metadata Handler
        metadataHandler.init();

        return (() => {
            if (window.radioPlayerMetadataInterval) {
                clearInterval(window.radioPlayerMetadataInterval);
            }

        });

    }, [editData.stations]);

    return (
        <div className="radio-player-form">

            <div className="radio-player-form-main">

                <div className="radio-player-form-settings">
                    <div className="settings-tabs">
                        {Object.keys(tabs).map((tabKey, index) => (
                            <button key={index} type={'button'}
                                    className={`radio-player-form-tab ${tab === tabKey ? 'active' : ''} tab-${tabKey}`}
                                    onClick={() => setTab(tabKey)}
                            >
                                <span className={`tab-icon`}/>
                                {tabs[tabKey]}
                            </button>
                        ))}
                    </div>

                    {'stations' === tab && <Stations/>}
                    {'skins' === tab && <Skins/>}
                    {'controls' === tab && <Controls/>}
                    {'appearance' === tab && <Appearance/>}
                    {'schedules' === tab && <Schedules/>}

                    <div className="settings-footer">

                        {/* Back */}
                        <button
                            data-tooltip-content={wp.i18n.__('Previous step', "radio-player")}
                            data-tooltip-id={'previous'}
                            type={'button'}
                            disabled={'stations' === tab}
                            className={`radio-player-btn ${'stations' === tab ? 'disabled' : ''}`}
                            onClick={() => {
                                const tabKeys = Object.keys(tabs);
                                const tabIndex = tabKeys.findIndex(item => item === tab);
                                setTab(tabKeys[tabIndex - 1]);
                            }}
                        >
                            <i className={`dashicons dashicons-arrow-left-alt2`}></i>
                            <span>{wp.i18n.__("Back", "radio-player")}</span>

                            <Tooltip
                                id="previous"
                                effect="solid"
                                place="top"
                            />
                        </button>

                        {/* Save Changes */}
                        <button type={'button'} className={`radio-player-btn ${isDirty ? '' : 'disabled'}`}
                                onClick={() => updatePlayer(editData)}>
                            {updating ? <Spinner/> : <i className={`dashicons dashicons-saved`}></i>}
                            <span>{isDirty ? wp.i18n.__("Save", "radio-player") : wp.i18n.__("Saved", "radio-player")}</span>
                        </button>

                        {/* Next Tab */}
                        <button
                            data-tooltip-content={wp.i18n.__('Next step', "radio-player")}
                            data-tooltip-id={'next'}
                            type={'button'}
                            disabled={'schedules' === tab}
                            className={`radio-player-btn ${'schedules' === tab ? 'disabled' : ''}`}
                            onClick={() => {
                                const tabKeys = Object.keys(tabs);
                                const tabIndex = tabKeys.findIndex(item => item === tab);
                                setTab(tabKeys[tabIndex + 1]);
                            }}>
                            <span>{wp.i18n.__("Next", "radio-player")}</span>
                            <i className={`dashicons dashicons-arrow-right-alt2`}></i>

                            <Tooltip
                                id="next"
                                effect="solid"
                                place="top"
                            />
                        </button>
                    </div>
                </div>

                <div className={`radio-player-form-sidebar`}>
                    <div className={`radio-player-form-preview`}>
                        <div className={`preview-browser device-${previewDevice}`}>
                            <div className="preview-browser-header">

                                <div className="preview-browser-buttons">
                                    <div className="header-button close"></div>
                                    <div className="header-button minimize"></div>
                                    <div className="header-button maximize"></div>
                                </div>

                                <div className="preview-browser-url">
                                    <i className={`dashicons dashicons-visibility`}></i>
                                    <span>{wp.i18n.__('Live Preview', 'radio-player')}</span>
                                </div>

                                <div className="preview-browser-devices">
                                    <i onClick={() => setPreviewDevice('desktop')}
                                       className={`dashicons dashicons-desktop ${'desktop' === previewDevice ? 'active' : ''}`}/>
                                    <i className={`dashicons dashicons-smartphone ${'mobile' === previewDevice ? 'active' : ''}`}
                                       onClick={() => setPreviewDevice('mobile')}/>
                                </div>
                            </div>

                            <div className="browser-content">
                                <div className="radio_player" data-player-type="shortcode"
                                     data-data={base64Encode(JSON.stringify({playerType: 'shortcode', ...editData}))}>

                                    <Player
                                        data={{playerType: 'shortcode', ...editData,}}
                                        previewDevice={previewDevice}
                                    />

                                </div>
                            </div>
                        </div>
                    </div>
                </div>

            </div>

        </div>
    )
}