<?php

defined( 'ABSPATH' ) || exit();

final class Radio_Player {

    /**
     * The single instance of the class.
     *
     * @var Radio_Player
     * @since 1.0.0
     */
    protected static $instance = null;

    /**
     * Radio_Player constructor.
     */
    public function __construct() {

        $this->check_environment();
        $this->includes();
        $this->init_hooks();

        register_activation_hook( RADIO_PLAYER_FILE, array( $this, 'activate' ) );
        do_action( 'radio_player_loaded' );

    }

    public function activate() {

        if ( ! class_exists( 'Radio_Player_Install' ) ) {
            include_once RADIO_PLAYER_INCLUDES . '/class-install.php';
        }

        Radio_Player_Install::activate();
    }

    /**
     * Ensure theme and server variable compatibility
     *
     * @return void
     * @since 1.0.0
     *
     */
    public function check_environment() {
        $min_php = '5.6.0';
        $min_wp  = '5.0';

        if ( version_compare( PHP_VERSION, $min_php, '<' ) ) {
            deactivate_plugins( plugin_basename( RADIO_PLAYER_FILE ) );
            wp_die( sprintf( 'Unsupported PHP version. Minimum required PHP version: %s', esc_html( $min_php ) ) );
        }

        if ( version_compare( get_bloginfo( 'version' ), $min_wp, '<' ) ) {
            add_action( 'admin_notices', function () use ( $min_wp ) {

                $message = '
                <div class="notice notice-error is-dismissible">
                    <p>
                        <strong>Radio Player</strong> requires WordPress version <strong>' . esc_html( $min_wp ) . '</strong> or higher. Please update your WordPress.
                    </p>           
                </div>
                ';

                echo wp_kses_post( $message );

            } );

            add_action( 'after_plugin_row_radio-player/plugin.php', function () use ( $min_wp ) {

                $message = '
                <tr class="plugin-update-tr active">
                    <td class="plugin-update colspanchange" colspan="4">
                        <div class="update-message notice inline notice-error notice-alt">
                            <p>
                                <strong>Radio Player</strong> requires WordPress version <strong>' . esc_html( $min_wp ) . '</strong> or higher. Please update your WordPress.
                            </p>           
                        </div>
                    </td>
                </tr>
                ';

                echo wp_kses_post( $message );

            } );
        }
    }

    /**
     * Include required core files used in admin and on the frontend.
     *
     * @return void
     * @since 1.0.0
     *
     */
    public function includes() {

        // Core includes
        include_once RADIO_PLAYER_INCLUDES . '/functions.php';
        include_once RADIO_PLAYER_INCLUDES . '/class-enqueue.php';
        include_once RADIO_PLAYER_INCLUDES . '/class-hooks.php';
        include_once RADIO_PLAYER_INCLUDES . '/class-shortcode.php';
        include_once RADIO_PLAYER_INCLUDES . '/class-ajax.php';
        require_once RADIO_PLAYER_INCLUDES . '/class-player-locations.php';
        require_once RADIO_PLAYER_INCLUDES . '/class-stream-data.php';

        // Elementor
        require_once RADIO_PLAYER_PATH . '/elementor/class-elementor.php';

        // Block
        include_once RADIO_PLAYER_PATH . '/block/class-block.php';

        // Admin includes
        if ( is_admin() ) {
            include_once RADIO_PLAYER_INCLUDES . '/class-admin.php';
        }

    }

    /**
     * Hook into actions and filters.
     *
     * @since 1.0.0
     */
    private function init_hooks() {
        add_action( 'admin_notices', [ $this, 'print_notices' ], 15 );
    }

    public function add_notice( $class, $message ) {

        $notices = get_option( sanitize_key( 'radio_player_notices' ), [] );
        if ( is_string( $message ) && is_string( $class ) && ! wp_list_filter( $notices, array( 'message' => $message ) ) ) {

            $notices[] = array(
                    'message' => $message,
                    'class'   => $class,
            );

            update_option( sanitize_key( 'radio_player_notices' ), $notices );
        }

    }

    /**
     * Prince admin notice
     *
     * @return void
     * @since 1.0.0
     *
     */
    public function print_notices() {
        $notices = get_option( sanitize_key( 'radio_player_notices' ), [] );

        foreach ( $notices as $notice ) { ?>
            <div class="notice notice-large is-dismissible radio-player-admin-notice notice-<?php echo esc_attr( $notice['class'] ); ?>">
                <?php

                echo wp_kses( $notice['message'], [
                        'div'    => [
                                'class' => [],
                        ],
                        'a'      => [
                                'href'   => [],
                                'class'  => [],
                                'target' => [],
                                'rel'    => [],
                        ],
                        'strong' => [
                                'class' => [],
                        ],
                        'br'     => [],
                        'em'     => [],
                        'img'    => [
                                'src'    => [],
                                'alt'    => [],
                                'width'  => [],
                                'height' => [],
                        ],
                        'h4'     => [
                                'class' => [],
                        ],
                        'h3'     => [
                                'class' => [],
                        ],
                        'ul'     => [
                                'class' => [],
                        ],
                        'ol'     => [],
                        'li'     => [],
                        'span'   => [
                                'class' => [],
                        ],
                        'i'   => [
                                'class' => [],
                        ],
                        'style'  => [],
                        'script' => [],
                        'p'      => [
                                'class' => [],
                        ],
                ] );

                ?>
            </div>
            <?php
            update_option( sanitize_key( 'radio_player_notices' ), [] );
        }
    }

    public function is_pro_active() {

        if ( ! function_exists( 'rp_fs' ) ) {
            return false;
        }

        return rp_fs()->can_use_premium_code__premium_only();
    }

    /**
     * Main Radio_Player Instance.
     *
     * Ensures only one instance of Radio_Player is loaded or can be loaded.
     *
     * @return Radio_Player - Main instance.
     * @since 1.0.0
     * @static
     */
    public static function instance() {

        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }

        return self::$instance;
    }
}

// kickoff radio_player
if ( ! function_exists( 'radio_player' ) ) {
    function radio_player() {
        return Radio_Player::instance();
    }
}

radio_player();