<?php
class All_Reviews_List_Table extends WP_List_Table{
    private $plugin_name;
    private $title_length;

    /**
     * The wp nonce of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $ays_quiz_nonce
     */
    public $ays_quiz_nonce;

    private $current_user_can_edit;


    /** Class constructor */
    public function __construct($plugin_name) {
        $this->plugin_name = $plugin_name;
        $this->title_length = Quiz_Maker_Admin::get_listtables_title_length('quiz_reviews');
        $this->current_user_can_edit = Quiz_Maker_Admin::quiz_maker_capabilities();

        parent::__construct( array(
            'singular' => __( 'Review', 'quiz-maker' ), //singular name of the listed records
            'plural'   => __( 'Reviews', 'quiz-maker' ), //plural name of the listed records
            'ajax'     => false //does this table support ajax?
        ) );
        add_action( 'admin_notices', array( $this, 'reviews_notices' ) );

        $this->ays_quiz_nonce = wp_create_nonce('ays_quiz_admin_all_reviews_list_table_nonce');

        if( empty($this->ays_quiz_nonce) ){
            add_action('init', function () {
                $this->ays_quiz_nonce = wp_create_nonce('ays_quiz_admin_all_reviews_list_table_nonce');
            }, 1);
        }

    }

    /**
     * Override of table nav to avoid breaking with bulk actions & according nonce field
     */
    public function display_tablenav( $which ) {
        ?>
        <div class="tablenav <?php echo esc_attr( $which ); ?>">

            <div class="alignleft actions">
                <?php $this->bulk_actions( $which ); ?>
            </div>

            <?php
            $this->extra_tablenav( $which );
            $this->pagination( $which );
            ?>
            <br class="clear" />
        </div>
        <?php
    }

    /**
     * Disables the views for 'side' context as there's not enough free space in the UI
     * Only displays them on screen/browser refresh. Else we'd have to do this via an AJAX DB update.
     *
     * @see WP_List_Table::extra_tablenav()
     */
    public function extra_tablenav($which) {
        global $wpdb;

        // Run a security check.
        if (empty($this->ays_quiz_nonce) || ! wp_verify_nonce( $this->ays_quiz_nonce, 'ays_quiz_admin_all_reviews_list_table_nonce' ) ) {
            // This nonce is not valid.
            wp_die('Nonce verification failed!');
        }

        $quizes_table  = esc_sql( $wpdb->prefix . "aysquiz_quizes" );

        $titles_sql = "SELECT {$quizes_table}.title,
                              {$quizes_table}.id 
                       FROM {$wpdb->prefix}aysquiz_quizes ORDER BY {$quizes_table}.title ASC";
        $quiz_titles = $wpdb->get_results($titles_sql);

        $quiz_id = null;

        if( isset( $_GET['filterby'] )){
            $quiz_id = absint( $_GET['filterby'] );
        }

        $quiz_reviews = array(
            "1" => 1,
            "2" => 2,
            "3" => 3,
            "4" => 4,
            "5" => 5,
        );

        $review_key = null;

        if( isset( $_GET['filterbyreview'] )){
            $review_key = absint( $_GET['filterbyreview'] );
        }

        $quiz_comments = array(
            "with_answer"   => __("With reviews", 'quiz-maker'),
            "without_answer" => __("Without reviews", 'quiz-maker'),
        );

        $comment_key = null;

        if( isset( $_GET['filterbycomment'] )){
            $comment_key = sanitize_text_field( wp_unslash($_GET['filterbycomment']) );
        }

        ?>

        <div id="quiz-filter-div-<?php echo esc_attr( $which ); ?>" class="alignleft actions bulkactions ays-quiz-review-filter-main-div">
            <select name="filterby-<?php echo esc_attr( $which ); ?>" id="bulk-action-quiz-selector-<?php echo esc_attr( $which ); ?>">
                <option value=""><?php echo esc_html__('Select Quiz','quiz-maker')?></option>
                <?php
                    foreach($quiz_titles as $key => $q_title){
                        $selected = "";
                        if($quiz_id === intval($q_title->id)){
                            $selected = "selected";
                        }
                        echo "<option ".esc_attr($selected)." value='".esc_attr($q_title->id)."'>".esc_html($q_title->title)."</option>";
                    }
                ?>
            </select>
            <select name="filterbyreview-<?php echo esc_attr( $which ); ?>" id="bulk-action-quiz-rate-selector-<?php echo esc_attr( $which ); ?>">
                <option value=""><?php echo esc_html__('Select Rate','quiz-maker')?></option>
                <?php
                    foreach($quiz_reviews as $key => $review) {
                        $selected = "";
                        if( $review_key === absint($review) ) {
                            $selected = "selected";
                        }
                        echo "<option ".esc_attr($selected)." value='".esc_attr( $key )."'>".esc_html($review)."</option>";
                    }
                ?>
            </select>

            <select name="filterbycomment-<?php echo esc_attr( $which ); ?>" id="bulk-action-quiz-rate-selector-<?php echo esc_attr( $which ); ?>">
                <option value=""><?php echo esc_html__('With/without reviews','quiz-maker'); ?></option>
                <?php
                    foreach($quiz_comments as $key => $quiz_comment) {
                        $selected = "";
                        if( $comment_key === sanitize_text_field($key) ) {
                            $selected = "selected";
                        }
                        echo "<option ".esc_attr($selected)." value='".esc_attr( $key )."'>".esc_html($quiz_comment)."</option>";
                    }
                ?>
            </select>
            <input type="button" id="doaction-quiz-<?php echo esc_attr( $which ); ?>" class="ays-quiz-question-tab-all-filter-button-<?php echo esc_attr( $which ); ?> button" value="<?php echo esc_html__( "Filter", 'quiz-maker' ); ?>">
        </div>

        <a style="" href="?page=<?php echo !empty($_REQUEST['page']) ? esc_attr( $_REQUEST['page'] ) : ''; ?>" class="button"><?php echo esc_html__( "Clear filters", 'quiz-maker' ); ?></a>
        <?php
    }

    protected function get_views() {

        // Run a security check.
        if (empty($this->ays_quiz_nonce) || ! wp_verify_nonce( $this->ays_quiz_nonce, 'ays_quiz_admin_all_reviews_list_table_nonce' ) ) {
            // This nonce is not valid.
            wp_die( esc_html__( 'Nonce verification failed!', 'quiz-maker' ) );
        }

        if( !is_user_logged_in()){
            wp_die(  esc_html__( 'Something went wrong', 'quiz-maker' ) );
        }

        // Verify unauthorized requests
        if( !current_user_can( $this->current_user_can_edit ) ){
            wp_die(  esc_html__( 'Something went wrong', 'quiz-maker' ) );
        }

        $all_count = $this->all_record_count();
        $filter_by_quiz = (isset($_GET['filterby']) && $_GET['filterby'] > 0) ? esc_attr($_GET['filterby']) : '';
        $avg_score = '';
        if($filter_by_quiz){
            $avg_score = $this->avg_review($filter_by_quiz);
            if($avg_score){
                $avg_score = "<span>AVG ( ".round($avg_score , 1)." ) </span>";
            }
        }
        
        $selected_all = "";
        $selected_0 = "";
        $selected_1 = "";
        if( isset( $_REQUEST['fstatus'] ) && is_numeric( $_REQUEST['fstatus'] ) && ! is_null( $_REQUEST['fstatus'] ) ){

            $fstatus  = absint( $_REQUEST['fstatus'] );

            switch( $fstatus ){
                case 0:
                    $selected_0 = " style='font-weight:bold;' ";
                    break;
                case 1:
                    $selected_1 = " style='font-weight:bold;' ";
                    break;
                default:
                    $selected_all = " style='font-weight:bold;' ";
                    break;
            }
        }else{
            $selected_all = " style='font-weight:bold;' ";
        }

        $status_links = array(
            "all" => "<a ".$selected_all." href='?page=". ( !empty($_REQUEST['page']) ? esc_attr( $_REQUEST['page'] ) : '' ) ."'>". __( 'All', 'quiz-maker' )." (".$all_count.")</a> ".$avg_score,
        );
        return $status_links;
    }

    /**
     * Retrieve customers data from the database
     *
     * @param int $per_page
     * @param int $page_number
     *
     * @return mixed
     */
    public function get_reviews( $per_page = 50, $page_number = 1 ) {
        global $wpdb;

        // Run a security check.
        if (empty($this->ays_quiz_nonce) || ! wp_verify_nonce( $this->ays_quiz_nonce, 'ays_quiz_admin_all_reviews_list_table_nonce' ) ) {
            // This nonce is not valid.
            wp_die('Nonce verification failed!');
        }

        if( !is_user_logged_in()){
            return array();
        }

        // Verify unauthorized requests
        if( !current_user_can( $this->current_user_can_edit ) ){
            return array();
        }

        $rates_table  = esc_sql( $wpdb->prefix . "aysquiz_rates" );

        $sql = "SELECT * FROM {$rates_table}";

        $sql .= $this->get_where_condition();

        if ( ! empty( $_REQUEST['orderby'] ) ) {

            $order_by  = ( isset( $_REQUEST['orderby'] ) && sanitize_text_field( wp_unslash($_REQUEST['orderby']) ) != '' ) ? sanitize_text_field( wp_unslash($_REQUEST['orderby']) ) : 'rate_date';
            $order_by .= ( ! empty( $_REQUEST['order'] ) && strtolower( $_REQUEST['order'] ) == 'asc' ) ? ' ASC' : ' DESC';

            $sql_orderby = sanitize_sql_orderby($order_by);

            if ( $sql_orderby ) {
                $sql .= ' ORDER BY ' . $sql_orderby;
            } else {
                $sql .= ' ORDER BY rate_date DESC';
            }

        }
        else{
            $sql .= ' ORDER BY rate_date DESC';
        }

        $sql .= " LIMIT $per_page";
        $sql .= ' OFFSET ' . ( $page_number - 1 ) * $per_page;


        $result = $wpdb->get_results( $sql, 'ARRAY_A' );

        return $result;
    }

    public function get_where_condition(){
        global $wpdb;

        // Run a security check.
        if (empty($this->ays_quiz_nonce) || ! wp_verify_nonce( $this->ays_quiz_nonce, 'ays_quiz_admin_all_reviews_list_table_nonce' ) ) {
            // This nonce is not valid.
            wp_die( esc_html__( 'Nonce verification failed!', 'quiz-maker' ) );
        }

        if( !is_user_logged_in()){
            return '';
        }

        // Verify unauthorized requests
        if( !current_user_can( $this->current_user_can_edit ) ){
            return '';
        }

        $where = array();
        $sql = '';

        $search = ( isset( $_REQUEST['s'] ) ) ? esc_sql( $wpdb->esc_like( sanitize_text_field( wp_unslash($_REQUEST['s']) ) ) ) : false;
        if( $search ){
            $s = array();
            $s[] = sprintf( " `id` LIKE '%%%s%%' ", esc_sql( $wpdb->esc_like( $search ) ) );
            $s[] = sprintf( " `user_name` LIKE '%%%s%%' ", esc_sql( $wpdb->esc_like( $search ) ) );
            $s[] = sprintf( " `user_email` LIKE '%%%s%%' ", esc_sql( $wpdb->esc_like( $search ) ) );
            $s[] = sprintf( " `user_id` LIKE '%%%s%%' ", esc_sql( $wpdb->esc_like( $search ) ) );
            $s[] = sprintf( " `review` LIKE '%%%s%%' ", esc_sql( $wpdb->esc_like( $search ) ) );

            $where[] = ' ( ' . implode(' OR ', $s) . ' ) ';
        }

        if( isset( $_REQUEST['filterby'] ) && intval( $_REQUEST['filterby'] ) > 0){
            $quiz_id = intval( $_REQUEST['filterby'] );
            $where[] = ' `quiz_id` = '. $quiz_id .' ';
        }

        if( isset( $_REQUEST['filterbyreview'] ) && intval( $_REQUEST['filterbyreview'] ) > 0){
            $review_key = intval( $_REQUEST['filterbyreview'] );
            $where[] = ' `score` = '. $review_key .' ';
        }

        if( isset( $_REQUEST['filterbycomment'] ) && sanitize_text_field( wp_unslash($_REQUEST['filterbycomment']) ) != ""){
            $comment_key = sanitize_text_field( wp_unslash($_REQUEST['filterbycomment']) );

            switch ( $comment_key ) {
                case 'with_answer':
                    $where[] = ' `review` != "" ';
                    break;
                case 'without_answer':
                default:
                    $where[] = ' `review` = "" ';
                    break;
            }
        }

        if( ! empty($where) ){
            $sql = " WHERE " . implode( " AND ", $where );
        }
        return $sql;
    }

    /**
     * Delete a customer record.
     *
     * @param int $id customer ID
     */
    public function delete_reviews( $id ) {
        global $wpdb;

        // Run a security check.
        if (empty($this->ays_quiz_nonce) || ! wp_verify_nonce( $this->ays_quiz_nonce, 'ays_quiz_admin_all_reviews_list_table_nonce' ) ) {
            // This nonce is not valid.
            wp_die( esc_html__( 'Nonce verification failed!', 'quiz-maker' ) );
        }

        if( !is_user_logged_in()){
            wp_die(  esc_html__( 'Something went wrong', 'quiz-maker' ) );
        }

        // Verify unauthorized requests
        if( !current_user_can( $this->current_user_can_edit ) ){
            wp_die(  esc_html__( 'Something went wrong', 'quiz-maker' ) );
        }

        $rates_table  = esc_sql( $wpdb->prefix . "aysquiz_rates" );

        $id = ( isset( $id ) && $id != '' ) ? absint( $id ) : null;

        if ( ! is_null( $id ) && $id > 0 ) {
            $wpdb->delete(
                $rates_table,
                array( 'id' => $id ),
                array( '%d' )
            );
        }
    }

    /**
     * Delete a customer review only.
     *
     * @param int $id customer ID
     */
    public function delete_only_reviews( $id ) {
        global $wpdb;

        // Run a security check.
        if (empty($this->ays_quiz_nonce) || ! wp_verify_nonce( $this->ays_quiz_nonce, 'ays_quiz_admin_all_reviews_list_table_nonce' ) ) {
            // This nonce is not valid.
            wp_die( esc_html__( 'Nonce verification failed!', 'quiz-maker' ) );
        }

        if( !is_user_logged_in()){
            wp_die(  esc_html__( 'Something went wrong', 'quiz-maker' ) );
        }

        // Verify unauthorized requests
        if( !current_user_can( $this->current_user_can_edit ) ){
            wp_die(  esc_html__( 'Something went wrong', 'quiz-maker' ) );
        }

        $rates_table  = esc_sql( $wpdb->prefix . "aysquiz_rates" );

        $id = ( isset( $id ) && $id != '' ) ? absint( $id ) : null;

        if ( ! is_null( $id ) && $id > 0 ) {
            $rates_result = $wpdb->update(
                $rates_table,
                array(
                    'review' => "",

                ),
                array( 'id' => $id ),
                array( '%s' ),
                array( '%d' )
            );
        }
    }


    /**
     * Returns the count of records in the database.
     *
     * @return null|string
     */
    public function record_count() {
        global $wpdb;

        // Run a security check.
        if (empty($this->ays_quiz_nonce) || ! wp_verify_nonce( $this->ays_quiz_nonce, 'ays_quiz_admin_all_reviews_list_table_nonce' ) ) {
            // This nonce is not valid.
            wp_die( esc_html__( 'Nonce verification failed!', 'quiz-maker' ) );
        }

        $rates_table  = esc_sql( $wpdb->prefix . "aysquiz_rates" );

        $sql = "SELECT COUNT(*) FROM {$rates_table}";
        $sql .= $this->get_where_condition();
        return $wpdb->get_var( $sql );
    }

    public function all_record_count() {
        global $wpdb;

        // Run a security check.
        if (empty($this->ays_quiz_nonce) || ! wp_verify_nonce( $this->ays_quiz_nonce, 'ays_quiz_admin_all_reviews_list_table_nonce' ) ) {
            // This nonce is not valid.
            wp_die( esc_html__( 'Nonce verification failed!', 'quiz-maker' ) );
        }

        $rates_table  = esc_sql( $wpdb->prefix . "aysquiz_rates" );

        $sql = "SELECT COUNT(*) FROM {$rates_table}";
        $sql .= $this->get_where_condition();

        return $wpdb->get_var( $sql );
    }

    public function avg_review($quiz_id) {
        global $wpdb;

        // Run a security check.
        if (empty($this->ays_quiz_nonce) || ! wp_verify_nonce( $this->ays_quiz_nonce, 'ays_quiz_admin_all_reviews_list_table_nonce' ) ) {
            // This nonce is not valid.
            wp_die( esc_html__( 'Nonce verification failed!', 'quiz-maker' ) );
        }

        $rates_table  = esc_sql( $wpdb->prefix . "aysquiz_rates" );

        $sql = "SELECT AVG(score) as avg_score FROM {$rates_table} WHERE quiz_id = %s";
        $sql = $wpdb->prepare($sql , $quiz_id);
        return $wpdb->get_var( $sql );  
    }

    public function get_quiz_by_id( $id ){
        global $wpdb;

        // Run a security check.
        if (empty($this->ays_quiz_nonce) || ! wp_verify_nonce( $this->ays_quiz_nonce, 'ays_quiz_admin_all_reviews_list_table_nonce' ) ) {
            // This nonce is not valid.
            wp_die( esc_html__( 'Nonce verification failed!', 'quiz-maker' ) );
        }

        $sql = "SELECT * FROM {$wpdb->prefix}aysquiz_quizes WHERE id=" . absint( intval( $id ) );

        $result = $wpdb->get_row($sql, 'ARRAY_A');

        return $result;
    }

    /** Text displayed when no customer data is available */
    public function no_items() {
        echo esc_html__( 'There are no reviews yet.', 'quiz-maker' );
    }


    /**
     * Render a column when no column specific method exist.
     *
     * @param array $item
     * @param string $column_name
     *
     * @return mixed
     */
    public function column_default( $item, $column_name ) {
        switch ( $column_name ) {
            case 'user_id':
            case 'quiz_id':
            case 'user_ip':
            case 'user_name':
            case 'user_email':
            case 'rate_date':
            case 'id':
            case 'score':
            case 'review':
                return $item[ $column_name ];
                break;
            default:
                return print_r( $item, true ); //Show the whole array for troubleshooting purposes
        }
    }

    /**
     * Render the bulk edit checkbox
     *
     * @param array $item
     *
     * @return string
     */
    function column_cb( $item ) {
        return sprintf(
            '<input type="checkbox" class="ays_result_delete" name="bulk-delete[]" value="%s" />', esc_attr( $item['id'] )
        );
    }


    /**
     * Method for name column
     *
     * @param array $item an array of DB data
     *
     * @return string
     */
    function column_user_id( $item ) {

        // Run a security check.
        if (empty($this->ays_quiz_nonce) || ! wp_verify_nonce( $this->ays_quiz_nonce, 'ays_quiz_admin_all_reviews_list_table_nonce' ) ) {
            // This nonce is not valid.
            wp_die( esc_html__( 'Nonce verification failed!', 'quiz-maker' ) );
        }

        $delete_nonce = wp_create_nonce( $this->plugin_name . '-delete-result' );
        $user_id = intval($item['user_id']);

        if($user_id == 0){
            $name = __( "Guest" , 'quiz-maker' );
        }else{
            $name = '';
            $user = get_userdata($user_id);
            if ($user !== false) {
                $name = $user->data->display_name;
            } else {
                $name = "<span style='color: red;'>" . __( "Deleted user", 'quiz-maker' ) . "</span>";
            }
        }

        $actions = array(
            'delete' => sprintf( '<a class="ays_confirm_del" data-message="this review" href="?page=%s&action=%s&result=%s&_wpnonce=%s">Delete</a>', esc_attr( $_REQUEST['page'] ), 'delete', absint( $item['id'] ), $delete_nonce )
        );

        return $name . $this->row_actions( $actions );

    }

    /**
     * Method for name column
     *
     * @param array $item an array of DB data
     *
     * @return string
     */
    function column_quiz_id( $item ) {
        $quiz_id = intval( $item['quiz_id'] );
        $quiz = $this->get_quiz_by_id( $quiz_id );

        $quiz_title = (isset( $quiz['title'] ) && $quiz['title'] != "") ? sanitize_text_field( wp_unslash($quiz['title']) ) : "";

        $result = "<span>". $quiz_title ."<span>";
        if ( $quiz_title != "" ) {
            $result = sprintf( '<a href="?page=%s&action=edit&quiz=%d" target="_blank">%s</a>', 'quiz-maker', $quiz_id, $quiz_title );
        }

        return $result;
    }

    /**
     * Method for name column
     *
     * @param array $item an array of DB data
     *
     * @return string
     */
    function column_review( $item ) {

        // Run a security check.
        if (empty($this->ays_quiz_nonce) || ! wp_verify_nonce( $this->ays_quiz_nonce, 'ays_quiz_admin_all_reviews_list_table_nonce' ) ) {
            // This nonce is not valid.
            wp_die( esc_html__( 'Nonce verification failed!', 'quiz-maker' ) );
        }

        $column_t = (isset( $item['review'] ) && $item['review'] != '') ? stripcslashes( nl2br( trim($item['review']) ) ) : '';
        $t = esc_attr($column_t);

        $review_title_length = intval( $this->title_length );
        
        $restitle = Quiz_Maker_Admin::ays_restriction_string("word", $column_t, $review_title_length);

        $title = sprintf( '<span title="%s">%s</span>', $t, $restitle );

        return $title;
    }

    /**
     *  Associative array of columns
     *
     * @return array
     */
    function get_columns() {
        $columns = array(
            'cb'          => '<input type="checkbox" />',
            'user_id'     => __( 'WP User', 'quiz-maker' ),
            'quiz_id'     => __( 'Quiz', 'quiz-maker' ),
            'user_ip'     => __( 'User IP', 'quiz-maker' ),
            'user_name'   => __( 'Name', 'quiz-maker' ),
            'user_email'  => __( 'Email', 'quiz-maker' ),
            'rate_date'   => __( 'Rate Date', 'quiz-maker' ),
            'score'       => __( 'Rate', 'quiz-maker' ),
            'review'      => __( 'Review', 'quiz-maker' ),
            'id'          => __( 'ID', 'quiz-maker' ),
        );

        return $columns;
    }


    /**
     * Columns to make sortable.
     *
     * @return array
     */
    public function get_sortable_columns() {
        $sortable_columns = array(
            'user_id'       => array( 'user_id', true ),
            'user_ip'       => array( 'user_ip', true ),
            'rate_date'     => array( 'rate_date', true ),
            'score'         => array( 'score', true ),
            'user_name'     => array( 'user_name', true ),
            'user_email'    => array( 'user_email', true ),
            'id'            => array( 'id', true ),
        );

        return $sortable_columns;
    }

    /**
     * Columns to make sortable.
     *
     * @return array
     */

    /**
     * Returns an associative array containing the bulk action
     *
     * @return array
     */
    public function get_bulk_actions() {
        $actions = array(
            'bulk-delete'        => __('Delete', 'quiz-maker'),
            'bulk-delete-review' => __('Delete only review', 'quiz-maker'),
        );

        return $actions;
    }


    /**
     * Handles data query and filter, sorting, and pagination.
     */
    public function prepare_items() {

        // Run a security check.
        if (empty($this->ays_quiz_nonce) || ! wp_verify_nonce( $this->ays_quiz_nonce, 'ays_quiz_admin_all_reviews_list_table_nonce' ) ) {
            // This nonce is not valid.
            wp_die( esc_html__( 'Nonce verification failed!', 'quiz-maker' ) );
        }

        $this->_column_headers = $this->get_column_info();

        /** Process bulk action */
        $this->process_bulk_action();

        $per_page     = $this->get_items_per_page( 'quiz_all_reviews_per_page', 50 );

        $current_page = $this->get_pagenum();
        $total_items  = $this->record_count();

        $this->set_pagination_args( array(
            'total_items' => $total_items, //WE have to calculate the total number of items
            'per_page'    => $per_page //WE have to determine how many items to show on a page
        ) );

        $this->items = $this->get_reviews( $per_page, $current_page );
    }

    public function process_bulk_action() {
        // Detect when a bulk action is being triggered.
        $action = $this->current_action();
        if ( ! $action ) {
            return;
        }

        if( !is_user_logged_in()){
            return;
        }

        // Verify unauthorized requests
        if( !current_user_can( $this->current_user_can_edit ) ){
            return;
        }

        if( current_user_can( $this->current_user_can_edit ) && is_user_logged_in() ){

            //Detect when a bulk action is being triggered...
            $message = 'deleted';
            if ( 'delete' === $this->current_action() ) {

                // In our file that handles the request, verify the nonce.
                $nonce = esc_attr( $_REQUEST['_wpnonce'] );

                if ( ! wp_verify_nonce( $nonce, $this->plugin_name . '-delete-result' ) ) {
                    die( 'Go get a life script kiddies' );
                }
                else {
                    $this->delete_reviews( absint( $_GET['result'] ) );

                    // esc_url_raw() is used to prevent converting ampersand in url to "#038;"
                    // add_query_arg() return the current url

                    $url = esc_url_raw( remove_query_arg(array('action', 'result', '_wpnonce')  ) ) . '&status=' . $message;
                    wp_safe_redirect( $url );
                    exit();
                }

            }

            // If the delete bulk action is triggered
            if ( ( isset( $_POST['action'] ) && $_POST['action'] == 'bulk-delete' )
                || ( isset( $_POST['action2'] ) && $_POST['action2'] == 'bulk-delete' )
            ) {

                $delete_ids = ( isset( $_POST['bulk-delete'] ) && ! empty( $_POST['bulk-delete'] ) ) ? esc_sql( $_POST['bulk-delete'] ) : array();

                // loop over the array of record IDs and delete them
                foreach ( $delete_ids as $id ) {
                    $this->delete_reviews( $id );

                }

                // esc_url_raw() is used to prevent converting ampersand in url to "#038;"
                // add_query_arg() return the current url

                $url = esc_url_raw( remove_query_arg(array('action', 'result', '_wpnonce')  ) ) . '&status=' . $message;
                wp_safe_redirect( $url );
                exit();
            } elseif ((isset($_POST['action']) && $_POST['action'] == 'bulk-delete-review')
                      || (isset($_POST['action2']) && $_POST['action2'] == 'bulk-delete-review')
            ) {

                $review_ids = ( isset( $_POST['bulk-delete'] ) && ! empty( $_POST['bulk-delete'] ) ) ? esc_sql( $_POST['bulk-delete'] ) : array();

                // loop over the array of record IDs and mark as read them

                foreach ( $review_ids as $id ) {
                    $this->delete_only_reviews( $id );
                }

                // esc_url_raw() is used to prevent converting ampersand in url to "#038;"
                // add_query_arg() return the current url
                $url = esc_url_raw( remove_query_arg(array('action', 'result', '_wpnonce')  ) ) . '&status=' . $message;
                wp_safe_redirect( $url );
                exit();
            }
        } else {
            return;
        }
    }

    public function reviews_notices(){

        // Run a security check.
        if (empty($this->ays_quiz_nonce) || ! wp_verify_nonce( $this->ays_quiz_nonce, 'ays_quiz_admin_all_reviews_list_table_nonce' ) ) {
            // This nonce is not valid.
            wp_die( esc_html__( 'Nonce verification failed!', 'quiz-maker' ) );
        }

        if( !is_user_logged_in()){
            return;
        }

        // Verify unauthorized requests
        if( !current_user_can( $this->current_user_can_edit ) ){
            return;
        }

        if( empty($_REQUEST['status']) ){
            return;
        }

        $status = (isset($_REQUEST['status'])) ? sanitize_text_field( $_REQUEST['status'] ) : '';

        if ( empty( $status ) )
            return;

        if ( 'created' == $status )
            $updated_message = esc_html( __( 'Quiz created.', 'quiz-maker' ) );
        elseif ( 'deleted' == $status )
            $updated_message = esc_html( __( 'Review(s) deleted.', 'quiz-maker' ) );

        if ( empty( $updated_message ) )
            return;

        ?>
        <div class="ays-quiz-admin-notice notice notice-success is-dismissible">
            <p> <?php echo esc_html($updated_message); ?> </p>
        </div>
        <?php
    }
}
