<?php

/**
 * @package: provesrc-plugin
 */

/**
 * Plugin Name: ProveSource
 * Description: ProveSource is a social proof marketing platform that works with your Wordpress and WooCommerce websites out of the box
 * Version: 4.0.0
 * Author: ProveSource LTD
 * Author URI: https://provesrc.com
 * License: GPLv3 or later
 * Text Domain: provesrc-plugin
 * 
 * WC requires at least: 3.0
 * WC tested up to: 10.1
 */

if (!defined('ABSPATH')) {
    die;
}

/** constants */
define('PROVESRC_HOST', 'https://api.provesrc.com');
define('PROVESRC_VERSION', '4.0.0');
define('PROVESRC_OPTIONS_GROUP', 'provesrc_options');

define('PROVESRC_OPTION_API_KEY', 'provesrc_api_key');
define('PROVESRC_OPTION_WEBHOOK_SECRET', 'provesrc_webhook_secret');
define('PROVESRC_OPTION_DEBUG_KEY', 'provesrc_debug');
define('PROVESRC_OPTION_EVENTS_KEY', 'provesrc_events');
define('PROVESRC_OPTION_TOS_KEY', 'provesrc_tos_accepted');
define('PROVESRC_OPTION_ANALYTICS_KEY', 'provesrc_analytics_consent');

/* hooks */
add_action('before_woocommerce_init', function () {
    if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
});
add_action('admin_menu', 'provesrc_admin_menu'); //1.5.0
add_action('admin_init', 'provesrc_admin_init'); //2.5.0
add_action('admin_notices', 'provesrc_admin_notice_html');
add_action('wp_head', 'provesrc_inject_code'); //1.2.0

// WooCommerce hooks
add_action('woocommerce_new_order', 'provesrc_woocommerce_hook_handler', 999, 1);
add_action('woocommerce_thankyou', 'provesrc_woocommerce_hook_handler', 999, 1);
add_action('woocommerce_checkout_create_order', 'provesrc_woocommerce_hook_handler', 999, 2);
add_action('woocommerce_checkout_order_processed', 'provesrc_woocommerce_hook_handler', 999, 3);
add_action('woocommerce_order_status_pending', 'provesrc_woocommerce_hook_handler', 999, 1);
add_action('woocommerce_order_status_processing', 'provesrc_woocommerce_hook_handler', 999, 1);
add_action('woocommerce_order_status_completed', 'provesrc_woocommerce_hook_handler', 999, 1);
add_action('woocommerce_payment_complete', 'provesrc_woocommerce_hook_handler', 999, 1);

register_uninstall_hook(__FILE__, 'provesrc_uninstall_hook');
register_activation_hook(__FILE__, 'provesrc_activation_hook');
register_deactivation_hook(__FILE__, 'provesrc_deactivation_hook');

add_action('wp_ajax_provesrc_import_orders', 'provesrc_import_orders');
add_action('wp_ajax_provesrc_debug_log', 'provesrc_debug_log');

function provesrc_admin_menu()
{
    add_menu_page('ProveSource Settings', 'ProveSource', 'manage_options', 'provesrc', 'provesrc_admin_menu_page_html', 'dashicons-provesrc');
}

function provesrc_admin_init()
{
    wp_enqueue_style('provesrc_admin_style', plugin_dir_url(__FILE__) . 'style.css', array(), PROVESRC_VERSION);
    // Register all settings first
    register_setting(PROVESRC_OPTIONS_GROUP, PROVESRC_OPTION_API_KEY, array(
        'type' => 'string',
        'sanitize_callback' => 'sanitize_text_field',
    ));
    register_setting(PROVESRC_OPTIONS_GROUP, PROVESRC_OPTION_WEBHOOK_SECRET, array(
        'type' => 'string',
        'sanitize_callback' => 'sanitize_text_field',
    ));
    register_setting(PROVESRC_OPTIONS_GROUP, PROVESRC_OPTION_DEBUG_KEY, array(
        'type' => 'boolean',
        'sanitize_callback' => 'rest_sanitize_boolean',
    ));
    register_setting(PROVESRC_OPTIONS_GROUP, PROVESRC_OPTION_EVENTS_KEY, array(
        'type' => 'array',
        'sanitize_callback' => 'provesrc_sanitize_events_array',
    ));
    register_setting(PROVESRC_OPTIONS_GROUP, PROVESRC_OPTION_TOS_KEY, array(
        'type' => 'boolean',
        'sanitize_callback' => 'rest_sanitize_boolean',
    ));
    register_setting(PROVESRC_OPTIONS_GROUP, PROVESRC_OPTION_ANALYTICS_KEY, array(
        'type' => 'boolean',
        'sanitize_callback' => 'rest_sanitize_boolean',
    ));
    wp_enqueue_style('dashicons-provesrc', plugin_dir_url(__FILE__) . '/assets/css/dashicons-provesrc.css', array(), PROVESRC_VERSION);

    if (isset($_POST['option_page']) && sanitize_text_field(wp_unslash($_POST['option_page'])) === PROVESRC_OPTIONS_GROUP) {
        // Verify nonce for settings form submission
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), PROVESRC_OPTIONS_GROUP . '-options')) {
            wp_die('Security check failed. Please try again.');
        }
        
        // Get submitted values
        $newApiKey = isset($_POST[PROVESRC_OPTION_API_KEY]) ? sanitize_text_field(wp_unslash($_POST[PROVESRC_OPTION_API_KEY])) : '';
        $newWebhookSecret = isset($_POST[PROVESRC_OPTION_WEBHOOK_SECRET]) ? sanitize_text_field(wp_unslash($_POST[PROVESRC_OPTION_WEBHOOK_SECRET])) : '';
        $newEventsSubmitted = isset($_POST[PROVESRC_OPTION_EVENTS_KEY]) ? array_map('sanitize_text_field', wp_unslash($_POST[PROVESRC_OPTION_EVENTS_KEY])) : array();
        $tosSubmitted = isset($_POST[PROVESRC_OPTION_TOS_KEY]) ? rest_sanitize_boolean(sanitize_text_field(wp_unslash($_POST[PROVESRC_OPTION_TOS_KEY]))) : false;
        
        // Terms of Service is always required
        if (!$tosSubmitted) {
            add_settings_error(
                PROVESRC_OPTION_API_KEY,
                'tos_not_accepted',
                'You must accept the Terms of Service to use ProveSource.',
                'error'
            );
            return;
        }

        // Only proceed if we have both API key and webhook secret
        if (empty($newApiKey) || empty($newWebhookSecret)) {
            return;
        }

        // Get previous values before WordPress saves the new ones
        $previousApiKey = provesrc_get_api_key();
        $previousWebhookSecret = provesrc_get_webhook_secret();
        $sendOrders = $previousApiKey !== $newApiKey || $previousWebhookSecret !== $newWebhookSecret;
        provesrc_api_key_updated($newApiKey, $newWebhookSecret, $sendOrders, $newEventsSubmitted);
    }
}

function provesrc_inject_code()
{
    // global $wp;
    // $url = home_url($wp->request);
    // if(strpos($url, 'exclude1') > -1 || strpos($url, 'exclude2') > -1) {
    //     return;
    // }
    $version = PROVESRC_VERSION;
    $apiKey = provesrc_get_api_key(); ?>

    <!-- Start of Async ProveSource Code (Wordpress / Woocommerce v<?php echo esc_html($version); ?>) --><script>!function(o,i){window.provesrc&&window.console&&console.error&&console.error("ProveSource is included twice in this page."),provesrc=window.provesrc={dq:[],display:function(){this.dq.push(arguments)}},o._provesrcAsyncInit=function(){provesrc.init({apiKey:"<?php echo esc_html($apiKey); ?>",v:"0.0.4"})};var r=i.createElement("script");r.async=!0,r["ch"+"ar"+"set"]="UTF-8",r.src="https://cdn.provesrc.com/provesrc.js";var e=i.getElementsByTagName("script")[0];e.parentNode.insertBefore(r,e)}(window,document);</script><!-- End of Async ProveSource Code -->
<?php
}

function provesrc_woocommerce_hook_handler($arg1, $arg2 = null, $arg3 = null)
{
    $selectedEvents = get_option(PROVESRC_OPTION_EVENTS_KEY, array());
    $currentEvent = current_filter();
    if (!$selectedEvents) {
        $selectedEvents = ['woocommerce_checkout_order_processed', 'woocommerce_order_status_completed'];
    }
    if (!in_array($currentEvent, (array)$selectedEvents)) {
        provesrc_log('order handler skipping event', ['current' => $currentEvent, 'selected' => $selectedEvents]);
        return;
    }
    try {
        switch ($currentEvent) {
            case 'woocommerce_checkout_create_order':
                provesrc_order_created_hook($arg1, $arg2);
                break;
            case 'woocommerce_checkout_order_processed':
                provesrc_order_processed($arg1, $arg2, $arg3);
                break;
            default:
                provesrc_order_id_hook($arg1);
                break;
        }
    } catch (Exception $err) {
        provesrc_handle_error('Failed to process order from event: ' . $currentEvent, $err, ['arg1' => $arg1, 'arg2' => $arg2, 'arg3' => $arg3]);
    }
}

function provesrc_order_created_hook($order, $data)
{
    try {
        provesrc_log('woocommerce order created', ['order' => $order]);
        provesrc_send_webhook($order);
    } catch (Exception $err) {
        provesrc_handle_error('failed to process order created', $err, ['order' => $order]);
    }
}

function provesrc_order_id_hook($id)
{
    try {
        $order = wc_get_order($id);
        provesrc_log('woocommerce order complete', ['id' => $id, 'order' => $order]);
        provesrc_send_webhook($order);
    } catch (Exception $err) {
        provesrc_handle_error('failed to process order complete', $err, ['orderId' => $id]);
    }
}

function provesrc_order_processed($id, $data, $order)
{
    try {
        if (!isset($id) || $id < 1) {
            provesrc_log('woocommerce order event (no id)', $order);
            provesrc_send_webhook($order);
        } else {
            provesrc_log('woocommerce order event (with id)', ['id' => $id, 'order' => $order]);
            provesrc_send_webhook(wc_get_order($id));
        }
    } catch (Exception $err) {
        provesrc_handle_error('failed to process order', $err, ['orderId' => $id]);
    }
}

function provesrc_uninstall_hook()
{
    if (!current_user_can('activate_plugins')) {
        return;
    }
    $apiKey = provesrc_get_api_key();
    if (!$apiKey) {
        return;
    }
}

function provesrc_activation_hook()
{
    if (!current_user_can('activate_plugins')) {
        return;
    }
    $apiKey = provesrc_get_api_key();
    if (!$apiKey) {
        return;
    }
}

function provesrc_deactivation_hook()
{
    if (!current_user_can('activate_plugins')) {
        return;
    }
    $apiKey = provesrc_get_api_key();
    if (!$apiKey) {
        return;
    }
}

function provesrc_api_key_updated($apiKey = null, $webhookSecret = null, $sendOrders = true, $selectedEvents = null)
{
    try {
        // Clear previous transients before making request
        delete_transient('provesrc_api_error');
        delete_transient('provesrc_success_message');

        // Use provided values or fetch from database
        if ($apiKey === null) {
            $apiKey = provesrc_get_api_key();
        }
        if ($webhookSecret === null) {
            $webhookSecret = provesrc_get_webhook_secret();
        }
        if ($selectedEvents === null) {
            $selectedEvents = get_option(PROVESRC_OPTION_EVENTS_KEY, array());
        }
        
        if ($apiKey == null) {
            provesrc_log('bad api key, settings update not sent');
            return array('success' => false, 'error' => 'Invalid API key');
        }
        if ($webhookSecret == null) {
            provesrc_log('bad webhook secret, settings update not sent');
            return array('success' => false, 'error' => 'Invalid webhook secret');
        }

        $data = array(
            'secret' => 'simple-secret',
            'woocommerce' => provesrc_has_woocommerce(),
            'siteUrl' => get_site_url(),
            'siteName' => get_bloginfo('name'),
            'multisite' => is_multisite(),
            'description' => get_bloginfo('description'),
        );
        if ($sendOrders) {
            $orders = array();
            if (provesrc_has_woocommerce()) {
                $wcOrders = wc_get_orders(array(
                    'limit' => 30,
                    'orderby' => 'date',
                    'order' => 'DESC'
                ));
                for ($i = 0; $i < count($wcOrders); $i++) {
                    $orders[] = provesrc_get_order_payload($wcOrders[$i], false);
                }
            }
            $data['orders'] = $orders;
        }

        $data['selectedEvents'] = $selectedEvents;

        $res = provesrc_send_request('/wp/setup', $data, false, $apiKey, $webhookSecret);
        $response_code = wp_remote_retrieve_response_code($res);
        $response_body = wp_remote_retrieve_body($res);
        $response_data = json_decode($response_body, true);
        if ($response_code != 200) {
            if (isset($response_data['error'])) {
                $error_message = $response_data['error'];
            } else {
                $error_message = 'unexpected error ' . $response_code;
            }
            provesrc_log('/wp/setup failed: ' . $error_message);
            set_transient('provesrc_api_error', $error_message);
            return array('success' => false, 'error' => $error_message, 'response_code' => $response_code);
        } else {
            $success_message = isset($response_data['successMessage']) ? $response_data['successMessage'] : 'Setup completed successfully';
            set_transient('provesrc_success_message', $success_message);
            provesrc_log('/wp/setup complete');
            return array('success' => true, 'message' => $success_message);
        }
    } catch (Exception $err) {
        provesrc_handle_error('failed updating settings', $err);
        return array('success' => false, 'error' => 'Exception: ' . $err->getMessage());
    }
}

/** hooks - END */

/** helpers */

function provesrc_send_webhook($order)
{
    try {
        $data = provesrc_get_order_payload($order);
        return provesrc_send_request('/webhooks/track/woocommerce', $data);
    } catch (Exception $err) {
        provesrc_handle_error('failed to send webhook', $err, $order);
    }
}

function provesrc_get_order_payload($order, $userInitiated = false)
{
    if (is_a($order, 'WC_Order_Refund')) {
        $order = wc_get_order($order->get_parent_id());
    }
    if (!is_a($order, 'WC_Order')) {
        return array();
    }
    $ip = $order->get_customer_ip_address();
    $ips = provesrc_get_ips();
    $location = null;
    if ($userInitiated) {
        $ips = [$ip];
        if (class_exists('WC_Geolocation')) {
            $geo = new WC_Geolocation();
            $userip = $geo->get_ip_address();
            $location = $geo->geolocate_ip($userip);
            $location['ip'] = $userip;
        }
    }
    if (!in_array($ip, $ips)) {
        array_unshift($ips, $ip);
    }
    $payload = array(
        'orderId' => $order->get_id(),
        'firstName' => $order->get_billing_first_name(),
        'lastName' => $order->get_billing_last_name(),
        'email' => $order->get_billing_email(),
        'ip' => $ips[0],
        'ips' => $ips,
        'siteUrl' => get_site_url(),
        'total' => (int) $order->get_total(),
        'currency' => $order->get_currency(),
        'products' => provesrc_get_products_array($order),
        'billingAddress' => $order->get_address('billing'),
        'shippingAddress' => $order->get_address('shipping'),
    );
    if ($location) {
        $payload['wooLocation'] = $location;
    }
    $countryCode = $order->get_billing_country();
    if (empty($countryCode)) {
        $countryCode = $order->get_shipping_country();
    }
    $city = $order->get_billing_city();
    if (empty($city)) {
        $city = $order->get_shipping_city();
    }
    $stateCode = $order->get_billing_state();
    if (empty($stateCode)) {
        $stateCode = $order->get_shipping_state();
    }
    $payload['location'] = array(
        'countryCode' => $countryCode,
        'stateCode' => $stateCode,
        'city' => $city,
    );
    if (method_exists($order, 'get_date_created')) {
        $date = $order->get_date_created();
        if (!empty($date) && method_exists($date, 'getTimestamp')) {
            $payload['date'] = $order->get_date_created()->getTimestamp() * 1000;
        }
    }
    return $payload;
}

function provesrc_get_products_array($order)
{
    $items = $order->get_items();
    $products = array();
    foreach ($items as $item) {
        try {
            $quantity = $item->get_quantity();
            $product = $item->get_product();
            if (!is_object($product)) {
                $p = array(
                    'id' => $item->get_id(),
                    'name' => $item->get_name(),
                );
            } else {
                $images_arr = wp_get_attachment_image_src($product->get_image_id(), array('72', '72'), false);
                $image = null;
                if ($images_arr !== null && $images_arr[0] !== null) {
                    $image = $images_arr[0];
                    if (is_ssl()) {
                        $image = str_replace('http', 'https', $image);
                    }
                }
                $p = array(
                    'id' => $product->get_id(),
                    'quantity' => (int) $quantity,
                    'price' => (int) $product->get_price(),
                    'name' => $product->get_title(),
                    'link' => get_permalink($product->get_id()),
                    'image' => $image,
                );
            }
            array_push($products, $p);
        } catch (Exception $err) {
            provesrc_log('failed processing line item', $err);
        }
    }
    return $products;
}

function provesrc_send_error($message, $err, $data = null)
{
    try {
        $payload = array(
            'message' => $message,
            'err' => provesrc_encode_exception($err),
            'data' => $data,
        );
        $apiKey = provesrc_get_api_key();
        $headers = array(
            'Content-Type' => 'application/json',
            'x-plugin-version' => PROVESRC_VERSION,
            'x-site-url' => get_site_url(),
            'Authorization' => "Bearer $apiKey"
        );
        return wp_remote_post(PROVESRC_HOST . '/webhooks/wp-error', array(
            'headers' => $headers,
            'body' => json_encode($payload),
        ));
    } catch (Exception $err) {
        provesrc_log('failed sending error', $err);
    }
}

function provesrc_send_request($path, $data, $ignoreAuth = false, $apiKey = null, $webhookSecret = null)
{
    try {
        $headers = array(
            'Content-Type' => 'application/json',
            'x-plugin-version' => PROVESRC_VERSION,
            'x-site-url' => get_site_url(),
            'x-wp-version' => get_bloginfo('version'),
            'x-api-version' => 2,
        );

        // Use provided values or fetch from database
        if ($apiKey === null) {
            $apiKey = provesrc_get_api_key();
        }
        if ($webhookSecret === null) {
            $webhookSecret = provesrc_get_webhook_secret();
        }
        
        if (!$ignoreAuth && ($apiKey == null || $webhookSecret == null)) {
            return;
        }
        if (!empty($apiKey)) {
            $headers['authorization'] = "Bearer $apiKey";
        }
        if(!empty($webhookSecret)) {
            $headers['x-webhook-secret'] = $webhookSecret;
        }

        if (provesrc_has_woocommerce()) {
            $headers['x-woo-version'] = WC()->version;
        }

        $url = PROVESRC_HOST . $path;
        $requestData = array(
            'headers' => $headers,
            'body' => json_encode($data),
        );
        provesrc_log('sending request', ['url' => $url]);
        $res = wp_remote_post($url, $requestData);
        provesrc_log('got response ' . $url, $res);
        return $res;
    } catch (Exception $err) {
        provesrc_handle_error('failed sending request', $err, $data);
    }
}

function provesrc_handle_error($message, $err, $data = null)
{
    provesrc_log($message, $err);
    provesrc_send_error($message, $err, $data);
}

function provesrc_get_api_key()
{
    return get_option(PROVESRC_OPTION_API_KEY);
}

function provesrc_get_webhook_secret()
{
    return get_option(PROVESRC_OPTION_WEBHOOK_SECRET);
}

function provesrc_get_debug()
{
    return get_option(PROVESRC_OPTION_DEBUG_KEY, false);
}

function provesrc_get_tos_accepted()
{
    return get_option(PROVESRC_OPTION_TOS_KEY, false);
}

function provesrc_get_analytics_consent()
{
    return get_option(PROVESRC_OPTION_ANALYTICS_KEY, false);
}

function provesrc_isvalid_api_key($apiKey)
{
    if (isset($apiKey) && strlen($apiKey) > 30) {
        $start = strpos($apiKey, '.');
        $end = strpos($apiKey, '.', $start + 1);
        $substr = substr($apiKey, $start + 1, $end - $start - 1);
        $json = json_decode(base64_decode($substr));

        if (is_object($json) && isset($json->accountId)) {
            return true;
        }
    }
    return false;
}

function provesrc_get_ips()
{
    $ips = [];
    if (isset($_SERVER['HTTP_CLIENT_IP'])) {
        array_push($ips, filter_var(wp_unslash($_SERVER['HTTP_CLIENT_IP']), FILTER_VALIDATE_IP));
    } else if (isset($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        array_push($ips, filter_var(wp_unslash($_SERVER['HTTP_X_FORWARDED_FOR']), FILTER_VALIDATE_IP));
    } else if (isset($_SERVER['HTTP_X_FORWARDED'])) {
        array_push($ips, filter_var(wp_unslash($_SERVER['HTTP_X_FORWARDED']), FILTER_VALIDATE_IP));
    } else if (isset($_SERVER['HTTP_FORWARDED_FOR'])) {
        array_push($ips, filter_var(wp_unslash($_SERVER['HTTP_FORWARDED_FOR']), FILTER_VALIDATE_IP));
    } else if (isset($_SERVER['HTTP_FORWARDED'])) {
        array_push($ips, filter_var(wp_unslash($_SERVER['HTTP_FORWARDED']), FILTER_VALIDATE_IP));
    } else if (isset($_SERVER['REMOTE_ADDR'])) {
        array_push($ips, filter_var(wp_unslash($_SERVER['REMOTE_ADDR']), FILTER_VALIDATE_IP));
    } else if (isset($_SERVER['HTTP_X_REAL_IP'])) {
        array_push($ips, filter_var(wp_unslash($_SERVER['HTTP_X_REAL_IP']), FILTER_VALIDATE_IP));
    }
    return $ips;
}

function provesrc_import_orders()
{
    if (!isset($_POST['security']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['security'])), 'provesrc_import_orders_nonce')) {
        wp_send_json_error('Invalid request');
        return;
    }

    $transient_key = 'provesrc_last_import_time';
    $rate_limit_seconds = 60;
    $last_import_time = get_transient($transient_key);
    if ($last_import_time) {
        $current_time = current_time('timestamp');
        $time_since_last_import = $current_time - $last_import_time;
        if ($time_since_last_import < $rate_limit_seconds) {
            wp_send_json_error('Importing past orders can only be triggered once per minute');
            return;
        }
    }

    provesrc_log('importing last orders manually');
    
    // Use provesrc_api_key_updated with sendOrders = true to send full setup with orders
    $result = provesrc_api_key_updated(null, null, true);
    
    if ($result['success']) {
        set_transient($transient_key, current_time('timestamp'), $rate_limit_seconds);
        wp_send_json_success('Import orders completed');
    } else {
        $error_message = isset($result['error']) ? $result['error'] : 'Unknown error';
        $response_code = isset($result['response_code']) ? $result['response_code'] : 500;
        wp_send_json_error('Failed to import orders: ' . $error_message, $response_code);
        provesrc_handle_error('failed sending request', $error_message);
    }
}

function provesrc_debug_log()
{
    if (!isset($_POST['security']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['security'])), 'provesrc_debug_log_nonce')) {
        wp_send_json_error('Invalid request');
        return;
    }

    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }

    $log_file = plugin_dir_path(__FILE__) . 'debug.log';

    if (file_exists($log_file)) {
        $log_content = file_get_contents($log_file);
        wp_send_json_success($log_content);
    } else {
        wp_send_json_error('Debug log file not found');
    }
}


function provesrc_has_woocommerce()
{
    return in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')));
}

function provesrc_admin_menu_page_html()
{
    if (!current_user_can('manage_options')) {
        return;
    }

    $apiKey = provesrc_get_api_key(); 
    $webhookSecret = provesrc_get_webhook_secret();
    $selectedEvents = get_option(PROVESRC_OPTION_EVENTS_KEY, array());
    $tosAccepted = $apiKey ? true : provesrc_get_tos_accepted();
    if (!$selectedEvents) {
        $selectedEvents = ['woocommerce_checkout_order_processed', 'woocommerce_order_status_completed'];
    }
    $woocommerce_hooks = [
        'woocommerce_order_status_completed' => 'Order Status Completed (Recommended)',
        'woocommerce_checkout_order_processed' => 'Checkout Order Processed (Recommended)',
        'woocommerce_order_status_pending' => 'Order Status Pending Payment',
        'woocommerce_order_status_processing' => 'Order Status Processing',
        'woocommerce_checkout_create_order' => 'Checkout Order Created',
        'woocommerce_payment_complete' => 'Payment Complete',
        'woocommerce_thankyou' => 'Thank You',
        'woocommerce_new_order' => 'New Order',
    ];
    ?>

    <div class="wrap" id="ps-settings">
        <!-- <h1><?php esc_html(get_admin_page_title()); ?></h1> -->
        <a href="https://provesrc.com">
            <img class="top-logo" src="<?php echo esc_url(plugin_dir_url(__FILE__) . 'assets/top-logo.png'); ?>">
        </a>
        <form action="options.php" method="post">
            <?php
            settings_fields(PROVESRC_OPTIONS_GROUP);
            do_settings_sections(PROVESRC_OPTIONS_GROUP);
            ?>
            <div class="ps-settings-container">
                <?php 
                if ($apiKey != null && $webhookSecret != null) { ?>
                    <div class="ps-success">ProveSource is Active</div>
                    <div class="ps-warning">
                        If you still see <strong>"waiting for data..."</strong> in the ProveSource dashboard open your website in <strong>incognito</strong> or <strong>clear cache</strong>
                        <br>If you have <strong>cache or security plugins</strong>, please <a href="http://help.provesrc.com/en/articles/4206151-common-wordpress-woocommerce-issues">see this guide</a> about possible issues and how to solve them
                    </div>
                <?php } elseif (empty($apiKey) || empty($webhookSecret)) { ?>
                    <div class="ps-red-warning">Add your API Key and Webhook Secret below</div>
                    <div class="account-link">If you don't have an account - <a href="https://console.provesrc.com/?utm_source=woocommerce&utm_medium=plugin&utm_campaign=woocommerce-signup#/signup" target="_blank">signup here!</a></div>

                <?php } ?>
                
                <table class="form-table ps-settings-table">
                    <tr>
                        <th scope="row">
                            <label for="provesrc_api_key">Your API Key <span style="color: #dc3232;">*</span></label>
                            <p class="description">Get your <a href="https://console.provesrc.com/#/settings" target="_blank">API Key here</a></p>
                        </th>
                        <td>
                            <input type="text" id="provesrc_api_key" class="ps-apikey regular-text" placeholder="eyJhbG..." name="<?php echo esc_attr(PROVESRC_OPTION_API_KEY); ?>" value="<?php echo esc_attr($apiKey); ?>" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="provesrc_webhook_secret">Webhook Secret <span style="color: #dc3232;">*</span></label>
                            <p class="description">Get your <a href="https://console.provesrc.com/#/settings" target="_blank">Webhook Secret here</a></p>
                        </th>
                        <td>
                            <input type="text" id="provesrc_webhook_secret" class="ps-apikey regular-text" placeholder="550e8400-e29b..." name="<?php echo esc_attr(PROVESRC_OPTION_WEBHOOK_SECRET); ?>" value="<?php echo esc_attr(get_option(PROVESRC_OPTION_WEBHOOK_SECRET)); ?>" />
                        </td>
                    </tr>
                    <?php if (provesrc_has_woocommerce()) { ?>
                    <tr>
                        <th scope="row">
                            <label for="woo_events">WooCommerce Events <span style="color: #dc3232;">*</span></label>
                            <p class="description">Select which WooCommerce order/checkout events ProveSource will track:</p>
                        </th>
                        <td>
                            <?php foreach ($woocommerce_hooks as $hook_value => $hook_label) { 
                                $isChecked = in_array($hook_value, (array) $selectedEvents);
                                ?>
                                <label style="display: block; margin-bottom: 5px;">
                                    <input id="woo_events" type="checkbox" 
                                        name="<?php echo esc_attr(PROVESRC_OPTION_EVENTS_KEY . '[]'); ?>" 
                                        value="<?php echo esc_attr($hook_value); ?>"
                                        <?php checked($isChecked); ?> />
                                    <?php echo esc_html($hook_label); ?>
                                </label>
                            <?php } ?>
                        </td>
                    </tr>
                    <?php } ?>
                    <tr>
                        <th scope="row">
                            <label for="ps-toggle">Enable Debug Mode</label>
                            <p class="description"><a href="#" id="download_debug_log">Download Debug Log</a></p>
                        </th>
                        <td>
                            <div class="ps-toggle">
                                <input type="checkbox" class="ps-toggle-checkbox" id="ps-toggle" tabindex="0"
                                    name="<?php echo esc_attr(PROVESRC_OPTION_DEBUG_KEY); ?>" <?php if (provesrc_get_debug()) { echo "checked"; } ?>>
                                <label class="ps-toggle-label" for="ps-toggle"></label>
                            </div>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="analytics_checkbox">Analytics Consent</label>
                        </th>
                        <td>
                            <label>
                                <input type="checkbox" name="<?php echo esc_attr(PROVESRC_OPTION_ANALYTICS_KEY); ?>" value="1" <?php checked(provesrc_get_analytics_consent()); ?> id="analytics_checkbox">
                                Allow analytics data about plugin activity and website data (optional)
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="tos_checkbox">Terms of Service <span style="color: #dc3232;">*</span></label>
                        </th>
                        <td>
                            <label>
                                <input type="checkbox" name="<?php echo esc_attr(PROVESRC_OPTION_TOS_KEY); ?>" value="1" <?php checked($tosAccepted); ?> required id="tos_checkbox">
                                By using the ProveSource plugin, you agree to our <a href="https://provesrc.com/terms/" target="_blank">Terms of Service</a>
                            </label>
                            <p class="description">(ProveSource will add provesrc.js to your website and automatically retrieve website name, description, URL and recent orders for initial setup).</p>
                        </td>
                    </tr>
                </table>
                <script type="text/javascript">
                    jQuery(document).ready(function($) {
                        $('#download_debug_log').on('click', function(e) {
                            e.preventDefault();
                            $.ajax({
                                url: ajaxurl,
                                type: 'POST',
                                data: {
                                    action: 'provesrc_debug_log',
                                    security: '<?php echo esc_js(wp_create_nonce("provesrc_debug_log_nonce")); ?>'
                                },
                                success: function(response) {
                                    if (response.success) {
                                        var blob = new Blob([response.data], {
                                            type: 'text/plain'
                                        });
                                        var downloadUrl = URL.createObjectURL(blob);
                                        var a = document.createElement('a');
                                        a.href = downloadUrl;
                                        a.download = 'debug.log';
                                        document.body.appendChild(a);
                                        a.click();
                                        document.body.removeChild(a);
                                        URL.revokeObjectURL(downloadUrl);
                                    } else {
                                        alert('Failed to download debug log: ' + response.data);
                                    }
                                },
                                error: function(xhr, status, error) {
                                    alert('An error occurred: ' + error);
                                }
                            });
                        });
                    });
                </script>
            </div>
            <style>
                .ps-spinner {
                    display: inline-block;
                    width: 16px;
                    height: 16px;
                    border: 2px solid #f3f3f3;
                    border-top: 2px solid #0073aa;
                    border-radius: 50%;
                    animation: ps-spin 1s linear infinite;
                    margin-right: 8px;
                }
                @keyframes ps-spin {
                    0% { transform: rotate(0deg); }
                    100% { transform: rotate(360deg); }
                }
                .ps-loading {
                    opacity: 0.6;
                    pointer-events: none;
                }
            </style>
            <script type="text/javascript">
                jQuery(document).ready(function($) {
                    function toggleButtons() {
                        var tosChecked = $('#tos_checkbox').is(':checked');
                        var apiKey = $('[name="<?php echo esc_js(PROVESRC_OPTION_API_KEY); ?>"]').val();
                        var webhookSecret = $('[name="<?php echo esc_js(PROVESRC_OPTION_WEBHOOK_SECRET); ?>"]').val();
                        
                        // Save button is enabled if Terms of Service is checked (API key and webhook secret are optional)
                        if (tosChecked) {
                            $('#submit').prop('disabled', false);
                        } else {
                            $('#submit').prop('disabled', true);
                        }
                        
                        // Import button is only enabled if API key is valid and webhook secret is provided
                        if (apiKey && webhookSecret && webhookSecret.trim() !== '' && tosChecked) {
                            $('#import_orders_button').prop('disabled', <?php echo esc_js(!provesrc_isvalid_api_key($apiKey) ? 'true' : 'false'); ?>);
                        } else {
                            $('#import_orders_button').prop('disabled', true);
                        }
                    }

                    // Add loading state to Save button
                    $('form').on('submit', function() {
                        var $submitBtn = $('#submit');
                        var originalText = $submitBtn.val();
                        
                        $submitBtn.addClass('ps-loading');
                        $submitBtn.val('Saving...');
                        $submitBtn.prepend('<span class="ps-spinner"></span>');
                        
                        // Re-enable after a delay in case of errors
                        setTimeout(function() {
                            $submitBtn.removeClass('ps-loading');
                            $submitBtn.val(originalText);
                            $submitBtn.find('.ps-spinner').remove();
                        }, 10000);
                    });

                    $('#tos_checkbox').on('change', toggleButtons);
                    $('[name="<?php echo esc_html(PROVESRC_OPTION_API_KEY); ?>"]').on('input', toggleButtons);
                    $('[name="<?php echo esc_html(PROVESRC_OPTION_WEBHOOK_SECRET); ?>"]').on('input', toggleButtons);
                    toggleButtons();
                });
            </script>
            <div style="display:flex; align-items:center">
                <div>
                    <?php submit_button('Save'); ?>
                </div>
                <div style="margin-top:7px; margin-left:20px; font-weight: bold">
                    <button
                        <?php echo esc_attr(!provesrc_isvalid_api_key($apiKey) ? 'disabled' : ''); ?>
                        type="button"
                        id="import_orders_button"
                        style=" padding-top:3px; padding-bottom:3px; background-color:#7825f3; border:none"
                        class="button button-primary ">
                        Re-import Last 30 Orders
                    </button>
                    <script type="text/javascript">
                        jQuery(document).ready(function($) {
                            $('#import_orders_button').on('click', function() {
                                var $btn = $(this);
                                var originalText = $btn.text();
                                
                                // Add loading state
                                $btn.addClass('ps-loading');
                                $btn.prop('disabled', true);
                                $btn.text('Importing...');
                                $btn.prepend('<span class="ps-spinner"></span>');
                                
                                $.ajax({
                                    url: ajaxurl,
                                    type: 'POST',
                                    data: {
                                        action: 'provesrc_import_orders',
                                        security: '<?php echo esc_js(wp_create_nonce("provesrc_import_orders_nonce")); ?>'
                                    },
                                    success: function(response) {
                                        // Remove loading state
                                        $btn.removeClass('ps-loading');
                                        $btn.prop('disabled', false);
                                        $btn.text(originalText);
                                        $btn.find('.ps-spinner').remove();
                                        
                                        if (response.success) {
                                            alert('Orders imported successfully!');
                                        } else {
                                            alert(response.data);
                                        }
                                    },
                                    error: function(xhr, status, error) {
                                        // Remove loading state
                                        $btn.removeClass('ps-loading');
                                        $btn.prop('disabled', false);
                                        $btn.text(originalText);
                                        $btn.find('.ps-spinner').remove();
                                        
                                        if (xhr.responseJSON && xhr.responseJSON.data) {
                                            alert(xhr.responseJSON.data);
                                        } else {
                                            alert(error);
                                        }
                                    }
                                });
                            });
                        });
                    </script>
                </div>
            </div>
        </form>
        <p class="ps-version-text">ProveSource WordPress Plugin v<?php echo esc_html(PROVESRC_VERSION); ?></p>
    </div>

<?php
}
function provesrc_admin_notice_html()
{
    $apiKey = provesrc_get_api_key();
    $webhookSecret = provesrc_get_webhook_secret();
    $error_message = get_transient('provesrc_api_error');
    $success_message = get_transient('provesrc_success_message');

    if ($apiKey != null && $webhookSecret != null && !$error_message && !$success_message) {
        return;
    }

    // $screen = get_current_screen();
    // if($screen !== null && strpos($screen->id, 'provesrc') > 0) return;

?>
    <div class="notice is-dismissible <?php echo esc_attr($success_message ? 'notice-success' : 'notice-error'); ?>">
        <?php if ($success_message): ?>
            <p class="ps-success"><?php echo esc_html($success_message); ?></p>
        <?php elseif ($apiKey == null): ?>
            <p class="ps-error">ProveSource API Key is missing! <a href="admin.php?page=provesrc">Click here</a> to configure your API key.</p>
        <?php elseif ($webhookSecret == null): ?>
            <p class="ps-error">ProveSource Webhook Secret is missing! <a href="admin.php?page=provesrc">Click here</a> to configure your webhook secret.</p>
        <?php elseif ($error_message): ?>
            <p class="ps-error"><a href="admin.php?page=provesrc">ProveSource</a> encountered an error: <?php echo esc_html($error_message); ?></p>
        <?php endif; ?>
    </div>
<?php
    if ($success_message) {
        delete_transient('provesrc_success_message');
    }
}

function provesrc_log($message, $data = null)
{
    $debug = provesrc_get_debug();
    if (!$debug) {
        return;
    }
    $log = current_time("Y-m-d\TH:i:s.u ");
    if (isset($data)) {
        $log .= "[ProveSource] " . $message . ": " . print_r($data, true);
    } else {
        $log .= "[ProveSource] " . $message;
    }
    $log .= "\n";
    error_log($log);

    $pluginlog = plugin_dir_path(__FILE__) . 'debug.log';
    error_log($log, 3, $pluginlog);
}

function provesrc_var_dump_str($data)
{
    ob_start();
    var_dump($data);

    return ob_get_clean();
}

function provesrc_encode_exception($err)
{
    if (!isset($err) || is_null($err)) {
        return [];
    }
    return [
        'message' => $err->getMessage(),
        'code' => $err->getCode(),
        'file' => $err->getFile() . ':' . $err->getLine(),
        'trace' => substr($err->getTraceAsString(), 0, 500),
    ];
}

/**
 * Sanitization callback for events array field
 * @param mixed $value The value to sanitize
 * @return array Sanitized array of event names
 */
function provesrc_sanitize_events_array($value)
{
    if (!is_array($value)) {
        return array();
    }
    
    // Define allowed event names for validation
    $allowed_events = array(
        'woocommerce_order_status_completed',
        'woocommerce_order_status_pending',
        'woocommerce_order_status_processing',
        'woocommerce_checkout_create_order',
        'woocommerce_checkout_order_processed',
        'woocommerce_payment_complete',
        'woocommerce_thankyou',
        'woocommerce_new_order'
    );
    
    $sanitized = array();
    for ($i = 0; $i < count($value); $i++) {
        $event = sanitize_key($value[$i]);
        if (in_array($event, $allowed_events)) {
            $sanitized[] = $event;
        }
    }
    
    return $sanitized;
}

/* helpers - END */

?>