<?php
/**
 * Prevent XSS Vulnerability Frontend.
 *
 * @package PreventXSSVulnerability
 */

/**
 * Make detections, redirect and add console message.
 */
class Prevent_XSS_Vulnerability_Frontend {
	/**
	 * JS file suffix (version number with with extension).
	 *
	 * @var string
	 */
	private $js_suffix = '-' . PREVENT_XSS_VULNERABILITY_VERSION . '.min.js';

	/**
	 * Initialize WordPress init Hook.
	 */
	public function init() {
		add_action( 'plugins_loaded', array( $this, 'prevent_reflected_xss' ) );
		add_action( 'wp', array( $this, 'escape_html' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'self_xss_script' ), 100 );
	}

	/**
	 * Remove XSS parameters from the URI (if enabled).
	 *
	 * @access private
	 * @since 1.0.0
	 *
	 * @param string $xss_detection Current page URI.
	 * @param array  $reflected_xss Settings saved from Reflected XSS page.
	 *
	 * @return string URI after removing XSS parameters (if enabled).
	 */
	private function block_xss_params( $xss_detection, $reflected_xss ) {
		if ( 1 === $reflected_xss['enable_blocking'] ) {
			// Entities that needs to be persist.
			$persist_entities  = array(
				'/',
				'=',
				'#',
				'%',
			);
			$remove_xss_params = array(
				'<',
				'>',
				'(',
				')',
				'[',
				']',
				'{',
				'|',
				'}',
			);
			$remove_xss_values = array(
				'%3C',
				'%3E',
				'%28',
				'%29',
				'%5B',
				'%5D',
				'%7B',
				'%7C',
				'%7D',
			);

			if ( isset( $reflected_xss['include_extra_entities'] )
				&& ! empty( $reflected_xss['include_extra_entities'] )
			) {
				$extra_entities = explode( ',', $reflected_xss['include_extra_entities'] );

				foreach ( $extra_entities as $single_entity ) {
					if ( ! empty( $single_entity )
						&& ! in_array( $single_entity, $remove_xss_params, true )
						&& ! in_array( $single_entity, $persist_entities, true )
					) {
						$remove_xss_params[] = $single_entity;
					}
				}
			}

			if ( is_admin() ) {
				unset( $remove_xss_params[4] );
				unset( $remove_xss_params[5] );
				unset( $remove_xss_values[4] );
				unset( $remove_xss_values[5] );
			}

			$exclude_block_keys = array();
			if ( isset( $reflected_xss['exclude_block_entities'] )
				&& ! empty( $reflected_xss['exclude_block_entities'] )
			) {
				$blocked_entities = preg_replace(
					'/\s+/',
					'',
					$reflected_xss['exclude_block_entities']
				);
				$blocked_entities = explode( ',', $blocked_entities );
				$entities_length  = count( $blocked_entities );
				$loop_count       = 0;

				while ( $loop_count < $entities_length ) {
					if ( isset( $blocked_entities[ $loop_count ] )
						&& in_array( $blocked_entities[ $loop_count ], $remove_xss_params, true )
					) {
						$key = array_search(
							$blocked_entities[ $loop_count ],
							$remove_xss_params,
							true
						);

						if ( isset( $key ) && ! empty( $key ) ) {
							$exclude_block_keys[ $key ] = $key;
						}
					}

					++$loop_count;
				}

				$remove_xss_params = array_diff_key(
					$remove_xss_params,
					$exclude_block_keys
				);
				$remove_xss_values = array_diff_key(
					$remove_xss_values,
					$exclude_block_keys
				);
			}

			$xss_detection = str_replace( $remove_xss_params, '', $xss_detection );
			$xss_detection = str_replace( $remove_xss_values, '', $xss_detection );
		}

		return $xss_detection;
	}

	/**
	 * Encode XSS parameters from the URI (if enabled).
	 *
	 * @access private
	 * @since 1.0.0
	 *
	 * @param string $xss_detection URI after removing blocking parameters.
	 * @param array  $reflected_xss Settings saved from Reflected XSS page.
	 *
	 * @return string URI after encoding XSS parameters (if enabled).
	 */
	private function encode_xss_params( $xss_detection, $reflected_xss ) {
		if ( 1 === $reflected_xss['enable_encoding'] ) {
			$encode_params = array(
				' ',
				'!',
				'"',
				'% ',
				"'",
				'(',
				')',
				'*',
				'<',
				'>',
				'`',
				'^',
				'[',
				']',
				'{',
				'|',
				'}',
			);

			$encode_values = array(
				'%20', // Space Encoded Value.
				'%21', // ! Encoded Value.
				'%22', // " Encoded Value.
				'%25', // % Encoded Value.
				'%27', // ' Encoded Value.
				'%28', // ( Encoded Value.
				'%29', // ) Encoded Value.
				'%2A', // * Encoded Value.
				'%3C', // < Encoded Value.
				'%3E', // > Encoded Value.
				'%60', // ` Encoded Value.
				'%5E', // ^ Encoded Value.
				'%5B', // [ Encoded Value.
				'%5D', // ] Encoded Value.
				'%7B', // { Encoded Value.
				'%7C', // | Encoded Value.
				'%7D', // } Encoded Value.
			);

			$exclude_encode_keys = array();
			if ( isset( $reflected_xss['exclude_encode_entities'] )
				&& ! empty( $reflected_xss['exclude_encode_entities'] )
			) {
				$encoded_entities = preg_replace(
					'/\s+/',
					'',
					$reflected_xss['exclude_encode_entities']
				);
				$encoded_entities = explode( ',', $encoded_entities );
				$entities_length  = count( $encoded_entities );
				$loop_count       = 0;

				while ( $loop_count < $entities_length ) {
					if ( isset( $encoded_entities[ $loop_count ] )
						&& in_array( $encoded_entities[ $loop_count ], $encode_params, true )
					) {
						$key = array_search(
							$encoded_entities[ $loop_count ],
							$encode_params,
							true
						);

						if ( isset( $key ) && ! empty( $key ) ) {
							$exclude_encode_keys[ $key ] = $key;
						}
					}

					++$loop_count;
				}

				$encode_params = array_diff_key(
					$encode_params,
					$exclude_encode_keys
				);
				$encode_values = array_diff_key(
					$encode_values,
					$exclude_encode_keys
				);
			}

			$xss_detection = str_replace(
				$encode_params,
				$encode_values,
				$xss_detection
			);
		}

		return $xss_detection;
	}

	/**
	 * Encode and remove the Entities in Page URI and match with Original URI.
	 * If both doesn't match so, it redirects the user to the Encoded
	 * and Removed URI.
	 *
	 * @access private
	 * @since 0.1
	 *
	 * @return void
	 */
	public function prevent_reflected_xss() {
		$reflected_xss = get_option(
			'prevent_xss_vulnerability_reflected_settings'
		);

		if ( ! $reflected_xss ) {
			return;
		}

		if ( is_string( $reflected_xss ) ) {
			$reflected_xss = maybe_unserialize( $reflected_xss );
		}

		$blocking_enabled = (int) $reflected_xss['enable_blocking'];
		$encoding_enabled = (int) $reflected_xss['enable_encoding'];
		if ( ! isset( $reflected_xss )
			|| empty( $reflected_xss )
			|| ( 1 !== $encoding_enabled && 1 !== $blocking_enabled )
		) {
			return;
		}

		$get_page_uri = '';
		if ( isset( $_SERVER['REQUEST_URI'] ) ) {
			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$get_page_uri = wp_unslash( $_SERVER['REQUEST_URI'] );
		}

		$xss_detection = $get_page_uri;
		$xss_detection = $this->block_xss_params( $xss_detection, $reflected_xss );
		$xss_detection = $this->encode_xss_params( $xss_detection, $reflected_xss );

		/*
		 * If the request page URI and the URI after XSS Detection is not the same
		 * then, redirect the user on the URI which doesn't contain XSS Parameters.
		 */
		if ( $get_page_uri !== $xss_detection ) {
			nocache_headers();
			wp_safe_redirect( $xss_detection );
			exit();
		}
	}

	/**
	 * Escaped HTML string in `$_GET`.
	 *
	 * @access public
	 * @since 0.2
	 *
	 * @return void
	 */
	public function escape_html() {
		$reflected_xss = get_option(
			'prevent_xss_vulnerability_reflected_settings'
		);

		if ( ! $reflected_xss ) {
			return;
		}

		if ( is_string( $reflected_xss ) ) {
			$reflected_xss = maybe_unserialize( $reflected_xss );
		}

		if ( isset( $reflected_xss, $reflected_xss['escape_html'] )
			&& 1 === $reflected_xss['escape_html']
		) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$get_all_params = $_GET;
			foreach ( $get_all_params as $key => $value ) {
				if ( is_array( $value ) ) {
					$escape_array = array();
					foreach ( $value as $array_key => $array_value ) {
						$escape_array[ $array_key ] = esc_html( $array_value );
					}

					$_GET[ $key ] = $escape_array;
				} else {
					$_GET[ $key ] = esc_html( $value );
				}
			}
		}
	}

	/**
	 * Add Self XSS message.
	 *
	 * @access public
	 * @since 0.3.0
	 *
	 * @return void
	 */
	public function self_xss_script() {
		$self_xss = get_option( 'prevent_xss_vulnerability_self_xss_settings' );

		if ( ! $self_xss ) {
			return;
		}

		if ( is_string( $self_xss ) ) {
			$self_xss = maybe_unserialize( $self_xss );
		}

		if ( isset( $self_xss, $self_xss['user_warning'] )
			&& 1 === $self_xss['user_warning']
		) {
			$message = '';
			if ( isset( $self_xss['warning_message'] )
				&& ! empty( $self_xss['warning_message'] )
			) {
				$message = esc_html( $self_xss['warning_message'] );
			}

			$js_filename = 'script' . $this->js_suffix;
			$plugin_url  = plugins_url( '/assets', PREVENT_XSS_VULNERABILITY_FILE );
			wp_enqueue_script(
				'prevent-xss-vulnerability',
				$plugin_url . '/js/' . $js_filename,
				array(),
				true,
				true
			);
			wp_localize_script(
				'prevent-xss-vulnerability',
				'selfXss',
				array(
					'message' => $message,
				)
			);
		}
	}
}
