<?php
/**
 * Prevent XSS Vulnerability Self Settings.
 *
 * @package PreventXSSVulnerability
 */

/**
 * Create HTML of Self Settings page.
 */
class Prevent_XSS_Vulnerability_Self_Settings {
	/**
	 * Call Reflected Settings Function.
	 */
	public function __construct() {
		$this->self_xss_settings_page();
	}

	/**
	 * Save Self XSS Settings
	 *
	 * @access private
	 * @since 1.0.0
	 */
	private function save_self_xss_settings() {
		$form_submit = filter_input( INPUT_POST, 'submit' );
		$user_id     = get_current_user_id();

		if ( $form_submit
			&& check_admin_referer(
				'prevent-xss-vulnerability-self_' . $user_id,
				'_prevent_xss_vulnerability_self_nonce'
			)
		) {
			$self_xss = array(
				'warning_message' => '',
				'user_warning'    => 0,
			);

			$warning_message = filter_input( INPUT_POST, 'warning_message' );
			if ( $warning_message ) {
				$self_xss['warning_message'] = wp_kses( $warning_message, array() );
			}

			$self_enabled = (int) filter_input( INPUT_POST, 'user_warning' );
			if ( $self_enabled && 1 === $self_enabled ) {
				$self_xss['user_warning'] = 1;
			}

			update_option( 'prevent_xss_vulnerability_self_xss_settings', $self_xss );
		}
	}

	/**
	 * Self-XSS Settings Page.
	 *
	 * @access private
	 * @since 0.3.0
	 */
	private function self_xss_settings_page() {
		if ( ! current_user_can( 'activate_plugins' ) ) {
			wp_die(
				esc_html_e(
					'You do not have sufficient permissions to access this page.',
					'prevent-xss-vulnerability'
				)
			);
		}

		$this->save_self_xss_settings();

		$user_id      = get_current_user_id();
		$xss_settings = get_option( 'prevent_xss_vulnerability_self_xss_settings' );

		if ( is_string( $xss_settings ) ) {
			$xss_settings = maybe_unserialize( $xss_settings );
		}

		$warning_message = '';
		if ( isset( $xss_settings['warning_message'] )
			&& ! empty( $xss_settings['warning_message'] )
		) {
			$warning_message = $xss_settings['warning_message'];
		}

		$user_warning = '';
		if ( isset( $xss_settings['user_warning'] )
			&& 1 === $xss_settings['user_warning']
		) {
			$user_warning = 'checked';
		}
		?>

		<div class="wrap">
			<h2>
				<?php esc_html_e( 'Self-XSS Settings', 'prevent-xss-vulnerability' ); ?>
			</h2>
			<div>
				<p>
					<?php
					esc_html_e( 'Self-XSS is a type of security vulnerability where an attacker tricks a user into executing malicious code within their own web browser. This can occur through social engineering tactics, such as convincing the user to copy and paste malicious code into their browser\'s console.', 'prevent-xss-vulnerability' );
					?>
				</p>
				<p>
					<?php
						esc_html_e( 'Here\'s a breakdown of how it works:', 'prevent-xss-vulnerability' );
					?>
				</p>
				<ol>
					<li>
						<?php
						esc_html_e( 'Social Engineering: The attacker employs social engineering techniques (e.g., phishing, promises of rewards) to convince the victim to perform a specific action.', 'prevent-xss-vulnerability' );
						?>
					</li>
					<li>
						<?php
						esc_html_e( 'Code Execution: The victim is tricked into executing the malicious code, often by pasting it into the browser\'s console or through other means.', 'prevent-xss-vulnerability' );
						?>
					</li>
					<li>
						<?php
						esc_html_e( 'Compromised Browser: Once executed, the malicious code can potentially steal sensitive information, hijack user sessions, or spread malware.', 'prevent-xss-vulnerability' );
						?>
					</li>
				</ol>

				<p>
					<?php
					esc_html_e( 'Self-XSS attacks exploit user trust and curiosity. It\'s crucial for users to be vigilant and avoid executing any code they don\'t fully understand or trust.', 'prevent-xss-vulnerability' );
					?>
				</p>
			</div>
			<form enctype="multipart/form-data" action="" method="POST" id="reflected-xss">
			<?php
			wp_nonce_field(
				'prevent-xss-vulnerability-self_' . $user_id,
				'_prevent_xss_vulnerability_self_nonce',
				true
			);
			?>
			<table class="prevent-xss reflected-xss">
				<caption>
				<?php
				esc_html_e( 'Warn users about self-XSS attacks', 'prevent-xss-vulnerability' );
				?>
				</caption>
				<tbody>
				<tr>
					<td>
					<?php // phpcs:ignore WordPress.WP.I18n.NonSingularStringLiteralText ?>
					<textarea name="warning_message" rows="5" cols="100"><?php esc_html_e( $warning_message ); ?></textarea>
					<small>
					<?php
						esc_html_e(
							'Customize the warning message that appears in the console. Leave empty to use the default message.',
							'prevent-xss-vulnerability'
						);
					?>
					</small>
					</td>
				</tr>
				<tr>
					<td class="enable">
					<?php // phpcs:ignore WordPress.WP.I18n.NonSingularStringLiteralText ?>
					<input type="checkbox" name="user_warning" value=1 <?php esc_html_e( $user_warning ); ?>/>
					<strong>
					<?php
						esc_html_e( 'Enable User Warning', 'prevent-xss-vulnerability' );
					?>
					</strong>
					</td>
				</tr>
				</tbody>
			</table>

			<p class="submit">
				<input type="submit" name="submit" id="submit" class="button button-primary" value="<?php esc_html_e( 'Save Changes', 'prevent-xss-vulnerability' ); ?>" />
			</p>
			</form>
		</div>

		<?php
	}
}
